#include <stdio.h>
#include <string.h>
#include <allegro.h>
#include "vm.h"

typedef struct {
  const char *name;
  int code;
  int has_param;
} insn_info;

static const insn_info info[INSN_COUNT] = {
  { "nop", INSN_NOP, 0 },
  { "end", INSN_END, 0 },
  { "yield", INSN_YIELD, 0 },
  { "pushtime", INSN_PUSHTIME, 0 },
  { "pushx", INSN_PUSHX, 0 },
  { "pushy", INSN_PUSHY, 0 },
  { "popx", INSN_POPX, 0 },
  { "popy", INSN_POPY, 0 },
  { "push", INSN_PUSH, 1 },
  { "pop", INSN_POP, 0 },
  { "add", INSN_ADD, 0 },
  { "sub", INSN_SUB, 0 },
  { "mul", INSN_MUL, 0 },
  { "div", INSN_DIV, 0 },
  { "mod", INSN_MOD, 0 },
  { "sin", INSN_SIN, 0 },
  { "cos", INSN_COS, 0 },
  { "rand", INSN_RAND, 0 },
  { "ajz", INSN_AJZ, 1 },
  { "rjz", INSN_RJZ, 1 },
  { "pushd", INSN_PUSHD, 0 },
  { "popd", INSN_POPD, 0 },
  { "puship", INSN_PUSHIP, 0 },
  { "peek", INSN_PEEK, 0 },
  { "poke", INSN_POKE, 0 },
  { "pushw", INSN_PUSHW, 0 },
  { "pushh", INSN_PUSHH, 0 },
  { "neg", INSN_NEG, 0 },
};


int add_insn(const char *insn, THREAD *thread)
{
  static char str[64];
  int n;
  for (n=0; n<INSN_COUNT; ++n) {
    if (info[n].has_param) {
      int param;
      int ret = sscanf(insn, "%s %d", str, &param);
      if (ret==2) {
        if (ustricmp(str,info[n].name)==0) {
          // found
          thread->code[thread->ip++] = info[n].code;
          thread->code[thread->ip++] = param;
          return 1;
        }
      }
    }
    else {
      int ret = sscanf(insn, "%s", str);
      if (ret==1) {
        if (ustricmp(str,info[n].name)==0) {
          // found
          thread->code[thread->ip++] = info[n].code;
          thread->code[thread->ip++] = 0;
          return 1;
        }
      }
    }
  }
  fprintf(stderr,"bad insn: %s\n",insn);
  alert("bad insn",insn,"","OK","",13,27);
  return 0;
}


THREAD *parse(const char *name, const char *prog)
{
  char *string = ustrdup(prog);
  THREAD *thread = create_thread(name);
  char *current = ustrtok(string, ";");
  while (current) {
    if (!add_insn(current, thread)) {
      destroy_thread(thread);
      thread = NULL;
      break;
    }
    current = ustrtok(NULL, ";");
  }
  free(string);
  // Check that last insn is END
  if (!thread || thread->ip<2 || thread->code[thread->ip-2]!=INSN_END) {
    static char culprit[265];
    sprintf(culprit,"insn %d: %s",thread->ip/2,prog);
    alert(name,"bad code",culprit,"OK","",13,27);
    fprintf(stderr,"%s: bad code, insn %d: %s\n",name,thread->ip/2,prog);
    destroy_thread(thread);
    thread = NULL;
  }
  if (thread) thread->ip=0;
  return thread;
}

