/*         ______   ___    ___ 
 *        /\  _  \ /\_ \  /\_ \ 
 *        \ \ \L\ \\//\ \ \//\ \      __     __   _ __   ___ 
 *         \ \  __ \ \ \ \  \ \ \   /'__`\ /'_ `\/\`'__\/ __`\
 *          \ \ \/\ \ \_\ \_ \_\ \_/\  __//\ \L\ \ \ \//\ \L\ \
 *           \ \_\ \_\/\____\/\____\ \____\ \____ \ \_\\ \____/
 *            \/_/\/_/\/____/\/____/\/____/\/___L\ \/_/ \/___/
 *                                           /\____/
 *                                           \_/__/
 *
 *      A simple game demonstrating the use of the Allegro library.
 *
 *      By Shawn Hargreaves.
 *
 *      See readme.txt for copyright information.
 */


#include <string.h>
#include <stdio.h>
#include <math.h>

#include "allegro.h"
#include "actions.h"
#include "colors.h"
#include "rmstar.h"
#include "vm.h"
#include "scripts.h"
#include "parser.h"
#include "swscroll.h"
#include "font.h"
#include "cache.h"

#include <allegro/internal/aintern.h>

#if ALLEGRO_VERSION < 4
#define fixmul fmul
#define fixdiv fdiv
#define fixsin fsin
#define fixcos fcos
#endif

void add_alien(const char*,int,int,int);
void generate_aliens();
char current_path[4096];
char last_error[4096]="";
int compute_height();
void cycle_weapon(int);
volatile int allow_cycling_weapons = 1;
void position_alien(int);
void clear_screen();
void display_filename(
  BITMAP *dest,FONT *font,int x,int y,int color,int time,const char *text
);
int laser_on=0;
int laser_hit_y=0;
int laser_hit_alien=-1;
int laser_hit_same_alien=0;
SAMPLE *zap = NULL;
BITMAP *default_image;


/* different ways to update the screen */
#define DOUBLE_BUFFER      1
#define PAGE_FLIP          2
#define RETRACE_FLIP       3
#define TRIPLE_BUFFER      4
#define DIRTY_RECTANGLE    5

int animation_type = 0;


/* command line options */
int cheat = FALSE;
int turbo = FALSE;
int jumpstart = FALSE;


/* our graphics, samples, etc. */
DATAFILE *data;

BITMAP *s;

BITMAP *page1, *page2, *page3;
int current_page = 0;

int use_retrace_proc = FALSE;

PALETTE title_palette;


/* the current score */
volatile int score;
char score_buf[4096];


/* info about the state of the game */
int dead;
int pos;
int xspeed, yspeed, ycounter;
int ship_state;
int ship_burn;
int shot;
int skip_speed, skip_count;
volatile int frame_count, fps;
volatile int game_time;
int thread_time;

#define MAX_SPEED       32
#define SPEED_SHIFT     3


/* for the starfield */
#define MAX_STARS       128

volatile struct {
   fixed x, y, z;
   int ox, oy;
} star[MAX_STARS];


/* info about the aliens (they used to be aliens at least,
 * even if the current graphics look more like asteroids :-) 
 */
#define MAX_ALIENS      512

volatile struct {
   int x, y;
   int d;
   int state;
   int shot;
   char *filename;
   int attribute;
   int text_color;
   BITMAP *bmp;
   THREAD *thread;
} alien[MAX_ALIENS];

volatile int alien_count;
volatile int new_alien_count;


/* explosion graphics */
#define EXPLODE_FLAG    100
#define EXPLODE_FRAMES  64
#define EXPLODE_SIZE    80

RLE_SPRITE *explosion[EXPLODE_FRAMES];


/* position of the bullet */
#define BULLET_SPEED    6

volatile int bullet_type = 0;
volatile int selected_bullet_type = 1;
volatile int bullet_x, bullet_y; 


/* dirty rectangle list */
typedef struct DIRTY_RECT {
   int x, y, w, h;
} DIRTY_RECT;

typedef struct DIRTY_LIST {
   int count;
   DIRTY_RECT rect[2*(MAX_ALIENS+MAX_STARS+512)];
} DIRTY_LIST;

DIRTY_LIST dirty, old_dirty;


/* text messages (loaded from readme.txt) */
char *title_text;
int title_size;
int title_alloced;

char *end_text;


/* for doing stereo sound effects */
#define PAN(x)       (((x) * 256) / SCREEN_W)



/* timer callback for controlling the game speed */
void game_timer(void)
{
   game_time++;
}

END_OF_FUNCTION(game_timer);



/* timer callback for measuring the frames per second */
void fps_proc(void)
{
   fps = frame_count;
   frame_count = 0;
   score++;
   allow_cycling_weapons = 1;
}

END_OF_FUNCTION(fps_proc);



/* the main game update function */
void move_everyone(void)
{
   int c;

   if (shot) {
      /* player dead */
      if (skip_count <= 0) {
         if (ship_state >= EXPLODE_FLAG+EXPLODE_FRAMES-1)
            dead = TRUE;
         else
            ship_state++;

         if (yspeed)
            yspeed--;
      }
   }
   else {
      if (skip_count <= 0) {
         if (laser_on<-64) {
            if (key[KEY_L]) {
               laser_on=32;
               play_sample(zap, 64, PAN(pos>>SPEED_SHIFT), 1000, FALSE);
            }
         }
         else {
           --laser_on;
         }
         if (allow_cycling_weapons) {
            if (key[KEY_PGUP]) cycle_weapon(+1);
            if (key[KEY_PGDN]) cycle_weapon(-1);
         }
         if ((key[KEY_LEFT]) || (joy[0].stick[0].axis[0].d1)) {
            /* moving left */
            if (xspeed > -MAX_SPEED)
               xspeed -= 2;
         }
         else if ((key[KEY_RIGHT]) || (joy[0].stick[0].axis[0].d2)) {
            /* moving right */
            if (xspeed < MAX_SPEED)
               xspeed += 2;
         }
         else {
            /* decelerate */
            if (xspeed > 0)
               xspeed -= 2;
            else if (xspeed < 0)
               xspeed += 2;
         }
      }

      /* which ship sprite to use? */
      if (xspeed < -24)
         ship_state = ship_burn?SHIP1:SHIP1NE;
      else if (xspeed < -2)
         ship_state = ship_burn?SHIP2:SHIP2NE;
      else if (xspeed <= 2)
         ship_state = ship_burn?SHIP3:SHIP3NE;
      else if (xspeed <= 24)
         ship_state = ship_burn?SHIP4:SHIP4NE;
      else
         ship_state = ship_burn?SHIP5:SHIP5NE;

      /* move player */ 
      pos += xspeed;
      if (pos < (32 << SPEED_SHIFT)) {
         pos = (32 << SPEED_SHIFT);
         xspeed = 0;
      }
      else if (pos >= ((SCREEN_W-32) << SPEED_SHIFT)) {
         pos = ((SCREEN_W-32) << SPEED_SHIFT);
         xspeed = 0;
      }

      if (skip_count <= 0) {
         if ((key[KEY_UP]) || (joy[0].stick[0].axis[1].d1) || (turbo)) {
            /* firing thrusters */
            if (yspeed < MAX_SPEED) {
               if (yspeed == 0) {
                  play_sample(data[ENGINE_SPL].dat, 0, PAN(pos>>SPEED_SHIFT), 1000, TRUE);
               }
               else {
                  /* fade in sample while speeding up */
                  adjust_sample(data[ENGINE_SPL].dat, yspeed*64/MAX_SPEED, PAN(pos>>SPEED_SHIFT), 1000, TRUE);
               }
               yspeed++;
            }
            else {
               /* adjust pan while the sample is looping */
               adjust_sample(data[ENGINE_SPL].dat, 64, PAN(pos>>SPEED_SHIFT), 1000, TRUE);
            }

            ship_burn = TRUE;
            score++;
         }
         else {
            /* not firing thrusters */
            if (yspeed) {
               yspeed--;
               if (yspeed == 0) {
                  stop_sample(data[ENGINE_SPL].dat);
               }
               else {
                  /* fade out and reduce frequency when slowing down */
                  adjust_sample(data[ENGINE_SPL].dat, yspeed*64/MAX_SPEED, PAN(pos>>SPEED_SHIFT), 500+yspeed*500/MAX_SPEED, TRUE);
               }
            }

            ship_burn = FALSE;
         }
      }
   }

   /* if going fast, move everyone else down to compensate */
   if (yspeed) {
      ycounter += yspeed;

      while (ycounter >= (1 << SPEED_SHIFT)) {
         if (bullet_type)
            bullet_y++;

         for (c=0; c<MAX_STARS; c++) {
            if (++star[c].oy >= SCREEN_H)
               star[c].oy = 0;
         }

         for (c=0; c<alien_count; c++)
            alien[c].y++;

         ycounter -= (1 << SPEED_SHIFT);
      }

   }

   /* laser beam */
   if (laser_on > 0) {
      int hit = -1;
      laser_hit_y = 0;
      for (c=0; c<alien_count; c++) {
         if ((ABS((pos>>SPEED_SHIFT) - alien[c].x) < 20) && (!alien[c].shot)) {
            if (alien[c].y > laser_hit_y) {
               hit = c;
               laser_hit_y = alien[c].y;
            }
         }
      }
      if (hit != -1) {
         /* an alien was hit by a laser */
         if (laser_hit_alien == hit) {
           ++laser_hit_same_alien;
           if (laser_hit_same_alien >= 24) {
             /* shot enough, kill it */
             dead = TRUE;
             score += 10;
             play_sample(data[BOOM_SPL].dat, 255, PAN((pos>>SPEED_SHIFT)), 1000, FALSE);
             clear_screen();
             shoot_file(alien[hit].filename,selected_bullet_type);
             bullet_type = 0;
             laser_on = 0;
           }
         }
         else {
           laser_hit_same_alien = 0;
         }
         laser_hit_alien = hit;
      }
   }
   else {
     laser_hit_alien = -1;
     laser_hit_same_alien = 0;
   }

   /* move bullet */
   if (bullet_type) {
      bullet_y -= BULLET_SPEED;

      if (bullet_y < 8) {
         bullet_type = 0;
      }
      else {
         /* shot an alien? */
         for (c=0; c<alien_count; c++) {
            if ((ABS(bullet_y - alien[c].y) < 20) && (ABS(bullet_x - alien[c].x) < 20) && (!alien[c].shot)) {
               dead = TRUE;
               score += 10;
               play_sample(data[BOOM_SPL].dat, 255, PAN(bullet_x), 1000, FALSE);
               clear_screen();
               shoot_file(alien[c].filename,bullet_type);
               bullet_type = 0;
               /*
               alien[c].shot = TRUE;
               alien[c].state = EXPLODE_FLAG;
               */
               break; 
            }
         }
      }
   }

   /* move stars */
   for (c=0; c<MAX_STARS; c++) {
      if ((star[c].oy += ((int)star[c].z>>1)+1) >= SCREEN_H)
         star[c].oy = 0;
   }

   /* fire bullet? */
   if (!shot) {
      if ((key[KEY_SPACE]) || (joy[0].button[0].b) || (joy[0].button[1].b)) {
         if (!bullet_type) {
            bullet_x = (pos>>SPEED_SHIFT)-2;
            bullet_y = SCREEN_H-64;
            bullet_type = selected_bullet_type;
            play_sample(data[SHOOT_SPL].dat, 100, PAN(bullet_x), 1000, FALSE);
         } 
      }
   }

   /* move aliens */
   for (c=0; c<alien_count; c++) {
      if (alien[c].shot) {
         /* dying alien */
         if (skip_count <= 0) {
            if (alien[c].state < EXPLODE_FLAG+EXPLODE_FRAMES-1) {
               alien[c].state++;
               if (alien[c].state & 1)
                  alien[c].x += alien[c].d;
            }
            else {
               alien[c].x = 16 + rand() % (SCREEN_W-32);
               alien[c].y = -60 - (rand() & 0x3f);
               alien[c].d = (rand()&1) ? 1 : -1;
               alien[c].shot = FALSE;
               alien[c].state = -1;
            }
         }
      }
      else {
         alien[c].thread->x=alien[c].x;
         alien[c].thread->y=alien[c].y;
         run_thread(alien[c].thread,thread_time,512);
         alien[c].x=alien[c].thread->x;
         alien[c].y=alien[c].thread->y;
         if (alien[c].x < -60)
            alien[c].x = SCREEN_W;
         else if (alien[c].x > SCREEN_W)
            alien[c].x = -60;
      }

      /* move alien vertically */
      alien[c].y += 1;

      if (alien[c].y > SCREEN_H+30) {
         if (!alien[c].shot) {
            position_alien(c);
         }
      }
      else {
         /* alien collided with player? */
         if ((ABS(alien[c].x - (pos>>SPEED_SHIFT)) < 48) && (ABS(alien[c].y - (SCREEN_H-42)) < 32)) {
            if (!shot) {
               ship_state = EXPLODE_FLAG;
               shot = TRUE;
               dead = TRUE; /* don't wait for the explosion */
               stop_sample(data[ENGINE_SPL].dat);
               clear_screen();
               collide_file(alien[c].filename, alien[c].attribute);
            }
            /*
            if (!alien[c].shot) {
               alien[c].shot = TRUE;
               alien[c].state = EXPLODE_FLAG;
            }
            */
         }
      }
   }

   if (skip_count <= 0) {
      skip_count = skip_speed;
   }
   else
      skip_count--;

   /* yes, I know that isn't a very pretty piece of code :-) */
}



/* adds a new screen area to the dirty rectangle list */
void add_to_list(DIRTY_LIST *list, int x, int y, int w, int h)
{
   list->rect[list->count].x = x;
   list->rect[list->count].y = y;
   list->rect[list->count].w = w;
   list->rect[list->count].h = h;
   list->count++; 
}



/* qsort() callback for sorting the dirty rectangle list */
int rect_cmp(const void *p1, const void *p2)
{
   DIRTY_RECT *r1 = (DIRTY_RECT *)p1;
   DIRTY_RECT *r2 = (DIRTY_RECT *)p2;

   return (r1->y - r2->y);
}



/* clears the screen */
void clear_screen()
{
   if (animation_type == DOUBLE_BUFFER) {
     clear_to_color(s, background_color());
     clear_to_color(screen, background_color());
   }
   else if ((animation_type == PAGE_FLIP) || (animation_type == RETRACE_FLIP)) {
     clear_to_color(page1, background_color());
     clear_to_color(page2, background_color());
     show_video_bitmap(page1);
   }
   else if (animation_type == TRIPLE_BUFFER) {
     clear_to_color(page1, background_color());
     clear_to_color(page2, background_color());
     clear_to_color(page3, background_color());
     show_video_bitmap(page1);
   }
   else {
     clear_to_color(s, background_color());
     clear_to_color(screen, background_color());
     add_to_list(&dirty, 0,0,SCREEN_W,SCREEN_H);
   }
}

/* main screen update function */
void draw_screen(void)
{
   int c;
   int x, y;
   BITMAP *bmp;
   RLE_SPRITE *spr;
   char *animation_type_str;

   if (animation_type == DOUBLE_BUFFER) {
      /* for double buffering, draw onto the memory bitmap. The first step 
       * is to clear it...
       */
      animation_type_str = "double buffered";
      bmp = s;
      clear_to_color(bmp, background_color());
   }
   else if ((animation_type == PAGE_FLIP) || (animation_type == RETRACE_FLIP)) {
      /* for page flipping we draw onto one of the sub-bitmaps which
       * describe different parts of the large virtual screen.
       */ 
      if (animation_type == PAGE_FLIP) 
         animation_type_str = "page flipping";
      else
         animation_type_str = "synced flipping";

      if (current_page == 0) {
         bmp = page2;
         current_page = 1;
      }
      else {
         bmp = page1;
         current_page = 0;
      }
      clear_to_color(bmp, background_color());
   }
   else if (animation_type == TRIPLE_BUFFER) {
      /* triple buffering works kind of like page flipping, but with three
       * pages (obvious, really :-) The benefit of this is that we can be
       * drawing onto page a, while displaying page b and waiting for the
       * retrace that will flip across to page c, hence there is no need
       * to waste time sitting around waiting for retraces.
       */ 
      animation_type_str = "triple buffered";

      if (current_page == 0) {
         bmp = page2; 
         current_page = 1;
      }
      else if (current_page == 1) {
         bmp = page3; 
         current_page = 2; 
      }
      else {
         bmp = page1; 
         current_page = 0;
      }
      clear_to_color(bmp, background_color());
   }
   else {
      /* for dirty rectangle animation we draw onto the memory bitmap, but 
       * we can use information saved during the last draw operation to
       * only clear the areas that have something on them.
       */
      int color = background_color();
      animation_type_str = "dirty rectangles";
      bmp = s;

      for (c=0; c<dirty.count; c++) {
         if ((dirty.rect[c].w == 1) && (dirty.rect[c].h == 1)) {
            putpixel(bmp, dirty.rect[c].x, dirty.rect[c].y, color);
         }
         else {
            rectfill(bmp, dirty.rect[c].x, dirty.rect[c].y, 
                        dirty.rect[c].x + dirty.rect[c].w, 
                        dirty.rect[c].y+dirty.rect[c].h, color);
         }
      }

      old_dirty = dirty;
      dirty.count = 0;
   }

   acquire_bitmap(bmp);

   /* draw the stars */
   for (c=0; c<MAX_STARS; c++) {
      y = star[c].oy;
      x = ((star[c].ox-SCREEN_W/2)*(y/(4-star[c].z/2)+SCREEN_H)/SCREEN_H)+SCREEN_W/2;

      {
        int i = 16*(15-(int)star[c].z) + 15;
        putpixel(bmp, x, y, makecol(i, i, i));
      }

      if (animation_type == DIRTY_RECTANGLE)
         add_to_list(&dirty, x, y, 1, 1);
   }

   /* draw the laser beam */
   if (laser_on > 0) {
     int x = (pos>>SPEED_SHIFT) - 1; /* player pos */
     int red = makecol(255, 0, 0);
     int dark_red = makecol(128, 0, 0);
     int waves;
     const int MAX_WAVES = 3;
     vline(bmp, x, SCREEN_H-60, laser_hit_y, red);
     for (waves=0; waves<MAX_WAVES; ++waves) {
       int dx = 3*(waves+1);
       int y = SCREEN_H-60;
       int bx = x;
       for (;;) {
         int new_y = y-2-rand()%4;
         int new_bx = x+rand()%dx-dx/2;
         line(bmp, bx, y, new_bx, new_y, dark_red);
         y = new_y;
         bx = new_bx;
         if (new_y<laser_hit_y) break;
       }
     }
     if (animation_type == DIRTY_RECTANGLE)
        add_to_list(&dirty, x-MAX_WAVES*3/2, 0, MAX_WAVES*3, SCREEN_H);
   }

   /* draw the player */
   x = pos>>SPEED_SHIFT;

#if 0
   if ((ship_burn) && (ship_state < EXPLODE_FLAG)) {
      spr = (RLE_SPRITE *)data[ENGINE1+(retrace_count/4)%7].dat;
      draw_rle_sprite(bmp, spr, x-spr->w/2, SCREEN_H-24); 

      if (animation_type == DIRTY_RECTANGLE)
         add_to_list(&dirty, x-spr->w/2, SCREEN_H-24, spr->w, spr->h);
   }
#endif

   if (ship_state >= EXPLODE_FLAG)
      spr = (RLE_SPRITE *)data[SHIP3].dat;
   else
      spr = (RLE_SPRITE *)data[ship_state].dat;

   set_alpha_blender();
   draw_trans_rle_sprite(bmp, spr, x-spr->w/2, SCREEN_H-42-spr->h/2);

   if (animation_type == DIRTY_RECTANGLE)
      add_to_list(&dirty, x-spr->w/2, SCREEN_H-42-spr->h/2, spr->w, spr->h);

   /* draw the aliens */
   for (c=0; c<alien_count; c++) {
      x = alien[c].x;
      y = alien[c].y;

      spr = NULL;
      if (alien[c].state >= EXPLODE_FLAG) {
         spr = explosion[alien[c].state-EXPLODE_FLAG];

         draw_rle_sprite(bmp, spr, x-spr->w/2, y-spr->h/2);

         if (animation_type == DIRTY_RECTANGLE)
            add_to_list(&dirty, x-spr->h/2, y-spr->h/2, spr->w, spr->h);
      }
      else {
         BITMAP *spr = alien[c].bmp;

         set_alpha_blender();
         draw_trans_sprite(bmp, spr, x-spr->w/2, y-spr->h/2);
         display_filename(
           bmp,font,x,y,alien[c].text_color,thread_time,alien[c].filename
         );

         if (animation_type == DIRTY_RECTANGLE)
            add_to_list(&dirty, x-spr->h/2, y-spr->h/2, spr->w, spr->h);
      }
   }

   /* draw the bullet */
   if (bullet_type) {
      x = bullet_x;
      y = bullet_y;

      spr = (RLE_SPRITE *)data[ROCKET].dat;
      draw_rle_sprite(bmp, spr, x-spr->w/2, y-spr->h/2);

      if (animation_type == DIRTY_RECTANGLE)
         add_to_list(&dirty, x-spr->w/2, y-spr->h/2, spr->w, spr->h);
   }

   /* draw the score and fps information */
   sprintf(
     score_buf, "%s - last command status: %s - weapon: %s",
     current_path, last_error, get_weapon_executable(selected_bullet_type)
   );

   textout(bmp, font, score_buf, 0, 0, makecol(255,255,255));

   if (animation_type == DIRTY_RECTANGLE)
      add_to_list(&dirty, 0, 0, text_length(font, score_buf), 8);

   release_bitmap(bmp);

   if (animation_type == DOUBLE_BUFFER) {
      /* when double buffering, just copy the memory bitmap to the screen */
      blit(s, screen, 0, 0, 0, 0, SCREEN_W, SCREEN_H);
   }
   else if ((animation_type == PAGE_FLIP) || (animation_type == RETRACE_FLIP)) {
      /* for page flipping we scroll to display the image */ 
      show_video_bitmap(bmp);
   }
   else if (animation_type == TRIPLE_BUFFER) {
      /* make sure the last flip request has actually happened */
      do {
      } while (poll_scroll());

      /* post a request to display the page we just drew */
      request_video_bitmap(bmp);
   }
   else {
      /* for dirty rectangle animation, only copy the areas that changed */
      for (c=0; c<dirty.count; c++)
         add_to_list(&old_dirty, dirty.rect[c].x, dirty.rect[c].y, dirty.rect[c].w, dirty.rect[c].h);

      /* sorting the objects really cuts down on bank switching */
      if (!gfx_driver->linear)
         qsort(old_dirty.rect, old_dirty.count, sizeof(DIRTY_RECT), rect_cmp);

      acquire_screen();

      for (c=0; c<old_dirty.count; c++) {
         if ((old_dirty.rect[c].w == 1) && (old_dirty.rect[c].h == 1)) {
            putpixel(screen, old_dirty.rect[c].x, old_dirty.rect[c].y, 
                     getpixel(bmp, old_dirty.rect[c].x, old_dirty.rect[c].y));
         }
         else {
            blit(bmp, screen, old_dirty.rect[c].x, old_dirty.rect[c].y, 
                            old_dirty.rect[c].x, old_dirty.rect[c].y, 
                            old_dirty.rect[c].w, old_dirty.rect[c].h);
         }
      }

      release_screen();
   }
}



/* timer callback for controlling the speed of the scrolling text */
volatile int scroll_count;

void scroll_counter(void)
{
   scroll_count++;
}

END_OF_FUNCTION(scroll_counter);



/* sets up and plays the game */
int play_game(void)
{
   int c;
   int esc = FALSE;

   stop_midi();

   /* set up a load of globals */
   dead = shot = FALSE;
   pos = (SCREEN_W/2)<<SPEED_SHIFT;
   xspeed = yspeed = ycounter = 0;
   ship_state = SHIP3;
   ship_burn = FALSE;
   frame_count = fps = 0;
   bullet_type = 0;
   score = 0;
   laser_on = 0;
   laser_hit_y = 0;
   laser_hit_alien = -1;
   laser_hit_same_alien = 0;
   old_dirty.count = dirty.count = 0;

   skip_count = 0;
   if (SCREEN_W < 400)
      skip_speed = 0;
   else if (SCREEN_W < 700)
      skip_speed = 1;
   else if (SCREEN_W < 1000)
      skip_speed = 2;
   else
      skip_speed = 3;

   for (c=0; c<MAX_STARS; c++) {
      star[c].ox = rand() % SCREEN_W;
      star[c].oy = rand() % SCREEN_H;
      star[c].z = rand() & 7;
   }

   generate_aliens();
   new_alien_count = 0;
   thread_time=0;

   play_midi(data[GAME_MUSIC].dat, TRUE);

   text_mode(0);
   clear_to_color(screen, background_color());
   clear_to_color(s, background_color());
   draw_screen();

#ifndef DONT_WAIT
   if ((!keypressed()) && (!joy[0].button[0].b) && (!joy[0].button[1].b)) {
      do {
      } while (midi_pos < 17);
   }
#endif

   clear_keybuf();

   set_palette(data[GAME_PAL].dat);
   position_mouse(SCREEN_W/2, SCREEN_H/2);

   /* set up the interrupt routines... */
   install_int(fps_proc, 1000);

   if (use_retrace_proc)
      retrace_proc = game_timer;
   else
      install_int(game_timer, 6400/SCREEN_W);

   game_time = 0;

   /* main game loop */
   while (!dead) {
      poll_keyboard();
      poll_joystick();

      while (game_time > 0) {
         move_everyone();
         game_time--;
         ++thread_time;
      }

      if (!dead) draw_screen();
      frame_count++;

      if (key[KEY_ESC])
         esc = dead = TRUE;
   }

   /* cleanup */
   remove_int(fps_proc);

   if (use_retrace_proc)
      retrace_proc = NULL;
   else
      remove_int(game_timer);

   stop_sample(data[ENGINE_SPL].dat);

   if (esc) {
      while (current_page != 0)
         if (!dead) draw_screen();

      show_video_bitmap(screen);

      do {
         poll_keyboard();
      } while (key[KEY_ESC]);

      fade_out(5);
      return 0;
   }

   fps = 0;
   if (!dead) draw_screen();

   if ((animation_type == PAGE_FLIP) || (animation_type == RETRACE_FLIP) || (animation_type == TRIPLE_BUFFER)) {
      while (current_page != 0)
         if (!dead) draw_screen();

      show_video_bitmap(screen);

      blit(screen, s, 0, 0, 0, 0, SCREEN_W, SCREEN_H);
   }

   fade_out(5);

   return 1;
}



/* the explosion graphics are pregenerated, using a simple particle system */
void generate_explosions(void)
{
   BITMAP *bmp = create_bitmap(EXPLODE_SIZE, EXPLODE_SIZE);
   unsigned char *p;
   int c, c2;
   int x, y;
   int xx, yy;
   int color;

   #define HOTSPOTS  64

   struct HOTSPOT
   {
      int x, y;
      int xc, yc;
   } hot[HOTSPOTS];

   for (c=0; c<HOTSPOTS; c++) {
      hot[c].x = hot[c].y = (EXPLODE_SIZE/2)<<16;
      hot[c].xc = (rand()&0xFFFF)-0x7FFF;
      hot[c].yc = (rand()&0xFFFF)-0x7FFF;
   }

   for (c=0; c<EXPLODE_FRAMES; c++) {
      clear_to_color(bmp, background_color());

      color = ((c<16) ? c*4 : (80-c)) >> 2;

      for (c2=0; c2<HOTSPOTS; c2++) {
         for (x=-6; x<=6; x++) {
            for (y=-6; y<=6; y++) {
               xx = (hot[c2].x>>16) + x;
               yy = (hot[c2].y>>16) + y;
               if ((xx>0) && (yy>0) && (xx<EXPLODE_SIZE) && (yy<EXPLODE_SIZE)) {
                  p = bmp->line[yy] + xx;
                  *p += (color >> ((ABS(x)+ABS(y))/3));
                  if (*p > 63)
                     *p = 63;
               }
            }
         }
         hot[c2].x += hot[c2].xc;
         hot[c2].y += hot[c2].yc;
      }

      for (x=0; x<EXPLODE_SIZE; x++) {
         for (y=0; y<EXPLODE_SIZE; y++) {
            c2 = bmp->line[y][x];
            if (c2 < 8)
               bmp->line[y][x] = 0;
            else
               bmp->line[y][x] = 16+c2/4;
         }
      }

      explosion[c] = get_rle_sprite(bmp);
   }

   destroy_bitmap(bmp);
}


void generate_sounds()
{
   float osc1, osc2, freq1, freq2, vol;
   char *p;
   int i;

   /* zap (firing sound) consists of multiple falling saw waves */
   zap = create_sample(8, FALSE, 22050, 8192);

   p = (char *)zap->data;

   osc1 = 0;
   freq1 = 0.02;

   osc2 = 0;
   freq2 = 0.025;

   for (i=0; i<zap->len; i++) {
      vol = (float)(zap->len - i) / (float)zap->len * 127;

      *p = 128 + (fmod(osc1, 1) + fmod(osc2, 1) - 1) * vol;

      osc1 += freq1;
      freq1 -= 0.000001;

      osc2 += freq2;
      freq2 -= 0.00000125;

      p++;
   }
}


/* modifies the palette to give us nice colors for the GUI dialogs */
void set_gui_colors(void)
{
   static RGB black = { 0,  0,  0,  0 };
   static RGB grey  = { 48, 48, 48, 0 };
   static RGB white = { 63, 63, 63, 0 };

   set_color(0, &black);
   set_color(16, &black);
   set_color(1, &grey); 
   set_color(255, &white); 

   gui_fg_color = palette_color[0];
   gui_bg_color = palette_color[1];
}



/* BEGIN SPEEDHACK CODE REUSE MARKER */

/* d_list_proc() callback for the animation mode dialog */
char *anim_list_getter(int index, int *list_size)
{
   static char *s[] =
   {
      "Double buffered",
      "Page flipping",
      "Synced flips",
      "Triple buffered",
      "Dirty rectangles"
   };

   if (index < 0) {
      *list_size = 5;
      return NULL;
   }

   return s[index];
}



/* dialog procedure for the animation type dialog */
extern DIALOG anim_type_dlg[];


int anim_list_proc(int msg, DIALOG *d, int c)
{
   int sel, ret;

   sel = d->d1;

   ret = d_list_proc(msg, d, c);

   if (sel != d->d1)
      ret |= D_REDRAW;

   return ret;
}



/* dialog procedure for the animation type dialog */
int anim_desc_proc(int msg, DIALOG *d, int c)
{
   static char *double_buffer_desc[] = 
   {
      "Draws onto a memory bitmap,",
      "and then uses a brute-force",
      "blit to copy the entire",
      "image across to the screen.",
      NULL
   };

   static char *page_flip_desc[] = 
   {
      "Uses two pages of video",
      "memory, and flips back and",
      "forth between them. It will",
      "only work if there is enough",
      "video memory to set up dual",
      "pages.",
      NULL
   };

   static char *retrace_flip_desc[] = 
   {
      "This is basically the same",
      "as page flipping, but it uses",
      "the vertical retrace interrupt",
      "simulator instead of retrace",
      "polling. Only works in mode-X,",
      "and not under win95.",
      NULL
   };

   static char *triple_buffer_desc[] = 
   {
      "Uses three pages of video",
      "memory, to avoid wasting time",
      "waiting for retraces. Only",
      "some drivers and hardware",
      "support this, eg. VBE 3.0 or",
      "mode-X with retrace sync.",
      NULL
   };

   static char *dirty_rectangle_desc[] = 
   {
      "This is similar to double",
      "buffering, but stores a list",
      "of which parts of the screen",
      "have changed, to minimise the",
      "amount of drawing that needs",
      "to be done.",
      NULL
   };

   static char **descs[] =
   {
      double_buffer_desc,
      page_flip_desc,
      retrace_flip_desc,
      triple_buffer_desc,
      dirty_rectangle_desc
   };

   char **p;
   int y;

   if (msg == MSG_DRAW) {
      rectfill(screen, d->x, d->y, d->x+d->w, d->y+d->h, d->bg);
      text_mode(d->bg);

      p = descs[anim_type_dlg[2].d1];
      y = d->y;

      while (*p) {
         textout(screen, font, *p, d->x, y, d->fg);
         y += 8;
         p++;
      } 
   }

   return D_O_K;
}



DIALOG anim_type_dlg[] =
{
   /* (dialog proc)     (x)   (y)   (w)   (h)   (fg)  (bg)  (key) (flags)     (d1)  (d2)  (dp)                 (dp2) (dp3) */
   { d_shadow_box_proc, 0,    0,    281,  151,  0,    1,    0,    0,          0,    0,    NULL,                NULL, NULL  },
   { d_ctext_proc,      140,  8,    1,    1,    0,    1,    0,    0,          0,    0,    "Animation Method",  NULL, NULL  },
   { anim_list_proc,    16,   28,   153,  44,   0,    1,    0,    D_EXIT,     4,    0,    anim_list_getter,    NULL, NULL  },
   { anim_desc_proc,    16,   90,   248,  48,   0,    1,    0,    0,          0,    0,    0,                   NULL, NULL  },
   { d_button_proc,     184,  28,   80,   16,   0,    1,    13,   D_EXIT,     0,    0,    "OK",                NULL, NULL  },
   { d_button_proc,     184,  50,   80,   16,   0,    1,    27,   D_EXIT,     0,    0,    "Cancel",            NULL, NULL  },
   { d_yield_proc,      0,    0,    0,    0,    0,    0,    0,    0,          0,    0,    NULL,                NULL, NULL  },
   { NULL,              0,    0,    0,    0,    0,    0,    0,    0,          0,    0,    NULL,                NULL, NULL  }
};

/* END SPEEDHACK CODE REUSE MARKER */


/* allows the user to choose an animation type */
int pick_animation_type(int *type)
{
   int ret;

   centre_dialog(anim_type_dlg);

   clear_to_color(screen, background_color());

   /* we set up colors to match screen color depth */
   for (ret = 0; anim_type_dlg[ret].proc; ret++) {
      anim_type_dlg[ret].fg = palette_color[0];
      anim_type_dlg[ret].bg = palette_color[1];
   }

   ret = do_dialog(anim_type_dlg, 2);

   *type = anim_type_dlg[2].d1 + 1;

   return (ret == 5) ? -1 : ret;
}



/* for parsing readme.txt */
typedef struct TEXT_LIST
{
   char *text;
   struct TEXT_LIST *next;
} TEXT_LIST;


typedef struct README_SECTION
{
   TEXT_LIST *head;
   TEXT_LIST *tail;
   char *flat;
   char *desc;
} README_SECTION;



/* formats a list of TEXT_LIST structure into a single string */
char *format_text(TEXT_LIST *head, char *eol, char *gap)
{
   TEXT_LIST *l;
   int size = 0;
   char *s;

   l = head;
   while (l) {
      if (l->text[0])
         size += strlen(l->text) + strlen(eol);
      else
         size += strlen(gap) + strlen(eol);
      l = l->next;
   }

   s = malloc(size+1);
   s[0] = 0;

   l = head;
   while (l) {
      if (l->text[0])
         strcat(s, l->text);
      else
         strcat(s, gap);
      strcat(s, eol);
      l = l->next;
   }

   return s;
}



/* loads the scroller message from readme.txt */
void load_text(void)
{
   README_SECTION sect[] =
   {
      { NULL, NULL, NULL, "Introduction" },
      { NULL, NULL, NULL, "Features"     },
      { NULL, NULL, NULL, "Copyright"    },
      { NULL, NULL, NULL, "Contact info" }
   };

   #define SPLITTER  "                                                      "

   static char intro_msg[] =
     "A long time ago, at an installation far, far away... "
     SPLITTER
     "It is a time of intra-system war, as forces of the User Alliance "
     "struggle to break the iron grip of the evil Admin Empire. Now, "
     "striking from a hidden directory, they win their first victory. "
     SPLITTER
     "During the battle, User spies manage to snarf source of the Empire's "
     "ultimate weapon; the dreaded `rm-star', a privileged root program with "
     "the power to destroy an entire file system at a keystroke. "
      SPLITTER
     "Now, hotly pursued by the Empire's sinister audit trail, Princess LA36 "
     "races aboard her shellscript -- custodian of the stolen listings that "
     "can save her people and restore freedom and games to the network... "
      SPLITTER
     "Your controls: the arrow keys to move left and right, the up arrow to accelerate (the faster you go, the more score you get), and the space bar to fire. Change weapons with PageUp and PageDown. "
     SPLITTER
     "UnixWars can be found somewhere on the Internet, at ESR's site, "
     "probably somewhere at snark.thyrsus.com "
     SPLITTER;

   static char splitter[] = SPLITTER;
   static char marker[] = "--------";
   TEXT_LIST *l, *p;
   int i;

   title_size = strlen(intro_msg);

   for (i=0; i<(int)(sizeof(sect)/sizeof(sect[0])); i++) {
      if (sect[i].head) {
         sect[i].flat = format_text(sect[i].head, " ", splitter);
         title_size += strlen(sect[i].flat) + strlen(sect[i].desc) + 
                       strlen(splitter) + strlen(marker)*2 + 2;
      }
   }

   title_text = malloc(title_size+1);
   title_alloced = TRUE;

   strcpy(title_text, intro_msg);

   for (i=0; i<(int)(sizeof(sect)/sizeof(sect[0])); i++) {
      if (sect[i].flat) {
         strcat(title_text, marker);
         strcat(title_text, " ");
         strcat(title_text, sect[i].desc);
         strcat(title_text, " ");
         strcat(title_text, marker);
         strcat(title_text, splitter);
         strcat(title_text, sect[i].flat);
      }
   }

   for (i=0; i<(int)(sizeof(sect)/sizeof(sect[0])); i++) {
      l = sect[i].head;
      while (l) {
         free(l->text);
         p = l;
         l = l->next;
         free(p);
      }
      if (sect[i].flat)
         free(sect[i].flat);
   }
}




volatile int swscroller_time=0;
static void swscroller_timer()
{
  ++swscroller_time;
}
END_OF_FUNCTION(swscroller_timer);

int find_color(int x, int y)
{
  if (x < SCREEN_W/2-160) return 0;
  if (y < SCREEN_H/2-96) return 0;
  if (x >= SCREEN_W/2-160+320) return 0;
  if (y >= SCREEN_H/2-96+128) return 0;
  return getpixel(data[TITLE_BMP].dat, x-(SCREEN_W/2-160), y-(SCREEN_H/2-96));
}

/* displays the title screen */
int title_screen(void)
{
   static int color = 0;
   int c, c2, star_color;
   BITMAP *buffer;
   RGB rgb;

   /* for the starfield */
   fixed x, y;
   int ix, iy;
   int star_count = 0;
#ifdef ENABLE_TITLE_STAR_FIELD
   int star_count_count = 0;
#endif

   /* for the text scroller */
   char buf[2] = " ";
   int text_char = 0xFFFF;
   int text_width = 0;
   int text_pix = 0;
   int text_scroll = 0;
   BITMAP *text_bmp;
   SWSCROLLER *sws;

   play_midi(data[TITLE_MUSIC].dat, TRUE);

   /* Star wars scroller */
   if (!jumpstart) {
     LOCK_FUNCTION(swscroller_timer);
     LOCK_VARIABLE(swscroller_time);
     sws = create_swscroller();
     install_int(&swscroller_timer,20);
     swscroller_time = 0;
     while ((!keypressed()) && (!joy[0].button[0].b) && (!joy[0].button[1].b)) {
       poll_joystick();
       poll_keyboard();
       while (swscroller_time > 0) {
         if (!update_swscroller(sws)) goto end_scroller;
         --swscroller_time;
       }
     }
  end_scroller:
     remove_int(&swscroller_timer);
     destroy_swscroller(sws);
   }

   for (c=0; c<MAX_STARS; c++) {
      star[c].z = 0;
      star[c].ox = star[c].oy = -1;
   }

   for (c=0; c<8; c++)
      title_palette[c] = ((RGB *)data[TITLE_PAL].dat)[c];

   /* set up the colors differently each time we display the title screen */
   for (c=8; c<PAL_SIZE/2; c++) {
      rgb = ((RGB *)data[TITLE_PAL].dat)[c];
      switch (color) {
         case 0:
            rgb.b = rgb.r;
            rgb.r = 0;
            break;
         case 1:
            rgb.g = rgb.r;
            rgb.r = 0;
            break;
         case 3:
            rgb.g = rgb.r;
            break;
      }
      title_palette[c] = rgb;
   }

   for (c=PAL_SIZE/2; c<PAL_SIZE; c++)
      title_palette[c] = ((RGB *)data[TITLE_PAL].dat)[c];

   color++;
   if (color > 3)
      color = 0;

   clear_to_color(screen, makecol(0, 0, 0));
   set_palette(title_palette);

   blit(data[TITLE_BMP].dat, screen, 0, 0, SCREEN_W/2-160, SCREEN_H/2-96, 320, 128);

   buffer = create_bitmap(SCREEN_W, SCREEN_H);
   clear_to_color(buffer, makecol(0, 0, 0));

   text_bmp = create_bitmap(SCREEN_W, 24);
   clear_to_color(text_bmp, makecol(0, 0, 0));

   old_dirty.count = dirty.count = 0;

   clear_keybuf();

   scroll_count = 0;

   if (use_retrace_proc)
      retrace_proc = scroll_counter;
   else
      install_int(scroll_counter, 6);

   do {
      do {
      } while (scroll_count < 0);

      while (scroll_count > 0) {
         /* animate the starfield */
         for (c=0; c<star_count; c++) {
            if (star[c].z <= itofix(1)) {
               x = itofix(rand()&0xff);
               y = itofix(((rand()&3)+1)*SCREEN_W);

               star[c].x = fixmul(fixcos(x), y);
               star[c].y = fixmul(fixsin(x), y);
               star[c].z = itofix((rand() & 0x1f) + 0x20);
            }

            x = fixdiv(star[c].x, star[c].z);
            y = fixdiv(star[c].y, star[c].z);

            ix = (int)(x>>16) + SCREEN_W/2;
            iy = (int)(y>>16) + SCREEN_H/2;

            if ((ix >= 0) && (ix < SCREEN_W) && (iy >= 0) && (iy <= SCREEN_H)) {
               star[c].ox = ix;
               star[c].oy = iy;
               star[c].z -= 4096;
            }
            else {
               star[c].ox = -1;
               star[c].oy = -1;
               star[c].z = 0;
            }
         }

         /* wake up new stars */
#ifdef ENABLE_TITLE_STAR_FIELD
         if (star_count < MAX_STARS) {
            if (star_count_count++ >= 32) {
               star_count_count = 0;
               star_count++;
            }
         }
#endif

         /* move the scroller */
         text_scroll += (use_retrace_proc ? 2 : 1);

         scroll_count--;
      }

      /* clear dirty rectangle list */
      for (c=0; c<dirty.count; c++) {
         if ((dirty.rect[c].w == 1) && (dirty.rect[c].h == 1)) {
            putpixel(buffer, dirty.rect[c].x, dirty.rect[c].y, 0);
         }
         else {
            rectfill(buffer, dirty.rect[c].x, dirty.rect[c].y, 
                             dirty.rect[c].x + dirty.rect[c].w, 
                             dirty.rect[c].y+dirty.rect[c].h, 0);
         }
      }

      old_dirty = dirty;
      dirty.count = 0;

      /* draw the text scroller */
      blit(text_bmp, text_bmp, text_scroll, 0, 0, 0, SCREEN_W, 24);
      rectfill(text_bmp, SCREEN_W-text_scroll, 0, SCREEN_W, 24, 0);

      while (text_scroll > 0) {
         text_pix += text_scroll;
         textout(text_bmp, data[TITLE_FONT].dat, buf, SCREEN_W-text_pix, 0, -1);
         if (text_pix >= text_width) {
            text_scroll = text_pix - text_width;
            text_char++;
            if (text_char >= title_size)
               text_char = 0;
            buf[0] = title_text[text_char];
            text_pix = 0;
            text_width = text_length(data[TITLE_FONT].dat, buf);
         }
         else
            text_scroll = 0;
      }

      blit(text_bmp, buffer, 0, 0, 0, SCREEN_H-24, SCREEN_W, 24);
      add_to_list(&dirty, 0, SCREEN_H-24, SCREEN_W, 24);

      /* draw the starfield */
      for (c=0; c<star_count; c++) {
         c2 = 7-(int)(star[c].z>>18);
         c2 *= 256/8;
         c2 = MID(0, c2, 255);
         star_color = makecol(c2, c2, c2);
         putpixel(buffer, star[c].ox, star[c].oy, star_color);
         add_to_list(&dirty, star[c].ox, star[c].oy, 1, 1);
      }

      /* draw the Allegro logo over the top */
      //draw_sprite(buffer, data[TITLE_BMP].dat, SCREEN_W/2-160, SCREEN_H/2-96);
      blit(data[TITLE_BMP].dat,buffer,0,0,SCREEN_W/2-160,SCREEN_H/2-96,320,128);

      /* dirty rectangle screen update */
      for (c=0; c<dirty.count; c++)
         add_to_list(&old_dirty, dirty.rect[c].x, dirty.rect[c].y, dirty.rect[c].w, dirty.rect[c].h);

      if (!gfx_driver->linear)
         qsort(old_dirty.rect, old_dirty.count, sizeof(DIRTY_RECT), rect_cmp);

      acquire_screen();

      for (c=0; c<old_dirty.count; c++) {
         if ((old_dirty.rect[c].w == 1) && (old_dirty.rect[c].h == 1)) {
            putpixel(screen, old_dirty.rect[c].x, old_dirty.rect[c].y, 
                     getpixel(buffer, old_dirty.rect[c].x, old_dirty.rect[c].y));
         }
         else {
            blit(buffer, screen, old_dirty.rect[c].x, old_dirty.rect[c].y, 
                                 old_dirty.rect[c].x, old_dirty.rect[c].y, 
                                 old_dirty.rect[c].w, old_dirty.rect[c].h);
         }
      }

      release_screen();

      poll_joystick();

   } while ((!keypressed()) && (!joy[0].button[0].b) && (!joy[0].button[1].b));

   if (use_retrace_proc)
      retrace_proc = NULL;
   else
      remove_int(scroll_counter);

   fade_out(5);

   destroy_bitmap(buffer);
   destroy_bitmap(text_bmp);

   while (keypressed())
      if ((readkey() & 0xff) == 27)
         return FALSE;

   return TRUE;
}



int main(int argc, char *argv[])
{
   int c, w, h, cdepth;
   static char buf[4096], buf2[4096];
   int conv;

   for (c=1; c<argc; c++) {
      if (stricmp(argv[c], "-cheat") == 0)
         cheat = TRUE;

      if (stricmp(argv[c], "-turbo") == 0)
         turbo = TRUE;

      if (stricmp(argv[c], "-jumpstart") == 0)
         jumpstart = TRUE;
   }

   allegro_init();
   install_keyboard();
   install_timer();
   install_mouse();
   font = get_font(FONT_SIZE_SMALL);
   init_actions();

   if (install_sound(DIGI_AUTODETECT, MIDI_AUTODETECT, argv[0]) != 0) {
      allegro_message("Error initialising sound\n%s\n", allegro_error);
      install_sound (DIGI_NONE, MIDI_NONE, NULL);
   }

   if (install_joystick(JOY_TYPE_AUTODETECT) != 0) {
      allegro_message("Error initialising joystick\n%s\n", allegro_error);
      install_joystick (JOY_TYPE_NONE);
   }

   #ifdef ALLEGRO_CONSOLE_OK
      if (!jumpstart)
         fade_out(4);
   #endif

   if (set_gfx_mode(GFX_SAFE, 320, 200, 0, 0) != 0) {
      set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);
      allegro_message("Unable to set any graphic mode\n%s\n", allegro_error);
      return 1;
   }

   load_text();

#if 0
   /* Can't play not loaded music */
   if (!jumpstart) {
      /* horrible hack to use MIDI music if we have a nice MIDI driver, or
       * play a sample otherwise, but still compile on platforms that don't
       * know about the MIDI drivers we are looking for. Urgh, I'm quite
       * ashamed to admit that I actually wrote this :-)
       */
      #ifdef ALLEGRO_WINDOWS

         play_midi(data[INTRO_MUSIC].dat, FALSE);

      #else

         if ((0) ||
            #ifdef MIDI_DIGMID
               (midi_driver->id == MIDI_DIGMID) ||
            #endif
            #ifdef MIDI_AWE32
               (midi_driver->id == MIDI_AWE32) ||
            #endif
            #ifdef MIDI_MPU
               (midi_driver->id == MIDI_MPU) ||
            #endif
            #ifdef MIDI_SB_OUT
               (midi_driver->id == MIDI_SB_OUT) ||
            #endif
            (0))
         play_midi(data[INTRO_MUSIC].dat, FALSE);

      #endif
   }
#endif

   c = retrace_count;

#ifndef DONT_WAIT
   if (!jumpstart) {
      do {
      } while (retrace_count-c < 120);

      fade_out(1);
   }
#endif

   clear_to_color(screen, background_color());
   set_gui_colors();
   set_mouse_sprite(NULL);

   while (1) {
     //if (!gfx_mode_select_ex(&c, &w, &h, &cdepth))
     if (!gfx_mode_select_ex(&graphic_mode, &w, &h, &cdepth))
        exit(1);
     if (cdepth==8) {
       alert("8 bit color not supported",NULL,NULL,"OK",NULL,0,0);
     }
     else break;
   };

   if (pick_animation_type(&animation_type) < 0)
      exit(1);

   set_color_depth(cdepth);
   //if (set_gfx_mode(c, w, h, 0, 0) != 0) {
   if (set_gfx_mode(graphic_mode, w, h, 0, 0) != 0) {
      set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);
      allegro_message("Error setting %dbpp graphics mode\n%s\n", cdepth, allegro_error);
      exit(1);
   }

   get_executable_name(buf, sizeof(buf));
   replace_filename(buf2, buf, "rmstar.dat", sizeof(buf2));
   conv = COLORCONV_TOTAL;
   conv &= ~(COLORCONV_32A_TO_8);
   conv &= ~(COLORCONV_32A_TO_15);
   conv &= ~(COLORCONV_32A_TO_16);
   conv &= ~(COLORCONV_32A_TO_24);
   set_color_conversion(conv);
   data = load_datafile(buf2);
   if (!data) {
      set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);
      allegro_message("Error loading %s\n", buf2);
      exit(1);
   }
   {
      static char tmp[4096];
      static char dest[4096];
      static char fname[4096];
      get_executable_name(tmp, sizeof(tmp));
      fix_filename_path(dest,tmp,sizeof(dest));
      replace_filename(fname, dest, "gfx/whitestar.tga", sizeof(fname));
      default_image = load_bitmap(fname, NULL);
      if (!default_image) {
         set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);
         allegro_message("Error loading %s\n", fname);
         exit(1);
      }
   }

   generate_explosions();
   generate_sounds();

   /* BEGIN SPEEDHACK CODE REUSE MARKER */

   switch (animation_type) {

      case PAGE_FLIP:
         /* set up page flipping bitmaps */
         page1 = create_video_bitmap(SCREEN_W, SCREEN_H);
         page2 = create_video_bitmap(SCREEN_W, SCREEN_H);

         if ((!page1) || (!page2)) {
            set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);
            allegro_message("Not enough video memory for page flipping\n");
            exit(1);
         }
         break;

      case RETRACE_FLIP: 
         /* set up retrace-synced page flipping bitmaps */
         if (!timer_can_simulate_retrace()) {
            set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);
            allegro_message("Retrace syncing is not possible in this environment\n");
            exit(1);
         }

         timer_simulate_retrace(TRUE);

         page1 = create_video_bitmap(SCREEN_W, SCREEN_H);
         page2 = create_video_bitmap(SCREEN_W, SCREEN_H);

         if ((!page1) || (!page2)) {
            set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);
            allegro_message("Not enough video memory for page flipping\n");
            exit(1);
         }
         break;

      case TRIPLE_BUFFER: 
         /* set up triple buffered bitmaps */
         if (!(gfx_capabilities & GFX_CAN_TRIPLE_BUFFER))
            enable_triple_buffer();

         if (!(gfx_capabilities & GFX_CAN_TRIPLE_BUFFER)) {
            strcpy(buf, gfx_driver->name);
            set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);

            #ifdef ALLEGRO_DOS
               allegro_message("The %s driver does not support triple buffering\n\nTry using mode-X in clean DOS mode (not under Windows)\n", buf);
            #else
               allegro_message("The %s driver does not support triple buffering\n", buf);
            #endif

            exit(1);
         }

         page1 = create_video_bitmap(SCREEN_W, SCREEN_H);
         page2 = create_video_bitmap(SCREEN_W, SCREEN_H);
         page3 = create_video_bitmap(SCREEN_W, SCREEN_H);

         if ((!page1) || (!page2) || (!page3)) {
            set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);
            allegro_message("Not enough video memory for triple buffering\n");
            exit(1);
         }
         break;
   }

   /* END SPEEDHACK CODE REUSE MARKER */

   use_retrace_proc = timer_is_using_retrace();

   LOCK_VARIABLE(game_time);
   LOCK_FUNCTION(game_timer);

   LOCK_VARIABLE(scroll_count);
   LOCK_FUNCTION(scroll_counter);

   LOCK_VARIABLE(score);
   LOCK_VARIABLE(frame_count);
   LOCK_VARIABLE(fps);
   LOCK_VARIABLE(allow_cycling_weapons);
   LOCK_FUNCTION(fps_proc);

   s = create_bitmap(SCREEN_W, SCREEN_H);

   text_mode(0);
   clear_to_color(screen, background_color());

   title_screen();

   while (play_game());

   destroy_bitmap(s);

   if ((animation_type == PAGE_FLIP) || (animation_type == RETRACE_FLIP)) {
      destroy_bitmap(page1);
      destroy_bitmap(page2);
   }
   else if (animation_type == TRIPLE_BUFFER) {
      destroy_bitmap(page1);
      destroy_bitmap(page2);
      destroy_bitmap(page3);
   }

   for (c=0; c<EXPLODE_FRAMES; c++)
      destroy_rle_sprite(explosion[c]);

   stop_midi();

   set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);

   if (end_text) {
      allegro_message("%shttp://alleg.sourceforge.net/\n\n", end_text);
      free(end_text);
   }

   if ((title_text) && (title_alloced))
      free(title_text);

   unload_datafile(data);

   allegro_exit();

   return 0;
}

END_OF_MAIN();


void add_alien(const char *filename,int attribute,int x,int y)
{
  if (alien_count>=MAX_ALIENS) return;

  alien[alien_count].x=x;
  alien[alien_count].y=y;
  alien[alien_count].d=0;
  alien[alien_count].state=-1;
  alien[alien_count].shot=FALSE;
  alien[alien_count].filename=strdup(filename);
  alien[alien_count].attribute=attribute;
  {
    int r=255, g=255, b=255;
    const char *image = get_file_image(filename, attribute);
    if (image) {
      get_file_color(filename, attribute, &r, &g, &b);
      alien[alien_count].bmp = get_bitmap(image, r, g, b);
      if (!alien[alien_count].bmp) alien[alien_count].bmp = default_image;
    }
    else {
      alien[alien_count].bmp = default_image;
    }
    r = 255-r;
    g = 255-g;
    b = 255-b;
    get_file_textcolor(filename, attribute, &r, &g, &b);
    alien[alien_count].text_color = makecol(r, g, b);
  }
  {
    static char name[80];
    const char *thread_code;
    THREAD *thread=NULL;
    sprintf(name,"code%d",1+rand()%4);
    thread_code=get_config_string("rmstar_scripts",name,0);
    if (thread_code) thread=parse(filename,thread_code);
    if (thread) {
      alien[alien_count].thread=thread;
    }
    else {
      alien[alien_count].thread=create_thread(filename);
    }
  }
  ++alien_count;
}

static void dogen(const char *filename,int attribute,int param)
{
  add_alien(get_filename(filename), attribute, 0, 0);
}

void clear_aliens()
{
  int n;
  for (n=0;n<alien_count;++n) {
    destroy_thread(alien[n].thread);
    free(alien[n].filename);
  }
  alien_count=0;
}

void position_alien(int c)
{
  int h = compute_height();
  alien[c].x = rand()%SCREEN_W;
  alien[c].y = -rand() %h;
  alien[c].d = (rand()&1) ? 1 : -1;
}

void position_aliens()
{
  int n;
  for (n=0;n<alien_count;++n) {
    position_alien(n);
  }
}

void generate_aliens()
{
#if (DEVICE_SEPARATOR != 0) && (DEVICE_SEPARATOR != '\0')
  int drive = _al_getdrive();
#else
  int drive = 0;
#endif
  static char pattern[4096];

  clear_aliens();

  _al_getdcwd(drive, current_path, sizeof(current_path) - ucwidth(OTHER_PATH_SEPARATOR));
  fix_filename_case(current_path);
  fix_filename_slashes(current_path);
  put_backslash(current_path);

  strcpy(pattern,current_path);
#if defined ALLEGRO_DOS || defined ALLEGRO_WINDOWS
  strcat(pattern,"*.*");
#else
  strcat(pattern,"*");
#endif
  for_each_file(pattern,INT_MAX,&dogen,0);

  position_aliens();
}

int compute_height()
{
  return 1 + alien_count*256;
}

void cycle_weapon(int delta)
{
  const char *exe;

  if (!allow_cycling_weapons) return;

  selected_bullet_type += delta;
  exe = get_weapon_executable(selected_bullet_type);
  if (!exe) {
    if (delta == 1) {
      selected_bullet_type = 1;
    }
    else {
      /* it's possible to define weapons while editing the cfg ! :) */
      int n;
      for (n=1;get_weapon_executable(n);++n);
      selected_bullet_type = n-1;
    }
  }
  allow_cycling_weapons = 0;
}

void display_filename(
  BITMAP *dest,FONT *font,int x,int y,int color,int time,const char *text
)
{
   const int MAX_WIDTH=48;
   const float SINE_FACTOR=0.05f;
   int full_length;
   int x1,x2,dx;
   float sine;
   const char *ptr;
   int cx;
   int alpha;
   int i,j;
   BITMAP *scratch;
   float length_sine_factor;

   full_length=text_length(font,text);
   dx=(full_length-(MAX_WIDTH-8))/2;
   if (dx<0) dx=0;
   x1=x-MAX_WIDTH/2;
   x2=x+MAX_WIDTH/2;
   if (dx<0) dx=0;
   length_sine_factor=32.0f/full_length;
   if (length_sine_factor>1.0f) length_sine_factor=1.0f;
   sine=(float)sin(time*SINE_FACTOR*length_sine_factor);

   cx=x+dx*sine;
   cx-=full_length/2;
   scratch=create_bitmap_ex(8,8,8);
   for (ptr=text;*ptr;++ptr) {
     char slice[2]=" ";
     *slice=*ptr;
     if (cx>=x1-8) {
       if (cx>=x2) break;
       clear_bitmap(scratch);
       textout(scratch,font,slice,0,0,1);
       for (j=0;j<scratch->h;++j) for (i=0;i<scratch->w;++i) {
         if (cx+i >= x1 && cx+i <= x2) {
           if (_getpixel(scratch,i,j)) {
             if (cx+i < x) {
               alpha=16*((cx+i)-x1);
             }
             else {
               alpha=16*(x2-(cx+i));
             }
             alpha=MID(0,alpha,255);
             set_trans_blender(128,128,128,alpha);
             drawing_mode(DRAW_MODE_TRANS,NULL,0,0);
             putpixel(dest,cx+i,y+j,color);
           }
         }
       }
     }
     cx+=text_length(font,slice);
   }
   destroy_bitmap(scratch);
   solid_mode();
}
