UNIT mingro;
(*       _    _
 *      | \  / | O  _ ___  _____  _ __  _____
 *      |  \/  || || `_  \/  _  \| `__|/  _  \
 *      | |\/| || || | | || |_| || |   | |_| |
 *      |_|  |_||_||_| |_||___  ||_|   \____/
 *			    __| |
 *			   \___/
 * 
 *      Defines the whole Mingro API and implementation.
 *
 *      By uo Martnez.
 *      Some parts of the IMPLEMENTATION section were translated from the
 *	Allegro Game Library sources by Shawn Hargreaves and others. 
 *
 *      See readme.txt for copyright information.
 *)

INTERFACE

{$PACKRECORDS C}

  USES
    mgBase,
    mgVTable; { Needed by the MG_BITMAP definition. }



  CONST
  { Identifies the library version. }
    MG_VERSION = 0.0; { This is a very early Work In Progress version }
    MG_MAJOR_V = 0;   { so "major" is 0.                              }
    MG_MINOR_V = 0;



(**********
 * System *
 **********)
  TYPE
  { To define procedures to be used as "callback". }
    MG_CALLBACK_SIMPLE_PROC = PROCEDURE; CDECL;

  VAR
    mgErrno: LONGINT; { Returns the error status and identification. }

  FUNCTION mgStartup: BOOLEAN; { Must be called at the beggining. }
  PROCEDURE mgShutdown;        { Must be called before the "END." keyword. }

(* mgSleep:
 *   Stops the program the given milliseconds. *)
  PROCEDURE mgSleep (MilliSeconds: INTEGER); CDECL;



(*********
 * Timer *
 *********)

{ Give the number of seconds between each tick. }
  FUNCTION mgSecsToTimer (x: LONGINT): LONGINT;
{ Give the number of milliseconds between each tick. }
  FUNCTION mgMsecToTimer (x: LONGINT): LONGINT;
{ Give the number of ticks each second. }
  FUNCTION mgBpsToTimer  (x: LONGINT): LONGINT;
{ Give the number of ticks per minute. }
  FUNCTION mgBpmToTimer  (x: LONGINT): LONGINT;

{ Installs the given interruption at given speed. }
  FUNCTION mgInstallInterruption (proc: MG_CALLBACK_SIMPLE_PROC; speed: LONGINT): LONGINT; CDECL;
{ Removes the given interruption. }
  PROCEDURE mgRemoveInterruption (proc: MG_CALLBACK_SIMPLE_PROC); CDECL;



(***********
 * Bitmaps *
 ***********)
  TYPE
    MG_BITMAPptr = ^MG_BITMAP;
    MG_BITMAP = RECORD
      w, h: LONGINT;            (* width and height in pixels *)
      clip: LONGINT;            (* flag if clipping is turned on *)
      cl, cr, ct, cb: LONGINT;  (* clip left, right, top and bottom values *)
{ The next conditional prevents the fpdoc utility to index all fields. }
{$IFNDEF DOCMODE}
      vtable: MG_GFX_VTABLEptr; (* drawing functions *)
      write_bank: POINTER;      (* C func on some machines, asm on i386 *)
      read_bank: POINTER;       (* C func on some machines, asm on i386 *)
      dat: POINTER;             (* the memory we allocated for the bitmap *)
      id: DWORD;                (* for identifying sub-bitmaps *)
      extra: POINTER;           (* points to a structure with more info *)
{$ENDIF}
      x_ofs: LONGINT;           (* horizontal offset (for sub-bitmaps) *)
      y_ofs: LONGINT;           (* vertical offset (for sub-bitmaps) *)
{ The next conditional prevents the fpdoc utility to index all fields. }
{$IFNDEF DOCMODE}
      seg: LONGINT;             (* bitmap segment *)
      line: POINTER;		(* ZERO_SIZE_ARRAY(unsigned char *, line); *)
{$ENDIF}
    END;

{ Creates bitmaps. }
  FUNCTION mgCreateBitmap (w, h: LONGINT): MG_BITMAPptr; CDECL;
  FUNCTION mgCreateBitmapEx (w, h, bpp: LONGINT): MG_BITMAPptr; CDECL;
  FUNCTION mgCreateSubBitmap (parent: MG_BITMAPptr; x, y, w, h: LONGINT): MG_BITMAPptr; CDECL;
{ Frees the bitmap resources.  }
  PROCEDURE mgDestroyBitmap (bmp: MG_BITMAPptr); CDECL;
{ Gets bitmap information. }
  FUNCTION mgBitmapColorDepth (bmp: MG_BITMAPptr): LONGINT;
  FUNCTION mgBitmapMaskColor (bmp: MG_BITMAPptr): LONGINT; { For sprites. }
  FUNCTION mgIsSameBitmap (bmp1, bmp2: MG_BITMAPptr): BOOLEAN;
  FUNCTION mgIsScreenBitmap (bmp: MG_BITMAPptr): BOOLEAN;
  FUNCTION mgIsSubBitmap (bmp: MG_BITMAPptr): BOOLEAN;



(******************
 * Graphics modes *
 ******************)
  VAR
  { To access to the video memory. }
    mgScreen: MG_BITMAPptr; EXTERNAL ALLEGRO_SHARED_LIBRARY_NAME NAME 'screen';
  { Screen size. }
    mgScreenW, mgScreenH: LONGINT;



{ Sets the graphics mode. }
  FUNCTION mgSetGfxMode (w, h, bpp: LONGINT): BOOLEAN;
  FUNCTION mgSetGfxModeEx (w, h, bpp: LONGINT; windowed: BOOLEAN): BOOLEAN;
{ "Unsets" the graphics mode. }
  PROCEDURE mgExitGfxMode;

{ Hardware control. }
  VAR
  { If the retrace simulator is installed, this count is incremented on each
    vertical retrace; otherwise, if the refresh rate is known, the count is
    incremented at the same rate (ignoring retraces); otherwise, it is
    incremented 70 times a second. This provides a way of controlling the speed
    of your program without installing user timer functions. }
    mgRetraceCount: LONGINT; EXTERNAL ALLEGRO_SHARED_LIBRARY_NAME NAME 'retrace_count';
  PROCEDURE mgWaitVerticalRetrace; CDECL;



(**********************
 * Color and palettes *
 **********************)

  CONST
    MG_COLOR_PALETTE_SIZE = 256;

  TYPE
  { RGB color descriptor. }
    MG_RGBptr = ^MG_RGB;
    MG_RGB = RECORD
      r : BYTE;
      g : BYTE;
      b : BYTE;
      filler : BYTE;
    END;
  { Color palette description for indexed modes (8bpp). }
    MG_COLOR_PALETTEptr = ^MG_COLOR_PALETTE;
    MG_COLOR_PALETTE = ARRAY [0..MG_COLOR_PALETTE_SIZE-1] OF MG_RGB;

{ Make a "color" from his RGB description. }
  FUNCTION mgColor (r, g, b: LONGINT): LONGINT; CDECL;
  FUNCTION mgColorDepth (color_depth, r, g, b: LONGINT): LONGINT; CDECL;
{ "Unmake" a color to his RGB description. }
  PROCEDURE mgRGB (c: LONGINT; VAR r, g, b: LONGINT);
  PROCEDURE mgRGBDepth (color_depth, c: LONGINT; VAR r, g, b: LONGINT);

{ Set the screen color palette. }
  PROCEDURE mgSetColor (index: LONGINT; p: MG_RGBptr); CDECL;
  PROCEDURE mgSetPalette (p: MG_COLOR_PALETTE); CDECL;
{ Get the screen color palette. }
  PROCEDURE mgGetColor (index: LONGINT; p: MG_RGBptr); CDECL;
  PROCEDURE mgGetPalette (p: MG_COLOR_PALETTE); CDECL;
{ Color effects. }
  PROCEDURE mgFadeFrom (source, dest: MG_COLOR_PALETTE; speed: LONGINT); CDECL;
  PROCEDURE mgFadeIn (p: MG_COLOR_PALETTE; speed: LONGINT); CDECL;
  PROCEDURE mgFadeOut (speed: LONGINT); CDECL;
{ Create palettes. }
(* mgGenerateOptimizedColorPalette:
 *    Generates a 256-color palette suitable for making a reduced color version
 *    of the specified truecolor image. *)
  TYPE
    MG_RSVDCOLSTYPE = ARRAY [0..255] OF BYTE;
  FUNCTION mgGenerateOptimizedColorPalette (image: MG_BITMAPptr;
					    pal: MG_COLOR_PALETTE;
					    rsvdcols: MG_RSVDCOLSTYPE
					   ): LONGINT; CDECL;
(* mgGenerate332ColorPalette:
 *   Constructs a fake truecolor palette, using three bits for red and green
 *   and two for the blue. *)
  PROCEDURE mgGenerate332ColorPalette (pal: MG_COLOR_PALETTE); CDECL;



(**********************
 * Drawing primitives *
 **********************)

{ Draw clipping. }
  PROCEDURE mgSetClip (bmp: MG_BITMAPptr; x1, y1, x2, y2: LONGINT); CDECL;
{ Using next procedures would accelerate the drawing in some cases. }
  PROCEDURE mgAcquireBitmap (bmp: MG_BITMAPptr);
  PROCEDURE mgReleaseBitmap (bmp: MG_BITMAPptr);
  PROCEDURE mgAcquireScreen;
  PROCEDURE mgReleaseScreen;
{ Clears the given bitmap. }
  PROCEDURE mgClearBitmap (bmp: MG_BITMAPptr);
  PROCEDURE mgClearBitmapToColor (bmp: MG_BITMAPptr; c: LONGINT);
{ Drawing. }
  FUNCTION  mgGetPixel    (bmp: MG_BITMAPptr; x, y: LONGINT): LONGINT;
  PROCEDURE mgPutPixel    (bmp: MG_BITMAPptr; x, y, color: LONGINT);
  PROCEDURE mgLine        (bmp: MG_BITMAPptr; x1, y1, x2, y2, color: LONGINT);
  PROCEDURE mgRect        (bmp: MG_BITMAPptr; x1, y1, x2, y2, color: LONGINT);
  PROCEDURE mgRectFill    (bmp: MG_BITMAPptr; x1, y1, x2, y2, color: LONGINT);
  PROCEDURE mgCircle      (bmp: MG_BITMAPptr; x, y, r, color: LONGINT);
  PROCEDURE mgCircleFill  (bmp: MG_BITMAPptr; x, y, r, color: LONGINT);
  PROCEDURE mgEllipse     (bmp: MG_BITMAPptr; x, y, rx, ry, color: LONGINT);
  PROCEDURE mgEllipseFill (bmp: MG_BITMAPptr; x, y, rx, ry, color: LONGINT);
  PROCEDURE mgFill	  (bmp: MG_BITMAPptr; x, y, color: LONGINT);



(*****************************
 * Loading and saving images *
 *****************************)

{ This function can load BMP, PCX, TGA and LBM files. }
  FUNCTION mgLoadBitmap (CONST FileName: STRING; VAR Palette: MG_COLOR_PALETTE): MG_BITMAPptr;
{ This function can save BMP, PCX and TGA files. }
  FUNCTION mgSaveBitmap (CONST FileName: STRING; Bmp: MG_BITMAPptr; Palette: MG_COLOR_PALETTE): BOOLEAN;



(*************************
 * Sprites and blitting. *
 *************************)
(* Copy a bitmap or a subbitmap. *)
  PROCEDURE mgBlit (Orig, Dest: MG_BITMAPptr; x, y: LONGINT);
  PROCEDURE mgBlitEx (Orig, Dest: MG_BITMAPptr; ox, oy, dx, dy, w, h: LONGINT);
(* Draw a sprite. *)
  PROCEDURE mgDrawSprite (bmp, sprite: MG_BITMAPptr; x, y: LONGINT);
  PROCEDURE mgDrawSpriteHflip (bmp, sprite: MG_BITMAPptr; x, y: LONGINT);
  PROCEDURE mgDrawSpriteVflip (bmp, sprite: MG_BITMAPptr; x, y: LONGINT);
  PROCEDURE mgDrawSpriteVHflip (bmp, sprite: MG_BITMAPptr; x, y: LONGINT);
(* The angle is between 0 and 255. *)
  PROCEDURE mgDrawRotatedSprite (bmp, sprite: MG_BITMAPptr; x, y: LONGINT; angle: LONGINT);
  PROCEDURE mgDrawRotatedVflipSprite (bmp, sprite: MG_BITMAPptr; x, y: LONGINT; angle: LONGINT);



(****************
 * Text drawing *
 ****************)

  TYPE
    MG_TEXT_FONTptr = POINTER;

  VAR
    mgDefaultFont: MG_TEXT_FONTptr; EXTERNAL ALLEGRO_SHARED_LIBRARY_NAME NAME 'font';
    mg404Char: LONGINT; EXTERNAL ALLEGRO_SHARED_LIBRARY_NAME NAME 'allegro_404_char';

{ Length of text in pixels. }
  FUNCTION mgTextLength (const text: STRING; const font: MG_TEXT_FONTptr): LONGINT;
{ Height of the text in pixels. }
  FUNCTION mgTextHeight (const font: MG_TEXT_FONTptr): LONGINT; CDECL;
{ Text output. }
  PROCEDURE mgTextout (bmp: MG_BITMAPptr; font: MG_TEXT_FONTptr; x, y, c: LONGINT; text: STRING);



(************
 * Keyboard *
 ************)

CONST
{ Max number of keys handled. }
  MG_KEY_MAX             = 115;

VAR
  mgKey: ARRAY [0..MG_KEY_MAX-1] OF CHAR; EXTERNAL ALLEGRO_SHARED_LIBRARY_NAME NAME 'key';
  mgKeyShifts: LONGINT; EXTERNAL ALLEGRO_SHARED_LIBRARY_NAME NAME 'key_shifts';
  mgThreeFingerFlag: INTEGER; EXTERNAL ALLEGRO_SHARED_LIBRARY_NAME NAME 'three_finger_flag';
  mgKeyLedFlag: INTEGER; EXTERNAL ALLEGRO_SHARED_LIBRARY_NAME NAME 'key_led_flag';

{ Waits for a keyboard input. }
  FUNCTION mgReadKey: LONGINT; CDECL;
{ UNICODE version of mgReadKey. }
  FUNCTION mgUReadKey (scancode: PLONGINT): LONGINT; CDECL; 
{ Retruns TRUE if a key was pressed. }
  FUNCTION mgKeyPressed: BOOLEAN; CDECL;
{ Clears the keyboard input fuffer }
  PROCEDURE mgClearKeyBuf; CDECL;
{ Updates the keyboard state variables.
  Must be used to update the mgKey array. }
  PROCEDURE mgPollKeyboard; CDECL;
{ Simulate keyboard use. }
  PROCEDURE mgSimulateKeyPress (keycode: LONGINT); CDECL;
{ UNICODE version of mgSimulateKeyPress. }
  PROCEDURE mgSimulateUKeyPress (keycode, scancode: LONGINT); CDECL;
{ Set keyboard repeat rate. }
  PROCEDURE mgSetKeyboardRate (delay, _repeat: LONGINT); CDECL;
{ }
  FUNCTION mgScanCodeToASCII (scancode: LONGINT): LONGINT; CDECL;


{ The next conditional prevents the fpdoc utility to index all keyboard
  constants. }
{$IFNDEF DOCMODE}
CONST
{ Special keys flags. }
  MG_KB_SHIFT_FLAG	 = $0001;
  MG_KB_CTRL_FLAG	 = $0002;
  MG_KB_ALT_FLAG         = $0004;
  MG_KB_LWIN_FLAG        = $0008;
  MG_KB_RWIN_FLAG        = $0010;
  MG_KB_MENU_FLAG        = $0020;
  MG_KB_SCROLOCK_FLAG    = $0100;
  MG_KB_NUMLOCK_FLAG     = $0200;
  MG_KB_CAPSLOCK_FLAG    = $0400;
  MG_KB_INALTSEQ_FLAG    = $0800;
  MG_KB_ACCENT1_FLAG     = $1000;
  MG_KB_ACCENT2_FLAG     = $2000;
  MG_KB_ACCENT3_FLAG     = $4000;
  MG_KB_ACCENT4_FLAG     = $8000;
{ Key identifiers. }
  MG_KEY_A		= 1;
  MG_KEY_B		= 2;
  MG_KEY_C              = 3;
  MG_KEY_D              = 4;
  MG_KEY_E              = 5;
  MG_KEY_F              = 6;
  MG_KEY_G		= 7;
  MG_KEY_H		= 8;
  MG_KEY_I              = 9;
  MG_KEY_J              = 10;
  MG_KEY_K              = 11;
  MG_KEY_L              = 12;
  MG_KEY_M              = 13;
  MG_KEY_N              = 14;
  MG_KEY_O              = 15;
  MG_KEY_P              = 16;
  MG_KEY_Q              = 17;
  MG_KEY_R              = 18;
  MG_KEY_S              = 19;
  MG_KEY_T              = 20;
  MG_KEY_U              = 21;
  MG_KEY_V              = 22;
  MG_KEY_W              = 23;
  MG_KEY_X              = 24;
  MG_KEY_Y              = 25;
  MG_KEY_Z              = 26;
  MG_KEY_0	        = 27;
  MG_KEY_1	        = 28;
  MG_KEY_2		= 29;
  MG_KEY_3	        = 30;
  MG_KEY_4		= 31;
  MG_KEY_5		= 32;
  MG_KEY_6		= 33;
  MG_KEY_7		= 34;
  MG_KEY_8		= 35;
  MG_KEY_9		= 36;
  MG_KEY_0_PAD		= 37;
  MG_KEY_1_PAD		= 38;
  MG_KEY_2_PAD		= 39;
  MG_KEY_3_PAD		= 40;
  MG_KEY_4_PAD		= 41;
  MG_KEY_5_PAD		= 42;
  MG_KEY_6_PAD		= 43;
  MG_KEY_7_PAD  	= 44;
  MG_KEY_8_PAD  	= 45;
  MG_KEY_9_PAD		= 46;
  MG_KEY_F1		= 47;
  MG_KEY_F2		= 48;
  MG_KEY_F3		= 49;
  MG_KEY_F4		= 50;
  MG_KEY_F5		= 51;
  MG_KEY_F6		= 52;
  MG_KEY_F7		= 53;
  MG_KEY_F8		= 54;
  MG_KEY_F9		= 55;
  MG_KEY_F10		= 56;
  MG_KEY_F11		= 57;
  MG_KEY_F12		= 58;
  MG_KEY_ESC		= 59;
  MG_KEY_TILDE		= 60;
  MG_KEY_MINUS		= 61;
  MG_KEY_EQUALS		= 62;
  MG_KEY_BACKSPACE	= 63;
  MG_KEY_TAB		= 64;
  MG_KEY_OPENBRACE	= 65;
  MG_KEY_CLOSEBRACE	= 66;
  MG_KEY_ENTER		= 67;
  MG_KEY_COLON		= 68;
  MG_KEY_QUOTE		= 69;
  MG_KEY_BACKSLASH	= 70;
  MG_KEY_BACKSLASH2	= 71;
  MG_KEY_COMMA  	= 72;
  MG_KEY_STOP		= 73;
  MG_KEY_SLASH  	= 74;
  MG_KEY_SPACE  	= 75;
  MG_KEY_INSERT  	= 76;
  MG_KEY_DEL  	     	= 77;
  MG_KEY_HOME  	     	= 78;
  MG_KEY_END  	     	= 79;
  MG_KEY_PGUP  	     	= 80;
  MG_KEY_PGDN  	     	= 81;
  MG_KEY_LEFT  		= 82;
  MG_KEY_RIGHT  	= 83;
  MG_KEY_UP  	     	= 84;
  MG_KEY_DOWN  	     	= 85;
  MG_KEY_SLASH_PAD      = 86;
  MG_KEY_ASTERISK       = 87;
  MG_KEY_MINUS_PAD      = 88;
  MG_KEY_PLUS_PAD       = 89;
  MG_KEY_DEL_PAD  	= 90;
  MG_KEY_ENTER_PAD      = 91;
  MG_KEY_PRTSCR  	= 92;
  MG_KEY_PAUSE  	= 93;
  MG_KEY_ABNT_C1  	= 94;
  MG_KEY_YEN  	  	= 95;
  MG_KEY_KANA  	     	= 96;
  MG_KEY_CONVERT  	= 97;
  MG_KEY_NOCONVERT      = 98;
  MG_KEY_AT  	      	= 99;
  MG_KEY_CIRCUMFLEX     = 100;
  MG_KEY_COLON2  	= 101;
  MG_KEY_KANJI  	= 102;

  MG_KEY_MODIFIERS      = 103;

  MG_KEY_LSHIFT  	= 103;
  MG_KEY_RSHIFT 	= 104;
  MG_KEY_LCONTROL       = 105;
  MG_KEY_RCONTROL       = 106;
  MG_KEY_ALT  	      	= 107;
  MG_KEY_ALTGR  	= 108;
  MG_KEY_LWIN  	      	= 109;
  MG_KEY_RWIN  	      	= 110;
  MG_KEY_MENU  	      	= 111;
  MG_KEY_SCRLOCK  	= 112;
  MG_KEY_NUMLOCK  	= 113;
  MG_KEY_CAPSLOCK       = 114;
{$ENDIF}

{ KEY_MAX was defined above. }



(*******************
 * Sound and music *
 *******************)

 TYPE
   MG_SAMPLEptr = POINTER;
   MG_MIDIptr = POINTER;

(* mgSetSoundVolume:
 *   Alters the global sound output volume. Specify volumes for both digital
 *   samples and MIDI playback, as integers from 0 to 255, or pass a negative
 *   value to leave one of the settings unchanged. Values bigger than 255 will
 *   be reduced to 255. *)
PROCEDURE mgSetSoundVolume (DigiVolume, MidiVolume: LONGINT); CDECL;

(* mgLoadSample:
 *   Loads a WAV or VOC sample. *)
FUNCTION mgLoadSample (Filename: STRING): MG_SAMPLEptr;

(* mgDestroySample:
 *   Frees the resources used by the sample. *)
PROCEDURE mgDestroySample (Sample: MG_SAMPLEptr); CDECL;

(* mgPlaySample:
 *   Triggers a sample at the specified volume, pan position, and frequency.
 *   The parameters `vol' and `pan' range from 0 (min/left) to 255 (max/right).
 *   Frequency is relative rather than absolute: 1000 represents the frequency
 *   that the sample was recorded at, 2000 is twice this, etc. If `loop' is
 *   TRUE, the sample will repeat until you call mgStopSample, and can be
 *   manipulated while it is playing by calling mgAdjustSample. Returns the
 *   voice number that was allocated for the sample or negative if no voices
 *   were available.*)
FUNCTION mgPlaySample (Sample: MG_SAMPLEptr; Vol, Pan, Freq: INTEGER; Loop: BOOLEAN): INTEGER;

(* mgAdjustSample:
 *   Alters the parameters of a sample while it is playing (useful for
 *   manipulating looped sounds).  You can alter the volume, pan, and frequency,
 *   and can also clear the loop flag, which will stop the sample when it next
 *   reaches the end of its loop.  The values of the parameters are just like
 *   those of mgPlaySample().  If there are several copies of the same sample
 *   playing, this will adjust the first one it comes across.  If the sample is
 *   not playing it has no effect. *)
PROCEDURE mgAdjustSample (Sample: MG_SAMPLEptr; Vol, Pan, Freq: INTEGER; Loop: BOOLEAN);

(* mgStopSample:
 *   Kills off a sample, which is required if you have set a sample going in
 *   looped mode.  If there are several copies of the sample playing, it will
 *   stop them all. *)
PROCEDURE mgStopSample (Sample: MG_SAMPLEptr); CDECL;

(* mgLoadMIDI:
 *   Loads a MID file. *)
FUNCTION mgLoadMIDI (Filename: STRING): MG_MIDIptr;

(* mgDestroyMIDI:
 *   Frees the resources used by the MIDI. *)
PROCEDURE mgDestroyMIDI (MIDI: MG_MIDIptr); CDECL;

(* mgPlayMIDI:
 *   Starts playing the specified MIDI file, first stopping whatever music was
 *   previously playing.  If the loop flag is set to TRUE, the data will be
 *   repeated until replaced with something else, otherwise it will stop at the
 *   end of the file.  Passing a NILL pointer will stop whatever music is
 *   currently playing.  Returns FALSE if an error occurs. *)
FUNCTION mgPlayMIDI (MIDI: MG_MIDIptr; Loop: BOOLEAN): BOOLEAN;

(* mgStopMIDI:
 *   Stops whatever music is currently playing. *)
PROCEDURE mgStopMIDI;

(* TODO -oNiunio:  There are a lot of MIDI functions that would be done as send
 *		  MIDI messages, track MIDI events, etc. *)


(**************
 * Data files *
 **************)

  TYPE
    MG_DATAFILE_PROPERTYptr = ^MG_DATAFILE_PROPERTY;
    MG_DATAFILE_PROPERTY = RECORD
      dat : PCHAR;		{ pointer to the data  }
      _type : LONGINT;		{ property type  }
    END;
    MG_DATAFILEptr = ^MG_DATAFILE;
    MG_DATAFILE = RECORD
      dat : POINTER;		{ pointer to the data  }
      _type : LONGINT;		{ object type  }
      size : LONGINT;		{ size of the object  }
      prop : MG_DATAFILE_PROPERTYptr; { object properties  }
    END;

  FUNCTION mgLoadDatafile (filename: STRING): MG_DATAFILEptr;
  PROCEDURE mgUnloadDatafile (dat: MG_DATAFILEptr); CDECL;

  FUNCTION mgLoadDatafileObject (filename, objectname: STRING): MG_DATAFILEptr;
  PROCEDURE mgUnloadDatafileObject (dat: MG_DATAFILEptr); CDECL;



IMPLEMENTATION

{$INLINE ON}

  USES allegAPI; { Interfaces with Allegro Game Library. }



(* ConvertToASCIIZ:
 *   Allegro uses null terminated strings (ASCII-Z), so Pascal strings must be
 *   converted before to use them. *)
FUNCTION ConvertToASCIIZ (const text: STRING): PCHAR; INLINE;
VAR
  Tmp: STRING;
BEGIN
{ Convert to null terminated string. }
  Tmp := text + #0;
  ConvertToASCIIZ := @Tmp[1];
END;



(**********
 * System *
 **********)

(* mgStartup:
 *   Library initialization.  It installs the basic services too. *)
FUNCTION mgStartup: BOOLEAN;
BEGIN
{ Default Allegro initialisation. }
  set_uformat ($41534338); { Uses UNICODE ASCII (Allegro's U_ASCII ident.). }
  mgStartup := install_allegro (0, @mgErrno, NIL) = 0;
  IF mgStartup THEN
  BEGIN
  { Disables any color conversion when loading bitmap files. }
    set_color_conversion (0);
  { Installs additional stuff. }
    install_keyboard;
    install_timer;
    install_sound (-1, -1, NIL);
  END;
END;



(* mgShutdown:
 *   Clean up. *)
procedure mgShutdown;
BEGIN
(* Uninstall Allegro.  This function uninstalls the services. *)
  allegro_exit;
END;



(* mgSleep:
 *   Stops the program the given milliseconds. *)
PROCEDURE mgSleep (MilliSeconds: INTEGER); CDECL;
  EXTERNAL ALLEGRO_SHARED_LIBRARY_NAME NAME 'rest';


(*********
 * Timer *
 *********)

CONST
  TIMERS_PER_SECOND     = 1193181;



(* mgSecsToTimer:
 *   Give the number of seconds between each tick. *)
FUNCTION mgSecsToTimer (x: LONGINT): LONGINT;
BEGIN
  mgSecsToTimer := x * TIMERS_PER_SECOND;
END;



(* mgMsecToTimer:
 *   Give the number of milliseconds between ticks. *)
FUNCTION mgMsecToTimer (x: LONGINT): LONGINT;
BEGIN
  mgMsecToTimer := TRUNC (x * (TIMERS_PER_SECOND / 1000));
END;



(* mgBpsToTimer:
 *   Give the number of ticks each second. *)
FUNCTION mgBpsToTimer  (x: LONGINT): LONGINT;
BEGIN
  mgBpsToTimer := TIMERS_PER_SECOND DIV x;
END;



(* mgBpmToTimer:
 *  Give the number of ticks per minute. *)
FUNCTION mgBpmToTimer  (x: LONGINT): LONGINT;
BEGIN
  mgBpmToTimer  := (60 * TIMERS_PER_SECOND) DIV x;
END;



(* mgInstallInterruption:
 *   Installs the given interruption at given speed. *)
FUNCTION mgInstallInterruption (proc: MG_CALLBACK_SIMPLE_PROC; speed: LONGINT): LONGINT; CDECL;
  EXTERNAL ALLEGRO_SHARED_LIBRARY_NAME NAME 'install_int_ex';



(* mgRemoveInterruption:
 *   Removes the given interruption. *)
PROCEDURE mgRemoveInterruption (proc: MG_CALLBACK_SIMPLE_PROC); CDECL;
  EXTERNAL ALLEGRO_SHARED_LIBRARY_NAME NAME 'remove_int';



(***********
 * Bitmaps *
 ***********)

{ Creates bitmaps. }
FUNCTION mgCreateBitmap (w, h: LONGINT): MG_BITMAPptr; CDECL;
  EXTERNAL ALLEGRO_SHARED_LIBRARY_NAME NAME 'create_bitmap';
FUNCTION mgCreateBitmapEx (w, h, bpp: LONGINT): MG_BITMAPptr; CDECL;
  EXTERNAL ALLEGRO_SHARED_LIBRARY_NAME NAME 'create_bitmap_ex';
FUNCTION mgCreateSubBitmap (parent: MG_BITMAPptr; x, y, w, h: LONGINT): MG_BITMAPptr; CDECL;
  EXTERNAL ALLEGRO_SHARED_LIBRARY_NAME NAME 'create_sub_bitmap';
{ Frees the bitmap resources.  }
PROCEDURE mgDestroyBitmap (bmp: MG_BITMAPptr); CDECL;
  EXTERNAL ALLEGRO_SHARED_LIBRARY_NAME NAME 'destroy_bitmap';



(* mgBitmapColorDepth:
 *   Returns the color depth of the given bitmap. *)
FUNCTION mgBitmapColorDepth (bmp: MG_BITMAPptr): LONGINT;
BEGIN
  mgBitmapColorDepth := bmp^.vtable^.color_depth;
END;



(* mgBitmapMaskColor:
 *   Returns the color used as "transparent". *)
FUNCTION mgBitmapMaskColor (bmp: MG_BITMAPptr): LONGINT;
BEGIN
  mgBitmapMaskColor := bmp^.vtable^.mask_color;
END;



(* mgIsSameBitmap:
 *   Checks if both bitmaps are the same bitmap or sub-bitmaps of
 *   the same bitmap. *)
FUNCTION mgIsSameBitmap (bmp1, bmp2: MG_BITMAPptr): BOOLEAN;
VAR
  m1, m2: DWORD;
BEGIN
  IF (bmp1 = NIL) OR (bmp2 = NIL) THEN
    mgIsSameBitmap := FALSE
  ELSE
    IF bmp1 = bmp2 THEN
      mgIsSameBitmap := TRUE
    ELSE BEGIN
      m1 := (bmp1^.id AND $01FFFFFF);
      m2 := (bmp2^.id AND $01FFFFFF);
      mgIsSameBitmap := ((m1 <> 0) AND (m1 = m2));
    END;
END;



(* mgIsScreenBitmap:
 *   Check if the given bitmap is the screen or a sub-bitmap of the screen. *)
FUNCTION mgIsScreenBitmap (bmp: MG_BITMAPptr): BOOLEAN;
BEGIN
  mgIsScreenBitmap := mgIsSameBitmap (mgScreen, bmp);
END;



(* mgIsSubBitmap:
 *  Check if the given bitmap is a sub-bitmap of any other. *)
FUNCTION mgIsSubBitmap (bmp: MG_BITMAPptr): BOOLEAN;
BEGIN
  mgIsSubBitmap := (bmp^.id AND $20000000) <> 0;
END;



(******************
 * Graphics modes *
 ******************)
 
(* mgSetGfxMode:
 *   Initializes the requested graphics mode. *)
FUNCTION mgSetGfxMode (w, h, bpp: LONGINT): BOOLEAN;
BEGIN
{ Sets color depth. }
  set_color_depth (bpp);
{ Tryes a full screen mode. }
  mgSetGfxMode := set_gfx_mode (1, w, h, 0, 0) = 0;
  IF NOT mgSetGfxMode THEN
  BEGIN
  { Tryes a windowed mode. }
    mgSetGfxMode := set_gfx_mode (2, w, h, 0, 0) = 0;
    IF NOT mgSetGfxMode THEN
    { Tryes a "safe" mode.  The ID was taken from Allegro's ID_SAFE macro. }
      mgSetGfxMode := set_gfx_mode ($53414645, w, h, 0, 0) = 0;
  END;
  IF mgSetGfxMode THEN
  BEGIN
  { Get screen information. }
    mgScreenW := w; mgScreenH := h;
  END;
END;



(* mgSetGfxModeEx:
 *   An diferent version of the mgSetGfxMode function. *)
FUNCTION mgSetGfxModeEx (w, h, bpp: LONGINT; windowed: BOOLEAN): BOOLEAN;
VAR
  ID: LONGINT;
BEGIN
{ Selects the driver identification. }
  if windowed THEN ID := 2 ELSE ID := 1;
{ Sets color depth. }
  set_color_depth (bpp);
{ Sets the graphics mode. }
  mgSetGfxModeEx := set_gfx_mode (ID, w, h, 0, 0) = 0;
  IF mgSetGfxModeEx THEN
  BEGIN
  { Get screen information. }
    mgScreenW := w; mgScreenH := h;
  END;
END;



(* mgExitGfxMode:
 *   Closes the current graphics mode. *)
PROCEDURE mgExitGfxMode;
BEGIN
  set_gfx_mode (0, 0, 0, 0, 0);
  mgScreenW := 0; mgScreenH := 0;
END;


PROCEDURE mgWaitVerticalRetrace; CDECL;
  EXTERNAL ALLEGRO_SHARED_LIBRARY_NAME NAME 'vsync';



(**********************
 * Color and palettes *
 **********************)

(* mgColor & mgColorDepth:
 *   Make a "color" from his RGB description. *)
FUNCTION mgColor (r, g, b: LONGINT): LONGINT; CDECL;
  EXTERNAL ALLEGRO_SHARED_LIBRARY_NAME NAME 'makecol';
FUNCTION mgColorDepth (color_depth, r, g, b: LONGINT): LONGINT; CDECL;
  EXTERNAL ALLEGRO_SHARED_LIBRARY_NAME NAME 'makecol_depth';



(* mgRGB & mgRGBDepth:
 *   "Unmake" a color to his RGB description. *)
PROCEDURE mgRGB (c: LONGINT; VAR r, g, b: LONGINT);
BEGIN
  r := getr (c);
  g := getg (c);
  b := getb (c);
END;

PROCEDURE mgRGBDepth (color_depth, c: LONGINT; VAR r, g, b: LONGINT);
BEGIN
  r := getr_depth (color_depth, c);
  g := getg_depth (color_depth, c);
  b := getb_depth (color_depth, c);
END;



PROCEDURE mgSetColor (index: LONGINT; p: MG_RGBptr); CDECL;
  EXTERNAL ALLEGRO_SHARED_LIBRARY_NAME NAME 'set_color';
PROCEDURE mgSetPalette (p: MG_COLOR_PALETTE); CDECL;
  EXTERNAL ALLEGRO_SHARED_LIBRARY_NAME NAME 'set_palette';
PROCEDURE mgGetColor (index: LONGINT; p: MG_RGBptr); CDECL;
  EXTERNAL ALLEGRO_SHARED_LIBRARY_NAME NAME 'get_color';
PROCEDURE mgGetPalette (p: MG_COLOR_PALETTE); CDECL;
  EXTERNAL ALLEGRO_SHARED_LIBRARY_NAME NAME 'get_palette';


  
(* TODO -oNiunio: These color effects should be implemented as to be used with
   "high color" and "true color" depths.  Of course that "high" and "true"
   depths don't need color palettes, so the API would change a bit.  It
   shouln't be too hard.  There are some code snipets about this at Allegro.cc
   and PixWiki. *)
PROCEDURE mgFadeFrom (source, dest: MG_COLOR_PALETTE; speed: LONGINT); CDECL;
   EXTERNAL ALLEGRO_SHARED_LIBRARY_NAME NAME 'fade_from';
PROCEDURE mgFadeIn (p: MG_COLOR_PALETTE; speed: LONGINT); CDECL;
  EXTERNAL ALLEGRO_SHARED_LIBRARY_NAME NAME 'fade_in';
PROCEDURE mgFadeOut (speed: LONGINT); CDECL;
  EXTERNAL ALLEGRO_SHARED_LIBRARY_NAME NAME 'fade_out';



(* mgGenerateOptimizedColorPalette:
 *    Generates a 256-color palette suitable for making a reduced color version
 *    of the specified truecolor image. *)
FUNCTION mgGenerateOptimizedColorPalette (image: MG_BITMAPptr; pal: MG_COLOR_PALETTE; rsvdcols: MG_RSVDCOLSTYPE): LONGINT; CDECL;
  EXTERNAL ALLEGRO_SHARED_LIBRARY_NAME NAME 'generate_optimized_palette';

(* mgGenerate332ColorPalette:
 *   Constructs a fake truecolor palette, using three bits for red and green
 *   and two for the blue. *)
PROCEDURE mgGenerate332ColorPalette (pal: MG_COLOR_PALETTE); CDECL;
  EXTERNAL ALLEGRO_SHARED_LIBRARY_NAME NAME 'generate_332_palette';

  

(**********************
 * Drawing primitives *
 **********************)

{ Draw clipping. }
PROCEDURE mgSetClip (bmp: MG_BITMAPptr; x1, y1, x2, y2: LONGINT); CDECL;
  EXTERNAL ALLEGRO_SHARED_LIBRARY_NAME NAME 'set_clip';



(* mgAcquireBitmap:
 *   Prepares the given bitmap to an output operation. *)
PROCEDURE mgAcquireBitmap (bmp: MG_BITMAPptr);
BEGIN
  IF bmp <> NIL THEN
    bmp^.vtable^.acquire (bmp);
END;



(* mgReleaseBitmap:
 *   Releases the given bitmap from an output operation. *)
PROCEDURE mgReleaseBitmap (bmp: MG_BITMAPptr);
BEGIN
  IF bmp <> NIL THEN
    bmp^.vtable^.release (bmp);
END;



(* mgAcquireScreen:
 *   Prepares the screen bitmap to an output operation. *)
PROCEDURE mgAcquireScreen;
BEGIN
  IF mgScreen <> NIL THEN
    mgScreen^.vtable^.acquire (mgScreen);
END;



(* mgReleaseScreen:
 *   Releases the screen bitmap from an output operation. *)
PROCEDURE mgReleaseScreen;
BEGIN
  IF mgScreen <> NIL THEN
    mgScreen^.vtable^.release (mgScreen);
END;



(* mgClearBitmap:
 *   Clears the bitmap. *)
PROCEDURE mgClearBitmap (bmp: MG_BITMAPptr);
BEGIN
  bmp^.vtable^.clear_to_color (bmp, 0);
END;



(* mgClearBitmapToColor:
 *   Clears the bitmap to the given color. *)
PROCEDURE mgClearBitmapToColor (bmp: MG_BITMAPptr; c: LONGINT);
BEGIN
  bmp^.vtable^.clear_to_color (bmp, c);
END;


(* Drawing: *)

FUNCTION  mgGetPixel (bmp: MG_BITMAPptr; x, y: LONGINT): LONGINT;
BEGIN
  mgGetPixel := bmp^.vtable^.getpixel (bmp, x, y);
END;



PROCEDURE mgPutPixel (bmp: MG_BITMAPptr; x, y, color: LONGINT);
BEGIN
  bmp^.vtable^.putpixel (bmp, x, y, color);
END;



PROCEDURE mgLine (bmp: MG_BITMAPptr; x1, y1, x2, y2, color: LONGINT);
BEGIN
  bmp^.vtable^.line (bmp, x1, y1, x2, y2, color);
END;



PROCEDURE mgRect (bmp: MG_BITMAPptr; x1, y1, x2, y2, color: LONGINT);
BEGIN
  bmp^.vtable^.rect (bmp, x1, y1, x2, y2, color);
END;



PROCEDURE mgRectFill (bmp: MG_BITMAPptr; x1, y1, x2, y2, color: LONGINT);
BEGIN
  bmp^.vtable^.rectfill (bmp, x1, y1, x2, y2, color);
END;



PROCEDURE mgCircle (bmp: MG_BITMAPptr; x, y, r, color: LONGINT);
BEGIN
  bmp^.vtable^.circle (bmp, x, y, r, color);
END;



PROCEDURE mgCircleFill (bmp: MG_BITMAPptr; x, y, r, color: LONGINT);
BEGIN
  bmp^.vtable^.circlefill (bmp, x, y, r, color);
END;



PROCEDURE mgEllipse (bmp: MG_BITMAPptr; x, y, rx, ry, color: LONGINT);
BEGIN
  bmp^.vtable^.ellipse (bmp, x, y, rx, ry, color);
END;



PROCEDURE mgEllipseFill (bmp: MG_BITMAPptr; x, y, rx, ry, color: LONGINT);
BEGIN
  bmp^.vtable^.ellipsefill (bmp, x, y, rx, ry, color);
END;



PROCEDURE mgFill (bmp: MG_BITMAPptr; x, y, color: LONGINT);
BEGIN
  bmp^.vtable^.floodfill (bmp, x, y, color);
END;



(*****************************
 * Loading and saving images *
 *****************************)

(* mg_load_bitmap:
 *   This function can load BMP, PCX, TGA and LBM files. *)
FUNCTION mgLoadBitmap (CONST FileName: STRING; VAR Palette: MG_COLOR_PALETTE): MG_BITMAPptr;
VAR
  PPal: MG_RGBptr;
  T: MG_RGB;
  Cnt: INTEGER;
BEGIN
{ Some initialisation. }
  GetMem (PPal, SizeOf (T) * 256);
{ Loads the bitmap. }
  mgLoadBitmap := load_bitmap (ConvertToASCIIZ (filename), PPal);
{ If the bitmap was loaded, it needs some conversion. }
  IF mgLoadBitmap <> NIL THEN
  { Ports the palette. }
    FOR Cnt := 0 TO 255 DO
    BEGIN
      Palette[Cnt].r := PPal[Cnt].r;
      Palette[Cnt].g := PPal[Cnt].g;
      Palette[Cnt].b := PPal[Cnt].b;
    END;
{ Frees resources. }
  Dispose (PPal);
END;



(* mg_save_bitmap:
 *   This function can save BMP, PCX and TGA files. *)
FUNCTION mgSaveBitmap (CONST FileName: STRING; Bmp: MG_BITMAPptr; Palette: MG_COLOR_PALETTE): BOOLEAN;
VAR
  PPal: MG_RGBptr;
  T: MG_RGB;
  Cnt: INTEGER;
BEGIN
{ Ports the palette. }
  GetMem (PPal, SizeOf (T) * 256);
  FOR Cnt := 0 TO 255 DO
  BEGIN
    PPal[Cnt].r := Palette[Cnt].r;
    PPal[Cnt].g := Palette[Cnt].g;
    PPal[Cnt].b := Palette[Cnt].b;
  END;
{ Saves the bitmap. }
  mgSaveBitmap := save_bitmap (ConvertToASCIIZ (FileName), bmp, PPal) = 0;
{ Frees the resources. }
  Dispose (PPal);
END;



(*************************
 * Sprites and blitting. *
 *************************)

(* mgBlit:
 *   Blits a bitmap onto another bitmap. *)
PROCEDURE mgBlit (Orig, Dest: MG_BITMAPptr; x, y: LONGINT);
BEGIN
  blit (Orig, Dest, 0, 0, x, y, Orig^.w, Orig^.h);
END;



(* mgBlitEx:
 *   Blits a bitmap onto another bitmap. *)
PROCEDURE mgBlitEx (Orig, Dest: MG_BITMAPptr; ox, oy, dx, dy, w, h: LONGINT);
BEGIN
  blit (Orig, Dest, ox, oy, dx, dy, w, h);
END;


(* mgDrawSprite:
 *   Draws the given sprite in the given bitmap. *)
PROCEDURE mgDrawSprite (bmp, sprite: MG_BITMAPptr; x, y: LONGINT);
BEGIN
  ASSERT (bmp <> NIL);
  ASSERT (sprite <> NIL);

  if sprite^.vtable^.color_depth = 8 THEN
    bmp^.vtable^.draw_256_sprite (bmp, sprite, x, y)
  ELSE BEGIN
    ASSERT (bmp^.vtable^.color_depth = sprite^.vtable^.color_depth);
    bmp^.vtable^.draw_sprite (bmp, sprite, x, y);
  END;
END;



(* Draws flipped and rotatedsprites. *)
PROCEDURE mgDrawSpriteHflip (bmp, sprite: MG_BITMAPptr; x, y: LONGINT);
BEGIN
  ASSERT (bmp <> NIL);
  ASSERT (sprite <> NIL);
  ASSERT (bmp^.vtable^.color_depth = sprite^.vtable^.color_depth);

  bmp^.vtable^.draw_sprite_h_flip (bmp, sprite, x, y);
END;

PROCEDURE mgDrawSpriteVflip (bmp, sprite: MG_BITMAPptr; x, y: LONGINT);
BEGIN
  ASSERT (bmp <> NIL);
  ASSERT (sprite <> NIL);
  ASSERT (bmp^.vtable^.color_depth = sprite^.vtable^.color_depth);

  bmp^.vtable^.draw_sprite_v_flip (bmp, sprite, x, y);
END;

PROCEDURE mgDrawSpriteVHflip (bmp, sprite: MG_BITMAPptr; x, y: LONGINT);
BEGIN
  ASSERT (bmp <> NIL);
  ASSERT (sprite <> NIL);
  ASSERT (bmp^.vtable^.color_depth = sprite^.vtable^.color_depth);

  bmp^.vtable^.draw_sprite_vh_flip (bmp, sprite, x, y);
END;



PROCEDURE mgDrawRotatedSprite (bmp, sprite: MG_BITMAPptr; x, y: LONGINT; angle: LONGINT);
BEGIN
  ASSERT (bmp <> NIL);
  ASSERT (sprite <> NIL);

  bmp^.vtable^.pivot_scaled_sprite_flip (bmp, sprite, (x SHL 16) + (sprite^.w * $10000) DIV 2,
						      (y SHL 16) + (sprite^.h * $10000) DIV 2,
						      sprite^.w SHL 15, sprite^.h SHL 15,
						      angle SHL 16, $10000, 0);
END;

PROCEDURE mgDrawRotatedVflipSprite (bmp, sprite: MG_BITMAPptr; x, y: LONGINT; angle: LONGINT);
BEGIN
  ASSERT (bmp <> NIL);
  ASSERT (sprite <> NIL);

  bmp^.vtable^.pivot_scaled_sprite_flip (bmp, sprite, (x SHL 16) + (sprite^.w * $10000) DIV 2,
						      (y SHL 16) + (sprite^.h * $10000) DIV 2,
						      sprite^.w SHL 15, sprite^.h SHL 15,
						      angle SHL 16, $10000, NOT 0);
END;



(***************
 * Text output *
 ***************)

(* mgTextLength:
 *   Length of text in pixels. *)
FUNCTION mgTextLength (const text: STRING; const font: MG_TEXT_FONTptr): LONGINT;
BEGIN
  mgTextLength := text_length (font, ConvertToASCIIZ (text));
END;



(* mgTextHeight:
 *   Height of the text in pixels. *)
FUNCTION mgTextHeight (const font: MG_TEXT_FONTptr): LONGINT; CDECL;
  EXTERNAL ALLEGRO_SHARED_LIBRARY_NAME NAME 'text_height';



(* mgTextout:
 *   Text output. *)
PROCEDURE mgTextout (bmp: MG_BITMAPptr; font: MG_TEXT_FONTptr; x, y, c: LONGINT; text: STRING);
BEGIN
{ Background is always transparent. }
  textout_ex (bmp, font, ConvertToASCIIZ (text), x, y, c, -1);
END;



(************
 * Keyboard *
 ************)

(* MgReadKey:
 *   Waits for a keyboard input. *)
FUNCTION mgReadKey: LONGINT; CDECL;
  EXTERNAL ALLEGRO_SHARED_LIBRARY_NAME NAME 'readkey';

(* mgUReadKey:
 *   UNICODE version of mgReadKey. *)
FUNCTION mgUReadKey (scancode: PLONGINT): LONGINT; CDECL;
  EXTERNAL ALLEGRO_SHARED_LIBRARY_NAME NAME 'ureadkey';

(* mgKeyPressed:
 *   Retruns TRUE if a key was pressed. *)
FUNCTION mgKeyPressed: BOOLEAN; CDECL;
  EXTERNAL ALLEGRO_SHARED_LIBRARY_NAME NAME 'keypressed';

(* mgClearKeyBuf:
 *   Clear the keyboard input fuffer. *)
PROCEDURE mgClearKeyBuf; CDECL;
  EXTERNAL ALLEGRO_SHARED_LIBRARY_NAME NAME 'clear_keybuf';

(* mgPollKeyboard:
 *   Updates the keyboard state variables. *)
PROCEDURE mgPollKeyboard; CDECL;
  EXTERNAL ALLEGRO_SHARED_LIBRARY_NAME NAME 'poll_keyboard';

(* mgSimulateKeyPress:
 *   Simulate keyboard use. *)
PROCEDURE mgSimulateKeyPress (keycode: LONGINT); CDECL;
  EXTERNAL ALLEGRO_SHARED_LIBRARY_NAME NAME 'simulate_keypress';

(* mgSimulateUKeyPress:
 *   UNICODE version of mgSimulateKeyPress. *)
PROCEDURE mgSimulateUKeyPress (keycode, scancode: LONGINT); CDECL;
  EXTERNAL ALLEGRO_SHARED_LIBRARY_NAME NAME 'simulate_ukeypress';

(* mgSetKeyboardRate:
 *   Set keyboard repeat rate. *)
PROCEDURE mgSetKeyboardRate (delay, _repeat: LONGINT); CDECL;
  EXTERNAL ALLEGRO_SHARED_LIBRARY_NAME NAME 'set_keyboard_rate';

(* mgScanCodeToASCII:
 * *)
FUNCTION mgScanCodeToASCII (scancode: LONGINT): LONGINT; CDECL;
  EXTERNAL ALLEGRO_SHARED_LIBRARY_NAME NAME 'scancode_to_ascii';



(*******************
 * Sound and music *
 *******************)

(* mgSetSoundVolume:
 *   Alters the global sound output volume. Specify volumes for both digital
 *   samples and MIDI playback, as integers from 0 to 255, or pass a negative
 *   value to leave one of the settings unchanged. Values bigger than 255 will
 *   be reduced to 255. *)
PROCEDURE mgSetSoundVolume (DigiVolume, MidiVolume: LONGINT); CDECL;
  EXTERNAL ALLEGRO_SHARED_LIBRARY_NAME NAME 'set_volume';

(* mgLoadSample:
 *   Loads a WAV or VOC sample. *)
FUNCTION mgLoadSample (Filename: STRING): MG_SAMPLEptr;
BEGIN
  mgLoadSample := load_sample (ConvertToASCIIZ (Filename));
END;

(* mgDestroySample:
 *   Frees the resources used by the sample. *)
PROCEDURE mgDestroySample (Sample: MG_SAMPLEptr); CDECL;
  EXTERNAL ALLEGRO_SHARED_LIBRARY_NAME NAME 'destroy_sample';

(* mgPlaySample:
 *   Triggers a sample at the specified volume, pan position, and frequency.
 *   The parameters `vol' and `pan' range from 0 (min/left) to 255 (max/right).
 *   Frequency is relative rather than absolute: 1000 represents the frequency
 *   that the sample was recorded at, 2000 is twice this, etc. If `loop' is
 *   TRUE, the sample will repeat until you call mgStopSample, and can be
 *   manipulated while it is playing by calling mgAdjustSample. Returns the
 *   voice number that was allocated for the sample or negative if no voices
 *   were available.*)
FUNCTION mgPlaySample (Sample: MG_SAMPLEptr; Vol, Pan, Freq: INTEGER; Loop: BOOLEAN): INTEGER;
VAR
  FR: LONGINT;
BEGIN
  IF Loop THEN FR := 1 ELSE FR := 0;
  mgPlaySample := play_sample (Sample, Vol, Pan, Freq, FR);
END;

(* mgAdjustSample:
 *   Alters the parameters of a sample while it is playing (useful for
 *   manipulating looped sounds).  You can alter the volume, pan, and frequency,
 *   and can also clear the loop flag, which will stop the sample when it next
 *   reaches the end of its loop.  The values of the parameters are just like
 *   those of mgPlaySample().  If there are several copies of the same sample
 *   playing, this will adjust the first one it comes across.  If the sample is
 *   not playing it has no effect. *)
PROCEDURE mgAdjustSample (Sample: MG_SAMPLEptr; Vol, Pan, Freq: INTEGER; Loop: BOOLEAN);
VAR
  FR: LONGINT;
BEGIN
  IF Loop THEN FR := 1 ELSE FR := 0;
  adjust_sample (Sample, Vol, Pan, Freq, FR);
END;

(* mgStopSample:
 *   Kills off a sample, which is required if you have set a sample going in
 *   looped mode.  If there are several copies of the sample playing, it will
 *   stop them all. *)
PROCEDURE mgStopSample (Sample: MG_SAMPLEptr); CDECL;
  EXTERNAL ALLEGRO_SHARED_LIBRARY_NAME NAME 'stop_sample';

(* mgLoadMIDI:
 *   Loads a MID file. *)
FUNCTION mgLoadMIDI (Filename: STRING): MG_MIDIptr;
BEGIN
  mgLoadMIDI := load_midi (ConvertToASCIIZ (Filename));
END;

(* mgDestroyMIDI:
 *   Frees the resources used by the MIDI. *)
PROCEDURE mgDestroyMIDI (MIDI: MG_MIDIptr); CDECL;
  EXTERNAL ALLEGRO_SHARED_LIBRARY_NAME NAME 'destroy_midi';

(* mgPlayMIDI:
 *   Starts playing the specified MIDI file, first stopping whatever music was
 *   previously playing.  If the loop flag is set to TRUE, the data will be
 *   repeated until replaced with something else, otherwise it will stop at the
 *   end of the file.  Passing a NULL pointer will stop whatever music is
 *   currently playing.  Returns FALSE if an error occurs. *)
FUNCTION mgPlayMIDI (MIDI: MG_MIDIptr; Loop: BOOLEAN): BOOLEAN;
VAR
  FR: LONGINT;
BEGIN
  IF Loop THEN FR := 1 ELSE FR := 0;
  mgPlayMIDI := play_midi (MIDI, FR) = 0;
END;

(* mgStopMIDI:
 *   Stops whatever music is currently playing. *)
PROCEDURE mgStopMIDI;
BEGIN
  play_midi (NIL, 0);
END;



(**************
 * Data files *
 **************)

FUNCTION mgLoadDatafile (filename: STRING): MG_DATAFILEptr;
BEGIN
  mgLoadDatafile := load_datafile (ConvertToASCIIZ (filename));
END;

PROCEDURE mgUnloadDatafile (dat: MG_DATAFILEptr); CDECL;
  EXTERNAL ALLEGRO_SHARED_LIBRARY_NAME NAME 'unload_datafile';

FUNCTION mgLoadDatafileObject (filename, objectname: STRING): MG_DATAFILEptr;
BEGIN
  mgLoadDatafileObject := load_datafile_object (ConvertToASCIIZ (filename),
						ConvertToASCIIZ (objectname));
END;

PROCEDURE mgUnloadDatafileObject (dat: MG_DATAFILEptr); CDECL;
   EXTERNAL ALLEGRO_SHARED_LIBRARY_NAME NAME 'unload_datafile_object';



END.

