UNIT damero;
(* Programa: Reversi, versin 2.x
 * Archivo: damero.pas
 * Descripcin: Facilita la creacin de dameros para juegos de mesa.  Es un
 *		mdulo externo ya que puede servir para otros juegos.
 * Autor: uo Martnez < >
 *
    Copyright (C) uo Martnez, 2005

    Este programa es un Programa de Libre Distribucin; usted
    puede redistribuirlo y/o modificarlo bajo los trminos de
    la "Licencia de Libre Distribucin de Programas (LLDP)".

    Este programa es distribuido con la esperanza de que le ser
    til, pero SIN NINGUNA GARANTA. Vea la "Licencia de Libre
    Distribucin de Programas" para ms detalles.

    Usted debe haber recibido una copia de la "Licencia de Libre
    Distribucin de Programas" junto con este programa, si no es
    as, escriba a:

    Guimarprog, Apdo. 4034 09080 Burgos, Espaa

 *)

INTERFACE

USES mingro, Sysutils;

TYPE
(* EXCEP_DAMERO:
 *   Para discriminar errores. *)
  EXCEP_DAMERO = CLASS (EXCEPTION) END;



(* CTABLERO:
 *   Encapsula un tablero genrico. *)
  TABLERO = CLASS (TOBJECT)
  PRIVATE
  { Definicin del tablero.  Se refieren al tamao de un solo lado, ya que
    siempre son cuadrados. }
    fNumEscaques, fTamEscaque: INTEGER; fEscaques: PCHAR;
    fx, fy: INTEGER; fBmp: MG_BITMAPptr; (* Dnde lo dibujar. *)
  PROTECTED
    FUNCTION ObtieneEscaque (x, y: INTEGER): CHAR;
    PROCEDURE PonEscaque (x, y: INTEGER; Valor: CHAR);
  PUBLIC
  (* Create:
   *   Constructor.  Crea un tablero con el nmero de escaques dado del tamao
   *   indicado.  Las coordenadas es donde lo dibujar. *)
    CONSTRUCTOR Create (NumEscaques, TamEscaques, x, y: INTEGER);
  (* Dibuja:
   *   Dibuja el damero completo pero vaco. *)
    PROCEDURE Dibuja;
  (* PosACasilla:
   *   Obtiene una posicin en pantalla y devuelve la casilla o negativo si
   *   est fuera. *)
    PROCEDURE PosACasilla (px, py: INTEGER; VAR cx, cy: INTEGER);
  (* Free:
   *   Destructor.  Libera los recursos utilizados por el tablero. *)
    DESTRUCTOR Free;
  (* Propiedades para acceder a los datos del tablero. *)
    PROPERTY Escaque[x, y: INTEGER]: CHAR READ ObtieneEscaque WRITE PonEscaque;
    PROPERTY Bmp: MG_BITMAPptr READ fBmp;
  END;


CONST
  Fuera = #0; { Devuelto por la funcin de acceso a escaques. }

IMPLEMENTATION

(* Tablero.ObtieneEscaque:
 *   Devuelve el escaque indicado por las coordenadas o Vacio si est fuera
 *   del tablero. *)
FUNCTION Tablero.ObtieneEscaque (x, y: INTEGER): CHAR;
BEGIN
{ Comprueba si est fuera del tablero. }
  IF (x < 0) OR (y < 0) OR (x >= fNumEscaques) OR (y >= fNumEscaques) THEN
    ObtieneEscaque := Fuera
  ELSE
    ObtieneEscaque := fEscaques[(y * fNumEscaques) + x];
END;



(* Tablero.PonEscaque:
 *   Establece el valor del escaque indicado por las coordenadas. *)
PROCEDURE Tablero.PonEscaque (x, y: INTEGER; Valor: CHAR);
BEGIN
{ Comprueba si est fuera del tablero. }
  IF (x < 0) OR (y < 0) OR (x >= fNumEscaques) OR (y >= fNumEscaques) THEN
    RAISE EXCEP_DAMERO.CREATE (
			'Se intent acceder a un escaque fuera del tablero');
{ En otro caso, asigna el nuevo valor. }
  fEscaques[(y * fNumEscaques) + x] := Valor;
END;

  

(* TABLERO.Create:
 *   Constructor.  Crea un tablero con el nmero de escaques dado del tamao
 *   indicado. *)
CONSTRUCTOR TABLERO.Create (NumEscaques, TamEscaques, x, y: INTEGER);
VAR
  Cnt: INTEGER;
BEGIN
{ Valores iniciales. }
  fEscaques := NIL; fBmp := NIL;
{ Dnde dibujar el tablero. }
  fx := x; fy := y;
{ Almacena los tamaos y comprueba que sean correctos. }
  fNumEscaques := TRUNC (SQRT (NumEscaques)); fTamEscaque := TamEscaques;
  if fNumEscaques * fNumEscaques <> NumEscaques THEN
    RAISE EXCEP_DAMERO.CREATE (
		'El nmero de escaques ha de ser un cuadrado perfecto');
{ Crea el tablero. }
  GETMEM (SELF.fEscaques, NumEscaques);
  IF fEscaques = NIL THEN
    RAISE EXCEP_DAMERO.CREATE ('No pudo crearse el tablero.');
{ Obtiene el subbitmap donde dibujar el tablero. }
  fBmp := mgCreateSubBitmap (mgScreen, x, y,
		fNumEscaques * TamEscaques, fNumEscaques * TamEscaques);
  IF fBmp = NIL THEN
    RAISE EXCEP_DAMERO.CREATE ('No pudo crearse el tablero.');
{ Definicin del tablero. }
  FOR Cnt := 0 TO NumEscaques - 1 DO
    fEscaques[Cnt] := #0;
END;



(* TABLERO.Dibuja:
 *   Dibuja el damero completo pero vaco. *)
PROCEDURE TABLERO.Dibuja;
VAR
  x, y, Color: INTEGER;
BEGIN
{ Aceleramos las cosas. }
  mgAcquireBitmap (fBmp);
  FOR y := 0 TO fNumEscaques - 1 DO
    FOR x := 0 TO fNumEscaques - 1 DO
    BEGIN
    { Si ambos escaques son pares o los dos son impares, es negro. }
      IF x MOD 2 = y MOD 2 THEN
	Color := mgColor (0, 0, 0)
    { En otro caso, es blanco. }
      ELSE
	Color := mgColor (255, 255, 255);
      mgRectFill (fBmp,
		  x * fTamEscaque, y * fTamEscaque,
		  (x * fTamEscaque) + fTamEscaque - 1,
		  (y * fTamEscaque) + fTamEscaque - 1,
		  Color);
    END;
{ Finaliza el dibujo. }
  mgReleaseBitmap (fBmp);
END;



(* TABLERO.PosACasilla:
 *   Obtiene una posicin en pantalla y devuelve la casilla o negativo si
 *   est fuera. *)
PROCEDURE TABLERO.PosACasilla (px, py: INTEGER; VAR cx, cy: INTEGER);
BEGIN
{ Pasa a coordenadas locales. }
  px := px - fBmp^.x_ofs; py := py - fBmp^.y_ofs;
{ Comprueba si est fuera. }
  IF (0 > px) OR (px > fBmp^.w) OR (0 > py) OR (py > fBmp^.h) THEN
  BEGIN
    cx := -1; cy := -1; EXIT;
  END;
{ Realiza el clculo. }
  cx := px DIV fTamEscaque; cy := py DIV fTamEscaque;
END;



(* TABLERO.Free:
 *   Destructor.  Libera los recursos utilizados por el tablero. *)
DESTRUCTOR TABLERO.Free;
BEGIN
{ Liberamos tablero. }
  IF fEscaques <> NIL THEN
  BEGIN
    FREEMEM (fEscaques, fNumEscaques * fNumEscaques);
    fEscaques := NIL;
  END;
  IF fBmp <> NIL THEN
  BEGIN
    mgDestroyBitmap (fBmp);
    fBmp := NIL;
  END;
END;

END.

