/*
 * This module handles the music and sound. If you don't like
 * libmodplug, define NO_SOUND to turn it all off. You don't get sound
 * effects either, primarily because I never bothered to make those a
 * separate switch.
 */

#ifdef NO_SOUND
void sound_init(void) { }
void sound_stop(void) { }
void sound_start(void) { }
void sound_update(void) { }
void sound_play(int sample) { }
void sound_nextmusic(void) { }
void sound_playtitlemusic(void) { }
void sound_restartmusic(void) { }
#else

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <allegro.h>
#include "modplug.h"
#include "main.h"
#include "config.h"
#include "sound.h"

#define NUMMUSICS 2 /* If you change this, change sound_init() below. */
static ModPlugFile *mods[NUMMUSICS+1]; /* Hack: mods[NUMMUSICS] is title. */
static int playing = NUMMUSICS; /* Title music. */
static int soundinited; /* If the sound has initialized or not. */
static int sounddisabled;
static int soundstopped; /* Temporarily stopped? (pause/gameover screens) */

static void *readfile(const char *filename, int *len)
{
	char *buf = NULL;
	int nbuf = 0, sbuf = 0;
	int c;
	FILE *fp = fopen(filename, "rb");

	if (fp == NULL) abort_with_error("File %s: cannot open", filename);

	while ((c = getc(fp)) != EOF)
	{
		XPND(buf, nbuf, sbuf);
		buf[nbuf++] = c;
	}
	fclose(fp);
	*len = nbuf;
	return buf;
}

#define MUSIC_BITS 16
#define MUSIC_MODPLUG_BITS 16
#define MUSIC_STEREO TRUE
#define MUSIC_FREQ 44100
#define MUSIC_VOL 255
#define MUSIC_PAN 128
#define MUSIC_CHANNELS (MUSIC_STEREO ? 2 : 1)
#define MUSIC_BYTES (MUSIC_BITS/8)

/*
 * The following settings are copied from mp2wav's defaults, with two
 * changes:
 *    1. surround enabled;
 *    2. linear instead of spline resampling (I was worried about CPU
 *       use).
 */

#define DEF_FLAGS          MODPLUG_ENABLE_OVERSAMPLING \
                         | MODPLUG_ENABLE_SURROUND
#define DEF_CHANNELS       (MUSIC_STEREO ? 2 : 1)
#define DEF_BITS           MUSIC_MODPLUG_BITS
#define DEF_SAMPLE_RATE    MUSIC_FREQ
#define DEF_RESAMPLE       MODPLUG_RESAMPLE_LINEAR
#define DEF_XBASS_AMOUNT   50 /* 0 to 100 */
#define DEF_XBASS_RANGE    55 /* cutoff in Hz */
#define DEF_SURROUND_DEPTH 75 /* 0 to 100 */
#define DEF_SURROUND_DELAY 20 /* usually 5 to 40 ms */
#define DEF_LOOP_COUNT     -1 /* loop forever */

static ModPlug_Settings settings =
{
	DEF_FLAGS,
	DEF_CHANNELS,
	DEF_BITS,
	DEF_SAMPLE_RATE,
	DEF_RESAMPLE,
	0, 0, /* reverb settings, unused for now */
	DEF_XBASS_AMOUNT,
	DEF_XBASS_RANGE,
	DEF_SURROUND_DEPTH,
	DEF_SURROUND_DELAY,
	DEF_LOOP_COUNT
};

static ModPlugFile *loadmod(const char *filename)
{
	void *filedata;
	int len;
	ModPlugFile *ret;

	filedata = readfile(filename, &len);
	ret = ModPlug_Load(filedata, len);
	if (ret == NULL)
		abort_with_error("Cannot load music file %s", filename);
	free(filedata);
	return ret;
}

static AUDIOSTREAM *stream;
#define SOUND_BUF_SIZE 16384 /* This is in sample frames */
#define SOUND_BUF_SAMPLES (SOUND_BUF_SIZE * MUSIC_CHANNELS)
#define SOUND_BUF_BYTES (SOUND_BUF_SIZE * MUSIC_BYTES * MUSIC_CHANNELS)
static short my16bitbuffer[SOUND_BUF_SAMPLES];

#define NUM_VOICES 4 /* Ought to be enough for anybody... */

/*
 * Reopens stream, or opens it, if it has never been open.
 *
 * This is useful for killing the current contents of the buffer when
 * restarting the music or playing a new song. Without killing the
 * buffer, you'd end up playing the old music for a little bit.
 */
static void reopen_stream(void)
{
	if (stream != NULL)
	{
		stop_audio_stream(stream);
		stream = NULL;
	}

	stream = play_audio_stream(SOUND_BUF_SIZE,
		MUSIC_BITS, MUSIC_STEREO, MUSIC_FREQ, MUSIC_VOL, MUSIC_PAN);
	if (stream == NULL)
	{
		abort_with_error("Error creating audio stream: %s",
			allegro_error);
	}
}

static void loadwavs(void); /* Defined below, called from here */

void sound_init(void)
{
	assert(!soundinited);

	/* Setup on the Allegro side */
	reserve_voices(NUM_VOICES, -1);
	if (install_sound(DIGI_AUTODETECT, MIDI_NONE, NULL) != 0)
	{
		/* Just play with no sound, then */
		sounddisabled = 1;
		return;
	}

	/* Setup on the modplug side */
	ModPlug_SetSettings(&settings);
	mods[0] = loadmod(SHAREPATH "Music-1.it");
	mods[1] = loadmod(SHAREPATH "Music-2.it");
	mods[2] = loadmod(SHAREPATH "Music-0.it");

	/* Load the wav files */
	loadwavs();

	/* More setup on the Allegro side */
	reopen_stream();

	soundinited = 1;
}

/* Saturator/limiter, to amplify the music without clipping it. */

#define SAT_GRAD 0.4f
#define SAT_GRADINV (1.0f / SAT_GRAD)
#define SAMP_RANGE 32767.0f
#define SAT_RANGE (SAMP_RANGE * (1.0f / ((SAT_GRAD + 1.0f) / 2.0f)))
static int saturate(double sample)
{
	sample /= SAT_RANGE;
	if (sample > 0.0f)
	{
		if (sample > SAT_GRAD)
		{
			sample = SAT_GRAD + (sample-SAT_GRAD)
				/ (1+((sample-SAT_GRAD)/(1-SAT_GRAD))
				   * ((sample-SAT_GRAD)/(1-SAT_GRAD)));
		}
		if (sample > 1.0f)
			sample = (SAT_GRAD + 1.0f) / 2;
	}
	else
	{
		sample = -sample;
		if (sample > SAT_GRAD)
		{
			sample = SAT_GRAD + (sample-SAT_GRAD)
				/ (1+((sample-SAT_GRAD)/(1-SAT_GRAD))
				   * ((sample-SAT_GRAD)/(1-SAT_GRAD)));
		}
		if (sample > 1.0f)
			sample = (SAT_GRAD + 1.0f) / 2;
		sample = -sample;
	}
	sample *= SAT_RANGE;
	return (int)sample;
}

static void reseekmod(void)
{
	ModPlug_Seek(mods[playing], 0); /* Back to the beginning. */
}

void sound_restartmusic(void)
{
	if (sounddisabled) return;
	assert(!soundstopped);
	reseekmod();
	reopen_stream();
}

#define MODPLUG_BUF_BYTES (SOUND_BUF_BYTES * MUSIC_MODPLUG_BITS / MUSIC_BITS)

/* The output from modplug is really quiet. This is a multiplier for it. */
#define AMP 6.0f

void sound_update(void)
{
	unsigned short *p;
	int nbytes, i;

	if (soundstopped || sounddisabled) return;

	p = get_audio_stream_buffer(stream);
	if (p == NULL) return; /* Nothing to do at the moment */

	nbytes = ModPlug_Read(mods[playing], my16bitbuffer, MODPLUG_BUF_BYTES);
	if (nbytes < MODPLUG_BUF_BYTES)
	{
		/* Mod must've ended, seek to beginning. */
		reseekmod();
		nbytes += ModPlug_Read(mods[playing], &my16bitbuffer[nbytes],
			MODPLUG_BUF_BYTES-nbytes);

		/* If we STILL don't have enough, something is wrong. */
		if (nbytes < MODPLUG_BUF_BYTES)
		{
			abort_with_error("Music: Wanted %d bytes, only got %d",
				MODPLUG_BUF_BYTES, nbytes);
		}
	}

	/* Amplify the sound and stuff. */
	for (i = 0; i < SOUND_BUF_SAMPLES; i++)
	{
		p[i] = (unsigned short)
			(saturate(my16bitbuffer[i] * AMP) + 32768);
	}

	free_audio_stream_buffer(stream);
}

void sound_stop(void)
{
	if (sounddisabled) return;
	soundstopped = 1;
	voice_stop(stream->voice);
}

void sound_start(void)
{
	if (sounddisabled) return;
	soundstopped = 0;
	voice_start(stream->voice);
}

const char *samplefilenames[NUMSAMPLES] =
{
	SHAREPATH "Sshoot.wav",
	SHAREPATH "Skill.wav",
	SHAREPATH "Sdis.wav",
	SHAREPATH "Stele.wav",
	SHAREPATH "Stime.wav",
	SHAREPATH "Srefl.wav",
	SHAREPATH "Stick.wav"
};

static SAMPLE *samples[NUMSAMPLES];

static void loadwavs(void)
{
	int i;
	for (i = 0; i < NUMSAMPLES; i++)
	{
		samples[i] = load_wav(samplefilenames[i]);
		if (samples[i] == NULL)
		{
			abort_with_error("Can't load sample %s: %s",
				samplefilenames[i], allegro_error);
		}
	}
}

void sound_play(int sample)
{
	static int voiceinuse = -1;
	int newvoice;

	if (sounddisabled) return;

	newvoice = play_sample(samples[sample], 255, 128, 1000, 0);
	if (newvoice == -1 && voiceinuse != -1)
	{
		/* Try to free up the last one. */
		deallocate_voice(voiceinuse);
		newvoice = play_sample(samples[sample], 255, 128, 1000, 0);
	}
	if (newvoice != -1) voiceinuse = newvoice;
}

/* Switch to the next music file, like on a new level. */
void sound_nextmusic(void)
{
	if (sounddisabled) return;
	assert(soundinited);

	/* Use first song after title screen. */
	if (playing == NUMMUSICS) playing--;

	playing = (playing + 1) % NUMMUSICS;
	assert(mods[playing] != NULL);

	reseekmod(); /* Start the new music at the beginning. */
	reopen_stream();
}

void sound_playtitlemusic(void)
{
	if (sounddisabled) return;
	playing = NUMMUSICS;
	reseekmod();
	reopen_stream();
}

#endif
