#include <stdio.h> /* for snprintf */
#include <string.h>
#include <assert.h>
#include <allegro.h>
#include "crandom.h"
#include "game.h" /* for TICS */
#include "field.h"
#include "images.h"
#include "main.h"
#include "colors.h"
#include "sound.h"
#include "game_int.h"

/*
 * Game logic and movement.
 * We also read keys and play sound effects from here.
 */

#define TIMELIMIT 60    /* Low, so you need monster-kill bonuses badly */
#define INITMONSTERS 16 /* The number of monsters initially */

train_t train;
monster_t monsters[MAXMONSTERS];
bullet_t bullets[MAXBULLETS];
int nummonsters;
int numlivingmonsters;
int numbullets;
int timetics;

/* Used to do things like "move the foo every 'bar' tics" */
static int tics;

#define clipfunc(name, constant)		\
int name(int n)					\
{						\
	while (n < 1) n += constant;		\
	while (n > constant) n -= constant;	\
	return n;				\
}
clipfunc(clipx, FIELD_WIDTH);
clipfunc(clipy, FIELD_HEIGHT);

static void spawn_train(int y, int x)
{
	train.y = train.spawny = y;
	train.x = train.spawnx = x;
	train.hdir = 1; /* east */
	train.vdir = 1; /* south */
	train.turn = 1; /* right */
	train.crashed = 0;
	train.steps = 0;
	train.justteleported = 0;
	train.justreflected = 0;
	train.canfire = 1;
	train.gameovercount = 0;
}

static void setup_train(void)
{
	int y, x, spawned = 0;
	for (y = 1; y < FIELD_HEIGHT; y++)
		for (x = 1; x < FIELD_WIDTH; x++)
			if (field_get(y, x) == TRAIN_START)
			{
				if (!spawned) spawn_train(y, x);
				spawned = 1;
				field_set(y, x, TRAIN_REFLECT);
			}
}

static int monsterdsty(int mt)
{
	return clipy(monsters[mt].y + monsters[mt].vdir);
}
static int monsterdstx(int mt)
{
	return clipx(monsters[mt].x + monsters[mt].hdir);
}

/*
 * Tests if a monster other than exempt is in, or moving into, the spot.
 * If not, returns -1. If so, returns the monster's number.
 */
static int monsterinspot(int y, int x, int exempt)
{
	int i;
	for (i = 0; i < nummonsters; i++)
	{
		if (!monsters[i].alive || i == exempt) continue;
		if (monsters[i].x == x && monsters[i].y == y) return i;
		if (monsters[i].steps >= 0
			&& monsterdstx(i) == x && monsterdsty(i) == y)
		{
			return i;
		}
	}
	return -1;
}

static void spawn_bullet(int y, int x, int vdir, int hdir)
{
	if (numbullets == MAXBULLETS) abort_with_error("Too many bullets");
	bullets[numbullets].y = y;
	bullets[numbullets].x = x;
	bullets[numbullets].hdir = hdir;
	bullets[numbullets].vdir = vdir;
	bullets[numbullets].inuse = 1;
	bullets[numbullets].steps = 0;
	numbullets++;
}

static void remove_bullet(int num)
{
	bullets[num].inuse = 0;
	while (numbullets > 0 && bullets[numbullets-1].inuse == 0)
		numbullets--;
}

#define MONSTERKILLTIMEBONUS 12
#define MONCOLLAPSESTEPS 3

static void newlevel(void); /* Defined below, but needed here. */

static void killmonster(int mon)
{
	monsters[mon].alive = -MONCOLLAPSESTEPS; /* hack: collapse */
	numlivingmonsters--;
	sound_play(SMP_KILL);
	timetics += MONSTERKILLTIMEBONUS * TICS;
}

static int possiblyremovebullet(int bt)
{
	int mon;
	if (field_get(bullets[bt].y, bullets[bt].x) != EMPTY)
	{
		remove_bullet(bt);
		sound_play(SMP_DISINTEGRATE);
		return 1;
	}
	if ((mon = monsterinspot(bullets[bt].y, bullets[bt].x, -1)) != -1)
	{
		remove_bullet(bt);
		if (monsters[mon].alive > 0) killmonster(mon);
		return 1;
	}
	return 0;
}

/*
 * Return the size of the open area at (y, x).
 * If the size is max or more, we don't care, so just say max.
 */
static int openingsize(int y, int x, int max)
{
	static int visited[FIELD_HEIGHT][FIELD_WIDTH];
	static int inside = 0;
	int outer = 0;
	int count;
	int i, j;

	y = clipy(y);
	x = clipx(x);
	if (max == 0 || field_get(y, x) != EMPTY)
		return 0;

	if (!inside)
	{
		outer = 1;
		inside = 1;
		for (i = 0; i < FIELD_HEIGHT; i++)
		for (j = 0; j < FIELD_WIDTH; j++)
		{
			visited[i][j] = 0;
		}
	}

	if (visited[y][x])
		return 0;
	visited[y][x] = 1;

	count = 1; /* This cell counts. */
	count += openingsize(y-1, x, max-count);
	count += openingsize(y+1, x, max-count);
	count += openingsize(y, x-1, max-count);
	count += openingsize(y, x+1, max-count);

	if (outer)
		inside = 0;

	return count;
}

static void spawn_monster(int y, int x, int vdir, int hdir)
{
	if (nummonsters == MAXMONSTERS) abort_with_error("Too many monsters");
	monsters[nummonsters].y = y;
	monsters[nummonsters].x = x;
	monsters[nummonsters].hdir = hdir;
	monsters[nummonsters].vdir = vdir;
	monsters[nummonsters].alive = 1;
	monsters[nummonsters].steps = -1;
	nummonsters++;
	numlivingmonsters++;
}

/*
 * Initially, the smallest number of spaces we want each
 * spawned monster to have access to.
 */
#define INIT_MINSIZE 3

/* 1 in this many chance of reducing minsize when it is not met. */
#define MINSIZE_REDUCE 8

/* Spawn num monsters in random empty spots. */
static void setup_monsters(int num)
{
	int y, x;
	int minsize = INIT_MINSIZE;
	nummonsters = 0;
	numlivingmonsters = 0;
	while (nummonsters < num)
	{
		y = rnd(FIELD_HEIGHT) + 1;
		x = rnd(FIELD_WIDTH)  + 1;
		if (field_get(y, x) == EMPTY
			&& monsterinspot(y, x, -1) == -1)
		{
			int movedir, hdir = NONE, vdir = NONE;

			/* Is this not enough open space? */
			if (openingsize(y, x, minsize) < minsize)
			{
				/* Consider being less choosy. */
				if (rnd(MINSIZE_REDUCE) == 0)
					minsize--;

				continue;
			}

			movedir = rnd(4);
			if (movedir == 0) hdir = LEFT;
			else if (movedir == 1) hdir = RIGHT;
			else if (movedir == 2) vdir = UP;
			else if (movedir == 3) vdir = DOWN;
			spawn_monster(y, x, vdir, hdir);
		}
	}
}

int gameover = 0;

/* Restart a level after having run out of time. */
static void restartlevel(void)
{
	spawn_train(train.spawny, train.spawnx);
	setup_monsters(INITMONSTERS);
	numbullets = 0;
	tics = 0;
	timetics = TIMELIMIT * TICS;
	sound_start();
	sound_restartmusic();
}

int titlescreen = 1;

static void newlevel(void)
{
	field_load();
	setup_train();
	setup_monsters(INITMONSTERS);
	numbullets = 0;
	tics = 0;
	timetics = TIMELIMIT * TICS;
	titlescreen = 0;

	setup_colors();
	sound_nextmusic();
}

void game_init(void)
{
	load_images();
}

/************************************************************/

#define TICS_PER_STEP 3 /* Train makes a "step" once every this many tics */
#define TICS_PER_MONSTER_STEP 6
#define TICS_PER_BULLET_STEP 3
#define MONSTER_WAIT 4      /* Monster waits this many steps out */
#define MONSTER_TURN_WAIT 2 /* Monster waits this many steps after turning */
#define CRASH do { train.crashed = 1; return; } while (0)
enum { MOV_HORIZ, MOV_VERT, MOV_ASCEND, MOV_DESCEND };

/*
 * Returns nonzero if the spot admits the desired kind of
 * movement.
 *
 * Note that Y-junctions do not admit any of these kinds of
 * movement in general. Furthermore, teleporters and
 * reflectors are not considered to admit any types of
 * movement.
 */
static int admits(spot_t spot, int mov)
{
	if (spot == RAIL_HORIZ)
		return mov == MOV_HORIZ;
	if (spot == RAIL_VERT)
		return mov == MOV_VERT;
	if (spot == RAIL_ASCEND)
		return mov == MOV_ASCEND;
	if (spot == RAIL_DESCEND)
		return mov == MOV_DESCEND;
	if (spot == JUNC_UNIV) return 1;
	if (spot == JUNC_HV)
		return mov == MOV_VERT || mov == MOV_HORIZ;
	if (spot == JUNC_DIAG)
		return mov == MOV_ASCEND || mov == MOV_DESCEND;
	return 0;
}

static int movtype(int vdir, int hdir)
{
	if (vdir == NONE) return MOV_HORIZ;
	if (hdir == NONE) return MOV_VERT;
	if (vdir == hdir) return MOV_DESCEND;
	return MOV_ASCEND;
}

static void doturn(int *vdir, int *hdir, int turn)
{
	if (turn == TURN_LEFT)
	{
		if (*vdir == NONE && *hdir == LEFT) *vdir = DOWN;
		else if (*vdir == DOWN && *hdir == LEFT) *hdir = NONE;
		else if (*vdir == DOWN && *hdir == NONE) *hdir = RIGHT;
		else if (*vdir == DOWN && *hdir == RIGHT) *vdir = NONE;
		else if (*vdir == NONE && *hdir == RIGHT) *vdir = UP;
		else if (*vdir == UP && *hdir == RIGHT) *hdir = NONE;
		else if (*vdir == UP && *hdir == NONE) *hdir = LEFT;
		else if (*vdir == UP && *hdir == LEFT) *vdir = NONE;
	}
	else
	{
		if (*vdir == NONE && *hdir == LEFT) *vdir = UP;
		else if (*vdir == UP && *hdir == LEFT) *hdir = NONE;
		else if (*vdir == UP && *hdir == NONE) *hdir = RIGHT;
		else if (*vdir == UP && *hdir == RIGHT) *vdir = NONE;
		else if (*vdir == NONE && *hdir == RIGHT) *vdir = DOWN;
		else if (*vdir == DOWN && *hdir == RIGHT) *hdir = NONE;
		else if (*vdir == DOWN && *hdir == NONE) *hdir = LEFT;
		else if (*vdir == DOWN && *hdir == LEFT) *vdir = NONE;
	}
}

/*
 * Look for a secondary connection for the train, supposing it
 * turns in the direction turn. Return 1 if one is found.
 *
 * Note: Y-junctions, teleporters, and reflectors are never
 * used for secondary connections.
 */
static int secondary(int turn)
{
	int dstx, dsty;
	int vdir, hdir, mov;

	/*
	 * First we decide what the train's hdir and vdir
	 * will be if it makes this turn.
	 */
	vdir = train.vdir;
	hdir = train.hdir;
	doturn(&vdir, &hdir, turn);

	/*
	 * Then we calculate its type of movement after the
	 * turn.
	 */
	mov = movtype(vdir, hdir);

	/* Finally its destination point. */
	dstx = clipx(train.x + hdir);
	dsty = clipy(train.y + vdir);

	/*
	 * The turn can only be made if the destination
	 * point admits that particular type of movement.
	 */
	return admits(field_get(dsty, dstx), mov);
}

static void orienttrain(void);

/* Check the train's destination -- where it'll be next move. */
static void checkdest(void)
{
	int dstx, dsty;
	int hdir = train.hdir, vdir = train.vdir;
	spot_t spot;

	dstx = clipx(train.x + hdir);
	dsty = clipy(train.y + vdir);

	spot = field_get(dsty, dstx);
	if (spot == TRAIN_REFLECT && !train.justreflected)
	{
		train.vdir *= -1;
		train.hdir *= -1;
		train.justreflected = 1;
		sound_play(SMP_REFLECT);
		orienttrain();
	}
	else if (spot != TRAIN_TELE
		&& !(spot == RAIL_HORIZ && hdir != NONE)
		&& !(spot == RAIL_VERT && vdir != NONE)
		&& !(spot == RAIL_ASCEND &&
			(hdir == NONE || vdir == NONE
			|| (vdir == UP && hdir == RIGHT)
			|| (vdir == DOWN && hdir == LEFT)))
		&& !(spot == RAIL_DESCEND &&
			(hdir == NONE || vdir == NONE
			|| (vdir == UP && hdir == LEFT)
			|| (vdir == DOWN && hdir == RIGHT)))
		&& !(spot == JUNC_UNIV)
		&& !(spot == JUNC_HV)
		&& !(spot == JUNC_DIAG)
		&& !(spot == JUNC_Y_UP &&
			((vdir == DOWN && hdir != NONE)
			|| (vdir == UP && hdir == NONE)))
		&& !(spot == JUNC_Y_DOWN &&
			((vdir == UP && hdir != NONE)
			|| (vdir == DOWN && hdir == NONE)))
		&& !(spot == JUNC_Y_LEFT &&
			((hdir == RIGHT && vdir != NONE)
			|| (hdir == LEFT && vdir == NONE)))
		&& !(spot == JUNC_Y_RIGHT &&
			((hdir == LEFT && vdir != NONE)
			|| (hdir == RIGHT && vdir == NONE))))
	{
		int found = -1;
		/* No primary connector, look for a secondary one. */
		if (train.turn == TURN_LEFT)
		{
			if (secondary(TURN_LEFT)) found = TURN_LEFT;
			else if (secondary(TURN_RIGHT)) found = TURN_RIGHT;
		}
		else
		{
			if (secondary(TURN_RIGHT)) found = TURN_RIGHT;
			else if (secondary(TURN_LEFT)) found = TURN_LEFT;
		}
		if (found != -1)
			doturn(&train.vdir, &train.hdir, found);
	}
}

static void teleporttrain(void)
{
	int y, x;

	y = train.y;
	x = train.x;

	/* If there is only one teleporter, we'll come back to it. */
	for (;;)
	{
		x = clipx(x + 1);
		if (train.x == x) y = clipy(y - 1);
		if (field_get(y, x) == TRAIN_TELE) break;
	}

	/* If we moved at all, play the teleport sound. */
	if (train.y != y || train.x != x)
		sound_play(SMP_TELEPORT);

	train.y = y;
	train.x = x;
}

/* Orient the train, decide where to next. Train may end up crashing. */
static void orienttrain(void)
{
	int vdir, hdir, turn;

	hdir = train.hdir;
	vdir = train.vdir;
	turn = train.turn;

	switch (field_get(train.y, train.x))
	{
		case RAIL_HORIZ: /* - */
		{
			vdir = NONE;
			if (hdir == NONE) CRASH;
			break;
		}
		case RAIL_VERT: /* | */
		{
			hdir = NONE;
			if (vdir == NONE) CRASH;
			break;
		}
		case RAIL_ASCEND: /* / */
		{
			if (hdir == RIGHT || vdir == UP)
			{
				if (hdir == LEFT || vdir == DOWN) CRASH;
				hdir = RIGHT;
				vdir = UP;
			}
			else
			{
				hdir = LEFT;
				vdir = DOWN;
			}
			break;
		}
		case RAIL_DESCEND: /* \ */
		{
			if (hdir == LEFT || vdir == UP)
			{
				if (hdir == RIGHT || vdir == DOWN) CRASH;
				hdir = LEFT;
				vdir = UP;
			}
			else
			{
				hdir = RIGHT;
				vdir = DOWN;
			}
			break;
		}
		case JUNC_UNIV: /* * */
			break; /* Universal junction, just keep on going */
		case JUNC_HV: /* + */
		{
			/*
			 * This junction can be entered from any direction.
			 * The turn value is taken into account here.
			 */
			if (vdir != NONE && hdir != NONE)
			{
				if (vdir == UP && hdir == LEFT
					&& turn == TURN_LEFT) vdir = NONE;
				else if (vdir == UP && hdir == LEFT
					&& turn == TURN_RIGHT) hdir = NONE;
				else if (vdir == UP && hdir == RIGHT
					&& turn == TURN_LEFT) hdir = NONE;
				else if (vdir == UP && hdir == RIGHT
					&& turn == TURN_RIGHT) vdir = NONE;
				else if (vdir == DOWN && hdir == LEFT
					&& turn == TURN_LEFT) hdir = NONE;
				else if (vdir == DOWN && hdir == LEFT
					&& turn == TURN_RIGHT) vdir = NONE;
				else if (vdir == DOWN && hdir == RIGHT
					&& turn == TURN_LEFT) vdir = NONE;
				else if (vdir == DOWN && hdir == RIGHT
					&& turn == TURN_RIGHT) hdir = NONE;
			}
			break;
		}
		case JUNC_DIAG: /* x */
		{
			/*
			 * This junction can be entered from any direction.
			 * The turn value is taken into account here.
			 */
			if (vdir == NONE || hdir == NONE)
			{
				if (vdir == UP && turn == TURN_LEFT)
					hdir = LEFT;
				else if (vdir == UP && turn == TURN_RIGHT)
					hdir = RIGHT;
				else if (vdir == DOWN && turn == TURN_LEFT)
					hdir = RIGHT;
				else if (vdir == DOWN && turn == TURN_RIGHT)
					hdir = LEFT;
				else if (hdir == LEFT && turn == TURN_LEFT)
					vdir = DOWN;
				else if (hdir == LEFT && turn == TURN_RIGHT)
					vdir = UP;
				else if (hdir == RIGHT && turn == TURN_LEFT)
					vdir = UP;
				else if (hdir == RIGHT && turn == TURN_RIGHT)
					vdir = DOWN;
			}
			break;
		}
		case JUNC_Y_UP: /* v */
		{
			if (vdir == DOWN && hdir != NONE) hdir = NONE;
			else if (!(vdir == UP && hdir == NONE)) CRASH;
			else if (turn == TURN_LEFT) hdir = LEFT;
			else hdir = RIGHT;
			break;
		}
		case JUNC_Y_DOWN: /* ^ */
		{
			if (vdir == UP && hdir != NONE) hdir = NONE;
			else if (!(vdir == DOWN && hdir == NONE)) CRASH;
			else if (turn == TURN_LEFT) hdir = RIGHT;
			else hdir = LEFT;
			break;
		}
		case JUNC_Y_LEFT: /* > */
		{
			if (hdir == RIGHT && vdir != NONE) vdir = NONE;
			else if (!(hdir == LEFT && vdir == NONE)) CRASH;
			else if (turn == TURN_LEFT) vdir = DOWN;
			else vdir = UP;
			break;
		}
		case JUNC_Y_RIGHT: /* < */
		{
			if (hdir == LEFT && vdir != NONE) vdir = NONE;
			else if (!(hdir == RIGHT && vdir == NONE)) CRASH;
			else if (turn == TURN_LEFT) vdir = UP;
			else vdir = DOWN;
			break;
		}
		case TRAIN_START: /* $ */
		case EMPTY:
		{
			CRASH;
			break;
		}
		case TRAIN_TELE: /* # */
		{
			/* Like an HV junction (+). Copied and pasted: */
			if (vdir != NONE && hdir != NONE)
			{
				if (vdir == UP && hdir == LEFT
					&& turn == TURN_LEFT) vdir = NONE;
				else if (vdir == UP && hdir == LEFT
					&& turn == TURN_RIGHT) hdir = NONE;
				else if (vdir == UP && hdir == RIGHT
					&& turn == TURN_LEFT) hdir = NONE;
				else if (vdir == UP && hdir == RIGHT
					&& turn == TURN_RIGHT) vdir = NONE;
				else if (vdir == DOWN && hdir == LEFT
					&& turn == TURN_LEFT) hdir = NONE;
				else if (vdir == DOWN && hdir == LEFT
					&& turn == TURN_RIGHT) vdir = NONE;
				else if (vdir == DOWN && hdir == RIGHT
					&& turn == TURN_LEFT) vdir = NONE;
				else if (vdir == DOWN && hdir == RIGHT
					&& turn == TURN_RIGHT) hdir = NONE;
			}

			/* Teleport, unless that's how you got here. */
			if (!train.justteleported)
			{
				teleporttrain();
				train.justteleported = 1;
				orienttrain();
				return;
			}
			break;
		}
		case TRAIN_REFLECT: /* @ */
			assert(0);
	}

	train.hdir = hdir;
	train.vdir = vdir;
	train.turn = turn;
	train.justteleported = 0;

	checkdest();

	train.justreflected = 0;
}

enum { NOBULLET, BUL_LEFT, BUL_RIGHT, BUL_DOWN, BUL_UP };

static void firebullet(int vdir, int hdir)
{
	int x, y;
	if (!train.canfire) return;
	y = clipy(train.y + vdir
		+ (train.steps >= STEPS_PER_MOVE/2 ? train.vdir : 0));
	x = clipx(train.x + hdir
		+ (train.steps >= STEPS_PER_MOVE/2 ? train.hdir : 0));
	spawn_bullet(y, x, vdir, hdir);
	if (!possiblyremovebullet(numbullets-1))
		sound_play(SMP_SHOOT);
	train.canfire = 0;
}

static void throwbullet(int direction)
{
	int bvdir = NONE, bhdir = NONE;
	switch (direction)
	{
		case BUL_LEFT: bhdir = LEFT; break;
		case BUL_RIGHT: bhdir = RIGHT; break;
		case BUL_UP: bvdir = UP; break;
		case BUL_DOWN: bvdir = DOWN; break;
	}
	firebullet(bvdir, bhdir);
}

static void movetrain(void)
{
	if (train.gameovercount)
		return;

	/*
	 * Playability hack: The train should never crash,
	 * but some level segments might have mistakes. In
	 * case we missed one, if the train crashes, respawn
	 * it in its starting location.
	 */
	if (train.crashed)
	{
#ifndef NO_LOG
		FILE *log = fopen("log.txt", "a");
		if (log != NULL)
		{
			fprintf(log, "crashed at y: %d, x: %d, "
				"vdir: %d, hdir: %d\n",
				train.y, train.x, train.vdir, train.hdir);
			fclose(log);
		}
#endif
		train.x = train.spawnx;
		train.y = train.spawny;
		train.vdir = DOWN;
		train.hdir = RIGHT;
		train.crashed = 0;
		train.steps = 0;
		train.canfire = 1;
	}

	if (++train.steps == STEPS_PER_MOVE)
	{
		train.steps = 0;
		train.x = clipx(train.x + train.hdir);
		train.y = clipy(train.y + train.vdir);
		orienttrain();
	}
	else if (train.steps == STEPS_PER_MOVE/2)
		train.canfire = 1;
}

static void explodetrain(void)
{
	if (train.crashed)
	{
		if (train.vdir || train.hdir)
		{
			train.vdir = train.hdir = NONE;
			train.turn = TURN_LEFT;
		}
		else if (train.turn == TURN_LEFT)
			train.turn = TURN_RIGHT;
	}
}

static void turnmonster(int mt)
{
	int newdir, oldhdir, oldvdir;

	oldhdir = monsters[mt].hdir;
	oldvdir = monsters[mt].vdir;

	do
	{
		newdir = rnd(4);
		if (newdir&1)
		{
			monsters[mt].vdir = NONE;
			monsters[mt].hdir = (newdir&2) ? LEFT : RIGHT;
		}
		else
		{
			monsters[mt].hdir = NONE;
			monsters[mt].vdir = (newdir&2) ? UP : DOWN;
		}
	} while (monsters[mt].hdir == oldhdir
		&& monsters[mt].vdir == oldvdir);

	monsters[mt].steps = -MONSTER_TURN_WAIT;
}

void checkmonstersanity(int mt)
{
	assert(monsters[mt].hdir >= -1 && monsters[mt].hdir <= 1);
	assert(monsters[mt].vdir >= -1 && monsters[mt].vdir <= 1);
	assert(monsters[mt].steps >= -MONSTER_WAIT);
	assert(monsters[mt].steps < STEPS_PER_MOVE);
	assert(monsters[mt].x == clipx(monsters[mt].x));
	assert(monsters[mt].y == clipx(monsters[mt].y));
}

void checkbulletsanity(int bt)
{
	assert(bullets[bt].hdir >= -1 && bullets[bt].hdir <= 1);
	assert(bullets[bt].vdir >= -1 && bullets[bt].vdir <= 1);
	assert(bullets[bt].steps >= 0);
	assert(bullets[bt].steps < STEPS_PER_MOVE);
	assert(bullets[bt].x == clipx(bullets[bt].x));
	assert(bullets[bt].y == clipx(bullets[bt].y));
}

static void movemonster(int mt)
{
	checkmonstersanity(mt);
	if (++monsters[mt].steps == STEPS_PER_MOVE)
	{
		monsters[mt].steps = -MONSTER_WAIT;
		monsters[mt].x = clipx(monsters[mt].x + monsters[mt].hdir);
		monsters[mt].y = clipy(monsters[mt].y + monsters[mt].vdir);
	}
	else if (monsters[mt].steps == 0)
	{
		int mdsty = monsterdsty(mt), mdstx = monsterdstx(mt);
		if (rnd(4) == 0
			|| field_get(mdsty, mdstx) != EMPTY
			|| monsterinspot(mdsty, mdstx, mt) != -1)
		{
			turnmonster(mt);
		}
	}
	checkmonstersanity(mt);
}

static void movemonsters(void)
{
	int i;
	for (i = 0; i < nummonsters; i++)
	{
		if (monsters[i].alive < 0)
		{
			monsters[i].alive++;
			if (monsters[i].alive == 0 && numlivingmonsters == 0)
				newlevel();
		}
		else if (monsters[i].alive)
			movemonster(i);
	}
}

static void movebullet(int bt)
{
	checkbulletsanity(bt);
	if (++bullets[bt].steps == STEPS_PER_MOVE)
	{
		bullets[bt].steps = 0;
		bullets[bt].x = clipx(bullets[bt].x + bullets[bt].hdir);
		bullets[bt].y = clipy(bullets[bt].y + bullets[bt].vdir);
		if (possiblyremovebullet(bt))
			return;
	}
	checkbulletsanity(bt);
}

static void movebullets(void)
{
	int i;
	for (i = 0; i < numbullets; i++)
		if (bullets[i].inuse)
			movebullet(i);
}

/************************************************************/

static void handlekey(int keynum)
{
	if (titlescreen)
	{
		if (keynum == KEY_ESC) gameover = 1;
		else if (keynum == KEY_SPACE) newlevel();
		else if (keynum == KEY_ENTER && key_shifts & KB_ALT_FLAG)
			toggle_fullscreen();
		return;
	}

	switch (keynum)
	{
		case KEY_LEFT:
			if (!train.crashed) train.turn = TURN_LEFT;
			break;
		case KEY_RIGHT:
			if (!train.crashed) train.turn = TURN_RIGHT;
			break;
		case KEY_6_PAD:
		case KEY_D: /* WASD */
		case KEY_K: /* UHJK */
			if (!train.crashed) throwbullet(BUL_RIGHT);
			break;
		case KEY_4_PAD:
		case KEY_A: /* WASD */
		case KEY_H: /* UHJK */
			if (!train.crashed) throwbullet(BUL_LEFT);
			break;
		case KEY_2_PAD:
		case KEY_5_PAD:
		case KEY_S: /* WASD */
		case KEY_J: /* UHJK */
			if (!train.crashed) throwbullet(BUL_DOWN);
			break;
		case KEY_8_PAD:
		case KEY_W: /* WASD */
		case KEY_U: /* UHJK */
			if (!train.crashed) throwbullet(BUL_UP);
			break;
		case KEY_SPACE:
			/*
			 * Pausing takes effect once logic completes.
			 * If the TIME OVER banner is up, don't pause,
			 * instead restart.
			 */
			if (train.gameovercount) restartlevel();
			else pause_game();
			break;
		case KEY_ESC:
			titlescreen = 1;
			reset_colors();
			sound_playtitlemusic();
			break;
		case KEY_ENTER:
			if (key_shifts & KB_ALT_FLAG)
				toggle_fullscreen();
			break;
		default:
			/* Whatever, man. I can't heeear you! */
			break;
	}
}

static void readkeys(void)
{
	while (keypressed())
		handlekey(readkey()>>8);
}

/* How long before you restart, after running out of time. */
#define GAMEOVERSECS 10

int game_run_logic_frame(void)
{
	int changed = 0; /* did anything actually change here? */

	if (titlescreen)
	{
		readkeys();
		return 0;
	}
	readkeys();
	tics++;
	if (tics % TICS_PER_STEP == 0)
	{
		movetrain();
		explodetrain();
		changed = 1;
	}
	if (tics % TICS_PER_MONSTER_STEP == 0)
	{
		movemonsters();
		changed = 1;
	}
	if (tics % TICS_PER_BULLET_STEP == 0)
	{
		movebullets();
		changed = 1;
	}
	if (tics == TICS_PER_STEP * TICS_PER_MONSTER_STEP
		* TICS_PER_BULLET_STEP) tics = 0;

	if (train.gameovercount)
	{
		train.gameovercount--;
		if (train.gameovercount == 0) restartlevel();
	}
	else if (--timetics == 0)
	{
		/* Bummer, man. You ran out of time. */
		sound_play(SMP_TIMEUP);
		train.crashed = 1;
		explodetrain();
		train.gameovercount = GAMEOVERSECS * TICS;

		/*
		 * This function name is misleading. It actually just
		 * stops the music, not the time-up sound effect.
		 */
		sound_stop();
	}
	else if (timetics <= 10*TICS && timetics % TICS == 0)
		sound_play(SMP_TICK); /* About out of time. */

	/* If we're returning to the title screen, it needs to be drawn. */
	if (titlescreen) changed = 1;

	return changed;
}
