/*
 * Timer routines used in main.c, for busy-wait loops.
 */

#include <stddef.h>
#ifndef _WIN32
#include <sys/types.h>
#include <sys/time.h>
#endif
#include "ctimer.h"

#if defined(_WIN32) || defined(__BEOS__)
typedef unsigned long long u_int64_t;
#endif

static int counter[MAXTIMERS];

static u_int64_t lasttime[MAXTIMERS], udelay[MAXTIMERS];
static int nexttimer = 0;

#ifdef _WIN32

/*
 * Stuff from winbase.h, but to avoid including tons of unneeded
 * Windows garbage too, it is included here.
 */
typedef struct _FILETIME
{
	unsigned long dwLowDateTime;
	unsigned long dwHighDateTime;
} FILETIME;
void __stdcall GetSystemTimeAsFileTime(FILETIME *);

/* Our own struct timeval, hooray. */
struct timeval
{
	long tv_sec;
	long tv_usec;
};

typedef union
{
	long long ns100; /* time since Jan. 1, 1601 in 100ns units */
	FILETIME ft;
} w32_ftv;

#define _W32_FT_OFFSET 116444736000000000LL

/* Fake gettimeofday for Windows. */
static void gettimeofday(struct timeval *tv, void *ignored)
{
	w32_ftv _now;
	GetSystemTimeAsFileTime(&(_now.ft));
	tv->tv_usec = (long)((_now.ns100 / 10LL) % 1000000LL);
	tv->tv_sec  = (long)((_now.ns100 - _W32_FT_OFFSET)/10000000LL);
	(void)ignored;
}
#endif

static u_int64_t get_usecs(void)
{
	struct timeval tv;
	(void)gettimeofday(&tv, NULL);
	return ((u_int64_t)tv.tv_sec * 1000000UL) + tv.tv_usec;
}

static void adjust_delay(int timer, u_int64_t newdelay)
{
	u_int64_t _lasttime = lasttime[timer], _udelay = udelay[timer];
	if (_lasttime > 0 && _udelay > 0)
	{
		/* Adjust as though change occurred at last tick. */
		u_int64_t curtime = get_usecs(), lastticktime;
		int tickspassed = (int)((curtime - _lasttime) / _udelay);
		lastticktime = curtime - (curtime - _lasttime) % _udelay;
		lasttime[timer] = lastticktime - newdelay*tickspassed;
	}
	udelay[timer] = newdelay;
}

int get_timer(void)
{
	if (nexttimer == MAXTIMERS)
		return -1;
	restart_timer(nexttimer);
	return nexttimer++;
}

void restart_timer(int timer)
{
	lasttime[timer] = get_usecs();
	counter[timer] = 0;
}

void set_delay(int timer, int ticks)
{
	adjust_delay(timer, 1000000UL / ticks);
}

static int get_ticks(int timer)
{
	u_int64_t curtime = get_usecs();
	int tickspassed = (int)((curtime - lasttime[timer]) / udelay[timer]);

	/* Set lasttime to time of last actual tick... */
	if (tickspassed > 0)
	{
		lasttime[timer] = curtime
			- (curtime - lasttime[timer]) % udelay[timer];
	}
	return tickspassed;
}

void update_counter(int timer)
{
	counter[timer] += get_ticks(timer);
}

int read_counter(int timer)
{
	int ret = counter[timer];
	counter[timer] = 0;
	return ret;
}

/*
 * Provide some sort of a time, just so rand_init() in crandom.c can
 * initialize the random generator with it.
 */
int get_rough_time(void)
{
	return (int)(get_usecs() & 0x7fffffff);
}
