/*
 * Self-righteousness has killed more people than smoking.
 *    -- John McCarthy
 */

/*
 * The glue that ties everything together. Handles page-flipping or
 * double-buffering, as well as timing and calling the game_* routines
 * as needed.
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <assert.h>
#include <allegro.h>
#include "main.h"
#include "game.h"
#include "ctimer.h"
#include "sound.h"

static BITMAP *pages[2]; /* For page-flipping */
static int     target;   /* Page being currently drawn to */
static int     newframe; /* Has game logic run since last redraw? */
static int     timer;    /* Timer we're counting on */
static int     paused;   /* Is the game paused? */

static enum
{
	NOT_ESTABLISHED,
	VIDEO_BITMAPS,
	SUB_BITMAP,
	DOUBLE_BUFFER
} draw_method = NOT_ESTABLISHED;

/************************************************************/

static void flip_screen(void)
{
	assert(draw_method == VIDEO_BITMAPS || draw_method == SUB_BITMAP);
	show_video_bitmap(pages[target]);
	target = !target;
}

void abort_with_error(const char *msg, ...)
{
	char str[1100];
	va_list argptr;

	va_start(argptr, msg);
	vsnprintf(str, sizeof str, msg, argptr);
	va_end(argptr);

	if (screen != NULL)
		set_gfx_mode(GFX_TEXT, 0, 0, 0, 0);
	allegro_message(str);
	exit(1);
}

static void update_logic(void)
{
	int nticks;

	for (;;)
	{
		update_counter(timer);
		nticks = read_counter(timer);

		if (nticks > 0)
		{
			while (nticks-- > 0)
				if (game_run_logic_frame())
					newframe = 1;
		}
		else break;
	}
}

void pause_game(void)
{
	paused = 1;
}

/* Wait for someone to press space, thereby unpausing the game. */
static void waitforunpause(void)
{
	int k;

	sound_stop();
	while ((k = readkey()>>8) != KEY_SPACE && k != KEY_ESC)
	{
		if (k == KEY_ENTER && key_shifts & KB_ALT_FLAG)
			toggle_fullscreen();
		rest(0);
	}
	clear_keybuf(); /* don't pause it all over again */
	paused = 0;
	sound_start();
}

static void redraw(void)
{
	assert(draw_method != NOT_ESTABLISHED);
	game_draw_frame(pages[target]);
	if (paused)
	{
		/* Display a PAUSED banner. */
		textout_centre_ex(pages[target], font, "   P A U S E D   ",
			SCRWIDTH/2, SCRHEIGHT/2, 14 /* yellow */, -1);
	}
	if (draw_method == VIDEO_BITMAPS || draw_method == SUB_BITMAP)
		flip_screen();
	else if (draw_method == DOUBLE_BUFFER)
	{
		vsync();
		blit(pages[target], screen, 0, 0, 0, 0, SCRWIDTH, SCRHEIGHT);
	}
	newframe = 0;
}

/* Free bitmaps used by the current drawing method. */
static void free_bitmaps(void)
{
	if (draw_method != NOT_ESTABLISHED) destroy_bitmap(pages[0]);
	if (draw_method == VIDEO_BITMAPS || draw_method == SUB_BITMAP)
		destroy_bitmap(pages[1]);
	draw_method = NOT_ESTABLISHED;
}

static void setup_graphics(int card)
{
	free_bitmaps();
	target = 0;

	/* Try the "sub-bitmap" method. */
	if (set_gfx_mode(card, SCRWIDTH, SCRHEIGHT, SCRWIDTH, 2*SCRHEIGHT) == 0)
	{
		pages[0] = create_sub_bitmap(screen, 0, 0, SCRWIDTH, SCRHEIGHT);
		pages[1] = create_sub_bitmap(screen, 0, SCRHEIGHT,
			SCRWIDTH, SCRHEIGHT);

		if (pages[0] != NULL && pages[1] != NULL)
		{
			target = 1;
			draw_method = SUB_BITMAP;
			return;
		}

		/* Failed, so free memory if part of it worked. */
		if (pages[0] != NULL) destroy_bitmap(pages[0]);
		if (pages[1] != NULL) destroy_bitmap(pages[1]);
	}

	/* If that didn't work, try the "video bitmap" method. */
	if (set_gfx_mode(card, SCRWIDTH, SCRHEIGHT, 0, 0) == 0)
	{
		pages[0] = create_video_bitmap(SCRWIDTH, SCRHEIGHT);
		pages[1] = create_video_bitmap(SCRWIDTH, SCRHEIGHT);

		if (pages[0] != NULL && pages[1] != NULL)
		{
			target = 1;
			draw_method = VIDEO_BITMAPS;
			return;
		}

		/* Failed, so free memory if part of it worked. */
		if (pages[0] != NULL) destroy_bitmap(pages[0]);
		if (pages[1] != NULL) destroy_bitmap(pages[1]);
	}
	else
		abort_with_error("Cannot set video mode: %s", allegro_error);

	/* As a last resort, try the "double buffering" method. */
	pages[0] = create_bitmap(SCRWIDTH, SCRHEIGHT);
	draw_method = DOUBLE_BUFFER;

	if (pages[0] == NULL)
		abort_with_error("Cannot double buffer: %s", allegro_error);
}

void toggle_fullscreen(void)
{
	int newmode;
	if (is_windowed_mode())
		newmode = GFX_AUTODETECT_FULLSCREEN;
	else
		newmode = GFX_AUTODETECT_WINDOWED;
	setup_graphics(newmode);
	redraw();
}

/************************************************************/

int main(void)
{
	if (allegro_init() != 0)
		exit(1);

	set_window_title("Rail Blaster");

	setup_graphics(GFX_AUTODETECT);

	if (install_keyboard() != 0)
	{
		abort_with_error("Cannot install keyboard driver: %s",
			allegro_error);
	}

	/* If the game is switched away from, make sure we pause it. */
	set_display_switch_callback(SWITCH_OUT, pause_game);
	if (set_display_switch_mode(SWITCH_BACKGROUND) == -1)
		(void)set_display_switch_mode(SWITCH_BACKAMNESIA);

	timer = get_timer();
	if (timer == -1)
		abort_with_error("Cannot get a timer");

	game_init();
	sound_init();

	set_delay(timer, TICS);
	restart_timer(timer);
	redraw();

	for (;;)
	{
		update_logic();
		sound_update();
		if (newframe || paused) redraw();
		if (gameover) break;
		if (paused)
		{
			waitforunpause();
			restart_timer(timer);
			redraw();
		}
	}

	return 0;
} END_OF_MAIN()
