#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <unistd.h> /* for access() */
#include "crandom.h"
#include "field.h"
#include "config.h"
#include "main.h"

/*
 * A simple abstraction layer for the playfield, with sanity checks.
 * Each tile is called a "spot."
 *
 * This is also where we assemble levels out of segments. The other
 * code files have no knowledge of segments; they just know that
 * somehow field_load() creates a new level.
 */

static spot_t field[FIELD_HEIGHT][FIELD_WIDTH];

spot_t field_get(int y, int x)
{
	assert(y >= 1);
	assert(x >= 1);
	assert(y <= FIELD_HEIGHT);
	assert(x <= FIELD_WIDTH);
	return field[y-1][x-1];
}

void field_set(int y, int x, spot_t spot)
{
	assert(y >= 1);
	assert(x >= 1);
	assert(y <= FIELD_HEIGHT);
	assert(x <= FIELD_WIDTH);
	field[y-1][x-1] = spot;
}

const char *symbols = "-|/\\*+xv^><$#@.";

#define SEG_WIDTH  16
#define SEG_HEIGHT 16

#if SEG_WIDTH > FIELD_WIDTH || SEG_HEIGHT > FIELD_HEIGHT \
    || FIELD_WIDTH % SEG_WIDTH > 0 || FIELD_HEIGHT % SEG_HEIGHT > 0
#error "Please choose a sane value for SEG_WIDTH and SEG_HEIGHT."
#endif

static void loadsegment(const char *filename, int ay, int ax)
{
	int y = 0, x = 0;
	int c;

	FILE *fp = fopen(filename, "r");
	if (fp == NULL) abort_with_error("Can't open file %s", filename);
	while ((c = getc(fp)) != EOF)
	{
		char *place = strchr(symbols, c);
		if (place == NULL) continue;
		field_set(y+ay, x+ax, place - symbols);
		x++;
		if (x == SEG_WIDTH)
		{
			x = 0;
			y++;
			if (y == SEG_HEIGHT)
				break;
		}
	}
	if (y < SEG_HEIGHT)
		abort_with_error("Level segment too small in %s", filename);
	fclose(fp);
}

/*
 * We pick segments from Seg-1 .. Seg-(numsegs).
 *
 * This function dynamically decides how many segments are present,
 * so segments can be added and changed.
 */
static int numsegs(void)
{
	char filename[1100];
	int numfound = 0;

	for (;;)
	{
		snprintf(filename, sizeof filename, "%sSeg-%d", SHAREPATH,
			numfound+1);
		if (access(filename, R_OK) == -1) break;
		numfound++;
	}

	if (numfound == 0)
		abort_with_error("No segments found");

	return numfound;
}

/*
 * A couple features for debugging levels with loops/crashes
 * are found here. One, if you compile with -DONESEG=n, then levels
 * will be made out of segment n repeated. This is a necessary but not
 * a sufficient condition for segment n to coexist nicely with the
 * other segments (i.e., work).
 *
 * Secondly, without NO_LOG, which is presently off by default,
 * field_load() will attempt to write a log.txt file telling which
 * segments were used to construct the current level and drawing a
 * map of the resulting arrangement.
 */

void field_load(void)
{
	int y, x, segnum;
	int segs;
	char filename[1100];
#ifndef NO_LOG
	FILE *log;

	log = fopen("log.txt", "a");
	if (log != NULL)
	{
		fprintf(log, "Making a new level with the following "
			"segments:\n");
	}
#endif

	segs = numsegs();

	for (y = 1; y <= FIELD_HEIGHT; y += SEG_HEIGHT)
		for (x = 1; x <= FIELD_WIDTH; x += SEG_WIDTH)
		{
#ifdef ONESEG /* Test by loading all one particular segment */
			segnum = ONESEG;
#else
			segnum = rnd(segs)+1;
#endif

#ifndef NO_LOG
			if (log != NULL)
				fprintf(log, "%d\n", segnum);
#endif

			snprintf(filename, sizeof filename,
				"%sSeg-%d", SHAREPATH, segnum);
			loadsegment(filename, y, x);
		}

#ifndef NO_LOG
	if (log != NULL)
	{
		for (y = 1; y <= FIELD_HEIGHT; y++)
		{
			for (x = 1; x <= FIELD_WIDTH; x++)
				putc(symbols[field_get(y,x)], log);
			putc('\n', log);
		}
	}
#endif

#ifndef NO_LOG
	if (log != NULL) fclose(log);
#endif
}
