#include <stdio.h> /* for snprintf */
#include <string.h>
#include <assert.h>
#include <allegro.h>
#include "game.h"
#include "field.h"
#include "images.h"
#include "font.h"
#include "colors.h"
#include "game_int.h"

/*
 * Drawing the game and title screens.
 */

/*
 * Sure, the levels are FIELD_WIDTH by FIELD_HEIGHT. But how
 * much is actually on-screen at a time?
 */
#define VIEW_WIDTH  15
#define VIEW_HEIGHT 15

#define SCRN_TOP ((SCRHEIGHT - (VIEW_HEIGHT*FONT_HEIGHT))/2)
#define SCRN_LEFT ((SCRWIDTH - (VIEW_WIDTH*FONT_WIDTH))/2)

static char spot2char(spot_t spot)
{
	const char *screenchars = "-|/\\*+xv^><$#@.";
	return screenchars[spot];
}

#define STATUS_H_MARGIN 2
#define STATUS_W (SCRN_LEFT - STATUS_H_MARGIN*2)

/* Left and right of status1 (left side) and status2 (right side). */
#define STATUS1_LEFT STATUS_H_MARGIN
#define STATUS1_RIGHT (STATUS_H_MARGIN + STATUS_W - 1)
#define STATUS1_CENTER ((STATUS1_LEFT+STATUS1_RIGHT)/2)
#define STATUS2_LEFT (SCRWIDTH - SCRN_LEFT + STATUS_H_MARGIN)
#define STATUS2_RIGHT (SCRWIDTH - STATUS_H_MARGIN)
#define STATUS2_CENTER ((STATUS2_LEFT+STATUS2_RIGHT)/2)

/* The small font's size. */
#define SFONT_WIDTH  8
#define SFONT_HEIGHT 8

#define VSPACE 2

#define LEGEND_MEDIAN 24
#define LEGEND_HEIGHT (3*FONT_HEIGHT + 11*VSPACE \
                    + 11*SFONT_HEIGHT + 2*LEGEND_MEDIAN)
#define LEGEND_TOP ((SCRHEIGHT-LEGEND_HEIGHT)/2)

/* Draw the legend, or "help screen." */
static void drawlegend(BITMAP *target)
{
	int y, x;

	y = LEGEND_TOP;
	x = STATUS1_CENTER;

	draw_sprite(target, image_train(DOWN, RIGHT, TURN_RIGHT),
		x - (FONT_WIDTH/2), y);
	y += FONT_HEIGHT + VSPACE;
	textout_centre_ex(target, font, "Train", x, y, status_em_color, -1);
	y += SFONT_HEIGHT + VSPACE;
	textout_centre_ex(target, font, "Turn with", x, y, status_color, -1);
	y += SFONT_HEIGHT + VSPACE;
	textout_centre_ex(target, font, "arrow", x, y, status_color, -1);
	y += SFONT_HEIGHT + VSPACE;
	textout_centre_ex(target, font, "keys.", x, y, status_color, -1);
	y += SFONT_HEIGHT + LEGEND_MEDIAN;

	draw_sprite(target, image_bullet, x - (FONT_WIDTH/2), y);
	y += FONT_HEIGHT + VSPACE;
	textout_centre_ex(target, font, "Acorns", x, y, status_em_color, -1);
	y += SFONT_HEIGHT + VSPACE;
	textout_centre_ex(target, font, "Fire them", x, y, status_color, -1);
	y += SFONT_HEIGHT + VSPACE;
	textout_centre_ex(target, font, "with WASD", x, y, status_color, -1);
	y += SFONT_HEIGHT + VSPACE;
	textout_centre_ex(target, font, "keys.", x, y, status_color, -1);
	y += SFONT_HEIGHT + LEGEND_MEDIAN;

	draw_sprite(target, image_enemy(NONE, RIGHT, 0), x - (FONT_WIDTH/2), y);
	y += FONT_HEIGHT + VSPACE;
	textout_centre_ex(target, font, "Squirrels", x, y, status_em_color, -1);
	y += SFONT_HEIGHT + VSPACE;
	textout_centre_ex(target, font, "Kill", x, y, status_color, -1);
	y += SFONT_HEIGHT + VSPACE;
	textout_centre_ex(target, font, "them all.", x, y, status_color, -1);
	y += SFONT_HEIGHT;
}

static char *gettimerstring(void)
{
	static char timerstring[] = "x:xx";
	unsigned int timesecs = (timetics + TICS-1) / TICS;

	/* This drawer only handles up to 9:59, natch. */
	assert(timesecs < 600);

	timerstring[0] = '0' + timesecs/60;
	timerstring[2] = '0' + timesecs%60/10;
	timerstring[3] = '0' + timesecs%10;
	return timerstring;
}

#define RSTATUS_MEDIAN1 20
#define RSTATUS_MEDIAN2 (SCRHEIGHT/2)
#define RSTATUS_HEIGHT (7*SFONT_HEIGHT + 4*VSPACE + RSTATUS_MEDIAN1 \
                        + RSTATUS_MEDIAN2)
#define RSTATUS_TOP ((SCRHEIGHT-RSTATUS_HEIGHT)/2)

#define MINIMAP_HEIGHT FIELD_HEIGHT
#define MINIMAP_WIDTH  FIELD_WIDTH

#define MINIMAP_BGCOLOR  0 /* black */
#define MINIMAP_TRAIN   10 /* green */
#define MINIMAP_MONSTER 12 /* red */

/* Draw the mini-map. */
static void drawminimap(BITMAP *target, int ystart, int yend)
{
	int y, x;
	int fy, fx; /* subtractive on the field */
	int mony, monx;
	int ix;

	y = (ystart + yend + MINIMAP_HEIGHT) / 2;
	x = STATUS2_CENTER - MINIMAP_WIDTH/2;

	rectfill(target, x, y, x + MINIMAP_WIDTH-1, y + MINIMAP_HEIGHT-1,
		MINIMAP_BGCOLOR);

	fy = clipy(train.y + train.vdir*(train.steps >= STEPS_PER_MOVE/2)
		- FIELD_HEIGHT/2) - 1;
	fx = clipx(train.x + train.hdir*(train.steps >= STEPS_PER_MOVE/2)
		- FIELD_WIDTH/2)  - 1;

	putpixel(target, x-1 + FIELD_WIDTH/2, y-1 + FIELD_HEIGHT/2,
		MINIMAP_TRAIN);

	for (ix = 0; ix < nummonsters; ix++)
	{
		if (!monsters[ix].alive) continue;
		mony = clipy(monsters[ix].y
			+ monsters[ix].vdir
			* (monsters[ix].steps >= STEPS_PER_MOVE/2) - fy);
		monx = clipx(monsters[ix].x
			+ monsters[ix].hdir
			* (monsters[ix].steps >= STEPS_PER_MOVE/2) - fx);
		putpixel(target, x-1 + monx, y-1 + mony, MINIMAP_MONSTER);
	}
}

/* Draw the right status bar: number of monsters living; timer. */
static void drawstatus(BITMAP *target)
{
	char monstr[STATUS_W/FONT_WIDTH + 1];
	int y, x;

	y = RSTATUS_TOP;
	x = STATUS2_CENTER;

	snprintf(monstr, sizeof monstr, "%d", numlivingmonsters);

	textout_centre_ex(target, font, "Squirrels", x, y, status_em_color, -1);
	y += SFONT_HEIGHT + VSPACE;
	textout_centre_ex(target, font, "Left", x, y, status_em_color, -1);
	y += SFONT_HEIGHT + VSPACE;
	textout_centre_ex(target, font, monstr, x, y, status_color, -1);
	y += SFONT_HEIGHT + RSTATUS_MEDIAN1;

	textout_centre_ex(target, font, "Time Left", x, y, status_em_color, -1);
	y += SFONT_HEIGHT + VSPACE;
	textout_centre_ex(target, font, gettimerstring(), x, y,
		status_color, -1);
	y += SFONT_HEIGHT;

	drawminimap(target, y, y + RSTATUS_MEDIAN2);
	y += RSTATUS_MEDIAN2;

	textout_centre_ex(target, font, "Rail", x, y, status_em_color, -1);
	y += SFONT_HEIGHT + VSPACE;
	textout_centre_ex(target, font, "Blaster", x, y, status_em_color, -1);
	y += SFONT_HEIGHT;
}

/* Draw the legend and status info. */
static void drawstatusnonsense(BITMAP *target)
{
	drawlegend(target);
	drawstatus(target);
}

/************************************************************/

/* Title screen drawing */

/* Draw a string in the big magnified "font." */
static void bigtext(BITMAP *target, int x, int y, const char *s,
	int color, int bg)
{
	x -= FONT_WIDTH * strlen(s) / 2;
	while (*s != '\0')
	{
		drawchar(target, x, y, *s, color, bg);
		x += FONT_WIDTH;
		s++;
	}
}

#define TITLE_COLOR 10 /* bright green */
#define INFO_COLOR 7 /* gray */
#define SPACE_COLOR 11 /* bright cyan */
#define TITLE_HEIGHT (FONT_HEIGHT*10 + SFONT_HEIGHT)
#define TITLE_TOP ((SCRHEIGHT-TITLE_HEIGHT)/2)

static void drawtitle(BITMAP *target)
{
	int y = TITLE_TOP, x = SCRWIDTH/2;

	clear_bitmap(target);

	bigtext(target, x, y, "Rail Blaster", TITLE_COLOR, -1);
	y += 3*FONT_HEIGHT;

	bigtext(target, x, y, "-|/\\    Tracks     ", INFO_COLOR, -1);
	y += FONT_HEIGHT;
	bigtext(target, x, y, "*+x     Junctions  ", INFO_COLOR, -1);
	y += FONT_HEIGHT;
	bigtext(target, x, y, "v^><    Y-junctions", INFO_COLOR, -1);
	y += FONT_HEIGHT;
	bigtext(target, x, y, "@       Reflectors ", INFO_COLOR, -1);
	y += FONT_HEIGHT;
	bigtext(target, x, y, "#       Teleporters", INFO_COLOR, -1);
	y += 3*FONT_HEIGHT;

	textout_centre_ex(target, font, "   P R E S S   S P A C E   ",
		x, y, SPACE_COLOR, -1);
}

/************************************************************/

/*
 * spoty and spotx are in tiles in the level. But they may be off a side.
 * viewy and viewx are in pixels relative to the top left of the screen.
 */
static void drawspot(BITMAP *target, int spoty, int spotx, int viewy, int viewx)
{
	spoty = clipy(spoty);
	spotx = clipx(spotx);
	drawchar(target, viewx, viewy,
		spot2char(field_get(spoty, spotx)), view_color, view_bg);
}

/* The train is always, always in the center of the screen. */
#define TRAINX ((SCRWIDTH  - FONT_WIDTH)  / 2)
#define TRAINY ((SCRHEIGHT - FONT_HEIGHT) / 2)

void game_draw_frame(BITMAP *target)
{
	int y, x, i;
	int trainx, trainy;
	int basespoty, basespotx;
	int basepixely, basepixelx;
	int numspotstodrawy = VIEW_HEIGHT, numspotstodrawx = VIEW_WIDTH;

	if (titlescreen)
	{
		drawtitle(target);
		return;
	}

	clear_to_color(target, status_bg);
	set_clip_rect(target, SCRN_LEFT, SCRN_TOP,
		SCRN_LEFT + VIEW_WIDTH*FONT_WIDTH - 1,
		SCRN_TOP + VIEW_HEIGHT*FONT_HEIGHT - 1);

	trainx = (SCRWIDTH  - FONT_WIDTH)  / 2;
	trainy = (SCRHEIGHT - FONT_HEIGHT) / 2;

	/* The spot that'll show in the top left of the screen */
	basespotx = train.x - VIEW_WIDTH/2;
	basespoty = train.y - VIEW_WIDTH/2;

	/* The pixels where this spot will show */
	basepixelx = SCRN_LEFT;
	basepixely = SCRN_TOP;

	/* Reconcile the base variables with train's sub-spot movement */
	if (train.steps != 0)
	{
		if (train.vdir != NONE) numspotstodrawy++;
		if (train.hdir != NONE) numspotstodrawx++;
		basepixelx -= train.hdir * train.steps * FONT_WIDTH
			/ STEPS_PER_MOVE;
		basepixely -= train.vdir * train.steps * FONT_HEIGHT
			/ STEPS_PER_MOVE;
		if (basepixelx > SCRN_LEFT)
		{
			basespotx--;
			basepixelx -= FONT_WIDTH;
		}
		if (basepixely > SCRN_TOP)
		{
			basespoty--;
			basepixely -= FONT_HEIGHT;
		}
	}

	/* Clip the base spot -- drawing enemies later on needs this. */
	basespotx = clipx(basespotx);
	basespoty = clipy(basespoty);

	for (y = basespoty; y < basespoty + numspotstodrawy; y++)
		for (x = basespotx; x < basespotx + numspotstodrawx; x++)
		{
			drawspot(target, y, x,
				(y-basespoty)*FONT_HEIGHT + basepixely,
				(x-basespotx)*FONT_WIDTH  + basepixelx);
		}

	masked_blit(image_train(train.vdir, train.hdir, train.turn),
		target,
		0, 0,
		trainx, trainy,
		FONT_WIDTH, FONT_HEIGHT);

	for (i = 0; i < nummonsters; i++)
	{
		int monx, mony;
		int imgstatus;

		if (!monsters[i].alive) continue;

		checkmonstersanity(i);

		mony = clipy(monsters[i].y - basespoty + 1) - 1;
		monx = clipx(monsters[i].x - basespotx + 1) - 1;

		/*
		 * Ensure that monsters just entering the view area
		 * from the top or left are visible.
		 */
		if (monsters[i].steps > 0)
		{
			if (mony == FIELD_HEIGHT-1
				&& monsters[i].vdir == DOWN)
			{
				mony -= FIELD_HEIGHT;
			}
			if (monx == FIELD_WIDTH-1
				&& monsters[i].hdir == RIGHT)
			{
				monx -= FIELD_WIDTH;
			}
		}

		mony *= FONT_HEIGHT;
		monx *= FONT_WIDTH;

		mony += basepixely;
		monx += basepixelx;

		if (monsters[i].steps > 0)
		{
			monx += monsters[i].hdir * monsters[i].steps
				* FONT_WIDTH / STEPS_PER_MOVE;
			mony += monsters[i].vdir * monsters[i].steps
				* FONT_HEIGHT / STEPS_PER_MOVE;
		}

		if (monsters[i].alive < 0) imgstatus = 2; /* collapsing */
		else if (monsters[i].steps > 0) imgstatus = 1; /* moving */
		else imgstatus = 0; /* still */

		/*
		 * Yes, I blit every monster, even though it might be
		 * totally offscreen. The view area is clipped, so
		 * it's ok. No blitting is actually getting done if
		 * the enemy is offscreen.
		 */
		masked_blit(image_enemy(monsters[i].vdir, monsters[i].hdir,
			imgstatus),
			target,
			0, 0,
			monx, mony,
			FONT_WIDTH, FONT_HEIGHT);		
	}

	/* XXX mostly a copy and paste from above */
	for (i = 0; i < numbullets; i++)
	{
		int bulx, buly;
		if (!bullets[i].inuse) continue;

		checkbulletsanity(i);

		buly = clipy(bullets[i].y - basespoty + 1) - 1;
		bulx = clipx(bullets[i].x - basespotx + 1) - 1;

		/*
		 * Ensure that bullets just entering the view area
		 * from the top or left are visible.
		 */
		if (bullets[i].steps > 0)
		{
			if (buly == FIELD_HEIGHT-1
				&& bullets[i].vdir == DOWN)
			{
				buly -= FIELD_HEIGHT;
			}
			if (bulx == FIELD_WIDTH-1
				&& bullets[i].hdir == RIGHT)
			{
				bulx -= FIELD_WIDTH;
			}
		}

		buly *= FONT_HEIGHT;
		bulx *= FONT_WIDTH;

		buly += basepixely;
		bulx += basepixelx;

		if (bullets[i].steps > 0)
		{
			bulx += bullets[i].hdir * bullets[i].steps
				* FONT_WIDTH / STEPS_PER_MOVE;
			buly += bullets[i].vdir * bullets[i].steps
				* FONT_HEIGHT / STEPS_PER_MOVE;
		}

		/*
		 * The same excuse above for blitting every monster
		 * applies here too.
		 */
		masked_blit(image_bullet,
			target, /* no pun intended */
			0, 0,
			bulx, buly,
			FONT_WIDTH, FONT_HEIGHT);		
	}

	set_clip_rect(target, 0, 0, SCRWIDTH-1, SCRHEIGHT-1);

	drawstatusnonsense(target);

	if (train.gameovercount)
	{
		/* Print a "TIME OVER" banner. */
		textout_centre_ex(target, font, "   T I M E   O V E R   ",
			SCRWIDTH/2, SCRHEIGHT/2, status_em_color, -1);
	}
}
