#include "std_include.h"
#include "zlib_src\\zlib.h"
#include "bam_structs.h"


// ======================================================================================
// uncompress 1 frame pixels into a BITMAP
// input :
//    bmp         = pointer on the allegro BITMAP to fill
//    data        = pointer on compressed frame pixels
//    compr_idx   = compressed color index
//    data_length = size of the data buffer
// output :
//    return 0 if ok, non-zero if error
// ======================================================================================
int bam_uncompress_frame(BITMAP * bmp, UBYTE * data, int compr_idx, long data_length)
{
   int i=0, x=0, y=0, c;


   for(;;)
   {
      if (i >= data_length)
         return -1;
      c = data[i];
      i++;
      if (c == compr_idx)
      {
         if (i >= data_length)
            return -1;
         c = data[i];
         i++;
         x += c + 1;
      }
      else
      {
         bmp->line[y][x] = c;
         x++;
      }
      while (x >= bmp->w)
      {
         x -= bmp->w;
         y++;
         if (y >= bmp->h)
            return 0;
      }
   }
}


// ======================================================================================
// decode all the frames of the BAM file
// input :
//    bam_anim      = pointer to the BAM_ANIMATION_S where to put the datas
//    buffer        = pointer to BAM file in mem
//    buffer_length = size of this BAM file
// output :
//    return 0 if sucess, non-zero otherwise
// ======================================================================================
int bam_decode_frames(BAM_ANIMATION_S * bam_anim, char * buffer, long buffer_length)
{
   BAM_FRAME_HEADER_S * frm          = NULL;
   BAM_CYCLE_HEADER_S * cyc          = NULL;
   BAM_PALETTE_S      * palette      = NULL;
   UWORD              * lookup_table = NULL;
   BAM_DATA_HEADER_S  * data         = (BAM_DATA_HEADER_S *) buffer;
   long               offset, end_offset;
   int                i, size, max, transparency, trans_idx=0, w, h, x, y, ret, c, f;
   UBYTE              r, g, b, * pixel;
   BITMAP             * bmp;
   DIRECTION_S        * dir;
   FRAME_S            * frame;


   // for safety
   if(buffer == NULL)
   {
      printf("bam_decode_frames() : buffer of BAM file is NULL\n");
      return -1;
   }

   // palette
   // =======
   
   // for safety
   if ( (bam_anim->palette_data.palette    != NULL) ||
        (bam_anim->palette_data.al_palette != NULL) ||
        (bam_anim->palette_data.pal_num    != 0)
      )
   {
      printf("bam_decode_frames() : bam_anim already has palettes datas\n");
      return -1;
   }

   // get palette pointer
   offset = data->palette_offset;
   if ((offset < 0) || (offset >= buffer_length))
   {
      printf("bam_decode_frames() : palette offset (%li) is behind EOF (%li)\n",
         offset, buffer_length);
      return -1;
   }
   end_offset = offset + sizeof(BAM_PALETTE_S);
   if ((end_offset < 0) || (end_offset > buffer_length))
   {
      printf("bam_decode_frames() : end of palette (%li) is behind EOF (%li)\n",
         end_offset, buffer_length);
      return -1;
   }
   palette = (BAM_PALETTE_S *) & buffer[offset];

   // malloc of BAM palette
   bam_anim->palette_data.pal_num = 1;
   size = sizeof(PALETTE_S);
   bam_anim->palette_data.palette = (PALETTE_S *) malloc(size);
   if (bam_anim->palette_data.palette == NULL)
   {
      printf("bam_decode_frames() : can't allocate %i bytes for BAM palette\n", size);
      return -1;
   }
   memset(bam_anim->palette_data.palette, 0, size);

   // malloc of allegro palette
   size = sizeof(PALETTE);
   bam_anim->palette_data.al_palette = (PALETTE *) malloc(size);
   if (bam_anim->palette_data.al_palette == NULL)
   {
      printf("bam_decode_frames() : can't allocate %i bytes for Allegro palette\n", size);
      return -1;
   }
   memset(bam_anim->palette_data.al_palette, 0, size);

   // read palette and search transparency index
   transparency = FALSE;
   for (i=0; i < 256; i++)
   {
      r = palette[0].color[i].r;
      bam_anim->palette_data.palette[0].color[i].r = r;
      bam_anim->palette_data.al_palette[0][i].r    = r >> 2;

      g = palette[0].color[i].g;
      bam_anim->palette_data.palette[0].color[i].g = g;
      bam_anim->palette_data.al_palette[0][i].g    = g >> 2;

      b = palette[0].color[i].b;
      bam_anim->palette_data.palette[0].color[i].b = b;
      bam_anim->palette_data.al_palette[0][i].b    = b >> 2;

      if (transparency == FALSE)
      {
         if ((r == 0) && (g == 255) && (b == 0))
         {
            trans_idx = i;
            transparency = TRUE;
         }
      }
   }
   if (transparency == FALSE)
      trans_idx = 0;

   // ANIM
   // ====

   // malloc of ANIM
   size = sizeof(ANIMATION_S);
   bam_anim->anim = (ANIMATION_S *) malloc(size);
   if (bam_anim->anim == NULL)
   {
      printf("bam_decode_frames() : can't allocate %i bytes for BAM anim struct\n", size);
      return -1;
   }
   memset(bam_anim->anim, 0, size);

   // get frame headers pointer
   offset = data->frame_entries_offset;
   if ((offset < 0) || (offset >= buffer_length))
   {
      printf("bam_decode_frames() : frames offset (%li) is behind EOF (%li)\n",
         offset, buffer_length);
      return -1;
   }
   end_offset = offset + sizeof(BAM_FRAME_HEADER_S) * data->num_frames;
   if ((end_offset < 0) || (end_offset > buffer_length))
   {
      printf("bam_decode_frames() : end of frames (%li) is behind EOF (%li)\n",
         end_offset, buffer_length);
      return -1;
   }
   frm = (BAM_FRAME_HEADER_S *) & buffer[offset];

   // malloc of ANIM BMP
   size = sizeof(BITMAP *) * data->num_frames;
   bam_anim->anim->bmp = (BITMAP **) malloc(size);
   if (bam_anim->anim->bmp == NULL)
   {
      printf("bam_decode_frames() : can't allocate %i bytes for anim.bmp table\n", size);
      return -1;
   }
   memset(bam_anim->anim->bmp, 0, size);
   bam_anim->anim->bmp_num = data->num_frames;

   // decode frames BITMAP
   for (i=0; i < data->num_frames; i++)
   {
      w = frm[i].width;
      h = frm[i].height;
      if ((w >= 1) && (h >= 1))
      {
         bmp = (w==0) || (h==0) ? NULL : create_bitmap_ex(8, w, h);
         if (bmp == NULL)
         {
            printf("can't create BITMAP of frame %i\n", i);
            return -1;
         }
         offset = frm[i].frame_data_offset & 0xEFFFFFFFLU;
         if ((offset < 0) || (offset >= buffer_length))
            continue;

         pixel = (UBYTE *) & buffer[offset];
         if (frm[i].frame_data_offset & 0x80000000LU)
         {
            // RAW frame

            // end offset
            end_offset = offset + (w * h);
            if ((end_offset < 0) || (end_offset > buffer_length))
            {
               printf("bam_decode_frames() : end frame data offset (%li) is behind EOF (%li)\n",
                  end_offset, buffer_length);
               return -1;
            }

            // fill it
            for (y=0; y < h; y++)
               for (x=0; x < w; x++)
                  bmp->line[y][x] = pixel[(y * w) + x];
         }
         else
         {
            // compressed frame
            clear_to_color(bmp, trans_idx);
            ret = bam_uncompress_frame(
               bmp,
               pixel,
               data->compressed_color_index,
               buffer_length
            );
            if (ret)
            {
               printf("bam_decode_frames() : can't uncompress frame %i pixels\n", i);
               return -1;
            }
         }
         bam_anim->anim->bmp[i] = bmp;
      }
   }

   // get cycles pointer
   offset = end_offset;
   if ((offset < 0) || (offset >= buffer_length))
   {
      printf("bam_decode_frames() : cycles offset (%li) is behind EOF (%li)\n",
         offset, buffer_length);
      return -1;
   }
   end_offset = offset + sizeof(BAM_CYCLE_HEADER_S) * data->num_cycles;
   if ((end_offset < 0) || (end_offset > buffer_length))
   {
      printf("bam_decode_frames() : end of cycles (%li) is behind EOF (%li)\n",
         end_offset, buffer_length);
      return -1;
   }
   cyc = (BAM_CYCLE_HEADER_S *) & buffer[offset];

   // get size of lookup table
   max = 0;
   for (i=0; i < data->num_cycles; i++)
   {
      size = cyc[i].start + cyc[i].size;
      if (size > max)
         max = size;
   }

   // get lookup pointer
   offset = data->lookup_table_offset;
   if ((offset < 0) || (offset >= buffer_length))
   {
      printf("bam_decode_frames() : lookup offset (%li) is behind EOF (%li)\n",
         offset, buffer_length);
      return -1;
   }
   end_offset = offset + sizeof(UWORD) * max;
   if ((end_offset < 0) || (end_offset > buffer_length))
   {
      printf("bam_decode_frames() : end of lookup (%li) is behind EOF (%li)\n",
         end_offset, buffer_length);
      return -1;
   }
   lookup_table = (UWORD *) & buffer[offset];

   // fill other datas

   // # of directions
   bam_anim->anim->direction_num = data->num_cycles;

   // malloc of directions buffer
   size = sizeof(DIRECTION_S) * data->num_cycles;
   bam_anim->anim->direction = (DIRECTION_S *) malloc(size);
   if (bam_anim->anim->direction == NULL)
   {
      printf("bam_decode_frames() : can't allocate %i bytes for directions buffer\n", size);
      return -1;
   }
   memset(bam_anim->anim->direction, 0, size);

   // for all directions
   for (c=0; c < data->num_cycles; c++)
   {
      // get a pointer on current direction
      dir = & bam_anim->anim->direction[c];

      // # of frames
      dir->frame_num = cyc[c].size;

      // malloc of frames
      size = sizeof(FRAME_S) * cyc[c].size;
      dir->frame = (FRAME_S *) malloc(size);
      if (dir->frame == NULL)
      {
         printf("bam_decode_frames() : can't allocate %i bytes for frames buffer of cycle %i\n",
            size, c);
         return -1;
      }
      memset(dir->frame, 0, size);

      // for all frames
      for (f=0; f < cyc[c].size; f++)
      {
         // get a pointer on current frame
         frame = & dir->frame[f];

         // frame header index
         i = lookup_table[ cyc[c].start + f ];
         if (i >= data->num_frames)
         {
            printf("bam_decode_frames() : frame %i want to use header %i, max is %i\n",
               f, i, data->num_frames);
            return -1;
         }
         frame->bmp_idx = i;

         // frame box
         x = - frm[i].offset_x;
         y = - frm[i].offset_y;
         w = frm[i].width;
         h = frm[i].height;
         frame->box.x1 = x;
         frame->box.y1 = y;
         frame->box.w  = w;
         frame->box.h  = h;
         frame->box.x2 = x + w;
         frame->box.y2 = y + h;

         // direction box
         if (f == 0)
         {
            dir->box.x1 = x;
            dir->box.y1 = y;
            dir->box.w  = w;
            dir->box.h  = h;
            dir->box.x2 = x + w;
            dir->box.y2 = y + h;
         }
         else
         {
            if (x < dir->box.x1)       dir->box.x1 = x;
            if (x + w > dir->box.x2)   dir->box.x2 = x + w;
            if (y < dir->box.y1)       dir->box.y1 = y;
            if (y + h > dir->box.y2)   dir->box.y2 = y + h;

            dir->box.w = dir->box.x2 - dir->box.x1;
            dir->box.h = dir->box.y2 - dir->box.y1;
         }
      }

      // animation box
      if (c == 0)
      {
         bam_anim->anim->box.x1 = dir->box.x1;
         bam_anim->anim->box.y1 = dir->box.y1;
         bam_anim->anim->box.x2 = dir->box.x2;
         bam_anim->anim->box.y2 = dir->box.y2;
         bam_anim->anim->box.w  = dir->box.w;
         bam_anim->anim->box.h  = dir->box.h;
      }
      else
      {
         if (dir->box.x1 < bam_anim->anim->box.x1)   bam_anim->anim->box.x1 = dir->box.x1;
         if (dir->box.x2 > bam_anim->anim->box.x2)   bam_anim->anim->box.x2 = dir->box.x2;
         if (dir->box.y1 < bam_anim->anim->box.y1)   bam_anim->anim->box.y1 = dir->box.y1;
         if (dir->box.y2 > bam_anim->anim->box.y2)   bam_anim->anim->box.y2 = dir->box.y2;

         bam_anim->anim->box.w = bam_anim->anim->box.x2 - bam_anim->anim->box.x1;
         bam_anim->anim->box.h = bam_anim->anim->box.y2 - bam_anim->anim->box.y1;
      }
   }

   // end
   return 0;
}


// ======================================================================================
// for decompressing a BAMC into a BAM
// Upon entry, dst_length is the total size of the destination buffer, which must be
//    large enough to hold the entire uncompressed data
// Upon exit, dst_length is the actual size of the compressed buffer.
// ======================================================================================
int extract_bam_blocks(char * src, long src_length, char * dst, long * dst_length)
{
   int ret;


   ret = uncompress(
      (unsigned char *) dst,
      (unsigned long *) dst_length,
      (unsigned char *) src,
      src_length
   );
   switch(ret)
   {
      case Z_OK :
         return 0;
         break;

      case Z_MEM_ERROR :
         printf("zlib uncompress() : not enough memory\n");
         return -1;
         break;

      case Z_BUF_ERROR :
         printf("zlib uncompress() : there was not enough room in the output buffer\n");
         return -1;
         break;

      case Z_DATA_ERROR :
         printf("zlib uncompress() : the input data was corrupted\n");
         return -1;
         break;

      default : 
         printf("zlib uncompress() : unknown error\n");
         return -1;
         break;
   }
}


// ======================================================================================
// destroy a BAM animation (release memory)
// always return NULL
// ======================================================================================
BAM_ANIMATION_S * destroy_bam_animation(BAM_ANIMATION_S * bam)
{
   int d, f;


   // no BAM ---> end
   if (bam == NULL)
      return NULL;

   // palettes
   if (bam->palette_data.palette)
      free(bam->palette_data.palette);

   if (bam->palette_data.al_palette)
      free(bam->palette_data.al_palette);

   // other datas
   if (bam->anim)
   {
      // free direction datas
      // --------------------
      if (bam->anim->direction)
      {
         // for all directions
         for (d=0; d < bam->anim->direction_num; d++)
         {
            if (bam->anim->direction[d].frame)
            {
               // free frames datas of this direction
               free(bam->anim->direction[d].frame);
            }
         }

         // free table of direction datas
         free(bam->anim->direction);
      }

      // free BITMAP
      // -----------
      if (bam->anim->bmp)
      {
         for (f=0; f < bam->anim->bmp_num; f++)
         {
            if (bam->anim->bmp[f])
               destroy_bitmap(bam->anim->bmp[f]);
         }

         // free table of BITMAP
         free(bam->anim->bmp);
      }

      // free anim
      // ---------
      free(bam->anim);
   }

   // free BAM
   free(bam);

   // end
   return NULL;
}


// ======================================================================================
// decode a BAM or a BAMC into a BAM_ANIMATION_S structure
// return NULL if error
// up to you to free this structure with the help of destroy_bam_animation()
// ======================================================================================
BAM_ANIMATION_S * decode_bam(char * buffer, long buffer_length)
{
   BAM_FILE_HEADER_S  * hdr               = NULL;          // small file header
   BAMC_DATA_HEADER_S * data_c            = NULL;          // header in case of BAMC
   char               * data              = NULL;          // BAM file buffer to decode
   long               uncompressed_length = buffer_length; // uncompressed length
   char               * decomp_buffer     = NULL;          // temp buffer in case of BAMC
   BAM_ANIMATION_S    * bam_anim          = NULL;          // the final animation struct
   int                ret,                                 // misc return value
                      size;                                // temp size


   // file header
   hdr = (BAM_FILE_HEADER_S *) buffer;

   // is it a compressed BAM ?
   if (strncmp(hdr->signature, "BAMC", 4) == 0)
   {
      // yep, so decompress it

      // use a BAMC type of header
      data_c = (BAMC_DATA_HEADER_S *) buffer;

      // get datas
      uncompressed_length = data_c->length;
      decomp_buffer       = (char *) malloc(uncompressed_length);
      memset(decomp_buffer, 0, uncompressed_length);
      if (decomp_buffer == NULL)
      {
         printf("can't allocate %li bytes for decompressing the BAM\n",
            uncompressed_length);
         return NULL;
      }

      // uncompress it (it'll become a full new BAM file)
      ret = extract_bam_blocks(
         (char *) & data_c->bamc_data_start, // * src
         buffer_length - 12,                 // src_length
         decomp_buffer,                      // * dst
         & uncompressed_length               // * dst_length
      );
      if (ret)
      {
         free(decomp_buffer);
         return NULL;
      }

      // we'll use this new BAM file
       data = decomp_buffer;
   }
   else
   {
      // a regular BAM
      data = buffer;
   }

   // make a new BAM_ANIMATION_S
   size     = sizeof(BAM_ANIMATION_S);
   bam_anim = (BAM_ANIMATION_S *) malloc(size);
   if (bam_anim == NULL)
   {
      if (decomp_buffer)
         free(decomp_buffer);
      printf("decode_bam() : can't allocate %i bytes for BAM_ANIMATION_S\n", size);
      return NULL;
   }
   memset(bam_anim, 0, size);

   // decode all frames, as well as other datas (such as palette)
   ret = bam_decode_frames(bam_anim, data, uncompressed_length);
   if (ret)
   {
      if (decomp_buffer)
         free(decomp_buffer);
      destroy_bam_animation(bam_anim);
      return NULL;
   }

   // end
   if (decomp_buffer)
      free(decomp_buffer);

   return bam_anim;
}
