/* Path Finding Demo
   Copyright (C) 2004 by David A. Capello

   See LICENSE for more information.
*/

#include <allegro.h>
#include "heap.h"
#include "path.h"

char map[H][W];		/* TRUE=wall, FALSE=floor */

int map_x1, map_y1;	/* origin position (from this position) */
int map_x2, map_y2;	/* destination position (to this position) */

int use_diag = FALSE;	/* don't use diagonal movement through walls */

/**********************************************************************/
/* NODE */

/* A is a best node of B */
#define is_best_node(a, b)  _is_best_node(((NODE *)(a)), ((NODE *)(b)))
#define _is_best_node(a, b) (a->g+a->h < b->g+b->h)

/* recalculate G (movement cost made) */
#define recalc_node_g(a) _recalc_g(((NODE *)(a)))
#define _recalc_g(a)						\
  a->g = a->parent ? a->parent->g+				\
    ((a->x != a->parent->x && a->y != a->parent->y)? 14: 10):0;

/* recalculate H (heuristic movement cost) */
#define recalc_node_h(a) _recalc_h(((NODE *)(a)))
#if 1				/* using Manhattan distance */
#define _recalc_h(a)						\
  a->h = (ABS (a->x-map_x2/TW) + ABS (a->y-map_y2/TH)) * 10;
#elif 1				/* using diagonal distance */
#define _recalc_h(a)						\
  a->h = MAX (ABS (a->x-map_x2/TW), ABS (a->y-map_y2/TH)) * 10;
#elif 1				/* using straight line distance */
#define _recalc_h(a)							\
  a->h = sqrt (pow (a->x-map_x2/TW, 2) + pow (a->y-map_y2/TH, 2)) * 10;
#endif

static NODE *create_node (int x, int y, NODE *parent)
{
  NODE *node = malloc (sizeof (NODE));
  node->x = x;
  node->y = y;
  node->parent = parent;
  recalc_node_g (node);
  recalc_node_h (node);
  return node;
}

static int cmp_nodes (const void *p1, const void *p2)
{
  return is_best_node (p1, p2) ? -1: 1;
}

/**********************************************************************/
/* PATH */

static PATH *create_path (void)
{
  PATH *path = malloc (sizeof (PATH));
  heap_init (&path->open, cmp_nodes);
  heap_init (&path->close, cmp_nodes);
  path->dest_node = NULL;
  return path;
}

static void add_node_to_path (PATH *path, NODE *node)
{
  int c;

  /* can't add the node here */
  if (map[node->y][node->x] != FALSE) {
    free (node);
    return;
  }

  /* is in the closed list */
  for (c=0; c<path->close.top; c++) {
    if (((NODE *)path->close.array[c])->x == node->x &&
	((NODE *)path->close.array[c])->y == node->y) {
      free (node);
      return;
    }
  }

  /* is in the opened list? */
  for (c=0; c<path->open.top; c++) {
    if (((NODE *)path->open.array[c])->x == node->x &&
	((NODE *)path->open.array[c])->y == node->y) {
      /* if the new node is a best path */
      if (is_best_node (node, path->open.array[c])) {
	/* replace the parent of the node that is already in the
	   opened list, and recalculate the movement cost */
	((NODE *)path->open.array[c])->parent = node->parent;
	recalc_node_g (path->open.array[c]);
	_heap_usher (&path->open, c);

	free (node);
	return;
      }
      else {
	free (node);
	return;
      }
    }
  }

  heap_push (&path->open, node);
}

#define ADD(u,v)							\
  add_node_to_path (path, create_node (node->x+u, node->y+v, node))

static void add_adjacent_nodes_to_path (PATH *path, NODE *node)
{
  if (node->y > 0) {
    if (node->x > 0) ADD (-1, -1);
    ADD (0, -1);
    if (node->x < W-1) ADD (+1, -1);
  }

  if (node->x > 0) ADD (-1, 0);
  if (node->x < W-1) ADD (+1, 0);

  if (node->y < H-1) {
    if (node->x > 0) ADD (-1, +1);
    ADD (0, +1);
    if (node->x < W-1) ADD (+1, +1);
  }
}

static void add_adjacent_nodes_to_path_check_diag (PATH *path, NODE *node)
{
  if (node->y > 0) {
    ADD (0, -1);
    if (!map[node->y-1][node->x]) {
      if (node->x > 0 && !map[node->y][node->x-1]) ADD (-1, -1);
      if (node->x < W-1 && !map[node->y][node->x+1]) ADD (+1, -1);
    }
  }

  if (node->x > 0) ADD (-1, 0);
  if (node->x < W-1) ADD (+1, 0);

  if (node->y < H-1) {
    ADD (0, +1);
    if (!map[node->y+1][node->x]) {
      if (node->x > 0 && !map[node->y][node->x-1]) ADD (-1, +1);
      if (node->x < W-1 && !map[node->y][node->x+1]) ADD (+1, +1);
    }
  }
}

static void check_best_node (PATH *path)
{
  NODE *cur;

  while ((cur = heap_pop (&path->open))) {
    heap_push (&path->close, cur);

    if (map_x2 >= cur->x*TW && map_x2 <= cur->x*TW+TW-1 &&
	map_y2 >= cur->y*TH && map_y2 <= cur->y*TH+TH-1) {
      path->dest_node = cur;
      break;
    }

    if (use_diag)
      add_adjacent_nodes_to_path (path, cur);
    else
      add_adjacent_nodes_to_path_check_diag (path, cur);
  }
}

PATH *find_path (void)
{
  PATH *path = create_path ();

  add_node_to_path (path, create_node (map_x1/TW, map_y1/TH, NULL));
  check_best_node (path);

  return path;
}

void destroy_path (PATH *path)
{
  heap_foreach (&path->open, free);
  heap_foreach (&path->close, free);
  heap_fini (&path->open);
  heap_fini (&path->close);
  free (path);
}
