/* Path Finding Demo
   Copyright (C) 2004 by David A. Capello

   See LICENSE for more information.
*/

#include <allegro.h>
#include <math.h>
#include "heap.h"
#include "path.h"

static PATH *path = NULL;	/* the path */

static void regen_path (void);

static void clear_map (void);
static void reset_map (void);
static void grid_map (void);
static void spiral_map (void);
static void random_map (void);

static void draw_node (BITMAP *bmp, NODE *node, int color);
static void thick_putpixel (BITMAP *bmp, int x, int y, int color);
static void map_putpixel (BITMAP *bmp, int x, int y, int color);

int main (void)
{
  int old_mouse_x, old_mouse_y;
  double angle;
  BITMAP *bmp;
  int x, y;

  allegro_init();
  install_timer();
  install_keyboard();
  install_mouse();

  if (set_gfx_mode (GFX_AUTODETECT, 320, 240, 0, 0) < 0) {
    allegro_message ("Error setting graphics mode\n%s", allegro_error);
    return 1;
  }

  clear (screen);
  text_mode (0);
  y=0;
  textout (screen, font, "A* (A star) Path Finding Demo", 0, 8*(y++), makecol (255, 255, 255));
  textout (screen, font, "Copyright (C) 2004 by David A. Capello", 0, 8*(y++), makecol (255, 255, 255));
  y++;
  textout (screen, font, "Tiles Reference:", 0, 8*(y++), makecol (255, 255, 255));
  textout (screen, font, "Black: floor", 8, 8*(y++), makecol (255, 255, 255));
  textout (screen, font, "Gray: wall", 8, 8*(y++), makecol (255, 255, 255));
  textout (screen, font, "White: final path", 8, 8*(y++), makecol (255, 255, 255));
  textout (screen, font, "Blue: nodes in closed list", 8, 8*(y++), makecol (255, 255, 255));
  textout (screen, font, "Red: nodes in opened list", 8, 8*(y++), makecol (255, 255, 255));
  y++;
  textout (screen, font, "Tiles Operations (in mouse position):", 0, 8*(y++), makecol (255, 255, 255));
  textout (screen, font, "Key <1>: change origin position", 8, 8*(y++), makecol (255, 255, 255));
  textout (screen, font, "Key <2>: change destination position", 8, 8*(y++), makecol (255, 255, 255));
  textout (screen, font, "Left button: add wall", 8, 8*(y++), makecol (255, 255, 255));
  textout (screen, font, "Right button: add floor (remove wall)", 8, 8*(y++), makecol (255, 255, 255));
  y++;
  textout (screen, font, "Entire Map Operations:", 0, 8*(y++), makecol (255, 255, 255));
  textout (screen, font, "Key <C>: Clear map", 8, 8*(y++), makecol (255, 255, 255));
  textout (screen, font, "Key <G>: Grid map", 8, 8*(y++), makecol (255, 255, 255));
  textout (screen, font, "Key <S>: Spiral map", 8, 8*(y++), makecol (255, 255, 255));
  textout (screen, font, "Key <R>: Random map", 8, 8*(y++), makecol (255, 255, 255));

  textout (screen, font, "Press any key to continue", 0, SCREEN_H-8, makecol (255, 255, 255));
  readkey ();

  bmp = create_bitmap (SCREEN_W, SCREEN_H);
  reset_map ();

  /* origin and destination positions */
  map_x1 = SCREEN_W/4;
  map_x2 = SCREEN_W*3/4;
  map_y1 = map_y2 = SCREEN_H/2;

  old_mouse_x = old_mouse_y = -1;

  regen_path ();

  do {
    clear (bmp);

    /* modify terrain */
    if (mouse_b) {
      if (old_mouse_x > 0)
	do_line (bmp, old_mouse_x/TW, old_mouse_y/TH, mouse_x/TW, mouse_y/TH,
		 mouse_b & 1 ? TRUE: FALSE, map_putpixel);
      else
	map_putpixel (bmp, mouse_x/TW, mouse_y/TH, mouse_b & 1 ? TRUE: FALSE);
	       
      old_mouse_x = mouse_x;
      old_mouse_y = mouse_y;

      regen_path ();
    }
    else
      old_mouse_x = old_mouse_y = -1;

    /* change origin or destination */
    if (key[KEY_1] || key[KEY_2]) {
      if (key[KEY_1]) {
	map_x1 = mouse_x;
	map_y1 = mouse_y;
      }
      if (key[KEY_2]) {
	map_x2 = mouse_x;
	map_y2 = mouse_y;
      }

      regen_path ();
    }

    /* entire map operations */
    if (key[KEY_C]) { clear_map (); regen_path (); }
    if (key[KEY_G]) { grid_map (); regen_path (); }
    if (key[KEY_S]) { spiral_map (); regen_path (); }
    if (key[KEY_R]) { random_map (); regen_path (); while (key[KEY_R]); }

    /* draw origin */
    angle = atan2 (map_y2-map_y1, map_x2-map_x1);
    triangle (bmp,
	      map_x1+cos(angle)*16, map_y1+sin(angle)*16,
	      map_x1+cos(angle+AL_PI*3/4)*16, map_y1+sin(angle+AL_PI*3/4)*16,
	      map_x1+cos(angle-AL_PI*3/4)*16, map_y1+sin(angle-AL_PI*3/4)*16,
	      makecol (100, 100, 100));

    /* draw destination */
    do_circle (bmp, map_x2, map_y2, 8,
	       makecol (100, 100, 100), thick_putpixel);

    /* draw map tiles (wall/floors) */
    for (y=0; y<H; y++)
      for (x=0; x<W; x++)
	if (map[y][x])
	  rectfill (bmp, x*TW, y*TH, x*TW+TW, y*TH+TH, makecol (128, 128, 128));
	else
	  rect (bmp, x*TW, y*TH, x*TW+TW, y*TH+TH, makecol (128, 128, 128));

    /* draw path */
    if (path) {
      NODE *node;
      int c;

      /* draw nodes in opened list */
      for (c=0; c<path->open.top; c++)
	draw_node (bmp, path->open.array[c], makecol (255, 0, 0));

      /* draw nodes in closed list */
      for (c=0; c<path->close.top; c++)
	draw_node (bmp, path->close.array[c], makecol (0, 0, 255));

      /* draw final path */
      for (node=path->dest_node; node; node=node->parent)
	draw_node (bmp, node, makecol (0, 255, 255));
    }

    /* draw mouse */
    hline (bmp, mouse_x-4, mouse_y, mouse_x+4, makecol(255, 255, 255));
    vline (bmp, mouse_x, mouse_y-4, mouse_y+4, makecol(255, 255, 255));

    /* blit to screen */
    blit (bmp, screen, 0, 0, 0, 0, SCREEN_W, SCREEN_H);
  } while (!key[KEY_ESC]);

  destroy_bitmap (bmp);

  allegro_exit();
  return 0;
}

END_OF_MAIN ();

static void regen_path (void)
{
  if (path)
    destroy_path (path);
  path = find_path ();
}

static void clear_map (void)
{
  int x, y;

  for (y=0; y<H; y++)
    for (x=0; x<W; x++)
      map[y][x] = FALSE;
}

static void reset_map (void)
{
  clear_map ();

  map[H/2-1][W/2] = TRUE;
  map[H/2  ][W/2] = TRUE;
  map[H/2+1][W/2] = TRUE;
}

static void grid_map (void)
{
  int x, y;

  clear_map ();

  for (y=0; y<H; y++)
    for (x=0; x<W; x++)
      map[y][x] = x & 1 && y & 1 ? TRUE: FALSE;
}

static void spiral_map (void)
{
  int c, i, x, y, d;

  for (y=0; y<H; y++)
    for (x=0; x<W; x++)
      map[y][x] = TRUE;

  c = 1;
  d = 0;
  x = W/2;
  y = H/2;

  while (x >= 0 && y >= 0 && x < W && y < H) {
    switch (d) {
      case 0:
	for (i=0; i<c; i++, x--)
	  map[y][x] = FALSE;
	d=1;
	break;
      case 1:
	for (i=0; i<c; i++, y--)
	  map[y][x] = FALSE;
	d=2;
	break;
      case 2:
	for (i=0; i<c; i++, x++)
	  map[y][x] = FALSE;
	d=3;
	break;
      case 3:
	for (i=0; i<c; i++, y++)
	  map[y][x] = FALSE;
	d=0;
	break;
    }
    c++;
  }
}

static void random_map (void)
{
  int x, y;

  for (y=0; y<H; y++)
    for (x=0; x<W; x++)
      map[y][x] = (rand () & 3) == 0 ? TRUE: FALSE;

  map[map_y1/TH][map_x1/TW] = FALSE;
  map[map_y2/TH][map_x2/TW] = FALSE;
}

static void draw_node (BITMAP *bmp, NODE *node, int color)
{
  rect (bmp,
	node->x*TW+1, node->y*TH+1,
	node->x*TW+TW-1, node->y*TH+TW-1, color);

  if (node->parent) {
    int dx = node->parent->x - node->x;
    int dy = node->parent->y - node->y;
    line (bmp,
	  node->x*TW+TW/2, node->y*TH+TH/2,
	  node->x*TW+TW/2+dx*TW/2,
	  node->y*TH+TH/2+dy*TH/2, color);
  }
}

static void thick_putpixel (BITMAP *bmp, int x, int y, int color)
{
  rectfill (bmp, x, y, x+1, y+1, color);
}

static void map_putpixel (BITMAP *bmp, int x, int y, int color)
{
  (void)bmp;
  map[y][x] = color;
}
