// Utility functions used by other, larger functions.
// Or, stuff that didn't fit anywhere else.

#include "headers.h"

// Sort the filenames using a bubble sort algorithm
void sort_filenames (char choices[][MAX_MENU_OPTION_LENGTH], int num_choices)
{
    int which, pass;
    char temp [512];

    for (pass = 0; pass < num_choices; pass++)
    {
        for (which = 0; which < num_choices - 1; which++)
        {
            if (strcmp (choices[which], choices[which + 1]) > 0)
            {
                // Swap the values of the strings
                strcpy (temp, choices[which]);      
                strcpy (choices[which], choices[which + 1]);
                strcpy (choices[which + 1], temp);
            }
        }
    }
}



// Finds the index of color_table that holds the color col. Returns -1 if the color
// was not found
int color_table_index(int col)
{
    int index;

    for (index = 0; index < MAX_COLOR_TABLE_COLORS; index++) {
//      if (color_table[index] == col) return index;
    }

    return -1;
}



// Add a color to the current color boxes
int add_current_color (int col)
{
    int which_color;
    
    // Search the array to find if col is already in it
    for (which_color = 0; which_color < MAX_CUR_COLORS; which_color++)
    {
        // If col is found, returns the index of the array where it is
        if (cur_color[which_color] == col) return which_color;
    }
    
    // The color is different than the first one in our array
    if (col != cur_color[0])
    {
        // Shift all the colors in our array forward once
        for (which_color = MAX_CUR_COLORS - 1; which_color > 0; which_color--)
        {
            cur_color[which_color] = cur_color[which_color - 1];
        }

        cur_color[0] = col;         // Assign the first color in the array
        draw_color = 0;             // Reset the drawing color

        refresh_cur_color_boxes();  // Refresh the current color boxes
    }

    return -1;  // col was not found in the array and was added
}



//
// Lighten or darken the current drawing color
// color_add: the number of shades to increment the drawing color
//
void increment_drawing_color (int color_add)
{
    int r, g, b;

    r = getr(separate_cur_color) + (color_add * 8);
    g = getg(separate_cur_color) + (color_add * 8);
    b = getb(separate_cur_color) + (color_add * 8);
    
    if (r < 0) r = 0;
    if (g < 0) g = 0;
    if (b < 0) b = 0;
 
    if (r > 255) r = 255;
    if (g > 255) g = 255;
    if (b > 255) b = 255;
    
    separate_cur_color = makecol(r, g, b);
}



// Copies one video page to another
void copy_page (BITMAP *source, BITMAP *dest)  {
    blit (source, dest, 0, 0, 0, 0, SCREEN_W, SCREEN_H);
}



// Prepares for an undo function by cycling the sprite array images.
// If forwards is TRUE, images are cycled forwards to prepare for undo.
// If forwards is FALSE, images are cycled backwards to execute an undo.
void cycle_sprite_array (int forwards)
{
    static int num_undos_left,
        which_image;

    if (forwards == PREPARE_FOR_UNDO)     // Cycle image forwards to prepare for undo
    {
        for (which_image = NUM_IMAGES - 1; which_image > 0; which_image--)
        {
            blit(image[which_image - 1], image[which_image], 0, 0, 0, 0, image[0]->w,
                image[0]->h);
        }

        if (num_undos_left < NUM_IMAGES) num_undos_left++;
    }
    else if (forwards == EXECUTE_UNDO)  // Cycle images backwards to execute undo
    {
        if (num_undos_left == 0) return;

        for (which_image = 0; which_image < NUM_IMAGES - 1; which_image++)
        {
            blit(image[which_image + 1], image[which_image], 0, 0, 0, 0, image[0]->w,
                image[0]->h);
        }

        num_undos_left--;
    }
    else if (forwards == PURGE_UNDO_IMAGES)     // Erase all undo images
    {
        for (which_image = 1; which_image < NUM_IMAGES - 1; which_image++)
        {
            clear_to_color(image[which_image], TRANSPARENT_COL);
        }

        num_undos_left = 0;
    }

/*
    // Test: display the undo images
    for (which_image = 0; which_image < NUM_IMAGES; which_image++) {
        blit(image[which_image], screen, 0, 0, 0, which_image * image[0]->h, image[0]->w,
            image[0]->h);
    }
    PAUSE;
*/
}



// Get a filename from the user and save the sprite to that filename
void save_image_to_file (void)
{
    int x1, y1, x2, y2;
    int width, height;
    unsigned int max_length = MAX_SAVE_FILENAME;
    char filename[200];
    
    show_mouse(NULL);       // Hide the mouse pointer

    // Determine the location of the window
    width = text_length(font, "_") * max_length + 10;
    height = text_height(font) * 2 + 10;
    x1 = mouse_x - width / 2;
    y1 = mouse_y - height / 2;
    if (x1 < 0) x1 = 0;
    if (y1 < 0) y1 = 0;
    if (x1 + width > SCREEN_W)
        x1 = SCREEN_W - width;
    if (y1 + height > SCREEN_H)
        y1 = SCREEN_H - height;
    x2 = x1 + width;
    y2 = y1 + height;

    // Draw the window
    rectfill(buffer, x1 + 1, y1 + 1, x2 - 1, y2 - 1, BLACK);
    rect(buffer, x1, y1, x2, y2, WHITE);
	
	//
    // If the current filename is too big to fit in the save box, chop off the
	// last 4 characters
	//
    if (strlen(cur_filename) > max_length)
	{
        cur_filename[strlen(cur_filename) - 4] = '\0';
    }
		
    // Tell the user what to enter and get the sprite filename
    TEXTPRINTF(buffer, font, x1 + 5, y1 + 5, TEXTPRINTF_COL, "Save as:");
    get_string(filename, x1 + 5, y1 + 5 + text_height(font), cur_filename, max_length);
    
    write_bitmap_file (filename, image[0]);

	// Copy this filename to the current filename var
    if (filename[0] != '(') strcpy(cur_filename, filename);
    
    redraw_core(); 
    show_mouse(buffer);     // Replace the mouse pointer

}



//
// Helper function for save_image_to_file
// PARAMETER filename       The filename entered by the user
// PARAMETER image          The bitmap image to save
//
void write_bitmap_file (char *filename, BITMAP *image)
{
    int x, y;
    int which_tile;
    int width, height;
    int number_length;
    BITMAP *tile;
    char *close_paren = strchr(filename, ')');
    char tile_filename [MAX_SAVE_FILENAME + 10];
    char number_str[10];

	// If the filename is empty, bail out
    if (strlen(filename) < 1) return;
		
	// Zoom out if we are in magnified mode
	if (magnified == TRUE) restore_magnified_area ();

    //
    // If the filename begins with ( and contains ), chop it into tiles and 
    // write them to individual bitmap files. Otherwise, just write the file.
    //
    if (filename[0] == '(' && close_paren != NULL)
    {
        width = atoi (&filename[1]);
        height = atoi (strchr(filename, ',') + 1);

        which_tile = 0;
    
        for (y = 0; y < image->h; y += height)
        {
            for (x = 0; x < image->w; x += width)
            {
                which_tile++; 

                tile = create_sub_bitmap (image, x, y, width, height);

                number_length = (int) ceil(log10(ceil((double)image->w / width)
                     * ceil((double)image->h / height)));

                get_str_from_int (number_str, which_tile, number_length);

                strcpy (tile_filename, close_paren + 1);
                strcat (tile_filename, " ");
                strcat (tile_filename, number_str);
                strcat (tile_filename, ".bmp");
                
                save_bmp (tile_filename, tile, NULL);
            }
        }
    }
    else
    {
        //
        // Append a .bmp extension to the sprite filename if it does not
        // already end with .bmp
        //
        if(strcmp(&filename[strlen(filename) - 4], ".bmp") != 0)
        {
            strcat(filename, ".bmp");
        }

        save_bmp(filename, image, NULL);
    }
}



//
// DESCRIPTIONL:    Creates a string representation of an integer.
// PARAMETER str:   The string to write a representation of the value to
// PARAMTER num:    The number to create a string rep. of
// PARAMTER length: The length of the finished string. If num contains fewer
//                  digits than length, zeroes are prepended onto the string.
//
void get_str_from_int (char *str, int num, int length)
{
    int digit;

    for (digit = 0; digit < length; digit++)
    {
        str[digit] = '0' + ((num / (int) pow(10, length - digit - 1)) % 10);
    }

    str[length] = '\0';
}



// Loads a sprite from a filename selected by the user
// Returns TRUE on success, FALSE on error
int load_image_from_file (char filename[])
{
    char choices [MAX_MENU_OPTIONS][MAX_MENU_OPTION_LENGTH]; // Holds menu choice strings

    struct al_ffblk file_info;  // Contains info about the file found

    // Results of a search for bitmap files. 0 = found, non-0 = none found.
    static int file_search_results,
        num_files,              // Number of bitmap files found in the current directory
        option,                 // Menu selection
        x1, y1, x2, y2,         // Used for positioning the menu window
        width,                  // Potential width of the menu window
        index;

    BITMAP *bmp;                // Load the image here

/*
struct al_ffblk
{
    int attrib;         // Actual attributes of the file found
    time_t time;        // Modification time of file
    long size;          // Size of file
    char name[512];     // Name of file
};
*/
    // Filename of "" means we are supposed to get it from the user.
    // Filename of "<" or ">" means advance to next/previous filename.
    if (strcmp (filename, "<") == 0 || strcmp (filename, ">") == 0 ||
        strlen(filename) == 0)
    {
        // Search for bitmap files in the current directory
        file_search_results = al_findfirst("*.bmp", &file_info, FA_RDONLY | FA_ARCH);

        // Loop until no more files are found
        for (num_files = 0; file_search_results == 0; num_files++)
        {
            strcpy(choices[num_files], file_info.name);     // Assign the file name
            file_search_results = al_findnext(&file_info);  // Continue searching for files
        }
        al_findclose(&file_info);                           // End the file search

        sort_filenames (choices, num_files);

        // Filename of "": activate the menu to get the filename
        if (strlen(filename) == 0)
        {
            strcpy(choices[num_files], "CANCEL");

            // Determine the location of the menu window
            width = get_width(choices, num_files);
            x1 = mouse_x - width / 2;
            x2 = x1 + width;
            y1 = mouse_y - (5 + text_height(font) / 2);
            y2 = AUTO_DETECT;

            if (num_files * text_height(font) + 10 > SCREEN_H)
            {
                y1 = 0;
                y2 = SCREEN_H - 1;
            }

            // Do not load if there are no files
            if (num_files == 0) return TRUE;

            option = get_menu_option(x1, y1, x2, y2, num_files + 1, choices);
        }
        // Filename of "" means we are supposed to get it from the user.
        else if (strcmp (filename, "<") == 0 || strcmp (filename, ">") == 0)
        {
            option = num_files + 1; // Cancel by default; do not load anything

            // Find the filename in the list
            for (index = 0; index < num_files; index++)
            {
                if (strcmp(choices[index], cur_filename) == 0)
                {
                    if (strcmp (filename, "<") == 0) option = index - 1;
                    if (strcmp (filename, ">") == 0) option = index + 1;

                    if (option < 0) option = num_files - 1;
                    if (option == num_files) option = 0;
                    
                    break;
                }
            }
        }
    }
    else        // If the filename is given
    {
        option = 0;         // Tell it to load from that filename
        num_files = 1;
        strcpy(choices[0], filename);
    }

    if (option < num_files)                  // "CANCEL" option was not selected
    {
        bmp = load_bmp(choices[option], NULL);

        if (bmp == NULL)
        {
            // Save the filename of the file that failed to load
            strcpy(load_fail_filename, choices[option]);
            return FALSE;
        }

        // Copy temp bitmap to image and destroy the temp bitmap
        destroy_bitmap (image[0]);
        image[0] = create_bitmap (bmp->w, bmp->h);
        blit(bmp, image[0], 0, 0, 0, 0, bmp->w, bmp->h);
        destroy_bitmap(bmp);

        strcpy(cur_filename, choices[option]);          // Reset the current filename
        
        init_newly_loaded_image ();
    }
    else
    {
        show_mouse (NULL);
        redraw_core ();
    }

    show_mouse(buffer);         // Show the mouse cursor

    return TRUE;
}



// Reset variables and redraw things after loading a new sprite image
void init_newly_loaded_image ()
{
    static int which_image;

    // Clear out all the sprite (undo) images
    for (which_image = 1; which_image < NUM_IMAGES; which_image++)
    {
        destroy_bitmap (image[which_image]);
        image [which_image] = create_bitmap (image[0]->w, image[0]->h);
        clear_to_color (image[which_image], TRANSPARENT_COL);
    }

    // Destroy and re-create the buffer image
    destroy_bitmap (buffer_image);
    buffer_image = create_bitmap (image[0]->w, image[0]->h);
    clear_to_color (buffer_image, TRANSPARENT_COL);

    init_grid_cell_size ();
    redraw_core();
}



// Sorts a string array alphabetically
void sort_string_array(char unsorted_strings[][MAX_MENU_OPTION_LENGTH], int num_elements)
{
    int which_string, passes;
    char temp_str[MAX_MENU_OPTION_LENGTH];
    char new_strings[MAX_MENU_OPTIONS][MAX_MENU_OPTION_LENGTH];

    // Copy the unsorted strings array to our new strings array
    for (which_string = 0; which_string < num_elements; which_string++ ){
        strcpy(new_strings[which_string], unsorted_strings[which_string]);
    }

    // Go through the array as many times as there are elements squared
    for (passes = 0; passes < num_elements * num_elements; passes++) {
        // Go through each string element
        for (which_string = 0; which_string < num_elements - 1; which_string++) {
            // If the next string is before the first one in the alphabet
            if (strcmp(new_strings[which_string], new_strings[which_string + 1]) > 0) {
                // Swap them
                strcpy(temp_str, new_strings[which_string]);
                strcpy(new_strings[which_string], new_strings[which_string + 1]);
                strcpy(new_strings[which_string + 1], temp_str);                
            }
        }
    }

    // Copy the sorted strings array back to the unsorted strings array (which will then
    // be sorted!)
    for (which_string = 0; which_string < num_elements; which_string++ ){
        strcpy(unsorted_strings[which_string], new_strings[which_string]);
    }

}



// Set the size of a sprite cell
void init_grid_cell_size (void)
{
    grid_cell_size = (SCREEN_W - 256) / image[0]->w;    // Init the cell size

    if ((SCREEN_H / image[0]->h) < grid_cell_size)      // If the cell size is too tall
    {
        grid_cell_size = SCREEN_H / image[0]->h;        // Resize to fis vertically
    }

    if (grid_cell_size < 1) grid_cell_size = 1;         // Keep it >= 1
}



// Display a line of text for the help screen
// If init = 0, start at top of screen
void help_text (char * text)
{
    static int y;

    if (strlen (text) > 0)
    {
        if (text[0] == '~')
		{
			y = 5;     // Init the text at the top of the screen
			return;
		}
    }

    TEXTPRINTF (buffer, font, 5, y, TEXTPRINTF_COL, "%s", text);

    y += text_height (font) + 3;
}



// Saves certain options to a local configuration file
void save_config_file (void)
{
    FILE *fp_out;

    fp_out = fopen("bse.cfg", "w");     // Open the file for output

    // Write the information
    fprintf (fp_out, "# Basic Sprite Editor v2.0 configuration file\n");
    fprintf (fp_out, "# Options are saved here so they can be loaded upon restart\n\n");
    fprintf (fp_out, "%s\n", cur_filename);
    fprintf (fp_out, "%d, %d, %d, %d, %d, %d, %d\n", SCREEN_W, 
        SCREEN_H, screen_type, grid_activated, background_style, color_strips_on,
        solid_paste);

    fclose(fp_out);                         // Close the file
}



//
// Save the magnified portion of the image in the
// original and restore it.
//
void restore_magnified_area (void)
{
	static int width, height;
	
	blit (image[0], magnify_original, 0, 0, mag_x_pixel, mag_y_pixel,
		image[0]->w, image[0]->h);

	width = magnify_original->w;
	height = magnify_original->h;

	destroy_bitmap (image[0]);
	image[0] = create_bitmap (width, height);
	blit (magnify_original, image[0], 0, 0, 0, 0, width, height);

	show_mouse (NULL);
	init_newly_loaded_image ();
	show_mouse (buffer);

	cycle_sprite_array (PURGE_UNDO_IMAGES);
}



// Prints a message on the upper left hand corner of the screen, pauses, and exits.
// Used in debugging.
void shout (char *str)
{
    TEXTPRINTF(screen, font, 0, 0, TEXTPRINTF_COL, "%s", str);
    PAUSE;
    exit(0);
}
void snum (int num)
{
    TEXTPRINTF(screen, font, 0, 0, TEXTPRINTF_COL, "%d", num);
    PAUSE;
    exit(0);
}
