// Sprite manipulation functions

#include "headers.h"

// Flip the image horizontally
void flip_horizontal (void)
{
    BITMAP *new_image;

    cycle_sprite_array(PREPARE_FOR_UNDO);
    
    new_image = create_bitmap (image[0]->w, image[0]->h);   // Create a new copy bitmap
    blit(image[0], new_image, 0, 0, 0, 0, image[0]->w, image[0]->h);    // Copy the image
    
    show_mouse (NULL);              // Hide the mouse cursor
    
    // Re-draw the sprite, flipped horizontally
    clear_to_color (image[0], TRANSPARENT_COL); 
    draw_sprite_h_flip(image[0], new_image, 0, 0);
    
    redraw_core(); 
    
    show_mouse(buffer);             // Show the mouse cursor

    destroy_bitmap(new_image);      // Destroy the copy bitmap
}



// Flip the image vertically
void flip_vertical (void)
{
    BITMAP *new_image;

    cycle_sprite_array(PREPARE_FOR_UNDO);
    
    new_image = create_bitmap (image[0]->w, image[0]->h);   // Create a new copy bitmap
    blit(image[0], new_image, 0, 0, 0, 0, image[0]->w, image[0]->h);    // Copy the image
    
    show_mouse (NULL);              // Hide the mouse cursor
    
    // Re-draw the sprite, flipped vertically
    clear_to_color (image[0], TRANSPARENT_COL);
    draw_sprite_v_flip(image[0], new_image, 0, 0);
    
    redraw_core(); 

    show_mouse(buffer);             // Show the mouse cursor

    destroy_bitmap(new_image);      // Destroy the copy bitmap
}



// Rotate the image 90 degrees
void rotate_90_degrees (void)
{
    int x1, y1, x2, y2, width, height, lmouse_down;
    BITMAP *new_image;

    // If the sprite's height is unequal to its width,
    if (image[0]->w != image[0]->h)  {

        // Position the window
        width = 10 + text_length(font, "A sprite can only be rotated 90 degrees");
        height = 10 + text_height(font) * 2;
        x1 = mouse_x - width / 2;       
        x2 = x1 + width - 1;
        y1 = mouse_y - 5;
        y2 = y1 + height - 1;

        if (x1 < 0) {                   // Keep the left edge of the window in bounds
            x2 -= x1;
            x1 = 0;
        }
        if (y1 < 0) {                   // Keep the right edge of the window in bounds
            y2 -= y1;
            y1 = 0;
        }
        if (x2 >= SCREEN_W) {           // Keep the right edge of the window in bounds
            x1 -= x2 - (SCREEN_W - 1);
            x2 = SCREEN_W - 1;
        }
        if (y2 >= SCREEN_H - 1) {       // Keep the bottom edge of the window in bounds
            y1 -= y2 - (SCREEN_H - 1);
            y2 = SCREEN_H - 1;
        }

        rectfill (buffer, x1, y1, x2, y2, BLACK);       // Draw the rectangles
        rect     (buffer, x1, y1, x2, y2, WHITE);

        TEXTPRINTF (buffer, font, x1 + 5, y1 + 5,  TEXTPRINTF_COL,  // Display the error msg
            "A sprite can only be rotated 90 degrees");
        TEXTPRINTF (buffer, font, x1 + 5, y1 + 5 + text_height(font), TEXTPRINTF_COL,
            "if its height is equal to its width!");
        
        copy_page(buffer, screen);  // Update the screen
        rest(1000);                 // Wait one second
        show_mouse(buffer);         // Reveal the mouse cursor

        // Init the left mouse down variable
        if (mouse_b & 1) lmouse_down = TRUE; else lmouse_down = FALSE;
        clear_keybuf();     // Clear the keyboard buffer

        // Wait until spacebar or enter keys are pressed, or left mouse button is clicked
        while (!key[KEY_ESC] && !key[KEY_SPACE] && !key[KEY_ENTER]) {
            if (!(mouse_b & 1)) {
                if (lmouse_down == TRUE) break;
            } else lmouse_down = TRUE;

            if (keyboard_needs_poll()) poll_keyboard();
            if (mouse_needs_poll()) poll_mouse();
            copy_page(buffer, screen);
        }

        show_mouse(NULL);           // Hide the mouse cursor
        redraw_core();
        show_mouse(buffer);         // Show the mouse cursor
        return;

    } else {    // If the sprite's height is equal to its width

        cycle_sprite_array(PREPARE_FOR_UNDO);

        new_image = create_bitmap (image[0]->w, image[0]->h);   // Create a copy bitmap
        blit(image[0], new_image, 0, 0, 0, 0, image[0]->w, image[0]->h);    // Copy image
    
        show_mouse (NULL);              // Hide the mouse cursor
    
        // Re-draw the sprite, rotated 90 degrees
        clear_to_color (image[0], TRANSPARENT_COL);
        rotate_sprite (image[0], new_image, 0, 0, itofix(64));

        destroy_bitmap(new_image);      // Destroy the copy bitmap
    }

    redraw_core();
    
    show_mouse(buffer);             // Show the mouse cursor
}



// Rotates the sprite by the specified angle
// If init == 0, start the rotation; 1, continue it; 2, end it
void rotate_spr (int init, int arg_angle)
{
    static BITMAP * new_image;

    static int cur_angle;

    if (init == 0)
    {
        cycle_sprite_array(PREPARE_FOR_UNDO);

        // Create a new bitmap and copy the current sprite onto it
        new_image = create_bitmap (image[0]->w, image[0]->h);

        blit (image[0], new_image, 0, 0, 0, 0, image[0]->w, image[0]->h);

        // Init the angle at 0
        cur_angle = 0;
    }
    else if (init == 1)
    {   
        show_mouse (NULL);

        // Increment the angle to rotate
        cur_angle += arg_angle;

        // Draw the rotated sprite
        clear_to_color (image[0], TRANSPARENT_COL);

        rotate_sprite (image[0], new_image, 0, 0, itofix (cur_angle));

        // Update the screen
        redraw_core();
    
        show_mouse (buffer);
    }
    else
    {
        destroy_bitmap (new_image);
    }
}



// Clears the sprite image to transparent color
void clear_sprite_image (void)
{
    cycle_sprite_array(PREPARE_FOR_UNDO);
    clear_to_color(image[0], TRANSPARENT_COL); // Clear the sprite
    redraw_core();
    show_mouse(buffer);         // Show the mouse cursor
}



// Shifts the sprite the specified number of pixels
// Only shifts the sprite x or y, not both!
void shift_sprite (int x_shift, int y_shift)
{
    BITMAP *shift_bmp;
    int x1, y1, x2, y2, width, height;
    char str[13] = "";
    int lmouse_down;

    if (x_shift == 0 && y_shift == 0) {     // Means "get values from user"

        x1 = mouse_x; y1 = mouse_y;
        width = 10 + text_length(font, "Move X:");      // Position the window
        height = 10 + text_height(font) * 2;
        x1 = mouse_x - width / 2;       
        x2 = x1 + width - 1;
        y1 = mouse_y - 10;
        y2 = y1 + height - 1;
    
        if (x1 < 0) {                   // Keep the left edge of the window in bounds
            x2 -= x1;
            x1 = 0;
        }
        if (y1 < 0) {                   // Keep the right edge of the window in bounds
            y2 -= y1;
            y1 = 0;
        }
        if (x2 >= SCREEN_W) {           // Keep the right edge of the window in bounds
            x1 -= x2 - (SCREEN_W - 1);
            x2 = SCREEN_W - 1;
        }
        if (y2 >= SCREEN_H - 1) {       // Keep the bottom edge of the window in bounds
            y1 -= y2 - (SCREEN_H - 1);
            y2 = SCREEN_H - 1;
        }
    
        // Draw the dialog window
        rectfill(buffer, x1, y1, x2, y2, BLACK);
        rect(buffer, x1, y1, x2, y2, WHITE);
        // Tell the user what to enter
        TEXTPRINTF(buffer, font, x1 + 5, y1 + 5, TEXTPRINTF_COL, "Move X:");
        strcpy(str, "1");   // Set the default string
        // Get the x shift parameter
        get_string (str, x1 + 5, y1 + 5 + text_height(font), str, 3);   
        x_shift = strtod (str, NULL);
        
        // Position the second window
        width = 10 + text_length(font, "Move Y:");      // Position the window
        height = 10 + text_height(font) * 2;
        x1 = mouse_x - width / 2;       
        x2 = x1 + width - 1;        
    
        if (x1 < 0) {                   // Keep the left edge of the window in bounds
            x2 -= x1;
            x1 = 0;
        }
        if (x2 >= SCREEN_W) {           // Keep the right edge of the window in bounds
            x1 -= x2 - (SCREEN_W - 1);
            x2 = SCREEN_W - 1;
        }       
    
        if (y2 + 10 + text_height(font) * 2 > SCREEN_H) {
            y2 = y1 - 5;
            y1 = y2 - (10 + text_height(font) * 2);
        } else {
            y1 = y2 + 5;
            y2 = y1 + (10 + text_height(font) * 2);
        }           
    
        // Draw the dialog window   
        rectfill(buffer, x1, y1, x2, y2, BLACK);
        rect(buffer, x1, y1, x2, y2, WHITE);
        // Tell the user what to enter
        TEXTPRINTF(buffer, font, x1 + 5, y1 + 5, TEXTPRINTF_COL, "Move Y:");
        strcpy(str, "1");   // Set the default string
        // Get the y shift parameter
        get_string (str, x1 + 5, y1 + 5 + text_height(font), str, 3);   
        y_shift = strtod (str, NULL);           

        // If the x or y shift parameters are out of bounds, tell the user so
        if (x_shift < -image[0]->w || x_shift > image[0]->w || 
            y_shift < -image[0]->h || y_shift > image[0]->h ||
            (x_shift == 0 && y_shift == 0))  {
            
            // Position the window
            width = 10 + text_length(font, "Can't resize sprite to (999, 999).");
            height = 10 + text_height(font) * 2;
            x1 = mouse_x - width / 2;       
            x2 = x1 + width - 1;
            y1 = y2 + 5;
            y2 = y1 + height - 1;
    
            if (x1 < 0) {                   // Keep the left edge of the window in bounds
                x2 -= x1;
                x1 = 0;
            }
            if (y1 < 0) {                   // Keep the right edge of the window in bounds
                y2 -= y1;
                y1 = 0;
            }
            if (x2 >= SCREEN_W) {           // Keep the right edge of the window in bounds
                x1 -= x2 - (SCREEN_W - 1);
                x2 = SCREEN_W - 1;
            }
            if (y2 >= SCREEN_H - 1) {       // Keep the bottom edge of the window in bounds
                y1 -= y2 - (SCREEN_H - 1);
                y2 = SCREEN_H - 1;
            }       

            rectfill (buffer, x1, y1, x2, y2, BLACK);       // Draw the rectangles
            rect     (buffer, x1, y1, x2, y2, WHITE);

            TEXTPRINTF (buffer, font, x1 + 5, y1 + 5,  TEXTPRINTF_COL,  // Display the error msg
                "Can't shift sprite (%d, %d).", x_shift, y_shift);
            TEXTPRINTF (buffer, font, x1 + 5, y1 + 5 + text_height(font), TEXTPRINTF_COL,
                "Maximum sprite shift is (%d, %d).", image[0]->w, image[0]->h);
            
            copy_page(buffer, screen);  // Update the screen
            rest(1000);                 // Wait one second
            show_mouse(buffer);         // Reveal the mouse cursor  
    
            // Init the left mouse down variable
            if (mouse_b & 1) lmouse_down = TRUE; else lmouse_down = FALSE;
            clear_keybuf();     // Clear the keyboard buffer
    
            // Wait until spacebar or enter keys are pressed, or left mouse button is clicked
            while (!key[KEY_ESC] && !key[KEY_SPACE] && !key[KEY_ENTER]) {
                if (!(mouse_b & 1)) {
                    if (lmouse_down == TRUE) break;
                } else lmouse_down = TRUE;  
    
                if (keyboard_needs_poll()) poll_keyboard();
                if (mouse_needs_poll()) poll_mouse();
                copy_page(buffer, screen);
            }   

            show_mouse(NULL);           // Hide the mouse cursor
            redraw_core();
            show_mouse(buffer);         // Show the mouse cursor
            return;
        }
    }
    
    cycle_sprite_array(PREPARE_FOR_UNDO);       // Prepare for an undo operation
    
    // Create a copy bitmap equal in size to the drawing image
    shift_bmp = create_bitmap(image[0]->w, image[0]->h);

    // blit(sprite, bmp, 0, 0, x, y, sprite->w, sprite->h)
    
    // Copy the shifted part of the drawing image to the copy bitmap
    if (x_shift > 0) {
        x1 = 0; y1 = 0;
        x2 = x_shift; y2 = 0;
    } else if (x_shift < 0) {
        x1 = -x_shift; y1 = 0;
        x2 = 0; y2 = 0;
    } else if (y_shift > 0) {
        x1 = 0; y1 = 0;
        x2 = 0; y2 = y_shift;
    } else if (y_shift < 0) {
        x1 = 0; y1 = -y_shift;
        x2 = 0; y2 = 0;
    }
    
    blit(image[0], shift_bmp, x1, y1, x2, y2, image[0]->w, image[0]->h);

    if (x_shift > 0) {
        x1 = image[0]->w - x_shift; y1 = 0;
        x2 = 0; y2 = 0;
        width = x_shift;
        height = image[0]->h;
    } else if (x_shift < 0) {
        x1 = 0; y1 = 0;
        x2 = image[0]->w - -x_shift; y2 = 0;
        width = -x_shift;
        height = image[0]->h;
    } else if (y_shift > 0) {
        x1 = 0; y1 = image[0]->h - y_shift;
        x2 = 0; y2 = 0;
        width = image[0]->w;
        height = y_shift;
    } else if (y_shift < 0) {
        x1 = 0; y1 = 0;
        x2 = 0; y2 = image[0]->h - -y_shift;
        width = image[0]->w;
        height = -y_shift;
    }
    
    blit(image[0], shift_bmp, x1, y1, x2, y2, width, height);
    
    // Copy the copy bitmap back to the original image
    blit(shift_bmp, image[0], 0, 0, 0, 0, image[0]->w, image[0]->h);

    // Destroy the shift bitmap
    destroy_bitmap(shift_bmp);
    
    // Refresh the sprite image
    refresh_image_grid();
}



// Enter new values for the sprite's size in pixels
void resize_sprite (void)
{
    static int x1, y1,
        x2, y2,
        width, height,
        lmouse_down,
        which_image,
        new_w, new_h;

    char str[13] = "";

    BITMAP *new_bmp;
    
    if (image[0]->w >= 100) {
        str[0] = '0' + image[0]->w / 100;
        str[1] = '0' + (image[0]->w / 10) % 10;
        str[2] = '0' + image[0]->w % 10;
        str[3] = '\0';
    } 
    else if (image[0]->w >= 10) {
        str[0] = '0' + image[0]->w / 10;
        str[1] = '0' + image[0]->w % 10;
        str[2] = '\0';
    }
    else {
        str[0] = '0' + image[0]->w % 10;
        str[1] = '\0';
    }
    
    width = 10 + text_length(font, "New width:");       // Position the window
    height = 10 + text_height(font) * 2;
    x1 = mouse_x - width / 2;       
    x2 = x1 + width - 1;
    y1 = mouse_y - 10;
    y2 = y1 + height - 1;

    if (x1 < 0) {                   // Keep the left edge of the window in bounds
        x2 -= x1;
        x1 = 0;
    }
    if (y1 < 0) {                   // Keep the right edge of the window in bounds
        y2 -= y1;
        y1 = 0;
    }
    if (x2 >= SCREEN_W) {           // Keep the right edge of the window in bounds
        x1 -= x2 - (SCREEN_W - 1);
        x2 = SCREEN_W - 1;
    }
    if (y2 >= SCREEN_H - 1) {       // Keep the bottom edge of the window in bounds
        y1 -= y2 - (SCREEN_H - 1);
        y2 = SCREEN_H - 1;
    }

    // Get the new width of the sprite
    rectfill (buffer, x1, y1, x2, y2, BLACK);
    rect (buffer, x1, y1, x2, y2, WHITE);

    TEXTPRINTF(buffer, font, x1 + 5, y1 + 5, TEXTPRINTF_COL, "New width:");
    
    get_string (str, x1 + 5, y1 + 5 + text_height(font), str, 3);   
    
    // Position the second window
    width = 10 + text_length(font, "New height:");      // Position the window
    height = 10 + text_height(font) * 2;
    x1 = mouse_x - width / 2;       
    x2 = x1 + width - 1;

    if (x1 < 0) {                   // Keep the left edge of the window in bounds
        x2 -= x1;
        x1 = 0;
    }
    if (x2 >= SCREEN_W) {           // Keep the right edge of the window in bounds
        x1 -= x2 - (SCREEN_W - 1);
        x2 = SCREEN_W - 1;
    }

    if (y2 + 10 + text_height(font) * 2 > SCREEN_H) {
        y2 = y1 - 5;
        y1 = y2 - (10 + text_height(font) * 2);
    } else {
        y1 = y2 + 5;
        y2 = y1 + (10 + text_height(font) * 2);
    }

    new_w =  strtod (str, NULL);
    if (image[0]->h >= 100) {
        str[0] = '0' + image[0]->h / 100;
        str[1] = '0' + (image[0]->h / 10) % 10;
        str[2] = '0' + image[0]->h % 10;
        str[3] = '\0';
    } 
    else if (image[0]->h>= 10) {
        str[0] = '0' + image[0]->h / 10;
        str[1] = '0' + image[0]->h % 10;
        str[2] = '\0';
    }
    else {
        str[0] = '0' + image[0]->h % 10;
        str[1] = '\0';
    }

    // Get the new height of the sprite
    rectfill(buffer, x1, y1, x2, y2, BLACK);
    rect(buffer, x1, y1, x2, y2, WHITE);

    TEXTPRINTF(buffer, font, x1 + 5, y1 + 5, TEXTPRINTF_COL, "New height:");
    get_string (str, x1 + 5, y1 + 5 + text_height(font), str, 3);   
    new_h = strtod (str, NULL);

    // If the new width and height are too small, tell the user so
    if (new_w < 1 || new_h < 1)
    {
        // Position the window
        width = 10 + text_length(font, "Can't resize sprite to (999, 999).");
        height = 10 + text_height(font);
        x1 = mouse_x - width / 2;       
        x2 = x1 + width - 1;
        y1 = y2 + 5;
        y2 = y1 + height - 1;

        if (x1 < 0) {                   // Keep the left edge of the window in bounds
            x2 -= x1;
            x1 = 0;
        }
        if (y1 < 0) {                   // Keep the right edge of the window in bounds
            y2 -= y1;
            y1 = 0;
        }
        if (x2 >= SCREEN_W) {           // Keep the right edge of the window in bounds
            x1 -= x2 - (SCREEN_W - 1);
            x2 = SCREEN_W - 1;
        }
        if (y2 >= SCREEN_H - 1) {       // Keep the bottom edge of the window in bounds
            y1 -= y2 - (SCREEN_H - 1);
            y2 = SCREEN_H - 1;
        }

        rectfill (buffer, x1, y1, x2, y2, BLACK);       // Draw the rectangles
        rect     (buffer, x1, y1, x2, y2, WHITE);

        TEXTPRINTF (buffer, font, x1 + 5, y1 + 5,  TEXTPRINTF_COL,  // Display the error msg
            "Can't resize sprite to (%d, %d).", new_w, new_h);
        
        copy_page(buffer, screen);  // Update the screen
        rest(1000);                 // Wait one second
        show_mouse(buffer);         // Reveal the mouse cursor

        // Init the left mouse down variable
        if (mouse_b & 1) lmouse_down = TRUE; else lmouse_down = FALSE;
        clear_keybuf();     // Clear the keyboard buffer

        // Wait until spacebar or enter keys are pressed, or left mouse button is clicked
        while (!key[KEY_ESC] && !key[KEY_SPACE] && !key[KEY_ENTER]) {
            if (!(mouse_b & 1)) {
                if (lmouse_down == TRUE) break;
            } else lmouse_down = TRUE;

            if (keyboard_needs_poll()) poll_keyboard();
            if (mouse_needs_poll()) poll_mouse();
            copy_page(buffer, screen);
        }

        show_mouse(NULL);           // Hide the mouse cursor
        redraw_core(); 
        show_mouse(buffer);         // Show the mouse cursor
        return;
    }

    
    // Create a new bitmap equal in size to the sprite and copy the sprite to it
    new_bmp = create_bitmap (image[0]->w, image[0]->h);
    blit (image[0], new_bmp, 0, 0, 0, 0, image[0]->w, image[0]->h);
    
    // Destroy all the undo images and re-create them
    for (which_image = 0; which_image < NUM_IMAGES; which_image++)
    {
        destroy_bitmap (image[which_image]);
        image[which_image] = create_bitmap (new_w, new_h);

        if (image[which_image] == NULL)
        {
            allegro_message ("Error creating one of the undo images!");
            exit (1);
        }

        clear_to_color (image[which_image], TRANSPARENT_COL);
    }

    // Destroy, re-create and initialize the buffer image
    destroy_bitmap (buffer_image);
    buffer_image = create_bitmap (new_w, new_h);
    clear_to_color (buffer_image, TRANSPARENT_COL);
    
    // Copy the old image to the new one
    blit (new_bmp, image[0], 0, 0, 0, 0, new_bmp->w, new_bmp->h);
    
    init_grid_cell_size ();

    redraw_core(); 
    show_mouse(buffer);         // Show the mouse cursor
}
  


// Copy the image, depending on the type specified
void mirror_image (int type, int mouse_x_cell, int mouse_y_cell)
{
	static BITMAP * tmp;
	static int median_width, median_height,
		x1, y1, x2, y2;
	
	median_width = image[0]->w / 2;
	median_height = image[0]->h / 2;
	
	// Copy the bitmap
	tmp = create_bitmap (image[0]->w, image[0]->h);
	blit  (image[0], tmp, 0, 0, 0, 0, image[0]->w, image[0]->h);
	
	if (type == 1) // Mirror horizontally
	{
		flip_horizontal ();
		
		y1 = 0;
		y2 = 0;
		
		if (mouse_x_cell < median_width)
		{
			x1 = 0;
			x2 = 0;
		}
		else
		{
			x1 = median_width;
			x2 = median_width;
		}
		
		blit (tmp, image[0], x1, y1, x2, y2, median_width, image[0]->h);
	}
	else if (type == 2) // Mirror vertically
	{
		flip_vertical ();
		
		x1 = 0;
		x2 = 0;
		
		if (mouse_y_cell < median_height)
		{
			y1 = 0;
			y2 = 0;
		}
		else
		{
			y1 = median_height;
			y2 = median_height;
		}
		
		blit (tmp, image[0], x1, y1, x2, y2, image[0]->w, median_height);
	}
	
	redraw_core ();
	
	destroy_bitmap (tmp);
}



// Draw a border around the sprite (black for now)
void draw_border (void)
{
	static int x, y,
		cur_pixel,
		prev_pixel,
		border_col;
	
	border_col = BLACK;		// For now, just draw a black border

	// Draw the border for the top and bottom edges
	for (x = 0; x < image[0]->w; x++)
	{
		prev_pixel = TRANSPARENT_COL;
		for (y = 0; y < image[0]->h; y++)
		{
			cur_pixel = getpixel (image[0], x, y);
			
			if (prev_pixel == TRANSPARENT_COL && cur_pixel != TRANSPARENT_COL &&
				cur_pixel != border_col && y > 0)
			{
				putpixel (image[0], x, y - 1, border_col);
			}

			if (cur_pixel == TRANSPARENT_COL && prev_pixel != TRANSPARENT_COL &&
				prev_pixel != border_col)
			{
				putpixel (image[0], x, y, border_col);
			}
			
			prev_pixel = cur_pixel;
		}
	}		

	// Draw the border for the left and right edges
	for (y = 0; y < image[0]->h; y++)
	{
		prev_pixel = TRANSPARENT_COL;
		for (x = 0; x < image[0]->w; x++)
		{
			cur_pixel = getpixel (image[0], x, y);
			
			if (prev_pixel == TRANSPARENT_COL && cur_pixel != TRANSPARENT_COL &&
				cur_pixel != border_col && x > 0)
			{
				putpixel (image[0], x - 1, y, border_col);
			}

			if (cur_pixel == TRANSPARENT_COL && prev_pixel != TRANSPARENT_COL &&
				prev_pixel != border_col)
			{
				putpixel (image[0], x, y, border_col);
			}
			
			prev_pixel = cur_pixel;			
		}
	}
}
