/*
Author:         Daniel Tenedorio
Date Started:   Sept. 4, 2005
Date Finished:  Sept. 25, 2005

Maintainence edits: 
	May 4, 2006 -   Added icon support, fixed menus, choppy mouse
	May 27, 2006 -  Added support for mice without a wheel
    Jun 3, 2006 -   Added support to save the current image chopped up into
                        tiles by saving as "(width,height)basename"
    Aug 15, 2006 -  Added magnification tool. Implemented better color picker.

Description: Basic Sprite Editor V2.0
    Supports creating sprites in true colors

Contact: E-mail materialism@gmail.com for comments, questions,
    bug reports, etc.
*/

#include "headers.h"

int main (int argc, char *argv[])
{
//  DATAFILE *dat;              // datafile object that holds the datafile info
//  FONT *game_font;            // font object to hold the nifty game font
//  FONT *title_font;           // font object to hold the nifty game font

    filename_was_passed = FALSE;

    if (argc > 2) {
        fprintf(stderr, "Usage:\n\tbse2\n\tbse2 imagefilename.bmp");
        exit(1);
    }
    else if (argc == 2) {   // If one argument was passed to this function
        if (strlen(argv[1]) >= 2000) {
            fprintf(stderr, "Filename argument cannot be longer than %d!\n", 2000);
        }
        else {
            strcpy(passed_filename, argv[1]);
            filename_was_passed = TRUE;
        }
    }

    // 640x480x16 bit colors (for now?)
    init_beginning(DEFAULT_SCREEN_W, DEFAULT_SCREEN_H, DEFAULT_COLOR_DEPTH);

    while (TRUE)
    {
        respond_to_keyboard_input ();
        respond_to_mouse_input ();
        update_info_area();
        copy_page(buffer, screen);                          // Update the screen
        rest(5);
    }

    return 0;
}

END_OF_MAIN ()              // Needed for allegro library


/*
Initializes the keyboard, mouse, timer, graphics mode, default directory,
and text mode.
*/
void init_beginning (int screen_x_size, int screen_y_size, int color_depth)
{
    int index;

    strcpy (cur_filename, "");
    pbrush_size = 2;
    num_icon_files = NUM_ICON_FILES;
    
    allegro_init();
    install_keyboard();
    install_timer();
    install_mouse();
    
    // Init graphics mode
    set_color_depth (color_depth);      // Value of 16 sets 65536 color mode    

    screen_type = GFX_AUTODETECT_FULLSCREEN;        // Set fullscreen display
    grid_activated = TRUE;                          // Set visible grid
    background_style = BLACK_STYLE;                 // Set black background pixels
    color_strips_on = TRUE;                         // Use color strips
    solid_paste = FALSE;                            // Use transparent paste

    load_config_file(&screen_x_size, &screen_y_size);   // Load info from the config file

    if (set_gfx_mode(screen_type, screen_x_size, screen_y_size, 0, 0) != 0) {
        allegro_message("Error setting graphics mode\n%s\n", allegro_error);
        exit (1);
    }

    drawing_color_base = RED;

    buffer = create_bitmap(SCREEN_W, SCREEN_H); // Create the video buffer
    clear_bitmap (buffer);                      // Clear the buffer to color 0
    clear_bitmap (screen);                      // Clear the screen to color 0
    SET_SOLID_TEXT;

    // Create the image bitmaps
    for (index = 0; index < NUM_IMAGES; index++) {
        image[index] = create_bitmap(DEFAULT_SPRITE_WIDTH, DEFAULT_SPRITE_HEIGHT);
        clear_to_color(image[index], TRANSPARENT_COL);  // Clear the image to transparent
    }
    // Create and initialize the buffer image
    buffer_image = create_bitmap(DEFAULT_SPRITE_WIDTH, DEFAULT_SPRITE_HEIGHT);
    clear_to_color(buffer_image, TRANSPARENT_COL);

    // Create a dummy copy image bitmap
    copy_image = create_bitmap(1, 1);

    load_icons();

    cur_tool = 0;                           // Init the current drawing tool to "pencil"

    draw_color = 0;                         // Init the current drawing color

    for (index = 0; index < MAX_CUR_COLORS; index++)    // Init the current colors array
    {
        cur_color[index] = TRANSPARENT_COL;
    }

    cur_color[0] = WHITE;                       // Init the first current color to white
    separate_cur_color = cur_color[0];          // Init separate current color to above

    init_grid_cell_size ();

    prev_x_cell = -1;       // Init the previous cell to -1 so the drawing functions 
    prev_y_cell = -1;       // know to start a new one

    drawing_operation = FALSE;      // Use image[0], not buffer_image
    
    set_mouse_sprite_focus(0, 0);   // Set what spot on the arrow actually points
    
    // If a filename to load was passed as a cmd line argument
    if (filename_was_passed == TRUE)
    {
        if (load_image_from_file(passed_filename) == FALSE)
        {
            fprintf(stderr, "Could not load \"%s\"!\n", passed_filename);
        }
    }
    // If the filename was saved from before
    else if (strlen(cur_filename) > 0)
    {
        // Load the file; if fail, clear the current filename
        if (load_image_from_file(cur_filename) == FALSE)
        {
            strcpy(cur_filename, "");
        }
    }

    magnified = FALSE;
	mirror_horizontal = FALSE;
	mirror_vertical = FALSE;
    
    show_mouse(NULL);

    redraw_core();
    display_splash_screen();
    redraw_core();

    show_mouse(buffer);
}



//
// Load the icon image files from the folder titled "icons" in the current
// working directory. If the folder or any of the files does not exist, icons
// are disabled.
//
void load_icons (void) 
{
    char filename[512];
    int which_file;
    
    char *icon_folder_name = "icons/";

    char log_filename[100];
    FILE *fp_out; 
    
    strcpy(log_filename, icon_folder_name);
    strcat(log_filename, "load_log.txt");

    if ((fp_out = fopen (log_filename, "w")) == NULL) {
		
		icons_enabled = FALSE;
        return;
	}

    fprintf (fp_out, "Loading icon images...\n");

    for (which_file = 0; which_file < NUM_ICON_FILES; which_file++) {

        // Determine the filename of the icon to load
        strcpy (filename, icon_folder_name);
        strcat (filename, icon_filenames[which_file]);
        fix_filename_slashes (filename);

        // Get info about the file and load the bitmap. If an error occurs,
        // print the error message to the log file and exit.
        errno = 0;
        if ((icon_images[which_file] = load_bitmap (filename, NULL)) == NULL) {

            icons_enabled = FALSE;

            // Print an error message to the log file
            fprintf (fp_out, "An error occurred loading file %s\n", filename);
            if (errno != 0) {
                fprintf (fp_out, "%s\n", strerror(errno));
            }
            fclose(fp_out);
            return;
        }
    }
    
    fclose (fp_out);
    remove (log_filename);
}



// Loads the game's palette into the pal object from the given filename
void load_palette (PALETTE *pal, char *filename)
{
    int io_results;
    FILE *fp_in;

    fp_in = fopen("palette.dat", "rb");     // Open the file for read binary access
    io_results = fread (pal, sizeof(*pal), 1, fp_in);   // Read the pal info
    
    if (io_results == 0)  {             // If the load failed, show error message
        allegro_message("Error loading palette from galxyinv.pal\n%s\n", 
            allegro_error);
        exit (1);                       // ...and quit the program
    }
    fclose (fp_in);                 // Close the file
}



// Loads the sprites from the files listed here into the sprite array
void load_sprites (BITMAP *sprites[])
{
    int which_file;
    char *filenames[NUM_SPRITES];

    // WARNING: This function will load the number of sprites specified by
    // NUM_SPRITES. Provide exactly as many valid filenames as NUM_SPRITES!
    // If NUM_SPRITES = 3, then there should be 3 lines of code here, etc.
    filenames[0] =  "test.bmp";

    for (which_file = 0; which_file < NUM_SPRITES; which_file++) {
        
        // Load the bitmap file into its corresponding sprite[] entry
        sprites[which_file] = load_bitmap(filenames[which_file], NULL);

        if (sprites[which_file] == NULL) {      // If the sprite did not load
            allegro_message("Error loading sprite #%d from file \"%s\"", 
                which_file, filenames[which_file]);
            exit (1);                           // Exit with error message
        }
    }
}



// Load options from the local configuration file
void load_config_file (int *screen_x, int *screen_y)
{
    int val[7];
    char str[2000];
    char filename[2000];
    
    FILE *fp_in;

    // Create the file if it does not exist
    fp_in = fopen("bse.cfg", "a"); fclose(fp_in);
    
    fp_in = fopen("bse.cfg", "r");      // Open the file for input

    if (fgets(str, 2000, fp_in) != NULL) {      // If the file is not empty

        fgets (str, 2000, fp_in);                // Read the info
        fgets (str, 2000, fp_in);
        
        fgets (filename, 2000, fp_in);
        fgets (str, 2000, fp_in);

        // Get the information from the last line of text
        sscanf(str, "%d, %d, %d, %d, %d, %d, %d", &val[0], &val[1], &val[2],
            &val[3], &val[4], &val[5], &val[6]);

        filename[strlen(filename) - 1] = '\0';  // Remove the trailing newline

        // If any of the information was modified
        if (val[0] < 640 || val[1] < 480 || val[0] > 1280 || val[1] > 1024 ||
            (val[2] != GFX_AUTODETECT_FULLSCREEN && val[2] != GFX_AUTODETECT_WINDOWED) || 
            (val[3] != TRUE && val[3] != FALSE) || 
            (val[4] != BLACK_STYLE && val[4] != CHECKER_STYLE) ||
            (val[5] != TRUE && val[5] != FALSE) || 
            (val[6] != TRUE && val[6] != FALSE))
        {
            // Do nothing
        }
        else
        {    // Set the values
            *screen_x = val[0];
            *screen_y = val[1];
            screen_type = val[2];
            grid_activated = val[3];
            background_style = val[4];
            color_strips_on = val[5];
            solid_paste = val[6];
            strcpy(cur_filename, filename);
        }
    }

    fclose(fp_in);                          // Close the file

}



//
// Scans for certain keypresses. If the keys are pressed, execute the
// appropriate actions.
//
// Currently used keys:
//      Number keys 0-9:        Select drawing tool
//      R:                      Rotate sprite 90 degrees
//      Ctrl+R:                 Resize current sprite
//      G:                      Toggle drawing grid
//      B:                      Toggle background color
//      T:                      View sprite tiled
//      P:                      Toggle transparent/solid paste
//      Q:                      Quit
//      K:                      Draw a black border around the sprite
//      Z, X:                   Change drawing tool
//      S, L:                   Save/load sprite
//		C, V:					Rotate sprite slowly
//		H, J:					Toggle horizontal/vertical mirroring
//      U/Ctrl+Z:               Undo
//      F1:                     Show help mode
//      F2:                     Save a screenshot
//      F5:                     Toggle fullscreen/window
//      Backspace/del.:         Erase current sprite to transparent col.
//      Arrow keys:             Move sprite one pixel
//      Ctrl+Arrow keys:        Flip sprite horizontally or vertically
//      Esc:                    Show main menu
//      Comma/Period:           Lighten/darken drawing color
//      >/<:                    Switch to previous/next image in the directory
//
void respond_to_keyboard_input ()                             
{
    //
    // Hold the state of keys between function calls so we can act on each
    // keypress instead of continuously.
    //
    static int esc_down, f5_down, bkspc_down, u_down,
        up_down, left_down, r_down, g_down, b_down, t_down, p_down, x_down,
        z_down, comma_down, stop_down, c_down, v_down, key_c, key_v, h_down,
		j_down, k_down;

    int prev_cur_tool;

    if (keyboard_needs_poll()) poll_keyboard();
   
    prev_cur_tool = cur_tool; 

    if (key[KEY_1]) cur_tool = 0;   // Set "Pencil" tool
    if (key[KEY_2]) cur_tool = 1;   // Set "Brush" tool
    if (key[KEY_3]) cur_tool = 2;   // Set "Line" tool
    if (key[KEY_4]) cur_tool = 3;   // Set "Rect." tool
    if (key[KEY_5]) cur_tool = 4;   // Set "Box" tool
    if (key[KEY_6]) cur_tool = 5;   // Set "Oval" tool
    if (key[KEY_7]) cur_tool = 6;   // Set "Fill" tool
    if (key[KEY_8]) cur_tool = 7;   // Set "Copy" tool
    if (key[KEY_9]) cur_tool = 8;   // Set "Cut" tool
    if (key[KEY_0]) cur_tool = 9;   // Set "Paste" tool

    if (cur_tool != prev_cur_tool) draw_icons();

	if (key[KEY_R])
    {                   
        if (r_down == FALSE)
        {
			r_down = TRUE;
            
            if (key[KEY_LCONTROL])
            {
                resize_sprite ();
            }
            else
            {
			    rotate_90_degrees();        // Rotate the sprite 90 degrees
			    refresh_image_grid();       // Re-draw the sprite and grid
            }
		}
	}
	else r_down = FALSE;

	if (key[KEY_G]) {                   // If Ctrl-G is pressed
        if (g_down == FALSE) {          // and it was previously up
			g_down = TRUE;              // Tell the function that it is now down
			toggle_grid();              // Toggle the grid
			show_mouse(NULL);           // Hide the mouse cursor
			redraw_core();
			show_mouse(buffer);         // Show the mouse cursor
		}
	}
	else g_down = FALSE;

	if (key[KEY_B]) {                   // If Ctrl-B is pressed
        if (b_down == FALSE) {          // and it was previously up
			b_down = TRUE;              // Tell the function that it is now down
			toggle_background();        // Toggle background style
			show_mouse(NULL);           // Hide the mouse cursor
			redraw_core();
			show_mouse(buffer);         // Show the mouse cursor
		}
	}
	else b_down = FALSE;

	if (key[KEY_T]) {                   // If Ctrl-T is pressed
        if (t_down == FALSE) {          // and it was previously up
			t_down = TRUE;              // Tell the function that it is now down
			show_mouse(NULL);           // Hide the mouse cursor
			view_sprite_tiled();        // View the sprite tiled
		}
	}
	else t_down = FALSE;

	if (key[KEY_P]) {                   // If Ctrl-P is pressed
        if (p_down == FALSE) {          // and it was previously up
			p_down = TRUE;              // Tell the function that it is now down
			toggle_solid_paste();       // Toggle between solid and transparent paste
		}
	}
	else p_down = FALSE;

	if (key[KEY_Q]) {                   // If Ctrl-Q pressed
		save_config_file();             // Save settings to a local configuration file
		exit(0);                        // Exit the program
	}
    
    if (key[KEY_Z]) {
        if (z_down == FALSE) {
            z_down = TRUE;
            if (key[KEY_LCONTROL])
            {
                cycle_sprite_array (EXECUTE_UNDO);   // Execute an undo
                refresh_image_grid();
            }
            else if (cur_tool > 0)
            {
                 cur_tool--;
                 draw_icons();
            }
        }
    }
    else z_down = FALSE;

    if (key[KEY_X]) {
        if (x_down == FALSE) {
            x_down = TRUE;
             if (cur_tool < 9) {
                 cur_tool++;
                 draw_icons();
             }
        }
    }
    else x_down = FALSE;

    if (key[KEY_S]) save_image_to_file();

    if (key[KEY_L]) {
        if (load_image_from_file("") == FALSE) {
            allegro_message("Error loading image file: %s\n%s", load_fail_filename,
                allegro_error);             
            clear_keybuf();                 // Clear keyboard buf
            strcpy(cur_filename, "");       // Empty the current filename
        }
    }

    if (key[KEY_F2]) {
        save_bmp ("screenshot.bm_", buffer, NULL);
    }
    
    if (key[KEY_BACKSPACE] || key[KEY_DEL]) {   // If Backspace or Del keys are pressed
        if (bkspc_down == FALSE) {              // and it was previously up
            bkspc_down = TRUE;                  // Tell the function that it is now down
            clear_sprite_image();           // Clear the sprite image to transparent color
        }
    }
    else bkspc_down = FALSE;

    if (key[KEY_U]) {                       // If  "U" key pressed
        if (u_down == FALSE) {              // and it was previously up
            u_down = TRUE;                  // Tell the function that it is now down
            cycle_sprite_array (EXECUTE_UNDO);   // Execute an undo
            refresh_image_grid();           // Re-draw the sprite and grid
        }
    }
    else u_down = FALSE;

    if (key[KEY_UP] || key[KEY_DOWN]) {
        if (up_down == FALSE) {             // and it was previously up
            up_down = TRUE;                 // Tell the function that it is now down
            if (key[KEY_LCONTROL]) {        // If the left Ctrl key is pressed
                flip_vertical();            // Flip the sprite vertically
                refresh_image_grid();       // Re-draw the sprite and grid
            } else {                        // If the left Ctrl key is NOT pressed
                if (key[KEY_UP]) shift_sprite(0, -1);       // Move up one pixel
                else if (key[KEY_DOWN]) shift_sprite(0, 1); // Move down one pixel
            }
        }
    }
    else up_down = FALSE;
    
    if (key[KEY_LEFT] || key[KEY_RIGHT]) {  // If Ctrl-left or Ctrl-right are pressed
        if (left_down == FALSE) {           // and it was previously up
            left_down = TRUE;               // Tell the function that it is now down
            if (key[KEY_LCONTROL]) {        // If the left Ctrl key is pressed
                flip_horizontal();          // Flip the sprite horizontally
                refresh_image_grid();       // Re-draw the sprite and grid
            } 
            else                            // If the left Ctrl key is NOT pressed
            {                        
                if (key[KEY_LEFT]) shift_sprite(-1, 0);         // Move one pixel left
                else if (key[KEY_RIGHT]) shift_sprite(1, 0);    // Move one pixel right
            }
        }
    }
    else left_down = FALSE;

    if (key[KEY_F1]) {                      // If F1 key is pressed
        display_help();                     // Show the help dialog
    }

    if (key[KEY_F5]) {                      // If F5 is pressed
        if (f5_down == FALSE) {             // and it was previously up
            f5_down = TRUE;                 // Tell the function that it is now down
            toggle_fullscreen_window();     // Toggle between fullscreen and window mode    
        }
    }
    else f5_down = FALSE;

    if (key[KEY_ESC]) {                     // If Esc pressed
        if (esc_down == FALSE) {            // and was previously up
            esc_down = TRUE;                // Tell the function that it is now down
            main_menu();                    // Esc key action
        }
    }
    else esc_down = FALSE;

	if (key[KEY_COMMA])
    {
        if (comma_down == FALSE)
        {
            comma_down = TRUE;

            if (key[KEY_LSHIFT])
            {
                load_image_from_file ("<");
            }
            else
            {
                increment_drawing_color (-1);
                refresh_cur_color_boxes();
            }
        }
	}
    else
    {
        comma_down = FALSE;
    }

	if (key[KEY_STOP])
	{
        if (stop_down == FALSE)
        {
            stop_down = TRUE;

            if (key[KEY_LSHIFT])
            {
                load_image_from_file (">");
            }
            else
            {
                increment_drawing_color (1);
                refresh_cur_color_boxes();
            }
        }
	}
    else
    {
        stop_down = FALSE;
    }
	
	// Toggle horizontal/vertical mirroring
	if (key[KEY_H])
	{
		if (h_down == FALSE)
		{
			h_down = TRUE;
			
			toggle_mirror_horizontal ();
			redraw_core ();
		}
	}
	else
	{
		h_down = FALSE;
	}

	if (key[KEY_J])
	{
		if (j_down == FALSE)
		{
			j_down = TRUE;
			
			toggle_mirror_vertical ();
			redraw_core ();
		}
	}
	else
	{
		j_down = FALSE;
	}
	
	// Draw a border around the sprite
	if (key[KEY_K])
	{
		if (k_down == FALSE)
		{
			k_down = TRUE;
			
			draw_border ();
			redraw_core ();
		}
	}
	else
	{
		k_down = FALSE;
	}
	
    // Rotate the sprite left or right
    key_c = key[KEY_C];
    key_v = key[KEY_V];

    if (!key_v)
    {
        if (key_c)
        {
            // If the key wasn't down, start the rotation. If it was, continue it.
            if (c_down == FALSE)
            {
                c_down = TRUE;
                rotate_spr (0, -1);
            }
            else
            {
                rotate_spr (1, -1);
            }
        }
        else
        {
            // If the key was down, end the rotation.
            if (c_down == TRUE) rotate_spr (2, -1);

            c_down = FALSE;
        }
    }
    
    if (!key_c)
    {
        if (key_v)
        {
            // If the key wasn't down, start the rotation. If it was, continue it.
            if (v_down == FALSE)
            {
                v_down = TRUE;
                rotate_spr (0, 1);
            }
            else
            {
                rotate_spr (1, 1);
            }
        }
        else
        {
            // If the key was down, end the rotation.
            if (v_down == TRUE) rotate_spr (2, 1);

            v_down = FALSE;
        }
    }
}



// Check and respond to mouse clicks, etc.
//
// The mouse buttons perform the following actions:
// Left click in sprite grid:           Use current drawing tool
// Ctrl + Left click in sprite grid:    Use copy tool
// Alt + Left click in sprite grid:     Use magnify tool
// Left click in color grid:            Change current drawing color
// Left click on drawing tool icons:    Change current drawing tool
// Left click on current color palette: Change current drawing color
// Right click in sprite grid:          Use highlighted drawing color
// Right click in color grid:           Toggle color grid modes
// Roll mouse wheel:                    Brighten/darken drawing color
// Shift + Roll mouse wheel:            Change current drawing tool
//
void respond_to_mouse_input ()
{
    static int lmouse_down, rmouse_down,
        mouse_x_cell, mouse_y_cell,
        color_over,
        index,
        width, height,
        x, y,
        which_icon,
        mouse_x_pixel, mouse_y_pixel;

    if (mouse_needs_poll()) poll_mouse();

    mouse_x_pixel = mouse_x;
    mouse_y_pixel = mouse_y;

    // Determine the cell the mouse pointer is hovering over
    mouse_x_cell = mouse_x_pixel / grid_cell_size;
    mouse_y_cell = mouse_y_pixel / grid_cell_size;

    if (mouse_x_cell < 0) mouse_x_cell = 0;     // Keep the cell in bounds
    if (mouse_y_cell < 0) mouse_y_cell = 0;

    if (mouse_x_cell > image[0]->w - 1) mouse_x_cell = image[0]->w - 1;
    if (mouse_y_cell > image[0]->h - 1) mouse_y_cell = image[0]->h - 1;

    if (mouse_b & 1) {  // Left mouse button is pressed

        // Mouse pointer is over the icons
        if (mouse_x_pixel > SCREEN_W - 256 && mouse_y_pixel > ICON_AREA_TOP &&
			mouse_y_pixel < ICON_AREA_TOP + ICONS_HEIGHT)
        {    
			if (icons_enabled == TRUE)
            {
                which_icon = (mouse_x_pixel - (SCREEN_W - 256)) /
					(icon_images[0]->w + 1);
				
                if (which_icon >= 0 && which_icon < 10)
                {
                    cur_tool = which_icon;
                    show_mouse (NULL);
                    draw_icons();
                    show_mouse (buffer);
	            }
            }
        }
        // Mouse pointer is over the color grid area
        else if (mouse_x_pixel > SCREEN_W - 256 && mouse_y_pixel <
            ICON_AREA_TOP) {

            show_mouse (NULL);
            color_over = getpixel (buffer, mouse_x_pixel, mouse_y_pixel);

            if (!color_strips_on && mouse_x_pixel >= SCREEN_W -
                COLOR_SLIDER_WIDTH)
            {
                drawing_color_base = color_over;

                refresh_colors_main_square ();
            }
            else if (!color_strips_on && mouse_x_pixel >= SCREEN_W -
                COLOR_SLIDER_WIDTH - 5)
            {

            }
            else
            {
                index = add_current_color (color_over);

                if (index == -1)
                {
                    // Copy the new draw color to the separate current color box
                    separate_cur_color = cur_color[0];
                }
                else
                {
                    // Copy the new draw color to the separate current color box
                    separate_cur_color = cur_color[draw_color];
                }

                refresh_cur_color_boxes();

                // Left mouse button was previously released
                if (lmouse_down == FALSE)
                {
                    lmouse_down = TRUE;        // It is now pressed

                    // Restrict mouse movement to the color grid area
                    set_mouse_range(SCREEN_W - 256, 0, SCREEN_W, SCREEN_H);
                }
            }

            show_mouse(buffer);                            // Show mouse pointer
        }
        // Mouse pointer is over the sprite grid
        else if (mouse_x_pixel < SCREEN_W - 256)
        {
            //
            // If the control key is also pressed, set the "copy"
            // drawing tool. Ditto for alt and "magnify".
            //
            if (key[KEY_LCONTROL])
            {
                cur_tool = 7;
            }
            else if (key[KEY_ALT])
            {
                if (cur_tool != 8) prev_tool = cur_tool;
                cur_tool = 8;
            }

            // Left mouse button was previously released
            if (lmouse_down == FALSE)
            {
                lmouse_down = TRUE;        // It is now pressed
                
                // Restrict mouse movement to the sprite grid area
                set_mouse_range(0, 0, image[0]->w * grid_cell_size - 1,
                    image[0]->h * grid_cell_size - 1);

                cycle_sprite_array(PREPARE_FOR_UNDO);   // Prepare for undo

                switch (cur_tool)
                {
                case 8:     // Magnify

                    if (magnified == TRUE)
                    {
						restore_magnified_area ();
						break;
                    }
					
                case 2:     // Line
                case 3:     // Rect.
                case 4:     // Box
                case 5:     // Oval
                case 7:     // Copy

                    prev_x_cell = mouse_x_cell;         // Set the starting point
                    prev_y_cell = mouse_y_cell;
                    drawing_operation = TRUE;           // Use buffer_image, not image[0]

                    // Copy the real image to the temporary drawing one
                    blit (image[0], buffer_image, 0, 0, 0, 0, image[0]->w, image[0]->h);

                    break;

                case 6:     // Fill

                    floodfill (image[0], mouse_x_cell, mouse_y_cell, separate_cur_color);
                    refresh_image_grid();

                    break;

                case 9:     // Paste

                    if (copy_image == NULL) break;

                    if (solid_paste)
                    {
                        // Copy directly from the copy bitmap to the drawing image
                        blit (copy_image, image[0], 0, 0, mouse_x_cell, mouse_y_cell, 
                            copy_image->w, copy_image->h);
                    }
                    else 
                    {
                        // Transparent paste
                        draw_sprite(image[0], copy_image, mouse_x_cell, mouse_y_cell);
                    }

                    refresh_image_grid();           // Redraw the sprite

                    break;
                }
            }
            else        // Mouse button is pressed and it was pressed before
            {
                 // Make sure the current color is added
                add_current_color (separate_cur_color);

                switch (cur_tool)
                {
                case 0:     // "Pencil" or "Brush" tool
                case 1:
                                        
                    // If the previous cell location is uninitialized
                    if (prev_x_cell == -1 || prev_y_cell == -1)
                    {
                        // Set it to the current cell location
                        prev_x_cell = mouse_x_cell;
                        prev_y_cell = mouse_y_cell;
                    }
   
                    // 
                    // Draw a line on the bitmap from the previous cell location to 
                    // the current location
                    //
                    line (image[0], prev_x_cell, prev_y_cell, mouse_x_cell, mouse_y_cell,
                        separate_cur_color);
                    
                    if (cur_tool == 1)    // If the "Brush" drawing tool is set
                    {
                        line (image[0], prev_x_cell + 1, prev_y_cell, mouse_x_cell + 1, 
                            mouse_y_cell, separate_cur_color);
                        line (image[0], prev_x_cell, prev_y_cell + 1, mouse_x_cell, 
                            mouse_y_cell + 1, separate_cur_color);
                        line (image[0], prev_x_cell + 1, prev_y_cell + 1, mouse_x_cell + 1,
                            mouse_y_cell + 1, separate_cur_color);
                    }

                    refresh_image_grid();

                    // Reset the previous cell location
                    prev_x_cell = mouse_x_cell;
                    prev_y_cell = mouse_y_cell;

                    break;

                case 2:     // "Line" tool

                    // Copy the real image to the temporary drawing one
                    blit (image[0], buffer_image, 0, 0, 0, 0, image[0]->w, image[0]->h);

                    // Draw a line on the temporary drawing image
                    line (buffer_image, prev_x_cell, prev_y_cell, mouse_x_cell,
                        mouse_y_cell, separate_cur_color);

                    refresh_image_grid();   // Display the results

                    break;

                case 3:     // "Rect." tool

                    // Copy the real image to the temporary drawing one
                    blit(image[0], buffer_image, 0, 0, 0, 0, image[0]->w, image[0]->h);

                    // Draw a rectangle on the temporary drawing image
                    rect (buffer_image, prev_x_cell, prev_y_cell, mouse_x_cell, 
                        mouse_y_cell, separate_cur_color);

                    refresh_image_grid();   // Display the results

                    break;

                case 4:     // "Box" tool
                    
                    // Copy the real image to the temporary drawing one
                    blit(image[0], buffer_image, 0, 0, 0, 0, image[0]->w, image[0]->h);
                    // Draw a rectangle on the temporary drawing image
                    rectfill(buffer_image, prev_x_cell, prev_y_cell, mouse_x_cell, 
                        mouse_y_cell, separate_cur_color);
                    refresh_image_grid();   // Display the results

                    break;
                
                case 5:     // "Oval" tool
                    
                    // Copy the real image to the temporary drawing one
                    blit(image[0], buffer_image, 0, 0, 0, 0, image[0]->w, image[0]->h);

                    // Draw an oval on the temporary drawing image
                    if (prev_x_cell == mouse_x_cell || prev_y_cell == mouse_y_cell)
                    {
                        line (buffer_image, prev_x_cell, prev_y_cell, mouse_x_cell,
                            mouse_y_cell, separate_cur_color);
                    }
                    else
                    {
                        ellipse (buffer_image, 
                            (prev_x_cell + mouse_x_cell) / 2, 
                            (prev_y_cell + mouse_y_cell) / 2, 
                            abs(prev_x_cell - mouse_x_cell) / 2,
                            abs(prev_y_cell - mouse_y_cell) / 2,
                            separate_cur_color);
                    }
                    
                    refresh_image_grid();   // Display the results

                    break;
                
                case 7:     // "Copy" or "Cut" tool
                case 8:
                    
                    // Copy the real image to the temporary drawing one
                    blit(image[0], buffer_image, 0, 0, 0, 0, image[0]->w, image[0]->h);

                    // Draw a dotted rectangle on the temporary drawing image
                    dotted_rect(buffer_image, prev_x_cell, prev_y_cell, mouse_x_cell,
                        mouse_y_cell);

                    refresh_image_grid();   // Display the results

                    break;
                }
            }
        }
        // Cur color boxes
        else if
            (
            mouse_x_pixel >= SCREEN_W - 27 &&
            mouse_x_pixel <= SCREEN_W - 5 &&
            mouse_y_pixel >= INFO_AREA_TOP + text_height(font) * 2 + 10 + CUR_COLOR_BOX_HEIGHT &&
            mouse_y_pixel < INFO_AREA_TOP + text_height(font) * 2 + 10 + CUR_COLOR_BOX_HEIGHT +
                (12 * (MAX_CUR_COLORS / 2))
            )
        {
                draw_color = ((mouse_y_pixel - 2) - INFO_AREA_TOP + text_height(font) * 2 + 10 + 
                    CUR_COLOR_BOX_HEIGHT) / 12 + 5;
                separate_cur_color = cur_color[draw_color];
                refresh_cur_color_boxes();
        } 
        // Cur color boxes
        else if
            (
            mouse_x_pixel >= SCREEN_W - 53 &&
            mouse_x_pixel <= SCREEN_W - 32 &&
            mouse_y_pixel >= INFO_AREA_TOP + text_height(font) * 2 + 10 + CUR_COLOR_BOX_HEIGHT &&
            mouse_y_pixel < INFO_AREA_TOP + text_height(font) * 2 + 10 + CUR_COLOR_BOX_HEIGHT +
                (12 * (MAX_CUR_COLORS / 2))
            )
        {
                draw_color = ((mouse_y_pixel - 2) - INFO_AREA_TOP + text_height(font) * 2 + 10 + 
                    CUR_COLOR_BOX_HEIGHT) / 12 - 6;
                separate_cur_color = cur_color[draw_color];
                refresh_cur_color_boxes();
        }
    }
    else        // Left mouse button is not pressed
    {
        if (lmouse_down == TRUE)    // Left mouse button was previously pressed
        {
            lmouse_down = FALSE;

            set_mouse_range(0, 0, SCREEN_W - 1, SCREEN_H - 1);  // Free mouse movement

            if (mouse_x_pixel < SCREEN_W - 256)     // Mouse pointer is over sprite grid
            {
           
                switch (cur_tool)
                {
                case 0:     // "Pencil" tool

                    break;

                case 2:     // "Line" tool

                    line (image[0], prev_x_cell, prev_y_cell, mouse_x_cell, mouse_y_cell,
                        separate_cur_color);

                    drawing_operation = FALSE;      // Use image[0], not buffer_image
                    refresh_image_grid();           // Redraw the sprite

                    break;

                case 3:     // "Rect" tool

                    rect(image[0], prev_x_cell, prev_y_cell, mouse_x_cell, mouse_y_cell,
                        separate_cur_color);

                    drawing_operation = FALSE;      // Use image[0], not buffer_image
                    refresh_image_grid();           // Redraw the sprite

                    break;

                case 4:     // "Box" tool

                    rectfill(image[0], prev_x_cell, prev_y_cell, mouse_x_cell, mouse_y_cell,
                        separate_cur_color);

                    drawing_operation = FALSE;      // Use image[0], not buffer_image
                    refresh_image_grid();           // Redraw the sprite

                    break;

                case 5:     // "Oval" tool

                    if (prev_x_cell == mouse_x_cell || prev_y_cell == mouse_y_cell)
                    {
                        line (image[0], prev_x_cell, prev_y_cell, mouse_x_cell,
                            mouse_y_cell, separate_cur_color);
                    }
                    else
                    {
                        ellipse(image[0], 
                            (prev_x_cell + mouse_x_cell) / 2, 
                            (prev_y_cell + mouse_y_cell) / 2, 
                            abs(prev_x_cell - mouse_x_cell) / 2,
                            abs(prev_y_cell - mouse_y_cell) / 2,
                            separate_cur_color);
                    }

                    drawing_operation = FALSE;      // Use image[0], not buffer_image
                    refresh_image_grid();           // Redraw the sprite

                    break;

                case 7:     // "Copy" tool
                case 8:     // "Cut" tool

                    // Determine the dimensions of the sub-image to be copied
                    width = abs(mouse_x_cell - prev_x_cell) + 1;
                    height = abs(mouse_y_cell - prev_y_cell) + 1;
                    
                    // Destroy the old copy image bitmap and create the new one
                    destroy_bitmap (copy_image);
                    copy_image = create_bitmap (width, height);
                    clear_bitmap (copy_image);

                    // Determine where to copy from
                    if (prev_x_cell < mouse_x_cell) x = prev_x_cell; else x = mouse_x_cell;
                    if (prev_y_cell < mouse_y_cell) y = prev_y_cell; else y = mouse_y_cell;
                
                    // Copy from the drawing image to the copy bitmap
                    blit(image[0], copy_image, x, y, 0, 0, width, height);

                    if (cur_tool == 8)      // If "Cut" tool
                    {
                        if (magnified == TRUE)
                        {
                            cur_tool = prev_tool;
                            magnified = FALSE;
                            copy_image = NULL;
                        }
                        else
                        {
                            magnified = TRUE;
                            mag_x_pixel = x;
                            mag_y_pixel = y;

                            // Save a copy of the original image
                            magnify_original = create_bitmap (image[0]->w, image[0]->h);

                            blit(image[0], magnify_original, 0, 0, 0, 0,
                                image[0]->w, image[0]->h);

                            //
                            // Erase the original image and replace it with the magnified
                            // portion
                            //
                            destroy_bitmap (image[0]);
                            image[0] = create_bitmap (width, height);
                            blit(copy_image, image[0], 0, 0, 0, 0, width, height);

                            show_mouse (NULL);
                            init_newly_loaded_image ();
                            show_mouse (buffer);

                            cur_tool = 0; // Switch to "pencil" drawing tool

                            cycle_sprite_array (PURGE_UNDO_IMAGES);
                        }
                    }
                    else
                    {
                        cur_tool = 9; // Switch to "Paste" drawing tool
                    }

                    drawing_operation = FALSE;      // Use image[0], not buffer_image
                    refresh_image_grid();           // Redraw the sprite
                    draw_icons();
                }
				
				// If the mirror flags are true, copy the image
				if (mirror_horizontal) mirror_image (1, mouse_x_cell,
					mouse_y_cell);
				if (mirror_vertical) mirror_image (2, mouse_x_cell, 
					mouse_y_cell);

                prev_x_cell = -1;   // Reset the previous cell location
                prev_y_cell = -1;
            }
        }
    }
    
    if (mouse_b & 2) {  // Right mouse button is pressed

        if (rmouse_down == FALSE) {    // Right mouse button was previously released
            rmouse_down = TRUE;        // It is now pressed
        }
        
        if (mouse_x_pixel < SCREEN_W - 256) {     // Mouse pointer is on the image grid
            // Get the color of that pixel of the drawing image
            color_over = getpixel(image[0], mouse_x_cell, mouse_y_cell);
            // Add the color to the current color grid
            index = add_current_color(color_over);  // Get the color
            if (index != -1) {                      // If the color is in the array
                // Copy the new draw color to the separate current color box
                separate_cur_color = cur_color[draw_color];
                draw_color = index;                 // Set the draw color to it
                refresh_cur_color_boxes();          // Re-draw the boxes
            } else {
                // Copy the new draw color to the separate current color box
                separate_cur_color = cur_color[draw_color];
            }
        
        // Mouse pointer is over the color grid area
        } else if (mouse_x_pixel > SCREEN_W - 256 && mouse_y_pixel < INFO_AREA_TOP - 5) {
            // Constant right mouse down action
        }
    } else {                            // Right mouse button is not pressed
        if (rmouse_down == TRUE) {     // And was previously pressed
            rmouse_down = FALSE;       // It is now released
            // If the cursor is over the color grid area
            if (mouse_x_pixel > SCREEN_W - 256 && mouse_y_pixel < INFO_AREA_TOP - 5) {
                // Switch between the color strips and the color map
                if (color_strips_on) color_strips_on = FALSE; else color_strips_on = TRUE;
                refresh_color_grid();
            }
        }
    }
	
	// If the middle mouse button is pressed, bring up the main menu
	if (mouse_b & 4) main_menu();

    if (mouse_z != 0) { // Mouse wheel is rolled
        
        if (key[KEY_LSHIFT]) {  // If the left Shift key is pressed
            draw_color -= mouse_z;  // Increment the drawing color by the number of times
                                    // the mouse wheel was rolled
        
            // Keep the drawing color in range
            while (draw_color < 0) draw_color += MAX_CUR_COLORS;
            while (draw_color >= MAX_CUR_COLORS) draw_color -= MAX_CUR_COLORS;

            // Copy the new draw color to the separate current color box
            separate_cur_color = cur_color[draw_color];

        }
        else if (key[KEY_LCONTROL]) {   // If the left control key is pressed
            cur_tool -= mouse_z;        // Increment the drawing tool

            while (cur_tool < 0) cur_tool += 10;        // Keep the drawing
            while (cur_tool >= 10) cur_tool -= 10;      // tool in range

            show_mouse (NULL);
            draw_icons ();          // Re-draw the icons
            show_mouse (buffer);
        }
        else {
            increment_drawing_color (mouse_z);
        }
        refresh_cur_color_boxes();  // Re-draw the current color boxes
        position_mouse_z(0);        // Reset the mouse wheel to zero
    }
}
