// Drawing.c - All functions directly related to drawing go here.

#include "headers.h"

// Draws the image grid and color tables
void draw_grids (void)
{
    static int x, y,
        x1, y1, x2, y2;

    clear_bitmap(buffer);                   // Clear the buffer to black

    // Draw the image grid
    for (y = 0; y < image[0]->h; y++)
    {
        for (x = 0; x < image[0]->w; x++)
        {
            x1 = x * grid_cell_size;            // Position the grid square
            y1 = y * grid_cell_size;
            x2 = x1 + grid_cell_size;
            y2 = y1 + grid_cell_size;

            // rect(buffer, x1, y1, x2, y2, DARK_GRAY);    // Draw the grid square
            
            do_line (buffer, x1, y1, x2, y1, 0, draw_pixel_t);
            do_line (buffer, x1, y1, x1, y2, 0, draw_pixel_t);
            do_line (buffer, x2, y1, x2, y2, 0, draw_pixel_t);
            do_line (buffer, x1, y2, x2, y2, 0, draw_pixel_t);
        }
    }

    refresh_color_grid();

    // Draw the rectangle around the sprite being edited
    if (image[0]->w <= MAX_SPRITE_X && image[0]->h <= MAX_SPRITE_Y)
    {
        rect(buffer, ORIGINAL_SIZE_X - 1, ORIGINAL_SIZE_Y - 1, ORIGINAL_SIZE_X
            + image[0]->w, ORIGINAL_SIZE_Y + image[0]->h, DARK_GRAY);
    }

    // Draw the rectangle around the color over box
    rect(buffer, COLOR_OVER_BOX_X, COLOR_OVER_BOX_Y, // Draw the rectangle
        COLOR_OVER_BOX_X + COLOR_OVER_BOX_WIDTH, 
        COLOR_OVER_BOX_Y + COLOR_OVER_BOX_HEIGHT, DARK_GRAY);

    // Draw the rectangle around the current color box
    rect(buffer, CUR_COLOR_BOX_X, CUR_COLOR_BOX_Y,  // Draw the rectangle
        CUR_COLOR_BOX_X + CUR_COLOR_BOX_WIDTH, 
        CUR_COLOR_BOX_Y + CUR_COLOR_BOX_HEIGHT, DARK_GRAY);
}



// Refresh the mouse coordinates and color over box
void update_info_area ()
{
    static int mouse_x_cell, mouse_y_cell,
        color_over;     // Color of the pixel that the mouse is currently hovering over

    // Determine the cell the mouse pointer is hovering over
    mouse_x_cell = mouse_x / grid_cell_size;
    mouse_y_cell = mouse_y / grid_cell_size;

    // If the reported cell information is in bounds
    if (mouse_x_cell >= 0 && mouse_y_cell >= 0 && mouse_x_cell < image[0]->w &&
        mouse_y_cell < image[0]->h)
    {
        // Display the information
        TEXTPRINTF_RIGHT(buffer, font, SCREEN_W - 2, INFO_AREA_TOP + CUR_COLOR_BOX_HEIGHT
            + 5, TEXTPRINTF_COL, "  %d, %d", mouse_x_cell + 1, mouse_y_cell + 1);
    }
    else    // If the info is out of bounds
    {    
        // Display as much
        TEXTPRINTF_RIGHT(buffer, font, SCREEN_W - 2, INFO_AREA_TOP + CUR_COLOR_BOX_HEIGHT
            + 5, TEXTPRINTF_COL, "  --, --");
    }

    // Display the current drawing tool
    TEXTPRINTF_RIGHT(buffer, font, SCREEN_W - 2,  INFO_AREA_TOP + CUR_COLOR_BOX_HEIGHT + 5
        + text_height(font), TEXTPRINTF_COL, "   %s", tool_descriptions[cur_tool]);

    // Refresh the color over box
    if (mouse_x < SCREEN_W - 256)
    {
        color_over = getpixel(image[0], mouse_x_cell, mouse_y_cell);
    }
    else    // If the mouse is not in the image grid area
    {        
        show_mouse(NULL);                                   // Hide mouse pointer
        color_over = getpixel(buffer, mouse_x, mouse_y);    // Get the color
        show_mouse(buffer);                                 // Show mouse pointer
    }

    if (color_over == TRANSPARENT_COL)  // If the color is the transparent color
    {
        checker_area(COLOR_OVER_BOX_X + 1, COLOR_OVER_BOX_Y + 1, COLOR_OVER_BOX_X + 
            COLOR_OVER_BOX_WIDTH, COLOR_OVER_BOX_Y + COLOR_OVER_BOX_HEIGHT);
    }
    else if (color_over != -1)   // If the color is not the transparent color
    {
        rectfill(buffer, COLOR_OVER_BOX_X + 1, COLOR_OVER_BOX_Y + 1,
                COLOR_OVER_BOX_X + COLOR_OVER_BOX_WIDTH - 1, 
                COLOR_OVER_BOX_Y + COLOR_OVER_BOX_HEIGHT - 1, color_over);
    }
}



// Re-draws the current color boxes
void refresh_cur_color_boxes (void)
{
    static int x1, y1,
        x2, y2,
        index,
        pass,
        col;

    int cell_width = 22, cell_height = 12;

    show_mouse(NULL);   // Hide the mouse pointer prior to drawing operation

    for (pass = 0; pass < 2; pass++) {  // Go through this loop twice
    for (index = 0; index < MAX_CUR_COLORS; index++) {
        
        // Determine where to draw the cell
        if (index < MAX_CUR_COLORS / 2) {
            x2 = SCREEN_W - (cell_width + 5) - 5;
            x1 = x2 - (cell_width - 1);
            y1 = INFO_AREA_TOP + text_height(font) * 2 + 10 + CUR_COLOR_BOX_HEIGHT +
                (index * cell_height);
            y2 = y1 + cell_height - 1;
        } else {
            x2 = SCREEN_W - 5;
            x1 = x2 - (cell_width - 1);
            y1 = INFO_AREA_TOP + text_height(font) * 2 + 10 + CUR_COLOR_BOX_HEIGHT +
                ((index - MAX_CUR_COLORS / 2) * cell_height);
            y2 = y1 + cell_height - 1;
        }

        if (pass == 0) {
            // Draw the cell itself
            col = cur_color[index];
            if (col == TRANSPARENT_COL) {   // If the color is the transparent color
                checker_area(x1, y1, x2, y2);       // Draw a checker pattern in the area
            } else {                        // If the color is not the transparent color
                rectfill(buffer, x1, y1, x2 - 1, y2 - 1, col);      // Draw the cell
            }
        
            // If the cell is not the current drawing cell
            if (draw_color != index) {
                // Draw the border around the cell
                rect(buffer, x1 - 1, y1 - 1, x2, y2, DARK_GRAY);
            }
        } else if (draw_color == index) {
            // Highlight the border around the cell if it is the current color
                // Draw the border around the cell
                rect(buffer, x1 - 1, y1 - 1, x2, y2, YELLOW);
        }
    }
    }

    // Refresh the separate current color box
    if (separate_cur_color == TRANSPARENT_COL)
    {
        checker_area (CUR_COLOR_BOX_X + 1, CUR_COLOR_BOX_Y + 1, CUR_COLOR_BOX_X
            + CUR_COLOR_BOX_WIDTH, CUR_COLOR_BOX_Y + CUR_COLOR_BOX_HEIGHT);
    }
    else
    {
        rectfill (buffer, CUR_COLOR_BOX_X + 1, CUR_COLOR_BOX_Y + 1,
            CUR_COLOR_BOX_X + CUR_COLOR_BOX_WIDTH - 1, 
            CUR_COLOR_BOX_Y + CUR_COLOR_BOX_HEIGHT - 1,
            separate_cur_color);
    }
    
    show_mouse (buffer);    // Replace the mouse pointer after drawing operation
}



// Re-draws the cells on the image grid based on the contents of the sprite
void refresh_image_grid (void)
{
    int x, y, x1, y1, x2, y2, col;

    show_mouse(NULL);               // Hide the mouse cursor prior to drawing operation
    
    for (y = 0; y < image[0]->h; y++)
    {
        for (x = 0; x < image[0]->w; x++)
        {
            x1 = x * grid_cell_size;                            // Position the grid square
            y1 = y * grid_cell_size;
            x2 = x1 + grid_cell_size;
            y2 = y1 + grid_cell_size;

            //
            // If a drawing operation is in progress, use the buffer image;
            // otherwise, use the real image
            //
            if (drawing_operation)
            {
                col = getpixel(buffer_image, x, y);
            }
            else
            {
                col = getpixel(image[0], x, y);
            }

            if (col == TRANSPARENT_COL)
            {
                // If the grid is on, don't draw over it
                if (grid_activated)
                {
                    x1++;
                    y1++;
                }

                if (background_style == BLACK_STYLE)
                {
                    rectfill(buffer, x1, y1, x2 - 1, y2 - 1, BLACK);
                }
                else    // Checker background style
                {
                    //
                    // Draw the checkerboard pattern in each grid transparent
                    // square with four rectangle drawing operations
                    //
                    rectfill(buffer, x1, y1, x1 + grid_cell_size / 2, y1 + grid_cell_size / 2,
                        makecol(32, 32, 32));
                    rectfill(buffer, x1 + grid_cell_size / 2, y1, x2 - 1, 
                        y1 + grid_cell_size / 2, makecol(42, 42, 42));
                    rectfill(buffer, x1, y1 + grid_cell_size / 2, x1 + grid_cell_size / 2,
                        y2 - 1, makecol(42, 42, 42));
                    rectfill(buffer, x1 + grid_cell_size / 2, y1 + grid_cell_size / 2, x2 - 1,
                        y2 - 1, makecol(32, 32, 32));
                }
            }
            else    // If the color is not the transparent color
            {
                if (grid_activated)
                {
                    rectfill(buffer, x1 + 1, y1 + 1, x2 - 1, y2 - 1, col);
                }
                else
                {
                    rectfill(buffer, x1, y1, x2 - 1, y2 - 1, col);
                }
            }
        }
    }

    // Refresh the normal size image
    if (image[0]->w <= MAX_SPRITE_X && image[0]->h <= MAX_SPRITE_Y)
    {
        // Clear black behind the sprite
        rectfill(buffer, ORIGINAL_SIZE_X, ORIGINAL_SIZE_Y, ORIGINAL_SIZE_X +
            image[0]->w - 1, ORIGINAL_SIZE_Y + image[0]->h - 1, BLACK);

        if (drawing_operation)
        {
            draw_sprite(buffer, buffer_image, ORIGINAL_SIZE_X, ORIGINAL_SIZE_Y);
        }
        else
        {
            draw_sprite(buffer, image[0], ORIGINAL_SIZE_X, ORIGINAL_SIZE_Y);
        }
    }
	
	//
	// If the mirror flags are set, draw dotted lines to indicate where the
	// mirroring happens
	//
	if (mirror_horizontal)
	{
		x = image[0]->w * grid_cell_size / 2;
		for (y = 0; y < image[0]->h * grid_cell_size; y++)
		{
			if (y % 2 == 0) col = WHITE; else col = BLACK;
			putpixel (buffer, x, y, col);
		}		
	}
	if (mirror_vertical)
	{
		y = image[0]->h * grid_cell_size / 2;
		for (x = 0; x < image[0]->w * grid_cell_size; x++)
		{
			if (x % 2 == 0) col = WHITE; else col = BLACK;
			putpixel (buffer, x, y, col);
		}		
	}

    show_mouse(buffer);             // Replace the mouse cursor
}



// Temporarily view the sprite tiled to see what it looks like
void view_sprite_tiled()
{
    static int x, y,
        x1, y1,
        x2, y2,
        width, height,
        lmouse_down;

    // Get current keyboard and mouse information
    if (keyboard_needs_poll()) poll_keyboard();
    if (mouse_needs_poll()) poll_mouse();

    width = 2 + image[0]->w * 5;
    height = 2 + image[0]->h * 5;
    x1 = mouse_x - width / 2;       // Position the window
    x2 = x1 + width - 1;
    y1 = mouse_y - 10;
    y2 = y1 + height - 1;

    if (x1 < 0) {                   // Keep the left edge of the window in bounds
        x2 -= x1;
        x1 = 0;
    }
    if (y1 < 0) {                   // Keep the right edge of the window in bounds
        y2 -= y1;
        y1 = 0;
    }
    if (x2 >= SCREEN_W) {           // Keep the right edge of the window in bounds
        x1 -= x2 - (SCREEN_W - 1);
        x2 = SCREEN_W - 1;
    }
    if (y2 >= SCREEN_H - 1) {       // Keep the bottom edge of the window in bounds
        y1 -= y2 - (SCREEN_H - 1);
        y2 = SCREEN_H - 1;
    }

    rectfill(buffer, x1, y1, x2, y2, BLACK); // Clear the viewing area
    rect(buffer, x1, y1, x2, y2, WHITE); // Draw a white rectangle around the viewing area
    
    for (y = y1 + 1; y + image[0]->h <= y2; y += image[0]->h) {     // Draw the sprite 25 times,
    for (x = x1 + 1; x + image[0]->w <= x2; x += image[0]->w) {     // tiling it
        draw_sprite(buffer, image[0], x, y);
    }
    }

    copy_page(buffer, screen);  // Update the screen
    
    rest(1000);     // Wait one second

    show_mouse(buffer);         // Show the mouse cursor

    // Init the left mouse down variable
    if (mouse_b & 1) lmouse_down = TRUE; else lmouse_down = FALSE;
    clear_keybuf();     // Clear the keyboard buffer

    // Wait until spacebar or enter keys are pressed, or left mouse button is clicked
    while (!key[KEY_ESC] && !key[KEY_SPACE] && !key[KEY_ENTER]) {
        if (!(mouse_b & 1)) {
            if (lmouse_down == TRUE) break;
        } else lmouse_down = TRUE;

        if (keyboard_needs_poll()) poll_keyboard();
        if (mouse_needs_poll()) poll_mouse();
        copy_page(buffer, screen);
    }

    show_mouse(NULL);           // Hide the mouse cursor
    redraw_core(); 
    show_mouse(buffer);         // Show the mouse cursor
}



// Draws a checker pattern for transparent color areas
void checker_area(int x1, int y1, int x2, int y2)
{
    static int x_pixel, y_pixel, mod, col;

    for (y_pixel = y1; y_pixel < y2; y_pixel++)
    {
        for (x_pixel = x1; x_pixel < x2; x_pixel++)
        {
            if (grid_cell_size == 1)
            {
                mod = (x_pixel + y_pixel * abs(y1 - y2)) % 2;
            }
            else
            {
                mod = (x_pixel / (grid_cell_size / 2)) % 2;

                if ((y_pixel / (grid_cell_size / 2)) % 2 == 1)
                {
                    if (mod == 0)
                    {
                        mod = 1;
                    }
                    else
                    {
                        mod = 0;
                    }
                }
            }

            if (mod == 0)
            {
                col = makecol(42, 42, 42);  // Dark gray
            }
            else
            {
                col = makecol(32, 32, 32);  // Darker gray
            }

            putpixel (buffer, x_pixel, y_pixel, col);    // Draw the pixel
        }
    }
}



// Re-draws the color grid
void refresh_color_grid (void)
{
    static int x, y,
        x1, y1,
        x2, y2,
        col, cell_size,
        r, g, b,
        gradient;

    cell_size = 8; 

    show_mouse(NULL);           // Hide the mouse cursor
    
    // Clear the color grid area
    // rectfill(buffer, SCREEN_W - 256, 0, SCREEN_W - 1, INFO_AREA_TOP - 5, BLACK);
    rectfill(buffer, SCREEN_W - 256, 0, SCREEN_W - 1, NUM_COLOR_STRIPS * cell_size, BLACK);
    
    if (color_strips_on) {      // Draw the color strips
        
        // Draw the color strips on the upper right corner
        for (y = 0; y < NUM_COLOR_STRIPS; y++) {        
        for (x = 0; x < 32; x++) {
                
            // Determine where to draw the cell
            x1 = (SCREEN_W - (cell_size * 32)) + (x * cell_size);
            y1 = y * cell_size;
            x2 = x1 + cell_size - 1;
            y2 = y1 + cell_size - 1;
    
            // Determine the color
            if (y == 0)  col = makecol(x * 8, x * 8, x * 8);        // Black to white
            if (y == 1)  col = makecol(x * 8, 0,     0);            // Black to red
            if (y == 2)  col = makecol(0,     x * 8, 0);            // Black to green 
            if (y == 3)  col = makecol(0,     0,     x * 8);        // Black to blue
            if (y == 4)  col = makecol(x * 8, x * 8, 0);            // Black to yellow
            if (y == 5)  col = makecol(x * 8, 0,     x * 8);        // Black to purple
            if (y == 6)  col = makecol(0,     x * 8, x * 8);        // Black to teal
            if (y == 7)  col = makecol(x * 8, x * 4, 0);            // Black to orange
            if (y == 8)  col = makecol(255,   x * 8, x * 8);        // Red to white
            if (y == 9)  col = makecol(x * 8, 255,   x * 8);        // Green to white
            if (y == 10) col = makecol(x * 8, x * 8, 255);          // Blue to white
            if (y == 11) col = makecol(255,   255,   x * 8);        // Yellow to white
            if (y == 12) col = makecol(255,   x * 8, 255);          // Teal to white 
            if (y == 13) col = makecol(255, 128 + (x * 4), x * 8);  // Orange to white
            if (y == 14) col = makecol(x * 8, 255,   255);          // Purple to white
            if (y == 15) col = makecol(255 - (x * 8), 255, 0);      // Yellow to green
            if (y == 16) col = makecol(255, x * 8, 0);              // Red to yellow
            if (y == 17) col = makecol(0, x * 8, 255);              // Blue to teal
            if (y == 18) col = makecol(x * 8, 255 - (x * 8), 255);  // Teal to purple
            if (y == 19) col = makecol(255 - (x * 8), 0, 255);      // Blue to purple
            if (y == 20) col = makecol(0, x * 8, 255 - (x * 8));    // Blue to green
            if (y == 21) col = makecol(255, 255 - (x * 8), x * 8);  // Yellow to purple

            // "Off color" color strips
            if (y == 22) col = makecol((x + 5) * 4, (x + 5) * 7, (x + 5) * 3);  // B to green
            if (y == 23) col = makecol((x + 5) * 7, (x + 5) * 2, (x + 5) * 4);  // B to red
            if (y == 24) col = makecol((x + 5) * 4, (x + 5) * 2, (x + 5) * 7);  // B to purple
            if (y == 25) col = makecol((x + 5) * 7, (x + 5) * 6, (x + 5) * 2);  // B to yellow
            if (y == 26) col = makecol(0, (x + 5) * 5, (x + 5) * 7);    // B to blue
    
            if (col == TRANSPARENT_COL)         // If the color is the transparent color
                col = REPLACEMENT_TRANSPARENT_COL;  // Change the color
            rectfill(buffer, x1, y1, x2, y2, col);  // Draw the square
            
//          color_table[y * 32 + x] = col;      // Save the color information in the table
        }
        }
    }
    else                    // Draw a wide range of colors
    {
        // Draw horizontal strips in the color grid area
        for (y = 0; y < NUM_COLOR_STRIPS * cell_size; y++)
        {
            // Determine the base color to draw (in the middle)
            gradient = (y % ((NUM_COLOR_STRIPS * cell_size) / 6)) * 
                (256 / ((NUM_COLOR_STRIPS * cell_size) / 6));

            switch (y / ((NUM_COLOR_STRIPS * cell_size) / 6)) {
            case 0: r = 255; g = gradient; b = 0;        break;
            case 1: r = 255 - gradient; g = 255; b = 0;  break;
            case 2: r = 0; g = 255; b = gradient;        break;
            case 3: r = 0; g = 255 - gradient; b = 255;  break;
            case 4: r = gradient; g = 0; b = 255;        break;
            case 5: r = 255; g = 0; b = 255 - gradient;  break;
            }

            // Draw the strips
            hline (buffer, SCREEN_W - COLOR_SLIDER_WIDTH, y, SCREEN_W - 1,
                makecol(r, g, b)); 
        }

        refresh_colors_main_square ();
    }

    show_mouse(buffer);     // Show the mouse cursor
}



// Refresh the color grid to reflect the user's choice on the slider
void refresh_colors_main_square ()
{
    static int x, y,
        r, g, b,
        r1, g1, b1,
        r2, g2, b2,
        x_max,
        y_max,
        cell_size;

    cell_size = 8;

    x_max = 256 - COLOR_SLIDER_WIDTH - 5;
    y_max = NUM_COLOR_STRIPS * cell_size;

    // Draw horizontal strips in the color grid area
    for (y = 0; y < y_max; y++)
    {
        for (x = 0; x < x_max; x++)
        {
            //
            // Manipulate the values for the red, green, and blue color
            // components, to create two color grids:
            //
            // 1) The top of the square is the base drawing color selected on
            //    the slider to the right; colors fade to black at the bottom
            //
            // 2) The upper left corner of the square is white, and fades to
            //    black on the other three corners
            //
            // Then, combine the two colors to create a square of colors
            // that resembles the color picker in Adobe Photoshop or The Gimp.
            // 
            r = getr (drawing_color_base);      // Get the values of the base
            g = getg (drawing_color_base);      // drawing color
            b = getb (drawing_color_base);

            r1 = 255 * (x_max - x) / x_max;     // Get the values for the color
            g1 = 255 * (x_max - x) / x_max;     // square with the white corner
            b1 = 255 * (x_max - x) / x_max;

            r1 = r1 * (y_max - y) / y_max;
            g1 = g1 * (y_max - y) / y_max;
            b1 = b1 * (y_max - y) / y_max;

            r2 = r * x / x_max;                 // Get the values for the color
            g2 = g * x / x_max;                 // square with the corner with
            b2 = b * x / x_max;                 // the base color

            r2 = r2 * (y_max - y) / y_max;
            g2 = g2 * (y_max - y) / y_max;
            b2 = b2 * (y_max - y) / y_max;

            r = r1 + r2;                        // Combine the color values we
            g = g1 + g2;                        // found and keep them in bounds
            b = b1 + b2;

            if (r > 255) r = 255;
            if (g > 255) g = 255;
            if (b > 255) b = 255;

            // Draw the pixel
            putpixel (buffer, (SCREEN_W - 256) + x, y, makecol (r, g, b));
        }
    }
}



// Draws a dotted rectangle on bitmap bmp with the specified coordinates
void dotted_rect (BITMAP *bmp, int x1, int y1, int x2, int y2)
{
    int x, y, sign, col;

    if (x2 >= x1) sign = 1; else sign = -1;
    for (x = x1; x != x2; x += sign) {
        if ((x + y1) % 2 == 0) col = BLACK; else col = WHITE;
        putpixel(bmp, x, y1, col);
        if ((x + y2) % 2 == 0) col = BLACK; else col = WHITE;
        putpixel(bmp, x, y2, col);
    }

    if (y2 >= y1) sign = 1; else sign = -1;
    for (y = y1; y != y2; y += sign) {
        if ((x1 + y) % 2 == 0) col = BLACK; else col = WHITE;
        putpixel(bmp, x1, y, col);
        if ((x2 + y) % 2 == 0) col = BLACK; else col = WHITE;
        putpixel(bmp, x2, y, col);
    }
}



// Draw the icons on the screen
void draw_icons (void)
{
    int which_icon;
    int index;

    int x = SCREEN_W - 256;             // Initial pos. of icons
    int y = ICON_AREA_TOP;

    if (icons_enabled == FALSE) return;

    for (which_icon = 0; which_icon < NUM_ICON_FILES; which_icon++)
    {
        //
        // Determine the sprite index of the sprite to draw. If the icon
        // represents a drawing tool currently in use, draw the icon pressed
        // in to indicate such.
        //
        index = which_icon;

        if (which_icon == cur_tool ||
            (which_icon == 8 && magnified == TRUE))
        {
            index += NUM_ICON_FILES / 2;
        }

        draw_sprite (buffer, icon_images[index], x + which_icon *
            (icon_images[0]->w + 1), y);
    }
}



// Redraw the main visual elements - the sprite and color grids, the icons, the
// image data, and the current color boxes
void redraw_core (void) 
{
    draw_grids();
    draw_icons();
    refresh_image_grid();
    refresh_cur_color_boxes();
}



// Draws a pixel at (x, y) either gray or dark gray
void draw_pixel_t (BITMAP * bmp, int x, int y, int d)
{
    static int col;

    if ((x + y) % 2 == 0) col = DARK_DARK_GRAY; else col = DARK_GRAY;

    putpixel (bmp, x, y, col);
}



// Bring up a menu to change the video mode
void change_video_mode()
{    
    char choices[MAX_MENU_OPTIONS][MAX_MENU_OPTION_LENGTH] = {  // Hold the menu options
        "640x480 Fullscreen  ",
        "640x480 Window      ",
        "800x600 Fullscreen  ",
        "800x600 Window      ",
        "1024x768 Fullscreen ",
        "1024x768 Window     ",
        "1280x960 Fullscreen ",
        "1280x960 Window     ",
        "1280x1024 Fullscreen",
        "1280x1024 Window    ",
        "CANCEL           Esc"};
    
    int num_choices = 11;
    int x1, y1, x2, y2, width;
    int option;
    int new_width, new_height, new_screen_type; 
    
    width = get_width(choices, num_choices);                // Position the menu window
    x1 = mouse_x - width / 2;
    y1 = mouse_y - (5 + (text_height(font)));
    x2 = AUTO_DETECT;
    y2 = AUTO_DETECT;

    option = get_menu_option(x1, y1, x2, y2, num_choices, choices);     // Get the option

    if (option == 0) {
        new_width = 640;
        new_height = 480;
        new_screen_type = GFX_AUTODETECT_FULLSCREEN;
    }
    if (option == 1) {
        new_width = 640;
        new_height = 480;
        new_screen_type = GFX_AUTODETECT_WINDOWED;
    }
    if (option == 2) {
        new_width = 800;
        new_height = 600;
        new_screen_type = GFX_AUTODETECT_FULLSCREEN;
    }
    if (option == 3) {
        new_width = 800;
        new_height = 600;
        new_screen_type = GFX_AUTODETECT_WINDOWED;
    }
    if (option == 4) {
        new_width = 1024;
        new_height = 768;
        new_screen_type = GFX_AUTODETECT_FULLSCREEN;
    }
    if (option == 5) {
        new_width = 1024;
        new_height = 768;
        new_screen_type = GFX_AUTODETECT_WINDOWED;
    }
    if (option == 6) {
        new_width = 1280;
        new_height = 960;
        new_screen_type = GFX_AUTODETECT_FULLSCREEN;
    }
    if (option == 7) {
        new_width = 1280;
        new_height = 960;
        new_screen_type = GFX_AUTODETECT_WINDOWED;
    }
    if (option == 8) {
        new_width = 1280;
        new_height = 1024;
        new_screen_type = GFX_AUTODETECT_FULLSCREEN;
    }
    if (option == 9) {
        new_width = 1280;
        new_height = 1024;
        new_screen_type = GFX_AUTODETECT_WINDOWED;
    }

    if (option != 10) {     // Not "CANCEL" menu option
        // Try setting the new video mode
        set_color_depth (DEFAULT_COLOR_DEPTH);      // Value of 16 sets 65536 color mode    
        if (set_gfx_mode(new_screen_type, new_width, new_height, 0, 0) != 0)    {
            // If the video mode could not be set, display an error message
            allegro_message("Error setting graphics mode\n%s\n", allegro_error);
        } else {
            destroy_bitmap(buffer);                             // Destroy prev. buffer bitmap
            buffer = create_bitmap(SCREEN_W, SCREEN_H);         // Create the video buffer
            clear_bitmap (buffer);                              // Clear the buffer to color 0
            clear_bitmap (screen);                              // Clear the screen to color 0
            init_grid_cell_size ();
            screen_type = new_screen_type;                      // Save screen type
        }
    }
}
