// GUI functions

#include "headers.h"

// Returns the width of a dialog-to-be
int get_width(char choices[][MAX_MENU_OPTION_LENGTH], int num_choices)
{
    int index, width;

    for (index = 0, width = 10; index < num_choices; index++) {
        if (text_length(font, choices[index]) + 10 > width) 
            width = text_length(font, choices[index]) + 10;
    }

    return width;
}



// Gets a string from the keyboard, starting at location (x, y)
void get_string(char *return_string, int x, int y, char *default_str, 
                unsigned int max_chars)
{
    int cur_key;                        // The key most recently read from the keyboard
    char str[MAX_MENU_OPTION_LENGTH];       // Temporary string
    unsigned int cur_letter = strlen(default_str);
    int s_down;         // Used to wait until the S key is released if initially pressed
    
    if (key[KEY_S]) s_down = TRUE; else s_down = FALSE;

    strcpy(str, default_str);                   // Start with the passed default string
    
    while ((cur_key >> 8) != KEY_ENTER)  {      // Continue until enter key pressed

        // Clear the area where the text goes
        rectfill (buffer, x, y, x + text_length(font, "_") * max_chars,
            y + text_height(font), BLACK);
        
        if (strlen(str) == max_chars)                   // If the string is max length:
            TEXTPRINTF(buffer, font, x, y, TEXTPRINTF_COL, "%s", str);  // Display with no "_"
        else                                                // Shorter than max length:
            TEXTPRINTF(buffer, font, x, y, TEXTPRINTF_COL, "%s_", str); // Display with an "_"

        copy_page (buffer, screen); // Copy memory buffer to the screen

        if (s_down) {                               // If "S" key was previously pressed
            while (key[KEY_S])
            {
                if (keyboard_needs_poll()) poll_keyboard();
            }
            // Wait until it is released
            s_down = FALSE;                         // It is now released
            clear_keybuf();                         // Clear the keyboard buffer
        }

        cur_key = readkey();                            // Read a key from the keyboard
        if (                                            // If the key pressed is:
            ( ((cur_key >> 8)   >= KEY_A) & ((cur_key >> 8)   <= KEY_Z)) || // A-Z
            ( ((cur_key & 0xff) >= '0')   & ((cur_key & 0xff) <= '9'))   || // 0-9
            ( ((cur_key & 0xff) == '(' || (cur_key & 0xff) == ')')) ||      // ( or )
            ( ((cur_key & 0xff) == ',' || (cur_key & 0xff) == ',')) ||      // ,
            ( ((cur_key & 0xff) == '-' || (cur_key & 0xff) == ' ')) ||      // - or " "
            ( ((cur_key >> 8)   == KEY_STOP) & (strchr(str, (int) '.') == 0)))   // .
        {
            // If the string is short enough, and Ctrl and Alt are not pressed
            if (cur_letter < max_chars && !key[KEY_LCONTROL] && !key[KEY_ALT]) 
            {
                str[cur_letter] = (char) cur_key;   // Add the letter to the string
                str[cur_letter + 1] = '\0';         // End the string after the letter
                cur_letter++;                       // Advance to the next letter
            }
        }
        else if (((cur_key >> 8) == KEY_BACKSPACE) & (cur_letter > 0)) {    // Backspace
            cur_letter--;                           // Retreat to previous letter
            str[cur_letter] = '\0';                 // End the string after the letter
        }
    }

    strcpy(return_string, str); // Copy the string to the return string
}



// Gets a menu option from a list
// opens a window and returns the user's choice from it
int get_menu_option (int x1, int y1, int x2, int y2, int num_choices,
                     char choices[][MAX_MENU_OPTION_LENGTH])
{
    static int index,
        width,
        window_choices,
        x, y,
        top, bottom,    // For determining if an entry is highlighted
        menu_choice,
        blurb_col,
        esc_down, up_down, down_down, space_down,
        highlighted,
        reset_x, reset_y,
        final_choice;

    int display_start = 0;
    int click_flag = 0;
    int reset_mouse = FALSE;

    char blurb [MAX_MENU_OPTION_LENGTH];

    if (key[KEY_ESC]) esc_down = TRUE; else esc_down = FALSE;
    if (key[KEY_UP]) up_down = TRUE; else up_down = FALSE;
    if (key[KEY_DOWN]) down_down = TRUE; else down_down = FALSE;
    if (key[KEY_SPACE] || key[KEY_ENTER]) space_down = TRUE; else space_down = FALSE;

    show_mouse (NULL);      // Hide the mouse cursor

    if (ALLEGRO420 != TRUE) SET_TRANS_TEXT;

    // If x2 or y2 are passed as -1, autodetect the width and/or height
    if (x2 == AUTO_DETECT) {        
        x2 = x1 + 10;
        for (index = 0; index < num_choices; index++) {
            width = text_length(font, choices[index]) + 10;
            if ((x2 - x1) < width) x2 = x1 + width;
        }
    }
    if (y2 == AUTO_DETECT) {
        y2 = y1 + text_height(font) * num_choices + 9;
    }
    
    if (x1 < 0) {                   // Keep the left edge of the window in bounds
        x2 -= x1;
        x1 = 0;
    }
    if (y1 < 0) {                   // Keep the right edge of the window in bounds
        y2 -= y1;
        y1 = 0;

    }
    if (x2 >= SCREEN_W) {           // Keep the right edge of the window in bounds
        x1 -= x2 - (SCREEN_W - 1);
        x2 = SCREEN_W - 1;
    }
    if (y2 >= SCREEN_H - 1) {       // Keep the bottom edge of the window in bounds
        y1 -= y2 - (SCREEN_H - 1);
        y2 = SCREEN_H - 1;
    }

    // Draw the window
    rectfill (buffer, x1 + 1, y1 + 1, x2 - 1, y2 - 1, BLACK);
    rect (buffer, x1, y1, x2, y2, WHITE);
    
    // Position the mouse at the center of the first entry
    position_mouse((x2 + x1) / 2, y1 + 5 + text_height(font) / 2);

    // Determine the number of choices that can fit in one window
    window_choices = (y2 - y1 - 9) / text_height(font);
    if (window_choices > num_choices) window_choices = num_choices;
		
    // Position the level start marker at where the current filename is
    for (index = 0; index < num_choices; index++)
    {
        if (strcmp(choices[index], cur_filename) == 0)
        {
            display_start = index - (window_choices / 2);

            if (display_start < 0) display_start = 0;
            if (display_start + window_choices > num_choices)
                display_start = num_choices - window_choices;

            reset_mouse = TRUE;     // Tell the function to reset the mouse later
            reset_x = mouse_x;      // And tell it where.
            reset_y = mouse_y;

            // Start the mouse in our little window
            position_mouse ((x1 + x2) / 2, y1 + 10 + ((index - display_start) * 
                text_height(font)));

            break;
        }
    }
	
    click_flag = 0;
    do {
        if (keyboard_needs_poll()) poll_keyboard();    // Update the keyboard info
        if (mouse_needs_poll()) poll_mouse();       // Update the mouse info

        // If the left mouse button is not pressed and click_flag is at initial 
        // condition 0, change click_flag to 1
        if( !(mouse_b & 1) && click_flag == 0 ) click_flag = 1;

        // If the left mouse button is not pressed and click_flag is at initial
        // condition 2, change click_flag to 3
        if( !(mouse_b & 1) && click_flag == 2 ) break;

        // If the left mouse button is pressed and was at least once earlier, change
        // click_flag to 2
        if( (mouse_b & 1) && click_flag == 1 ) click_flag = 2; //exit(1);
        
        show_mouse (NULL);  // Hide the mouse cursor
        
        menu_choice = 0;    // Set the currently highlighted menu choice to none
        
        // Display all the menu choices in the current window
        for (index = 0; index < window_choices; index++)  {
            
            // Copy the string for the current menu option
            strcpy (blurb, choices[index + display_start]);

            // Set the text's position on the screen
            x = x1 + 5;
            y = y1 + 5 + index * text_height(font);
            
            // If we're at the last menu choice
            if (index + 1 == window_choices && num_choices > window_choices && 
                display_start + window_choices < num_choices) {
                strcpy(blurb, " \\/");              // Turn it into a down arrow
            }
            if (display_start > 0 && index == 0) {  // If we're at the first menu choice
                strcpy(blurb, " /\\");              // Turn it into an up arrow
            }

            // Determine if the choice is highlighted
            top = y1 + 4 + index * text_height(font);
            bottom = y1 + 4 + ((index + 1) * text_height(font)) - 1;
            if (index == 0) top = y1;
            if (index + 1 == window_choices) bottom = y2;

            // If the choice is not highlighted, draw it white
            highlighted = FALSE;
            blurb_col = makecol(192, 192, 192);
            
            // If the mouse cursor is in the highlighting area
            if (mouse_x >= x1 && mouse_x <= x2 && mouse_y >= top && mouse_y <= bottom) {
                
                highlighted = TRUE;
                blurb_col = WHITE;
                
                if (strcmp(blurb, " \\/") == 0) {   // If we are highlighting the arrow
                    display_start++;                // Scroll down
                    rest(5);                        // Wait 5 milliseconds
                    if( display_start > num_choices - window_choices + 1)
                        display_start = num_choices - window_choices + 1;
                }

                if( strcmp(blurb, " /\\") == 0 ){   // If we are highlighting the arrow
                    display_start--;                // Scroll up
                    rest(5);                        // Wait 5 milliseconds
                    if( display_start < 0 ) display_start = 0;
                }

                menu_choice = index;
            }

            if (highlighted)    // If the text is hightlighted
                 // Blue background
                rectfill (buffer, x - 2, y, x2 - 2, y + 7, makecol(0, 0, 164));
            else
                rectfill (buffer, x - 2, y, x2 - 2, y + 7, BLACK);  // Black background

            // Display the message on screen
            TEXTPRINTF (buffer, font, x, y, blurb_col TEXTPRINTF_OTHER_COL, blurb);
        }

        show_mouse (buffer);            // Show the mouse cursor

        copy_page (buffer, screen);     // Copy memory buffer to the screen

        if (mouse_z != 0)  {            // Mouse wheel is rolled
            display_start -= mouse_z * 3;
            if (display_start < 0) display_start = 0;
            if (display_start > num_choices - window_choices)
                display_start = num_choices - window_choices;
            position_mouse_z(0);
        }
        
        if (key[KEY_LCONTROL] && key[KEY_Q]) exit(0);   // Emergency quit on Ctrl-Q

        if (esc_down) {                             // If Esc key was previously pressed
            while (key[KEY_ESC])
            {
                if (keyboard_needs_poll()) poll_keyboard();   // Wait until it is released
            }
            esc_down = FALSE;                       // It is now released
            clear_keybuf();                         // Clear the keyboard buffer
        }

        if (space_down) {               // If space or enter keys were previously pressed
                                        // Wait until they are released
            while (key[KEY_SPACE] || key[KEY_ENTER])
            {
                if (keyboard_needs_poll()) poll_keyboard();
            }
            space_down = FALSE;                     // They are now released
            clear_keybuf();                         // Clear the keyboard buffer
        }

        if (key[KEY_UP]) {                  // If up arrow key is pressed
            if (up_down == FALSE) {         // and it was previously up
                up_down = TRUE;             // Tell the function that it is now down
                // If the mouse cursor is not too high up
                if (mouse_y > y1 + 5 + text_height(font))
                    // Move the mouse cursor upwards one line of text
                    position_mouse(mouse_x, mouse_y - text_height(font));
            }
        }
        else up_down = FALSE;

        if (key[KEY_DOWN]) {                // If down arrow key is pressed
            if (down_down == FALSE) {       // and it was previously up
                down_down = TRUE;           // Tell the function that it is now down
                // If the mouse cursor is not too high up
                if (mouse_y < y2 - (5 + text_height(font)))
                    // Move the mouse cursor downwards one line of text
                    position_mouse(mouse_x, mouse_y + text_height(font));
            }
        }
        else down_down = FALSE;

        if (key[KEY_SPACE] || key[KEY_ENTER]) break;

        if( key[KEY_ESC] ){                 // If the esc key is pressed
            menu_choice = num_choices - 1;          // Exit, selecting the last choice
            break;
        }

        if (key[KEY_F2]) {      // If F2 key is pressed
            save_bmp ("screenshot.bm_", buffer, NULL);      // Save a screenshot
        }
        
		rest (0);    // Allow time for smooth mouse movement

    // Loop until the user has clicked the left button once or pressed the spacebar or
    // enter key
    } while (click_flag != 3);

    show_mouse (NULL);  // Hide the mouse cursor

    SET_SOLID_TEXT;
    
    // Reset the mouse range to the entire screen
    // set_mouse_range(0, 0, SCREEN_W - 1, SCREEN_H - 1);
    
    // Reset the mouse cursor position if we moved it earlier
    if (reset_mouse) position_mouse(reset_x, reset_y);

    final_choice = menu_choice + display_start;
    if (mouse_x < x1 || mouse_x > x2 || mouse_y < y1 || mouse_y > y2) {
        return num_choices - 1; // Return last option - should be "CANCEL"
    }
    else {
        return menu_choice + display_start;
    }
}



// Displays the main menu
void main_menu (void)
{
    char choices[MAX_MENU_OPTIONS][MAX_MENU_OPTION_LENGTH] = {  // Hold the menu options
        "Undo             V",
        "Save             S",
        "Load             L",
        "Tools            >",
        "Sprite           >",
        "View             >",
        "Help            F1",
        "Exit             Q",
        "CANCEL         Esc"};
    int num_choices = 9;
    int x1, y1, x2, y2;
    int option;
    int width;

    width = get_width(choices, num_choices);                // Position the menu window
    x1 = mouse_x - width / 2;
    y1 = mouse_y - (5 + (text_height(font)));
    x2 = AUTO_DETECT;
    y2 = AUTO_DETECT;

    option = get_menu_option(x1, y1, x2, y2, num_choices, choices);     // Get the option
    
    if (option == 0) {                      // "Undo"
        cycle_sprite_array(EXECUTE_UNDO);
    } else if (option == 1) {               // "Save"
        save_image_to_file();
    } else if (option == 2) {               // "Load"
        load_image_from_file("");
    } else if (option == 3) {               // "Tools"
        do_tools_menu();    
    } else if (option == 4) {               // "Sprite"
        do_sprite_menu();   
    } else if (option == 5) {               // "View"
        do_view_menu(); 
    } else if (option == 6) {               // "Help"
        display_help();
    } else if (option == 7) {               // "Exit"
        save_config_file();                 // Save settings to a local configuration file
        exit(0);
    } else if (option == 8) {               // "CANCEL"
        // Do nothing
    }

    redraw_core(); 
    show_mouse(buffer);         // Show the mouse cursor
}



// Displays a menu enabling the user to select an image editing tool
void do_tools_menu (void)
{
    char choices[MAX_MENU_OPTIONS][MAX_MENU_OPTION_LENGTH] = {  // Hold the menu options
        "Pencil         1",
        "Brush          2",
        "Line           3",
        "Rectangle      4",
        "Filled box     5",
        "Oval           6",
        "Flood fill     7",
        "Copy           8",
        "Cut            9",
        "Paste          0",
        "CANCEL       Esc"};
    int num_choices = 11;
    int x1, y1, x2, y2, width;
    int option;

    width = get_width(choices, num_choices);                // Position the menu window
    x1 = mouse_x - width / 2;
    y1 = mouse_y - (5 + (text_height(font)));
    x2 = AUTO_DETECT;
    y2 = AUTO_DETECT;

    option = get_menu_option(x1, y1, x2, y2, num_choices, choices);     // Get the option

    if (option >= 0 && option <= 8) {
        cur_tool = option;          // Set the drawing tool
        draw_icons();
    }
    
    redraw_core(); 
    show_mouse(buffer);         // Show the mouse cursor
}



// Displays a menu enabling the user to flip the image horizontally or vertically,
// or rotate it 90 degrees
void do_sprite_menu (void)
{
    char choices[MAX_MENU_OPTIONS][MAX_MENU_OPTION_LENGTH] = {  // Hold the menu options
        "Flip horionztally      Ctrl+Right/Left",
        "Flip vertically           Ctrl+Up/Down",
        "Move (x, y) pixels        (Arrow keys)",
        "Rotate 90 degrees                    R",
        "Resize                          Ctrl+R",
		"Draw black border                    K",
        "Erase                              Del",
        "CANCEL                             Esc"};
    int num_choices = 8;
    int x1, y1, x2, y2, width;
    int option;

    width = get_width(choices, num_choices);                // Position the menu window
    x1 = mouse_x - width / 2;
    y1 = mouse_y - (5 + (text_height(font)));
    x2 = AUTO_DETECT;
    y2 = AUTO_DETECT;

    option = get_menu_option(x1, y1, x2, y2, num_choices, choices);     // Get the option

	switch (option)
	{
    case 0: flip_horizontal();				break;
    case 1: flip_vertical();				break;
    case 2: shift_sprite(0, 0);				break;
    case 3: rotate_90_degrees();			break;
    case 4: resize_sprite();				break;
    case 5: draw_border();					break;
    case 6: clear_sprite_image();			break;
	}
    
    redraw_core(); 
    show_mouse(buffer);         // Show the mouse cursor
}



// Displays a menu enabling the user to modify the view options
void do_view_menu (void)
{
    char choices[MAX_MENU_OPTIONS][MAX_MENU_OPTION_LENGTH] = {  // Hold the menu options
        "Toggle grid                             G",
        "Toggle transparent background           B",
        "Toggle fullscreen/window mode          F5",
        "Toggle horizontal mirroring             H",
        "Toggle vertical mirroring               J",
        "Toggle solid paste                      P",
        "View sprite tiled                       T",
        "Change video mode                        ",
        "CANCEL                                Esc"};
    int num_choices = 9;
    int x1, y1, x2, y2, width;
    int option;
    
    width = get_width(choices, num_choices);            // Position the menu window
    x1 = mouse_x - width / 2;
    y1 = mouse_y - (5 + (text_height(font)));
    x2 = AUTO_DETECT;
    y2 = AUTO_DETECT;

    // Get the option
    option = get_menu_option(x1, y1, x2, y2, num_choices, choices);

	switch (option)
	{
    case 0: toggle_grid();					break;
    case 1: toggle_background();			break;
    case 2: toggle_fullscreen_window();		break;
    case 3: toggle_mirror_horizontal();		break;
    case 4: toggle_mirror_vertical();		break;
    case 5: toggle_solid_paste();			break;
    case 6: view_sprite_tiled();			break;
    case 7: change_video_mode();			break;
	}

    redraw_core();

    show_mouse(buffer);         // Show the mouse cursor
}



// Display help information about the program
void display_help (void)
{
    int lmouse_down;
   
    show_mouse (NULL);          // Hide the mouse cursor

    clear_bitmap (buffer);      // Clear the screen

    help_text ("~");
    help_text ("BASIC SPRITE EDITOR - This is a utility for creating sprites, or relatively ");
	help_text ("small images typically used in computer games. To create a sprite, click and drag");
	help_text ("the mouse pointer over the grid squares on the left side, which represent the");
	help_text ("sprite's pixels. As you edit, you can see what the actual sprite looks like to the");
	help_text ("right of the sprite grid. To select colors for editing, click on the color selection");
	help_text ("area on the right side. Selected colors will appear in the palette squares on the");
	help_text ("lower right corner. Use the icons to the right to select drawing tools.");
    help_text ("");
    
    if (SCREEN_H > 480)
    {
        help_text ("MOUSE COMMANDS");
    }
    else
    {
        help_text ("MOUSE / KEYBOARD COMMANDS");
    }

    help_text ("");
    help_text ("Left click in sprite grid:              Use current drawing tool on the sprite");
    help_text ("Ctrl + Left click in sprite grid:       Use the copy tool on the sprite");
    help_text ("Alt + Left click in sprite grid:        Use the magnify tool on the sprite");
    help_text ("Left click in color grid:               Change current drawing color");
    help_text ("Left click on drawing tool icons:       Change current drawing tool");
    help_text ("Left click on current color palette:    Change current drawing color");
    help_text ("Right click in sprite grid:             Use highlighted drawing color");
    help_text ("Right click in color grid:              Toggle color grid modes");
    help_text ("Roll mouse wheel:                       Brighten/darken drawing color");
    help_text ("Shift + Roll mouse wheel:               Change current drawing tool");
    help_text ("");

    if (SCREEN_H > 480)
    {
        help_text ("KEYBOARD COMMANDS");
        help_text ("");
    }

    help_text ("Esc:                Show main menu");
    help_text ("Number keys:        Select current drawing tool");
    help_text ("Arrow keys:         Move sprite one pixel up, down, left, or right");
    help_text ("Ctrl + Arrow keys:  Flip sprite horizontally or vertically");
    help_text ("U / Ctrl+Z:         Undo");
    help_text ("S, L:               Save or load sprite");
    help_text ("Z, X:               Change drawing tool");
    help_text ("C, V:               Rotate sprite a small amount");
    help_text ("H, J:               Toggle horizontal/vertical mirroring");
    help_text ("K:                  Draw a border around the sprite");
    help_text ("R:                  Rotate sprite 90 degrees");
    help_text ("Ctrl+R:             Resize sprite");
    help_text ("G:                  Toggle drawing grid");
    help_text ("B:                  Toggle background color");
    help_text ("T:                  View sprite tiled");
    help_text ("P:                  Toggle transparent paste");
    help_text ("Q:                  Quit");
    help_text ("F1:                 Show help mode");
    help_text ("F5:                 Toggle fullscreen/window");
    help_text ("Backspace / Del.:   Erase current sprite                           ");
    help_text ("Comma, Period:      Lighten or darken drawing color                (Left click ");
    help_text ("< / >:              Switch to previous/next sprite                 to continue)");

    show_mouse (buffer);        // Show the mouse cursor

    // Init the left mouse down variable
    if (mouse_b & 1) lmouse_down = TRUE; else lmouse_down = FALSE;

	//
    // Wait until spacebar or enter keys are pressed, or left mouse button is
	// clicked
	//
    while (TRUE)
	{
        if (!(mouse_b & 1))
		{
            if (lmouse_down == TRUE) break;
        }
		else
		{
			lmouse_down = TRUE;
		}

        if (mouse_needs_poll()) poll_mouse();
			
        copy_page(buffer, screen);
		
		rest (5);
    }

    show_mouse(NULL);           // Hide mouse cursor
    redraw_core();
    show_mouse(buffer);         // Show mouse cursor
}



// Displays the about info
void display_splash_screen (void)
{
    int x1, y1, x2, y2, width, height;

    width = 15 + text_length(font, "BASIC SPRITE EDITOR v2.0");     // Position the window
    height = 10 + text_height(font) * 3;
    x1 = SCREEN_W / 2 - (width / 2);
    x2 = SCREEN_W / 2 + (width / 2);
    y1 = SCREEN_H / 2 - (height / 2);
    y2 = SCREEN_H / 2 + (height / 2);
    
    rectfill(buffer, x1, y1, x2, y2, BLACK);        // Draw a box around the splash screen
    rect(buffer, x1, y1, x2, y2, WHITE);

    TEXTPRINTF_CENTRE(buffer, font, SCREEN_W / 2, y1 + 6, TEXTPRINTF_COL,   // Display the text
        "BASIC SPRITE EDITOR v2.0");
    TEXTPRINTF_CENTRE(buffer, font, SCREEN_W / 2, y1 + 5 + text_height(font) * 2, TEXTPRINTF_COL, 
        "By Daniel Tenedorio");

    copy_page(buffer, screen);      // Copy it all to the screen

    show_mouse(screen);
    position_mouse(SCREEN_W / 2, y1 - 25);
    rest(2000);                     // Wait two seconds
	show_mouse(NULL);
}



// Toggles between solid and transparent paste
void toggle_solid_paste (void)
{
    if (solid_paste) solid_paste = FALSE; else solid_paste = TRUE;
}



// Toggle the grid squares from viewable to not viewable
void toggle_grid()
{
    grid_activated = grid_activated ? FALSE : TRUE;
}



// Toggle the background squares from black to checker color
void toggle_background() {
    if (background_style == BLACK_STYLE)
        background_style = CHECKER_STYLE;
    else
        background_style = BLACK_STYLE;
}



void toggle_mirror_horizontal ()
{
	mirror_horizontal = mirror_horizontal ? FALSE : TRUE;
}



void toggle_mirror_vertical ()
{
	mirror_vertical = mirror_vertical ? FALSE : TRUE;
}



// Toggle between fullscreen and window mode
void toggle_fullscreen_window (void)
{
    // Switch between fullscreen and window
    if (screen_type == GFX_AUTODETECT_WINDOWED)
        screen_type = GFX_AUTODETECT_FULLSCREEN;
    else
        screen_type = GFX_AUTODETECT_WINDOWED;

    show_mouse(NULL);       // Hide the mouse cursor

    // Set the graphics mode; if not successful
    if (set_gfx_mode(screen_type, SCREEN_W, SCREEN_H, 0, 0) != 0)
    {
        // Display an error message
        allegro_message("Error setting graphics mode\n%s\n", allegro_error);
        exit (1);   // Exit with error status
    }

    show_mouse(buffer);     // Replace the mouse cursor
}
