/*
 * high level fli decoding/encoding
 *
 */

#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <allegro.h>

#include "flilib.h"
#include "flitype.h"
#include "flifile.h"
#include "flisubs.h"    /* subroutines:  files...etc... */
#include "fli.h"


/*** helper functions ***/
void write_subchunk( FLI *f, int subchunk_type, int compress  );
static void  read_subchunk_frame( FLI * );
void debug_printf( FLI *f, BYTE errlevel, const char *fmt, ... );
/*** helper functions ***/


_FRAME_TYPE  frame1    = {0,0,0,0,{0,0,0,0,0,0}};




/* open a FLI file from a file.
 * Input: Filename
 * colormetric  = ( 0 = rgb range 0-63 (VGA), 1 = rgb range 0-255 )
 *        verbose_flag = ( 0=quiet, 1=verbose )
 * Output: Pointer to Allocated FLI structure
 *         Returns NULL on memory allocation error
 *         sets error code in f->errnumber on error ( different from FERR_OK )
 */
FLI *fli_open_file_read( char *filename, char colormetric, int verbose_flag )
{
    int     i;
    FLI     *f = NULL;
    char    *creatorname, *typename;


    f = (FLI *) alloc_fli_struc( verbose_flag );
    if(f==NULL){ return NULL;}      // memory allocation error

    if( tfile_exist(filename)==0 ){
    if(verbose_flag==1)
        debug_printf(f,0,"file %s don't exist", filename);
    f->errnumber = FERR_FILEDONTEXIST;
    return f;
    }
    f->read_write=0;    /* open for read */
    f->verbose = verbose_flag;
    f->debuglevel1[0]=0;
    f->debuglevel2[0]=0;
    f->debuglevel3[0]=0;
    f->debuglevel4[0]=0;
    f->colormetric = colormetric;
    f->filename  =  filename;

    /* open file, type 0 */
    f->fp = openfliload( filename, f );     // errnumber may be set on error
    if(f->fp==NULL){
        if(f->verbose==1){
        debug_printf(f,0,"openfliload error");
//        allegro_message("openfliload error ");
        }
    return f;
    }


    read_fli_header( f );   // errnumber may be set on error
    if( f->errnumber != FERR_OK ){
        if(f->verbose==1){
        debug_printf(f,0,"read_fli_header error");
//        allegro_message("read_fli_header error");
        }
    tclose(f->fp); f->fp=NULL;
    return f;
    }


//    f->errnumber = FERR_OK;




    /*** set bytes per line (1/8/15/16/24/32) ***/
    switch( f->depth ){
    case 1:
        f->stride = f->width/8;     /*  1 bit */
        f->update_palette = 1;
    break;
    case 4:
        f->stride = f->width/2;     /*  4bit  */
    break;
    case 8:
        f->stride = f->width;       /*  8bit  */
    break;
    case 15:
    case 16:
        f->stride = f->width * 2;   /* 15 or 16 bits */
    break;
    case 24:
        f->stride = f->width * 3;   /* 24 bit, BGR  */
    break;
    case 32:
        f->stride = f->width * 4;   /* 32 bit, BGRA */
    break;
    default:
        if(f->verbose==1){
            debug_printf(f,0,"error, strange bitdepth: <%d> is unsupported...", f->depth );
//        allegro_message("error, strange bitdepth: <%d> is unsupported...", f->depth );
        }
        f->errnumber = FERR_BITDEPTHUNSUP;
        tclose(f->fp);  f->fp = NULL;
        return f;    /* unsupported bit depth */
    break;
    }
    f->speedflag = 0;   /* time unit to use: 0-use frames, 1-milliseconds */
                        /* use frames initially */
    f->speedunit = f->frames;   /* unit in frames or milliseconds */
    f->framesize = f->stride*f->height; /* size in bytes of one frame */


    if(f->verbose==1){
    /* display header info */
    creatorname="unknown";
    if(f->creator==0x45474900) creatorname="EGI";
    if(f->creator==0x464c4942) creatorname="FlicLib";
    if(f->creator==0x42494c46) creatorname="FLicLib - release 2";
    if(f->creator==0x41544542) creatorname="Micrografx Simply 3D";
    if(f->creator==0x30314c46) creatorname="Kinetix 3DStudio MAX";

    typename="unknown";
    if(f->type==TYPE_FLI ) typename="FLI file";
    if(f->type==TYPE_FLC ) typename="Standard FLC, 8 bit depth";
    if(f->type==TYPE_EFLC) typename="FLC with bitdepth other than 8";
    if(f->type==TYPE_CFLC) typename="FLC with Huffman or BWT compression";

    puts("------------------------------");
    printf("size x: <%d>, size y: <%d>\n",f->width, f->height );
    printf("flic type, typename:<0x%x>,<%s>\n",f->type, typename);
    printf("flic size in bytes: <0x%x>\n", (int) f->size);
    printf("number of frames:   <%d>\n",  f->frames);
    printf("speed:              <%d> (in 1/70 seconds increments)\n", (int)  f->speed);
    printf("bytes per scanline: <%d>\n",  f->stride);
    printf("bitdepth (bpp):     <%d>\n",  f->depth);
    printf("creator number,name:<0x%x>,<%s>\n",(int) f->creator, creatorname);
    printf("updater number:     <0x%x>\n",(int) f->updater);
    printf("offset frame1:      <0x%x>\n",(int) f->oframe1);
    printf("offset frame2:      <0x%x>\n",(int) f->oframe2);
    printf("byte size / frame:  <%d>\n",  (int) f->framesize);
    printf("palette size:       <%d>\n", (1<<(f->depth))*3 );
    puts("------------------------------");
    printf("palette_entries:    <%d>\n",(1<<f->depth) );
    puts  ("note: palettes are valid only for bpp <= 8");
    puts("------------------------------");
    }

    f->images_to_go = f->frames;

    f->palentry = 1 << f->depth;
    if( f->depth >8) f->palentry=0;
    if( f->depth==1) f->palentry=2;
    if( f->depth==4) f->palentry=16;
    if( f->depth==8) f->palentry=256;

     /**** fail on common errors for FLx files ****/


    // case1: the colour depth is set to zero



    if(f->type!=TYPE_FLI){
        if( f->oframe1==0 && f->oframe2==0 ){
            if(f->verbose==1){
            debug_printf(f,0,"error, offset to first and second frames are zero");
//            allegro_message("error, offset to first and second frames are zero" );
            }
        f->errnumber = FERR_FIRSTSECONDOFFZERO;
        tclose(f->fp);  f->fp = NULL;
        return f;
        }



/*
        if( f->oframe1==f->oframe2 ){   // may be equal if only one single image is saved
            if(f->verbose==1){
            debug_printf(f,0,"error, offset to first and second frames are equal");
            allegro_message("error, offset to first and second frames are equal" );
            }
            tclose(f->fp); return NULL;
        }
*/


        if( f->oframe1==0 ){
            if(f->verbose==1){
            debug_printf(f,0,"error, offset to first frame is zero");
//            allegro_message("error, offset to first frame is zero" );
            }
            f->errnumber = FERR_FIRSTFRAMEZERO;
            tclose(f->fp);  f->fp = NULL;
            return f;
        }

/*
        if( f->oframe2==0 ){
            if(f->verbose==1){
            debug_printf(f,0,"error, offset to second frame is zero");
            allegro_message("error, offset to second frame is zero" );
            }
            tclose(f->fp);  f->fp = NULL;
            return f;
        }
*/


    }
     /**** fail on common errors for FLx files ****/


    if(f->type!=TYPE_FLI  && f->type!=TYPE_FLC  &&\
       f->type!=TYPE_EFLC && f->type!=TYPE_CFLC && f->type!=TYPE_SFLC ){
    if(f->verbose==1){
        debug_printf(f,0,"unknown fli type 0x%x",f->type);
//        allegro_message("unknown fli type 0x%x",f->type );
        }
    f->errnumber = FERR_UNKNOWNFLITYPE;
    tclose(f->fp);  f->fp = NULL;
    return f;
    }  /* unknown FLIC type */


    /* fli seems to be ok. */
    /* special processing for each different FLI file type */
    if(f->type == TYPE_FLI  ){     /* standard FLI file 320x200, 256 colors */
    }
    if(f->type == TYPE_FLC  ){     /* standard FLC file ---x---, 256 colors */
    }
    if(f->type == TYPE_EFLC ){     /* dta FLC file, bpp's: 1,8,15,16,24,32 */
    }

    if(f->type == TYPE_CFLC ){   /* FLC file, Huffman or BWT compression */
    if(f->verbose==1){
        debug_printf(f,0,"error, compressed FLC (huffman/BWT) is unsupported...");
        }
    f->errnumber = FERR_CFBWTUNSUPPORTED;
    tclose(f->fp);  f->fp = NULL;
    return f;       /* we don't support this type */
    }
    if(f->type == TYPE_SFLC ){   /* FLC file, "frame shift" or Huffman or BWT compression */
    if(f->verbose==1){
        debug_printf(f,0,"error, compressed FLC frame shift (huffman/BWT) is unsupported...");
        }
    f->errnumber = FERR_SFBWTUNSUPPORTED;
    tclose(f->fp);  f->fp = NULL;
    return f;       /* we don't support this type */
    }




    /* allocate space for image and palette */
    f->image = (BYTE *) malloc(  (f->framesize + f->stride*10)+999  );
    if( f->image==NULL ){
        if(f->verbose==1){
        debug_printf(f,0,"image frame allocation error");
//        allegro_message("image frame allocation error" );
        }
        f->errnumber = FERR_IMAGEALLOCERROR;
        tclose(f->fp);  f->fp = NULL;
        return f;
        }
    for(i=0;i< f->framesize ;i++){ f->image[i]=0;}

    f->palette = NULL;
    if( f->depth<=8 ){
    f->palette = (BYTE *) malloc(
    ( (f->palentry * 3) + 768+3) );
        if( f->palette==NULL ){
            if(f->verbose==1){
                debug_printf(f,0,"palette allocation error");
//                allegro_message("palette allocation error" );
                }
        f->errnumber = FERR_PALETTEALLOCERROR;
        tclose(f->fp);  f->fp = NULL;
        return f;
        }
        f->palette[0] = 0;      // provide default palette
        f->palette[1] = 0;
        f->palette[2] = 0;
        f->palette[3] = 255;
        f->palette[4] = 255;
        f->palette[5] = 255;
    }

    if(f->type!=TYPE_FLI){
    /* move to first frame, skipping prefix chunks (if any), since we don't
       care about them. We don't want CEL data either.    */
    tseek( f->fp, (f->oframe1), SEEK_SET );
    }
    f->fileposition = ttell( f->fp );  /* get our file position for rewinding */
    f->errnumber = FERR_OK;
    return f;
}


































/* Rewind to first frame
 * Input:
 * Output:
 *
 */
FLI *fli_rewind( FLI *f )
{
    f->images_to_go = f->frames;
    tclearerr( f->fp );            /* clear all errors */

    tseek( f->fp, f->fileposition, SEEK_SET );
/*
    if(f->type==TYPE_FLI)
        tseek( f->fp, f->fileposition , SEEK_SET );
    else    // move to second frame suitable for looping animation
        tseek( f->fp, f->fileposition , SEEK_SET );
//        tseek( f->fp, (f->oframe2) , SEEK_SET );
*/
    return f;
}





/* Read a frame.
 * Input:   FLI structure
 * Output:  FLI structure
 *         Returns ___ on error;
 */
FLI *fli_read_frame( FLI *f )
{
    int i;
    DWORD chunk_size, file_position;
    WORD  chunk_type;
    WORD  subchunks;

    if( f->images_to_go == 0 ) return f;     /* return if no more images */


//    if(f->verbose==1)
//        printf("****>fli_read_frame()\n");

    tclearerr( f->fp );

    /* safe chunks, in case subchunks size do not add to full chunk size */
    file_position = ttell( f->fp );
    frame1.size = chunk_size = r_dword( f->fp );
    frame1.type = chunk_type = r_word( f->fp );

    if(f->verbose==1){
//        printf("file_position <0x%x>\n", (int)  file_position);
//        printf("chunk_size:   <0x%x>:\n",(int)  chunk_size  );
//        printf("chunk_type:   <0x%x>: ", (int)  chunk_type  );
    }


    switch( chunk_type) {
/*
#define PREFIX_TYPE    0xf100
#define FRAME_TYPE     0xf1fa
#define SEGMENT_TABLE  0xf1fb
#define HUFFMAN_TABLE  0xf1fc
// subchunk types.....
#define COLOR_256   4
#define COLOR_64   11
#define BLACK      13
#define BYTE_RUN   15
#define DELTA_FLI  12
#define FLI_COPY   16
#define DELTA_FLC   7
#define DTA_BRUN   25
#define DTA_COPY   26
#define DTA_LC     27
// odd types .....
#define CEL_DATA    3
#define PSTAMP     18
#define LABEL      31
#define BMP_MASK   32
#define MLEV_MASK  33
#define SEGMENT    34
#define KEY_IMAGE  35
#define KEY_PAL    36
#define REGION     37
#define WAVE       38
#define USERSTRING 39
*/

    case FRAME_TYPE:
    frame1.chunks = subchunks = r_word( f->fp );
    frame1.delay              = r_word( f->fp );
    r_dword( f->fp );r_word( f->fp );   /* read past reserved[6] */

    if(f->verbose==1){
        debug_printf(f,2,"chunk type: FRAME_TYPE, subchunks %d, delay %d ", subchunks,frame1.delay);
        }
    if( frame1.delay !=0 ) {
    f->speedflag = 1;             /* time unit to use: 0-use frames, 1-milliseconds */
    f->speedunit = frame1.delay;  /* unit in milliseconds */
    }
    f->update_palette = 0;
    f->img_begin = 0;           /* fullscreen update */
    f->img_end   = f->height;   /* fullscreen update */


    if( subchunks == 0 ){       // simple frame
        f->images_to_go--;      /* decreased for each image */
    }

    for(i=0; i<subchunks; i++) {    // if subchunks is 0 then this is not executed
        read_subchunk_frame( f );   /* read subchunks and decode */
    }

    /* one frame is ready for displaying */
    break;



    case PREFIX_TYPE:       /* prefix chunk                    */
    if(f->verbose==1)
        debug_printf(f,2,"chunk type: PREFIX_TYPE - unsupported...");
    break;
    case SEGMENT_TABLE:     /* segment table chunk             */
    if(f->verbose==1)
        debug_printf(f,2,"chunk type: SEGMENT_TABLE - unsupported... ");
        f->images_to_go--;  /* decreased for each image */
    break;
    case HUFFMAN_TABLE:     /* Huffman compression table chunk */
    if(f->verbose==1)
        debug_printf(f,2,"chunk type: HUFFMAN_TABLE - unsupported... ");
    break;
    default:                /* unknown chunk type, ignore      */
    if(f->verbose==1)
        debug_printf(f,2,"chunk type: UNKNOWN 0x%x ", chunk_type);
    break;
    }

    /* this is to avoid padding bullshit and broken chunks */
    if( feof( f->fp) ){ f->images_to_go = 0; return f;}     /* return on file cutted */
    tseek( f->fp, (file_position+chunk_size), SEEK_SET );
    if( feof( f->fp) ){ f->images_to_go = 0; return f;}     /* return on file cutted */
    return f;
}









/* takes care of alignment and dispatching chunks to the respective subroutines */
static void  read_subchunk_frame( FLI *f )
{
    DWORD       file_position, subchunk_size;
    short       packets=0, x, copy;
    int             skip,y,i,j,k, type,c;
    unsigned char   d=0;
    DWORD           subchunk1_size;     // (unused)
    WORD            subchunk1_type;     // (unused)


    tclearerr( f->fp );

    /* safe chunks, in case subchunks size do not add to full chunk size */
    file_position = ttell( f->fp );
    subchunk1_size = subchunk_size = r_dword( f->fp );
    subchunk1_type = type          = r_word( f->fp );

    if(f->verbose==1){
        printf("\n-read_subchunk_frame()\n");
        printf("file_position:  <0x%x>\n", (int) file_position );
        printf(" subchunk size: <0x%x>\n", (int) subchunk_size );
        printf(" subchunk type: <0x%x>: ", (int) type );
//        debug_addprintf( f, "read_subchunk_frame()\n");
//        debug_addprintf( f, "file_position:  <0x%x>\n", (int) file_position );
//        debug_addprintf( f, " subchunk size: <0x%x>\n", (int) subchunk_size );
//        debug_addprintf( f, " subchunk type: <0x%x>: ", (int) type );
    }


    switch( type ){
    case BLACK:
    if(f->verbose==1)
        debug_printf(f,1,"subchunk type: BLACK");
        f->img_begin = 0;           /* fullscreen update */
        f->img_end   = f->height;   /* fullscreen update */
        for(i=0;i< f->framesize ; i++) f->image[i] = 0;
        f->images_to_go--;  /* decreased for each image, should this count as a image? */
    break;
    case FLI_COPY:
    if(f->verbose==1)
        debug_printf(f,1,"subchunk type: FLI_COPY");
        f->img_begin = 0;           /* fullscreen update */
        f->img_end   = f->height;   /* fullscreen update */
        for(i=0;i< f->framesize ; i++) f->image[i] = r_byte( f->fp );
        f->images_to_go--;  /* decreased for each image */
    break;


    case COLOR_64:
        f->update_palette = 1;
        f->upd_pal_begin  = -1;
        f->upd_pal_end    = -1;
        packets = r_word( f->fp );
        if(f->verbose==1) debug_printf(f,1,"subchunk type: COLOR64, packets: %d",packets);
        j=i=c=0;
        for( x=0 ; x < packets ; x++ )
        {
        skip    = r_byte( f->fp );
        copy    = r_byte( f->fp );  /* If this byte is zero it is interpreted to mean 256.*/
        if(copy==0){
                     copy = 255;    /* old value: 256 */
                     f->upd_pal_begin = 0;
                     f->upd_pal_end   = 255;
                   }
        if(f->verbose==1) printf(" COLOR skip<%d> copy<%d>\n",skip,copy);
        j = j + skip;
        if(f->upd_pal_begin== -1 ) f->upd_pal_begin=j;
        j = j * 3;
        i=0;
        for( y=0 ; y < copy ; y++ ) {
        if(f->colormetric==0)  d=0; /* vga, no shift                 */
        if(f->colormetric==1)  d=2; /* full range, shift to position */
        k = j+i;  f->palette[ k ] = r_byte( f->fp )<<d; i++; /* shift up */
        k = j+i;  f->palette[ k ] = r_byte( f->fp )<<d; i++;
        k = j+i;  f->palette[ k ] = r_byte( f->fp )<<d; i++;
        }
        j = j / 3;
        j = j + copy;
        }
        if(f->upd_pal_end== -1 ) f->upd_pal_end=j;
    break;

    case COLOR_256:
        f->update_palette = 1;
        f->upd_pal_begin  = -1;
        f->upd_pal_end    = -1;
        packets = r_word( f->fp );
        if(f->verbose==1){
            debug_printf(f,1,"subchunk type: COLOR256, packets: %d",packets);
        }
        j=i=c=0;
        for( x=0 ; x < packets ; x++ )
        {
        skip    = r_byte( f->fp );
        copy    = r_byte( f->fp );  /* If this byte is zero it is interpreted to mean 256.*/
        if(copy==0){
                     copy = 256;    // TESTING       /* old value: 256 */
                     f->upd_pal_begin=0;
                     f->upd_pal_end=255;
                   }
        if(f->verbose==1) printf("  COLOR skip<%d> copy<%d>\n",skip,copy);
        j = j + skip;
        if(f->upd_pal_begin==-1) f->upd_pal_begin=j;
        j = j * 3;
        i=0;
        for( y=0 ; y < copy ; y++ ) {
        if(f->colormetric==0)  d=2; /* vga, shift to position */
        if(f->colormetric==1)  d=0; /* full range, no shift   */
        k = j+i;  f->palette[ k ] = r_byte( f->fp )>>d; i++; /* shift down */
        k = j+i;  f->palette[ k ] = r_byte( f->fp )>>d; i++;
        k = j+i;  f->palette[ k ] = r_byte( f->fp )>>d; i++;
        }
        j = j / 3;
        j = j + copy;
        }
        if(f->upd_pal_end==-1) f->upd_pal_end=j;
    break;



    case BYTE_RUN:
    if(f->verbose==1)
        puts("  BYTE_RUN");
        debug_printf(f,1,"subchunk type: BYTE_RUN");
        do_fli_unbrun( f );
        f->images_to_go--;  /* decreased for each image */
    break;
    case DELTA_FLI:
    if(f->verbose==1)
        puts("  DELTA_FLI");
        debug_printf(f,1,"subchunk type: DELTA_FLI");
        do_delta_unfli( f );
        f->images_to_go--;  /* decreased for each image */
    break;
    case DELTA_FLC:
    if(f->verbose==1)
        debug_printf(f,1,"subchunk type: DELTA_FLC");
        do_delta_unflc( f );
        f->images_to_go--;  /* decreased for each image */
    break;
    case DTA_BRUN:
    if(f->verbose==1)
        debug_printf(f,1,"subchunk type: DTA_BRUN" );
        do_dta_unbrun( f );
        f->images_to_go--;  /* decreased for each image */
    break;
    case DTA_COPY:      /* erroneous...must be pixels, not bytes */
    if(f->verbose==1)
        debug_printf(f,1,"subchunk type: DTA_COPY" );
        f->img_begin = 0;           /* fullscreen update */
        f->img_end   = f->height;   /* fullscreen update */

        for(i=0;i< f->framesize ; i++){
            f->image[i] = r_byte( f->fp );
        }

        f->images_to_go--;  /* decreased for each image */
    break;
    case DTA_LC:
    if(f->verbose==1)
        debug_printf(f,1,"subchunk type: DTA_LC" );
        do_dta_unlc( f );
        f->images_to_go--;  /* decreased for each image */
    break;

    case PSTAMP:
    if(f->verbose==1)
        debug_printf(f,1,"subchunk type: PSTAMP (postage stamp) - unsupported..." );
    break;

    default:
    if(f->verbose==1)
        debug_printf(f,1,"subchunk type: unknown  0x%x", type );
        f->images_to_go--;  /* decreased for each image */
    break;
    }

    /* this is to avoid padding bullshit and broken subchunks */
    tseek( f->fp, (file_position + subchunk_size), SEEK_SET );
}



/*
 * write a subchunk (no compression ready yet.)
 *
 * supports only subchunk types:
 *            BLACK, FLI_COPY, DTA_COPY.
 *
 * compress: (0 = no compress 'as-is' , 1 = apply compress )
 *
 *
 *
 */
void write_subchunk( FLI *f, int subchunk_type, int compress  )
{
    int pixel, x, d=0, position=0, i,j;
    unsigned char *image = NULL, *palette = NULL;


    position = ttell( f->fp );
    w_dword( f->fp,0 );                /* subchunk_size, unknown yet */
    w_word( f->fp, subchunk_type );    /* type          */


    switch( subchunk_type ){

    case BLACK:
    if(f->verbose==1)
        debug_printf(f,1,"subchunk type: BLACK" );
    for(i=0;i< f->framesize ; i++) w_byte( f->fp, 0 );
    f->frames++;
    break;

    case FLI_COPY:
    if(f->verbose==1)
        debug_printf(f,1,"subchunk type: FLI_COPY" );
    for(i=0;i< f->framesize ; i++)  w_byte( f->fp, f->image[i] );
    f->frames++;
    break;

    case DTA_COPY:
    if(f->verbose==1)
        debug_printf(f,1,"subchunk type: DTA_COPY" );

        /* write pixels... */

        image = f->image;

        if( f->depth == 1 ){

        j = f->width / 8;
        j = j * f->height;
        for(i=0; i < j  ; i++)  w_byte( f->fp, *image++ );

        }else{

        j = f->width * f->height;      /* pixel count */
        for(i=0; i < j ; i++){

        if( f->depth==8 ){
            pixel = *image++;   w_byte( f->fp,  pixel );
        }

        if( f->depth==15 || f->depth==16 ){
            pixel = *image++;   w_byte( f->fp,  pixel );
            pixel = *image++;   w_byte( f->fp,  pixel );
        }

        if( f->depth==24 ){
            pixel = *image++;   w_byte( f->fp,  pixel );
            pixel = *image++;   w_byte( f->fp,  pixel );
            pixel = *image++;   w_byte( f->fp,  pixel );
        }

        if( f->depth==32 ){ /*  do this bitdepth have an alpha channel?  */
            pixel = *image++;   w_byte( f->fp,  pixel );
            pixel = *image++;   w_byte( f->fp,  pixel );
            pixel = *image++;   w_byte( f->fp,  pixel );
            pixel = *image++;   w_byte( f->fp,  pixel ); /* alpha??? */
        }
        }/* for loop */
        }/* bpp 1 else... */
    f->frames++;
    break;




    /************ not ready yet *****************/
    case COLOR_64:
    if(f->verbose==1)
        debug_printf(f,1,"subchunk type: COLOR_64" );
    i=0;
    if(f->palentry>=255) i=0;
    if(f->palentry==16) i=16*3;
    if(f->palentry==2) i=6;
    w_word( f->fp,1 );   /* 1 packet  */
    w_byte( f->fp,0 );   /* zero skip */
    w_byte( f->fp,i );   /* copy 256  */
    palette = f->palette;
    for( x=0 ; x < f->palentry * 3 ; x++ )
    {
        if(f->colormetric==0)  d=0; /* vga, no shift                 */
        if(f->colormetric==1)  d=2; /* full range, shift to position */
        pixel = *palette++;
        pixel = (pixel >> d) & 63;
        w_byte( f->fp, pixel );
    }
    break;



    case COLOR_256:
    i=0;
    if(f->palentry>=255) i=0;
    if(f->palentry==16) i=16*3;
    if(f->palentry==2) i=2*3;
    w_word (f->fp,1 );   /* 1 packet  */
    w_byte( f->fp,0 );   /* zero skip */
    w_byte( f->fp, i );  /* copy 256  */
    palette = f->palette;
    for( x=0 ; x < f->palentry * 3; x++ )
    {
        if(f->colormetric==0)  d=2; /* vga, shift to position */
        if(f->colormetric==1)  d=0; /* full range, no shift   */
        pixel = *palette++;
        pixel = (pixel << d) & 255;
        w_byte( f->fp, pixel );
    }
    break;


    case BYTE_RUN:
    if(f->verbose==1)
        debug_printf(f,1,"subchunk type: BYTE_RUN" );
        do_fli_brun( f );
    f->frames++;
    break;
    case DELTA_FLI:
    if(f->verbose==1)
        debug_printf(f,1,"subchunk type: DELTA_FLI" );
        do_delta_fli( f );
    f->frames++;
    break;
    case DELTA_FLC:
    if(f->verbose==1)
        debug_printf(f,1,"subchunk type: DELTA_FLC" );
        do_delta_flc( f );
    f->frames++;
    break;
    case DTA_BRUN:
    if(f->verbose==1)
        debug_printf(f,1,"subchunk type: DTA_BRUN" );
        do_dta_brun( f );
    f->frames++;
    break;

    case DTA_LC:
    if(f->verbose==1)
        debug_printf(f,1,"subchunk type: DTA_LC" );
        do_dta_lc( f );
    f->frames++;
    break;

    case PSTAMP:
    if(f->verbose==1)
        debug_printf(f,1,"subchunk type: PSTAMP (postage stamp) - unsupported..." );
    break;

/*#endif*/

    default:
    if(f->verbose==1)
        debug_printf(f,1,"subchunk type: unknown" );
    break;
    }


    /* must do trim to subchunk */
    trim_chunk_header( f, position );
}


/*
 *  Close and deallocate FLI structure
 *
 */
void fli_close( FLI *f )
{
    FLI     *retval = NULL;
    /* only for write operations */
    if( f->read_write==1 ){
        fflush( f->fp );
        f->size = ttell( f->fp );      /* get filesize */
        retval = write_fli_header( f );    /* write header at last close */
        if(retval == NULL );
    }
    f->errnumber = FERR_OK;
    free_fli_struc( f );
    return;
}


/*          *************************************************************
 *          *************************************************************
 *          *************************************************************
 *          *************************************************************
 *
 *          WRITE    WRITE    WRITE     WRITE     WRITE    WRITE    WRITE
 *
 *          *************************************************************
 *          *************************************************************
 *          *************************************************************
 */


/* open a new FLI for writing, creating a new file and writing a FLx header.
 *
 * Input:
 *     Filename = name of output file
 *          bpp = bit per plane (1/4/8/15/16/24/32) - not yet 32 bits.
 *                bpp 4 is not valid?
 *           sx = size x (not more than 32768)
 *           sy = size y (not more than 32678)
 *        image = unsigned char array to image, no padding used
 *      palette = optional palette, valid for bpp equal or less than 8
 * colormetric  = ( 0 = rgb range 0-63 (VGA), 1 = rgb range 0-255 )
 * verbose_flag = ( 0=quiet, 1=verbose )
 *
 * Output: Pointer to Allocated FLI structure
 *         Buffer for image will be allocated.
 *         Buffer for palette will be allocated if bpp is 8 or less.
 *         Returns NULL on memory allocation error
 *         Initially the offset to first and second images are equal.
 *
 */
FLI *fli_open_file_write( char *filename, char bpp, int sx, int sy, char colormetric ,int verbose_flag )
{
    int     i;
    FLI     *f=NULL;
    char    *creatorname, *typename;


    f = (FLI *) alloc_fli_struc(verbose_flag);
    if(f==NULL) return NULL;

    f->read_write=1;    /* open for write */
    f->verbose = verbose_flag;
    f->debuglevel1[0]=0;
    f->debuglevel2[0]=0;
    f->debuglevel3[0]=0;
    f->debuglevel4[0]=0;
    f->colormetric = colormetric;
    if(bpp==0){
        if(f->verbose==1)
            debug_printf(f,0,"error, bitdepth is zero" );
    f->errnumber = FERR_BITDEPTHISZERO;
    return f;
    }


    if(sx<=0) sx=1;
    if(sy<=0) sy=1;
    if(filename==NULL){
        if(f->verbose==1)
            debug_printf(f,0,"error, no filename given" );
    f->errnumber = FERR_NOFILENAMEGIVEN;
    return f;
    }

    f->depth  = bpp;     /* set bitdepth */
    f->width  = sx;
    f->height = sy;

    /* does not write FLI or compressed FLC */
    if( f->depth==1  || f->depth==4 ||f->depth==15 ||  f->depth==16 ||\
        f->depth==24 || f->depth==32 ) f->type = TYPE_EFLC;
    if(f->depth==8) f->type = TYPE_FLC;     // paletted 8-bits

    f->filename  =  filename;

    /* open file, type 0 */
    f->fp = openflisave( filename, f );     // errnumber may already be set
        if(f->fp==NULL){
        if(f->verbose==1)
            debug_printf(f,0,"openflisave error" );
        return f;
        }

    /*** set bytes per line (1/8/15/16/24/32) ***/
    switch( f->depth ){
    case 1:
        f->stride = f->width/8;     /*  1 bit */
    break;
    case 4:
        f->stride = f->width/2;     /*  4bit (nonstandard?) */
    break;
    case 8:
        f->stride = f->width;       /*  8bit  */
    break;
    case 15:
    case 16:
        f->stride = f->width*2;     /* 15 or 16 bits */
    break;
    case 24:
        f->stride = f->width * 3;   /* 24 bit, BGR  */
    break;
    case 32:
        f->stride = f->width * 4;   /* 32 bit, BGRA */
    break;
    default:
        if(f->verbose==1)
            debug_printf(f,0,"error, strange bitdepth: <%d> is unsupported...", f->depth  );
        f->errnumber = FERR_BITDEPTHUNSUP;
        tclose(f->fp);  f->fp = NULL;
        return f;    /* unsupported bit depth */
    }
    f->speedflag = 0;   /* time unit to use: 0-use frames, 1-milliseconds */
                        /* use frames initially */
    f->speedunit = 0;   /* unit in frames or milliseconds */
    f->framesize = f->stride * f->height; /* size in bytes of one frame */

    f->palentry = 1<<f->depth;
    if( f->depth==1) f->palentry=2;
    if( f->depth==4) f->palentry=16;
    if( f->depth==8) f->palentry=256;   // testing...
    if( f->depth> 8) f->palentry=0;     // no palette above bpp 8

    /* allocate space for image and palette */
    f->image      = (BYTE *)malloc(  (f->framesize+f->stride*2)  );
    if( f->image==NULL ){
        if(f->verbose==1)
            debug_printf(f,0,"error allocating memory for image\n"  );
        f->errnumber = FERR_IMAGEALLOCERROR;
        tclose(f->fp);  f->fp = NULL;
        return f;
        }
    f->image_prev = (BYTE *)malloc(  (f->framesize+f->stride*2)  );
    if( f->image_prev==NULL ){  // image is for delta compression
        if(f->verbose==1)
            debug_printf(f,0,"error allocating memory for 2nd image"  );
        f->errnumber = FERR_IMAGE2ALLOCERROR;
        tclose(f->fp);  f->fp = NULL;
        return f;
        }
    // clear images
    for(i=0;i< f->framesize ;i++){ f->image[i]=f->image_prev[i]=0;}

    if( f->palentry != 0 ){
    f->palette = (BYTE *)malloc(  f->palentry*3 + 768  );
        if( f->palette==NULL ){
        if(f->verbose==1)
            debug_printf(f,0,"palette allocation error"  );
        f->errnumber = FERR_PALETTEALLOCERROR;
        tclose(f->fp);  f->fp = NULL;
        return f;
        }
        f->palette_prev = (BYTE *)malloc(  f->palentry*3 + 768  );
        if( f->palette_prev==NULL ){
        if(f->verbose==1)
            debug_printf(f,0,"2nd palette allocation error"  );
        f->errnumber = FERR_PALETTE2ALLOCERROR;
        tclose(f->fp);  f->fp = NULL;
        return f;
        }
        f->palette[0] = 0;
        f->palette[1] = 0;
        f->palette[2] = 0;
        f->palette[3] = 255;
        f->palette[4] = 255;
        f->palette[5] = 255;
    }

    /* fli seems to be ok. */
    /* special processing for each different FLI file type */
    if(f->type == TYPE_FLI  ){}  /* standard FLI file 320x200, 256 colors */
    if(f->type == TYPE_FLC  ){}  /* standard FLC file          256 colors */
    if(f->type == TYPE_EFLC ){}  /* DTA FLC file, bpp's: 1,15,16,24,32    */
    if(f->type == TYPE_CFLC ){}  /* FLC file, Huffman or BWT compression  */

    if(f->verbose==1){
    /* display header info */
    creatorname="unknown";
    if(f->creator==0x45474900) creatorname="EGI";
    if(f->creator==0x464c4942) creatorname="FlicLib";
    if(f->creator==0x42494c46) creatorname="FLicLib - release 2";
    if(f->creator==0x41544542) creatorname="Micrografx Simply 3D";
    if(f->creator==0x30314c46) creatorname="Kinetix 3DStudio MAX";
    typename="unknown";
    if(f->type==TYPE_FLI ) typename="FLI file";
    if(f->type==TYPE_FLC ) typename="Standard FLC, 8 bit depth";
    if(f->type==TYPE_EFLC) typename="FLC with bitdepth other than 8";
    if(f->type==TYPE_CFLC) typename="FLC with Huffman or BWT compression";
    puts("------------------------------");
    printf("size x: <%d>, size y: <%d>\n",f->width, f->height );
    printf("flic type, typename:<0x%x>,<%s>\n",f->type, typename);
    printf("flic size in bytes: <0x%x>\n",(int) f->size);
    printf("number of frames:   <%d>\n",  f->frames);
    printf("speed:              <%d> (in 1/70 seconds increments)\n", (int) f->speed);
    printf("bytes per scanline: <%d>\n",  f->stride);
    printf("bitdepth (bpp):     <%d>\n",  f->depth);
    printf("creator number,name:<0x%x>,<%s>\n", (int) f->creator, creatorname);
    printf("updater number:     <0x%x>\n", (int) f->updater);
    printf("offset frame1:      <0x%x>\n", (int) f->oframe1);
    printf("offset frame2:      <0x%x>\n", (int) f->oframe2);
    printf("byte size / frame:  <%d>\n",   (int) f->framesize);
    printf("palette size:       <%d>\n", (1<<(f->depth))*3 );
    puts("------------------------------");
    printf("palette_entries:    <%d>\n",(1<<f->depth) );
    puts  ("note: palettes are valid only for bpp <= 8");
    puts("------------------------------");
    }

    write_fli_header( f );
        if( f->errnumber != FERR_OK ){
        if(f->verbose==1)
            debug_printf(f,0,"error, write_fli_header() 1st time"  );
        f->errnumber = FERR_WRITEFLI1HEADER;
        tclose(f->fp); f->fp = NULL;
        return f;
        }

    f->fileposition = ttell( f->fp );  /* get our file position for rewinding */
    f->oframe1 = f->fileposition;
    f->oframe2 = f->fileposition;

    tseek( f->fp, 0 , SEEK_SET );
    write_fli_header( f );
        if( f->errnumber != FERR_OK ){
        if(f->verbose==1)
            debug_printf(f,0,"error, write_fli_header() 2nd time"  );
        f->errnumber = FERR_WRITEFLI2HEADER;
        tclose(f->fp); f->fp = NULL;
        return f;
        }
    f->errnumber = FERR_OK;
    return f;
}


/* Write a frame. (not ready yet)
 * Input:   FLI structure
 * Output:  FLI structure
 *         Returns ___ on error;
 */
FLI *fli_write_frame( FLI *f )
{
    int retval1;
    int position;
    WORD  subchunks;

//    if(f->verbose==1)
//        printf("\n****>fli_write_frame()\n");
    tclearerr( f->fp );
    /* safe chunks, in case subchunks size do not add to full chunk size */

    /* first frame, write a full palette and a full image */
    if( f->frames == 0 ){     /***************** first frame *****************/
        if( f->depth <= 8 ) subchunks=2; else subchunks=1;
        position = write_chunk_header( f, FRAME_TYPE, subchunks );
        if( subchunks==2 ){
        if( f->colormetric==0)      write_subchunk( f, COLOR_64, 0 );
        if( f->colormetric==1)      write_subchunk( f, COLOR_256,0 );
        }
        if( f->type==TYPE_FLC || f->type==TYPE_FLI ){
            write_subchunk( f, BYTE_RUN,0 );  /* 8 bit bpp, compressed */
//            write_subchunk( f, FLI_COPY,0 );  /* 8 bit bpp, uncompressed */
            }
        if( f->type == TYPE_EFLC ){
            write_subchunk( f, DTA_BRUN,0 );  // errbone
//            write_subchunk( f, DTA_COPY,0 );
            }
        trim_chunk_header( f, position );
    return f;
    }                         /***************** first frame *****************/


    if( f->frames>=1) {   /**************** 2nd and more frame ****************/
        /* set seek position to second image */
        if(f->frames==1) f->oframe2 = ttell( f->fp );   /* 2nd frame */

    /* compress palette and compare to last palette */
    retval1 = compress_palette( f );      /* returns 0 on no change */
    if(retval1==0) subchunks = 1;         /* only image */
    else    subchunks = 2;                /* palette and image */

    if( subchunks == 2 && f->depth > 8 ) subchunks=1;   /* no palette for bpp > 8 */

    position = write_chunk_header( f, FRAME_TYPE, subchunks );
    if( subchunks == 2 ){
        if( f->colormetric==0 )     write_subchunk( f, COLOR_64, 0 );
        if( f->colormetric==1 )     write_subchunk( f, COLOR_256,0 );
    }
        if( f->type==TYPE_FLC || f->type==TYPE_FLI ){
            write_subchunk( f, BYTE_RUN,0 );  /*  2nd choise   */
//            write_subchunk( f, FLI_COPY,0 );  /* 8 bit bpp, uncompressed */
//            write_subchunk( f, DELTA_FLI,0 );  /* best choise */
            }
        if( f->type == TYPE_EFLC ){
            write_subchunk( f, DTA_BRUN,0 );  /* 2nd choise   */
//            write_subchunk( f, DTA_COPY,0 );  /* other bitdepths, uncompressed */
//            write_subchunk( f, DTA_LC,0 );  /* best choise */
            }

#ifdef NEVER
    /* compress image and compare to last image */
    retval2 = compress_image( f );  // DTA_BRUN or BYTE_RUN
        write_subchunk( f, retval2,1 );

        write_subchunk( f, BLACK    ,1);
        write_subchunk( f, BYTE_RUN ,1);/* FLI_BRUN */
        write_subchunk( f, FLI_COPY ,1);/* FLI_COPY */
        write_subchunk( f, DELTA_FLI,1);/* FLI_LC   */
        write_subchunk( f, DELTA_FLC,1);/* FLI_SS2  */
        write_subchunk( f, DTA_BRUN ,1);
        write_subchunk( f, DTA_COPY ,1);
        write_subchunk( f, DTA_LC   ,1);
#endif


    trim_chunk_header( f, position );
    return f;
    }                    /**************** 2nd and more frame ****************/

    return f;
}





void debug_printf( FLI *f, BYTE errlevel, const char *fmt, ... )
{
    BYTE *debuglevel = NULL;
    va_list args;

    errlevel &= 3;
    if(errlevel==0) debuglevel = f->debuglevel1;
    if(errlevel==1) debuglevel = f->debuglevel2;
    if(errlevel==2) debuglevel = f->debuglevel3;
    if(errlevel==3) debuglevel = f->debuglevel4;
    va_start(args, fmt);
    _vsnprintf( debuglevel ,80, fmt, args );    debuglevel[80] = 0;
    va_end(args);
}





/*
        strcpy( (char *)&stmp2, pathptr );
        st = strrchr( stickypathbkmap, '\\');
        *st++='\\';
        *st++=0;
*/


void debug_addprintf( FLI *f, const char *fmt, ... )
{
    int     len,led,i,x,y;
    BYTE    commandline[80+1], c;
    va_list args;

    va_start(args, fmt);

    _vsnprintf( commandline ,80, fmt, args ); commandline[80]=0;
    va_end(args);

    // copy to destination buffer
    len = strlen( commandline );
    led = strlen( f->debugappend );
    // append to buffer
    for(i=0;i<len;i++){
    f->debugappend[ led + i ] = commandline[ i ];
    if( (led+i) > 8190 ){ f->debugappend[ led + i ] = 0; break; }
    }

    // print out
//    textprintf( screen, font, 0,0, 125, f->debugappend );

    commandline[1] = 0;
    led = strlen( f->debugappend );
    x = y = 0;
    for(i=0;i<led;i++){
        c = f->debugappend[i];
        commandline[0] = c;
            if( c==0 || c=='\n' ){
            y += 8; x=0;
        }else{
            if( y < screen->h )
            textprintf( screen, font, x,y, 125, commandline );
            x += 8;
        }
    }
}


















/* from fli to BITMAP */
void fli_compose_image( BITMAP *bild, void *_ptr, int top, int down, int sx, int depth )
{
    unsigned char uch, ub, u;
    BYTE  uc1,uc2;
    int r=0,g=0,b=0,a=0,i;
    int x,y;
    int si;
    unsigned char *ptr= (unsigned char *)_ptr;
    static const BYTE bits[]={128,64,32,16,8,4,2,1,0};

    down++;

    if(depth==1){
        for(y = top; y< down ;y++)      {
        for(x = 0; x< (sx/8) ;x++)  {
        u = ptr[ x+y*(sx/8) ];
        for(i=0;i<8;i++){
            uch = u;
            ub = uch & bits[i]; if(ub!=0) ub=1;
            putpixel(bild,  x*8+i , y, ub );
            }
        }}
    }

    if(depth==8){
        for(y = top; y< down ;y++) {
        for(x = 0; x< sx ;x++) {
        si = x + (y*sx);
        i = ptr[si];
        putpixel(bild,  x,y, i );
        }}
    }

    if( depth==15||depth==16 ){
        for(y = top; y< down ;y++)      {
        for(x = 0; x<sx*2 ;x+=2) {
        si = x + (y*sx*2);
        uc2  = ptr[si+0];
        uc1  = ptr[si+1];
        if(depth==16){
        b = uc2 & 0x1f;
        b = b<<3;   /* scale to fit in 8 bit*/
        r = uc1 & 0xf8;
        g = ( (uc1 << 5) | (uc2 >>3) ) & 0xf8;
        }
        if(depth==15){
        b = uc2 & 0x1f;
        b = b<<3;   /* scale to fit in 8 bit*/
        r = (uc1<<1) & 0xf8;
        g = ( (uc1 << 6) | (uc2 >>2) ) & 0xf8;
        }
        putpixel(bild,  x/2,y, makecol(r,g,b) );
        }}
    }

    if(depth==24){
        for(y = top; y< down ;y++)      {
        for(x = 0; x<sx*3 ;x+=3) {
        si = x + (y*sx*3);
        b   = ptr[si+0];
        g   = ptr[si+1];
        r   = ptr[si+2];
        putpixel(bild,  x/3,y, makecol(r,g,b) );
        }}
    }

    if(depth==32){
        for(y = top; y< down ;y++)      {
        for(x = 0; x<sx*4 ;x+=4) {
        si = x + (y*sx*4);
        b   = ptr[si+0];    // bgra or abgr ?
        g   = ptr[si+1];
        r   = ptr[si+2];
        a   = ptr[si+3];
        putpixel(bild, x/4,y, makecol(r,g,b) );
        }}
    }

    return;
}




/* from BITMAP to fli */
void fli_reverse_compose_image ( FLI *fli, BITMAP *img )
{
    int     r=0,g=0,b=0, uc1=0, uc2=0,index=0, lineoffset=0;
    int     x,y,c,color,i;
    BYTE    icolor=0;
    BYTE     nor[]={128,64,32,16,8,4,2,1};
    BYTE    nand[]={127,191,223,239,247,251,253,254};



    if( fli->depth == 1){      // BPP1 IS NOT READY YET


    for(i=0;i< fli->framesize ;i++) fli->image[ i ] = 0xaa;

    for(y=0;y<fli->height;y++){
    lineoffset = y * fli->stride;
    i = 0;
    for(x=0,i=0,index=0;x< fli->width ;x++,index++){
        color = getpixel(img,x,y) & 255;
        index &= 7;         // range 0 to 7
        if( color != 0 )
            icolor = icolor | (nor[index]);
        else
            icolor = icolor & (nand[index]);
        if( index==7 ){
        fli->image[ lineoffset + i ] = icolor;
        i++;
        }
    }}

    }


    if( fli->depth == 8){
    for(y=0;y<fli->height;y++){
    for(x=0;x<fli->width;x++){
    r = getpixel(img,x,y) & 255;
    fli->image[ y * fli->stride + x ] = r;
    }}
    }


    if( fli->depth == 15 || fli->depth == 16 ){
    for(y=0;y< fli->height ;y++){
    for(x=0;x< fli->width  ;x++){
    c = getpixel(img,x,y);
    r = getr(c);
    g = getg(c);
    b = getb(c);
        if( fli->depth==16 ){
        b = ((b>>3) & 0x1f);
        g = ((g>>2) & 0x2f)<<5;
        r = ((r>>3) & 0x1f)<<11;
        }
        if( fli->depth==15 ){
        b = ((b>>3) & 0x1f);
        g = ((g>>3) & 0x1f)<<5;
        r = ((r>>3) & 0x1f)<<10;
        }
        uc1 = uc2 = (r | g | b);
        uc1 &= 0xff;
        uc2 = (uc2 >> 8)&0xff;
    fli->image[ (y* fli->stride) + x*2+0 ] = uc1;
    fli->image[ (y* fli->stride) + x*2+1 ] = uc2;
    }}
    }


    if( fli->depth == 24 ){
    for(y=0;y<fli->height;y++){
    for(x=0;x<fli->width;x++){
    c = getpixel(img,x,y);
    r = getr(c);
    g = getg(c);
    b = getb(c);
    fli->image[ (y * fli->stride) + x*3+0 ] = b;
    fli->image[ (y * fli->stride) + x*3+1 ] = g;
    fli->image[ (y * fli->stride) + x*3+2 ] = r;
    }}
    }

    if( fli->depth == 32 ){
    for(y=0;y<fli->height;y++){
    for(x=0;x<fli->width;x++){
    c = getpixel(img,x,y);
    r = getr(c);
    g = getg(c);
    b = getb(c);
    fli->image[ (y * fli->stride) + x*4+0 ] = b;
    fli->image[ (y * fli->stride) + x*4+1 ] = g;
    fli->image[ (y * fli->stride) + x*4+2 ] = r;
    fli->image[ (y * fli->stride) + x*4+3 ] = 0;
    }}
    }

/*
    putpixel( screen,0,0,1 );
    putpixel( screen,10,10,1 );
    putpixel( screen,20,20,1 );
    putpixel( screen,30,30,1 );
    fli_compose_image( screen, fli->image, 0, fli->height, fli->width, fli->depth );
    putpixel( screen,0,0,1 );
    putpixel( screen,10,10,1 );
    putpixel( screen,20,20,1 );
    putpixel( screen,30,30,1 );
*/

}



/* from fli to PALETTE */
void fli_copy_pal( FLI *fli, RGB *pal )
{
    int i,j;

    if( fli == NULL || pal == NULL ) return;
    if( fli->palentry == 0 ) return;
    if( fli->palette == NULL ) return;
    for(i=0,j=0;i < (fli->palentry-1) ; i++ ){
//    for(i=0,j=0;i < (fli->palentry) ; i++ ){
        pal[i].r = fli->palette[j++];
        pal[i].g = fli->palette[j++];
        pal[i].b = fli->palette[j++];
    }
}

/* from PALETTE to fli */
void fli_reverse_copy_pal( FLI *fli, RGB *pal )
{
    int i,j;

    if( fli == NULL || pal == NULL ) return;
    if( fli->palette == NULL ) return;
    for(i=0,j=0;i<256;i++){
        fli->palette[j++] = pal[i].r;
        fli->palette[j++] = pal[i].g;
        fli->palette[j++] = pal[i].b;
    }
}




/*
 * from errnumber into a string
 */
const char *fli_return_errstring( FLI *f )
{
    if( f->errnumber== FERR_OK ) return "ok\0";
    if( f->errnumber== FERR_CANTREADHEADER ) return "can't read header ( 128 bytes)";
    if( f->errnumber== FERR_CANTWRITEHEADER ) return "can't write header (128 bytes)";
    if( f->errnumber== FERR_CANTOPENFILEREAD  ) return "can't open file for read";
    if( f->errnumber== FERR_CANTOPENFILEWRITE ) return "can't open file for write";
    if( f->errnumber== FERR_FILEDONTEXIST ) return "file don't exist";
    if( f->errnumber== FERR_BITDEPTHUNSUP ) return "strange bitdepth is unsupported";
    if( f->errnumber== FERR_FIRSTSECONDOFFZERO ) return "offset to first and second frames are zero";
    if( f->errnumber== FERR_UNKNOWNFLITYPE ) return "unknown fli type";
    if( f->errnumber== FERR_CFBWTUNSUPPORTED ) return "compressed FLI (Huffman/BWT) is unsupported";
    if( f->errnumber== FERR_SFBWTUNSUPPORTED  ) return "compressed FLI (Frame Shift/Huffman/BWT) is unsupported";
    if( f->errnumber== FERR_IMAGEALLOCERROR ) return "image frame allocation error";
    if( f->errnumber== FERR_IMAGE2ALLOCERROR ) return "second image frame allocation error";
    if( f->errnumber== FERR_PALETTEALLOCERROR ) return "palette allocation error";
    if( f->errnumber== FERR_PALETTE2ALLOCERROR ) return "second palette allocation error";
    if( f->errnumber== FERR_FIRSTFRAMEZERO ) return "offset to first frame is zero";
    if( f->errnumber== FERR_BITDEPTHISZERO ) return "bitdepth is zero";
    if( f->errnumber== FERR_NOFILENAMEGIVEN ) return "no filename is given";
    if( f->errnumber== FERR_WRITEFLI1HEADER ) return "write_fli_header() 1st time";
    if( f->errnumber== FERR_WRITEFLI2HEADER ) return "write_fli_header() 2nd time";
    if( f->errnumber== FERR_EOF  ) return "End-Of-File encountered";
    return "unknown error";
}





