/*

GBMTIF.C  Microsoft/Aldus Tagged Image File Format support

Reads and writes 1, 4, 8 and 24 bit colour files.
Supports uncompressed, Packbits and LZW compressed files only.
Input option: index=N (default is 0)
Output options: artist=,software=,make=,model=,host=,documentname=,pagename=,
		imagedescription=,pal1bpp

Added support to allow SamplePerPixel >= 3 for RGB data (TIFF 6.0 new feature).
Added rejection test of FillOrder!=1.
Added rejection test of PlanarConfiguration!=1.
Added support for CMYK images.
Changed write of 1bpp data to Baseline black and white write.
Added pal1bpp output option meaning don't force Baseline write of 1bpp data.
Improved error messages substantially.
Fixed Packbits compression.
Added support for PlanarConfiguration==2 for RGB images only.
Added Predictor==2 support for bps==8, spp=1 case.
Added Predictor==2 support for bps==8, spp>=3 case.
Removed Too Many Strips limitation.

*/

/*...includes:0:*/
#include <stdio.h>
#include <string.h>
#include "standard.h"
#include "dxegbm.h"
#include "tifh.h"

#define INIT_CODE_SIZE  9
#define	CLEAR_CODE	0x100
#define	EOI_CODE	0x101
#define INIT_HASH(p)    (((p)+3)*301)
#define L_BUF 1024

/*...e*/
static GBMFT dxe_gbmft =
	{
	"TIFF",
	"Tagged Image File Format support (almost TIFF 6.0 Baseline)",
	"TIF",
    GBM_FT_R1|GBM_FT_R4|GBM_FT_R8|GBM_FT_R24|\
    GBM_FT_W1|GBM_FT_W4|GBM_FT_W8|GBM_FT_W24,
	};


/*...find_word:0:*/

/*...error codes:0:*/
#define	GBM_ERR_TIF_VERSION		((GBM_ERR) 800)
#define	GBM_ERR_TIF_N_TAGS		((GBM_ERR) 801)
#define GBM_ERR_TIF_TAG_TYPE    ((GBM_ERR) 802)
#define	GBM_ERR_TIF_HEADER		((GBM_ERR) 803)
#define GBM_ERR_TIF_MISSING_TAG ((GBM_ERR) 804)
#define	GBM_ERR_TIF_SPP_BIT		((GBM_ERR) 805)
#define	GBM_ERR_TIF_BPS_BIT		((GBM_ERR) 806)
#define	GBM_ERR_TIF_SPP_RGB		((GBM_ERR) 807)
#define	GBM_ERR_TIF_BPS_RGB		((GBM_ERR) 808)
#define	GBM_ERR_TIF_SPP_PAL		((GBM_ERR) 809)
#define	GBM_ERR_TIF_BPS_PAL		((GBM_ERR) 810)
#define GBM_ERR_TIF_SPP_CMYK    ((GBM_ERR) 811)
#define GBM_ERR_TIF_BPS_CMYK    ((GBM_ERR) 812)
#define GBM_ERR_TIF_COMP_1D_MH  ((GBM_ERR) 813)
#define	GBM_ERR_TIF_COMP_T4		((GBM_ERR) 814)
#define	GBM_ERR_TIF_COMP_T6		((GBM_ERR) 815)
#define	GBM_ERR_TIF_COMP		((GBM_ERR) 816)
#define GBM_ERR_TIF_COLORMAP    ((GBM_ERR) 817)
#define	GBM_ERR_TIF_CORRUPT		((GBM_ERR) 818)
#define GBM_ERR_TIF_PREDICTOR   ((GBM_ERR) 819)
#define GBM_ERR_TIF_PHOTO_TRANS ((GBM_ERR) 821)
#define	GBM_ERR_TIF_PHOTO_Y_Cb_Cr	((GBM_ERR) 822)
#define	GBM_ERR_TIF_PHOTO		((GBM_ERR) 823)
#define GBM_ERR_TIF_FILLORDER   ((GBM_ERR) 824)
#define	GBM_ERR_TIF_PLANARCONFIG_1	((GBM_ERR) 825)
#define	GBM_ERR_TIF_PLANARCONFIG_12	((GBM_ERR) 826)
#define	GBM_ERR_TIF_INKSET		((GBM_ERR) 827)
#define	GBM_ERR_TIF_ORIENT		((GBM_ERR) 828)
#define	GBM_ERR_TIF_INDEX		((GBM_ERR) 829)
/*...e*/
/* for TIF */
#define MAX_STRIPS  ((PRIV_SIZE-9*sizeof(int)-0x100*sizeof(GBMRGB))/sizeof(long))


typedef struct
	{
	GBMRGB gbmrgb [0x100];
	int rps, spp, bps, comp, photo, orient, planar, predictor, inx;
	long so [MAX_STRIPS];
	} TIF_PRIV;

#define	ENC_NONE	((int) 1)
#define	ENC_G3_1D_MH	((int) 2)
#define	ENC_T4		((int) 3)
#define	ENC_T6		((int) 4)
#define	ENC_LZW		((int) 5)
#define	ENC_PACKBITS	((int) 32773)

#define	PHOTO_BIT0	0
#define	PHOTO_BIT1	1
#define	PHOTO_RGB	2
#define	PHOTO_PAL	3
#define	PHOTO_TRANS	4
#define	PHOTO_CMYK	5
#define	PHOTO_Y_Cb_Cr	6

/*...rgb_bgr:0:*/
static void rgb_bgr(byte *p, byte *q, int n)
	{
	while ( n-- )
		{
		byte	r = *p++;
		byte	g = *p++;
		byte	b = *p++;
        *q++ = b;
		*q++ = g;
		*q++ = r;
		}
	}
/*...e*/

/*...tif_qft:0:*/
DLLEXPORT GBM_ERR dxe_qft(GBMFT *gbmft)
	{
    *gbmft = dxe_gbmft;
	return ( GBM_ERR_OK );
	}
/*...e*/
/*...tif_rhdr:0:*/
/*...value_of_tag_def:0:*/
static long value_of_tag_def(IFD *ifd, short type, long def)
	{
	TAG	*tag;

	if ( (tag = locate_tag(ifd, type)) != NULL )
		return ( value_of_tag(tag) );
	else
		return ( def );
	}
/*...e*/

DLLEXPORT GBM_ERR dxe_rhdr(char *fn, int fd, GBM *gbm, char *opt)
	{
	TIF_PRIV *tif_priv = (TIF_PRIV *) gbm -> priv;
	TAG *tag_w, *tag_h, *tag_so;
	GBM_ERR rc;
	IFH	*ifh;
	IFD	*ifd;
	int	inx = 0, strip, n_strips, fillorder;
	char	*index;

    if ( (index = u_find_word_prefix(opt, "index=")) != NULL )
		sscanf(index + 6, "%d", &inx);
	tif_priv -> inx = inx;

	fn=fn; /* Suppress 'unref arg' compiler warnings */

    u_lseek(fd, 0L, SEEK_SET);
	switch ( read_ifh_and_ifd(fd, inx, &ifh) )
		{
		case TE_OK:		rc = GBM_ERR_OK;		break;
		case TE_MEM:		rc = GBM_ERR_MEM;		break;
		case TE_VERSION:	rc = GBM_ERR_TIF_VERSION;	break;
		case TE_N_TAGS:		rc = GBM_ERR_TIF_N_TAGS;	break;
		case TE_TAG_TYPE:	rc = GBM_ERR_TIF_TAG_TYPE;	break;
		case TE_N_IFD:		rc = GBM_ERR_TIF_INDEX;		break;
		default:		rc = GBM_ERR_TIF_HEADER;	break;
		}

	if ( rc != GBM_ERR_OK )
		return ( rc );

	ifd = ifh -> ifd;

	if ( (tag_w  = locate_tag(ifd, T_IMAGEWIDTH  )) == NULL ||
	     (tag_h  = locate_tag(ifd, T_IMAGELENGTH )) == NULL ||
	     (tag_so = locate_tag(ifd, T_STRIPOFFSETS)) == NULL )
		{
		free_ifh(ifh);
		return ( GBM_ERR_TIF_MISSING_TAG );
		}

	gbm -> w           = (int) value_of_tag(tag_w);
	gbm -> h           = (int) value_of_tag(tag_h);
	tif_priv -> photo  = (int) value_of_tag_def(ifd, T_PHOTOMETRIC    , 1L);
	tif_priv -> rps    = (int) value_of_tag_def(ifd, T_ROWSPERSTRIP   , (long) gbm -> h);
	tif_priv -> spp    = (int) value_of_tag_def(ifd, T_SAMPLESPERPIXEL, 1L);
	tif_priv -> bps    = (int) value_of_tag_def(ifd, T_BITSPERSAMPLE  , 1L);
	tif_priv -> comp   = (int) value_of_tag_def(ifd, T_COMPRESSION    , 1L);
	tif_priv -> orient = (int) value_of_tag_def(ifd, T_ORIENTATION    , 1L);
	tif_priv -> planar = (int) value_of_tag_def(ifd, T_PLANARCONFIG   , 1L);

	rc = GBM_ERR_OK;
	switch ( tif_priv -> photo )
		{
/*...PHOTO_BITx    - bitmap or greyscale:16:*/
case PHOTO_BIT0:
case PHOTO_BIT1:
	if ( tif_priv -> spp != 1 )
		rc = GBM_ERR_TIF_SPP_BIT;
	else if ( tif_priv -> bps != 1 && tif_priv -> bps != 4 && tif_priv -> bps != 8 )
		rc = GBM_ERR_TIF_BPS_BIT;
	else
		{
		int	i, n_pal;

		n_pal = ( 1 << tif_priv -> bps );
		for ( i = 0; i < n_pal; i++ )
			{
			tif_priv -> gbmrgb [i].r =
			tif_priv -> gbmrgb [i].g =
			tif_priv -> gbmrgb [i].b = (byte) ((0xff * i) / (n_pal - 1));
			}
		if ( tif_priv -> photo == PHOTO_BIT0 )
			for ( i = 0; i < n_pal; i++ )
				{
				tif_priv -> gbmrgb [i].r ^= 0xff;
				tif_priv -> gbmrgb [i].g ^= 0xff;
				tif_priv -> gbmrgb [i].b ^= 0xff;
				}
		}
	gbm -> bpp = tif_priv -> bps;
	break;
/*...e*/
/*...PHOTO_RGB     - 24 bit data:16:*/
/* It is possible for sample per pixel to be greater than 3.
   This implies there are extra samples (which we will ignore).
   This is a new TIFF 6.0 feature. */

case PHOTO_RGB:
	if ( tif_priv -> spp < 3 )
		rc = GBM_ERR_TIF_SPP_RGB;
	else if ( tif_priv -> bps != 8 )
		rc = GBM_ERR_TIF_BPS_RGB;
	gbm -> bpp = 24;
	break;
/*...e*/
/*...PHOTO_PAL     - palettised:16:*/
/*
There are 2 known bugs in commonly available TIFF files today.
UBU will only write a ColorMap tag with a length field of 256 (not 256 * 3).
This bug is fixed inside my TIFF library itself!
OS/2 Image Support will write all palette entrys as bytes!
*/

case PHOTO_PAL:
	if ( tif_priv -> spp != 1 )
		rc = GBM_ERR_TIF_SPP_PAL;
	else if ( tif_priv -> bps != 1 && tif_priv -> bps != 4 && tif_priv -> bps != 8 )
		rc = GBM_ERR_TIF_BPS_PAL;
	else
		{
		int	i, n_pal;
		TAG	*tag_cm;

		n_pal = (1 << tif_priv -> bps);
		if ( (tag_cm = locate_tag(ifd, T_COLORMAP)) != NULL )
			{
			GBMRGB	*gbmrgb;

			for ( i = 0, gbmrgb = tif_priv -> gbmrgb; i < n_pal; i++, gbmrgb++ )
				{
				gbmrgb -> r = (byte) (value_of_tag_n(tag_cm,             i) >> 8);
				gbmrgb -> g = (byte) (value_of_tag_n(tag_cm,     n_pal + i) >> 8);
				gbmrgb -> b = (byte) (value_of_tag_n(tag_cm, 2 * n_pal + i) >> 8);
				}
/*...fix for OS/2 Image Support (and others):40:*/
{
byte	bugfix = 0;

for ( i = 0, gbmrgb = tif_priv -> gbmrgb; i < n_pal; i++, gbmrgb++ )
	bugfix |= (gbmrgb -> r | gbmrgb -> g | gbmrgb -> b);

if ( bugfix == 0 )
	for ( i = 0, gbmrgb = tif_priv -> gbmrgb; i < n_pal; i++, gbmrgb++ )
		{
		gbmrgb -> r = (byte) value_of_tag_n(tag_cm,             i);
		gbmrgb -> g = (byte) value_of_tag_n(tag_cm,     n_pal + i);
		gbmrgb -> b = (byte) value_of_tag_n(tag_cm, 2 * n_pal + i);
		}
}
/*...e*/
			}
		else
			rc = GBM_ERR_TIF_COLORMAP;
		}
	gbm -> bpp = tif_priv -> bps;
	break;
/*...e*/
/*...PHOTO_TRANS   - transparency mask:16:*/
case PHOTO_TRANS:
	rc = GBM_ERR_TIF_PHOTO_TRANS;
	break;
/*...e*/
/*...PHOTO_CMYK    - CMYK or other seperated image:16:*/
/* This is a colour seperated image.
   Typically expect 4 seperations, for CMYK.
   Can be other numbers, and possibly 4 with non standard ink colours.
   Ignore all but 4 seperations which are CMYK.
   Consider this a 24 bit RGB, mapping will occur from CMYK to RGB. */

case PHOTO_CMYK:
	if ( tif_priv -> spp != 4 )
		rc = GBM_ERR_TIF_SPP_CMYK;
	else if ( tif_priv -> bps != 8 )
		rc = GBM_ERR_TIF_BPS_CMYK;
	else if ( value_of_tag_def(ifd, T_INKSET, 1L) != 1 )
		rc = GBM_ERR_TIF_INKSET;
	else
		gbm -> bpp = 24;
	break;
/*...e*/
/*...PHOTO_Y_Cb_Cr - Y-Cb-Cr colour space:16:*/
case PHOTO_Y_Cb_Cr:
	rc = GBM_ERR_TIF_PHOTO_Y_Cb_Cr;
	break;
/*...e*/
/*...default       - wierd PhotometricInterpretation:16:*/
default:
	rc = GBM_ERR_TIF_PHOTO;
	break;
/*...e*/
		}

	if ( rc != GBM_ERR_OK )
		{ free_ifh(ifh); return ( rc ); }

	/* Remember where strips are, and how big they are */

	n_strips = (gbm -> h + (tif_priv -> rps - 1)) / tif_priv -> rps;
	if ( tif_priv -> photo == PHOTO_RGB && tif_priv -> planar == 2 )
		n_strips *= 3;

	if ( n_strips <= MAX_STRIPS )
		for ( strip = 0; strip < n_strips; strip++ )
			tif_priv -> so [strip] = value_of_tag_n(tag_so, strip);

	if ( tif_priv -> comp != ENC_NONE     &&
	     tif_priv -> comp != ENC_PACKBITS &&
	     tif_priv -> comp != ENC_LZW      )
/*...reject compression type:16:*/
{
free_ifh(ifh);
switch ( tif_priv -> comp )
	{
	case ENC_G3_1D_MH:	return ( GBM_ERR_TIF_COMP_1D_MH );
	case ENC_T4:		return ( GBM_ERR_TIF_COMP_T4    );
	case ENC_T6:		return ( GBM_ERR_TIF_COMP_T6    );
	default:		return ( GBM_ERR_TIF_COMP       );
	}
}
/*...e*/

	if ( tif_priv -> orient != 1 && tif_priv -> orient != 4 )
		{ free_ifh(ifh); return ( GBM_ERR_TIF_ORIENT ); }

	fillorder = (int) value_of_tag_def(ifd, T_FILLORDER, 1L);
	if ( fillorder != 1 )
		{ free_ifh(ifh); return ( GBM_ERR_TIF_FILLORDER ); }

	if ( tif_priv -> photo == PHOTO_RGB )
		/* Allow photo of 1 or 2 */
		{
		if ( tif_priv -> planar != 1 && tif_priv -> planar != 2 )
			{ free_ifh(ifh); return ( GBM_ERR_TIF_PLANARCONFIG_12 ); }
		}
	else
		/* Allow photo of 1 only */
		{
		if ( tif_priv -> planar != 1 )
			{ free_ifh(ifh); return ( GBM_ERR_TIF_PLANARCONFIG_1 ); }
		}

	tif_priv -> predictor = (int) value_of_tag_def(ifd, T_PREDICTOR, 1L);

	/* Only allow predictor of 1, unless a special case we handle */
	if ( tif_priv -> predictor != 1 &&
	     !(tif_priv -> comp == ENC_LZW &&
	       tif_priv -> bps == 8 &&
	       (tif_priv -> spp == 1 || tif_priv -> spp >= 3)) )
		{ free_ifh(ifh); return ( GBM_ERR_TIF_PREDICTOR ); }

	free_ifh(ifh);

	return ( GBM_ERR_OK );
	}

/*...e*/
/*...tif_rpal:0:*/
DLLEXPORT GBM_ERR dxe_rpal( int fd, GBM *gbm, GBMRGB *gbmrgb )
	{
	TIF_PRIV *tif_priv = (TIF_PRIV *) gbm -> priv;

	fd=fd; /* Suppress 'unref arg' compiler warning */

	if ( gbm -> bpp != 24 )
        memcpy( gbmrgb, tif_priv -> gbmrgb, (1 << gbm -> bpp ) * sizeof(GBMRGB) );

	return ( GBM_ERR_OK );
	}

/*...e*/
/*...tif_rdata:0:*/
/*
TIFF data is usually stored top-left-origin based.
ie: We ignore the private "Orientation" tag.
Our data format in memory has bigger padding than TIFF.
It has 'istride' bytes per line, GBM uses/requires 'stride' bytes per line.
Therefore we read it to the part of the strip area and then expand downwards.
*/

/*...get_strip_packbits - get bytes with packbits decompression:0:*/
static GBM_ERR get_strip_packbits(int fd, byte *dest, long n_bytes)
	{
	AHEAD	*ahead;

    if ( (ahead = u_create_ahead(fd)) == NULL )
		return ( GBM_ERR_MEM );

	while ( n_bytes > 0 )
		{
        byte b = u_next(ahead);

		if ( b < 0x80 )
			{
			unsigned int count = b + 1;

			do
                *dest++ = u_next(ahead);
			while ( --count > 0 );
			n_bytes -= (b + 1);
			}
		else if ( b > 0x80 )
			{
			unsigned int count = 0x101 - (unsigned int) b;
            byte c = u_next(ahead);

			memset(dest, c, count);
			dest += count;
			n_bytes -= count;
			}
		}

    u_destroy_ahead(ahead);
	return ( GBM_ERR_OK );
	}
/*...e*/
/*...get_strip_lzw      - get bytes with TIFF style LZW decompression:0:*/
/*
This code run on output of FotoTouch and some sample TIFFs from an afs
directory on the IBM IP-network.
*/

/*...read context:0:*/
typedef struct
	{
    int fd;             /* File descriptor to read */
	int inx, size;			/* Index and size in bits */
	byte buf [255+3];		/* Buffer holding bits */
	int code_size;			/* Number of bits to return at once */
	word read_mask;			/* 2^code_size-1 */
	} READ_CONTEXT;

static word read_code(READ_CONTEXT *c)
	{
	int raw_code, byte_inx;

	if ( c -> inx + c -> code_size > c -> size )
/*...not enough bits in buffer, refill it:16:*/
/* Not very efficient, but infrequently called */

{
int bytes_to_lose = (c -> inx >> 3);
int bytes;

/* Note biggest code size is 12 bits */
/* And this can at worst span 3 bytes */
memcpy(c -> buf, c -> buf + bytes_to_lose, 3);
(c -> inx) &= 7;
(c -> size) -= (bytes_to_lose << 3);
bytes = 255 - ( c -> size >> 3 );
if ( (bytes = u_read(c -> fd, c -> buf + (c -> size >> 3), bytes)) <= 0 )
	return ( 0xffff );
(c -> size) += (bytes << 3);
}
/*...e*/

	byte_inx = (c -> inx >> 3);
	raw_code = ((c -> buf [byte_inx    ]) << 16) +
		   ((c -> buf [byte_inx + 1]) <<  8) +
	           ( c -> buf [byte_inx + 2]       ) ;
	raw_code <<= ((c -> inx) & 7);
	(c -> inx) += (byte) (c -> code_size);
	raw_code >>= ( 24 - c -> code_size );
	return ( (word) raw_code & c -> read_mask );
	}
/*...e*/


static GBM_ERR get_strip_lzw(int fd, byte *dest, long n_bytes)
	{
	word max_code;			/* 1 << code_size */
	word clear_code;		/* Code to clear table */
	word eoi_code;			/* End of information code */
	word first_free_code;		/* First free code */
	word free_code;			/* Next available free code slot */
	int i, out_count = 0;
    word code, cur_code, old_code=0, in_code, fin_char=0;
	word *prefix, *suffix, *outcode;
	READ_CONTEXT c;
	BOOLEAN table_full = FALSE;

	n_bytes=n_bytes; /* Suppress 'unref arg' compiler warning */

    if ( (prefix = (word *) u_malloc(4096 * sizeof(word))) == NULL )
		return ( GBM_ERR_MEM );
    if ( (suffix = (word *) u_malloc(4096 * sizeof(word))) == NULL )
		{
        u_free(prefix);
		return ( GBM_ERR_MEM );
		}
    if ( (outcode = (word *) u_malloc(4097 * sizeof(word))) == NULL )
		{
        u_free(suffix);
        u_free(prefix);
		return ( GBM_ERR_MEM );
		}

	/* Initial read context */

	c.inx            = 0;
	c.size           = 0;
	c.fd             = fd;
	c.code_size      = INIT_CODE_SIZE;
	c.read_mask      = (word) ( (1 << INIT_CODE_SIZE) - 1 );

	/* 2^min_code size accounts for all colours in file */

	clear_code = (word) ( 1 << (INIT_CODE_SIZE - 1) );
	eoi_code = (word) (clear_code + 1);
	free_code = first_free_code = (word) (clear_code + 2);

	max_code = (word) ( 1 << INIT_CODE_SIZE );

	while ( (code = read_code(&c)) != eoi_code && code != 0xffff )
		{
		if ( code == clear_code )
			{
			c.code_size = INIT_CODE_SIZE;
			c.read_mask = (word) ( (1 << INIT_CODE_SIZE) - 1);
			max_code = (word) ( 1 << INIT_CODE_SIZE );
			free_code = first_free_code;
			cur_code = old_code = code = read_code(&c);
			if ( code == eoi_code || code == 0xffff )
				break;
			fin_char = cur_code;
			*dest++ = (byte) fin_char;
			table_full = FALSE;
			}
		else
			{
			cur_code = in_code = code;
			if ( cur_code >= free_code )
				{
				cur_code = old_code;
				outcode [out_count++] = fin_char;
				}
			while ( cur_code > 0xff )
				{
				if ( out_count > 4096 )
					{
                    u_free(outcode);
                    u_free(suffix);
                    u_free(prefix);
					return ( GBM_ERR_TIF_CORRUPT );
					}
				outcode [out_count++] = suffix [cur_code];
				cur_code = prefix [cur_code];
				}
			fin_char = cur_code;
			outcode [out_count++] = fin_char;
			for ( i = out_count - 1; i >= 0; i-- )
				*dest++ = outcode [i];
			out_count = 0;

			/* Update dictionary */

			if ( !table_full )
				{
				prefix [free_code] = old_code;
				suffix [free_code] = fin_char;

				/* Advance to next free slot */

				if ( ++free_code >= max_code - 1 )
					{
					if ( c.code_size < 12 )
						{
						c.code_size++;
						max_code <<= 1;
						c.read_mask = (word) (( 1 << c.code_size ) - 1);
						}
					else
						table_full = TRUE;
					}
				}
			old_code = in_code;
			}
		}

    u_free(outcode);
    u_free(suffix);
    u_free(prefix);

	if ( code == 0xffff )
		return ( GBM_ERR_READ );

	return ( GBM_ERR_OK );
	}
/*...e*/
/*...get_strip_lzw_pred - get_strip_lzw with non 1 predictor fixup:0:*/
static GBM_ERR get_strip_lzw_pred(
    int fd,
	byte *dest,
	long n_bytes,
	TIF_PRIV *tif_priv,
	int w, int h
	)
	{
	GBM_ERR rc;

	if ( (rc = get_strip_lzw(fd, dest, n_bytes)) != GBM_ERR_OK )
		return ( rc );

	if ( tif_priv -> predictor == 2 )
		/* Note: This is only allowed if bps==8 */
			{
			int x, y, spp = tif_priv -> spp;
			for ( y = 0; y < h; y++, dest += w * spp )
				for ( x = spp; x < w * spp; x++ )
					dest [x] += dest [x - spp];
			}

	return ( GBM_ERR_OK );
	}
/*...e*/
/*...get_strip_comp     - get strip, dealing with any compression:0:*/
/* n_bytes is passed in as istride * n_lines */

static GBM_ERR get_strip_comp(
    int fd,
	byte *dest,
	long so, long n_bytes,
	TIF_PRIV *tif_priv,
	int w, int h
	)
	{
    u_lseek(fd, so, SEEK_SET);
	switch ( tif_priv -> comp )
		{
/*...ENC_NONE     - no compression:16:*/
case ENC_NONE:
    return ( ( (int) n_bytes == u_read(fd, dest, (int) n_bytes) ) ?
			GBM_ERR_OK : GBM_ERR_READ );
/*...e*/
/*...ENC_PACKBITS - packbits:16:*/
case ENC_PACKBITS:
	return ( get_strip_packbits(fd, dest, n_bytes) );
/*...e*/
/*...ENC_LZW      - lzw:16:*/
case ENC_LZW:
	return ( get_strip_lzw_pred(fd, dest, n_bytes, tif_priv, w, h) );
/*...e*/
		}
	return ( GBM_ERR_NOT_SUPP );
	}
/*...e*/
/*...get_strip          - get strip, discarding extra samples, and CMYK mapping:0:*/
/*
If there are too many samples per pixel, this code can ignore the extra ones.
Also, if CMYK data is being read, it will read the CMYK, and convert.
This requires a temporary buffer, to read the original data in.
The original data is then 'converted'.
*/

static GBM_ERR get_strip(
    int fd,
	byte *dest,
	long so, long n_bytes,
	TIF_PRIV *tif_priv,
	int w, int h
	)
	{
	byte *buf = dest;
	GBM_ERR rc;

	if ( tif_priv -> photo == PHOTO_CMYK )
/*...allocate space for CMYK image:16:*/
{
n_bytes = (long) w * 4L * (long) h;
if ( (buf = u_malloc(n_bytes)) == NULL )
	return ( GBM_ERR_MEM );
}
/*...e*/
	else if ( tif_priv -> photo == PHOTO_RGB && tif_priv -> spp > 3 )
/*...allocate space for image + extra samples:16:*/
{
n_bytes = (long) w * tif_priv -> spp * (long) h;
if ( (buf = u_malloc(n_bytes)) == NULL )
	return ( GBM_ERR_MEM );
}
/*...e*/

	if ( (rc = get_strip_comp(fd, buf, so, n_bytes, tif_priv, w, h)) != GBM_ERR_OK )
		{
		if ( buf != dest )
            u_free(buf);
		return ( rc );
		}

	if ( tif_priv -> photo == PHOTO_CMYK )
/*...convert from CMYK to RGB:16:*/
{
int x, yy;
byte *buf_p = buf, *dest_p = dest;
for ( yy = 0; yy < h; yy++ )
	for ( x = 0; x < w; x++ )
		{
		word c = *buf_p++;
		word m = *buf_p++;
		word y = *buf_p++;
		word k = *buf_p++;

		/* Exploit 8 bit modulo arithmetic by biasing by + 0x100 */

		word r = 0x1ff - (c + k);
		word g = 0x1ff - (m + k);
		word b = 0x1ff - (y + k);

		if ( r < 0x100 ) r = 0x100;
		if ( g < 0x100 ) g = 0x100;
		if ( b < 0x100 ) b = 0x100;

		*dest_p++ = (byte) r;
		*dest_p++ = (byte) g;
		*dest_p++ = (byte) b;
		}

u_free(buf);
}
/*...e*/
	else if ( tif_priv -> photo == PHOTO_RGB && tif_priv -> spp > 3 )
/*...extract, ignoring extra-samples:16:*/
{
int x, y, skip = tif_priv -> spp - 2;
byte *buf_p = buf, *dest_p = dest;
for ( y = 0; y < h; y++ )
	for ( x = 0; x < w; x++ )
		{
		*dest_p++ = *buf_p++;
		*dest_p++ = *buf_p++;
		*dest_p++ = *buf_p  ;
		buf_p += skip;
		}

u_free(buf);
}
/*...e*/

	return ( GBM_ERR_OK );
	}
/*...e*/
/*...get_image          - get all strips, result is whole image:0:*/
/*
This routine calls get_strip() to get strips data one after another until it
has read the entire images worth of data. Of course, scan lines are aligned on
byte boundaries, and the data is usually considered to be image top to bottom.
*/

static GBM_ERR get_image(
    int fd,
	byte *dest,
	TIF_PRIV *tif_priv,
	long *so,
	GBM *gbm,
	int *strip
	)
	{
	GBM_ERR	rc;
	int y, istride = ((gbm -> w * gbm -> bpp + 7) / 8);

	for ( y = 0; y < gbm -> h; y += tif_priv -> rps, (*strip)++ )
		{
		int n_lines = min(tif_priv -> rps, gbm -> h - y);
		if ( (rc = get_strip(fd, dest + y * istride,
				     so [*strip],
				     (long) n_lines * (long) istride,
				     tif_priv,
				     gbm -> w, n_lines)) != GBM_ERR_OK )
			return ( rc );
		}

	return ( GBM_ERR_OK );
	}
/*...e*/
/*...get_image_planar   - get all strips, allowing for PlanarConfiguration:0:*/
/*
get_image() will assume the data is in PlanarConfiguration==1, ie: chunky
pixel mode. This is TRUE most of the time. But sometimes we will actually
allow PlanarConfiguration==2. In this case, use get_image() and then fix-up
the results.
*/

static GBM_ERR get_image_planar(
    int fd,
	byte *dest,
	TIF_PRIV *tif_priv,
	long *so,
	GBM *gbm
	)
	{
	int strip = 0;

	if ( tif_priv -> photo == PHOTO_RGB &&
	     tif_priv -> planar == 2 )
/*...read 3 seperate planes, and combine them:16:*/
/*
If PhotometricInterpretation==RGB and
   SamplesPerPixel>=3 and
   BitsPerSample==8 then
	we allow PlanarConfiguration==2

I have successfully read in a PlanarConfiguration==2 RGB TIFF file by using
the "read 3 images" logic below. This image had RowsPerStrip==1, and so
technically either fold below would have worked. I think the read 3 images
logic is a better interpretation of the TIFF 6.0 spec., but until I find
some other images I can handle, I will keep the alternative peice of code.
*/

{
GBM_ERR rc;
GBM gbm_planar;
int saved_spp = tif_priv -> spp;
int n_bytes = gbm -> w * gbm -> h;
int x, y;
byte *buf, *p [3];

if ( (buf = u_malloc(n_bytes * 3)) == NULL )
	return ( GBM_ERR_MEM );
p [0] = buf;
p [1] = p [0] + n_bytes;
p [2] = p [1] + n_bytes;

tif_priv -> spp = 1;
/*...read 3 images:16:*/
{
int i;

gbm_planar.w   = gbm -> w;
gbm_planar.h   = gbm -> h;
gbm_planar.bpp = 8;
for ( i = 0; i < 3; i++ )
	if ( (rc = get_image(fd, p [i], tif_priv, so, &gbm_planar, &strip)) != GBM_ERR_OK )
		{
		tif_priv -> spp = saved_spp;
        u_free(buf);
		return ( rc );
		}
}
/*...e*/
#ifdef NEVER
/*...read single image 3x too high:16:*/
gbm_planar.w   = gbm -> w;
gbm_planar.h   = gbm -> h * 3;
gbm_planar.bpp = 8;
if ( (rc = get_image(fd, buf, tif_priv, so, &gbm_planar, &strip)) != GBM_ERR_OK )
	{
	tif_priv -> spp = saved_spp;
    u_free(buf);
	return ( rc );
	}
/*...e*/
#endif
tif_priv -> spp = saved_spp;

for ( y = 0; y < gbm -> h; y++ )
	for ( x = 0; x < gbm -> w; x++ )
		{
		*dest++ = *(p [0])++;
		*dest++ = *(p [1])++;
		*dest++ = *(p [2])++;
		}
u_free(buf);
return ( GBM_ERR_OK );
}
/*...e*/
	else
		return ( get_image(fd, dest, tif_priv, so, gbm, &strip) );
	}
/*...e*/
/*...get_image_orient   - get all strips, correctly orientated:0:*/
static GBM_ERR get_image_orient(
    int fd,
	byte *dest,
	TIF_PRIV *tif_priv,
	long *so,
	GBM *gbm
	)
	{
	switch ( tif_priv -> orient )
		{
/*...1 - usual Baseline required case:16:*/
/*
File has array [scanlines_down] of array [pixels_across] of pixel.
GBMs bitmap data is array [scanlines_up] of array [pixels_across] of pixel.
So call get_image_planar(), and vertically reflect resulting data.
*/

case 4:
	{
	int istride = ((gbm -> bpp * gbm -> w + 7) / 8);
	byte *p0, *p1, *p2;
	GBM_ERR rc;
	if ( (rc = get_image_planar(fd, dest, tif_priv, so, gbm)) != GBM_ERR_OK )
		return ( rc );
    if ( (p0 = u_malloc(istride)) == NULL )
		return ( GBM_ERR_MEM );
	for ( p1 = dest, p2 = p1 + (gbm -> h - 1) * istride;
	      p1 < p2;
	      p1 += istride, p2 -= istride )
		{
		memcpy(p0, p1, istride);
		memcpy(p1, p2, istride);
		memcpy(p2, p0, istride);
		}
    u_free(p0);
	return ( GBM_ERR_OK );
	}
/*...e*/
/*...4 - vertically swapped case we can easily handle:16:*/
/*
File has array [scanlines_up] of array [pixels_across] of pixel.
Exactly matches GBMs layout of a bitmap.
So simply call get_image() and be done with.
*/

case 1:
	return ( get_image_planar(fd, dest, tif_priv, so, gbm) );
/*...e*/
		}
	return ( GBM_ERR_NOT_SUPP ); /* Shouldn't get here */
	}
/*...e*/
/*...get_image_strippy  - get all strips, when there are loads of them:0:*/
static GBM_ERR get_image_strippy(
    int fd,
	byte *dest,
	TIF_PRIV *tif_priv,
	GBM *gbm
	)
	{
	int n_strips = (gbm -> h + (tif_priv -> rps - 1)) / tif_priv -> rps;
	long *so = tif_priv -> so;

	if ( n_strips > MAX_STRIPS )
/*...re-read TIFF file header:16:*/
{
GBM_ERR rc;
int strip;
IFH *ifh;
IFD *ifd;
TAG *tag_so;

if ( (so = u_malloc(n_strips * sizeof(long))) == NULL )
	return ( GBM_ERR_MEM );

u_lseek(fd, 0L, SEEK_SET);
if ( read_ifh_and_ifd(fd, tif_priv -> inx, &ifh) != TE_OK )
	{
    u_free(so);
	return ( GBM_ERR_MEM );
	}
ifd = ifh -> ifd;
tag_so = locate_tag(ifd, T_STRIPOFFSETS);
for ( strip = 0; strip < n_strips; strip++ )
	so [strip] = value_of_tag_n(tag_so, strip);
free_ifh(ifh);

rc = get_image_orient(fd, dest, tif_priv, so, gbm);

u_free(so);
return ( rc );
}
/*...e*/
	else
		return ( get_image_orient(fd, dest, tif_priv, so, gbm) );
	}
/*...e*/

DLLEXPORT GBM_ERR dxe_rdata(int fd, GBM *gbm, byte *data)
	{
	TIF_PRIV *tif_priv = (TIF_PRIV *) gbm -> priv;
	int stride = ((gbm -> bpp * gbm -> w + 31) / 32) * 4;
	int istride = ((gbm -> bpp * gbm -> w + 7) / 8);
	int bias = gbm -> h * (stride - istride);
	GBM_ERR	rc;

	/* Read in data, packed close, and upside down */

	if ( (rc = get_image_strippy(fd, data + bias, tif_priv, gbm)) != GBM_ERR_OK )
		return ( rc );

/*...now expand out from byte padding to dword padding:8:*/
if ( bias )
	{
	int y;
	byte *dest = data, *src  = data + bias;

	for ( y = 0; y < gbm -> h; y++, dest += stride, src += istride )
		memcpy(dest, src, istride);
	}
/*...e*/
/*...now RGB -\62\ BGR if 24 bit data returned:8:*/
if ( gbm -> bpp == 24 )
	{
	int y;
	byte *p = data;

	for ( y = 0; y < gbm -> h; y++, p += stride )
		rgb_bgr(p, p, gbm -> w);
	}
/*...e*/

	return ( GBM_ERR_OK );
	}
/*...e*/
/*...tif_w:0:*/
/*
Write out data in a single large strip for now.
Note that the palette entrys are written as ((r << 8) | r).
This means they are 257/256 too big (insignificant).
Most programs only look at the top 8 bits (ie: no error).
A few (incorrectly) look at the bottom 8 bits.
Therefore we cater for all programs, with minimal fuss.
*/

/*...user_tag:0:*/
static BOOLEAN user_tag(IFD *ifd, char *name, short type, char *opt, char *def)
	{
	char	buf [200+1], *s;

    if ( (s = u_find_word_prefix(opt, name)) != NULL )
		sscanf(s + strlen(name), "%s", buf);
	else
		strcpy(buf, def);

	if ( *buf == '\0' )
		return ( TRUE );

	return ( add_ascii_tag(ifd, type, buf) );
	}
/*...e*/
/*...write_strip:0:*/
static GBM_ERR write_strip(int fd, int w, int h, int bpp, byte *data)
	{
	int stride = ((bpp * w + 31) / 32) * 4;
	int ostride = ((bpp * w + 7) / 8);
	int y;
	data += ((h - 1) * stride);

	if ( bpp == 24 )
/*...reverse rgb/bgr ordering and write:16:*/
{
byte *line;

if ( (line = u_malloc(ostride)) == NULL )
	return ( GBM_ERR_MEM );
for ( y = 0; y < h; y++, data -= stride )
	{
	rgb_bgr(data, line, w);
    if ( u_write(fd, line, ostride) != ostride )
		{
        u_free(line);
		return ( GBM_ERR_WRITE );
		}
	}
u_free(line);
}
/*...e*/
	else
/*...write:16:*/
for ( y = 0; y < h; y++, data -= stride )
    if ( u_write(fd, data, ostride) != ostride )
		return ( GBM_ERR_WRITE );
/*...e*/
	return ( GBM_ERR_OK );
	}
/*...e*/
/*...write_strip_lzw:0:*/
/*
This will be a tricky bit of code to get right.
This code blantantly copied and hacked from that in gbmgif.c!
*/


/*...write context:0:*/

typedef struct
	{
    int fd;             /* Open file descriptor to write to */
	int inx;			/* Bit index into buf */
	int code_size;			/* Code size in bits */
	byte buf [L_BUF+2];		/* Biggest block + overflow space */
	} WRITE_CONTEXT;

static BOOLEAN write_code(int code, WRITE_CONTEXT *w)
	{
	byte *buf = w -> buf + (w -> inx >> 3);

	code <<= (24-w->code_size);
	code >>= (w->inx&7);
	*buf++ |= (byte) (code >> 16);
	*buf++  = (byte) (code >>  8);
	*buf    = (byte)  code       ;

	(w -> inx) += (w -> code_size);
	if ( w -> inx >= L_BUF * 8 )
		/* Flush out full buffer */
		{
        if ( u_write(w -> fd, w -> buf, L_BUF) != L_BUF )
			return ( FALSE );
		memcpy(w -> buf, w -> buf + L_BUF, 2);
		memset(w -> buf + 2, 0, L_BUF);
		(w -> inx) -= (L_BUF * 8);
		}

	return ( TRUE );
	}

static BOOLEAN flush_code(WRITE_CONTEXT *w)
	{
	int bytes = ((w -> inx + 7) >> 3);

	if ( bytes )
		{
        if ( u_write(w -> fd, w -> buf, bytes) != bytes )
			return ( FALSE );
		}

	return ( TRUE );
	}
/*...e*/


#ifdef NEVER
 #define MAX_STRING  64      /* Just to prevent huge strings */
 #define MAX_STRINGS 64000       /* Small enough for 16 bit */
 #define MAX_HASH    8193        /* Prime, and much > 4096 */
#endif


static GBM_ERR write_strip_lzw(int fd, int w, int h, int bpp, byte *data)
	{
	int stride = ((bpp * w + 31) / 32) * 4;
	int ostride = ((bpp * w + 7) / 8);
	WRITE_CONTEXT wc;
    word lastentry=0;
	int x, y, nextstring, numentries, numrealentries;
	unsigned int hashcode, lenstring;
	byte string [MAX_STRING-2], *strings, **hashtable;

    if ( (strings = (byte *) u_malloc(MAX_STRINGS)) == NULL )
		return ( GBM_ERR_MEM );

    if ( (hashtable = (byte **) u_malloc(MAX_HASH * sizeof(byte *))) == NULL )
		{
        u_free(strings);
		return ( GBM_ERR_MEM );
		}

	/* Setup write context */

	wc.fd        = fd;
	wc.inx       = 0;
	wc.code_size = INIT_CODE_SIZE;
	memset(wc.buf, 0, sizeof(wc.buf));

	if ( !write_code(CLEAR_CODE, &wc) )
		{
        u_free(hashtable);
        u_free(strings);
		return ( GBM_ERR_WRITE );
		}

	numentries     = 0;
	numrealentries = 0;
	nextstring     = 0;
	lenstring      = 0;
	for ( hashcode = 0; hashcode < MAX_HASH; hashcode++ )
		hashtable [hashcode] = NULL;

	data += ( (h - 1) * stride );
	for ( y = h - 1; y >= 0; y--, data -= stride )
		{
		int inx1 = 0, inx2 = 2;
		for ( x = 0; x < ostride; x++ )
			{
			byte col;
/*...get byte to write to col:24:*/
if ( bpp == 24 )
	/* Have to handle rgb/bgr reverse as we go along */
	{
	col = data[inx1+inx2];
	if ( --inx2 < 0 )
		{
		inx1 += 3; inx2 = 2;
		}
	}
else
	col = data[x];
/*...e*/
/*...lzw encode:24:*/
string [0] = ++lenstring;
string [lenstring] = col;
if ( lenstring == 1 )
	{
	lastentry = col;
	hashcode = INIT_HASH(col);
	}
else
	{
	unsigned int i, j;

	/* Find current string within strings, by generating hash of string,
	   and looking in strings for it starting at hashcode */

	hashcode *= ( col + lenstring );
	j = (hashcode % MAX_HASH);
	for ( i = 0; i < MAX_HASH; i++ )
		{
		if ( ++j >= MAX_HASH )
			j = 0;
		if ( hashtable [j] == NULL )
			break;
		if ( !memcmp(hashtable [j]+2, string, 1 + lenstring) )
			break;
		}
	if ( hashtable [j] != NULL && 2 + 1 + lenstring < MAX_STRING )
		/* Found in the strings table */
		{
		byte *ptr = hashtable [j];
		lastentry = (word) ptr [0] + ((word) ptr [1] << 8);
		}
	else
		/* Not found, or length too long */
		{
		if ( !write_code(lastentry, &wc) )
			{
            u_free(hashtable);
            u_free(strings);
			return ( GBM_ERR_WRITE );
			}
		numentries++;
		if ( hashtable [j] == NULL )
			/* Add string */
			{
			word entry = (word) (EOI_CODE + numentries);
			byte *ptr = strings + nextstring;
			hashtable [j] = ptr;
			ptr [0] = (byte)  entry;
			ptr [1] = (byte) (entry >> 8);
			memcpy(ptr + 2, string, 1 + lenstring);
			nextstring += (2 + 1 + lenstring); 
			numrealentries++;
			}
		lenstring = string [0] = (byte) 1;
		lastentry = string [1] = (byte) col;
		hashcode = INIT_HASH(col);

		if ( EOI_CODE + numentries == (1 << wc.code_size) - 1 )
			/* Next code will be written longer */
			wc.code_size++;

		if ( EOI_CODE + numentries > 4093 ||
		     nextstring + MAX_STRING > MAX_STRINGS )
			/* Reset tables */
			{
			if ( !write_code(lastentry, &wc) )
				{
                u_free(hashtable);
                u_free(strings);
				return ( GBM_ERR_WRITE );
				}
			if ( !write_code(CLEAR_CODE, &wc) )
				{
                u_free(hashtable);
                u_free(strings);
				return ( GBM_ERR_WRITE );
				}
			numentries     = 0;
			numrealentries = 0;
			nextstring     = 0;
			lenstring      = 0;
			wc.code_size   = INIT_CODE_SIZE;
			for ( j = 0; j < MAX_HASH; j++ )
				hashtable [j] = NULL;
			}
		}
	}
/*...e*/
			}
		}

    u_free(hashtable);
    u_free(strings);

	if ( !write_code(lastentry, &wc) )
		return ( GBM_ERR_WRITE );
	if ( !write_code(EOI_CODE, &wc) )
		return ( GBM_ERR_WRITE );
	if ( !flush_code(&wc) )
		return ( GBM_ERR_WRITE );

	return ( GBM_ERR_OK );
	}
/*...e*/

DLLEXPORT GBM_ERR dxe_w(char *fn, int fd, GBM *gbm, GBMRGB *gbmrgb, byte *data, char *opt)
	{
    BOOLEAN baseline = ( u_find_word(opt, "pal1bpp") == NULL );
    BOOLEAN lzw      = ( u_find_word(opt, "lzw"    ) != NULL );
	IFH	*ifh;
	IFD	*ifd;
	long	w = gbm -> w;
	long	h = gbm -> h;
	long	stripoffset, stripbytecount;
	short	samplesperpixel, bitspersample [3], photo, comp;
	short	colormap [0x100+0x100+0x100];
	BOOLEAN	ok;
	GBM_ERR	rc;

	fn=fn; /* Suppress 'unref arg' compiler warning */

	if ( (ifh = make_ifh()) == NULL )
		return ( GBM_ERR_MEM );

	ifd = ifh -> ifd;

	if ( gbm -> bpp == 1 && baseline )
		{
		word k0 = (word) gbmrgb [0].r + (word) gbmrgb [0].g + (word) gbmrgb [0].b;
		word k1 = (word) gbmrgb [1].r + (word) gbmrgb [1].g + (word) gbmrgb [1].b;
		samplesperpixel   = 1;
		bitspersample [0] = 1;
		photo = ( k0 < k1 ) ? PHOTO_BIT1 : PHOTO_BIT0; /* Black is zero : White is zero */
		}
	else if ( gbm -> bpp == 24 )
		{
		samplesperpixel   = 3;
		bitspersample [0] = 
		bitspersample [1] = 
		bitspersample [2] = 8;
		photo = PHOTO_RGB;
		}
	else
		{
		samplesperpixel   = 1;
		bitspersample [0] = (short) gbm -> bpp;
		photo = PHOTO_PAL;
		}

	comp = ( lzw ) ? ENC_LZW : ENC_NONE;

	ok = add_long_tag(ifd, T_IMAGEWIDTH, &w, 1) &&
	     add_long_tag(ifd, T_IMAGELENGTH, &h, 1) &&
	     add_long_tag(ifd, T_STRIPOFFSETS, &stripoffset, 1) &&
	     add_long_tag(ifd, T_STRIPBYTECOUNTS, &stripbytecount, 1) &&
	     add_short_tag(ifd, T_SAMPLESPERPIXEL, &samplesperpixel, 1) &&
	     add_short_tag(ifd, T_BITSPERSAMPLE, bitspersample, samplesperpixel) &&
	     add_short_tag(ifd, T_PHOTOMETRIC, &photo, 1) &&
	     add_short_tag(ifd, T_COMPRESSION, &comp, 1) &&
	     user_tag(ifd, "artist=", T_ARTIST, opt, "") &&
	     user_tag(ifd, "software=", T_MAKE, opt, "Generalised Bitmap Module") &&
	     user_tag(ifd, "make=", T_MAKE, opt, "") &&
	     user_tag(ifd, "model=", T_MODEL, opt, "") &&
	     user_tag(ifd, "hostcomputer=", T_HOSTCOMPUTER, opt, "") &&
	     user_tag(ifd, "documentname=", T_DOCNAME, opt, "") &&
	     user_tag(ifd, "pagename=", T_PAGENAME, opt, "") &&
	     user_tag(ifd, "imagedescription=", T_DESCRIPTION, opt, "");

	if ( gbm -> bpp != 24 )
		{
		int	i, n_cols = (1 << gbm -> bpp);

		for ( i = 0; i < n_cols; i++ )
			{
			short	r = (short) gbmrgb [i].r;
			short	g = (short) gbmrgb [i].g;
			short	b = (short) gbmrgb [i].b;

			colormap [             i] = ((r << 8) | r);
			colormap [    n_cols + i] = ((g << 8) | g);
			colormap [2 * n_cols + i] = ((b << 8) | b);
			}
		if ( gbm -> bpp != 1 || !baseline )
			ok &= add_short_tag(ifd, T_COLORMAP, colormap, n_cols * 3);
		}

	if ( !ok )
		{
        free_ifh(ifh);
		return ( GBM_ERR_MEM );
		}

	if ( !write_ifh_and_ifd(ifh, fd) )
		{
		free_ifh(ifh);
		return ( GBM_ERR_WRITE );
		}

    stripoffset = u_lseek(fd, 0L, SEEK_CUR);

	if ( lzw )
		rc = write_strip_lzw(fd, gbm->w, gbm->h, gbm->bpp, data);
	else
		rc = write_strip(fd, gbm->w, gbm->h, gbm->bpp, data);
	
	if ( rc != GBM_ERR_OK )
		{
		free_ifh(ifh);
		return ( rc );
		}

    stripbytecount = u_lseek(fd, 0L, SEEK_CUR) - stripoffset;

	update_long_tag(ifd, T_STRIPOFFSETS, &stripoffset);
	update_long_tag(ifd, T_STRIPBYTECOUNTS, &stripbytecount);

	if ( !update_ifd(ifd, fd) )
		{
		free_ifh(ifh);
		return ( GBM_ERR_WRITE );
		}

	free_ifh(ifh);

	return ( GBM_ERR_OK );
	}
/*...e*/
/*...tif_err:0:*/
DLLEXPORT char *dxe_err(GBM_ERR rc)
	{
	switch ( (int) rc )
		{
		case GBM_ERR_TIF_VERSION:
			return ( "version number not 42" );
		case GBM_ERR_TIF_N_TAGS:
			return ( "too many tags in file" );
		case GBM_ERR_TIF_TAG_TYPE:
			return ( "bad tag type" );
		case GBM_ERR_TIF_HEADER:
			return ( "corrupt header" );
		case GBM_ERR_TIF_MISSING_TAG:
			return ( "ImageWidth, ImageLength or StripOffsets tag missing" );
		case GBM_ERR_TIF_SPP_BIT:
			return ( "SamplesPerPixel tag must be 1 for bitmap or greyscale file" );
		case GBM_ERR_TIF_BPS_BIT:
			return ( "BitsPerSample tag must be 1,4 or 8 for bitmap or greyscale file" );
		case GBM_ERR_TIF_SPP_RGB:
			return ( "SamplesPerPixel tag must be 3 or more for RGB file" );
		case GBM_ERR_TIF_BPS_RGB:
			return ( "BitsPerSample tag must be 8 for RGB file" );
		case GBM_ERR_TIF_SPP_PAL:
			return ( "SamplesPerPixel tag must be 1 for palettised file" );
		case GBM_ERR_TIF_BPS_PAL:
			return ( "BitsPerSample tag must be 1,4 or 8 for paletteised file" );
		case GBM_ERR_TIF_SPP_CMYK:
			return ( "SamplesPerPixel tag must be 4 for CMYK file" );
		case GBM_ERR_TIF_BPS_CMYK:
			return ( "BitsPerSample tag must be 8 for CMYK file" );
		case GBM_ERR_TIF_COMP_1D_MH:
			return ( "Compression tag is CCITT 1D Modified Huffman, not supported" );
		case GBM_ERR_TIF_COMP_T4:
			return ( "Compression tag is CCITT T.4 G3 Facsimile, not supported");
		case GBM_ERR_TIF_COMP_T6:
			return ( "Compression tag is CCITT T.6 G4 Facsimile, not supported");
		case GBM_ERR_TIF_COMP:
			return ( "Compression tag not uncompressed, PackBits or LZW, not supported" );
		case GBM_ERR_TIF_COLORMAP:
			return ( "ColorMap tag missing" );
		case GBM_ERR_TIF_CORRUPT:
			return ( "encoded data is corrupt" );
		case GBM_ERR_TIF_PREDICTOR:
			return ( "Predictor tag bad" );
		case GBM_ERR_TIF_PHOTO_TRANS:
			return ( "PhotometricInterpretation tag is transparency mask, not supported" );
		case GBM_ERR_TIF_PHOTO_Y_Cb_Cr:
			return ( "PhotometricInterpreation tag is Y-Cb-Cr colour space, not supported" );
		case GBM_ERR_TIF_PHOTO:
			return ( "PhotometricInterpretation tag unsupported/bad" );
		case GBM_ERR_TIF_FILLORDER:
			return ( "FillOrder tag must be 1" );
		case GBM_ERR_TIF_PLANARCONFIG_1:
			return ( "PlanarConfiguration tag must be 1 for non RGB files" );
		case GBM_ERR_TIF_PLANARCONFIG_12:
			return ( "PlanarConfiguration tag must be 1 or 2 for RGB files" );
		case GBM_ERR_TIF_INKSET:
			return ( "InkSet tag indicates non-CMYK colour seperations" );
		case GBM_ERR_TIF_ORIENT:
			return ( "Orientation tag must be 1 or 4" );
		case GBM_ERR_TIF_INDEX:
			return ( "less bitmaps in file than index requested" );
		}
	return ( NULL );
	}
/*...e*/















// ----------------------------------------------------------------------














/*

GBMTIFH.C  Routines to handle TIFF file headers

*/

/*...sincludes:0:*/
/*#include <stdlib.h>*/
#include <string.h>
#include "standard.h"
#define	_GBMTIFH_
#include "tifh.h"
//#include "gbm.h"    /* ???... */

/*...vgbmtifh\46\h:0:*/
/*...e*/

/*...susefull:0:*/
#define ifd_malloc()    ((IFD *) u_malloc(sizeof(IFD)))
#define ifd_free(ifd)   u_free((char *) ifd)
#define ifh_malloc()    ((IFH *) u_malloc(sizeof(IFH)))
#define ifh_free(ifh)   u_free((char *) ifh)

static int sizeof_data_type(short data_type)
	{
	switch ( data_type )
		{
		case D_BYTE:
		case D_SBYTE:
		case D_ASCII:
		case D_UNDEFINED:
			return ( 1 );
		case D_SHORT:
		case D_SSHORT:
			return ( sizeof(short) );
		case D_LONG:
		case D_SLONG:
			return ( sizeof(long) );
		case D_RATIONAL:
		case D_SRATIONAL:
			return ( sizeof(rational) );
		case D_FLOAT:
			return ( 4 );
		case D_DOUBLE:
			return ( 8 );
		}
	return ( 1 );
	}

static void tag_free(TAG *tag)
	{
    u_free(tag -> value);
	}

/*
This finds the slot for the new tag. It returns NULL if the limit of
MAX_TAGS tags is curruntly defined.
*/

static TAG *get_tag_slot(short type, IFD *ifd)
	{
	int	i;

	if ( ifd -> n_tags == MAX_TAGS )
		return ( NULL );

	for ( i = ifd -> n_tags;
	      i > 0 && ifd -> tags [i - 1].type >= type;
	      i-- )
		memcpy(ifd -> tags + i,
		       ifd -> tags + i - 1,
		       sizeof(TAG));

	ifd -> n_tags++;

	/* now i == slot, with greater than elements moved up */

	return ( &(ifd -> tags [i]) );
	}
/*...e*/

/*...sread_ifh_and_ifd:0:*/
/*...sread_long:0:*/

static long read_long(int fd, BOOLEAN motorola)
	{
	byte	b [4];

    u_read(fd, b, 4);
	return ( ( motorola ) ?
		make_long(b [0], b [1], b [2], b [3]) :
		make_long(b [3], b [2], b [1], b [0]) );
	}
/*...e*/


/*...sread_short:0:*/
static short read_short(int fd, BOOLEAN motorola)
	{
	byte	b [2];

    u_read(fd, b, 2);
	return ( ( motorola ) ?
		make_short(b [0], b [1]) :
		make_short(b [1], b [0]) );
	}
/*...e*/
/*...sread_rational:0:*/
static void read_rational(int fd, BOOLEAN motorola, rational *r)
	{
	r -> numerator   = read_long(fd, motorola);
	r -> denominator = read_long(fd, motorola);
	}
/*...e*/
/*...sread_tag:0:*/
static int read_tag(int fd, BOOLEAN motorola, TAG *tag)
	{
	int	i, s, n;
	long	len;
    long    seek_to, old_pos=0;

	tag -> type      = read_short(fd, motorola);
	tag -> data_type = read_short(fd, motorola);
	tag -> length    = read_long(fd, motorola);

	if ( tag -> type & 0x8000 )
		/* proprietry tag */
		{
        u_lseek(fd, 4L, SEEK_CUR);    /* skip data */
		return ( TE_OK );		/* assumed ok */
		}

	n   = (int) tag -> length;

/*...sbugfix for UBU\39\s writing of ColorMap tag:8:*/
/* UBU writes out a length feild of 256 when it should write 768 */

if ( tag -> type == T_COLORMAP && (n / 3) * 3 != n )
	n *= 3;
/*...e*/

	s   = sizeof_data_type(tag -> data_type);
	len = s * n;

	if ( len > 4 )
		/* will have to seek for data */
		{
		seek_to = read_long(fd, motorola);
        old_pos = u_lseek(fd, 0L, SEEK_CUR);
        u_lseek(fd, seek_to, SEEK_SET);
		}

    if ( (tag -> value = u_malloc((int) len)) == NULL )
		return ( TE_MEM );

	switch ( tag -> data_type )
		{
		case D_BYTE:
		case D_SBYTE:
            u_read(fd, tag -> value, n);
			break;
		case D_ASCII:
            u_read(fd, tag -> value, n);
			break;
		case D_SHORT:
		case D_SSHORT:
			{
			short	*short_ptr = (short *) tag -> value;

			for ( i = 0; i < n; i++ )
				*short_ptr++ = read_short(fd, motorola);
			}
			break;
		case D_LONG:
		case D_SLONG:
			{
			long	*long_ptr = (long *) tag -> value;

			for ( i = 0; i < n; i++ )
				*long_ptr++ = read_long(fd, motorola);
			}
			break;
		case D_RATIONAL:
		case D_SRATIONAL:
			{
			rational *rational_ptr = (rational *) tag -> value;

			for ( i = 0; i < n; i++ )
				read_rational(fd, motorola, rational_ptr++);
			}
			break;
		case D_FLOAT:
			/* Skip 4 byte IEEE floating point */
            u_lseek(fd, 4 * len, SEEK_CUR);
			break;
		case D_DOUBLE:
			/* Skip 8 byte IEEE double precision floating point */
            u_lseek(fd, 8 * len, SEEK_CUR);
			break;
		default:
            u_read(fd, tag -> value, (int) len);
			break;
		}

	if ( len > 4 )
        u_lseek(fd, old_pos, SEEK_SET);
	else if ( len < 4 )
        u_lseek(fd, 4L - len, SEEK_CUR);  /* advance past gap */

	return ( TE_OK );
	}
/*...e*/
/*...sread_ifd:0:*/
/*
For the time being we will assume there is only one IFD in
a given TIFF file. When this code was written, the author
knew of no software packages that support multiple IFDs.
*/

/*...sclean_up_ifd:0:*/
static void clean_up_ifd(IFD *ifd, int n)
	{
	int	i;
	TAG	*tag;

	for ( i = 0; i < n; i++ )
		{
		tag = &(ifd -> tags [i]);
		if ( !(tag -> type & 0x8000) )	/* its not read in */
			tag_free(tag);
		}
	ifd_free(ifd);
	}
/*...e*/

static int read_ifd(int fd, BOOLEAN motorola, IFD **ifd_return)
	{
	IFD	*ifd;
	int	i, ecode;

    if ( (ifd = ifd_malloc()) == NULL )
		return ( TE_MEM );

	/* ensure we can handle all the tags */

	if ( (ifd -> n_tags = read_short(fd, motorola)) > MAX_TAGS )
		{
		ifd_free(ifd); return ( TE_N_TAGS );
		}

	/* get the tags */

	for ( i = 0; i < ifd -> n_tags; i++ )
		if ( (ecode = read_tag(fd, motorola, &(ifd -> tags [i]))) != TE_OK )
			{
			clean_up_ifd(ifd, i);
			return ( ecode );
			}

	*ifd_return = ifd;

	return ( TE_OK );
	}
/*...e*/
/*...sskip_ifd:0:*/
/* Returns TRUE if there is another IFD afterwards */

static BOOLEAN skip_ifd(int fd, BOOLEAN motorola)
	{
	short n_tags = read_short(fd, motorola);
	long offset_ifd;
    u_lseek(fd, 12L * n_tags, SEEK_CUR);
	offset_ifd = read_long(fd, motorola);
	if ( offset_ifd == 0L )
		return ( FALSE );
    u_lseek(fd, offset_ifd, SEEK_SET);
	return ( TRUE );
	}
/*...e*/

int read_ifh_and_ifd(int fd, int n_ifds_to_skip, IFH **ifh_return)
	{
	IFH	*ifh;
	long	offset_ifd;
	BOOLEAN	motorola;
	int	ecode;

	if ( (ifh = ifh_malloc()) == NULL )
		return ( TE_MEM );

    u_read(fd, (char *) &(ifh -> byte_order), sizeof(short));
	motorola = ( ifh -> byte_order == ('M' << 8) + 'M' );

	/* Apparently, the following number has great univeral significance! */
	/* See the TIFF 5.0 spec. for details! */

	if ( (ifh -> version_no = read_short(fd, motorola)) != 42 )
		{
		ifh_free(ifh); return ( TE_VERSION );
		}

	offset_ifd = read_long(fd, motorola);
    u_lseek(fd, offset_ifd, SEEK_SET);
	while ( n_ifds_to_skip-- > 0 )
		if ( !skip_ifd(fd, motorola) )
			return ( TE_N_IFD );

	if ( (ecode = read_ifd(fd, motorola, &(ifh -> ifd))) != TE_OK )
		{
		ifh_free(ifh); return ( ecode );
		}

	*ifh_return = ifh;

	return ( TE_OK );
	}
/*...e*/
/*...slocate_tag:0:*/
TAG	*locate_tag(IFD *ifd, short type)
	{
	int	i;

	for ( i = 0; i < ifd -> n_tags; i++ )
		if ( ifd -> tags [i].type == type )
			return ( &(ifd -> tags [i]) );
	return ( NULL );
	}
/*...e*/
/*...snumeric_tag:0:*/
BOOLEAN	numeric_tag(TAG *tag)
	{
	short t = tag -> data_type;
	return ( t == D_BYTE  ||
		 t == D_SHORT || t == D_SSHORT ||
		 t == D_LONG  || t == D_SLONG  );
	}
/*...e*/
/*...svalue_of_tag_n:0:*/
/*
For a numeric tag, return the value of the nth item in it.
Upto the caller to know that tag is signed or unsigned.
*/

long	value_of_tag_n(TAG *tag, int n)
	{
	switch ( tag -> data_type )
		{
		case D_BYTE:
			{
			unsigned char *p = (unsigned char *) tag -> value;
			return ( (long) (unsigned long) p [n] );
			}
		case D_SBYTE:
			{
			signed char *p = (signed char *) tag -> value;
			return ( (long) p [n] );
			}
		case D_SHORT:
			{
			unsigned short *p = (unsigned short *) tag -> value;
			return ( (long) (unsigned long) p [n] );
			}
		case D_SSHORT:
			{
			signed short *p = (signed short *) tag -> value;
			return ( (long) p [n] );
			}
		case D_LONG:
			{
			unsigned long *p = (unsigned long *) tag -> value;
			return ( (long) p [n] );
			}
		case D_SLONG:
			{
			signed long *p = (signed long *) tag -> value;
			return ( (long) p [n] );
			}
		}
	return ( 0L );
	}
/*...e*/
/*...svalue_of_tag:0:*/
/*
For a numeric tag, return the value of the 1st value in it.
This is usefull for tags that typically only have 1 value anyway.
*/

long	value_of_tag(TAG *tag)
	{
	return ( value_of_tag_n(tag, 0) );
	}
/*...e*/
/*...sfree_ifh:0:*/
void	free_ifh(IFH *ifh)
	{
	IFD	*ifd;

	ifd = ifh -> ifd;
	clean_up_ifd(ifd, ifd -> n_tags);
	ifh_free(ifh);
	}
/*...e*/
/*...smake_ifh:0:*/
/*
Creates an empty IFH set up for the image.
Also creates an IFD as part of the IFH.
Use add_?_tag() routines to add tags to IFH's IFD.
*/

IFH	*make_ifh(void)
	{
	IFH	*ifh;
	IFD	*ifd;

	if ( (ifh = ifh_malloc()) == NULL )
		return ( NULL );

	if ( (ifh -> ifd = ifd = ifd_malloc()) == NULL )
		{
		ifh_free(ifh);
		return ( NULL );
		}

	ifh -> byte_order = ('I' << 8) + 'I';
	ifh -> version_no = 42;

	ifd -> n_tags = 0;

	return ( ifh );
	}
/*...e*/
/*...sadd_byte_tag:0:*/
BOOLEAN	add_byte_tag(IFD *ifd, short type, byte *value, int n)
	{
	byte	*byte_ptr;
	TAG	*tag;

    if ( (byte_ptr = (byte *) u_malloc(n * sizeof(byte))) == NULL )
		return ( FALSE );
	if ( (tag = get_tag_slot(type, ifd)) == NULL )
		return ( FALSE );
	tag -> type      = type;
	tag -> data_type = D_BYTE;
	tag -> length    = (long) n;
	if ( value != NULL )
		memcpy(tag -> value = (char *) byte_ptr,
		       value,
		       n * sizeof(byte));
	return ( TRUE );
	}
/*...e*/
/*...sadd_ascii_tag:0:*/
BOOLEAN	add_ascii_tag(IFD *ifd, short type, char *value)
	{
	char	*ascii_ptr;
	TAG	*tag;
	int	n;

	n = strlen(value) + 1;
    if ( (ascii_ptr = (char *) u_malloc(n)) == NULL )
		return ( FALSE );
	if ( (tag = get_tag_slot(type, ifd)) == NULL )
		return ( FALSE );
	tag -> type      = type;
	tag -> data_type = D_ASCII;
	tag -> length    = (long) n;
	strcpy(tag -> value = ascii_ptr, value);
	return ( TRUE );
	}
/*...e*/
/*...sadd_short_tag:0:*/
BOOLEAN	add_short_tag(IFD *ifd, short type, short *value, int n)
	{
	short	*short_ptr;
	TAG	*tag;

    if ( (short_ptr = (short *) u_malloc(n * sizeof(short))) == NULL )
		return ( FALSE );
	if ( (tag = get_tag_slot(type, ifd)) == NULL )
		return ( FALSE );
	tag -> type      = type;
	tag -> data_type = D_SHORT;
	tag -> length    = (long) n;
	if ( value != NULL )
		memcpy(tag -> value = (char *) short_ptr,
		       value,
		       n * sizeof(short));
	return ( TRUE );
	}
/*...e*/
/*...sadd_long_tag:0:*/
BOOLEAN	add_long_tag(IFD *ifd, short type, long *value, int n)
	{
	long	*long_ptr;
	TAG	*tag;

    if ( (long_ptr = (long *) u_malloc(n * sizeof(long))) == NULL )
		return ( FALSE );
	if ( (tag = get_tag_slot(type, ifd)) == NULL )
		return ( FALSE );
	tag -> type      = type;
	tag -> data_type = D_LONG;
	tag -> length    = (long) n;
	if ( value != NULL )
		memcpy(tag -> value = (char *) long_ptr,
		       value,
		       n * sizeof(long));
	return ( TRUE );
	}
/*...e*/
/*...sadd_rational_tag:0:*/
BOOLEAN	add_rational_tag(IFD *ifd, short type, rational *value, int n)
	{
	rational *rational_ptr;
	TAG	*tag;

    if ( (rational_ptr = (rational *) u_malloc(n * sizeof(rational))) == NULL )
		return ( FALSE );
	if ( (tag = get_tag_slot(type, ifd)) == NULL )
		return ( FALSE );
	tag -> type      = type;
	tag -> data_type = D_RATIONAL;
	tag -> length    = (long) n;
	if ( value != NULL )
		memcpy(tag -> value = (char *) rational_ptr,
		       value,
		       n * sizeof(rational));
	return ( TRUE );
	}
/*...e*/
/*...swrite_ifh_and_ifd:0:*/
/*...spad:0:*/
static BOOLEAN pad(int fd, int n)
	{
	static char padding [] = { 0, 0, 0, 0 };

    return ( u_write(fd, padding, n) == n );
	}
/*...e*/
/*...swrite_short:0:*/
static BOOLEAN write_short(int fd, short s)
	{
	byte	b [2];

	b [0] = (byte)  (s & 0x00ff);
	b [1] = (byte) ((s & 0xff00) >> 8);

    return ( u_write(fd, b, 2) == 2 );
	}
/*...e*/
/*...swrite_long:0:*/
static BOOLEAN write_long(int fd, long l)
	{
	byte	b [4];

	b [0] = (byte)  (l & 0x000000ffL);
	b [1] = (byte) ((l & 0x0000ff00L) >>  8);
	b [2] = (byte) ((l & 0x00ff0000L) >> 16);
	b [3] = (byte) ((l & 0xff000000L) >> 24);

    return ( u_write(fd, b, 4) == 4 );
	}
/*...e*/
/*...swrite_rational:0:*/
static BOOLEAN write_rational(int fd, rational *rational)
	{
	return ( write_long(fd, rational -> numerator  ) &&
		 write_long(fd, rational -> denominator) );
	}
/*...e*/
/*...swrite_tag:0:*/
static BOOLEAN write_tag(int fd, TAG *tag, long *offset_upto)
	{
	BOOLEAN	ok;
	int	s, i, n, len;
    long    offset_return_to=0;

	ok = write_short(fd, tag -> type) &&
	     write_short(fd, tag -> data_type) &&
	     write_long(fd, tag -> length);

	if ( !ok )
		return ( FALSE );

	/* if we can fit the tag into 4 bytes, do so */
	/* else we will have to allocate some disc space */

	s = sizeof_data_type(tag -> data_type);
	n = (int) tag -> length;
	len = s * n;

	if ( len > 4 )
		{
		if ( !write_long(fd, *offset_upto) )
			return ( FALSE );
        offset_return_to = u_lseek(fd, 0L, SEEK_CUR);
        u_lseek(fd, *offset_upto, SEEK_SET);
		}		

	/* actually write the tag */

	switch ( tag -> data_type )
		{
		case D_BYTE:
		case D_ASCII:
            if  ( u_write(fd, tag -> value, n) != n )
				return ( FALSE );
			break;
		case D_SHORT:
			{
			short	*short_ptr = (short *) tag -> value;

			for ( i = 0; i < n; i++ )
				if ( !write_short(fd, *short_ptr++) )
					return ( FALSE );
			}
			break;
		case D_LONG:
			{
			long	*long_ptr = (long *) tag -> value;

			for ( i = 0; i < n; i++ )
				if ( !write_long(fd, *long_ptr++) )
					return ( FALSE );
			}
			break;
		case D_RATIONAL:
			{
			rational *rational_ptr = (rational *) tag -> value;

			for ( i = 0; i < n; i++ )
				if ( !write_rational(fd, rational_ptr++) )
					return ( FALSE );
			}
			break;
		}

	if ( len > 4 )
		{
        if ( (*offset_upto = u_lseek(fd, 0L, SEEK_CUR)) & 1L )
			/* pad to make next offset even */
			{
			if ( !pad(fd, 1) )
				return ( FALSE );
			(*offset_upto)++;
			}
        u_lseek(fd, offset_return_to, SEEK_SET);
		}
	else if ( len < 4 )
		if ( !pad(fd, 4 - len) )
			return ( FALSE );
	return ( TRUE );
	}
/*...e*/
/*...swrite_ifd:0:*/
/*
Given an IFD, write it out to disc.
Also patch the IFH (which we know will be at the start of the file).
In writing out a tag we may need some more disc space other than
that for the IFD table. This occurs when a field is larger than
4 bytes. What we do is to keep a pointer to the next free space
(after the table) and write_tag() will advance it if it uses any
extra space.
*/

BOOLEAN write_ifd(int fd, IFD *ifd)
	{
	int	i, n;
	long	offset_upto;

	if ( !write_short(fd, n = ifd -> n_tags) )
		return ( FALSE );

	/* write out tags */

    offset_upto = u_lseek(fd, 0L, SEEK_CUR) + n * 12L + 4L;
		/* leave space for each tag plus next IFD ptr */

	for ( i = 0; i < n; i++ )
		if ( !write_tag(fd, &(ifd -> tags [i]), &offset_upto) )
			return ( FALSE );

	/* done writing out the IFD, now put null next IFD pointer */

	if ( !write_long(fd, 0L) )
		return ( FALSE );
	
    u_lseek(fd, offset_upto, SEEK_SET);

	return ( TRUE );
	}
/*...e*/

BOOLEAN write_ifh_and_ifd(IFH *ifh, int fd)
	{
	return ( write_short(fd, ifh -> byte_order) &&
		 write_short(fd, ifh -> version_no) &&
		 write_long(fd, 8L) &&
		 write_ifd(fd, ifh -> ifd) );
	}
/*...e*/
/*...supdate_byte_tag:0:*/
void	update_byte_tag(IFD *ifd, short type, byte *value)
	{
	TAG	*tag;
	int	n;

	tag = locate_tag(ifd, type);
	n = (int) tag -> length;
	memcpy(tag -> value, value, n * sizeof(byte));
	}
/*...e*/
/*...supdate_ascii_tag:0:*/
void	update_ascii_tag(IFD *ifd, short type, char *value)
	{
	TAG	*tag;
	int	n;

	tag = locate_tag(ifd, type);
	n = (int) tag -> length;
	memcpy(tag -> value, value, n);
	}
/*...e*/
/*...supdate_short_tag:0:*/
void	update_short_tag(IFD *ifd, short type, short *value)
	{
	TAG	*tag;
	int	n;

	tag = locate_tag(ifd, type);
	n = (int) tag -> length;
	memcpy(tag -> value, value, n * sizeof(short));
	}
/*...e*/
/*...supdate_long_tag:0:*/
void	update_long_tag(IFD *ifd, short type, long *value)
	{
	TAG	*tag;
	int	n;

	tag = locate_tag(ifd, type);
	n = (int) tag -> length;
	memcpy(tag -> value, value, n * sizeof(long));
	}
/*...e*/
/*...supdate_rational_tag:0:*/
void	update_rational_tag(IFD *ifd, short type, rational *value)
	{
	TAG	*tag;
	int	n;

	tag = locate_tag(ifd, type);
	n = (int) tag -> length;
	memcpy(tag -> value, value, n * sizeof(rational));
	}
/*...e*/
/*...supdate_ifd:0:*/
/*
Go back to the IFD, and rewrite it.
*/

BOOLEAN update_ifd(IFD *ifd, int fd)
	{
    u_lseek(fd, 8L, SEEK_SET);
	return ( write_ifd(fd, ifd) );
	}
/*...e*/
/*...serror_string:0:*/
static char *tiff_errlist [] =
	{
	NULL,
	"out of memory",
	"unsupported TIFF file version",
	"too many tags in TIFF file",
	"bad tag data type",
	};

char	*error_string(int rc)
	{
	return ( tiff_errlist [rc] );
	}
/*...e*/


