/*
        Calculate the rectangle between two overlapping rectangles
        returns TRUE on overlapping.
        returns FALSE if rectangles don't overlap.


        The structure VRECT is used to transfer data.

        The rectangle1 (input) and
            rectangle2 (input) is compared, and data is output on
            rectangle3 (output);



            x = screen coordinate x;
            y = screen coordinate x;
            sx = size x;
            sy = size y;

        You should call the function like this:

        retval = rect_intersection( (VRECT *)&u1, (VRECT *)&u2, (VRECT *)&u3 );


        Where retval is the return value of type int and usr1 and usr2 are
        two pointers to two structures where rectangle dimensions are defined.
        data is output on usr3.


        If the return value is FALSE then the values in the output members
        are invalid, and not changed.

    return value:
        The returned values are the intersection rentangle.


    Coordinates:
        Coordinates are relative to top, left.
        X increases towards right.
        Y increases towards bottom.
        Horizontal Lines of type 1 and 3 are drawn from left to right.
        Vertical lines of type 2 and 4 are draw from top to bottom.

        example code:

        #include "rect.h"
        #include <allegro.h>
        void main()
        {
             int   ret;
             VRECT u1, u2, out;
             i1.x = 10;      i2.x = 15;
             i1.y = 10;      i2.y = 15;
             i1.sx = 10;     i2.sx = 10;
             i1.sy = 10;     i2.sy = 10;
             ret = rect_coords_overlap( &i1, &i2, &out );
             if ( ret==TRUE ){
                // intersection valid, data is returned in  out
                }
             return;
        }


           x1 y1                x2 y1
             #--------------------#
             |p1     line1      p2|
             |                    |
             |                    |
       line4 |                    | line2
             |                    |
             |                    |
             |p4     line3      p3|
             #--------------------#
           x1 y2                x2 y2



*/
#include <allegro.h>
#include "rect.h"
#define ONE_ 1
#define ZERO 0



/**** prototypes ****/
static int check_inside( int p, int x1, int x2 );
/**** prototypes ****/





/**************************************************************
* check to see if pixels of rectangle u2 are inside u1.       *
*                                                             *
*    bug: does not detect crossings                           *
* return: FALSE - no overlapping                              *
*         TRUE - overlapping has occured                      *
*                                                             *
*     pixs:                                                   *
*         0 - no overlapping.                                 *
*         1 - 1 pixel  of u2 is  inside u1                    *
*         2 - 2 pixels of u2 are inside u1                    *
*         4 - 4 pixels of u2 are inside u1 (type is invalid)  *
*                                                             *
*      type: 1 to 4 to indicate which corner is inside.       *
*            if pix value is 4 then this value has no meaning.*
*            0 = invalid                                      *
*            1 = corner 1                                     *
*            2 = corner 2                                     *
*            3 = corner 3                                     *
*            4 = corner 4                                     *
*                                                             *
*    corners, 1 pixel is inside, type is:                     *
*                                                             *
*         1------      ------2    ------     ------           *
*         |      |    |      |   |      |   |      |          *
*         |      |    |      |   |      |   |      |          *
*          ------      ------     ------3   4------           *
*                                                             *
*    corners, 2 pixels are inside, type is:                   *
*                                                             *
*         1------1     ------2    ------    4------           *
*         |      |    |      |   |      |   |      |          *
*         |      |    |      |   |      |   |      |          *
*          ------      ------2   3------3   4------           *
**************************************************************/















// return 1 if point p is inside rectangle  of coordinates  x1,y1,x2,y2
// return 0 if point p is outside rectangle of coordinates  x1,y1,x2,y2
static int check_inside( int p, int x1, int x2 )
{
    if( p >= x1 && p <= x2 ) return 1;
    return 0;
}




/* ********************************************************
* return the intersection between two rectangles          *
*  returns the intersection rectangle if one rectangle    *
*   overlap the other rectangle.                          *
*  return:                                                *
*          FALSE, no overlapping has occured              *
*          TRUE , values are returned in out              *
*    Line types from 1 to 4:                                  *
*                                                             *
*          111111      ------2    ------    4------           *
*         |      |    |      2   |      |   4      |          *
*         |      |    |      2   |      |   4      |          *
*          ------      ------2   33333333   4------           *

line type 1:
        ---------->
line type 2:
            |
            |
           <|>
line type 3:
        ---------->
line type 4:
            |
            |
           <|>

********************************************************* */
// u1 = u_     u2 = v_
int rect_intersection( VRECT *u_, VRECT *v_, VRECT *out )
{
    int     x1=0,y1=0,x2=0,y2=0;
    int     u_x1, u_y1, u_x2, u_y2;
    int     v_x1, v_y1, v_x2, v_y2;
    int     p1inside=0,    p2inside=0,    p3inside=0,    p4inside=0;
    int     psum;

    if( u_==NULL || v_==NULL || out==NULL ) return FALSE;
    // width or heigth of 0 or less is invalid, oh yes
    if( u_->sx <=0 || u_->sy <=0 || v_->sx <=0 || v_->sy <=0 ) return FALSE;

    out->line1_valid = 0;
    out->line2_valid = 0;
    out->line3_valid = 0;
    out->line4_valid = 0;

    // form point coordinates for each rectangle
    u_x1 = u_->x;
    u_y1 = u_->y;
    u_x2 = u_->x + u_->sx+1;
    u_y2 = u_->y + u_->sy+1;
    v_x1 = v_->x;
    v_y1 = v_->y;
    v_x2 = v_->x + v_->sx+1;
    v_y2 = v_->y + v_->sy+1;

    x1 = check_inside( u_x1,   v_x1, v_x2 );
    y1 = check_inside( u_y1,   v_y1, v_y2 );
    x2 = check_inside( u_x2,   v_x1, v_x2 );
    y2 = check_inside( u_y2,   v_y1, v_y2 );
    if( x1==1 && y1==1 ) p1inside = 1;
    if( x2==1 && y1==1 ) p2inside = 1;
    if( x2==1 && y2==1 ) p3inside = 1;
    if( x1==1 && y2==1 ) p4inside = 1;

    out->p1inside = p1inside;
    out->p2inside = p2inside;
    out->p3inside = p3inside;
    out->p4inside = p4inside;
    out->psum     = psum = (p1inside + p2inside + p3inside + p4inside);


    out->x  = u_->x;
    out->y  = u_->y;
    out->sx = u_->sx;
    out->sy = u_->sy;

    out->line1_x1 = u_x1;   // horizontal, top
    out->line1_y1 = u_y1;
    out->line1_x2 = u_x2;
    out->line1_y2 = u_y1;

    out->line2_x1 = u_x2;   // vertical, right
    out->line2_y1 = u_y1;
    out->line2_x2 = u_x2;
    out->line2_y2 = u_y2;

    out->line3_x1 = u_x1;   // horizontal, bottom
    out->line3_y1 = u_y2;
    out->line3_x2 = u_x2;
    out->line3_y2 = u_y2;

    out->line4_x1 = u_x1;   // vertical, left
    out->line4_y1 = u_y1;
    out->line4_x2 = u_x1;
    out->line4_y2 = u_y2;




    // now determine cases

    if( psum == 4 ){   // all points inside
        out->line1_valid = 1;
        out->line2_valid = 1;
        out->line3_valid = 1;
        out->line4_valid = 1;
    return TRUE;
    }


    if( psum == 2 ){    // two points inside
        if( p3inside == 1 && p4inside == 1 ){
            out->sy          = u_y2 - v_y1;
            out->line2_valid = 1;
            out->line3_valid = 1;
            out->line4_valid = 1;
            out->line2_y1 = v_y1;
            out->line4_y1 = v_y1;
        }
        if( p1inside == 1 && p4inside == 1 ){
            out->sx          = v_x2 - u_x1;
            out->line1_valid = 1;
            out->line3_valid = 1;
            out->line4_valid = 1;
            out->line1_x2 = v_x2;
            out->line3_x2 = v_x2;
        }
        if( p1inside == 1 && p2inside == 1 ){
            out->sy          = v_y2 - v_y1;
            out->line1_valid = 1;
            out->line2_valid = 1;
            out->line4_valid = 1;
            out->line2_y2 = v_y2;
            out->line4_y2 = v_y2;
        }
        if( p2inside == 1 && p3inside == 1 ){
            out->x  = v_x1;
            out->sx = u_x2 - v_x1;
            out->line1_valid = 1;
            out->line2_valid = 1;
            out->line3_valid = 1;
            out->line1_x1 = v_x1;
            out->line3_x1 = v_x1;
        }
    return TRUE;
    }


    if( psum == 1 ){    // one edge of rectangle1 is inside rectangle2
        if( p1inside == 1){
            out->sx = v_x2 - u_x1;
            out->sy = v_y2 - u_y1;
            out->line1_valid = 1;
            out->line4_valid = 1;
            out->line1_x2 = v_x2;
            out->line4_y2 = v_y2;
      }
        if( p2inside == 1){
            out->x  = v_x1;
            out->sx = u_x2 - v_x1;
            out->sy = v_y2 - u_y1;
            out->line1_valid = 1;
            out->line2_valid = 1;
            out->line1_x1 = v_x1;
            out->line2_y2 = v_y2;
        }
        if( p3inside == 1){
            out->x  = v_x1;
            out->y  = v_y1;
            out->sx = u_x2 - v_x1+1;
            out->sy = u_y2 - v_y1+1;
            out->line2_valid = 1;
            out->line3_valid = 1;
            out->line2_y1 = v_y1;
            out->line3_x1 = v_x1;
        }
        if( p4inside == 1){
            out->y  = v_y1;
            out->sx = v_x2 - u_x1;
            out->sy = u_y2 - v_y1;
            out->line3_valid = 1;
            out->line4_valid = 1;
            out->line3_x2 = v_x2;
            out->line4_y1 = v_y1;
        }
    return TRUE;
    }



    if( psum == 0 ){   // all points outside

        // check for horizontal overlap
        //...
        if( u_x1 < v_x1 && u_x2 > v_x2 ){
            if( u_y1 < v_y1  && u_y2 <= v_y2 && u_y2 >= v_y1 ){  // top
            out->x  = v_x1;
            out->y  = v_y1;
            out->sx = v_x2 - v_x1;
            out->sy = u_y2 - v_y1;
            out->line3_valid = 1;
            out->line3_x1 = v_x1;
            out->line3_x2 = v_x2;
            }
            if( u_y1 >= v_y1 && u_y2 <= v_y2 ){                 // middle
            out->x  = v_x1;
            out->sx = v_x2 - v_x1;
            out->line1_valid = 1;
            out->line3_valid = 1;
            out->line1_x1 = v_x1;
            out->line1_x2 = v_x2;
            out->line3_x1 = v_x1;
            out->line3_x2 = v_x2;
            }
            if( u_y1 >= v_y1 && u_y1 <= v_y2 && u_y2 > v_y2 ){  // bottom
            out->x  = v_x1;
            out->sx = v_x2 - v_x1;
            out->sy = v_y2 - u_y1;
            out->line1_valid = 1;
            out->line1_x1 = v_x1;
            out->line1_x2 = v_x2;
            }

        }


        // check for vertical   overlap
        //...
        if( u_y1 < v_y1 && u_y2 > v_y2 ){
            if( u_x1 < v_x1  && u_x2 <= v_x2 && u_x2 >= v_x1 ){  // left
            out->x  = v_x1;
            out->y  = v_y1;
            out->sx = u_x2 - v_x1;
            out->sy = v_y2 - v_y1;
            out->line2_valid = 1;
            out->line2_y1 = v_y1;
            out->line2_y2 = v_y2;
            }
            if( u_x1 >= v_x1 && u_x2 <= v_x2 ){                 // middle
            out->y  = v_y1;
            out->sy = v_y2 - v_y1;
            out->line2_valid = 1;
            out->line4_valid = 1;
            out->line2_y1 = v_y1;
            out->line2_y2 = v_y2;
            out->line4_y1 = v_y1;
            out->line4_y2 = v_y2;
            }
            if( u_x1 >= v_x1 && u_x1 <= v_x2 && u_x2 > v_x2 ){  // right
            out->y  = v_y1;
            out->sx = v_x2 - u_x1;
            out->sy = v_y2 - v_y1;
            out->line4_valid = 1;
            out->line4_y1 = v_y1;
            out->line4_y2 = v_y2;
            }

        }

    return TRUE;
    }


    return FALSE;
}

