/*
 * Font renderer for GRX font files. (compatible with Xwindows).
 * Open fonts in memory or file.
 * For use with alleg.
 * users must include "fnt.h" in their files.
 * "plugin type"
 * Uses Allegro for graphic rendering.
 *
 *  FNT *fnt_load_mem(GRR *grf, int condens, int verbose_flag)
 *  FNT *fnt_load_file(char *filename, int condens, int verbose_flag)
 *  int fnt_text_length(FNT *f, unsigned char *str)
 *  int fnt_text_height(FNT *f)
 *  void fnt_textout(BITMAP *bmp, FNT *fnt, char *str, int _x, int _y,
 *                   int fg, int bg, int verbose_flag)
 *  void fnt_destroy_font( FNT *fnt )
 *
 *
 *
 *
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <allegro.h>
#include "fnt.h"
#include "ed.h"
#include "subs.h"
#include "dlg.h"
#include "data.h"
#include "butil.h"


/* helper function prototypes */
FNT *fnt_load_mem(GRR *,int,int);
FNT *fnt_load_file(char *,int,int);
int fnt_text_length(FNT *,unsigned char *);
int fnt_text_height(FNT *);
void fnt_textout(BITMAP *,FNT *,char *,int,int,int,int,int);
static int does_file_exist(char *);
static void print_char(BITMAP *,FNT *,int,int,int,int,int,int,int);
/* helper function prototypes */

/* function prototypes for the file wrapper functions */
static FILE               *ra_open( char *, char);         /* open          */
static void               ra_close( FILE * );              /* close         */
static unsigned char      ra_getc( FILE * );               /* get character */
static unsigned short int ra_getw( FILE * );               /* get word      */
static unsigned       int ra_getd( FILE * );               /* get dword     */
static int                ra_read( void *,int, FILE*);     /* read buffer   */
/* function prototypes for the file wrapper functions */






/*
 * load a font from a memory location.
 * The fonts are GRX (Xwindows) compatible.
 * The fonts loaded from memory are created with the 'fnt2c'
 * utility from the GRX package.
 * Only fixed fonts are supported.
 * untested on scalable fonts (may not work).
 *
 * condens = how many pixels to add between characters when writing text.
 *           use 0 for default.
 * verbose_flag = 1-verbose, 0-quiet.
 *
 * return NULL on error, else a pointer to a FNT structure.
 */
FNT *fnt_load_mem(GRR *grf, int condens, int verbose_flag)
{
    int i,j,l;
    FNT *f=NULL;

    if(grf==NULL){
    if(verbose_flag==1)
        puts ("*ERROR* no font structure given, '(null)'");
    return NULL;
    }
    if(verbose_flag==1)
        puts("\n***> entering fnt_load_from_mem()");
    f = (FNT *) malloc( sizeof(FNT) );    if( f==NULL ) return NULL;
    /***** INITIALIZATION *****/
    f->g.h.name         = grf->g.h.name        ;
    f->g.h.family       = grf->g.h.family      ;
    f->g.h.proportional = grf->g.h.proportional;
    f->g.h.scalable     = grf->g.h.scalable    ;
    f->g.h.preloaded    = grf->g.h.preloaded   ;
    f->g.h.modified     = grf->g.h.modified    ;
    f->g.h.width        = grf->g.h.width       ;
    f->g.h.height       = grf->g.h.height      ;
    f->g.h.baseline     = grf->g.h.baseline    ;
    f->g.h.ulpos        = grf->g.h.ulpos       ;
    f->g.h.ulheight     = grf->g.h.ulheight    ;
    f->g.h.minchar      = grf->g.h.minchar     ;
    f->g.h.numchars     = grf->g.h.numchars    ;
    f->g.bitmap    = grf->g.bitmap  ;
    f->g.auxmap    = grf->g.auxmap  ;
    f->g.minwidth  = grf->g.minwidth;
    f->g.maxwidth  = grf->g.maxwidth;
    f->g.auxsize   = grf->g.auxsize ;
    f->g.auxnext   = grf->g.auxnext ;
    f->filename = NULL;
    f->fp = NULL;
    f->fontsize = 0;
    i = f->g.h.width;
    j = f->g.h.height;
    l = i%8;
    if(l!=0) l=1;
    f->bytes_per_char = ((i/8)+l) * j;
    f->topchar = grf->g.h.minchar + grf->g.h.numchars;
    f->g.chrinfo[0].width  = grf->g.chrinfo[0].width;
    f->g.chrinfo[0].offset = grf->g.chrinfo[0].offset;
    for(i=0;i<f->g.h.numchars-1;i++){
    f->rest[i].width  = grf->rest[i].width;
    f->rest[i].offset = grf->rest[i].offset;
    }
    f->condens = condens;
    f->glyph = create_bitmap( f->g.maxwidth, f->g.h.height );
    if(f->glyph==NULL) return NULL;
    if(verbose_flag==1)
        puts("<*** leaving fnt_load_from_mem()");
    return (FNT *) f;
}






GrFontFileHeaderGRX *fnt_load_header( char *filename )
{
    int                 numchars=0, offset, i=0, j=0, headersize=0, prop;
    FILE                *fp = NULL;
    GrFontFileHeaderGRX *grxh = NULL;
    unsigned char       stringline[120], *st;



    if(filename==NULL) return NULL;
    if(does_file_exist (filename) == -1) return NULL;

    grxh = (GrFontFileHeaderGRX *) malloc( sizeof(GrFontFileHeaderGRX) );
    if( grxh == NULL ) goto failure;

    fp = ra_open (filename,0);  if(fp == NULL) goto failure;

//    fseek (fp, 0L, SEEK_END);
//    fontsize = ftell (fp);
//    fseek (fp, 0L, SEEK_SET);

    /**********************************/
    /**** READ AND VALIDATE HEADER ****/
    grxh->magic   = ra_getd(fp);    /* font file magic number */
    if( grxh->magic != GRX_FONTMAGIC ) goto failure;
    grxh->bmpsize = ra_getd(fp);  /* character bitmap size */
    grxh->width   = ra_getw(fp);  /* width (average for proportional) */
    grxh->height  = ra_getw(fp);  /* font height */
    grxh->minchar = ra_getw(fp);  /* lowest character code in font */
    grxh->maxchar = ra_getw(fp);  /* highest character code in font */
    grxh->isfixed = ra_getw(fp);  /* nonzero if fixed font */
    grxh->reserved= ra_getw(fp);  /* ??? */
    grxh->baseline= ra_getw(fp);  /* baseline from top of font */
    grxh->undwidth= ra_getw(fp);  /* underline width (at bottom) */
     /* font file name (w/o path) */
    i = ra_read((char *)&grxh->fnname,GRX_NAMEWIDTH,fp);    if(i!=GRX_NAMEWIDTH) goto failure;
    /* font family name */
    i = ra_read((char *)&grxh->family,GRX_NAMEWIDTH,fp);    if( i != GRX_NAMEWIDTH ) goto failure;
    grxh->manufacturer = NULL;
    grxh->familyname   = NULL;
    grxh->bold         = NULL;
    grxh->italic       = NULL;
    grxh->fontsize     = NULL;
    grxh->flag_bold    = 0;
    grxh->flag_italic  = 0;
    /**** READ AND VALIDATE HEADER ****/
    /**********************************/


    headersize  = ftell(fp);
    numchars    = grxh->maxchar - grxh->minchar;
    if(grxh->isfixed)           // fixed font
        prop = 0;
    else                        // proportional font
        prop = 2 + numchars*2;
    offset      = headersize  + prop + grxh->bmpsize;
    fseek (fp, (long) offset, SEEK_SET);



//    parse_FONT_line( (char *)&stringline, fp );
// *******************************************************************
// FONT -Adobe-Helvetica-Medium-R-Normal--18-180-75-75-P-98-ISO8859-1
    for(;;){
    grxh->infoline[0] = 0;
    stringline[0]=0;
    fgets( (char *)&stringline, sizeof(stringline)-1, fp );
    st = strrchr( (char *)&stringline, '\n'); // reverse search for enter character
    *st++=0;    // remove enter character from line
        if( stringline[0] != ' '){  // we got something else than copyright info
        if( stringline[0] =='F' &&  stringline[1] =='O' &&  stringline[2] =='N' &&  stringline[3] =='T' ){

        strncpy( (char *)&grxh->infoline, (char *)&stringline[6], sizeof(grxh->infoline)-1 );
//        // we got FONT data, scan it
//        sscanf( (char *)&stringline, "FONT -%s", (char *)&grxh->infoline );
        break;
        }}
    if(feof(fp)) break;
    }
// Adobe-Helvetica-Medium-R-Normal--18-180-75-75-P-98-ISO8859-1
// *******************************************************************

/*
    puts("---------------------------------");
    printf("%s\n", grxh->infoline );
*/

    split_adobe_string( (char *)&grxh->infoline, grxh );

    if( grxh->bold != NULL ){
        i = strcmp( grxh->bold, "Bold" );
        j = strcmp( grxh->bold, "DemiBold" );
        if( i==0 || j==0 ) grxh->flag_bold = 1;  // set bold flag
    }
    if( grxh->italic != NULL ){
        i = strcmp( grxh->italic, "I" );
        j = strcmp( grxh->italic, "O" );
        if( i==0 || j==0 ) grxh->flag_italic = 1;  // set italic flag
    }
/*
    printf("manufacturer <%s>\n", grxh->manufacturer );
    printf("familyname <%s>\n",   grxh->familyname );
    printf("bold <%s>\n",         grxh->bold  );
    printf("italic <%s>\n",       grxh->italic );
    printf("fontsize <%s>\n",     grxh->fontsize );
    printf("flag_bold:   %d\n",  grxh->flag_bold   );
    printf("flag_italic: %d\n",  grxh->flag_italic );
    puts("---------------------------------");
*/

    return (GrFontFileHeaderGRX *) grxh;
failure:
    if( fp!=NULL ) ra_close( fp );
    if(grxh!=NULL) free( grxh );
    return NULL;
}




// splits a line 'Adobe-Helvetica-Medium-R-Normal--18-180-75-75-P-98-ISO8859-1'
//
void split_adobe_string( char *infoline, GrFontFileHeaderGRX *grxh )
{
    int     i,len;
    char    c, *oldp;
    char    *dummy1 = NULL, *dummy2 = NULL;

//    printf("%s\n", infoline );

    if( infoline==NULL ) return;
    if( infoline[0]=='\0' ) return;
    len = strlen( infoline );
    oldp = infoline;
    for(i=0;i<len;i++){
    c = infoline[i];
        if( c=='-'){    // replace and assign
        infoline[i]=0;
        if( grxh->fontsize == NULL   && dummy2 != NULL ){     grxh->fontsize = oldp; break;}
        if( dummy2   == NULL   && dummy1 != NULL )      dummy2 = oldp;
        if( dummy1   == NULL   && grxh->italic != NULL )      dummy1 = oldp;
        if( grxh->italic   == NULL   && grxh->bold != NULL      )   grxh->italic = oldp;
        if( grxh->bold     == NULL   && grxh->familyname != NULL  ) grxh->bold = oldp;
        if( grxh->familyname == NULL && grxh->manufacturer != NULL) grxh->familyname = oldp;
        if( grxh->manufacturer == NULL )                      grxh->manufacturer = oldp;
        oldp = &infoline[i+1];
        }
    }
}
// splits a line 'helv29b.fnt:FONT -Adobe-Helvetica-Bold-R-Normal--24-240-75-75-P-138-ISO8859-1'
//
void split_adobe_string2( GrFontFileHeaderGRX *grxh )
{
    int     i,j,len;
    char    c, *oldp, *iline;
    char    *dummy1 = NULL, *dummy2 = NULL, *dummy3 = NULL;

    iline = (char *)&grxh->infoline[0];
    grxh->filename     = NULL;
    grxh->manufacturer = NULL;
    grxh->familyname   = NULL;
    grxh->bold         = NULL;
    grxh->italic       = NULL;
    grxh->fontsize     = NULL;
    grxh->flag_bold    = 0;
    grxh->flag_italic  = 0;
    grxh->bolditalic[0]= ' ';
    grxh->bolditalic[1]= ' ';
    grxh->bolditalic[2]= 0;
    if( iline==NULL ) return;
    if( iline[0]=='\0' ) return;
    len = strlen( iline );
    oldp = iline;
    for(i=0;i<len;i++){
    c = iline[i];
        if( c==':'){    // replace and assign
        iline[i]=0;
        if( grxh->filename == NULL ) grxh->filename = oldp;   // filename
        oldp = &iline[i+1];
        }

        if( c=='-'){    // replace and assign
        iline[i]=0;
        if( grxh->fontsize == NULL   && dummy2 != NULL ){     grxh->fontsize = oldp; break;}
        if( dummy2   == NULL   && dummy1 != NULL )      dummy2 = oldp;
        if( dummy1   == NULL   && grxh->italic != NULL )      dummy1 = oldp;
        if( grxh->italic   == NULL   && grxh->bold != NULL      )   grxh->italic = oldp;
        if( grxh->bold     == NULL   && grxh->familyname != NULL  ) grxh->bold = oldp;
        if( grxh->familyname == NULL && grxh->manufacturer != NULL) grxh->familyname = oldp;
        if( grxh->manufacturer == NULL && dummy3 != NULL) grxh->manufacturer = oldp;
        if( dummy3 == NULL ) dummy3 = oldp;
        oldp = &iline[i+1];
        }
    }
    grxh->bolditalic[0] = grxh->bolditalic[1]=' ';
    if( grxh->bold != NULL ){
        i = strcmp( grxh->bold, "Bold" );
        j = strcmp( grxh->bold, "DemiBold" );
        if( i==0 || j==0 ){ grxh->flag_bold = 1; grxh->bolditalic[0] = 'B'; }  // set bold flag
    }
    if( grxh->italic != NULL ){
        i = strcmp( grxh->italic, "I" );
        j = strcmp( grxh->italic, "O" );
        if( i==0 || j==0 ){ grxh->flag_italic = 1; grxh->bolditalic[1] = 'I'; } // set italic flag
    }
}



































/*
 * load a font from a file.
 * The fonts are GRX (Xwindows) compatible.
 *
 * condens = how many pixels to add between characters when writing text.
 *           use 0 for default.
 * verbose_flag = 1-verbose, 0-quiet.
 *
 * return NULL on error, else a pointer to a FNT structure.
 */
FNT *fnt_load_file( char *filename, int condens, int verbose_flag )
{
    unsigned int        magic;
    int                 i,j,k,l, mi,ma;
    unsigned char       *p, *o;
    FNT                 *f;
    GrFontFileHeaderGRX grxh;
/*
 *  font file structure:
 *  +-----------------------+
 *  |     FILE HEADER       |
 *  +-----------------------+
 *  |     PROPORTIONAL      |
 *  |     WIDTH TABLE       |
 *  |     (16 bit ints)     |
 *  |     (prop fonts only) |
 *  +-----------------------+
 *  |     BITMAP            |
 *  +-----------------------+
 */
    if(filename==NULL){
    if(verbose_flag==1)
        msgbox_ ("*ERROR* no filename given, '(null)'");
        return NULL;
    }
    if(verbose_flag==1)
        puts("\n***> entering fnt_load() for read");
    f = (FNT *) malloc( sizeof(FNT) );    if( f==NULL ) return NULL;
     /***** INITIALIZATION *****/

    f->filename = filename;
    if(does_file_exist (f->filename) == -1) {
    if(verbose_flag==1)
        msgbox_ ("*ERROR* InputFile does not exist, '%s'",f->filename);
        return NULL;
    }
    f->fp = ra_open (f->filename,0);
        if(f->fp == NULL) {
        if(verbose_flag==1)
            msgbox_ ("File open error, '%s'\n", f->filename);
        return NULL;
    }


    fseek (f->fp, 0L, SEEK_END);
    f->fontsize = ftell (f->fp);
    fseek (f->fp, 0L, SEEK_SET);


    /**********************************/
    /**** READ AND VALIDATE HEADER ****/
    grxh.magic = magic = ra_getd(f->fp);    /* font file magic number */
    if( magic != GRX_FONTMAGIC ){
        if(verbose_flag==1)
            msgbox_ ("Not a FNT file '%s', bad magic: 0x%x\n",f->filename, magic);
    ra_close(f->fp);
    return NULL;
    }
    grxh.bmpsize = ra_getd(f->fp);  /* character bitmap size */
    grxh.width   = ra_getw(f->fp);  /* width (average for proportional) */
    grxh.height  = ra_getw(f->fp);  /* font height */
    grxh.minchar = ra_getw(f->fp);  /* lowest character code in font */
    grxh.maxchar = ra_getw(f->fp);  /* highest character code in font */
    grxh.isfixed = ra_getw(f->fp);  /* nonzero if fixed font */
    grxh.reserved= ra_getw(f->fp);  /* ??? */
    grxh.baseline= ra_getw(f->fp);  /* baseline from top of font */
    grxh.undwidth= ra_getw(f->fp);  /* underline width (at bottom) */
    i = ra_read((char *)&grxh.fnname,GRX_NAMEWIDTH,f->fp);  /* font file name (w/o path) */
    if(i!=GRX_NAMEWIDTH){
        if(verbose_flag==1)
            msgbox_ ("error reading %d bytes \n", GRX_NAMEWIDTH);
        return NULL;
        }
    i = ra_read((char *)&grxh.family,GRX_NAMEWIDTH,f->fp);  /* font family name */
    if(i!=GRX_NAMEWIDTH){
        if(verbose_flag==1)
            msgbox_ ("error reading %d bytes \n", GRX_NAMEWIDTH);
        ra_close(f->fp);
        return NULL;
        }
    grxh.infoline[0]=0;     // my own little thingy, (FONT -Adobe-Helvetica-Medium-R-Normal--18-180-75-75-P-98-ISO8859-1)
    /**** READ AND VALIDATE HEADER ****/
    /**********************************/


    f->g.h.name   = p = (unsigned char *) malloc( GRX_NAMEWIDTH * sizeof(char)+99 );
    f->g.h.family = o = (unsigned char *) malloc( GRX_NAMEWIDTH * sizeof(char)+99 );
    if(p==NULL||o==NULL){
        if(verbose_flag==1)
            msgbox_ ("error allocating memory");
        ra_close(f->fp);
        return NULL;
        }
    strncpy(p,(char *)&grxh.fnname, GRX_NAMEWIDTH );         /* font name */
    strncpy(o,(char *)&grxh.family, GRX_NAMEWIDTH );         /* font family name */
    f->g.h.proportional = grxh.isfixed;                 /* characters have varying width    */
    f->g.h.scalable     = 0;                            /* derived from a scalable font     */
    f->g.h.preloaded    = 0;                            /* set when linked into program     */
    f->g.h.modified     = 0;                            /* "tweaked" font (resized, etc..)  */
    f->g.h.width        = grxh.width ;                  /* width (proportional=>average)    */
    f->g.h.height       = grxh.height ;                 /* font height                      */
    f->g.h.baseline     = grxh.baseline ;               /* baseline pixel pos (from top)    */
    f->g.h.ulpos        = 0;                            /* underline pixel pos (from top)   */
    f->g.h.ulheight     = grxh.undwidth;                /* underline width                  */
    f->g.h.minchar      = grxh.minchar ;                /* lowest character code in font    */
    f->g.h.numchars     = grxh.maxchar - grxh.minchar;  /* number of characters in font     */
    f->g.bitmap     = NULL;             /* character bitmap array           */
    f->g.auxmap     = NULL;             /* map for rotated & underline chrs */
    f->g.minwidth   = 0;                /* width of narrowest character     */
    f->g.maxwidth   = 0;                /* width of widest character        */
    f->g.auxsize    = 0;                /* allocated size of auxiliary map  */
    f->g.auxnext    = 0;                /* next free byte in auxiliary map  */
    l = grxh.width % 8;
    if( l != 0 ) l = 1;
    f->bytes_per_char = ((grxh.width/8)+l) * grxh.height;
    if(verbose_flag==1)
        printf("bytes_per_char:    <%d>\n",f->bytes_per_char );
    if(verbose_flag==1){
    puts  ("----HEADER INFO:----");
    printf("bmpsize:    <%d>\n",grxh.bmpsize );
    printf("width:      <%d>\n",grxh.width   );
    printf("height:     <%d>\n",grxh.height  );
    printf("minchar:    <%d>\n",grxh.minchar );
    printf("maxchar:    <%d>\n",grxh.maxchar );
    printf("isfixed:    <%d>\n",grxh.isfixed );
    printf("baseline:   <%d>\n",grxh.baseline);
    printf("undwidth:   <%d>\n",grxh.undwidth);
    printf("fnname:     <%s>\n",p);
    printf("family:     <%s>\n",o);
    puts  ("----HEADER INFO:----");
    printf("numchars:   <%d>\n",f->g.h.numchars);
    printf("isfixed<%s>\n", f->g.h.proportional ? "1:fixed width" : "0:varying width" );
    printf("<%s>\n",        f->g.h.proportional ? "no prop info in file" : "file contains prop info");
    }
    f->topchar = grxh.maxchar;
    /**************************/
    /**** read font bitmap ****/

    if( f->g.h.proportional==0 ){
    /***** proportional *****/
    if(verbose_flag==1)
        puts("reading prop info...");
    mi=99999;
    ma=0;
    k=0;
    j =  ra_getw(f->fp);
    if( j < mi ) mi = j;
    if( j > ma ) ma = j;
    f->g.chrinfo[0].width  = j;
    f->g.chrinfo[0].offset = k;
    l = j % 8;
    if(l!=0) l=1;
    k += ((j/8)+l) * f->g.h.height;

    for(i=0; i< f->g.h.numchars ; i++){
        j =  ra_getw(f->fp);
        if( j < mi ) mi = j;
        if( j > ma ) ma = j;
        f->rest[i].width = j;
        f->rest[i].offset= k;
        l = j % 8;
        if(l!=0) l=1;
        k += ((j/8)+l) * f->g.h.height;
    }

    if(verbose_flag==1)
        puts("reading prop bitmap...");

    f->g.bitmap = (unsigned char *) malloc( grxh.bmpsize+99 );
    if( f->g.bitmap ==NULL){
        if(verbose_flag==1)
            msgbox_ ("error allocating memory");
        ra_close(f->fp);
        return NULL;
        }
    /* read all bitmap */
    ra_read( f->g.bitmap, grxh.bmpsize, f->fp );
    f->g.minwidth  = mi;
    f->g.maxwidth  = ma;

    }else{

    /***** non-prop, fixed width *****/
    if(verbose_flag==1)
        puts("reading fixed bitmap...");

    k=0;
    f->g.chrinfo[0].width  = f->g.h.width;
    f->g.chrinfo[0].offset = k;
    k += f->bytes_per_char;
    for(i=0; i< f->g.h.numchars ; i++){         /* fix info structure */
    f->rest[i].width  = f->g.h.width ;
    f->rest[i].offset = k ;
    k += f->bytes_per_char;
    }

    f->g.bitmap = (unsigned char *) malloc( grxh.bmpsize+99 );
    if( f->g.bitmap ==NULL){
        if(verbose_flag==1)
            msgbox_ ("error allocating memory");
        ra_close(f->fp);
        return NULL;
        }
    /* read all bitmap */
    ra_read( f->g.bitmap, grxh.bmpsize, f->fp );

    f->g.minwidth  = f->g.h.width;
    f->g.maxwidth  = f->g.h.width;

    }/* if prop... */
    /**** read font bitmap ****/
    /**************************/

    ra_close(f->fp);
    if(verbose_flag==1)
        puts("<*** leaving fnt_load() for read\n");
    if(verbose_flag==1){
    puts("{width, offset}");
    for(i=k=0; i< f->g.h.numchars+1 ; i++){             /* fix info structure */
    printf("{%d, %d},\n",f->rest[i].width, f->rest[i].offset);
    }
    }
    f->glyph = create_bitmap( f->g.maxwidth, f->g.h.height );
    if(f->glyph==NULL) return NULL;

    f->condens = condens;
    return (FNT *) f;
}





/*   Returns the length (in pixels) of a string in the specified font.*/
int fnt_text_length(FNT *f, unsigned char *str)
{
    int     i, k, x=0, y=0, w=0;
    short   c;

    if(f==NULL) return 0;
    if(f==NULL||str==NULL) return 0;
    if(x<0||y<0) return 0;

    x=0;
    for(i=0;i<strlen(str);i++){
    c = str[i];
    if( c <  f->g.h.minchar || c >= (f->topchar+1) ) x += w;    /* inc. to next, don't display */
    if( c >= f->g.h.minchar && c <  (f->topchar+1) ){           /* display */
        k = c - f->g.h.minchar;
        if(k==0){
            x += f->g.chrinfo[0].width;
        }else{
            k--;
            x += f->rest[ k ].width;
        }
        x += f->condens;
        }
    }
    return x;
}




/*   Returns the height (in pixels) of the specified font.*/
int fnt_text_height(FNT *f)
{
    if(f==NULL) return 0;
    return f->g.h.height;
}



/*
void fnt_textout(BITMAP *bmp, FNT *fnt, char *str, int _x, int _y, int fg, int bg, int verbose_flag)
   Writes the string str onto the bitmap at position _x, _y.
   fg = foreground color;
   bg = background color, if negative, does not write on graphic;
   verbose_flag = textual output if 1, if 0 quiet.
*/
void fnt_textout(BITMAP *bmp, FNT *fnt, char *str, int _x, int _y, int fg, int bg, int verbose_flag)
{
    int offset, height, i, k,l, x=_x, y=_y, w, h, len;
    short   c;
    int     linewidth_pixel;
    int     linewidth_bytes;

    if(bmp==NULL||fnt==NULL||str==NULL) return;
    if(x<0||y<0) return;
    len = strlen(str);
    w = fnt->g.h.width;       /* glyph width  */
    h = fnt->g.h.height;      /* glyph height */
    x = _x;
    y = _y;
    for(i=0;i<len;i++){
    c = str[i];
    if( c <  fnt->g.h.minchar || c >= (fnt->topchar+1) ) x += w;  /* inc. to next, don't display */
    if( c >= fnt->g.h.minchar && c <  (fnt->topchar+1) ){    /* display */
        k = c - fnt->g.h.minchar;   /* k = rel. */
        if(verbose_flag==1)
            printf("char c: <%d>'%c' k <%d>  ",c,c,k);
        if(k==0){
            linewidth_pixel = fnt->g.chrinfo[0].width;
            l = linewidth_pixel%8;
            if(l!=0) l=1;
            linewidth_bytes = (linewidth_pixel/8) + l;
            offset          = fnt->g.chrinfo[0].offset;
            height          = fnt->g.h.height;
            }else{
        k--;
        linewidth_pixel = fnt->rest[ k ].width;
        l = linewidth_pixel%8;
        if(l!=0) l=1;
        linewidth_bytes = (linewidth_pixel/8) + l;
        offset          = fnt->rest[ k ].offset;
        height          = fnt->g.h.height;
        }
        if(verbose_flag==1){
            printf("linewidth_pixel: <%d>\n", linewidth_pixel );
            printf("linewidth_bytes: <%d>\n", linewidth_bytes );
            printf("offset         : <%d>\n", offset          );
            printf("height         : <%d>\n", height          );
        }
        print_char( bmp, fnt, x, y, offset, linewidth_pixel, linewidth_bytes, fg, bg );
        x += linewidth_pixel;
        x += fnt->condens;
        }
    }
    return;
}


/*
 * Deallocate a FNT structure.
 *
 */
void fnt_destroy_font( FNT *fnt )
{
    if(fnt==NULL) return;
    if(fnt->g.h.preloaded==1) return;   /* linked fonts cannot be killed */
    if(fnt->g.h.name   != NULL) free( fnt->g.h.name   );
    if(fnt->g.h.family != NULL) free( fnt->g.h.family );
    if(fnt->g.bitmap   != NULL) free( fnt->g.bitmap   );
    free(fnt); fnt=NULL;
}

/***************************************
    Function Returns 0 if file exist,
    else returns -1.
****************************************/
static int does_file_exist(char * filename)
{
    FILE *f;
    f = fopen(filename,"rb");    if(f == NULL){return -1;}
    fclose(f);
    return 0;
}


static void print_char(BITMAP *bmp,FNT *fnt,int _x,int _y,int _offset,int linewidth_pixel,int linewidth_bytes, int fg, int bg )
{
    int x,y,x_pix, color, j;
    unsigned char  data, bit, *bitmap = NULL;
    unsigned char mask[]={128,64,32,16,8,4,2,1,0};

    if(bmp==NULL||fnt==NULL) return;
    if( bg < 0 ) blit(bmp,fnt->glyph,_x,_y,0,0,linewidth_pixel, fnt->g.h.height);

    if(_x<0||_y<0) return;
    bitmap = fnt->g.bitmap;
    bitmap = bitmap + _offset;
    for( y=0 ; y < fnt->g.h.height ; y++ ){
    x_pix=0;
    for( x=0 ; x < linewidth_bytes ; x++ ){
    data = *bitmap++;   /* get byte from buffer */
    for( j=0 ; j<8 ; j++ ){
    bit = data;
    bit &= mask[j];
    if(bit==0) color = bg;
    if(bit!=0) color = fg;
    if(color >= 0)
        putpixel( fnt->glyph, x_pix, y, color);
        if( x_pix == linewidth_pixel ){ j = 9; break; }
        x_pix++;
    }
    }}
    blit(fnt->glyph,bmp,0,0,_x,_y,linewidth_pixel, fnt->g.h.height);
}

/*****************************************************************************/
/*****************************************************************************/
/**************** FILE.C - READ AHEAD FILE WRAPPER FUNCTIONS *****************/
/*****************************************************************************/
/*****************************************************************************/

/*
 * easy wrapping filefunctions
 * Buffered writes.
 * READ AHEAD (ra)
 * current version 0.4
 */
#include <stdio.h>
#include <stdlib.h>


/************ wrappers for unistd.h, fcntl.h ************/
static FILE *z_open(char *filename, char *type)
{
    FILE *fp;
    fp = fopen(filename, type);
    setvbuf(fp, NULL, _IOFBF, 70000 );   /* full buffer, QUICK writes */
    return fp;
}
/* wrapper version of unix's read(...) */
static int z_read(FILE *fp, void *_ptr, int size)
{
    unsigned char *ptr=_ptr;
    int i;
    i = fread(ptr,size,1,fp);
    if(i==EOF) return EOF;
    return size;
}
/* wrapper version of unix's write(...) */
/************ wrappers for unistd.h, fcntl.h ************/





/*********************************************************
 *
 *      OPEN, SEEK
 *
 *********************************************************/
/*
 * type: 0=read, 1=write;
 *
 */
static FILE *ra_open( char *filename, char type )
{
    type &= 1;
    if( type == 0 ) return z_open(filename, "rb");
    return z_open(filename, "wb");
}
static void ra_close( FILE *fp )
{
    fflush(fp);
    fclose(fp);
}

/*********************************************************
 *
 *      READING
 *
 *********************************************************/
/* read character (8 bit) */
static unsigned char ra_getc( FILE *fp )
{
    unsigned char c;
    fread((unsigned char *)&c,1,1,fp);
    return c;
}
/* read word (16 bit) */
static unsigned short int ra_getw( FILE *fp )
{
    unsigned short int us,u1,u2;
    u1 =(unsigned short int) ra_getc(fp);
    u2 =(unsigned short int) ra_getc(fp);
    us = (u2<<8) | u1;
    return us;
}
/* read dualword (32 bit) */
static unsigned int ra_getd( FILE *fp )
{
    unsigned int us,u1,u2;
    u1 =(unsigned long int) ra_getw(fp);
    u2 =(unsigned long int) ra_getw(fp);
    us =(u2<<16) | u1;
    return us;
}
/*
 * ptr = pointer
 * size = size
 * fp = file pointer
 */
static int ra_read( void *ptr, int size, FILE *fp )
{
    int i;

    i = z_read(fp, ptr, size);
    if(i==EOF) return EOF;
    return size;
}

/*****************************************************************************/
/*****************************************************************************/
/**************** FILE.C - READ AHEAD FILE WRAPPER FUNCTIONS *****************/
/*****************************************************************************/
/*****************************************************************************/










/*

char11.fnt:FONT -Bitstream-Charter-Medium-R-Normal--8-80-75-75-P-45-ISO8859-1

*/


GrFontFileHeaderGRX * fnt_make_branches( char *filename )
{
    int                 j=0;
    FILE                *fp = NULL;
    GrFontFileHeaderGRX *grxh = NULL;
    char                *st;



    if(filename==NULL) return NULL;
    if(does_file_exist (filename) == -1) return NULL;


    fp = ra_open (filename,0);  if(fp == NULL) goto failure;




//    parse_FONT_line( (char *)&stringline, fp );
// *******************************************************************
    for(j=0;;j++){
    grxh = (GrFontFileHeaderGRX *) malloc( sizeof(GrFontFileHeaderGRX) );    if( grxh == NULL ) goto failure;
    grxh->infoline[0]=0;
//    grxh->bolditalic[2]=0;


    fgets( (char *)&grxh->infoline, sizeof(grxh->infoline)-1, fp );
//    if( i <= 0 ) break;
    if(feof(fp)) break;
    st = strrchr( (char *)&grxh->infoline, '\n'); // reverse search for enter character
    *st++=0;    // remove enter character from line


// helv29b.fnt:FONT -Adobe-Helvetica-Bold-R-Normal--24-240-75-75-P-138-ISO8859-1


    // ..........
    // extract filename

//    printf("<%s>\n", grxh->infoline );

    // ..........
    split_adobe_string2( grxh );

/*
    puts("---------------------------------");
    printf("filename  <%s>\n",    grxh->filename    );
    printf("manufacturer <%s>\n", grxh->manufacturer);
    printf("familyname <%s>\n",   grxh->familyname );
    printf("bold <%s>\n",         grxh->bold       );
    printf("italic <%s>\n",       grxh->italic     );
    printf("fontsize <%s>\n",     grxh->fontsize   );
    printf("flag_bold:   %d\n",   grxh->flag_bold  );
    printf("flag_italic: %d\n",   grxh->flag_italic);
    printf("bolditalic: <%s>\n",  grxh->bolditalic );
    puts("---------------------------------");
*/
    add_in_tree( grxh );

//    if(j==13)
//        break;
    }




// Adobe-Helvetica-Medium-R-Normal--18-180-75-75-P-98-ISO8859-1
// *******************************************************************



//    view_tree( grxh );






    return (GrFontFileHeaderGRX *) grxh;
failure:
    if( fp!=NULL ) ra_close( fp );
    if(grxh!=NULL) free( grxh );
    return NULL;
}















TREEITEM *alloc_treeitem()
{
    TREEITEM    *t;

    t = (TREEITEM *) malloc( sizeof(TREEITEM)*2+99 );    if(t==NULL) return NULL;
    t->prev   = NULL;
    t->next   = NULL;
    t->data   = NULL;
    t->branch = NULL;
    return t;
}

void    free_treeitem( TREEITEM *t )
{
    if(t==NULL) return;
    if(t->data != NULL ) free( t->data );
    free( t );
}


static TREEITEM *fonts = NULL;      // on top of the tree



TREEITEM *add_in_tree( GrFontFileHeaderGRX *grxh  )
{
    TREEITEM    *t=NULL, *family, *size, *thickness, *filename;
    TREEITEM    *prev;

    grxh->bolditalic[2]=0;
    if( fonts== NULL ){ // append as first
        if( (fonts     = alloc_treeitem())==NULL) return NULL;
        if( (family    = alloc_treeitem())==NULL) return NULL;
        if( (size      = alloc_treeitem())==NULL) return NULL;
        if( (thickness = alloc_treeitem())==NULL) return NULL;
        if( (filename  = alloc_treeitem())==NULL) return NULL;
        fonts->branch = family;     // first
        family->branch = size;
        size->branch = thickness;
        thickness->branch = filename;
        family->data    = (char *) grxh->familyname;
        size->data      = (char *) grxh->fontsize;
        thickness->data = (char *)&grxh->bolditalic;
        filename->data  = (char *) grxh->filename;

    }else{      // append in tree, regular case
        family = prev = fonts->branch;
        for(;;){    // now search for a family duplicate
            if( family == NULL ){  // unique name, append family tree

// ----------------------------------------
        if( (family    = alloc_treeitem())==NULL) return NULL;
        if( (size      = alloc_treeitem())==NULL) return NULL;
        if( (thickness = alloc_treeitem())==NULL) return NULL;
        if( (filename  = alloc_treeitem())==NULL) return NULL;
    prev->next = family;
    family->prev =  prev;
    family->branch = size;
    size->branch = thickness;
    thickness->branch = filename;
        family->data    = (char *) grxh->familyname;
        size->data      = (char *) grxh->fontsize;
        thickness->data = (char *)&grxh->bolditalic;
        filename->data  = (char *) grxh->filename;
// ----------------------------------------
            break;
            }
            if( strcmp( family->data, grxh->familyname ) == 0 ){
            append_family_tree( family, grxh );
            break;
            }
        prev = family;
        family = family->next;   // go to next family in link
        }// for(;;)

    }

    return t;
}










TREEITEM *append_thickness_tree( TREEITEM *_thickness, GrFontFileHeaderGRX *grxh )
{
    TREEITEM    *filename;


    filename = _thickness->branch;      // go down a branch
    if( filename==NULL)
        if( (filename  = alloc_treeitem())==NULL) return NULL;
    filename->data  = (char *) grxh->filename;
    return _thickness;
}









TREEITEM *append_size_tree( TREEITEM *_size, GrFontFileHeaderGRX *grxh )
{
    TREEITEM    *thickness, *filename;
    TREEITEM    *prev = NULL;
    int         i;

//        if( (filename  = alloc_treeitem())==NULL) return NULL;

    thickness = _size->branch;  // go down a branch

    for(;;){    // search for a size duplicate or unique one thickness
        if( thickness == NULL ){        // unique name, append tree

//allegro_message("*c*");
// -----------------------------------------------
        // new thickness ->> new filename also
        if( (thickness = alloc_treeitem())==NULL) return NULL;
        if( (filename  = alloc_treeitem())==NULL) return NULL;

    prev->next = thickness;
    thickness->prev  = prev;
    thickness->branch = filename;
        thickness->data = (char *)&grxh->bolditalic;
        filename->data  = (char *) grxh->filename;

// -----------------------------------------------

        break;
        }
    i = strcmp( thickness->data, (char *)&grxh->bolditalic );  // equal sizes
        if( i==0 ){    // we got a match!

//allegro_message("*d*");
        append_thickness_tree( thickness, grxh );

        break;
        }
       prev = thickness;
       thickness = thickness->next;   // go to next size in link
    }//for(;;)
    return _size;
}









TREEITEM *append_family_tree( TREEITEM *_family, GrFontFileHeaderGRX *grxh )
{
    TREEITEM    *size, *thickness, *filename;
    TREEITEM    *prev = NULL;
    int         i;

    size = _family->branch;  // go down a branch
    for(;;){        // search for a size duplicate or unique one
        if( size == NULL ){ // didn't found a matching size.
// -------------------------------------------
//allegro_message("*a*");
        if( (size      = alloc_treeitem())==NULL) return NULL;
        if( (thickness = alloc_treeitem())==NULL) return NULL;
        if( (filename  = alloc_treeitem())==NULL) return NULL;
    size->prev = prev;
    prev->next = size;
    size->branch = thickness;
    thickness->branch = filename;
        size->data      = (char *) grxh->fontsize;
        thickness->data = (char *)&grxh->bolditalic;
        filename->data  = (char *) grxh->filename;
// -------------------------------------------
        break;
        }


    i = strcmp( size->data, grxh->fontsize );  // equal sizes
        if( i==0 ){    // we got a match!

//allegro_message("*b*");

        append_size_tree( size, grxh );     // errbone
        break;
        }
        prev = size;
        size = size->next;   // go to next size in link
    }//for(;;)
    return _family;
}


int amount_familynames()
{
    TREEITEM    *family;
    int         i=0;

    family = fonts->branch;     // begin with first family
    if (family==NULL) return 0;
    for(i=0;;i++){
    family = family->next;
    if( family==NULL ) return i;
    }
    return 0;
}

/*
char *seek_familyname_data( int index )
{
    TREEITEM    *family, *prev;
    int i,len;

    family = fonts->branch;     // begin with first family

    len = amount_familynames();
    for(i=0;i< len ; i++){
    if( i==index ) break;
    prev = family;
    family = family->next;
    if( family == NULL ){ family = prev; break;}
    }

    return (family->data);
}
*/
TREEITEM *seek_familyname_by_index( int index )
{
    TREEITEM    *family, *prev;
    int i,len;

    family = fonts->branch;     // begin with first family
    len = amount_familynames();
    for(i=0;i< len ; i++){
    if( i==index ) break;
    prev = family;
    family = family->next;
    if( family == NULL ){ family = prev; break;}
    }

    return family;
}
TREEITEM *seek_familyname_by_string( char *string )
{
    TREEITEM    *family, *prev;
    int i,len;

    family = fonts->branch;     // begin with first family
    len = amount_familynames();
    for(i=0;i< len ; i++){
    if( strcmp(family->data, string)==0 ) break;
    prev = family;
    family = family->next;
    if( family == NULL ){ family = prev; break;}
    }
    return family;
}





// ********************************************************
// ********************************************************
// ********************************************************


void  print_sizes( TREEITEM *family )
{
    TREEITEM    *size;

    size = family->branch;
//    printf("sizes:\n");
    for(;;){
    if( size==NULL) break;
//    printf("<%s>",  size->data );
    size = size->next;
    }
    puts("");

}
void  print_thickness( TREEITEM *family )
{
    TREEITEM    *size, *thickness;

    size      = family->branch;
    thickness = size->branch;
//    printf("Thickness:\n");
    for(;;){
    if( thickness==NULL) break;
//    printf("<%s>",  thickness->data );
    thickness = thickness->next;
    }
    puts("");
}
void  print_filename( TREEITEM *family )
{
    TREEITEM    *size, *thickness, *filename;

    size      = family->branch;
    thickness = size->branch;
    for(;;){
    if( thickness==NULL) break;
    filename =  thickness->branch;
//    printf("<%s>",  filename->data );
    thickness = thickness->next;
    }
    puts("");
}



void view_tree( GrFontFileHeaderGRX *grxh   )   // for debugging
{
    TREEITEM     *family;

    if( fonts== NULL ) return;  // tree don't exist

    family = fonts->branch;

//    printf("families:\n");
    for(;;){
    if( family==NULL) break;
//    printf("\n*** <%s>\n",  family->data );

    print_sizes( family );
    print_thickness( family );
    print_filename( family );

    family = family->next;
    }
//    puts("");
}






/* ***************************************************************************
 *****************************************************************************
 *****************************************************************************


                         DIALOG WINDOW

    void dlg_select_font( int mx, int my )

 *****************************************************************************
 *****************************************************************************
 *************************************************************************** */

/**** Common Data for all dialog functions ****/
//static  PALETTE opal;
//static  int file_gui1,   file_gui2;
//static  RGB file_rgb1 = {20,30,34};
//static  RGB file_rgb2 = {40,50,60};
//static  RGB file_rgb3 = {50,50,20};

//static char *pathptr, *stickypath=NULL, *stickypathbkmap=NULL;
/**** Common Data for all dialog functions ****/

// common thingys for all dialogs
static int  dlg_ret_val=0;              // used by all dialogs,return values for dlgs,0=cancel,1=ok
static void s_btn_ok(DIALOG *d);        // used by all dialogs
static void s_btn_cancel(DIALOG *d);    // used by all dialogs
static void s_btn_ok(DIALOG *d)
{
    wait_for_mousebutton_release();     // ok pressed
    dlg_ret_val = 1;
}
static void s_btn_cancel(DIALOG *d)     // cancel pressed
{
    wait_for_mousebutton_release();
    dlg_ret_val = 0;
}


// ---------------------------------------------------------------------------




/** helper functions **/
static  char *str_fonts(int index, int *list_size);
static  char *str_sizes(int index, int *list_size);

/** helper functions **/

static  char numstring1[64];
static int flag_italic = FALSE;
static int flag_bold   = FALSE;


static int s_list_proc(int msg, DIALOG *d, int c);

// int d_check_proc(int msg, DIALOG *d, int c);

//(dialog proc)(x)(y)(w)(h)(fg)(bg)(key)(flags)(d1)(d2)(dp)(dp2)(dp3)
static DIALOG dlg_fnt[]={
{ ed_win        ,0,0,320,220,0,0,  0,     0, 1,  1, "[ Font options ]",NULL,NULL},
{ d_text_proc   ,0,0, 32, 12,0,0,  0,     0, 0,  0, "Fonts:",  NULL,NULL},
{ d_text_proc   ,0,0, 32, 12,0,0,  0,     0, 0,  0, "Sizes:", NULL,NULL},
{ s_list_proc   ,0,0,150,130,0,0,  0,     0, 0, 0,  &str_fonts,NULL,NULL},
{ d_list_proc   ,0,0, 60,100,0,0,  0,  D_HIDDEN, 0, 0,  &str_sizes,NULL,NULL},

{ ed_check      ,0,0, 12, 12,0,0,  0,     0, 0,  0, "Italic",  (int *)&flag_italic,NULL},
{ ed_check      ,0,0, 12, 12,0,0,  0,     0, 0,  0, "Bold",    (int *)&flag_bold,NULL},

{ d_text_proc   ,0,0, 32, 12,0,0,  0,     0, 0,  0, "Condens(+/-):",   NULL,NULL},
{ d_edit_proc   ,0,0, 35, 18,0,0,  0,     0, 3,  0, numstring1,NULL,NULL},

{ ed_button_exit,0,0, 80, 22,0,0,'o',     0, 0,  0, "OK",s_btn_ok,NULL},
{ ed_button_exit,0,0, 80, 22,0,0, 27,     0, 0,  0, "CANCEL",s_btn_cancel,NULL},
{ NULL }};

#define DLISTFONTNAMES 3
#define DLISTFONTSIZES 4
static DIALOGXYCOORDS dlg_fnt2[]={
{   0,    0 },
{  10,   20+5 },
{ 170,   20+5 },
{  10,   30+5 },
{ 170,   30+5 },

{ 235,  110+5 },
{ 235,  130+5 },

{ 170,  150+5 },
{ 170,  165+5 },

{ 10,   190 },
{ 180,  190 },


{   0,    0 }};


/********************************************/
/**                                        **/
/**                                        **/
/**    Popup  selection                    **/
/**                                        **/
/**                                        **/
void dlg_select_font()
{
    int     i=0,j=0, condens=0;
    char    tmpstring[299];
    TREEITEM    *family, *size, *thickness, *filenamed, *prev=NULL;
    char        *filename, bolditalic[3];

//    flag_italic = FALSE;
//    flag_bold   = FALSE;

    dlg_ret_val = 0;            // assume cancel was pressed
    for(i=0;i<sizeof(numstring1)-1;i++) numstring1[i]=0;
    numstring1[0]='0';

    dlg_fnt[DLISTFONTSIZES].flags = D_HIDDEN;
    dlg_fnt[DLISTFONTSIZES].flags = 0;

    prepare_prefix_dlgs( dlg_fnt, dlg_fnt2,-1,-1,128 );
    do_dialog( dlg_fnt, -1 );

    if( dlg_ret_val==1 ){   // OK was pressed
    // check to see if file exist
    condens = atoi( numstring1 );

    bolditalic[0] = bolditalic[1] = ' ';
    bolditalic[2] = 0;
    if( flag_bold   == TRUE) bolditalic[0] = 'B';
    if( flag_italic == TRUE) bolditalic[1] = 'I';

    i = dlg_fnt[DLISTFONTNAMES].d1;
    j = dlg_fnt[DLISTFONTSIZES].d1;
    family    = seek_familyname_by_index( i );
    size      = family->branch;
    // -----------------
    // seek fitting size
    for(i=0;;i++){
    if( i == j ) break;
    if( size==NULL ){ size = prev; break;}
//    if(strcmp( size->data, (char *)&iksipiksi ) == 0) break;
    prev = size;
    size = size->next;
    }
    // -----------------


    thickness = size->branch;
    // -----------------
    // seek fitting thickness
    for(i=0;i<4;i++){
    if( thickness==NULL ){ thickness = prev; break;}
    if(strcmp( thickness->data, (char *)&bolditalic ) == 0) break;
    prev = thickness;
    thickness = thickness->next;
    }
    // -----------------

    filenamed = thickness->branch;
    filename = filenamed->data;

    strcpy( (char *)&tmpstring, GRX_FONTSUBDIR );
    strcat( (char *)&tmpstring, filename );
//    filename = fix_filename_case( (char *)&tmpstring );
//    filename = fix_filename_slashes( filename );

//allegro_message("filename: <%s>", tmpstring );

    fnt8 = fnt_load_file( (char *)&tmpstring, condens, 1 );
    if( fnt8==NULL ){ allegro_message("Loading failed!" );
    }


    }
    prepare_suffix_dlgs( dlg_fnt, dlg_fnt2 );
}




static char *selectedfamily = NULL;



// --------------------------

static int  iks=0;




// calls d_list_proc
static int s_list_proc(int msg, DIALOG *d, int c)
{
    TREEITEM    *t;
    char        *ch;
    int ret;

    switch( msg ){

    case MSG_CLICK:
        ret = d_list_proc(msg, d,  c);

    t = seek_familyname_by_index( d->d1 );
    ch = t->data;
    selectedfamily = ch;

    iks = 1;
    dlg_fnt[DLISTFONTSIZES].flags = 0;
        object_message( &dlg_fnt[DLISTFONTSIZES], MSG_END, 0 );
        object_message( &dlg_fnt[DLISTFONTSIZES], MSG_START, 0 );
        object_message( &dlg_fnt[DLISTFONTSIZES], MSG_DRAW, 0 );

// allegro_message("text: <%s>",ch);

    break;

//    case MSG_DCLICK:
//        dlg_ret_val = 1;    // ok pressed
//    ret = D_EXIT;
//    break;

    default:
    ret = d_list_proc(msg, d,  c);
    break;
    }
    return ret;
}









/*
int amount_familynames()
{
    TREEITEM    *family;
    int         i=0;

    family = fonts->branch;     // begin with first family
    if (family==NULL) return 0;
    for(i=0;;i++){
    family = family->next;
    if( family==NULL ) return i;
    }
    return 0;
}
*/







char *seek_sizes( int index, char *string )
{
    TREEITEM    *size, *family, *prev;
    int         i;

    if( string == NULL ) return "???";
    family = seek_familyname_by_string( string );

    // now seek the sizes that correspond to this family
    size = family->branch;

    for(i=0;;i++){
    if( i == index ) break;
    prev = size;
    size = size->next;
    if( size == NULL ){ size = prev; break;}
    }


    return (size->data);
}





int amount_sizes( char *string )
{
    TREEITEM    *size, *family, *prev;
    int         i;

    if(string==NULL) return 1;
    family = seek_familyname_by_string( string );
    // selectedfamily
    size = family->branch;

    for(i=0;;i++){
    prev = size;
    size = size->next;
    if( size == NULL ){ size = prev; break;}
    }

    return i;
}












static char *str_fonts(int index, int *list_size)
{   /*
   If index is zero or positive, the function should return a pointer to the
   string which is to be displayed at position index in the list. If index 
   is negative, it should return NULL and list_size should be set to the 
   number of items in the list. 
    */
    TREEITEM    *family;


    if(index<0){ *list_size = amount_familynames(); return NULL; }
//    return "New Century Schoolbook";

    family = seek_familyname_by_index( index );
//    selectedfamily = family->data;
    return family->data;
}



static char *str_sizes(int index, int *list_size)
{   /*
   If index is zero or positive, the function should return a pointer to the
   string which is to be displayed at position index in the list. If index 
   is negative, it should return NULL and list_size should be set to the 
   number of items in the list. 
    */

    if(index<0){ *list_size = amount_sizes( selectedfamily ); return NULL; }
    return seek_sizes( index, selectedfamily );            // return string

    if( iks==0)
    return "hello";
    if( iks==1)
    return "world";
}





