/*
 * low level fli decoding/encoding
 *
 */
#include <stdio.h>
#include <stdlib.h>
#include <allegro.h>
#include "flilib.h"
#include "flifile.h"
#include "flisubs.h"        /* subroutines:  files...etc... */
#include "fli.h"        /* for DEBUG */

/*** helpers ***/
static void master_unbrun( FLI *f, int bytes_per_pixel );
static void master_unlc( FLI *f, int bytes_per_pixel );
/*** helpers ***/





/*
 * Allocate a FLI structure in memory and sets default values.
 * Returns allocated FLI structure or NULL on error.
 * input: verbose_flag = ( 0=quiet, 1=verbose )
 *
 * returns NULL on allocation error.
 *
 *
 */
FLI *alloc_fli_struc( int verbose_flag )
{
    int i;
    FLI *f;

    f = (FLI *) malloc( sizeof(FLI) );
    if(f==NULL){
   //     if(verbose_flag==1)
   //         debug_printf(f,0,"malloc error, while allocating FLI structure");
        return NULL;
    }
    /* standard FLx header */
    f->size=0;
    f->type=0;
    f->frames=0;
    f->width=0;
    f->height=0;
    f->depth=0;
    f->flags=0;
    f->speed=0;
    f->reserved1=0;
    f->created=0;
    f->creator=0;
    f->updated=0;
    f->updater=0;
    f->aspect_dx=0;
    f->aspect_dy=0;
    f->ext_flags=0;
    f->keyframes=0;
    f->totalframes=0;
    f->req_memory=0;
    f->max_regions=0;
    f->transp_num=0;
    for(i=0;i<24;i++) f->reserved2[i]=0;
    f->oframe1=0;
    f->oframe2=0;
    for(i=0;i<40;i++) f->reserved3[i]=0;
    /* end standard FLx header */

    f->verbose=0;
    f->fp=NULL;
    f->filename=NULL;
    f->stride=0;
    f->speedflag=0;
    f->speedunit=0;
    f->current_frame=0;
    f->framesize=0;
    f->palentry=0;
    f->read_write=0;    // open for read (default)
    f->images_to_go=0;
    f->fileposition=0;
    f->colormetric=0;

    f->update_palette=0;
    f->upd_pal_begin=0;
    f->upd_pal_end=0;
    f->img_begin=0;
    f->img_end=0;

    f->image      = NULL;
    f->image_prev = NULL;
    f->image_compr = NULL;
    f->palette      = NULL;
    f->palette_prev = NULL;
    f->palette_compr = NULL;

    f->debuglevel1[0]=0;
    f->debuglevel2[0]=0;
    f->debuglevel3[0]=0;
    f->debuglevel4[0]=0;
    f->debugappend[0]=0;
    f->errnumber = FERR_OK;

    return f;
}

/*
 * Free FLI structure
 *
 */
void free_fli_struc(FLI *f)     // error happens while freeing, means
{                               // there is some error overwriting
    if(f==NULL) return;

    if(f->palette!=NULL){       free(f->palette);      f->palette =   NULL;  }
    if(f->palette_prev!=NULL){  free(f->palette_prev); f->palette_prev=NULL; }
    if(f->palette_compr!=NULL){ free(f->palette_compr);f->palette_compr=NULL; }
    if(f->image!=NULL){         free(f->image);        f->image =     NULL;  }
    if(f->image_prev!=NULL){    free(f->image_prev);   f->image_prev =NULL;  }
    if(f->image_compr!=NULL){   free(f->image_compr);  f->image_compr=NULL; }
    if(f->fp!=NULL){            tclose(f->fp);         f->fp =        NULL; }
    free(f);    f = NULL;
    return;
}










/* open a file from a file
 *
 * Input     filename
 *
 * output    File stream pointer
 */
FILE *openfliload( char *filename, FLI *f )
{
    f->filename = filename;
    f->fp = topen( filename,"rb" );
    if( f->fp == NULL ){ f->errnumber = FERR_CANTOPENFILEREAD; return NULL;}
    return f->fp;
}










/* read fli header information
 * input   p = file handler
 *         f = pointer to allocated FLI header.
 * output  f pointer is returned
 *         NULL on error
 *
 */
FLI *read_fli_header( FLI *f )
{
    FILE *p;
    p = f->fp;

    trewind(p);
    tclearerr(p);    clearerr( p );
    f->size        = r_dword(p);   if(ferror(p)!=0){ f->errnumber = FERR_CANTREADHEADER; return f; }
    f->type        = r_word(p);    if(ferror(p)!=0){ f->errnumber = FERR_CANTREADHEADER; return f; }
    f->frames      = r_word(p);    if(ferror(p)!=0){ f->errnumber = FERR_CANTREADHEADER; return f; }
    f->width       = r_word(p);    if(ferror(p)!=0){ f->errnumber = FERR_CANTREADHEADER; return f; }
    f->height      = r_word(p);    if(ferror(p)!=0){ f->errnumber = FERR_CANTREADHEADER; return f; }
    f->depth       = r_word(p);    if(ferror(p)!=0){ f->errnumber = FERR_CANTREADHEADER; return f; }
    f->flags       = r_word(p);    if(ferror(p)!=0){ f->errnumber = FERR_CANTREADHEADER; return f; }
    f->speed       = r_dword(p);   if(ferror(p)!=0){ f->errnumber = FERR_CANTREADHEADER; return f; }
    f->reserved1   = r_word(p);    if(ferror(p)!=0){ f->errnumber = FERR_CANTREADHEADER; return f; }
    f->created     = r_dword(p);   if(ferror(p)!=0){ f->errnumber = FERR_CANTREADHEADER; return f; }
    f->creator     = r_dword(p);   if(ferror(p)!=0){ f->errnumber = FERR_CANTREADHEADER; return f; }
    f->updated     = r_dword(p);   if(ferror(p)!=0){ f->errnumber = FERR_CANTREADHEADER; return f; }
    f->updater     = r_dword(p);   if(ferror(p)!=0){ f->errnumber = FERR_CANTREADHEADER; return f; }
    f->aspect_dx   = r_word(p);    if(ferror(p)!=0){ f->errnumber = FERR_CANTREADHEADER; return f; }
    f->aspect_dy   = r_word(p);    if(ferror(p)!=0){ f->errnumber = FERR_CANTREADHEADER; return f; }
    f->ext_flags   = r_word(p);    if(ferror(p)!=0){ f->errnumber = FERR_CANTREADHEADER; return f; }
    f->keyframes   = r_word(p);    if(ferror(p)!=0){ f->errnumber = FERR_CANTREADHEADER; return f; }
    f->totalframes = r_word(p);    if(ferror(p)!=0){ f->errnumber = FERR_CANTREADHEADER; return f; }
    f->req_memory  = r_dword(p);   if(ferror(p)!=0){ f->errnumber = FERR_CANTREADHEADER; return f; }
    f->max_regions = r_word(p);    if(ferror(p)!=0){ f->errnumber = FERR_CANTREADHEADER; return f; }
    f->transp_num  = r_word(p);    if(ferror(p)!=0){ f->errnumber = FERR_CANTREADHEADER; return f; }
                     r_bytes(p,24);if(ferror(p)!=0){ f->errnumber = FERR_CANTREADHEADER; return f; }
    f->oframe1     = r_dword(p);   if(ferror(p)!=0){ f->errnumber = FERR_CANTREADHEADER; return f; }
    f->oframe2     = r_dword(p);   if(ferror(p)!=0){ f->errnumber = FERR_CANTREADHEADER; return f; }
                     r_bytes(p,40);if(ferror(p)!=0){ f->errnumber = FERR_CANTREADHEADER; return f; }
    return f;
}




/*
void  do_fli_unbrun( FLI *f );
void  do_delta_unflc( FLI *f );
void  do_delta_unfli( FLI *f );
void  do_dta_unbrun( FLI *f );
void  do_dta_copy( FLI *f );
void  do_dta_unlc( FLI *f );
*/

void  do_fli_unbrun( FLI *f )   /* works 100% */
{
    int             packets, i,x,y, curr;
    signed short    size;
    char            size0;
    BYTE            uc;

#ifdef DEBUG
printf("do_fli_unbrun()\n{...\n");
printf("line stride <%d> bytes\n",f->stride);
#endif

    if(f->verbose==1)
        debug_printf(f,4,"do_fli_unbrun()");

    for(y=0 ; y < f->height ; y++)      /* for each line... */
    {
    packets = r_byte( f->fp );         /*    printf("packets:<%d>\n",packets);*/
    if(feof( f->fp )){ f->errnumber=FERR_EOF; return;}       // end-of-file occured
    i = y * f->stride;  /* linebytes */
    if( packets!=0 ){    /*  TYPE   FLI */
    /* one line*/
    while (packets-- > 0)       /*** do rle packets( one line ) ***/
    {
    size0 = (char) r_byte( f->fp );   /* packet count */
    size = (signed short) size0;
    if(feof( f->fp )){ f->errnumber=FERR_EOF; return;}       // end-of-file occured
        if( size > 0 ){
            /* If size_count is positive the data consists of a single byte which is repeated size_count times. */
            uc = r_byte( f->fp );
            for(x=0; x < size ;x++) f->image[ i++ ] = uc;
        }else{
            /* If size_count is negative there are -size_count bytes of data which are copied to the screen.*/
            size = -size;
            for(x=0; x < size ;x++) f->image[ i++ ] = r_byte( f->fp );
        }
    }   /* one line */

    }else{    /* TYPE FLC */    /*** do rle packets ***/

    curr=0;
    /* one line*/
    while ( curr < (f->stride-1) )  /*** do rle packets( one line ) ***/
    {
    size0 = (char) r_byte( f->fp );   /* packet count */
    size = (signed short) size0;
    if(feof( f->fp )){ f->errnumber=FERR_EOF;  return;}       // end-of-file occured
        if( size > 0 ){
            curr += size;
            /* If size_count is positive the data consists of a single byte which is repeated size_count times. */
            uc = r_byte( f->fp );
            for(x=0; x < size ;x++) f->image[ i++ ] = uc;
        }else{
            /* If size_count is negative there are -size_count bytes of data which are copied to the screen.*/
            size = -size;
            curr += size;
            for(x=0; x < size ;x++) f->image[ i++ ] = r_byte( f->fp );
        }
    }   /* one line */
    }   /* type FLI/FLC+ */

    }   /* all lines */
}












void  do_delta_unfli( FLI *f )        /* ok 100% */
{
    int             packets, i,x,y;
    unsigned short  lineskip, linedata, skip;
    signed short    size;
    char            size0;
    BYTE            uc;


    lineskip = r_word( f->fp );    /* skip count */
    linedata = r_word( f->fp );    /* data lines */

    f->img_begin = lineskip;


#ifdef DEBUG
printf("do_delta_fli()\n{...\n");
printf("lineskip:<%d> linedata:<%d>\n",lineskip,linedata);
#endif
    if(f->verbose==1)
        debug_printf(f,4,"do_delta_unfli() lineskip:<%d> linedata:<%d>\n",lineskip,linedata);


    i=0;
    for( y=lineskip ; y < (lineskip+linedata) ; y++)      /* for each line... */
    {
    packets = r_byte( f->fp );    if(feof( f->fp )){ f->errnumber=FERR_EOF;  return;}       // end-of-file occured
    i = y * f->stride;  /* linebytes */


    while (packets-- > 0)       /*** do rle packets( one line ) ***/
    {
    skip = r_byte( f->fp );
    size0 = (char) r_byte( f->fp );   /* packet count */
    size = (signed short) size0;
    if(feof( f->fp )){ f->errnumber=FERR_EOF;  return; }      // end-of-file occured


    if( size==0 ){  /* special case: skip count is bigger than 255 */
    i+=skip;
    skip =  r_byte( f->fp );
    size0 = (char) r_byte( f->fp );   /* packet count */
    size = (signed short) size0;
    if(feof( f->fp )){ f->errnumber=FERR_EOF;  return;}       // end-of-file occured
    }

    i += skip;

    if( size > 0 ){
        /*If it is positive, that many bytes of data follow and are to be copied to the screen.*/
        for(x=0; x < size ;x++) f->image[ i++ ] = r_byte( f->fp );
        }else{
        /*If it's negative a single byte follows, and is repeated -skip_count times.*/
        uc = r_byte( f->fp );
        size = -size;
        for(x=0; x < size ;x++) f->image[ i++ ] = uc;
        }
    }/*line packets */
    }   /* all lines */

    f->img_end   = y;

#ifdef DEBUG
printf("...}\n");
#endif
}





void  do_delta_unflc( FLI *f )
{
    int             packets, i,x,y;
    WORD            linedata, skip, opcode, optype;
    signed short    size;
    char            size0;
    BYTE            uc, uc2 ,last_byte=0, oddflag;


    f->img_begin = 0;
    f->img_end   = 0;


#ifdef DEBUG
printf("do_delta_flc()\n{...\n");
#endif

    linedata = r_word( f->fp );    /* data lines */
    if(feof( f->fp )){ f->errnumber=FERR_EOF;  return;}       // end-of-file occured
    if(linedata==0) return;

#ifdef DEBUG
printf("linedata:<%d>\n",linedata);
#endif

    if(f->verbose==1)
        debug_printf(f,4,"do_delta_unflc() linedata:<%d>\n",linedata);


    y=0;
    for(;;)      /* for each line... */
    {   /* add infinitum, exit when linedata==0 */
    i = y * f->stride;  /* linebytes */

    skip = packets = 0;
    oddflag = 0;

    for(;;)     /* handle opcodes */
    {
    opcode = r_word( f->fp );
    optype = (opcode & 0xc000);
    opcode = opcode & 0x3fff;       /* extract lower data */

#ifdef DEBUG
/* too much data... */
/*    printf("optype: <0x%0x>, opcode <0x%0x>...",optype, opcode);*/
#endif

    if( optype==0x0000 ){   /* packet count*/
        packets = opcode;

        #ifdef DEBUG
        /*        printf("-packet_count:<%d>-\n",packets);*/
        #endif

        break;
        }


    if( optype==0x4000 ){
        debug_printf( f, 4, "Undefined code in delta_flc" );
        return;
        }
    if( optype==0x8000 ){
        if(f->img_begin == 0) f->img_begin = y;
/*        puts("-add low byte to end of line-");*/
        /* low byte of opcode in last pixel of line */
        /* width is odd */
        last_byte = (opcode & 0xff);
        oddflag = 1;
        }
    if( optype==0xc000 ){   /* line skip count, signed */
        if( (opcode&0x2000) !=0 ) opcode=0x4000-opcode; /* (*sight*)...if stupid people could fly, there would be no sunshine... */
#ifdef DEBUG
/*        printf("-skiplines abs value<%d>-\n",opcode);*/
#endif
        y = y + opcode;
        i = y * f->stride;  /* linebytes */
        }
    }

    if( oddflag==1 ){
        if(f->img_begin == 0) f->img_begin = y;
        f->image[ (i+f->stride-1) ] = last_byte;
    }

    if( packets==0 ){ /* if no packets, then ignore line */
    y++;              /* ignore line */
    i = y * f->stride;  /* linebytes */
    }

    if( packets != 0 ){ /* if there are packets, process them */
    if( linedata==0 ) return;   /* exit when all lines are processed */
    linedata--;

    skip = size = 0;
    while (packets-- > 0)       /*** do rle packets( one line ) ***/
    {
    skip = r_byte( f->fp );
    size0 = (char) r_byte( f->fp );   /* packet count */
    size = (signed short) size0;
    if(feof( f->fp )){ f->errnumber=FERR_EOF;  return;}       // end-of-file occured
    if( size==0 ){  /* special case: skip count is bigger than 255 */
        i+=skip;
        skip =  r_byte( f->fp );
        size0 = (char) r_byte( f->fp );   /* packet count */
        size = (signed short) size0;
        }
        i+=skip;
    if( size > 0 ){
        /*If it is positive, that many words of data follow and are to be copied to the screen.*/
        if(f->img_begin == 0) f->img_begin = y;
        f->img_end = y;
        for(x=0; x<size ;x++){
            f->image[i++] = r_byte( f->fp );
            f->image[i++] = r_byte( f->fp );
            }
        }else{
        /*If it's negative a single word follows, and is repeated -skip_count times.*/
        uc=r_byte( f->fp );  uc2=r_byte( f->fp );
        if(feof( f->fp )){ f->errnumber=FERR_EOF;  return; }      // end-of-file occured
        size = -size;
        if(f->img_begin == 0) f->img_begin = y;
        f->img_end = y;
        for(x=0; x<size ;x++){
            f->image[i++]=uc;
            f->image[i++]=uc2;}
            }
    }/*line packets */
    }/* if packets != 0 ... */
    y++;
    if(y> f->height){
#ifdef DEBUG
    puts("on image break...(should not happen)...");
#endif
    break;}
    }   /* all lines */

    f->img_end = y;

#ifdef DEBUG
printf("...}\n");
#endif
}








void  do_dta_unbrun( FLI *f )
{
    int bpp=0;

    if(f->verbose==1)
        debug_printf(f,4,"do_dta_unbrun()");


    switch( f->depth ){
    case 15:
    case 16:    bpp=2;
    break;
    case 24:    bpp=3;
    break;
    case 32:    bpp=4;
    break;
    default:    bpp=1;
    break;
    }
    master_unbrun(f, bpp);
}
void  do_dta_copy( FLI *f ) /* unused */
{
    if(f->verbose==1)
        debug_printf(f,4,"do_dta_copy() (unused)");
}
void  do_dta_unlc( FLI *f )
{
    int bpp;

    if(f->verbose==1)
        debug_printf(f,4,"do_dta_unlc()");
    switch( f->depth ){
    case 15:
    case 16:    bpp=2;
    break;
    case 24:    bpp=3;
    break;
    case 32:    bpp=4;
    break;
    default:    bpp=1;
    break;
    }
    master_unlc( f, bpp);
}






static void master_unbrun( FLI *f, int bytes_per_pixel )
{
    int          packets, i,x,y, curr=0;
    signed short size=0;
    char         size0;
    BYTE         uc1=0,uc2=0,uc3=0,uc4=0;


    if(f->verbose==1)
        debug_printf(f,4,"master_unbrun() line stride <%d> bytes",f->stride);



    for(y=0 ; y < f->height ; y++)      /* for each line... */
    {

    packets = r_byte( f->fp );         /*    printf("packets:<%d>\n",packets);*/
    i = y * f->stride;                 /* linebytes */


    if( packets!=0 ){    /* FLI */

    /* one line*/
    while (packets-- > 0)       /*** do rle packets( one line ) ***/
    {
    size0 = (char) r_byte( f->fp );   /* packet count */
    size = (signed short) size0;
    if(feof( f->fp )){ f->errnumber=FERR_EOF;  return; }      // end-of-file occured

        if( size > 0 ){
        /* If size is positive the data consists of a single pixel which is repeated size times. */

        if(bytes_per_pixel==1){  uc1=r_byte( f->fp );                        }
        if(bytes_per_pixel==2){  uc1=r_byte( f->fp );   uc2=r_byte( f->fp ); }
        if(bytes_per_pixel==3){
            uc1=r_byte( f->fp ); uc2=r_byte( f->fp );  uc3=r_byte( f->fp );
            }
        if(bytes_per_pixel==4){
            uc1=r_byte( f->fp );   uc2=r_byte( f->fp );
            uc3=r_byte( f->fp );   uc4=r_byte( f->fp );   }


        for(x=0; x < size ;x++){
            if(bytes_per_pixel==1) f->image[i++]=uc1;
            if(bytes_per_pixel==2){
            f->image[i++]=uc1;
            f->image[i++]=uc2;
            }
            if(bytes_per_pixel==3){
            f->image[i++]=uc1;
            f->image[i++]=uc2;
            f->image[i++]=uc3;
            }
            if(bytes_per_pixel==4){
            f->image[i++]=uc1;  f->image[i++]=uc2;
            f->image[i++]=uc3;  f->image[i++]=uc4;
            }
        }

        if(feof( f->fp )){ f->errnumber=FERR_EOF;  return;}       // end-of-file occured


        }else{


        /* If size_count is negative there are -size_count pixels of data which are copied to the screen.*/
        size = -size;

        for(x=0; x < size ;x++){
            if(bytes_per_pixel==1)  f->image[ i++ ] = r_byte( f->fp );
            if(bytes_per_pixel==2){
            f->image[ i++ ] = r_byte( f->fp );
            f->image[ i++ ] = r_byte( f->fp );
            }
            if(bytes_per_pixel==3){
            f->image[ i++ ] = r_byte( f->fp );
            f->image[ i++ ] = r_byte( f->fp );
            f->image[ i++ ] = r_byte( f->fp );
            }
            if(bytes_per_pixel==4){
            f->image[ i++ ] = r_byte( f->fp );
            f->image[ i++ ] = r_byte( f->fp );
            f->image[ i++ ] = r_byte( f->fp );
            f->image[ i++ ] = r_byte( f->fp );
            }
        if(feof( f->fp )){ f->errnumber=FERR_EOF;  return;}       // end-of-file occured
        }


        }
    }   /* one line */





    }else{    /* TYPE FLC */    /*** do rle packets ***/



    curr=0;
    /* one line*/
    while ( curr < (f->stride) )  /*** do rle packets( one line ) ***/
    {

    size0 = (char) r_byte( f->fp );   /* packet count */
    size = (signed short) size0;
    if(feof( f->fp )){ f->errnumber=FERR_EOF;  return; }      // end-of-file occured
    if( size > 0 ){

    curr += size;   /* pixel counter */
        /* If size_count is positive the data consists of a single pixel which is repeated size_count times. */
            if(bytes_per_pixel==1)  uc1=r_byte( f->fp );
            if(bytes_per_pixel==2){ uc1=r_byte( f->fp );   uc2=r_byte( f->fp );   }
            if(bytes_per_pixel==3){
            uc1=r_byte( f->fp );
            uc2=r_byte( f->fp );
            uc3=r_byte( f->fp );
            }
            if(bytes_per_pixel==4){
            uc1=r_byte( f->fp );   uc2=r_byte( f->fp );
            uc3=r_byte( f->fp );   uc4=r_byte( f->fp );   }

        for(x=0; x < size ;x++){
            if(bytes_per_pixel==1) f->image[i++]=uc1;
            if(bytes_per_pixel==2){
            f->image[i++]=uc1;
            f->image[i++]=uc2;
            }
            if(bytes_per_pixel==3){
            f->image[i++]=uc1;
            f->image[i++]=uc2;
            f->image[i++]=uc3;
            }
            if(bytes_per_pixel==4){
            f->image[i++]=uc1;  f->image[i++]=uc2;
            f->image[i++]=uc3;  f->image[i++]=uc4;  }
        if(feof( f->fp )){ f->errnumber=FERR_EOF;  return; }      // end-of-file occured

        }/* for x */


    }else{
        /* If size_count is negative there are -size_count bytes of data which are copied to the screen.*/
        size = -size;
        curr += size;   /* pixel counter */


        for(x=0; x < size ;x++){
            if(bytes_per_pixel==1)  f->image[ i++ ] = r_byte( f->fp );
            if(bytes_per_pixel==2){
            f->image[ i++ ] = r_byte( f->fp );
            f->image[ i++ ] = r_byte( f->fp );
            }
            if(bytes_per_pixel==3){
            f->image[ i++ ] = r_byte( f->fp );
            f->image[ i++ ] = r_byte( f->fp );
            f->image[ i++ ] = r_byte( f->fp );
            }
            if(bytes_per_pixel==4){
            f->image[ i++ ] = r_byte( f->fp );
            f->image[ i++ ] = r_byte( f->fp );
            f->image[ i++ ] = r_byte( f->fp );
            f->image[ i++ ] = r_byte( f->fp );
            }
        if(feof( f->fp )){ f->errnumber=FERR_EOF;  return; }      // end-of-file occured
        }/*for x */



        }/* while curr < width... else...*/
    }   /* one line */
    }   /* type FLI/FLC+ */
    }   /* for each line */

}





























static void master_unlc( FLI *f, int bytes_per_pixel )
{
    int packets, i,x,y, rept;
    unsigned short linedata, skip, opcode, optype;
    signed short size;
    char size0;
    BYTE uc1=0,uc2=0,uc3=0,uc4=0,last_byte=0, oddflag;

        f->img_begin = 0;
        f->img_end   = 0;


    linedata = r_word( f->fp );    /* data lines */
    if(feof( f->fp )){ f->errnumber=FERR_EOF;  return;}       // end-of-file occured
    if(linedata==0) return;


    if(f->verbose==1)
        debug_printf(f,4,"master_unlc() linedata:<%d>\n",linedata);




    y=0;
    for(;;)      /* for each line... */
    {   /* add infinitum, exit when linedata==0 */
    i = y * f->stride;  /* linebytes */


    skip = packets = 0;
    oddflag = 0;
    for(;;)     /* handle opcodes */
    {
    opcode = r_word( f->fp );
    optype = (opcode & 0xc000);
    opcode = opcode & 0x3fff;       /* extract lower data */
    if(feof( f->fp )){ f->errnumber=FERR_EOF;  return;}       // end-of-file occured

#ifdef DEBUG
/*    printf("optype: <0x%0x>, opcode <0x%0x>...",optype, opcode);*/
#endif

    if( optype==0x0000 ){ /* packet count*/
        packets = opcode;

#ifdef DEBUG
/*        printf("-packet_count:<%d>-\n",packets);*/
#endif
        break;
        }
    if( optype==0x4000 ){
        debug_printf( f, 4, "Undefined code in delta_flc" );
        return;}
    if( optype==0x8000 ){
        if(f->img_begin == 0) f->img_begin = y;
        /* low byte of opcode in last pixel of line */
        /* width is odd */
        last_byte = (opcode & 0xff);
        oddflag = 1;
        }
    if( optype==0xc000 ){   /* line skip count, signed */
        if( (opcode&0x2000) !=0 ) opcode=0x4000-opcode; /* (*sight*)...if stupid people could fly, there would be no sunshine... */

#ifdef DEBUG
/*        printf("-skiplines abs value<%d>-\n",opcode);*/
#endif

        y = y + opcode;
        i = y * f->stride;  /* linebytes */
        }
    }

    if( oddflag==1 ){
        if(f->img_begin == 0) f->img_begin = y;
        f->img_end = y;
        f->image[ (i+f->stride-1) ] = last_byte;
    }

    if( packets==0 ){ /* if no packets, then ignore line */
    y++;              /* ignore line */
    i = y * f->stride;  /* linebytes */
    }


    if( packets != 0 ){ /* if there are packets, process them */
    if( linedata==0 ){ return;}   /* exit when all lines are processed */
    linedata--;

    skip = size = 0;
    while (packets-- > 0)       /*** do rle packets( one line ) ***/
    {
    skip = r_byte( f->fp );
    size0 = (char) r_byte( f->fp );   /* packet count */
    size = (signed short) size0;
    if(feof( f->fp )){ f->errnumber=FERR_EOF;  return; }      // end-of-file occured
    if( size==0 ){  /* special case: skip count is bigger than 255 */
        rept = skip;
        rept *= bytes_per_pixel;
        i+=rept;    /* skip pixels instead */
        skip =  r_byte( f->fp );
        size0 = (char) r_byte( f->fp );   /* packet count */
        size = (signed short) size0;
        }
        rept = skip;
        rept *= bytes_per_pixel;
        i+=rept;
    if( size > 0 ){


        /*If it is positive, that many pixels of data follow and are to be copied to the screen.*/


        if(f->img_begin == 0) f->img_begin = y;
        f->img_end = y;
        for(x=0; x < size ;x++){
        if(bytes_per_pixel==1)  f->image[ i++ ] = r_byte( f->fp );
        if(bytes_per_pixel==2){
            f->image[ i++ ] = r_byte( f->fp );
            f->image[ i++ ] = r_byte( f->fp ); }
        if(bytes_per_pixel==3){
            f->image[ i++ ] = r_byte( f->fp );
            f->image[ i++ ] = r_byte( f->fp );
            f->image[ i++ ] = r_byte( f->fp );
            }
        if(bytes_per_pixel==4){
            f->image[ i++ ] = r_byte( f->fp );
            f->image[ i++ ] = r_byte( f->fp );
            f->image[ i++ ] = r_byte( f->fp );
            f->image[ i++ ] = r_byte( f->fp ); }
        if(feof( f->fp )){ f->errnumber=FERR_EOF;  return; }      // end-of-file occured
        }/*for x */


        }else{
        /*If it's negative a single pixels follows, and is repeated -skip_count times.*/
        size = -size;

        if(bytes_per_pixel==1) uc1=r_byte( f->fp );
        if(bytes_per_pixel==2){
            uc1=r_byte( f->fp );
            uc2=r_byte( f->fp );
            }
        if(bytes_per_pixel==3){
            uc1=r_byte( f->fp );
            uc2=r_byte( f->fp );
            uc3=r_byte( f->fp );
            }
        if(bytes_per_pixel==4){
            uc1=r_byte( f->fp );
            uc2=r_byte( f->fp );
            uc3=r_byte( f->fp );
            uc4=r_byte( f->fp );
            }

        if(f->img_begin == 0) f->img_begin = y;
        f->img_end = y;
        for(x=0; x<size ;x++){
        if(bytes_per_pixel==1) f->image[i++]=uc1;
        if(bytes_per_pixel==2){
            f->image[i++]=uc1;
            f->image[i++]=uc2;
            }
        if(bytes_per_pixel==3){
            f->image[i++]=uc1;
            f->image[i++]=uc2;
            f->image[i++]=uc3;
            }
        if(bytes_per_pixel==4){
            f->image[i++]=uc1;
            f->image[i++]=uc2;
            f->image[i++]=uc3;
            f->image[i++]=uc4;
            }
        if(feof( f->fp )){ f->errnumber=FERR_EOF;  return; }      // end-of-file occured
        }
        }
    }/*line packets */
    }/* if packets != 0 ... */
    y++;
    if(y> f->height){

#ifdef DEBUG
    puts("on image break...(should not happen)...");
#endif

    break;}
    }   /* all lines */

    f->img_end = y;
}


/*          *************************************************************
 *          *************************************************************
 *          *************************************************************
 *          *************************************************************
 *
 *          WRITE    WRITE    WRITE     WRITE     WRITE    WRITE    WRITE
 *
 *          *************************************************************
 *          *************************************************************
 *          *************************************************************
 */


/* write fli header information
 * p = file handler
 * input:  f = pointer to allocated FLI header.
 * output: f pointer is returned back.
 *         NULL on error
 *
 */
FLI *write_fli_header( FLI *f )
{
    FILE *p;

    p = f->fp;
    trewind(p);
    tclearerr(p);
    w_dword(p,f->size        ); if(ferror(p)){ f->errnumber = FERR_CANTWRITEHEADER; return f; }
    w_word(p ,f->type        ); if(ferror(p)){ f->errnumber = FERR_CANTWRITEHEADER; return f; }
    w_word(p ,f->frames      ); if(ferror(p)){ f->errnumber = FERR_CANTWRITEHEADER; return f; }
    w_word(p ,f->width       ); if(ferror(p)){ f->errnumber = FERR_CANTWRITEHEADER; return f; }
    w_word(p ,f->height      ); if(ferror(p)){ f->errnumber = FERR_CANTWRITEHEADER; return f; }
    w_word(p ,f->depth       ); if(ferror(p)){ f->errnumber = FERR_CANTWRITEHEADER; return f; }
    w_word(p ,f->flags       ); if(ferror(p)){ f->errnumber = FERR_CANTWRITEHEADER; return f; }
    w_dword(p,f->speed       ); if(ferror(p)){ f->errnumber = FERR_CANTWRITEHEADER; return f; }
    w_word(p ,f->reserved1   ); if(ferror(p)){ f->errnumber = FERR_CANTWRITEHEADER; return f; }
    w_dword(p,f->created     ); if(ferror(p)){ f->errnumber = FERR_CANTWRITEHEADER; return f; }
    w_dword(p,f->creator     ); if(ferror(p)){ f->errnumber = FERR_CANTWRITEHEADER; return f; }
    w_dword(p,f->updated     ); if(ferror(p)){ f->errnumber = FERR_CANTWRITEHEADER; return f; }
    w_dword(p,f->updater     ); if(ferror(p)){ f->errnumber = FERR_CANTWRITEHEADER; return f; }
    w_word(p ,f->aspect_dx   ); if(ferror(p)){ f->errnumber = FERR_CANTWRITEHEADER; return f; }
    w_word(p ,f->aspect_dy   ); if(ferror(p)){ f->errnumber = FERR_CANTWRITEHEADER; return f; }
    w_word(p ,f->ext_flags   ); if(ferror(p)){ f->errnumber = FERR_CANTWRITEHEADER; return f; }
    w_word(p ,f->keyframes   ); if(ferror(p)){ f->errnumber = FERR_CANTWRITEHEADER; return f; }
    w_word(p ,f->totalframes ); if(ferror(p)){ f->errnumber = FERR_CANTWRITEHEADER; return f; }
    w_dword(p,f->req_memory  ); if(ferror(p)){ f->errnumber = FERR_CANTWRITEHEADER; return f; }
    w_word(p ,f->max_regions ); if(ferror(p)){ f->errnumber = FERR_CANTWRITEHEADER; return f; }
    w_word(p ,f->transp_num  ); if(ferror(p)){ f->errnumber = FERR_CANTWRITEHEADER; return f; }
    w_bytes(p,24);              if(ferror(p)){ f->errnumber = FERR_CANTWRITEHEADER; return f; }
    w_dword(p,f->oframe1     ); if(ferror(p)){ f->errnumber = FERR_CANTWRITEHEADER; return f; }
    w_dword(p,f->oframe2     ); if(ferror(p)){ f->errnumber = FERR_CANTWRITEHEADER; return f; }
    w_bytes(p,40);              if(ferror(p)){ f->errnumber = FERR_CANTWRITEHEADER; return f; }
    fflush(p);
    return f;
}





/* open a file to write
 *
 * Input:    filename
 *
 * output:   File stream pointer
 */
FILE *openflisave( char *filename, FLI *f )
{
    f->filename = filename;
    f->fp  = topen( filename,"wb" );
    if( f->fp == NULL ){ f->errnumber = FERR_CANTOPENFILEWRITE; return NULL;}
    return f->fp;
}




/* write chunk header
 * input:
 *  frame_type = only FRAME_TYPE is supported
 * output:
 * returns file position at chunk head
 */
long write_chunk_header( FLI *f, int frame_type, int subchunks  )
{
    DWORD file_position;

    file_position = ttell( f->fp );
    w_dword( f->fp, 0 );         /* Size of the chunk, including subchunks */
    w_word(  f->fp, frame_type); /* Chunk type: 0xF1FA */
    w_word(  f->fp, subchunks ); /* Number of subchunks */
    w_word(  f->fp, 0 );         /* Delay in milliseconds */
    w_dword( f->fp, 0 ); w_word( f->fp, 0 );  /* Reserved, set to 0 */
    return  file_position;
}

/*
 * trim chunk header size (any chunk or subchunk)
 * inserting padding 0 if necessary at the end
 * This function should be called when a chunk header has been written and
 * all of its data has been written.
 *
 * input:
 *     frame_type = unused...
 *       position = file position to bagin of header
 *
 *
 */
void trim_chunk_header( FLI *f, DWORD oldposition )
{
    DWORD file_position, diff;
    int odd=0;

    file_position = ttell(f->fp);  /* get current file position */

    diff = file_position - oldposition;
    odd = diff;
    odd = odd & 1;
    if(odd==1){
        diff++;                 /* add extra */
        w_byte( f->fp, diff ); /* write extra padding zero */
        file_position++;
    }

    tseek( f->fp, oldposition , SEEK_SET );
    w_dword( f->fp, diff );        /* write new file size */

    tseek( f->fp, file_position , SEEK_SET ); /* reset current file position */
    return;
}





/* returns 0 on no palette change */
int compress_palette( FLI *f)
{
    return 0;
}
/* returns subchunk type  */
int compress_image( FLI *f)
{
    return DTA_COPY;
}






void  do_fli_brun( FLI *f )
{
    int     i,x,xx,y, lineoffset;
    BYTE    data0,data1,data2,data3;
    char    packets=0,data_count=0;
    int     fileposition=0, current_fileposition=0;
    int     flag = FALSE;

    if(f->verbose==1)
        debug_printf(f,4,"do_fli_brun()");


    for(y=0 ; y < f->height ; y++){      // for each line...
    packets = 0;
    fileposition = ttell( f->fp );
    w_byte( f->fp, packets );    // unknown number of packets for this line
    lineoffset = y * f->stride;
    // scan data and compare if equal, output packets
    for(x=0;x< f->stride ; x++){
    data0 = f->image[ lineoffset + x     ];
    data1 = f->image[ lineoffset + x + 1 ];
        if( data0 == data1 ){     // let's see how many pixels are equal
            for(xx=0;xx< 127 ; xx++){
                data2 = f->image[ lineoffset + x + xx     ];
                data3 = f->image[ lineoffset + x + xx + 1 ];
            if(data2 != data3) break;
            if( ( x + xx ) >= (f->stride-1) ){ flag=TRUE; break;}     // don't scan beyond a line
            } // for xx
            // equal data until here.  xx = run length
            data_count = xx;
            // If packet count is positive the data consists of a single byte which is repeated packet count times.
            w_byte( f->fp, data_count );        // positive packet count
            w_byte( f->fp, data2 );     // data that is repeated
            packets++;
            x = x + xx;     // new x
            if(x!=0) x--;
            if(flag==TRUE){ flag=FALSE; x=f->stride; break;}
        }else{                  // let's see how many pixels are unequal
            for(xx=0;xx< 127 ; xx++){
            data2 = f->image[ lineoffset + x + xx     ];
            data3 = f->image[ lineoffset + x + xx + 1 ];
            if(data2 == data3) break;
            if( ( x + xx ) >= (f->stride-1) ){ flag=TRUE; break;}     // don't scan beyond a line
            } // for xx
            // unequal data until here.  xx = run length
            data_count = xx;
            data_count = -data_count;
            w_byte( f->fp, data_count );         // negative packet count
            for(i=0;i< xx ; i++)     // multiple data copy
                w_byte( f->fp, f->image[ lineoffset + x + i ] );
            packets++;
            x = x + xx;     // new x
            if(x!=0) x--;
            if(flag==TRUE){ flag=FALSE; x=f->stride; break;}
        }
    }   // for x
        if( f->type==TYPE_FLI ){    // only for FLI
        // ---------------------
        current_fileposition = ttell( f->fp );
        tseek( f->fp, fileposition , SEEK_SET );
        w_byte( f->fp, packets );   // number of packets for this line
        tseek( f->fp, current_fileposition, SEEK_SET );
        // ---------------------
        }
    } // for y,  all lines


}





typedef struct {
    BYTE    b,g,r;
}TBYTE __attribute__ ((packed));



DWORD mkdword( TBYTE  *pixel )
{
    DWORD   a=0,b=0,c=0;

    a = pixel->b;
    b = pixel->g;
    c = pixel->r;
    return ((c<<16) | (b<<8) | (a));
}



void  do_dta_brun( FLI *f )     // not ready yet
{
    int     i,x,xx,y,lineoffset;
    char    data_count=0;
    DWORD   *dbuffer = NULL, dpixel0, dpixel1, dpixel2=0, dpixel3;
    WORD    *wbuffer = NULL, wpixel0, wpixel1, wpixel2=0, wpixel3;
    BYTE   *bbuffer = NULL;
    TBYTE   bpixel0, bpixel1, bpixel2, bpixel3;
    BYTE    data0,data1,data2,data3;
    int     flag = FALSE;

    if(f->verbose==1)
        debug_printf(f,4,"do_dta_brun()  depth=%d", f->depth );


    dbuffer = (DWORD *) f->image;   // re-cast for 24 bpp
    wbuffer = (WORD *)  f->image;   // re-cast for 15 & 16 bpp
    bbuffer = (BYTE *)  f->image;   // re-cast for 1 bpp


    if( f->depth == 1 ){    // for depth 1 do_fli_brun() should be used instead

    for(y=0 ; y < f->height ; y++){      // for each line...

    w_byte( f->fp, 0 );             // set to zero number of packets
    lineoffset = y * f->stride;
    // scan data and compare if equal, output packets

    for(x=0;x< f->stride ; x++){
    data0 = bbuffer[ lineoffset + x     ];
    data1 = bbuffer[ lineoffset + x + 1 ];


        if( data0 == data1 ){     // let's see how many pixels are equal
            for(xx=0;xx< 127 ; xx++){
                data2 = bbuffer[ lineoffset + x + xx     ];
                data3 = bbuffer[ lineoffset + x + xx + 1 ];
            if(data2 != data3) break;
            if( ( x + xx ) >= (f->stride - 0) ){ flag=TRUE; break;}     // don't scan beyond a line
            } // for xx
            // equal data until here.  xx = run length
            data_count = xx;
            // If packet count is positive the data consists of a single byte which is repeated packet count times.
            w_byte( f->fp, data_count );        // positive packet count
            w_byte( f->fp, data2 );     // data that is repeated
            x = x + xx;     // new x
            if(x!=0) x--;
            if(flag==TRUE){ flag=FALSE; x=f->stride; break;}

        }else{                  // let's see how many pixels are unequal
            for(xx=0;xx< 127 ; xx++){
            data2 = bbuffer[ lineoffset + x + xx     ];
            data3 = bbuffer[ lineoffset + x + xx + 1 ];
            if(data2 == data3) break;
            if( ( x + xx ) >= (f->stride - 0) ){flag=TRUE; break;}     // don't scan beyond a line
            } // for xx

            // unequal data until here.  xx = run length
            data_count = xx;
            data_count = -data_count;
            w_byte( f->fp, data_count );         // negative packet count
            for(i=0;i< xx ; i++)     // multiple data copy
                w_byte( f->fp, bbuffer[ lineoffset + x + i ] );
            x = x + xx;     // new x
            if(x!=0) x--;
            if(flag==TRUE){ flag=FALSE; x=f->stride; break;}
        }

    }   // for x
    } // for y,  all lines


    }







    if( f->depth == 8 ){
allegro_message("ERROR: Reached here on bpp = 8");
    }





    if( f->depth == 15 || f->depth == 16){ // 2 bytes per pixel
    for(y=0 ; y < f->height ; y++){         // for each line...
    w_byte( f->fp, 0 );           // set zero packets
    lineoffset = y * (f->stride/2);
    // scan data and compare if equal, output packets
    for(x=0;x< f->stride ; x++){
    wpixel0 = wbuffer[ lineoffset + x     ];
    wpixel1 = wbuffer[ lineoffset + x + 1 ];
        if( wpixel0 == wpixel1 ){     // let's see how many wpixels are equal
            for(xx=0;xx< 127 ; xx++){
                wpixel2 = wbuffer[ lineoffset + x + xx     ];
                wpixel3 = wbuffer[ lineoffset + x + xx + 1 ];
            if(wpixel2 != wpixel3) break;
            if( ( x + xx ) >= (f->stride - 0) ){ flag=TRUE; break;}     // don't scan beyond a line
            } // for xx
            // equal data until here.  xx = run length
            data_count = xx;
            // If packet count is positive the data consists of a single byte which is repeated packet count times.
            w_byte( f->fp, data_count );        // positive packet count
            w_word( f->fp, wpixel2 );     // data that is repeated
            x = x + xx;     // new x
            if(x!=0) x--;
            if(flag==TRUE){ flag=FALSE; x=f->stride; break;}
        }else{                  // let's see how many wpixels are unequal
            for(xx=0;xx< 127 ; xx++){
            wpixel2 = wbuffer[ lineoffset + x + xx     ];
            wpixel3 = wbuffer[ lineoffset + x + xx + 1 ];
            if(wpixel2 == wpixel3) break;
            if( ( x + xx ) >= (f->stride - 0) ){ flag=TRUE; break;}     // don't scan beyond a line
            } // for xx
            // unequal data until here.  xx = run length
            data_count = xx;
            data_count = -data_count;
            w_byte( f->fp, data_count );         // negative packet count
            for(i=0;i< xx ; i++)     // multiple data copy
                w_word( f->fp, wbuffer[ lineoffset + x + i ] );
            x = x + xx;     // new x
            if(x!=0) x--;
            if(flag==TRUE){ flag=FALSE; x=f->stride; break;}
        }
    }   // for x
    } // for y,  all lines
    }
    // ---------------------------



    if( f->depth == 24 ){       // 3 bytes per pixel
    for(y=0 ; y < f->height*3 ; y++){         // for each line...
    w_byte( f->fp, 0 );           // set zero packets
    lineoffset = y * (f->stride/3);
    // scan data and compare if equal, output packets
    for(x=0;x< (f->stride/3) ; x++){  // note: we are counting pixels instead of bytes.
    bpixel0.b = bbuffer[ lineoffset + x*3       +0];
    bpixel0.g = bbuffer[ lineoffset + x*3       +1];
    bpixel0.r = bbuffer[ lineoffset + x*3       +2];
    bpixel1.b = bbuffer[ lineoffset + x*3 + 1*3 +0];
    bpixel1.g = bbuffer[ lineoffset + x*3 + 1*3 +1];
    bpixel1.r = bbuffer[ lineoffset + x*3 + 1*3 +2];
        if( mkdword(&bpixel0) == mkdword(&bpixel1) ){     // let's see how many bpixels are equal
            for(xx=0;xx< 127 ; xx++){
                bpixel2.b = bbuffer[ lineoffset + x*3 + xx*3       +0];
                bpixel2.g = bbuffer[ lineoffset + x*3 + xx*3       +1];
                bpixel2.r = bbuffer[ lineoffset + x*3 + xx*3       +2];
                bpixel3.b = bbuffer[ lineoffset + x*3 + xx*3 + 1*3 +0];
                bpixel3.g = bbuffer[ lineoffset + x*3 + xx*3 + 1*3 +1];
                bpixel3.r = bbuffer[ lineoffset + x*3 + xx*3 + 1*3 +2];
            if(mkdword(&bpixel2) != mkdword(&bpixel3)) break;
            if( ( x + xx ) >= (f->stride/3) ){ flag=TRUE; break;}   // don't scan beyond a line
            } // for xx
            // equal data until here.  xx = run length
            data_count = xx;
            // If packet count is positive the data consists of a single byte which is repeated packet count times.
            w_byte( f->fp, data_count );        // positive packet count
            w_byte( f->fp, bpixel2.b );     // data that is repeated
            w_byte( f->fp, bpixel2.g );     // data that is repeated
            w_byte( f->fp, bpixel2.r );     // data that is repeated
            x = x + xx;         // new x
            if(x!=0) x--;
            if(flag==TRUE){ flag=FALSE; x=f->stride; break;}
        }else{                  // let's see how many bpixels are unequal
            for(xx=0;xx< 127; xx++){
            bpixel2.b = bbuffer[ lineoffset + x*3 + xx*3       +0];
            bpixel2.g = bbuffer[ lineoffset + x*3 + xx*3       +1];
            bpixel2.r = bbuffer[ lineoffset + x*3 + xx*3       +2];
            bpixel3.b = bbuffer[ lineoffset + x*3 + xx*3 + 1*3 +0];
            bpixel3.g = bbuffer[ lineoffset + x*3 + xx*3 + 1*3 +1];
            bpixel3.r = bbuffer[ lineoffset + x*3 + xx*3 + 1*3 +2];
            if(mkdword(&bpixel2) == mkdword(&bpixel3)) break;
            if( ( x + xx ) >= (f->stride/3) ){ flag=TRUE; break;}     // don't scan beyond a line
            } // for xx
            // unequal data until here.  xx = run length
            data_count = xx;
            data_count = -data_count;
            w_byte( f->fp, data_count );    // negative packet count
            for(i=0;i< xx ; i++){           // multiple data copy
                w_byte( f->fp, bbuffer[ lineoffset + x*3 + i*3 +0] );
                w_byte( f->fp, bbuffer[ lineoffset + x*3 + i*3 +1] );
                w_byte( f->fp, bbuffer[ lineoffset + x*3 + i*3 +2] );
            }
            x = x + xx;         // new x
            if(x!=0) x--;
            if(flag==TRUE){ flag=FALSE; x=f->stride; break;}
        }
    }   // for x
    } // for y,  all lines
    }







    if( f->depth == 32 ){       // 4 bytes per pixel
    for(y=0 ; y < f->height ; y++){         // for each line...
    w_byte( f->fp, 0 );           // set zero packets
    lineoffset = y * (f->stride/4);
    // scan data and compare if equal, output packets
    for(x=0;x< (f->stride/4) ; x++){
    dpixel0 = dbuffer[ lineoffset + x     ];
    dpixel1 = dbuffer[ lineoffset + x + 1 ];
        if( dpixel0 == dpixel1 ){     // let's see how many dpixels are equal
            for(xx=0;xx< 127 ; xx++){
                dpixel2 = dbuffer[ lineoffset + x + xx     ];
                dpixel3 = dbuffer[ lineoffset + x + xx + 1 ];
            if(dpixel2 != dpixel3) break;
            if( ( x + xx ) >= f->stride/4 ){ flag=TRUE; break;}      // don't scan beyond a line
            } // for xx
            // equal data until here.  xx = run length
            data_count = xx;
            // If packet count is positive the data consists of a single byte which is repeated packet count times.
            w_byte( f->fp, data_count );        // positive packet count
            w_dword( f->fp, dpixel2 );     // data that is repeated
            x = x + xx;     // new x
            if(x!=0) x--;
            if(flag==TRUE){ flag=FALSE; x=f->stride; break;}
        }else{                  // let's see how many dpixels are unequal
            for(xx=0;xx< 127 ; xx++){
            dpixel2 = dbuffer[ lineoffset + x + xx     ];
            dpixel3 = dbuffer[ lineoffset + x + xx + 1 ];
            if(dpixel2 == dpixel3) break;
            if( ( x + xx ) >= f->stride/4 ){ flag=TRUE; break;}     // don't scan beyond a line
            } // for xx
            // unequal data until here.  xx = run length
            data_count = xx;
            data_count = -data_count;
            w_byte( f->fp, data_count );         // negative packet count
            for(i=0;i< xx ; i++)     // multiple data copy
                w_dword( f->fp, dbuffer[ lineoffset + x + i ] );
            x = x + xx;     // new x
            if(x!=0) x--;
            if(flag==TRUE){ flag=FALSE; x=f->stride; break;}
        }
    }   // for x
    } // for y,  all lines
    }

}








void  do_delta_flc( FLI *f )
{

}
void  do_dta_lc( FLI *f )
{

}



/*
 The first 16-bit word behind the chunk header is the line number 
 for the first line in the image that differs from the previous frame.
 In other words, it is a line skip count. 

 The second word is the number of lines in the chunk. 
 The data for the lines themselves follow these two words. 

Each line starts with a packet count for the line. 
Unlike the BYTE_RUN chunk, this value is significant because 
the "delta" need not to store the complete width of the frame image.

The packets themselves follow the packet count. 
Each packet consists of a column skip count, an RLE count byte and 
zero or more data bytes. 

The column skip count is the number of pixels to skip (the number of 
pixels that did not change between the current frame and the previous 
frame) from the current position in the line
It can have a value of zero.


The RLE count byte and the data bytes are similar to the BYTE_RUN encoding: 
if the count is positive, that number of bytes is to be copied from the 
data to the image (literal run); if the count is negative, its absolute 
value is the replication count of the one data byte that follows 
(replicate run). 


Two pitfalls: -------------------

The negative/positive meaning of the RLE count byte is reversed from 
that of the BYTE_RUN encoding. 

The RLE count byte can be zero. This occurs when the number of pixels to 
skip is more than 255. In this case, skip count is split over two packets. 
The first of these contains a skip count of 255, a RLE count of 0, and no 
data following the RLE count. The skip count of the second packet holds 
the remaining number of pixels to skip. 

------------------------

      word             word
[line skip count] [number of lines]

for each line:

[packet count] - [column skip count] [RLE count+-] [data bytes]



*/

void  do_delta_fli( FLI *f )        // Not REady Yet
{
    int     x,y,lineoffset=0, first_difference = -1, number_of_lines=0 ;
    int     column_skip_count=0, packet_count=0, rle_count=0;
    BYTE    data0,data1, delta;

    if(f->verbose==1)
        debug_printf(f,4,"do_delta_fli()");

    // find first word, line skip.
    for(y=0 ; y < f->height ; y++){

    // scan data and compare if unequal

    for(x=0;x< f->stride ; x++){
    data0 = f->image_prev[ lineoffset + x     ];
    data1 = f->image[ lineoffset + x + 1 ];
    delta = data0 - data1;
        if( delta != 0 ){   // first difference
        if( first_difference == -1 ) first_difference = y;  // only occurs once
        number_of_lines = 0;    // unknown so far.
        packet_count=0;         // unknown so far.
        column_skip_count = x;
        rle_count = 0;
        break;
        }

    }   // for x


    } // for y,  all lines
}

