/**************************************\
* ZEUS.C                               *
* Clone of Viacom New Media's          *
* Zoop(TM)                             *
* By Damian Yerrick                    *
\**************************************/

/* Notice

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation; either version 2 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be useful and fun, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the License for more details.

You should have received a copy of the License along with this program,
in the file COPYING; if not, write to the Free Software Foundation, Inc. /
59 Temple Place - Suite 330 / Boston, MA  02111-1307 / USA
or view the License online at http://www.gnu.org/copyleft/gpl.html

The author can be reached by
  usps:Damian Yerrick / Box 398 / 5500 Wabash Avenue / Terre Haute, IN 47803
  mailto:dosarena@pineight.8m.com
    http://come.to/yerrick

DOSArena is a trademark of Damian Yerrick. All other trademarks belong to
their respective owners.

*/

#define alleg_mouse_unused
#define alleg_3d_unused
#define alleg_joystick_unused
#define alleg_flic_unused

#include "dosarena.h"

// Constants /////////////////////////////////////////////////////////////////

enum {
  FACING_RIGHT = 0, FACING_UP, FACING_LEFT, FACING_DOWN,
  COLOR_PURPLE = 0, COLOR_ORANGE, COLOR_GREEN, COLOR_BLUE,
    COLOR_SPRING, COLOR_BOMB, COLOR_GEAR, COLOR_SPLOTCH,
  STATE_INACTIVE = 0, STATE_MOVING, STATE_SHOOTING, STATE_RETURNING
};

// Data Types ////////////////////////////////////////////////////////////////

typedef struct ZeusGlobals
{
  DATAFILE *pix, *wav;
  BITMAP   *bkgnd;
  BITMAP   *sprites;
  BITMAP   *dest;
  FONT     *lcd;

  unsigned char grid[14][18];

  short          left2Zoop;     // left to clear in this level
  unsigned short zooped;        // cleared
  int         score;
  int         newPieceClock;    // when a new piece appears
  int         thouClock;        // when the thousands disappear
  int         gameStart;        // when the game was started
  short         yBase;          // base of screen
  unsigned char thousands;      // thousand to redraw
  unsigned char newPiece;       // new piece to redraw
  unsigned char warning;        // is the game currently in warning?
  unsigned char level;          // current game level
  unsigned char nSprings;       // number of springs held

} ZeusGlobals;

typedef struct Player
{
  short returnX, returnY; // in 8.8
  short x, y;
  short destX, destY;

  unsigned char facing;         // direction facing
  unsigned char destFacing;     // direction will face when it returns
  unsigned char color;          // color of block
  unsigned char returnColor;    // color of last block cleared

  unsigned char state;          // state of player
  unsigned char column;         // column
  unsigned char scoreFac;       // score multiplier
  unsigned char want2Shoot;     // player wants to shoot when it gets a chance
  unsigned char destTime;       // how long it will go before it stops
} Player;

typedef struct Column
{
  char baseX, baseY, dx, dy;
  char length;
  char r[3];
} Column;

// Globals ///////////////////////////////////////////////////////////////////

static const Column gCol[16] =
{
  // right side
  {17,  5, -1,  0, 7},
  {17,  6, -1,  0, 7},
  {17,  7, -1,  0, 7},
  {17,  8, -1,  0, 7},
  // top
  { 7,  0,  0,  1, 5},
  { 8,  0,  0,  1, 5},
  { 9,  0,  0,  1, 5},
  {10,  0,  0,  1, 5},
  // left side
  { 0,  5,  1,  0, 7},
  { 0,  6,  1,  0, 7},
  { 0,  7,  1,  0, 7},
  { 0,  8,  1,  0, 7},
  // bottom
  { 7, 13,  0, -1, 5},
  { 8, 13,  0, -1, 5},
  { 9, 13,  0, -1, 5},
  {10, 13,  0, -1, 5}
};
static const char leftCol[16] =
{7, 0, 1, 2, 8, 4, 5, 6, 9, 10, 11, 12, 13, 14, 15, 3};
static const char rightCol[16] =
{1, 2, 3, 15, 5, 6, 7, 0, 4, 8, 9, 10, 11, 12, 13, 14};

static const SoundRec gSplotchSound[] =
{
  {65, 1}, {69, 1}, {72, 1}, {77, 1},
  {67, 1}, {71, 1}, {74, 1}, {79, 1},
  {69, 1}, {73, 1}, {76, 1}, {81, 1}, {255, 255}
};

static RGB winBar[8] =
{
  { 0,  0, 63},
  {10, 10, 63},
  {20, 20, 63},
  {30, 30, 63},
  {40, 40, 63},
  {30, 30, 63},
  {20, 20, 63},
  {10, 10, 63}
};

static ZeusGlobals g;
static Player p[2];
static unsigned char curTurn = 0;
#define ME p[curTurn]

////////////////

void ZeusPlaySound(const char *name, short vol)
{
  SAMPLE *wav;
  char str1[32];
  short pan;

  if(g.wav)
  {
    strcpy(str1, name);
    strcat(str1, "_WAV");

    wav = GetResource(g.wav, str1);
    if(vol == 255) // only warning uses this exact volume
      pan = gCol[g.newPiece].baseX * 256 / 18;
    else
      pan = ME.x / 18;

    if(wav)
      play_sample(wav, vol, pan, 1000, 0);
  }
}

static void NewPlayer(void)
{
  ME.destX = ME.x = ME.returnX = 7 << 8;
  ME.destY = ME.y = ME.returnY = 7 << 8;
  ME.destFacing = ME.facing = FACING_UP;
  ME.color = COLOR_BLUE;
  ME.state = STATE_MOVING;
}

static void NewGame(void)
{
  short x, y;

  set_palette(pal);

  curTurn = 0;
  NewPlayer();
  p[1].state = STATE_INACTIVE;
  g.newPieceClock = retrace_count;
  g.score = g.zooped = g.nSprings = g.thousands = g.thouClock = 0;

  // trigger an immediate NewLevel
  g.level = g.left2Zoop = 0;

  // clear out the world
  for(y = 0; y < 14; y++)
    for(x = 0; x < 18; x++)
      g.grid[y][x] = 0;
  g.gameStart = retrace_count;
}

static void DrawMeters(void)
{
  short i;

  text_mode(-1);

  // Draw the score.
  textprintf(g.dest, g.lcd, 0, 3, -1, "%07d", g.score); // draw numbers
  blit(g.dest, screen, 0, 3, 16, 3 + g.yBase, 112, 25);

  // Draw the left2Zoop.
  if(g.left2Zoop < 0)
    g.left2Zoop = 0;
  textprintf(g.dest, g.lcd, 224, 3, -1, "%04d", g.left2Zoop); // draw numbers
  blit(g.dest, screen, 224, 3, 240, 3 + g.yBase, 64, 25);

  // Draw the nSprings.
  blit(g.bkgnd, g.dest, 16, 168, 16, 168, 80, 16);
  i = g.nSprings;
  while(i--)
    masked_blit(g.sprites, g.dest, 0, 56, 16 + 16 * i, 168, 16, 14);
  blit(g.dest, screen, 16, 168, 32, 168 + g.yBase, 80, 16);

  // Draw the level.
  textprintf(g.dest, g.lcd, 256, 168, -1, "%02d", g.level); // draw numbers
  blit(g.dest, screen, 256, 168, 272, 168 + g.yBase, 32, 25);
}

static void DrawPlayer(void)
{
//  static const char directions[] = "ENWS";

  int playerX = ((int)p[0].x * 16) >> 8;
  int playerY = ((int)p[0].y * 14) >> 8;

  blit(g.bkgnd, g.dest, 112, 70, 112, 70, 64, 56);

  if(p[0].state != STATE_INACTIVE)
  {
    if(p[0].color < 4)
      masked_blit(g.sprites, g.dest, (int)p[0].facing * 16, (int)p[0].color * 14,
                  playerX, playerY, 16, 14);
    else
      masked_blit(g.sprites, g.dest, (int)(p[0].color - 4) * 16, 56,
                  playerX, playerY, 16, 14);
  }

  playerX = ((int)p[1].x * 16) >> 8;
  playerY = ((int)p[1].y * 14) >> 8;

  if(p[1].state != STATE_INACTIVE)
  {
    if(p[1].color < 4)
      masked_blit(g.sprites, g.dest, (int)p[1].facing * 16, (int)p[1].color * 14,
                  playerX, playerY, 16, 14);
    else
      masked_blit(g.sprites, g.dest, (int)(p[1].color - 4) * 16, 56,
                  playerX, playerY, 16, 14);
  }

  if(g.thousands)
  {
    char str1[16];
    short strwid;

    sprintf(str1, "%d000", (int)g.thousands);
    strwid = text_length(chicago, str1);
    textout(g.dest, chicago, str1, 145 - strwid / 2, 71, 16);
    textout(g.dest, chicago, str1, 144 - strwid / 2, 70, 15);
  }

  blit(g.dest, screen, 112, 70, 128, 70 + g.yBase, 64, 56);
}

static void EraseCol(short col)
{
  short left, top, width, height;

  left = gCol[col].baseX;
  top  = gCol[col].baseY;
  width  = gCol[col].length * gCol[col].dx;
  height = gCol[col].length * gCol[col].dy;

  if(width < 0)
  {
    left += width + 1; // for example, left 17 width -7 -> left 11 width 7
    width = -width;
  }
  else if(width == 0)
    width = 1;

  if(height < 0)
  {
    top += height + 1;
    height = -height;
  }
  else if(height == 0)
    height = 1;

  blit(g.bkgnd, g.dest, left * 16, top * 14, left * 16, top * 14,
       width * 16, height * 14);
}

static void DrawCol(short col)
{
  short x, y, i, block;

  x = gCol[col].baseX;
  y = gCol[col].baseY;
  i = gCol[col].length;

  do {
    switch((block = g.grid[y][x]) & 0x0c)
    {
      case 4: // Regular blocks
        masked_blit(g.sprites, g.dest, 64 + (col >> 2) * 16, (block - 4) * 14,
                    x * 16, y * 14, 16, 14);
      case 8: // Extra blocks
        masked_blit(g.sprites, g.dest, (block - 8) * 16, 56,
                    x * 16, y * 14, 16, 14);
    }
    x += gCol[col].dx;
    y += gCol[col].dy;
  } while(i--); // do i + 1 times
}

static void BlitCol(short col)
{
  short left, top, width, height;

  left = gCol[col].baseX;
  top  = gCol[col].baseY;
  width  = gCol[col].length * gCol[col].dx;
  height = gCol[col].length * gCol[col].dy;

  if(width < 0)
  {
    left += width; // for example, left 17 width -7 -> left 10 width 7
    width = -width;
  }
  if(height < 0)
  {
    top += height;
    height = -height;
  }

  blit(g.dest, screen, left * 16, top * 14, 16 + left * 16, g.yBase + top * 14,
       (width + 1) * 16, (height + 1) * 14);
}

static void RedrawAll(void)
{
  short i;

  blit(g.bkgnd, g.dest, 0, 0, 0, 0, 288, 196);

  DrawMeters();
  DrawPlayer();
  for(i = 0; i < 16; i++)
    DrawCol(i);

  blit(g.dest, screen, 0, 0, 16, g.yBase, 288, 196);
}

static void NewLevel(void)
{
  BITMAP *bmp = NULL;
  char filename[16];

  if(g.level)
    ZeusPlaySound("NEWLEVEL", 253);
  else
    ZeusPlaySound("IGOTTAGO", 224);

  g.level++;

  // change the music by levels
  if(jukeboxMode < 0)
    switch(g.level % 7)
    {
      case 1:
        play_midi(GetResource(music_dat, "DOSTHEME_MID"), TRUE);
        break;
      case 3:
        play_midi(GetResource(music_dat, "ZEUS_02_MID"), TRUE);
        break;
      case 4:
        play_midi(GetResource(music_dat, "SNAZZY_MID"), TRUE);
        break;
      case 6:
        play_midi(GetResource(music_dat, "DEF-CONL_MID"), TRUE);
        break;
      case 0:
        play_midi(GetResource(music_dat, "ACTION3_MID"), TRUE);
        break;
    }

  clear(g.bkgnd);

  // get the new level background
  sprintf(filename, "BG%02d_PCX", (int)g.level);
  bmp = GetResource(g.pix, filename);
  if(bmp == NULL)
    bmp = GetResource(g.pix, "BG10_PCX");
  if(bmp != NULL)
    blit(bmp, g.bkgnd, 0, 0, 0, 0, bmp->w, bmp->h);
  rect(g.bkgnd, 112, 70, 175, 125, 8);

  // this is Zoop's formula for determining the speed and length of a level
  g.left2Zoop = (g.level < 3) ? (20 * g.level + 20) : (10 * g.level + 60);

  RedrawAll();
}

/* CheckWarning()
 * Checks if a column is full.
 */
char CheckWarning(short col)
{
  short x = gCol[col].baseX;
  short y = gCol[col].baseY;
  short n = gCol[col].length;

  do {
    if(g.grid[y][x] == 0)
      return 0;
    x += gCol[col].dx;
    y += gCol[col].dy;
  } while(--n);

  return 1;
}

static char GetNewPiece(void)
{
  short x, y, color, i;
  char done = 0;

  // Don't put a new piece right in front of an approaching player.
  for(i = 0; i < 2; i++)
    if(g.newPiece == p[i].column && p[i].state == STATE_SHOOTING)
      return 0;

  g.newPieceClock = retrace_count + 420 / (g.level + 4);
  i = g.newPiece & 3;
  color = rand() & 0x7f;

  switch(g.newPiece >> 2)
  {
    case FACING_RIGHT:
      rectfill(screen, 304, 70 + g.yBase, 305, 125 + g.yBase, 0);
      y = i + 5;
      x = 17;

      // Find the first hole.
      while(g.grid[y][x])
        x--;
      if(x < 11) // If a block falls in...
      {
        if(g.grid[y][11] >= 8) // If a bonus block falls in, kill it.
          x++;
        else
          done = 1;          // game over
      }

      // Move this row to the right one space.
      i = x;
      while(i < 17)
      {
        g.grid[y][i] = g.grid[y][i + 1];
        i++;
      }

      // place new piece
      if((color & 0x40) && g.grid[y][16] >= 4 && g.grid[y][16] < 8)
        g.grid[y][17] = g.grid[y][16];
      else
      {
        color &= 0x3f;
        if(color < 4)
          g.grid[y][17] = color | 8;
        else
          g.grid[y][17] = (color & 3) | 4;
      }

      break;

    case FACING_UP:
      rectfill(screen, 128, g.yBase - 2, 191, g.yBase - 1, 0);
      x = i + 7;
      y = 0;

      // Find the first hole.
      while(g.grid[y][x])
        y++;
      if(y > 4) // If a block falls in...
      {
        if(g.grid[4][x] >= 8) // If a bonus block falls in, kill it.
          y--;
        else
          done = 1;          // game over
      }

      // Move this row up one space.
      i = y;
      while(i > 0)
      {
        g.grid[i][x] = g.grid[i - 1][x];
        i--;
      }

      // place new piece
      if((color & 0x40) && g.grid[1][x] >= 4 && g.grid[1][x] < 8)
        g.grid[0][x] = g.grid[1][x];
      else
      {
        color &= 0x3f;
        if(color < 4)
          g.grid[0][x] = color | 8;
        else
          g.grid[0][x] = (color & 3) | 4;
      }

      break;

    case FACING_LEFT:
      rectfill(screen, 14, 70 + g.yBase, 15, 125 + g.yBase, 0);
      y = i + 5;
      x = 0;

      // Find the first hole.
      while(g.grid[y][x])
        x++;
      if(x > 6) // If a block falls in...
      {
        if(g.grid[y][6] >= 8) // If a bonus block falls in, kill it.
          x--;
        else
          done = 1;          // game over
      }

      // Move this row to the left one space.
      i = x;
      while(i > 0)
      {
        g.grid[y][i] = g.grid[y][i - 1];
        i--;
      }

      // place new piece
      if((color & 0x40) && g.grid[y][1] >= 4 && g.grid[y][1] < 8)
        g.grid[y][0] = g.grid[y][1];
      else
      {
        color &= 0x3f;
        if(color < 4)
          g.grid[y][0] = color | 8;
        else
          g.grid[y][0] = (color & 3) | 4;
      }

      break;

    case FACING_DOWN:
      rectfill(screen, 128, 196 + g.yBase, 191, 197 + g.yBase, 0);
      x = i + 7;
      y = 13;

      // Find the first hole.
      while(g.grid[y][x])
        y--;
      if(y < 9) // If a block falls in...
      {
        if(g.grid[9][x] >= 8) // If a bonus block falls in, kill it.
          y++;
        else
          done = 1;          // game over
      }

      // Move this row up one space.
      i = y;
      while(i < 13)
      {
        g.grid[i][x] = g.grid[i + 1][x];
        i++;
      }

      // place new piece
      if((color & 0x40) && g.grid[12][x] >= 4 && g.grid[12][x] < 8)
        g.grid[13][x] = g.grid[12][x];
      else
      {
        color &= 0x3f;
        if(color < 4)
          g.grid[13][x] = color | 8;
        else
          g.grid[13][x] = (color & 3) | 4;
      }

      break;
  }

  // Draw the changed column.
  EraseCol(g.newPiece);
  DrawCol(g.newPiece);
  BlitCol(g.newPiece);

  if(CheckWarning(g.newPiece))
  {
    ZeusPlaySound("WARNING", 255);
    g.warning = 1;
  }

  // find a spot for the next piece
  g.newPiece = ((g.newPiece & 0x0c) + 4 + (rand() % 12)) & 0x0f;
  i = g.newPiece & 3;

  // draw new indicator
  switch(g.newPiece >> 2)
  {
    case FACING_RIGHT:
      rectfill(screen, 304, 70 + g.yBase + 14 * i, 305, 83 + g.yBase + 14 * i, 15);
      break;
    case FACING_UP:
      rectfill(screen, 128 + 16 * i, g.yBase - 2, 143 + 16 * i, g.yBase - 1, 15);
      break;
    case FACING_LEFT:
      rectfill(screen, 14, 70 + g.yBase + 14 * i, 15, 83 + g.yBase + 14 * i, 15);
      break;
    case FACING_DOWN:
      rectfill(screen, 128 + 16 * i, 196 + g.yBase, 143 + 16 * i, 197 + g.yBase, 15);
      break;
  }

  return done;
}

void AnimateColors(void)
{
  short i;
  short delta;

  switch(g.level)
  {
    case 10: // sky background
      delta = (retrace_count >> 3);
      for(i = 32; i < 40; i++)
      {
        set_color(i, &winBar[(i + delta) & 7]);
      }
  }

  if(g.warning)
  {
    RGB warnColor = {0, 0, 0};

    g.warning = 0;
    for(i = 0; i < 16; i++)
      if(CheckWarning(i))
      {
        g.warning = 1;
        i = 16;
      }

    if(g.warning)
    {
      i = abs((retrace_count & 0x1f) - 0x10);
      if(i >= 0x10)
        i = 0x0f;
      warnColor.r = i << 2;
    }
    set_color(0, &warnColor);
  }

}

/* PlayerMovement() ********************
 * Handle player control.
 */

static int PlayerMovement(void)
{
  int keys;
  static int lastKeys[2] = {0x0e, 0x0e};

  // read the keyboard and the joystick
  keys = ReadKPad(curTurn + 1 - gLaptopControl) | ReadJPad(curTurn);

  if(ME.state == STATE_INACTIVE)
  {
    if((keys & 0x0c) == 0x0c)
      NewPlayer();
    return 0;
  }

  if(keys & ~lastKeys[curTurn] & 0x0e)
  {
    ME.want2Shoot = 1;
    lastKeys[curTurn] = keys;
    return keys;
  }

  if(ME.state == STATE_SHOOTING)
  {
    lastKeys[curTurn] = keys;
    return keys;
  }

  if(keys & 0x80) // move up
  {
    ME.destFacing = 1;
    if(ME.destTime == 0 && ME.state == STATE_MOVING && ME.destY > 0x500)
    {
      ME.destY -= 0x100;
      ME.destTime = 8;
    }
  }

  if(keys & 0x40) // move down
  {
    ME.destFacing = 3;
    if(ME.destTime == 0 && ME.state == STATE_MOVING && ME.destY < 0x800)
    {
      ME.destY += 0x100;
      ME.destTime = 8;
    }
  }

  if(keys & 0x20) // move left
  {
    ME.destFacing = 2;
    if(ME.destTime == 0 && ME.state == STATE_MOVING && ME.destX > 0x700)
    {
      ME.destX -= 0x100;
      ME.destTime = 8;
    }
  }

  if(keys & 0x10) // move right
  {
    ME.destFacing = 0;
    if(ME.destTime == 0 && ME.state == STATE_MOVING && ME.destX < 0xa00)
    {
      ME.destX += 0x100;
      ME.destTime = 8;
    }
  }

  lastKeys[curTurn] = keys;
  return keys;
}

static void GameLoop(void)
{
  short kee;
  char done = 0;
  int lastClock = retrace_count;

  while(done == 0)
  {
    kee = PlayerMovement(); // handle keyboard

    if(kee & 0x100)
    {
      lastClock -= retrace_count;
      g.newPieceClock -= retrace_count;
      fade_out_range(8, 17, 254);
      if(DA_Pause())
      {
        ME.state = STATE_INACTIVE;
        if(p[1 - curTurn].state == STATE_INACTIVE)
          done = 1;
      }
      while(ReadKPad(curTurn) & 0x100);
      fade_in_range(pal, 8, 17, 254);
      lastClock += retrace_count;
      g.newPieceClock += retrace_count;
    }

    switch(ME.state)
    {
      case STATE_MOVING:
        ME.facing = ME.destFacing;
        if(ME.want2Shoot)
        {
          ME.returnX = ME.x = ME.destX;
          ME.returnY = ME.y = ME.destY;
          ME.destFacing ^= 2; // turn backwards when player comes back
          ME.state = STATE_SHOOTING;
          ME.want2Shoot = ME.destTime = ME.returnColor = ME.scoreFac = 0;

          // find column
          if(ME.facing & 1) // facing up or down?
            ME.column = (ME.facing << 2) + (ME.x >> 8) - 7;
          else // facing left or right?
            ME.column = (ME.facing << 2) + (ME.y >> 8) - 5;

          // play shoot sound
          if(digiPresent)
            play_sample(GetResource(g.wav, "SHOOT_WAV"), 160, 128,
                        (ME.color > COLOR_SPRING) ? 1500 : 1000, 0);
        }
        else if(ME.destTime)
        {
          ME.x += (int)(ME.destX - ME.x) / (int)ME.destTime;
          ME.y += (int)(ME.destY - ME.y) / (int)ME.destTime;
          ME.destTime--;
        }

// Handle cheats, etc. ///////////////////////////////////////////////////////
        if(keypressed())
        {
          kee = readkey();
        }

        DrawPlayer();
        break;

// Handle shooting ///////////////////////////////////////////////////////////
      case STATE_SHOOTING:
      {
        // slow down the player during debugging
//        lastClock += 20;

        // if player hit the wall, bounce
        if((ME.x <= 0) || (ME.x >= 17 << 8) ||
           (ME.y <= 0) || (ME.y >= 13 << 8))
        {
          ME.state = STATE_RETURNING;
        }
        else
        {
          short x1, y1;

          ME.x -= (int)gCol[ME.column].dx << 8;
          ME.y -= (int)gCol[ME.column].dy << 8;
          x1 = ME.x >> 8;
          y1 = ME.y >> 8;

          // skip air
          if(!g.grid[y1][x1])
            ;
          else

          // handle springs
          if(g.grid[y1][x1] == 8)
          {
            g.grid[y1][x1] = 0;
            g.nSprings++;
            g.score += 100;
            g.left2Zoop--;
            g.zooped++;
            ZeusPlaySound("SPRING", 253);
          } else

          // handle proxbomb use
          if(ME.color == COLOR_BOMB)
          {
            short x2, y2;

            ME.returnColor = g.grid[y1][x1];
            for(y2 = y1 - 1; y2 <= y1 + 1; y2++)
              for(x2 = x1 - 1; x2 <= x1 + 1; x2++)
                if(x2 >= 0 && x2 <= 17 && y2 >= 0 && y2 <= 17 &&
                   g.grid[y2][x2] != 0)
                {
                  if(g.grid[y2][x2] == 8)
                  {
                    g.nSprings++;
                    ZeusPlaySound("SPRING", 253);
                  }
                  else
                    ME.scoreFac = 1;
                  g.grid[y2][x2] = 0;
                  g.score += 100;
                  g.left2Zoop--;
                  g.zooped++;
                }
            ME.state = STATE_RETURNING;

            // Possibly changed these columns on the board, so redraw them.
            EraseCol(leftCol[ME.column]);
            DrawCol(leftCol[ME.column]);
            BlitCol(leftCol[ME.column]);
            EraseCol(rightCol[ME.column]);
            DrawCol(rightCol[ME.column]);
            BlitCol(rightCol[ME.column]);
          } else

          // handle color bomb use
          if(ME.color == COLOR_SPLOTCH)
          {
            short nCol = 4, col, x, y, i;

            ME.returnColor = g.grid[y1][x1];
            ME.state = STATE_RETURNING;
            col = ME.column & 0x0c;
            do {
              x = gCol[col].baseX;
              y = gCol[col].baseY;
              i = gCol[col].length;
              do {
                if(ME.returnColor == g.grid[y][x])
                {
                  g.grid[y][x] = 0;
                  g.score += 100;
                  g.left2Zoop--;
                  g.zooped++;
                  ME.scoreFac = 1;
                }
                x += gCol[col].dx;
                y += gCol[col].dy;
              } while(--i);
              EraseCol(col);
              DrawCol(col);
              BlitCol(col);
              col++;
            } while(--nCol);
          } else

          // line bomb and normal clear
          if((ME.color == COLOR_GEAR && g.grid[y1][x1] != 0) ||
             (ME.color < 4 && g.grid[y1][x1] - 4 == ME.color))
          {
            if(ME.returnColor != g.grid[y1][x1])
              ME.scoreFac = 0;
            ME.scoreFac++;
            ME.returnColor = g.grid[y1][x1];
            g.grid[y1][x1] = 0;
            g.score += 100 * (int)ME.scoreFac;
            g.left2Zoop--;
            g.zooped++;
          } else

          // If nothing has cleared, bounce.
          if(g.grid[y1][x1])
          {
            ME.returnColor = g.grid[y1][x1];
            g.grid[y1][x1] = ME.color + 4;
            ME.state = STATE_RETURNING;
          }
        }

// Handle the start of the return stroke /////////////////////////////////////
        if(ME.state == STATE_RETURNING)
        {
          if(ME.returnColor)
          {
            int pitch, now = GetNow();

            ME.color = ME.returnColor - 4;
            switch(ME.color)
            {
              case COLOR_BOMB:
                ZeusPlaySound("BOMB", 224);
                break;
              case COLOR_GEAR:
                for(pitch = 0; pitch < 12; pitch += 2)
                {
                  Note(1, 72 + pitch, 125 - 4 * pitch, now++, 2);
                  Note(1, 77 + pitch, 125 - 4 * pitch, now++, 2);
                }
                break;
              case COLOR_SPLOTCH:
                SendSound(gSplotchSound);
                break;
            }
          }
          ME.facing ^= 2;
          if(ME.scoreFac)
            ZeusPlaySound("POP", 253);

          // If player cleared a whole line of the same color...
          if(ME.scoreFac == gCol[ME.column].length)
          {
            if(gCol[ME.column].length == 5)
            {
              ZeusPlaySound("FIVE", 253);
              g.score += 5000;
              g.thousands = 5;
            }
            else
            {
              ZeusPlaySound("TEN", 253);
              g.score += 10000;
              g.thousands = 10;
            }
            g.thouClock = lastClock + 40;
          }
        }

        DrawMeters();
        EraseCol(ME.column);
        DrawCol(ME.column);
        DrawPlayer();
        BlitCol(ME.column);
      } break;

// Handle the return stroke //////////////////////////////////////////////////
      case STATE_RETURNING:
        ME.x += (int)gCol[ME.column].dx << 8;
        ME.y += (int)gCol[ME.column].dy << 8;

        if(ME.x == ME.returnX && ME.y == ME.returnY)
          ME.state = STATE_MOVING;

        EraseCol(ME.column);
        DrawCol(ME.column);
        DrawPlayer();
        BlitCol(ME.column);
    }

    curTurn = !curTurn;
    if(curTurn == 0)
    {
// Handle stuff that should only take place on Player 1's turn to avoid nasty
// bugs.
      while(lastClock - retrace_count > 0)
        yield_timeslice();
      lastClock++;
      AnimateColors();

// play THOUSANDS sound
      if(g.left2Zoop <= 0 &&
         p[0].state <= STATE_MOVING && p[1].state <= STATE_MOVING)
        NewLevel();

      if(g.thouClock != 0 && (lastClock - g.thouClock > 0))
      {
        ZeusPlaySound("THOUSAND", 253);
        g.thouClock = 0;
        g.thousands = 0;
      }

// Handle five springs clears the screen
      if(g.nSprings >= 5 &&
        p[0].state <= STATE_MOVING && p[1].state <= STATE_MOVING)
      {
        short x1, y1;

        g.nSprings = 0;
        for(y1 = 0; y1 < 14; y1++)
          for(x1 = 0; x1 < 18; x1++)
            if(g.grid[y1][x1] != 0)
            {
              g.grid[y1][x1] = 0;
              g.left2Zoop--;
              g.zooped++;
              g.score += 100;
            }
        lastClock += 70;
        RedrawAll();
      }

// Make a new piece
      if(g.newPieceClock - lastClock <= 0)
        done = GetNewPiece();
    }
  }
}

static void GameOver(void)
{
  HighScore entry =
  {
    {
      g.score >> 16,
      g.score & 0xffff,
      g.zooped,
      (retrace_count - g.gameStart) / 70
    },
    ""
  };

  text_mode(255);
  ZeusPlaySound("OHSH", 253);
  rest(1000);
  textout(screen, chicago, " end of game ", 108, 90 + g.yBase, 15);
  fade_out_range(2, 16, 255);

  rest(1000);

  // ask for the player's name if he or she has the high score
  clear(screen);
  set_palette(pal);
  clear_keybuf();
  AddHighScore(SCORE_ZEUS, &entry);
  ShowHighScores(SCORE_ZEUS);
  textout_centre(screen, small, "Press a key", 160, 86 + g.yBase, 4);

  // clean up
  readkey();
  fade_out(4);
}

static void ZeusExit(void)
{
  if(g.pix)
    unload_datafile(g.pix);
  if(g.wav)
    unload_datafile(g.wav);
  if(g.dest)
    destroy_bitmap(g.dest);
  if(g.bkgnd)
    destroy_bitmap(g.bkgnd);
}

static short ZeusInit(void)
{
  // setup therm bar
  clear(screen);
  set_palette(pal);
  textout(screen, font, "Loading...              (so get over it)",
          0, 148, 15); // after Skullmonkeys and Oddworld
  rectfill(screen, 0, 156, 319, 163, 2);

  // load pictures
  g.pix = load_skinned_datafile("zeus.dat");
  if(g.pix == NULL)
  {
    textout(screen, chicago, "This skin has no zeus.dat", 0, 0, 15);
    return 1;
  }
  rectfill(screen, 0, 156, 125, 163, 14);

  g.sprites = GetResource(g.pix, "sprites_pcx");
  g.lcd = GetResource(dosarena_dat, "lcdno_pcx");
  if(g.sprites == NULL || GetResource(g.pix, "bg10_pcx") == NULL)
  {
    textout(screen, chicago, "zeus.dat is missing something", 0, 0, 15);
    return 1;
  }

  rectfill(screen, 126, 156, 127, 163, 14);

  if(digiPresent)
    g.wav = load_skinned_datafile("zeuswav.dat");
  else
    g.wav = NULL;

  rectfill(screen, 128, 156, 255, 163, 14);

  g.bkgnd = create_bitmap(288, 196);
  g.dest = create_bitmap(288, 196);
  if(g.bkgnd == NULL || g.dest == NULL)
  {
    clear(screen);
    textout(screen, chicago, "Not enough RAM", 0, 0, 15);
    return 1;
  }

  set_palette(black_palette);

  if(jukeboxMode >= 0)
    play_midi(music_dat[jukeboxMode].dat, TRUE);

  g.yBase = (SCREEN_H - 196) / 2;

  return 0;
}

static char ZeusTitle(void)
{
  int gotkey = 0;

  clear(screen);

  text_mode(0);
  textout(screen, font, "ZEUS 1.0", 0, 0, 15);
  textout(screen, font, "For 1 or 2 players", 0, 16, 15);
  textout(screen, font, "Player 2 can join in with both flip keys", 0, 28, 15);
  textout(screen, font, "See Options for key bindings.", 0, 36, 15);
  textout(screen, font, "Shoot at pieces of your color to clear", 0, 48, 15);
  textout(screen, font, "them; shoot at a piece of a different", 0, 56, 15);
  textout(screen, font, "color to swap with it.", 0, 64, 15);
  textout(screen, font, "Press Enter to play or Esc to exit", 0, 192, 15);
  fade_in(pal, 2);

  while (gotkey != 13 && gotkey != 27)
  {
    gotkey = readkey() & 0xff;
  }

  clear(screen);

  return (gotkey == 27);
}

void ZeusMain(void)
{
  if(ZeusInit())
  {
    ZeusExit();
    readkey();
    return;
  }

  InitPlayEvent(136);
  while(ZeusTitle() == 0)
  {
    NewGame();
    GameLoop();
    GameOver();
  }
  remove_event();

  ZeusExit();
}

