/**************************************\
* TILEWLD.H                            *
* Tile-based scrolling engine for      *
* DJGPP and Allegro                    *
* By Damian Yerrick                    *
\**************************************/

// makes sure this file is only #included once
#ifndef _TILEWLD_H
#define _TILEWLD_H

#include <stdio.h>
#include <stdlib.h>
#include <time.h>

// bits in tile flags that tell what a certain tile is doing
enum {
  kTileDirty = 1, // has been changed or drawn over
  kTileSeen = 2
};

// rectangle from Mac OS QuickDraw
typedef struct Rect
{
  short top, left, bottom, right;
} Rect;

// tile world
typedef struct TileWorld
{
  BITMAP *tileImg;   // the pictures of the tiles
  BITMAP *tileDest;  // buffer containing the current tile background
  BITMAP *tileMap;   // which tile is in which space
  BITMAP *tileFlags; // should a tile be redrawn?
  // variables used internally for scrolling.  Just make sure to set these
  // all to 0 when the game starts.
  short xPixOffset, yPixOffset, xTileOffset, yTileOffset;
  short xMove, yMove;
  short xScrT, yScrT; // number of tiles on screen
  unsigned short xTileSize, yTileSize; // size of individual tiles
} TileWorld;

// Rectangle used for animation of sprites
typedef struct SelRect
{
  Rect r;           // the pixels
  short hotX, hotY; // hot spots on the frames of one animation line up
  short gunX, gunY; // where the hand, gun, etc. fits in
} SelRect;

// Control structure for a bitmap-based sprite (moving object)
typedef struct Sprite
{
  // TileWorld uses only these fields.
  BITMAP  *pic;
  SelRect *sel;
  short    x, y, frame;

  // Collision management systems may use these fields.
  short    angle;      // facing angle
  long     xLoc, yLoc; // subpixel movement (1/65536 tile from top left)
  long     xVel, yVel; // velocity (tiles per 65536 seconds)
} Sprite;

/* You can add your own fields in a wrapper structure.  For example:

typedef struct Actor
{
  Sprite base;
  short  aiType, aiState;
  long   stateTime;
  short  health, armor;
} Actor;

*/

// used in tile-sprite collision detection
extern unsigned char gSolidTiles[256];

// used in speed compensation
extern int gSlowness;

// set tiles under a pixel rectangle dirty
void SetDirty(TileWorld *tw, short left, short top, short right, short bottom);
// draw a tile
void DrawTile(TileWorld *tw, short x, short y, short color);
// refresh all tiles marked as dirty and unmark them
void RefreshDirty(TileWorld *tw);
// set a tile range as dirty
void SetDirtyT(TileWorld *tw, short left, short top, short right, short bottom);
// scroll the tiles
void ScrollTiles(TileWorld *tw, short x, short y);
// scroll to a particular upper left pixel corner
void ScrollTo(TileWorld *tw, short x, short y);
// set all tiles dirty at the beginning
void SetAllDirty(TileWorld *tw);
// change a tile and mark it dirty
void WriteTile(TileWorld *tw, short x, short y, unsigned char color);
// draw a sprite to the screen
void DrawSprite(TileWorld *tw, Sprite *spr);
// see if two sprites collide
char CollideSprite(Sprite *left, Sprite *right);
// see if a sprite collides with a rectangle
char CollideRect(Sprite *left, Rect *dst);
// see if a sprite collides with any solid tiles
char CollideArea(Sprite *spr, TileWorld *tw);
// see where the action point of a sprite sits
unsigned char ActPnt2Tile(Sprite *spr, TileWorld *tw, short *x, short *y);
// move a sprite's pixel location to its subpixel world space location
void MoveToSub(Sprite *p, TileWorld *tw1);
// move a sprite onto its parent
void MoveOnto(Sprite *spr, Sprite *parent);

// setup a clock
void SetupClock(short fps);
// call after each frame is done
int FrameDone(void);

TileWorld *NewTileWorld(short xTileSize, short yTileSize,
                       short xScrSize, short yScrSize, BITMAP *tileMap);

void FreeTileWorld(TileWorld *tw);

// invert palette
void MacInvert(BITMAP *bmp, RGB *pal);

#endif
