/**************************************\
* SOUND.C                              *
* DOSArena sound support               *
* Copr. 1999 Damian Yerrick            *
\**************************************/

/* Notice

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation; either version 2 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be useful and fun, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the License for more details.

You should have received a copy of the License along with this program,
in the file COPYING; if not, write to the Free Software Foundation, Inc. /
59 Temple Place - Suite 330 / Boston, MA  02111-1307 / USA
or view the License online at http://www.gnu.org/copyleft/gpl.html

The author can be reached by
  usps:Damian Yerrick / Box 398 / 5500 Wabash Avenue / Terre Haute, IN 47803
  mailto:dosarena@pineight.8m.com
    http://come.to/yerrick

DOSArena is a trademark of Damian Yerrick. All other trademarks belong to
their respective owners.

*/

#include "dosarena.h"

#ifdef ALLEGRO_DOS
int digiVol = 180;
#else
int digiVol = 252;
#endif
int midiVol = 248;

/**************************************\
* Custom streaming event handler       *
\**************************************/

/*
This is a layer on top of Allegro's midi_out() interface.  It allows
scheduling MIDI events to happen at any time in the future and played when the
time comes.
*/

static MIDIEvent midiQ[SIZE_midiQ] = {{0}};
static volatile int midiQRemain = -1;
static volatile int tix32 = 0;

/* PlayEvent_int() *********************
 * Called by timer.  Plays the next MIDIEvent when its time comes.
 * It plays all non-NoteOn events before playing NoteOn's to allow for more
 * polyphony.
 */
static void PlayEvent_int(void)
{
  short i;

  tix32++;

  // play events (other than NoteOn) from the queue
  if(midiQRemain)                // if there are still events left
  {
    int notesLeft = midiQRemain;

    for(i = 0; i < SIZE_midiQ; i++)
    {
      // if this event exists and is not an attack...
      if(midiQ[i].length != 0 &&
         ((midiQ[i].data[0] & 0xf0) != 0x90 || midiQ[i].data[2] == 0))
      {
        if(midiQ[i].when - tix32 <= 0) // and is set to occur now
        {
          midi_out(midiQ[i].data, midiQ[i].length); // then play it
          midiQ[i].length = 0;                      // and remove it
          midiQRemain--;
        }
        if(--notesLeft == 0)
          break;
      }
    }
  }

  // play more events from the queue (probably NoteOn's)
  if(midiQRemain)                // if there are still events left
  {
    int notesLeft = midiQRemain;

    for(i = 0; i < SIZE_midiQ; i++)
    {
      if(midiQ[i].length != 0)     // if this event exists
      {
        if(midiQ[i].when - tix32 <= 0) // and is set to occur now
        {
          midi_out(midiQ[i].data, midiQ[i].length); // then play it
          midiQ[i].length = 0;                      // and remove it
          midiQRemain--;
        }
        if(--notesLeft == 0)
          break;
      }
    }
  }
}
END_OF_FUNCTION(PlayEvent_int);

/* eventually put in this code for use with the internal speaker
  // Some compilers (DJGPP included) do not consistently support pow() on
  // FPU-less machines, so I had to create this power table.

  static const unsigned short SoundNote_freq[12] =
  {523, 554, 587, 622, 659, 698, 740, 784, 831, 880, 932, 988};
          short freq;

          // Find pitch within this octave.
          freq = SoundNote_freq[midiNote % 12];

          // If we have a low note, move it down an octave at a time.
          while(midiNote < 72)
          {
            freq >>= 1;
            midiNote += 12;
          }

          // If we have a high note, move it up an octave at a time.
          while(midiNote >= 84)
          {
            freq <<= 1;
            midiNote -= 12;
          }

          sound(freq);

*/


/* Put this in to pass audio when the sound is off
    {
      color.r = midiNote >> 1;
      color.b = 63 - color.r;
      set_color(0, &color);
    }
*/

void ResetPitchBends(void)
{
  unsigned char reset[] = {0xe0, 0, 0x40, 0xb0, 7, 112};
  short i;

  // reset volumes and pitch bends; Allegro 3.11 screwed this up
  for(i = 0; i < 16; i++)
  {
    midi_out(reset, 6);
    reset[0]++;
    reset[3]++;
  }
  set_volume(digiVol, midiVol);
}

/* InitPlayEvent() *********************
 * Clears the note cache and installs a timer handler that handles time
 * management.  Call it with the tempo (in quarter notes per minute) that you
 * want, and it will set up tix32 to be incremented every thirty-second note.
 */
void InitPlayEvent(int tempo)
{
  int speed;

  LOCK_VARIABLE(tix32);
  LOCK_VARIABLE(midiQ);
  LOCK_VARIABLE(midiQRemain);
  LOCK_VARIABLE(tix32);
  LOCK_FUNCTION(PlayEvent_int);

  // clear out events
  midiQRemain = 0;
  for(speed = 0; speed < 256; speed++)
  {
    midiQ[speed].length = 0;
  }
  tix32 = 0;

  speed = BPM_TO_TIMER(tempo << 3);
  install_int_ex(PlayEvent_int, speed);
}

/* PlayEvent() *************************
 * Adds a MIDIEvent to the list.  Returns the number of events in the list.
 */
short PlayEvent(MIDIEvent *event)
{
  int i;

  // add an event
  if(event != NULL && event->length != 0 &&
     midiQRemain >= 0 && midiQRemain < SIZE_midiQ)
  {
    if(event->length != 0)       // if this is a valid event
      for(i = 0; i < SIZE_midiQ; i++)   // look for a free space
        if(midiQ[i].length == 0) // if this event space is free
        {
          midiQRemain++;
          midiQ[i] = *event;     // add the event
          break;
        }
  }

  return midiQRemain;
}

/* Note() ******************************
 * Plays a MIDI note by constructing a MIDIEvent for the attack and release
 * phases.  For an immediate note (e.g. a beep sound), pass 0 in when.
 */
void Note(unsigned char channel, unsigned char noteNo, unsigned char vol,
          unsigned int when, unsigned int length)
{
  MIDIEvent event;

  if(when == 0)
    when = tix32 + 1;

  // build attack event
  event.when = when;
  event.length = 3;
  event.data[0] = 0x8f + channel; // Note On
  event.data[1] = noteNo;         // MIDI note number (middle C = 60)
  event.data[2] = vol;            // velocity
  PlayEvent(&event);

  // build release event
  event.when = when + length;
  event.data[2] = 0;              // velocity = 0 (key all the way up)
  PlayEvent(&event);
}

void remove_event(void)
{
  remove_int(PlayEvent_int);
  midiQRemain = -1;
}

/* InitMIDI() ***********************
 * Sets up MIDIEvent manager and instruments.
 * PORTING: This will stay pretty much the same for all General MIDI systems.
 */
void InitMIDI(short tempo, short nPatches, const unsigned char *patchMap)
{
  MIDIEvent event = {0, 2};
  short i;

// Cc nn       Load patch n into channel c.

  InitPlayEvent(tempo);
  load_midi_patches();
  if(nPatches)
    for(i = 0; i < nPatches; i++)
    {
      event.data[0] = 0xc0 | i;
      event.data[1] = patchMap[i];
      PlayEvent(&event);
    }
}

/* DisguiseRemoveSound() ***************
 * Covers up that annoying click when the Sound Bastard is turned off.
 */
void DisguiseRemoveSound(void)
{
  char mids[16] = {0xc0, 0x2f, 0x90, 0x20, 0x7c};

  midi_out(mids, 2);
  midi_out(mids + 2, 3);
  rest(100);
  mids[4] = 0;
  midi_out(mids + 2, 3);
#ifdef DJGPP
  sound(55);
#endif
  remove_sound();
#ifdef DJGPP
  nosound();
#endif
}

/* SendSound() *************************
 * Old 16-bit MS-DOS games used a function called SoundMgr that played a
 * SoundRec.  This new version of SoundMgr is functionally identical (to ease
 * porting) but plays through the MIDI card.
 */
void SendSound(const SoundRec *data)
{
  unsigned long nextTime;

  if(data)
  {
    nextTime = GetNow() + 1;
    while(data->note < 251)
    {
      if(data->note)
        Note(1, data->note, 125, nextTime, data->duration);
      nextTime += data->duration;
      data++;
    }
  }
}

/* unirand() ***************************
 * Compiler-independent pseudorandom number generator (PRN). Many compilers
 * have their own rand() functions, generally derived from the vendor's BASIC
 * or Pascal tools. The ANSI standard for rand() is this linear congruential
 * PRN:
 */
static unsigned long unirand_seed = 0;
short unirand(void)
{
  unirand_seed = (unirand_seed * 1103515245) + 12345;
  return (unirand_seed & 0x7fff0000) >> 16;
}

unsigned long GetNow(void)
{
  return tix32 + 1;
}


    //
   //
  // The DOSArena Jukebox
 //
//

static char midiNames[256][32];
static MIDI *midiSongs[256];
static int nSongs;

static void JukeboxInit(void)
{
  int i = 0;

  nSongs = 0;

  // Look for all the MIDI objects in the datafile.
  while(music_dat[i].type != DAT_END)
  {
    if(music_dat[i].type == DAT_MIDI)
    {
      strcpy(midiNames[nSongs],
             get_datafile_property(music_dat+i, DAT_ID('N','A','M','E')));
      midiSongs[nSongs] = music_dat[i].dat;
      nSongs++;
    }
    i++;
  }
  clear_to_color(screen, 15);
  set_palette(pal);
}

static char *GetSongList(int index, int *list_size)
{
  if (index < 0)
  {
    *list_size = nSongs;
    return NULL;
  }
  else
    return midiNames[index];
}

void Jukebox(void)
{
  short songPick = 0;

  DIALOG dlg[] =
  {
   /* (dialog proc)     (x)   (y)   (w)   (h)   (fg)  (bg)  (key) (flags)  (d1)  (d2)  (dp) */
   { DY_bitmap_proc,    8,    2,    96,   20,   0,    0,    0,    0,       0,    0,    GetResource(dosarena_dat, "LOGO_PCX") },
   { d_button_proc,     240,  96,   48,   12,   16,   15,   13,   D_EXIT,  0,    0,    "Play"},
   { d_button_proc,     240,  112,  48,   12,   16,   15,   27,   D_EXIT,  0,    0,    "Exit"},
   { d_list_proc,       32,   32,   192,  126,  14,   1,    0,    D_EXIT,  0,    0,    GetSongList },
   { DY_check_proc,     64,   160,  160,  8,    16,   15,   'l',  0,       0,    0,    "&Lock this song in" },
   { d_text_proc,       112,  4,    128,  16,   16,   15,   0,    0,       0,    0,    "Jukebox", chicago},
   { d_text_proc,       64,   176,  128,  16,   16,   15,   0,    0,       0,    0,    "Use arrows to move, then press Enter to play or Esc to quit.", small},
   { NULL,              0,    0,    0,    0,    0,    0,    0,    0,       0,    0,    NULL }
  };

  JukeboxInit();

  if(jukeboxMode >= 0)
  {
    dlg[4].flags |= D_SELECTED;
    dlg[3].flags = jukeboxMode;
    play_midi(midiSongs[jukeboxMode], TRUE);
  }

  while(songPick < nSongs)
  {
    // pick a song
    songPick = do_dialog(dlg, 3);

    if(songPick == 1 || songPick == 3)
    {
      // play it
      songPick = dlg[3].d1;
      play_midi(midiSongs[songPick], TRUE);
    }
    else
      songPick = nSongs;
  }
  jukeboxMode = (dlg[4].flags & D_SELECTED) ? dlg[3].d1 : -1;
}

