/**************************************\
* READTXT.C                            *
* Text buffer scrolling system         *
* Copr. 1999 Damian Yerrick            *
\**************************************/

/* Notice

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation; either version 2 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be useful and fun, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the License for more details.

You should have received a copy of the License along with this program,
in the file COPYING; if not, write to the Free Software Foundation, Inc. /
59 Temple Place - Suite 330 / Boston, MA  02111-1307 / USA
or view the License online at http://www.gnu.org/copyleft/gpl.html

The author can be reached by
  usps:Damian Yerrick / Box 398 / 5500 Wabash Avenue / Terre Haute, IN 47803
  mailto:dosarena@pineight.8m.com
    http://come.to/yerrick

DOSArena is a trademark of Damian Yerrick. All other trademarks belong to
their respective owners.

*/

#include "dosarena.h"

static unsigned long newlines[32767];

static void WriteLine(BITMAP *bmp, FONT *font, const char *buffer,
                      unsigned short lineNo, int x, int y, int c)
{
  char str1[256] = {0};
  int copylen;

  copylen = newlines[lineNo + 1] - newlines[lineNo];
  if(copylen > 254)
    copylen = 254;

  strncpy(str1, buffer + newlines[lineNo], copylen);
  textout(bmp, font, str1, x, y, c);
}

/* ReadTextFile() **********************
 * Display a text buffer.
 */
void ReadText(const char *buffer)
{
  unsigned short nLines = 1;
  unsigned       topDelta = 0;
  int            y = 0, i, curPos = 0, aimPos = 0;
  char           done = 0;
  int            lines = (SCREEN_H / 8) - 1, lastPos = lines;
  int            lastClock = 0;

  // Parse the text into lines.  This version doesn't wordwrap.
  newlines[0] = 0;
  while(buffer[topDelta])
  {
    if(buffer[topDelta] == '\n')
      newlines[nLines++] = topDelta;
    topDelta++;
  }
  newlines[nLines] = topDelta - 1;

  rectfill(screen, 0, 0, 319, 7, 1);
  text_mode(1);
  textout(screen, small, "Text file reader:  up and down scroll; esc exits", 1, 0, 14);
  text_mode(0);

  while(!done)
  {
    if(curPos != lastPos)
    {
      i = curPos;
      y = 8;
      if(curPos > lastPos && (curPos - lastPos) < lines)
      {
        int blitHt = (lastPos + lines - curPos) * 8;

        blit(screen, screen, 0, (curPos - lastPos) * 8 + 8, 0, 8, 320, blitHt);
        y += blitHt;
        i = lastPos + lines;
      }
      for(; i < curPos + lines; i++)
      {
        rectfill(screen, 0, y, 319, y + 7, 0);
        if(i < nLines)
          WriteLine(screen, small, buffer, i, 24, y, 15);
        y += 8;
      }
      lastPos = curPos;
    }

    while(keypressed())
      switch(readkey() >> 8)
      {
        case KEY_HOME:
          aimPos = 0;
          break;
        case KEY_UP:
          aimPos--;
          break;
        case KEY_PGUP:
          aimPos -= lines - 2;
          break;
        case KEY_END:
          aimPos = nLines - lines;
          break;
        case KEY_DOWN:
          aimPos++;
          break;
        case KEY_PGDN:
          aimPos += lines - 2;
          break;
        case KEY_ENTER:
        case KEY_ESC:
          done = 1;
      }
    if(aimPos < 0)
      aimPos = 0;
    if(aimPos >= nLines - lines - 1)
      aimPos = nLines - lines - 1;
    curPos = (aimPos + curPos) / 2 + (curPos < aimPos);

    while(lastClock == retrace_count)
      ;
    lastClock = retrace_count;
  }
}

/* ReadTextFile() **********************
 * Open a text file and read it.
 */
static void ReadTextFile(const char *filename)
{
  FILE *fp;
  char *theText;
  long fileSize;

  clear(screen);
  fp = fopen(filename, "r");
  if(fp == NULL)
    return;

  // Get the file size.
  fseek(fp, 0, SEEK_END);
  fileSize = ftell(fp);
  if(fileSize > 65528)
    fileSize = 65528;
  fseek(fp, 0, SEEK_SET);

  // Read the text into RAM.
  theText = malloc(fileSize + 1); // file size plus a terminator
  if(theText == NULL)
  {
    fclose(fp);
    return;
  }

  if(fread(theText, 1, fileSize, fp) < fileSize)
  {
    free(theText);
    fclose(fp);
    return;
  }
  theText[fileSize] = 0; // close off the text

  fclose(fp);
  ReadText(theText);
  free(theText);
}

void Copying(void)
{
  char fullPath[256];

  if(jukeboxMode >= 0)
    play_midi(music_dat[jukeboxMode].dat, TRUE);

  replace_filename(fullPath, progFolder, "copying", 255);
  ReadTextFile(fullPath);
}

