/**************************************\
* KEYBOARD.C                           *
* Set free*arena's key bindings        *
* Copr. 1999 Damian Yerrick            *
\**************************************/

// COPYRIGHT disclaimer here

#define alleg_3d_unused
#define alleg_math_unused
#define alleg_flic_unused
#include "dosarena.h"

typedef struct Key
{
  char name[12];
  unsigned char x, y, x2, y2;
} Key;

#define N_keys (KEY_MAX)

/* Updated for allegro 3.9.36 */
static const Key keys[N_keys] =
{
  {"No key",    0, 0},
  {"A",         9, 5},
  {"B",        27, 6},
  {"C",        19, 6},
  {"D",        17, 5},
  {"E",        16, 4},
  {"F",        21, 5},
  {"G",        25, 5},
  {"H",        29, 5},
  {"I",        36, 4},
  {"J",        33, 5},
  {"K",        37, 5},
  {"L",        41, 5},
  {"M",        35, 6},
  {"N",        31, 6},
  {"O",        40, 4},

  {"P",        44, 4},
  {"Q",         8, 4},
  {"R",        20, 4},
  {"S",        13, 5},
  {"T",        24, 4},
  {"U",        32, 4},
  {"V",        23, 6},
  {"W",        12, 4},
  {"X",        15, 6},
  {"Y",        28, 4},
  {"Z",        11, 6},
  {"0",        42, 3},
  {"1",         6, 3},
  {"2",        10, 3},
  {"3",        14, 3},
  {"4",        18, 3},

  {"5",        22, 3},
  {"6",        26, 3},
  {"7",        30, 3},
  {"8",        34, 3},
  {"9",        38, 3},
  {"Keypad0",  80, 7},
  {"Keypad1",  78, 6},
  {"Keypad2",  82, 6},
  {"Keypad3",  86, 6},
  {"Keypad4",  78, 5},
  {"Keypad5",  82, 5},
  {"Keypad6",  86, 5},
  {"Keypad7",  78, 4},
  {"Keypad8",  82, 4},
  {"Keypad9",  86, 4},
  {"F1",       10, 1},

  {"F2",       14, 1},
  {"F3",       18, 1},
  {"F4",       22, 1},
  {"F5",       28, 1},
  {"F6",       32, 1},
  {"F7",       36, 1},
  {"F8",       40, 1},
  {"F9",       46, 1},
  {"F10",      50, 1},
  {"F11",      54, 1},
  {"F12",      58, 1},
  {"Esc",       2, 1},
  {"~Tilde",    2, 3},
  {"-",        46, 3},
  {"=",        50, 3},
  {"Backspc",  56, 3},

  {"Tab",       3, 4},
  {"[",        48, 4},
  {"]",        52, 4},
  {"Enter",    55, 5},
  {";",        45, 5},
  {"'",        49, 5},
  {"\\",       57, 4},
  {"Alt. \\",   0, 0},
  {",",        39, 6},
  {".",        43, 6},
  {"/",        47, 6},
  {"Space",    30, 7},
  {"Insert",   64, 3},
  {"Delete",   64, 4},
  {"Home",     68, 3},
  {"End",      68, 4},

  {"PageUp",   72, 3},
  {"PageDown", 72, 4},
  {"Left",     64, 7},
  {"Right",    72, 7},
  {"Up",       68, 6},
  {"Down",     68, 7},
  {"Keypad/",  82, 3},
  {"Keypad*",  86, 3},
  {"Keypad-",  90, 3},
  {"Keypad+",  90, 5},
  {"Delete",   86, 7},
  {"Keypad+",  90, 7},
  {"PrntScrn", 64, 1},
  {"Pause",    72, 1},
  {"Abnt_C1",  40, 8},
  {"Yen",      44, 8},

  {"Kana",     48, 8},
  {"Convert",  52, 8},
  {"NoCnvrt",  56, 8},
  {"@",        60, 8},
  {"^",        64, 8},
  {": 2",      68, 8},
  {"Kanji",    68, 8},
  {"L.Shift",   5, 6},
  {"R.Shift",  55, 6},
  {"Control",   3, 7},
  {"R.Ctrl",   57, 7},
  {"Alt",      13, 7},
  {"AltGr",    47, 7},
  {"L.Win",     8, 7},
  {"R.Win",    52, 7},
  {"Menu",     52, 8},

  {"ScrLock",  68, 1},
  {"NumLock",  78, 3},
  {"CapsLock",  4, 5}
};


static unsigned char closestInit = 0;
static unsigned char closest[9][96];
static BITMAP *keytemp;

static int SetupClosest(void)
{
  int x, y, x2, y2, i;

  for(i = 1; i < N_keys; i++)
  {
    closest[keys[i].y2][keys[i].x2] = i;
    closest[keys[i].y][keys[i].x] = i;
  }

  do {
    y2 = 0;
    for(y = 0; y < 9; y++)
    {
      x2 = 0;
      for(x = 0; x < 96; x++)
      {
        if(closest[y][x] == 0)
        {
          if(closest[y][x + 1])
          {
            closest[y][x] = closest[y][x + 1];
            x2 = 0;
            y2++;
          }
          else if(closest[y][x - 1] && x2 == 0)
          {
            closest[y][x] = closest[y][x - 1];
            x2 = 1;
            y2++;
          }
        }
      }
    }
  } while(y2);

  closestInit = 1;

  return 0;
}

typedef struct Keypad
{
  char key[8][2];
} Keypad;

static Keypad keypads[3] =
{
  // Left player
  {{
    {KEY_R},
    {KEY_F},
    {KEY_D},
    {KEY_G},
    {KEY_LSHIFT},
    {KEY_Z},
    {KEY_LCONTROL},
    {KEY_ALT}
  }},
  // Right player
  {{
    {KEY_UP},
    {KEY_DOWN},
    {KEY_LEFT},
    {KEY_RIGHT},
    {KEY_QUOTE},
    {KEY_ENTER},
    {KEY_SLASH},
    {KEY_RSHIFT}
  }},
  // Middle player
  {{
    {KEY_I},
    {KEY_K},
    {KEY_J},
    {KEY_L},
    {KEY_U},
    {KEY_O},
    {KEY_SPACE},
    {KEY_ALTGR}
  }}
};

void ReadKeys(void)
{
  FILE *fp = fopen("frearena.kyb", "rb");

  if(fp)
  {
    fread(keypads, sizeof(Keypad), 3, fp);
    fclose(fp);
    fp = NULL;
  }
}

/* ReadKPad() **************************
 * Reads an area of the keyboard.
 * 0x80 up
 * 0x40 down
 * 0x20 left
 * 0x10 right
 * 0x08 left trigger
 * 0x04 right trigger
 * 0x02 A trigger
 * 0x01 B trigger
 */
int ReadKPad(int which)
{
  int out = 0;
  int theKey, i, j;

  for(i = 0; i < 8; i++)
  {
    out <<= 1;
    for(j = 0; j < 2; j++)
    {
      theKey = keypads[which].key[i][j];

      if(theKey && key[theKey])
        out |= 1;
    }
  }

  // pause key
  if(which == 1)
  {
    if(key[KEY_PAUSE] || key[KEY_END])
      out |= 0x0100;
  }
  else
    if(key[KEY_ESC])
      out |= 0x0100;

  return out;
}

static void DrawKeyboard(BITMAP *keybits, int under, const Keypad *keypad)
{
  int x1 = keys[under].x * 3;
  int y1 = keys[under].y * 10;
  int x2 = keys[under].x2 * 3;
  int y2 = keys[under].y2 * 10;
  int i, j;

  blit(keybits, keytemp, 0, 0, 0, 0, keybits->w, keybits->h);

  for(i = 0; i < 8; i++)
    for(j = 0; j < 2; j++)
    {
      int k = keypad->key[i][j];

      if(keys[k].x != 0)
        blit(keybits, keytemp, 12 * i, 100,
             keys[k].x * 3, keys[k].y * 10, 13, 11);
      if(keys[k].x2 != 0)
        blit(keybits, keytemp, 12 * i, 100,
             keys[k].x2 * 3, keys[k].y2 * 10, 13, 11);
    }

  ellipsefill(keytemp, x1 + 6, y1 + 5, 5, 4, 4);
  ellipsefill(keytemp, x2 + 6, y2 + 5, 5, 4, 6);
  blit(keytemp, screen, 0, 0, 0, 0, keybits->w, keybits->h);
}

void SetKeys(void)
{
  BITMAP *keybits = NULL;
  BITMAP *theCursor = NULL;
  int i = 0, last_i = 0;
  int curKey = 0;
  int showKey = -1;
  int curPad = 0;
  FILE *fp;

  clear_to_color(screen, 16);

  if(!closestInit)
    SetupClosest();

  keybits = GetResource(dosarena_dat, "keys_pcx");
  if(keybits == NULL)
  {
    set_palette(pal);
    textout(screen, font, "No keys.pcx", 0, 0, 15);
    readkey();
    return;
  }

  keytemp = create_bitmap(keybits->w, keybits->h);
  if(keytemp == NULL)
  {
    set_palette(pal);
    textout(screen, font, "No RAM for keyboard manager", 0, 0, 15);
    readkey();
    return;
  }

  theCursor = create_bitmap(13, 11);
  if(theCursor == NULL)
  {
    set_palette(pal);
    textout(screen, font, "No RAM for cursor manager", 0, 0, 15);
    readkey();
    destroy_bitmap(keytemp);
    keytemp = NULL;
    return;
  }

  set_palette(black_palette);
  DrawKeyboard(keybits, 0, &keypads[curPad]);

  text_mode(16);
  textout(screen, chicago, SHORT_PLATFORM_STR"Arena Keyboard Setup", 8, 120, 31);
  textout(screen, small,   "Click on a key icon, then left-click or right-click on the keyboard to set the key.", 8, 136, 30);
  textout(screen, small,   "Press Esc to exit.", 8, 143, 30);
  textout(screen, font,    "|  Player 1  |  Player 2  |  Player 3  |", 0, 151, 29);
  hline(screen, 104 * curPad + 4, 159, 104 * curPad + 107, 4);

  show_mouse(screen);
  fade_in(pal, 5);

  do {
    int x = mouse_x / 3;
    int y = mouse_y / 5;

    if(y >= 20)
    {
      if(showKey)
      {
        showKey = 0;
        set_mouse_sprite(NULL);
      }
      if(mouse_b)
      {
        switch(y)
        {
          case 20:
          case 21:
            if(x < 32)
              curKey = x / 4;
              while(mouse_b)
                ;
            break;
          case 30:
          case 31:
            curPad = x / 35;
            scare_mouse();
            hline(screen, 0, 159, 319, 16);
            hline(screen, 104 * curPad + 4, 159, 104 * curPad + 107, 4);
            DrawKeyboard(keybits, i, &keypads[curPad]);
            unscare_mouse();
            while(mouse_b)
              ;
            break;
        }
      }
    }
    else
    {
      if(x >= 96)
        x = 95;
      i = closest[(y + 1) / 2][x];

      if(showKey == 0)
      {
        showKey = 1;
        blit(keybits, theCursor, 12 * curKey, 100, 0, 0, 13, 11);
        set_mouse_sprite(theCursor);
      }

      if(i != last_i)
      {
        show_mouse(NULL);
        DrawKeyboard(keybits, i, &keypads[curPad]);
        text_mode(15);
        textprintf(screen, font, 224, 100, 16, "%8s", keys[i].name);
        show_mouse(screen);
        last_i = i;
      }
      if(mouse_b & 1)
      {
        keypads[curPad].key[curKey][0] = i;
        last_i = -1;
      }
      if(mouse_b & 2)
      {
        keypads[curPad].key[curKey][1] = i;
        last_i = -1;
      }
    }

  } while(!keypressed());
  readkey();

  fp = fopen("DOSArena.kyb", "wb");
  if(fp)
  {
    fwrite(keypads, sizeof(Keypad), 3, fp);
    fclose(fp);
  }
  else
    alert("Could not write your keyboard", 
          "bindings to the disk.  Make sure", 
          "that it is not write-protected.", 
          "Damn", NULL, 13, 0);

  // MUST set the mouse sprite back to an arrow before freeing the cursor
  // bitmap, or the program will crash.
  // Shawn: You might want to fix this. If the mouse's bitmap is not valid,
  // automatically set the sprite back to an arrow.

  set_mouse_sprite(NULL);

  destroy_bitmap(theCursor);
  theCursor = NULL;
  destroy_bitmap(keytemp);
  theCursor = NULL;


  return;
}
