/**************************************\
* INSANE.C                             *
* Insane 1.5 for DOSArena              *
* By Damian Yerrick                    *
\**************************************/

/* Notice

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation; either version 2 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be useful and fun, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the License for more details.

You should have received a copy of the License along with this program,
in the file COPYING; if not, write to the Free Software Foundation, Inc. /
59 Temple Place - Suite 330 / Boston, MA  02111-1307 / USA
or view the License online at http://www.gnu.org/copyleft/gpl.html

The author can be reached by
  usps:Damian Yerrick / Box 398 / 5500 Wabash Avenue / Terre Haute, IN 47803
  mailto:dosarena@pineight.8m.com
    http://come.to/yerrick

DOSArena is a trademark of Damian Yerrick. All other trademarks belong to
their respective owners.

*/

#include <ctype.h>
#include <math.h>

#define alleg_3d_unused
#define alleg_joystick_unused
#define alleg_flic_unused

#include "dosarena.h"
#include "tilewld.h"

// data types ////////////////////////////////////////////////////////////////

typedef struct Player
{
  BITMAP *map;
  BITMAP *gameWindow;
  TileWorld *tw;
  char shiftDir, cheated, tileset, r1;
  char repeatTime[8];
  unsigned short blocks, r2;
} Player;

typedef struct DemoPt
{
  unsigned char x, y, time, click;
} DemoPt;

typedef struct DemoLine
{
  char line1[48], line2[48];
  DemoPt points[16];
} DemoLine;

// constants /////////////////////////////////////////////////////////////////
enum {
  kModePuzzle = 0,
  kModeAction,
  kModeTA,
  kModeDemo,
  kModeExit
};

static const char gModeNames[kModeExit + 1][20] =
{
  "Puzzle",
  "Action",
  "Rising Attack",
  "Demo & Tips",
  "Exit to "SHORT_PLATFORM_STR"Arena"
};

static const SoundRec gBeepSnd[] = {{83, 1}, {79, 1}, {255, 255}};

static const SoundRec gGameOverSnd[] =
{
  {41, 1}, {47, 1}, {40, 16}, {39, 1}, {37, 1}, {35, 1}, {33, 1},
  {30, 1}, {27, 1}, {24, 1},
  {255, 255}
};


static unsigned char demoBlox[8][12] =
{
  {1, 3, 5, 3, 5, 4, 1, 2, 2, 3, 1, 2},
  {3, 4, 3, 1, 4, 1, 5, 5, 3, 3, 1, 1},
  {3, 2, 4, 2, 4, 4, 1, 5, 4, 4, 3, 3},
  {5, 1, 2, 4, 5, 5, 2, 5, 2, 3, 2, 4},
  {1, 3, 3, 5, 2, 1, 4, 4, 1, 5, 2, 2},
  {4, 2, 5, 5, 1, 2, 4, 2, 5, 4, 2, 3},
  {5, 5, 5, 2, 5, 5, 5, 3, 4, 4, 2, 3},
  {2, 5, 3, 4, 2, 3, 4, 2, 4, 4, 4, 4}
};

static DemoLine demo[] =
{
  {
    "Insane Demo (Esc to skip)",
    "In the center of the screen is a cursor.",
    {
      {5, 4, 100, 0},
      {255},
    }
  },
  {
    "Move the cursor with the arrow keys or",
    "the mouse.",
    {
      { 5, 4,100, 0},
      { 4, 4, 60, 0},
      { 5, 4, 40, 0},
      { 6, 4, 30, 0},
      { 7, 4, 20, 0},
      { 8, 4, 20, 0},
      { 8, 5, 20, 0},
      {255},
    }
  },
  {
    "To remove a group of identical tiles,",
    "press the spacebar or mouse button.",
    {
      { 8, 6,100, 30},
      { 9, 6, 30, 0},
      {10, 6, 30, 30},
      {10, 6, 30, 30},
      {255},
    }
  },
  {
    "Tiles will fall into empty spaces and",
    "shift into empty columns.",
    {
      { 9, 6,100, 0},
      { 8, 6, 20, 0},
      { 8, 5, 30, 30},
      { 8, 5, 30, 30},
      { 8, 6, 20, 0},
      { 8, 7, 30, 30},
      { 8, 6, 30, 30},
      {255},
    }
  },
  {
    "Use strategy to get lots of the",
    "same tile into one place and...",
    {
      { 9, 6,100, 0},
      {10, 5, 15, 0},
      {11, 5, 15, 10},
      { 0, 4, 15, 10},
      { 1, 4, 15, 10},
      { 1, 4, 15, 10},
      { 1, 3, 15, 10},
      { 1, 3, 15, 10},
      { 1, 4, 15, 10},
      { 2, 5, 15, 0},
      { 3, 5, 15, 0},
      { 4, 6, 15, 30},
      { 3, 5, 15, 0},
      {255},
    }
  },
  {
    "clear them all for big points.",
    "",
    {
      { 3, 5,100, 30},
      { 3, 5, 40, 30},
      {255},
    }
  },
  {
    "When you run out of matches,",
    "the game is over.",
    {
      { 3, 5,100, 30},
      { 4, 6, 40, 30},
      { 4, 6, 15, 70},
      {255},
    }
  },
  { "" }
};

// globals ///////////////////////////////////////////////////////////////////

static Player p[2];

static BITMAP *tiles, *tiles83, *title;
static DATAFILE *pix;

static int gXSize, gYSize, gameStart;
static unsigned char screensaving = 0;

unsigned char gSolidTiles[256] = {0};
static int curTurn = 0; // for forthcoming two player mode

#define ME p[curTurn]

// prototypes to tell the compiler how each function likes its parameters ////
static char ActionAddTile(short bottom, short offset);
static void Beep(void);
static void CopyAll(short offset);
static void CopyTile(short x, short y, short offset);
static void DisplayScore(unsigned long score);
static void DrawCursor(short x, short y, short offset);
static void GameOver(short gameType, unsigned long score);
static void InsaneShutDown(void);
static short InsaneStartUp(void);
static void Instructions(void);
static char MakemFall(short bottom);
static char MakemShift(short bottom);
static void NewGame(short);
static unsigned long Play(short);
void SetFadeStep(const RGB *, unsigned short pos,
                 unsigned short from, unsigned short to);
static void ShadowText(const char *str, short x, short y, short color);
static short TitleScr(short lastType);

// code //////////////////////////////////////////////////////////////////////

void InsaneMain(void)
{
  short gameType = 0;

  if(InsaneStartUp())
    return;

  while((gameType = TitleScr(gameType)) != kModeExit)
  {
    switch(gameType)
    {
      case kModePuzzle:
      case kModeAction:
      case kModeTA:
	// play a game and then put up the game over screen
	GameOver(gameType, Play(gameType));
	break;
      case kModeDemo:
        Play(kModeDemo);
        stop_midi();
        fade_out_range(2, 1, 255);
        play_midi(GetResource(music_dat, "DOSTHEME_MID"), TRUE);
	if(!screensaving)
	  Instructions();
	break;
      }
  }

  InsaneShutDown();
}

static char ActionAddTile(short bottom, short offset)
{
  short fillX, fillY;

  // Start looking at the bottom left.  If a space is found, return 0 for
  // success.  If the screen is full, it returns 1, which signals "game over"
  // to Play().

  for(fillY = bottom - 1; fillY >= 0; fillY--)
    for(fillX = 0; fillX < 12; fillX++)
      if(getpixel(ME.tw->tileMap, fillX, fillY) == 0)
      {
        WriteTile(ME.tw, fillX, fillY, (rand() % 5) + 1);
        RefreshDirty(ME.tw);
	CopyTile(fillX, fillY, offset);
	return 0;
      }

  return 1;
}

// either beep or change the border color
static void Beep(void)
{
  SendSound(gBeepSnd);
}

// copy all tiles from TileWorld to screen
static void CopyAll(short offset)
{
  short rTop = offset - gYSize * ME.tw->yTileOffset; // scrolling

  scare_mouse();
  blit(ME.tw->tileDest, ME.gameWindow, 0, rTop, 0, 0,
       ME.gameWindow->w, ME.gameWindow->h);
  unscare_mouse();
}

// copy a single tile to screen
static void CopyTile(short x, short y, short offset)
{
  Rect r = {0, 0, 0, 24};

  r.top = gYSize * (y - ME.tw->yTileOffset);
  r.left = gXSize * x;

  scare_mouse();
  blit(ME.tw->tileDest, ME.gameWindow, r.left, r.top,
         x * gXSize, y * gYSize - offset,
         gXSize, gYSize);
  unscare_mouse();
}

static void DrawCursor(short x, short y, short offset)
{
  scare_mouse();
  masked_blit(ME.tw->tileImg, ME.gameWindow,
              0, 7 * gYSize,
              x * gXSize, y * gYSize - offset,
              gXSize, gYSize);
  unscare_mouse();
}

static void DisplayScore(unsigned long score)
{
  char str1[64];

  scare_mouse();
  rectfill(screen, 16, 16, 142, 31, 16); // erase the longest possible old score
  sprintf(str1, "Score: %ld", score);
  textout(screen, chicago, str1, 16, 16, 14);
  unscare_mouse();
}

// display game over message and maybe ask for a high score
static void GameOver(short gameType, unsigned long score)
{
  HighScore entry =
  {
    {
      score >> 16,
      score & 0xffff,
      ME.blocks,
      (retrace_count - gameStart) / 70
    },
    ""
  };

  // fade the tiles out for drama
  stop_midi();
  SendSound(gGameOverSnd);
  FreeTileWorld(ME.tw);
  destroy_bitmap(ME.gameWindow);

  if(ME.cheated)
  {
    text_mode(0);
    textout(screen, chicago, "You little cheater!", 16, 32, 12);
    fade_out_range(2, 38, 255);
    fade_out_range(2, 0, 37);
    return;
  }

  show_mouse(NULL);
  GameOver1P(gameType + SCORE_INSANE_PUZZLE, &entry);
  show_mouse(screen);

  if(midiPresent)
    play_midi(GetResource(music_dat, "DOSTHEME_MID"), TRUE);
}


static short InsaneStartUp(void)
{
  // load graphics
  pix = load_skinned_datafile("insane.dat");
  if(pix == NULL)
  {
    textout(screen, chicago, "This skin contains no insane.dat", 0, 0, 0);
    readkey();
    return 1;
  }

  tiles   = GetResource(pix, "tiles_pcx");
  tiles83 = GetResource(pix, "tiles83_pcx");
  title   = GetResource(pix, "title_pcx");
  if(tiles == NULL || title == NULL)
  {
    unload_datafile(pix);
    textout(screen, chicago, "insane.dat is missing something", 0, 0, 0);
    readkey();
    return 32;
  }

  curTurn = 0;

  ME.map = create_bitmap(12, 20);
  if(!ME.map)
  {
    unload_datafile(pix);
    textout(screen, chicago, "Not enough RAM to run Insane", 0, 0, 0);
    readkey();
    return 32;
  }

  text_mode(-1);
  fade_out(4);
  show_mouse(screen);
  InitPlayEvent(136);
  return 0;
}

static void InsaneShutDown(void)
{
  destroy_bitmap(ME.map);
  unload_datafile(pix);
  show_mouse(NULL);

  remove_event();
}

static void Instructions(void)
{
  show_mouse(NULL);
  clear_to_color(screen, 42);

  // initial help text
  text_mode(-1);
  textout(screen, chicago, "Tips", 8, 8, 15);

  ShadowText("Puzzle: Play until no matches are left.", 16, 32, 14);
  ShadowText("Action: New tiles drop in from the top; play until the screen is full.", 16, 40, 14);
  ShadowText("Rising Attack: The screen scrolls up; play until a tile hits the top center.", 16, 48, 14);

  ShadowText("Press Home or End on the keyboard to shift all the tiles sideways.", 16, 60, 12);
  ShadowText("Press Esc on the keyboard to pause the game.", 16, 68, 12);

  ShadowText("If you're a true purist, press T to use original black-and-white graphics.", 16, 80, 14);

  ShadowText("Press a key", 1, 192, 15);
  show_mouse(screen);

  fade_in(pal, 3);
  readkey();
  fade_out(3);
}

static char MakemFall(short bottom)
{
  short srcY, fillX, fillY;
  char rval = 0;

  for(fillX = 0; fillX < 12; fillX++)
  {
    // move to first hole
    fillY = bottom - 1;
    while((fillY >= 0) && getpixel(ME.map, fillX, fillY))
      fillY--;

    srcY = fillY;

    // move everything down
    while(srcY >= 0)
    {
      if(getpixel(ME.map, fillX, srcY)) // there's a tile here
      {
	// move it down
	WriteTile(ME.tw, fillX, fillY,
	          getpixel(ME.map, fillX, srcY));
	WriteTile(ME.tw, fillX, srcY, 0);
	fillY--;
	rval = 1;
      }
      srcY--;
    }
  }
  RefreshDirty(ME.tw);
  return rval;
}

// 0: did nothing
// 1: did something
static char MakemShift(short bottom)
{
  short srcX, fillX, fillY;
  char rval = 0;

  if(ME.shiftDir == 0)
  {
    // move to first hole
    fillX = 0;
    while((fillX < 12) && getpixel(ME.map, fillX, bottom - 1))
      fillX++;

    if(fillX >= 12)
      return 0; // nothing

    // move everything left
    srcX = fillX;
    while(srcX < 12)
    {
      if(getpixel(ME.map, srcX, bottom - 1)) // there's a tile here
      {
        // move it left
        for(fillY = 0; fillY < bottom; fillY++)
        {
          WriteTile(ME.tw, fillX, fillY,
                    getpixel(ME.map, srcX, fillY));
          WriteTile(ME.tw, srcX, fillY, 0);
        }
        fillX++;
        rval = 1;
      }
      srcX++;
    }
  }
  else
  {
    // move to first hole
    fillX = 11;
    while((fillX >= 0) && getpixel(ME.map, fillX, bottom - 1))
      fillX--;

    if(fillX < 0)
      return 0; // nothing

    // move everything right
    srcX = fillX;
    while(srcX >= 0)
    {
      if(getpixel(ME.map, srcX, bottom - 1)) // there's a tile here
      {
        // move it left
        for(fillY = 0; fillY < bottom; fillY++)
        {
          WriteTile(ME.tw, fillX, fillY,
                    getpixel(ME.map, srcX, fillY));
          WriteTile(ME.tw, srcX, fillY, 0);
        }
        fillX--;
        rval = 1;
      }
      srcX--;
    }
  }
 
  RefreshDirty(ME.tw);
  return rval;
}

void SetTiles(BITMAP *tileset)
{
  blit(tileset, ME.tw->tileImg, 0, 0, 0, 0, tileset->w, tileset->h);
}

static void NewGame(short type)
{
  int x, y = 0;

  ME.tw = NewTileWorld(24, 20, 256, 160, ME.map);
  ME.gameWindow = create_sub_bitmap(screen, 16, 32, 288, 160);

  if(!ME.tw || !ME.gameWindow)
  {
    allegro_exit();
    puts("couldn't get RAM for drawing tiles on player 1's screen");
    exit(1);
  }

  SetTiles(tiles);

  SetFadeStep(pal, 0, 1, 256);
  show_mouse(NULL);
  clear(screen);

  // create initial pseudo-random tile map
  switch(type)
  {
    case kModeTA:
    case kModeAction:
      rectfill(ME.map, 0, 0, 11, 3, 0);
      y = 4;
      break;
    case kModeDemo:
      for(y = 0; y < 8; y++)
        for(x = 0; x < 12; x++)
          putpixel(ME.map, x, y, demoBlox[y][x]);
      break;
    case kModePuzzle:
      y = 0;
      break;
  }
  while(y < 8)
  {
    for(x = 0; x < 12; x++)
      putpixel(ME.map, x, y, rand() % 5 + 1);
    y++;
  }
  /* When I add the instant replay feature, I'll have to use the C standard
   * random number generator.
   */

  /* TEMPORARY DEBUG CODE:
   * Write the tile map to the disk.
   */

#ifdef WRITE_TILEMAP
  {
    FILE *fp = fopen("insanedisk.c", "wt");

    fputs("static int demoBlox[8][12] =\n{", fp);
    for(y = 0; y < 8; y++)
    {
      fputs("\n  {", fp);
      for(x = 0; x < 12; x++)
      {
        int num = getpixel(ME.map, x, y);

        if(x > 0)
          fputs(", ", fp);
        fprintf(fp, "%d", num);
      }
      fputs("}", fp);
      if(y < 7)
        fputs(",\n", fp);

    }
    fputs("};", fp);
    fclose(fp);
  }
#endif

  // init tile world
  ME.tw->yTileOffset = ME.tileset = 0;
  SetAllDirty(ME.tw);
  RefreshDirty(ME.tw);
  CopyAll(0);

  // draw initial game objects
  if(type == kModeTA)
    rectfill(screen, 136, 28, 183, 31, 12); // red bar for rising attack limit

  DisplayScore(0);
  if(type != kModeDemo)
    show_mouse(screen);

  ME.shiftDir = ME.cheated = ME.blocks = 0;
  if(midiPresent)
  {
    if(jukeboxMode >= 0)
      play_midi(music_dat[jukeboxMode].dat, TRUE);
    else
      play_midi(GetResource(music_dat, "CHOPIN_MID"), TRUE);
  }

}

/* look for matches and return 1 if none is found.  In Puzzle mode, no matches
 * means game over.
 */
static char NoMatches(short bottom)
{
  short x, y;
  unsigned char c;

  // check for vertical pairs
  for(y = 1; y < bottom; y++)
    for(x = 0; x < 12; x++)
    {
      c = getpixel(ME.map, x, y);
      if(c != 0 && c == getpixel(ME.map, x, y - 1))
        return 0;
    }

  // check for horizontal pairs
  for(y = 0; y < bottom; y++)
    for(x = 0; x < 11; x++)
    {
      c = getpixel(ME.map, x, y);
      if(c != 0 && c == getpixel(ME.map, x + 1, y))
        return 0;
    }

  return 1;
}

static char Pause(short offset)
{
  short i, k;

  // fade the tiles out so player doesn't cheat
  for(i = 192; i >= 0; i -= 16)
  {
    clock_t lastClock;

    lastClock = retrace_count;
    while(lastClock == retrace_count)
      ;

    SetFadeStep(pal, i, 40, 255); // fade just the tiles
  }

  midi_pause();
  show_mouse(NULL);
  k = DA_Pause();
  show_mouse(screen);
  midi_resume();
  text_mode(-1);

  // fade the tiles back in
  for(i = 64; i <= 256; i += 16)
  {
    clock_t lastClock;

    lastClock = retrace_count;
    while(lastClock == retrace_count)
      ;

    SetFadeStep(pal, i, 40, 255); // fade just the tiles
  }

  return k;
}

static unsigned long Play(short gameType)
{
  unsigned long  score = 0;
  unsigned short bottom = 8, demoLine = 0, demoPt = 0;
  short          cursorX = 5, cursorY = 4, yScrollLeft = -5;
  int            goOn;
  int            lastClock = retrace_count;
  unsigned short fadeIntense = 0, yScroll = 0;
  char           done = 0, hiddenCursor = 0, lastButtons = 0, demoState = 0;
  char           buttons, clicked;
  short          k;
  FONT           *impact = GetResource(dosarena_dat, "IMPACT_PCX");

  NewGame(gameType);
  gXSize = ME.tw->xTileSize;
  gYSize = ME.tw->yTileSize;
  DrawCursor(cursorX, cursorY, yScroll);
  gameStart = retrace_count;
  goOn = gameStart + 70;
  if(impact == NULL)
    impact = chicago;

  while(!done)
  {
    while(lastClock - retrace_count > 0)
      yield_timeslice();
    lastClock++;

    // fade in
    if(fadeIntense < 256)
    {
      fadeIntense = (retrace_count - gameStart) * 4;
      if(fadeIntense > 256)
        fadeIntense = 256;
      SetFadeStep(pal, fadeIntense, 1, 256);
    }

    if(NoMatches(bottom)) // has player run out of matches?
    {
      if(gameType == kModePuzzle || gameType == kModeDemo)
	done = 1;
      else
        goOn = retrace_count;
    }

    buttons = 0;
    clicked = 0;

    // Handle keyboard commands.
    if(keypressed() && gameType != kModeDemo)
    {
      k = readkey() >> 8;
      switch(k)
      {
        case KEY_HOME:
          ME.shiftDir = 0;
          if(MakemShift(bottom))
          {
            CopyAll(yScroll);
            hiddenCursor = 1;
          }
          break;
        case KEY_END:
          ME.shiftDir = 1;
          if(MakemShift(bottom))
          {
            CopyAll(yScroll);
            hiddenCursor = 1;
          }
          break;
	case KEY_ESC:
	  // make sure pausing doesn't screw the timers up
          goOn -= retrace_count;
          gameStart -= retrace_count;

	  // done will be set to (user wanted to exit)
	  done = Pause(yScroll);

	  // clean up timers after pause
          goOn += retrace_count;
          gameStart += retrace_count;
	  hiddenCursor = 1;
	  break;

        case KEY_T:
          if(tiles83 != NULL)
          {
            ME.tileset = !ME.tileset;
            SetTiles(ME.tileset ? tiles83 : tiles);
            SetAllDirty(ME.tw);
            RefreshDirty(ME.tw);
            CopyAll(yScroll);
            hiddenCursor = 1;
          }
          break;
      }
    }

    // Handle game control on keyboard, mouse, and joystick.
    if(gameType != kModeDemo)
    {
      short mouseX, mouseY, pointingX, pointingY;

      k = ReadJPad(0) | ReadKPad(1) | ReadKPad(2);
      MakeRepeats(ME.repeatTime, k, 18, 2); // repeat 35 fps after 1/4 s

      if(ME.repeatTime[0] == 1 || ME.repeatTime[0] == 17) // move up
      {
	hiddenCursor = 1;
	CopyTile(cursorX, cursorY, yScroll);
	cursorY--;
	if(cursorY < (short)bottom - 8)
	  cursorY += 8;
      }
      if(ME.repeatTime[1] == 1 || ME.repeatTime[1] == 17) // move down
      {
	hiddenCursor = 1;
	CopyTile(cursorX, cursorY, yScroll);
	cursorY++;
	if(cursorY >= bottom)
	  cursorY -= 8;
      }
      if(ME.repeatTime[2] == 1 || ME.repeatTime[2] == 17) // move left
      {
	hiddenCursor = 1;
	CopyTile(cursorX, cursorY, yScroll);
        cursorX = (cursorX + 11) % 12;
      }
      if(ME.repeatTime[3] == 1 || ME.repeatTime[3] == 17) // move right
      {
        hiddenCursor = 1;
        CopyTile(cursorX, cursorY, yScroll);
	cursorX = (cursorX + 1) % 12;
      }

      buttons = (k & 0x0e) | mouse_b;
      if(gCheatCodes[1].flag && key[KEY_LCONTROL])
      {
        clicked = 1;
        ME.cheated = 1;
      }

      if(buttons & ~lastButtons)
	clicked = 1;
      lastButtons = buttons;

      mouseX = mouse_x;
      mouseY = mouse_y;

      if(mouseX >= 16 && mouseX < 304 && mouseY >= 32 && mouseY < 192)
      {
	pointingX = (mouseX - 16) / gXSize;
	pointingY = (mouseY - 32 + yScroll) / gYSize;

	if((pointingX != cursorX || pointingY != cursorY) &&
	   (pointingY >= bottom - 8))
	{
	  hiddenCursor = 1;
	  CopyTile(cursorX, cursorY, yScroll);
	  cursorX = pointingX;
	  cursorY = pointingY;
	}
      }
    }

    // handle time based processes such as ActionAddTile, rising, and the demo
    if(goOn - retrace_count <= 0) // if the timer has run out
    {
      long temp1;
      short x1;

      switch(gameType)
      {
	case kModeAction: // add another tile at the top
	  done |= ActionAddTile(bottom, yScroll);
	  temp1 = goOn - gameStart + 8000;
	  goOn += 180000 / temp1;
	  hiddenCursor = 1;
	  break;
	case kModeTA: // add another row at the bottom
	  temp1 = goOn - gameStart + 8000;
	  goOn += 2080000 / temp1;
	  yScrollLeft += gYSize;

	  // make new tiles
	  if(getpixel(ME.map, 5, bottom - 8) ||
	     getpixel(ME.map, 6, bottom - 8))
	    done = 1;

	  for(x1 = 0; x1 < 12; x1++)
	  {
	    if(bottom > 9)
	      WriteTile(ME.tw, x1, bottom - 9, 0);
	    WriteTile(ME.tw, x1, bottom + 1, 0);
	    WriteTile(ME.tw, x1, bottom, (rand() % 5) + 1);
	  }

	  RefreshDirty(ME.tw);

	  // move cursor
	  bottom++;
	  if(cursorY < (short)bottom - 8)
	    cursorY++;

	  break;

        // This is the complete code for handling the game demonstration.
        case kModeDemo:
          switch(demoState)
          {
            case 0:
              // draw the title
              if(demoPt == 0)
              {
                acquire_screen();
                rectfill(screen, 0, 0, 319, 31, 16);
                rectfill(screen, 0, 192, 319, 199, 16);
                text_mode(16);
                textout_centre(screen, impact, demo[demoLine].line1,
                               160, 0, 15);
                textout_centre(screen, impact, demo[demoLine].line2,
                               160, 16, 15);
                release_screen();
              }
              // move the cursor
              hiddenCursor = 1;
    	      CopyTile(cursorX, cursorY, yScroll);
              cursorX = demo[demoLine].points[demoPt].x;
              cursorY = demo[demoLine].points[demoPt].y;
              goOn += demo[demoLine].points[demoPt].time;
              demoState = 1;
              break;

            case 1:
              if(demo[demoLine].points[demoPt].click)
              {
                goOn += demo[demoLine].points[demoPt].click;
                clicked = 1;
              }
              demoState = 2;
              break;

            case 2:
              // go to next point
              demoPt++;
              while(keypressed())
                if(readkey() >> 8 == KEY_ESC)
                  done = 1;

              if(demo[demoLine].points[demoPt].x > 11)
              {
                if(screensaving)
                  goOn += 140;
                else
                {
                  // Wait for a keypress.
                  textout_centre(screen, font, "Press a key", 160, 192, -1);
                  readkey();
                }

                // go to next line
                demoPt = 0;
                demoLine++;
                if(demo[demoLine].line1[0] == 0)
                  done = 1;
              }
              goOn = retrace_count;
              demoState = 0;
              break;
          }
          break;
      }
    }

    if(clicked && getpixel(ME.map, cursorX, cursorY)) // did user click on a tile?
    {
      short nTiles = 0, fillX, fillY;
      unsigned char tileColor;

      // look for a group of tiles
      tileColor = getpixel(ME.map, cursorX, cursorY);
      floodfill(ME.map, cursorX, cursorY, 7);

      // count and clear tiles
      for(fillY = 0; fillY < bottom; fillY++)
        for(fillX = 0; fillX < 12; fillX++)
          if(getpixel(ME.map, fillX, fillY) == 7)
          {
            WriteTile(ME.tw, fillX, fillY, 0);
            nTiles++;
          }

      // put the tile back if only one tile was there
      if(nTiles <= 1)
        WriteTile(ME.tw, cursorX, cursorY, tileColor);
      else
      {
	RefreshDirty(ME.tw);
	hiddenCursor = 1;
	Beep();
	CopyAll(yScroll);

	// now make them fall
	if(MakemFall(bottom))
	  CopyAll(yScroll);

	// make them shift in non-Rising Attack modes
        if(MakemShift(bottom))
          CopyAll(yScroll);

	// restore display
	score += (nTiles - 1) * (nTiles - 1);
	if(gameType != kModeDemo)
	  DisplayScore(score);
        ME.blocks += nTiles;
      }
    }

    // in Rising Attack, handle scrolling
    if(yScrollLeft > 0)
    {
      yScrollLeft -= 2;
      yScroll += 2;

      if(yScroll > gYSize * (ME.tw->yTileOffset + 1))
      {
	// scroll window up one tile
	ScrollTiles(ME.tw, 0, 1);

        // work behind TileWorld's back to make a continuously scrolling world
	if(bottom >= 18)
	{
          short x1, y1;

          for(y1 = 0; y1 < 11; y1++)
            for(x1 = 0; x1 < 12; x1++)
            {
              putpixel(ME.map, x1, y1, getpixel(ME.map, x1, y1 + 9));
              putpixel(ME.tw->tileFlags, x1, y1,
                        getpixel(ME.tw->tileFlags, x1, y1 + 9));
            }

	  bottom -= 9;
          ME.tw->yTileOffset -= 9;
	  yScroll -= gYSize * 9;
	  cursorY -= 9;
          rectfill(ME.map, 0, bottom, 19, 11, 0);
	}

	RefreshDirty(ME.tw);
      }

      CopyAll(yScroll);
      hiddenCursor = 1;
    }

    if(hiddenCursor) // if anything happened, the cursor disappeared.
    {
      DrawCursor(cursorX, cursorY, yScroll);
      hiddenCursor = 0;
    }
  }
  return score;
}

// fade a block of colors
void SetFadeStep(const RGB *pal, unsigned short pos,
                 unsigned short from, unsigned short to)
{
  PALETTE temp;

  fade_interpolate(black_palette, (RGB*)pal, temp, pos >> 2, from, to - 1);
  set_palette_range(temp, from, to - 1, 0);
}

// draw shadowed text
void ShadowText(const char *str, short x, short y, short color)
{
  textout(screen, small, (char *)str, x + 1, y + 1, 16); // draw shadow
  textout(screen, small, (char *)str, x, y, color); // draw text
}

static short TitleScr(short thisType)
{
  short lastType = -1;
  int   lastMove = retrace_count;
  short i;
  char  lastButtons = 0, chosen = 0;

  // display title screen
  show_mouse(NULL);
  SetFadeStep(pal, 4, 1, 256);
  clear_to_color(screen, 16);
  blit(title, screen, 0, 0, 0, 0, 320, 200);
  text_mode(-1);

  ShadowText("Allegro version by Damian Yerrick", 1, 72, 14);
  ShadowText("Keyboard: Select game type with up and down arrows.  Press Enter to play.", 1, 168, 14);

  for(i = 0; i <= kModeExit; i++)
    ShadowText(gModeNames[i], 128, 112 + i * 8, 2);

  ShadowText("Mouse: Click on a game type to play.", 1, 176, 14);
  ShadowText("To use the keyboard, move the mouse pointer to the corner.", 1, 184, 12);
  show_mouse(screen);

  screensaving = 0;
  fade_in(pal, 4);

  while(chosen <= 0)
  {
    // erase last choice and draw this one
    if(lastType != thisType)
    {
      // erase old type
      scare_mouse();
      if(lastType >= 0)
      {
	ShadowText(gModeNames[lastType], 128, 112 + lastType * 8, 2);
        lastMove = retrace_count;
      }

      // draw new type
      lastType = thisType;
      ShadowText(gModeNames[lastType], 128, 112 + lastType * 8, 15);
      unscare_mouse();
    }

    // handle keypress
    if(keypressed())
    {
      switch(readkey() >> 8)
      {
        case KEY_ESC:
          lastType = kModeExit;
          // fall through
	case KEY_ENTER:
	  chosen = 1;
	  break;
	case KEY_UP:
	  thisType--;
	  break;
	case KEY_DOWN:
	  thisType++;
	  break;
      }
    }

    if(chosen == -1)
      chosen = 1;

    // process mouse
    {
      short mouseX, mouseY;
      char buttons;

      mouseX = mouse_x;
      mouseY = mouse_y;
      buttons = mouse_b;

      if(mouseY >= 112 && mouseY < 152)
      {
	thisType = (mouseY - 112) / 8;
	if(~buttons & lastButtons)
	  chosen = 1;
      }
      lastButtons = buttons;
    }

    // wrap around
    if(thisType < 0)
      thisType += (kModeExit + 1);
    if(thisType > kModeExit)
      thisType -= (kModeExit + 1);

    if(retrace_count - lastMove > 700) // if 10 seconds have passed, run demo
    {
      lastType = kModeDemo;
      chosen = 1;
      screensaving = 1;
    }
    yield_timeslice();
  }

  fade_out(4);
  return lastType;
}

