/**************************************\
* ABKEY.C                              *
* Keyboard practice game               *
* By abigail@pineight.8m.com           *
* Copr. 1999 Damian Yerrick            *
\**************************************/

/* Notice

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation; either version 2 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be useful and fun, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the License for more details.

You should have received a copy of the License along with this program,
in the file COPYING; if not, write to the Free Software Foundation, Inc. /
59 Temple Place - Suite 330 / Boston, MA  02111-1307 / USA
or view the License online at http://www.gnu.org/copyleft/gpl.html

The author can be reached by
  usps:Damian Yerrick / Box 398 / 5500 Wabash Avenue / Terre Haute, IN 47803
  mailto:dosarena@pineight.8m.com
    http://come.to/yerrick

DOSArena is a trademark of Damian Yerrick. All other trademarks belong to
their respective owners.

*/


#define alleg_3d_unused
#define alleg_flic_unused
#define alleg_joystick_unused
#define alleg_math_unused
#define alleg_mouse_unused
#include <time.h>
#include <ctype.h>
#include "dosarena.h"

/* ABKey Documentation

At each checkpoint, you get additional seconds on your clock.  The point is to
complete the early stages quickly enough so that you build time.

Phase 1: 60 seconds; 30 letters
Press the letter that appears. (1 point per letter)

Phase 2: 30 seconds; 20 words
Type the three- or four-letter word that appears and then the space bar.
All the letters are on the home row. (3 or 4 points per word)

Phase 3: 30 seconds; infinite
Adds words not on the home row.

*/

#define N_FONTS 4
const char fontNames[N_FONTS][2][20] =
{
  {"Arial (sans-serif)", "ARIAL_PCX"},
  {"Bhiana (Gaelic)",    "GAELIC_PCX"},
  {"Fraktur (German)",   "FRAKTUR_PCX"},
  {"Wingdings (NYC)",    "WINGDING_PCX"}
};

#define N_HOMEROW 13
#define N_EASY 28
#define N_WORDS 37
const char wordList[N_WORDS][5] =
{
  "sag", "lad", "lass","sad", "fad", "had", "ask", "dad", "lag", "hag",
  "fall","flag","sass",

                       "pal", "mad", "fell","rags","jack","loss","logs",
  "kill","god", "hat", "hand","dead","dam", "hog", "keg",
                                                          "big", "home",
  "hare","hate","eat", "come","free","swap","bid"
};

DATAFILE *abdata = NULL;

const int nLeftPerPhase[] =
{
  0, 52, N_HOMEROW, N_EASY, N_WORDS
},
phaseLeftPerPhase[] =
{
  0, 30, 15, 15, 99999
},
timePerPhase[] =
{
  0, 60, 20, 20, 20
};


#define MAX_RESPONSE_LEN 15
static int PlayABKEY(int abfont)
{
  char  nextChars[52][5], answer[5], response[1 + MAX_RESPONSE_LEN];
  int   nRight = 0, nWrong = 0;
  int   phase = 1, nLeft = nLeftPerPhase[1], phaseLeft = phaseLeftPerPhase[1];
  int   numFail, x, endTime;
  short k;
  FONT  *theFont = GetResource(abdata, fontNames[abfont][1]);


  if(font == NULL)
    return -1;

  clear_to_color(screen, 15);
  text_mode(15);
  textout(screen, chicago, "ABKEY Keyboard Trainer: prepare yourself", 0, 8, 16);
  rectfill(screen, 136, 40, 184, 160, 16);
  rest(1000);
  ellipsefill(screen, 160, 60, 18, 15, 12);
  rest(1000);
  ellipsefill(screen, 160, 100, 18, 15, 14);
  rest(1000);
  ellipsefill(screen, 160, 140, 18, 15, 10);
  rest(500);

  // create the deck for Phase 1
  for(x = 0; x < 26; x++)
  {
    nextChars[x][0] = x + 'A';      // uppercase
    nextChars[x + 26][0] = x +
      ((abfont == 1) ? 'A' : 'a'); // lowercase
    nextChars[x][1] = nextChars[x + 26][1] = 0;
  }

  clear_to_color(screen, 19);
  endTime = retrace_count + 70 * timePerPhase[1]; // one minute
  srand(endTime);

  do {
    // Pick next letter.
    x = rand() % nLeft;
    nLeft--;
    strcpy(answer, nextChars[x]);
    strcpy(nextChars[x], nextChars[nLeft]);
    strcpy(nextChars[nLeft], answer);

    text_mode(-1);
    rectfill(screen, 0, 32, 319, 31 + text_height(theFont), 16);
    rectfill(screen, 0, 116, 319, 115 + text_height(theFont), 16);
    textout(screen, theFont, answer, 16, 32, -1);

    strupr(answer); // Make it uppercase for comparison.
    numFail = 0;
    x = 0;
    response[0] = 0;

    // main loop
    do {
      yield_timeslice();
      if(keypressed())
      {
        k = readkey() & 0x00ff;
        if(islower(k))
          k = toupper(k);
        if(isupper(k))
        {
          switch(phase)
          {
            case 1:
              if(k != answer[0])
              {
                endTime -= 140; // lose 2 seconds for each mistake
                numFail++;
                k = 0;
              }
              else
              {
                nRight++;
                phaseLeft--;
                k = -1;
              }
              break;

            default:
              if(x < 15)
              {
                if(abfont != 1)
                  k = tolower(k);
                response[x++] = k;
                response[x] = 0;
              }
              k = 0;
              break;
          }
        }
        else if(k == ' ') // space
        {
          switch(phase)
          {
            case 1: // phase 1: do nothing
              k = 0;
              break;

            default: // phase 2-4: set word
              if(stricmp(answer, response) == 0) // if the response is right
              {
                nRight += strlen(answer);
                phaseLeft--;
                k = -1;
              }
              else
              {
                endTime -= 140; // lose 2 seconds for each mistake
                numFail++;
                k = 0;
                x = 0;
                response[0] = 0;
              }
              break;
          }
        }
        else if(k == 8) // backspace
        {
          if(x > 0)
            response[--x] = 0;
          k = 0;
        }
        else if(k == 27) // escape
          endTime = retrace_count;
        else
          k = 0;

        yield_timeslice();
        rectfill(screen, 0, 116, 319, 115 + text_height(theFont), 16);
        text_mode(-1);
        textout(screen, theFont, response, 16, 116, -1);
      }
      else
        k = 0;
      text_mode(8);
      textprintf(screen, chicago, 0, 0, 31,
                 "ABKey phase %2d  score %5d  time %3d",
                 phase, nRight, (endTime - retrace_count) / 70);
    } while(k == 0 && numFail < 3 && endTime - retrace_count > 0);
    if(numFail == 3)                 
      nWrong++;
    if(nLeft <= 0)
      nLeft = nLeftPerPhase[phase];

    if(phaseLeft <= 0)
    {
      phase++;
      phaseLeft = phaseLeftPerPhase[phase];
      endTime += 70 * timePerPhase[phase];
      nLeft = nLeftPerPhase[phase];
      switch(phase)
      {
        case 2:
          // create the deck for Phases 2-4
          for(x = 0; x < N_WORDS; x++)
          {
            strcpy(nextChars[x], wordList[x]);      // uppercase
            if(abfont == 1)
              strupr(nextChars[x]);
          }
          textout(screen, chicago, "press space after each word", 0, 16, 15);
          break;
      }
    }

  } while(endTime - retrace_count > 0);

  // debrief
  clear_to_color(screen, 15);
  text_mode(15);
  textout_centre(screen, chicago, "ABKEY: Game Over", 160, 8, 16);
  textprintf_centre(screen, chicago, 160, 32, 16, "Score: %d", nRight);
  rest(1000);
  clear_keybuf();
  textout_centre(screen, chicago, "Press a key", 160, 48, 16);
  readkey();

  return nRight;
}

static const char *GetAbList(int index, int *list_size)
{
  if(index < 0)
  {
    *list_size = N_FONTS;
    return NULL;
  }
  else
    return fontNames[index][0];
}

void AbMain(void)
{
  int gamePick = 0;

  DIALOG dlg[] =
  {
   /* (dialog proc)     (x)   (y)   (w)   (h)   (fg)  (bg)  (key) (flags)  (d1)  (d2)  (dp) */
   { d_clear_proc,      0,    0,    0,    0,    15,   16,   0,    0,       0,    0,    NULL },
   { d_button_proc,     164,  160,  48,   12,   29,   16,   13,   D_EXIT,  0,    0,    "Go"},
   { d_button_proc,     100,  160,  48,   12,   29,   16,   27,   D_EXIT,  0,    0,    "Exit"},
   { d_list_proc,       64,   96,   208,  62,   14,   1,    0,    D_EXIT,  0,    0,    GetAbList },
   { DY_bitmap_proc,    0,    0,    320,  64,   0,    0,    0,    0,       0,    0,    NULL },
   { d_text_proc,       8,    64,   320,  16,   14,   16,   0,    0,       0,    0,    "Typing tutor by abigail@pineight.8m.com"},
   { d_text_proc,       8,    176,  320,  8,    29,   16,   0,    0,       0,    0,    "Choose a font; then be prepared to type quickly and accurately."},
   { DY_idle_proc,      0,    0,    0,    0,    0,    0,    0,    0,       0,    0,    NULL },
   { NULL,              0,    0,    0,    0,    0,    0,    0,    0,       0,    0,    NULL }
  };

  if(jukeboxMode >= 0)
    play_midi(music_dat[jukeboxMode].dat, TRUE);
  else
    play_midi(GetResource(music_dat, "etkiss_mid"), TRUE);

  clear(screen);
  abdata = load_skinned_datafile("abkey.dat");
  if(abdata == NULL)
  {
    textout(screen, chicago, "Couldn't load abkey.dat from this skin", 0, 0, 15);
    rest(1500);
    return;
  }

  dlg[4].dp = GetResource(abdata, "abtitle_pcx");
  dlg[5].dp2 = chicago;
  dlg[6].dp2 = small;

  while(gamePick >= 0)
  {
    set_palette(pal);

    // pick a game
    gamePick = do_dialog(dlg, 3);

    if(gamePick == 1 || gamePick == 3)
      gamePick = dlg[3].d1;
    else
      gamePick = -1;
    text_mode(16);

    // play the game
    if(gamePick >= 0)
    {
      int score = PlayABKEY(gamePick);

      HighScore entry =
      {
        {
          score >> 16,
          score & 0xffff,
          0, 0
        },
        ""
      };
      clear(screen);
      set_palette(pal);
      clear_keybuf();
      AddHighScore(SCORE_ABKEY, &entry);
      ShowHighScores(SCORE_ABKEY);
      textout_centre(screen, small, "Press a key", 160, 88, 4);
      readkey();
    }
  }

  unload_datafile(abdata);
}

