/**************************************\
* TILEWLD.C                            *
* Tile-based scrolling engine for      *
* DJGPP and Allegro                    *
* By Damian Yerrick                    *
\**************************************/

/* Notice

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation; either version 2 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
details.

You should have received a copy of the GNU General Public License along with
this program (look in the file COPYING); if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

/* The progress of the TileWorld library

     Last time we left TileWorld, we were using Borland's Turbo C++ 3.0, a 16-
bit compiler, and DoomVid, a mode 13H graphics library.  Now, we are using
DJGPP, a 32-bit compiler, and Allegro, its VGA graphics library.  The
functions and structures have been ported to DJGPP and Allegro.

*/

#define alleg_mouse_unused
#define alleg_joystick_unused
#define alleg_flic_unused
#define alleg_sound_unused
#define alleg_gui_unused

#include "allegro.h"
#include "tilewld.h"

/* COMPILATION OPTIONS *****/
// Turn this on if you want to make a demo.
#define kMakeDemo    0

// Turn this on because Allegro 3.1 and 3.11 have a problem with overlapping
// blits that move pixels down and to the right.
#define FIXALLEGROBUG 1

/* start of code and data */

int gSlowness = 1;

/* SetDirty() **************************
 * Set a box of tiles in map space to dirty.
 */
void SetDirtyT(TileWorld *tw, short left, short top, short right, short bottom)
{
  unsigned char *dstBuffer;
  unsigned short widCD;
  BITMAP *tf;

  tf = tw->tileFlags;

  if(left < 0)
    left = 0;
  if(top < 0)
    top = 0;
  if(right > tf->w)
    right = tf->w;
  if(bottom > tf->h)
    bottom = tf->h;
  if(right <= left || bottom <= top)
    return;

  right -= left;
  bottom -= top;

  /* After this point,
   * top means current row
   * right means width
   * bottom means height remaining
   */
  do {
    dstBuffer = tf->line[top++] + left;
    widCD = right;
    do {
      *(dstBuffer++) |= kTileDirty;
    } while (--widCD);
  } while (--bottom);
}

/* SetDirty() **************************
 * Set the tiles under a box of pixels in map pixel space to dirty.
 */
void SetDirty(TileWorld *tw, short left, short top, short right, short bottom)
{
  SetDirtyT(tw,
            left / tw->xTileSize,
            top / tw->yTileSize,
            (right - 1) / tw->xTileSize + 1,
            (bottom - 1) / tw->yTileSize + 1);

}

/* DrawTile ****************************
 * Draws a given tile from a TileWorld's font to its screen.
 */
void DrawTile(TileWorld *tw, short x, short y, short color)
{
  short width, height;

  if((x < 0) || (y < 0))
  {
     return;
  }

  width = tw->xTileSize;
  x *= width;
  height = tw->yTileSize;
  y *= height;
  if((x >= tw->tileDest->w) || (y >= tw->tileDest->h))
  {
    return;
  }

  blit(tw->tileImg, tw->tileDest, 0, color * height, x, y, width, height);
}

void RefreshDirty(TileWorld *tw)
{
  unsigned short left, top, widCD, width, height, x1 = 0, y1 = 0;
  unsigned char  *tfiPix;
  unsigned char  *tmiPix;

  //
  // 1. handle accumulated scrolling
  //
  if(tw->xMove || tw->yMove)
  {
    short xx, yy;

    width = tw->tileDest->w;
    xx = -tw->xMove * tw->xTileSize;
    if(xx < 0)
    {
      width += xx;
      left = -xx;
      xx = 0;
    } else
    {
      width -= xx;
      left = 0;
    }

    height = tw->tileDest->h;
    yy = -tw->yMove * tw->yTileSize;
    if(yy < 0)
    {
      height += yy;
      top = -yy;
      yy = 0;
    } else
    {
      height -= yy;
      top = 0;
    }

#if FIXALLEGROBUG
    if(yy > 0) // if moving everything down, Allegro has a bug.
    {
      short yts;
      int y1, y2;

      yts = tw->yTileSize;
      y1 = top + height - yts;
      y2 = yy + height - yts;

      do {
        blit(tw->tileDest, tw->tileDest, left, y1, xx, y2, width, yts);
        height -= yts;
        y1 -= yts;
        y2 -= yts;
      } while(height > 0);
    }
    else
#endif
    {
      blit(tw->tileDest, tw->tileDest, left, top, xx, yy, width, height);
    }
    tw->xMove = 0;
    tw->yMove = 0;
  }

  //
  // 2. draw tiles under sprites and newly uncovered tiles
  //
  top = tw->yTileOffset;
  left = tw->xTileOffset;

  width = tw->xScrT;
  height = tw->yScrT;
  y1 = 0;

  do {
    x1 = 0;
    tmiPix = tw->tileMap->line[top] + left;
    tfiPix = tw->tileFlags->line[top] + left;
    widCD = width;
    do {
      if (*tfiPix & kTileDirty)
      {
	*tfiPix &= ~kTileDirty;
	DrawTile(tw, x1, y1, tmiPix[x1]);
      }
      tfiPix++;
      x1++;
    } while (--widCD);
    top++;
    y1++;
  } while (--height);
}

void ScrollTiles(TileWorld *tw, short x, short y)
{
  short xOff, yOff;

  if(y)
  {
    xOff = tw->xTileOffset;
    yOff = tw->yTileOffset;

    tw->yMove += y;
    tw->yTileOffset += y;
    if(y < 0)
      SetDirtyT(tw, xOff, yOff + y, xOff + tw->xScrT, yOff);
    else
      SetDirtyT(tw, xOff, yOff + tw->yScrT,
		    xOff + tw->xScrT, yOff + tw->yScrT + y);
    if(x)
      RefreshDirty(tw);
  }

  if(x)
  {
    xOff = tw->xTileOffset;
    yOff = tw->yTileOffset;

    tw->xMove += x;
    tw->xTileOffset += x;
    if(x < 0) // moving left; all moves right
      SetDirtyT(tw, xOff + x, yOff, xOff, yOff + tw->yScrT);
    else // moving left; all moves right
      SetDirtyT(tw, xOff + tw->xScrT, yOff,
		    xOff + tw->xScrT + x, yOff + tw->yScrT);
  }
}

void ScrollTo(TileWorld *tw, short xPix, short yPix)
{
  // handle x scrolling
  short pixBase = tw->xTileOffset * tw->xTileSize;

  // keep it on the map
  if(xPix < 0)
    xPix = 0;
  if(xPix > (tw->tileMap->w - tw->xScrT) * tw->xTileSize)
  {
    xPix = (tw->tileMap->w - tw->xScrT) * tw->xTileSize + 1;
  }
  while(xPix < pixBase)
  {
    ScrollTiles(tw, -1, 0);
    pixBase -= tw->xTileSize;
  }
  while(xPix - tw->xTileSize > pixBase)
  {
    ScrollTiles(tw, 1, 0);
    pixBase += tw->xTileSize;
  }

  // handle y scrolling
  pixBase = tw->yTileOffset * tw->yTileSize;

  // keep it on the map
  if(yPix < 0)
    yPix = 0;
  if(yPix > (tw->tileMap->h - tw->yScrT) * tw->yTileSize)
    yPix = (tw->tileMap->h - tw->yScrT) * tw->yTileSize + 1;

  while(yPix < pixBase)
  {
    ScrollTiles(tw, 0, -1);
    pixBase -= tw->yTileSize;
  }
  while(yPix - tw->yTileSize > pixBase)
  {
    ScrollTiles(tw, 0, 1);
    pixBase += tw->yTileSize;
  }
}

/* SetAllDirty *************************
 * Sets all the tileFlags in a TileWorld to
 * dirty on; seen off
 */
void SetAllDirty(TileWorld *tw)
{
  clear_to_color(tw->tileFlags, kTileDirty);
}

/* DrawSprite **************************
 * Draws a sprite to the screen, marking tiles under it as dirty.
 */
void DrawSprite(TileWorld *tw, Sprite *spr)
{
  Rect theRect = {0, 0, 0, 0};
  short xOut, yOut;
  SelRect *sel;

  sel = spr->sel + spr->frame;

  // where does sprite fit in on the destination?
  theRect.left = spr->x - sel->hotX;
  theRect.right = theRect.left + sel->r.right - sel->r.left;

  theRect.top = spr->y - sel->hotY;
  theRect.bottom = theRect.top + sel->r.bottom - sel->r.top;

  // draw sprite to destination
  xOut = tw->xTileOffset * tw->xTileSize;
  yOut = tw->yTileOffset * tw->yTileSize;

  masked_blit(spr->pic, tw->tileDest,
              sel->r.left, sel->r.top, 
	      theRect.left - xOut, theRect.top - yOut,
	      sel->r.right - sel->r.left, sel->r.bottom - sel->r.top);

  // mark background under sprite as to be redrawn
  SetDirty(tw, theRect.left, theRect.top, theRect.right, theRect.bottom);
}

/* CollideSprite ***********************
 * Tells if two sprites are overlapping.  First, it intersects their
 * rectangles and, if there is any shared area, it tests their bitmaps for
 * overlap.  It's the function that was _supposed_ to be in all the game
 * programming books.
 */

char CollideSprite(Sprite *left, Sprite *right)
{
  Rect     leftSrc, rightSrc, leftDst, rightDst, dst;
  SelRect *leftSel, *rightSel;
  BITMAP  *leftBits, *rightBits;
  short    height, width;

  leftSel = left->sel + left->frame;
  leftSrc = leftSel->r;
  leftDst.left   = left->x - leftSel->hotX;
  leftDst.top    = left->y - leftSel->hotY;
  leftDst.right  = leftDst.left + leftSrc.right  - leftSrc.left;
  leftDst.bottom = leftDst.top  + leftSrc.bottom - leftSrc.top;

  rightSel = right->sel + right->frame;
  rightSrc = rightSel->r;
  rightDst.left   = right->x - rightSel->hotX;
  rightDst.top    = right->y - rightSel->hotY;
  rightDst.right  = rightDst.left + rightSrc.right  - rightSrc.left;
  rightDst.bottom = rightDst.top  + rightSrc.bottom - rightSrc.top;

  // catch if their rectangles aren't even overlapping
  if ((leftDst.left >= rightDst.right) ||
      (rightDst.left >= leftDst.right) ||
      (leftDst.top >= rightDst.bottom) ||
      (rightDst.top >= leftDst.bottom))
    return 0;

  // find intersection of bounding rectangles
  if (leftDst.left > rightDst.left)
    dst.left = leftDst.left;
  else
    dst.left = rightDst.left;
  if (leftDst.right > rightDst.right)
    dst.right = rightDst.right;
  else
    dst.right = leftDst.right;

  if (leftDst.top > rightDst.top)
    dst.top = leftDst.top;
  else
    dst.top = rightDst.top;
  if (leftDst.bottom > rightDst.bottom)
    dst.bottom = rightDst.bottom;
  else
    dst.bottom = leftDst.bottom;

  // find this intersection in the sprites themselves
  leftSrc.left  += dst.left - leftDst.left;
  rightSrc.left += dst.left - rightDst.left;
  leftSrc.top   += dst.top  - leftDst.top;
  rightSrc.top  += dst.top  - rightDst.top;

  // find these areas in RAM
  leftBits  = left->pic;
  rightBits = right->pic;

  height = dst.bottom - dst.top;
  width = dst.right - dst.left;
  do {
    short widCD = width;
    unsigned char *leftPtr, *rightPtr;

    leftPtr = leftBits->line[leftSrc.top++] + leftSrc.left;
    rightPtr = rightBits->line[rightSrc.top++] + rightSrc.left;
    do {
      if(*leftPtr && *rightPtr)
	return 1;
      leftPtr++;
      rightPtr++;
    } while (--widCD);
  } while (--height);
  return 0;
}

/* CollideRect *************************
 * Tells if a sprite is overlapping a rectangle.  It intersects the passed
 * rectangle and the sprite's rectangle and tests for pixels in the result.
 */

char CollideRect(Sprite *left, Rect *dst)
{
  Rect     leftSrc, leftDst, rightDst;
  SelRect *leftSel;
  short    height, width;
  BITMAP  *leftBits;

  leftSel = left->sel + left->frame;
  leftSrc = leftSel->r;
  leftDst.left   = left->x - leftSel->hotX;
  leftDst.top    = left->y - leftSel->hotY;
  leftDst.right  = leftDst.left + leftSrc.right  - leftSrc.left;
  leftDst.bottom = leftDst.top  + leftSrc.bottom - leftSrc.top;

  rightDst = *dst;

  // catch if their rectangles aren't even overlapping
  if ((leftDst.left >= rightDst.right) ||
      (rightDst.left >= leftDst.right) ||
      (leftDst.top >= rightDst.bottom) ||
      (rightDst.top >= leftDst.bottom))
    return 0;

  // find intersection of bounding rectangles
  if (leftDst.left > rightDst.left)
    rightDst.left = leftDst.left;
  if (leftDst.right < rightDst.right)
    rightDst.right = leftDst.right;

  if (leftDst.top > rightDst.top)
    rightDst.top = leftDst.top;
  if (leftDst.bottom < rightDst.bottom)
    rightDst.bottom = leftDst.bottom;

  // find this intersection in the sprite itself
  leftSrc.left  += rightDst.left - leftDst.left;
  leftSrc.top   += rightDst.top  - leftDst.top;

  // find this area in RAM
  leftBits = left->pic;

  height = rightDst.bottom - rightDst.top;
  width = rightDst.right - rightDst.left;
  do {
    short widCD;
    unsigned char *leftPtr;

    widCD = width;
    leftPtr = leftBits->line[leftSrc.top++] + leftSrc.left;
    do {
      if(*leftPtr++)
	return 1;
    } while (--widCD);

  } while (--height);
  return 0;
}

/* CollideArea() ***********************
 * Is a sprite overlapping any tiles?
 */
char CollideArea(Sprite *spr, TileWorld *tw)
{
  unsigned char collideDir = 0, t;
  short left, top, right, bottom, y2, x2;
  SelRect *sel;
  unsigned char *dstPtr;
  BITMAP *mapBits = tw->tileMap;

  sel = spr->sel + spr->frame;

  // where does sprite fit in on the destination?
  left = spr->x - sel->hotX;
  if(left < 0)
  {
    left = 0;
    collideDir = 1;
  }
  right = (left + sel->r.right - sel->r.left - 1) / tw->xTileSize + 1;
  if(right > mapBits->w)
  {
    right = mapBits->w;
    collideDir = 1;
  }
  left /= tw->xTileSize;

  top = spr->y - sel->hotY;
  if(top < 0)
  {
    top = 0;
    collideDir = 1;
  }
  bottom = (top + sel->r.bottom - sel->r.top - 1) / tw->yTileSize + 1;
  if(bottom > mapBits->h)
  {
    bottom = mapBits->h;
    collideDir = 1;
  }
  top /= tw->yTileSize;

  for(y2 = top; y2 < bottom; y2++)
  {
    dstPtr = mapBits->line[y2] + left;
    for(x2 = left; x2 < right; x2++)
    {
      t = *dstPtr++;
      if(gSolidTiles[t])
      {
	Rect theRect;

	theRect.top = y2 * tw->yTileSize;
	theRect.left = x2 * tw->xTileSize;
	theRect.bottom = theRect.top + tw->xTileSize;
	theRect.right = theRect.left + tw->yTileSize;

	if(CollideRect(spr, &theRect))
          collideDir |= gSolidTiles[t];
      }
    }
  }

  return collideDir;
}

/* MoveToSub() *************************
 * Move a sprite's map pixel space location to its map fixed point space
 * location.
 * in: p->xLoc and p->yLoc, 16.16 fixed point numbers of tiles from the origin
 * (top left).
 * out: p->x and p->y, integer numbers of pixels from the origin.
 */
void MoveToSub(Sprite *p, TileWorld *tw)
{
  p->x = (p->xLoc * tw->xTileSize) / 65536;
  p->y = (p->yLoc * tw->yTileSize) / 65536;
}

/* The old BounceBall() function is stupid.  It will need to be copied from
 * Adventure 99.
 */
void BounceBall(Sprite *p, TileWorld *tw1)
{
  // short xTile, yTile;

  // Newton's first law
  p->xLoc += p->xVel * gSlowness / 60;

  MoveToSub(p, tw1);

  if(CollideArea(p, tw1))
  { // bounce
    p->xVel = -p->xVel;
    p->xLoc += p->xVel * gSlowness / 60;

    MoveToSub(p, tw1);
  }

  // then in y
  p->yLoc += p->yVel * gSlowness / 60;
  MoveToSub(p, tw1);

  if(CollideArea(p, tw1))
  { // bounce
    p->yVel = -p->yVel;
    p->yLoc += p->yVel * gSlowness / 60;

    MoveToSub(p, tw1);
  }
}

void MacInvert(BITMAP *bmp, RGB *pal)
{
  int wid, ht;
  RGB myRGB;

  // invert the pixels in the bitmap (XOR with 11111111)
  if(bmp != NULL)
  {
    wid = bmp->w;
    ht = bmp->h;

    xor_mode(TRUE);
    rectfill(bmp, 0, 0, wid - 1, ht - 1, 0xff);
    solid_mode();
  }

  // reverse the order of the colors
  if(pal != NULL)
  {
    int i;
    for(i = 0; i < 128; i++)
    {
      myRGB = pal[i];
      pal[i] = pal[255 - i];
      pal[255 - i] = myRGB;
    }
  }
}

/* ActPnt2Tile() ***********************
 * Gives the (x,y) address in map space of where a sprite's action point
 * rests, and the shape of the tile under the action point.
 */
unsigned char ActPnt2Tile(Sprite *spr, TileWorld *tw, short *x, short *y)
{
  SelRect *sel;
  short x1, y1;

  // find action point
  sel = spr->sel + spr->frame;
  x1 = (spr->x - sel->hotX + sel->gunX) / tw->xTileSize;

  // return address
  if(x != NULL)
    *x = x1;
  y1 = (spr->y - sel->hotY + sel->gunY) / tw->yTileSize;
  if(y != NULL)
    *y = y1;

  // return tile
  return getpixel(tw->tileMap, x1, y1);
}

/* MoveOnto() **************************
 * Moves a sprite's hot spot onto its parent's action point.
 */
void MoveOnto(Sprite *spr, Sprite *parent)
{
  SelRect *sel;

  sel = parent->sel + parent->frame;
  spr->x = (parent->x - sel->hotX + sel->gunX);
  spr->y = (parent->y - sel->hotY + sel->gunY);
}


void WriteTile(TileWorld *tw, short x, short y, unsigned char color)
{
  putpixel(tw->tileMap, x, y, color);
  SetDirtyT(tw, x, y, x+1, y+1);
}

// create and destroy TileWorlds

TileWorld *NewTileWorld(short xTileSize, short yTileSize,
                       short xScrSize, short yScrSize, BITMAP *tileMap)
{
  TileWorld *tw;

  // allocate memory for the TileWorld structure
  tw = malloc(sizeof(TileWorld));
  if(tw == NULL)
    return NULL;

  // compute the size of the working buffer
  tw->xScrT = (xScrSize - 1) / xTileSize + 2;
  tw->yScrT = (yScrSize - 1) / yTileSize + 2;

  // allocate memory for the bitmaps
  tw->tileImg   = create_bitmap(xTileSize, yTileSize * 256);
  tw->tileFlags = create_bitmap(tileMap->w, tileMap->h);
  tw->tileDest  = create_bitmap(tw->xScrT * xTileSize, tw->yScrT * yTileSize);
  if(tw->tileImg == NULL || tw->tileFlags == NULL ||
     tw->tileDest == NULL)
  {
    FreeTileWorld(tw);
    return NULL;
  }

  // set initial values of all fields
  tw->xTileSize = xTileSize;
  tw->yTileSize = yTileSize;
  tw->xMove = tw->yMove = 0;
  tw->xTileOffset = tw->xPixOffset = tw->yTileOffset = tw->yPixOffset = 0;
  tw->tileMap = tileMap;

  SetAllDirty(tw);
  return tw;
}

void FreeTileWorld(TileWorld *tw)
{
  if(tw != NULL)
  {
    if(tw->tileImg   != NULL)
      destroy_bitmap(tw->tileImg);
    if(tw->tileDest  != NULL)
      destroy_bitmap(tw->tileDest);
    if(tw->tileFlags != NULL)
      destroy_bitmap(tw->tileFlags);
    free(tw);
  }
}


/**************************************\
* Demo of TileWld                      *
* By Damian Yerrick                    *
* Uses the gem tiles and wing boy      *
* sprite from FIGHTRIS.PCX             *
\**************************************/

#if kMakeDemo
static BITMAP *fightris;
static PALETTE pal;

static SelRect gBallRects[3] =
{
  {
    {72, 0, 84, 12},
    0, 0,
    0, 0
  },
  {
    {264, 0, 276, 12},
    0, 0,
    0, 0
  },
  {
    {396, 0, 408, 12},
    0, 0,
    0, 0
  }
};

unsigned char gSolidTiles[256] =
{
  0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1,
  0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1,
  0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1,
  0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1
};

#define METERS * 65536

SelRect gWingBoyRects[2] =
{
  {
    {0, 0, 14, 7},
    0, 0,
    0, 0
  },
  {
    {0, 7, 14, 14},
    0, 0,
    0, 0
  }
};

enum { // scroll directions
  kUp    = 0x80,
  kDown  = 0x40,
  kLeft  = 0x20,
  kRight = 0x10,
  kStop  = 0xf0,
  kLRoll = 0x08,
  kRRoll = 0x04,
  kFire2 = 0x02,
  kFire1 = 0x01
};

/* SetupClock **************************
 * Sets up a 60 Hz clock in gClock.
 */

volatile int gClock;
static int lastClockRead;

void my_timer_handler()
{
  gClock++;
}
END_OF_FUNCTION(my_timer_handler);

void SetupClock(short fps)
{
  LOCK_VARIABLE(gClock);
  gClock = 0;
  lastClockRead = 0;
  LOCK_FUNCTION(my_timer_handler);
  install_int_ex(my_timer_handler, BPS_TO_TIMER(60));
}

int FrameDone(void)
{
  int myClock;

  myClock = gClock;
  gSlowness = myClock - lastClockRead;
  lastClockRead = myClock;
  return myClock;
}

static unsigned char ReadKPad(void)
{
  char stat = 0;
  if(key[KEY_T]) // up
    stat |= kUp;
  if(key[KEY_G]) // down
    stat |= kDown;
  if(key[KEY_F]) // left
    stat |= kLeft;
  if(key[KEY_H]) // right
    stat |= kRight;
  if(key[KEY_R]) // lRoll
    stat |= kLRoll;
  if(key[KEY_Y]) // rRoll
    stat |= kRRoll;
  if(key[KEY_CONTROL]) // fire 2
    stat |= kFire2;
  if(key[KEY_LSHIFT]) // fire 1
    stat |= kFire1;

  return stat;
}

static void PlayerControl(Sprite *p, TileWorld *tw1)
{
  static unsigned char lastPad = 0;
  unsigned char collideDir;
  int i;

  // handle player actions

  // read keyboard
  collideDir = ReadKPad();

  // Newton's second law

  switch(collideDir & (kLeft | kRight))
  {
    case kLeft:
      if(p->angle == 1)
	p->xVel -= 10000 * gSlowness;
      else
      {
	short oldFrame;

	oldFrame = p->angle;
	p->frame = p->angle = 1;
	if(CollideArea(p, tw1))
	{
	  p->frame = p->angle = oldFrame;
	}
      }
      break;
    case kRight:
      if(p->angle == 0)
	p->xVel += 10000 * gSlowness;
      else
      {
	short oldFrame;

	oldFrame = p->angle;
	p->frame = p->angle = 0;
	if(CollideArea(p, tw1))
	{
	  p->frame = p->angle = oldFrame;
	}
      }
      break;
    case kLeft | kRight:
      p->xVel = (p->xVel * 3) / 4;
      break;
  }

  switch(collideDir & (kUp | kDown))
  {
    case kUp:
      if((lastPad & kUp) == 0)
      {
        p->yVel = p->yVel - 3 METERS;
      }
      break;
  }

  lastPad = collideDir;

  // drag and gravity; 9.8 m/s/s = 10704/65536 m/s/tick
  i = gSlowness;
  if(i)
  {
    do {
      p->xVel = p->xVel / 9 * 89 / 10;
      p->yVel = p->yVel / 9 * 89 / 10 + 10704;
    } while(--i);
  }

  // then do collision detection using MoveActor and CollideArea

  // Newton's first law
  collideDir = 0;
  p->xLoc += p->xVel * gSlowness / 60;
  MoveToSub(p, tw1);

  while(CollideArea(p, tw1) && !key[1])
  { // push player out of the block
    if(p->xVel < -65536 || p->xVel > 65536)
    {
      p->xVel /= 2;
      p->xLoc -= p->xVel * gSlowness / 60;
    }
    else
    {
      p->xLoc -= p->xVel * gSlowness / 60;
      collideDir = 1;
    }
    MoveToSub(p, tw1);
  }
  if(collideDir)
    p->xVel = 0;

  // then in y
  collideDir = 0;
  p->yLoc += p->yVel * gSlowness / 60;
  MoveToSub(p, tw1);

  while(CollideArea(p, tw1) && !key[1])
  { // push player out of the block
    if(p->yVel < -65536 || p->yVel > 65536)
    {
      p->yVel /= 2;
      p->yLoc -= p->yVel * gSlowness / 60;
    }
    else
    {
      p->yLoc -= p->yVel * gSlowness / 60;
      collideDir = 1;
    }
    MoveToSub(p, tw1);
  }
  if(collideDir)
    p->yVel = 0;
}

void FightrisTile(void)
{
  Rect theRect;
  short x1, y1, yout = 0, done = 0;
  char str1[256];
  short xPix = 0, yPix = 0, framesDrawn = 0;
  int nextClock, lastClock;

  TileWorld tw =
  {
    NULL, NULL, NULL, NULL,
    0, 0, // pixel offset
    0, 0, // tile offset
    0, 0, // move offset
    28, 18, // number of tiles on screen
    12, 12 // size of tiles
  };

  Sprite spr =
  {
    NULL, // pic
    gBallRects, // sizes
    12, 12, // place
    0, // frame
    0, // angle
    65536, 65536, // place
    65536, 65536  // speed
  };

  Sprite bouncing =
  {
    NULL, // pic
    gBallRects, // sizes
    124, 24, // place
    1, // frame
    0, // angle
    524288, 65536, // place
    4 METERS, 3 METERS  // speed
  };

  Sprite wingBoy =
  {
    NULL, // pic
    gWingBoyRects, // sizes
    124, 24, // place
    1, // frame
    1,
    8 METERS, 2 METERS, // place
    4 METERS, 0 // speed
  };

  tw.tileImg = create_bitmap(12, 768);
  if(!tw.tileImg)
  {
    return;
  }
  tw.tileDest  = create_bitmap(336, 216);
  if(!tw.tileDest)
  {
    destroy_bitmap(tw.tileImg);
    return;
  }
  tw.tileMap   = create_bitmap(32, 20);
  if(!tw.tileMap)
  {
    destroy_bitmap(tw.tileDest);
    destroy_bitmap(tw.tileImg);
    return;
  }
  tw.tileFlags = create_bitmap(32, 20);
  if(!tw.tileFlags)
  {
    destroy_bitmap(tw.tileMap);
    destroy_bitmap(tw.tileDest);
    destroy_bitmap(tw.tileImg);
    return;
  }
  wingBoy.pic = create_bitmap(16, 14);

  bouncing.pic = spr.pic = tw.tileImg;

// read tiles
  for(y1 = 0; y1 < 4; y1++)
  {
    theRect.top = y1 * 12;
    theRect.left = 0;
    for(x1 = 0; x1 < 16; x1++)
    {
      blit(fightris, tw.tileImg,
           theRect.left, theRect.top,
           0, yout,
           12, 12);
      yout += 12;
      theRect.left += 12;
    }
  }

// read frames for wing boy
  blit(fightris, wingBoy.pic, 192, 0, 0, 0, 16, 14);

  clear(screen);

// generate random tile map with a yellow gem border
  rectfill(tw.tileMap, 0, 0, 31, 19, 0); // the yellow border
  for(y1 = 1; y1 < 19; y1++)
    for(x1 = 1; x1 < 31; x1++)
      putpixel(tw.tileMap, x1, y1, rand() & 48);

// make something to bounce off
  for(y1 = 0; y1 < 19; y1++)
    for(x1 = 0; x1 < 31; x1++)
    {
      unsigned char g;

      g = getpixel(tw.tileMap, x1, y1);
      if((g == getpixel(tw.tileMap, x1+1, y1  )) &&
         (g == getpixel(tw.tileMap, x1,   y1+1)) &&
         (g == getpixel(tw.tileMap, x1+1, y1+1)) &&
         (g & 15) == 0)
      {
        putpixel(tw.tileMap, x1,   y1,   g | 7);
        putpixel(tw.tileMap, x1+1, y1,   g | 9);
        putpixel(tw.tileMap, x1,   y1+1, g |13);
        putpixel(tw.tileMap, x1+1, y1+1, g |15);
      }
    }

  // draw initial tiles
  SetAllDirty(&tw);

  //
  // user controlled main loop
  //
  install_keyboard(); // because getch() sucks d**k
  SetupClock(60);
  lastClock = gClock;
  nextClock = lastClock + 300;
  do {
   
    // artificial idiocy for spr
    spr.x += (rand() % 5) - 2;
    if(spr.x < 12) spr.x = 12;
    if(spr.x > 360) spr.x = 360;
    spr.y += (rand() % 5) - 2;
    if(spr.y < 12) spr.y = 12;
    if(spr.y > 216) spr.y = 216;

    // artificial idiocy for bouncing
    BounceBall(&bouncing, &tw);

    // user control for arrows
    PlayerControl(&wingBoy, &tw);

    // user control for scrolling
    if((yPix > 0) && key[72]) // up
      yPix--;
    if((xPix > 0) && key[75]) // left
      xPix--;
    if((yPix < 44) && key[80]) // down
      yPix++;
    if((xPix < 64) && key[77]) // right
      xPix++;

    done = key[1]; // esc key

    // scroll and redraw background under sprites and newly exposed areas
    ScrollTo(&tw, xPix, yPix);
    RefreshDirty(&tw);

    // draw sprites
    DrawSprite(&tw, &spr);
    DrawSprite(&tw, &bouncing);
    DrawSprite(&tw, &wingBoy);
    if(CollideSprite(&spr, &bouncing))
  
    // copy to screen
    theRect.top = yPix - tw.yTileOffset * tw.yTileSize;
    theRect.left = xPix - tw.xTileOffset * tw.xTileSize;
    blit(tw.tileDest, screen,
         xPix - tw.xTileOffset * tw.xTileSize,
         yPix - tw.yTileOffset * tw.yTileSize,
         0, 0, 320, 192);

    // manage speed; slow down system if > 60 fps
    do {
      gSlowness = gClock - lastClock;
    } while (gSlowness == 0);
    lastClock = gClock;

    framesDrawn++;
    if(lastClock >= nextClock)
    {
      nextClock = lastClock + 300;
      sprintf(str1, "[Esc = Exit]  Frames/second: %d  ", (int)(framesDrawn / 5));
      textout(screen, font, str1, 8, 192, 7);
      framesDrawn = 0;
    }
  } while (!done);
  remove_keyboard();

// clean up
  destroy_bitmap(wingBoy.pic);
  destroy_bitmap(tw.tileImg);
  destroy_bitmap(tw.tileDest);
  destroy_bitmap(tw.tileMap);
  destroy_bitmap(tw.tileFlags);
}

int main(void)
{
  puts("TileWorld Demo by Damian Yerrick\n"
     "\nTileWorld is a layer built on the DJGPP compiler and the Allegro library (both"
     "\navailable at http://www.delorie.com/ to create games (such as Mario and Zelda)"
     "\nwith sprites (moving bitmaps) on a scrolling tiled background.\n"
     "\nDemo controls"
     "\n   T"
     "\n F G H  Move wing boy   Arrows  Scroll screen   Esc  Exit demo\n"
     "\nPress space bar to start");

  allegro_init();
  if(!getch()) getch();

  if(set_gfx_mode(GFX_MODEX, 320, 240, 0, 0))
  {
    allegro_exit();
    printf("Couldn't set VGA mode X graphics because\n%s\n\n", allegro_error);
    exit(1);
  }

  clrscr();
  fightris = load_pcx("fightris.pcx", pal);
  if(!fightris)
  {
    allegro_exit();
    perror("Couldn't load FIGHTRIS.PCX");
    exit(1);
  }

  MacInvert(fightris, pal);
  set_palette(pal);

  install_timer();
  FightrisTile();

  destroy_bitmap(fightris);
  allegro_exit();

  delay(500);
  return 0;
}
#endif
