/**************************************\
* TET.C                                *
* Source code for the DOSArena module  *
* Tetanus, a clone of Alexey           *
* Pajitnov's Tetris.                   *
* Copr. 1998-99 Damian Yerrick         *
\**************************************/

/* Notice

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation; either version 2 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be useful and fun, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the License for more details.

You should have received a copy of the License along with this program,
in the file COPYING; if not, write to the Free Software Foundation, Inc. /
59 Temple Place - Suite 330 / Boston, MA  02111-1307 / USA
or view the License online at http://www.gnu.org/copyleft/gpl.html

The author can be reached by
  usps:Damian Yerrick / Box 398 / 5500 Wabash Avenue / Terre Haute, IN 47803
  mailto:dosarena@pineight.8m.com
    http://come.to/yerrick

DOSArena is a trademark of Damian Yerrick. All other trademarks belong to
their respective owners.

*/

// INCLUDES
#define alleg_3d_unused
#define alleg_joystick_unused
#define alleg_flic_unused

#include "dosarena.h"

// CONSTANTS
#define NUM_PIECES    10
#define NUM_FLIPS      4
#define NUM_BLOCKS     4

enum {
  STATE_INACTIVE = 0,
  STATE_GET_NEW_PIECE,
  STATE_FALLING_PIECE,
  STATE_CHECK4LINES,
  STATE_WAIT,
  STATE_FALL,
  STATE_PUSHUP,
  STATE_GAMEOVER,
  STATE_HISCORE
};

enum {
  PIECE_T = 0,
  PIECE_Z, PIECE_S, PIECE_J, PIECE_L, PIECE_SQUARE, PIECE_SMALL_L,
  PIECE_STICK, PIECE_TINY_STICK, PIECE_SHORT_STICK,
  PIECE_BOMB,
  PIECE_GEAR,
  PIECE_BOMBER,
  PIECE_BLOCK,
  PIECE_MAGNET,
  PIECE_INVERSE
};

// DATA TYPES

typedef struct Field
{
  signed char b[21][10];
} Field;

typedef struct Player
{
  Field blockMap, carbonMap;
  HighScore entry;

  signed char  repeatTime[8];

  signed char  curPiece, curColor, curFlip, ninja;
  signed char  nextPiece, nextColor, coming, top;
  signed char  scoreFac, chainCount, vis, state;
  signed char  pieceDone, nextSpecial, inverse, handiLines;
  signed char  spinMove, r1;

  short x, y;
  short lines, wins, hsMode;
  int   score, gameStart, stateTime;

  signed char  blkX[20];
} Player;

typedef struct TetGlobals
{
  DATAFILE *pix;
  BITMAP *tetbits, *backbits;
  MIDI *bgm;
  int endTime, doneTime;
  signed char winLimit, timeLimit;
  short handicap, yBase;
} TetGlobals;

// DATA TABLES
static const char gXBlocks[NUM_PIECES][NUM_FLIPS][NUM_BLOCKS] =
{
  {{0,1,2,1},{1,1,1,0},{2,1,0,1},{1,1,1,2}},
  {{0,1,1,2},{2,2,1,1},{2,1,1,0},{1,1,2,2}},
  {{0,1,1,2},{0,0,1,1},{2,1,1,0},{1,1,0,0}},
  {{0,1,2,2},{1,1,1,0},{2,1,0,0},{1,1,1,2}},
  {{0,0,1,2},{0,1,1,1},{2,2,1,0},{2,1,1,1}},
  {{1,1,2,2},{2,1,1,2},{2,2,1,1},{1,2,2,1}},
  {{1,1,2,2},{2,1,1,2},{2,2,1,1},{1,2,2,1}},
  {{0,1,2,3},{1,1,1,1},{3,2,1,0},{1,1,1,1}},
  {{0,1,2,3},{1,1,1,1},{3,2,1,0},{1,1,1,1}},
  {{0,1,2,0},{1,1,1,0},{2,1,0,0},{1,1,1,0}}
};

static const char gYBlocks[NUM_PIECES][NUM_FLIPS][NUM_BLOCKS] =
{
  {{1,1,1,2},{0,1,2,1},{1,1,1,0},{2,1,0,1}},
  {{1,1,2,2},{0,1,1,2},{2,2,1,1},{2,1,1,0}},
  {{2,2,1,1},{0,1,1,2},{1,1,2,2},{2,1,1,0}},
  {{1,1,1,2},{0,1,2,2},{1,1,1,0},{2,1,0,0}},
  {{2,1,1,1},{0,0,1,2},{0,1,1,1},{2,2,1,0}},
  {{1,2,2,1},{1,1,2,2},{2,1,1,2},{2,2,1,1}},
  {{1,2,2,1},{1,1,2,2},{2,1,1,2},{2,2,1,1}},
  {{2,2,2,2},{0,1,2,3},{2,2,2,2},{3,2,1,0}},
  {{2,2,2,2},{0,1,2,3},{2,2,2,2},{3,2,1,0}},
  {{2,2,2,0},{1,2,3,0},{2,2,2,0},{3,2,1,0}}
};

static const char gCBlocks[NUM_PIECES][NUM_BLOCKS] =
{
  {2,1,2,2},
  {2,2,2,2},
  {2,2,2,2},
  {2,1,2,2},
  {2,2,1,2},
  {2,2,2,2},
  {2,2,0,2},
  {2,1,1,2},
  {0,2,2,0},
  {2,1,2,0}
};

#define NUM_gTips 7
static char gTips[NUM_gTips][2][32] =
{
  {"When reducing your stack, try", "taking advantage of gravity."},
  {"Lines of all one color", "(H or R) count double."},
  {"Try sliding one block", "under another block."},
  {"Some days you win;", "some days it rains."},
  {"Baking cookies with butter (not", "margarine) improves the flavor."},
  {"To survive longer,", "lay sticks flat."},
  {"The best defense is a good", "offense; make tetrises often."}
};

static const SoundRec gFlipSound[] =
{
  {74, 1}, {0, 1}, {74, 1}, {255, 255}
};

/* thud */
static const SoundRec gDropSound[] =
{
  {42, 1}, {40, 1}, {36, 1}, {255, 255}
};

static const SoundRec gLineSound[] =
{
  {60, 1}, {57, 1}, {54, 1}, {51, 1}, {48, 1}, {255, 255}
};

static const SoundRec gTetrisSound[] =
{
  {71, 1}, {74, 1}, {77, 1}, {80, 1}, {0, 3},
  {74, 1}, {73, 1}, {72, 1}, {71, 1}, {70, 1}, {69, 1},
  {255, 255}
};

static const SoundRec gPauseSound[] =
{
  {83, 2}, {79, 2}, {83, 2}, {79, 10}, {255, 255}
};


/* GLOBAL VARIABLES */

static TetGlobals g = {NULL, NULL, NULL};
static Player p[2];
static int attackMode = 0, raceMode = 0, teflonMode = 0;
static int nPlayers = 2;
static int curTurn = 0;
#define ME p[curTurn] // to make accessing p[] easier

/* FUNCTION PROTOTYPES AND MACROS */

static char CarbonStillGoing(void);
void DrawPiece(short x, short y, short inPiece, short inFlip, short color, short style);
static void TeFloodFill(short x, short y, short c);
static void FloodFillLoop(short x, short y, short c, short uc);
static void GameLoop(void);
short MarkCarbon(int spinMove);
static void NewGame(void);
char CheckOverlap(short x, short y, short inPiece, short inFlip);
void PushUp(void);
void ScrollDown(short top, short bottom);
static void TeDrawBlock(short x, short y, short colour);
void TeShutDown(void);
short TeStartUp(void);
static short TitleScr(void);

/* main *******************************/
void TetanusMain(void)
{
  if(TeStartUp()) return;
  while(!TitleScr())
  {
    // Initialize playing field(s).
    for(curTurn = 0; curTurn <= attackMode; curTurn++)
      NewGame();

    // Display join in instructions in practice mode.
    if(!attackMode)
    {
      textout(screen, chicago, "    Tetanus", 200, 12, 15);
      textout(screen, chicago, " Practice Mode", 200, 28, 15);
      textout(screen, chicago, "   Player 2:", 200, 48, 12);
      textout(screen, chicago, "  Press Enter", 200, 64, 12);
      textout(screen, chicago, "  to join in", 204, 80, 12);
      textout(screen, chicago, "  See Options", 200, 104, 13);
      textout(screen, chicago, "for key binding", 200, 120, 5);
    }

    curTurn = 0;
    GameLoop();
  }
  TeShutDown();
}

/* CarbonStillGoing() ******************
 * Moves everything down one line if still floating.
 */
static char CarbonStillGoing(void)
{
  static int lastTime = 0;
  signed char flooded = 0;
  short x, d, e, y;

/* debugging information: enable if you touched any function with Carbon in
 * its name and your modifications make the program screw up */

/*
  for(y = 0; y <= 20; y++)
    for (x = 0; x < 10; x++)
    {
      gotoxy(x * 2 + 11, y + 1);
      cputs("%d", (int)ME.carbonMap.b[y][x]);
    }
*/

  /* move everything down if it is floating */
  for(x = 0; x < 10; x++)
  {
    e = ME.carbonMap.b[20][x];
    for(y = 20; y >= 0; y--)
    {
      d = e;
      if(y > 0)
	e = ME.carbonMap.b[y - 1][x];
      else
	e = 0;
      if((e != 1) && (d != 1))
      {
	ME.carbonMap.b[y][x] = e;
	if(y < 20)
	{
	  if (y == 0)
	    ME.blockMap.b[y][x] = 0;
	  else
	    ME.blockMap.b[y][x] = ME.blockMap.b[y - 1][x];
	  TeDrawBlock(x, y, ME.blockMap.b[y][x]);
	}
      }
    }
  }

  /* if something just hit the ground, mark it as ground */
  for(y = 19; y >= 0; y--)
    for(x = 0; x <= 9; x++)
      if((ME.carbonMap.b[y][x] > 1) && (ME.carbonMap.b[y + 1][x] == 1))
      {
	TeFloodFill(x, y, 1);
	flooded = 1;
      }

  /* if something hit the ground, play a thud */
  if(flooded)
    SendSound(gDropSound);

  /* if everything isn't ground, we're still going */
  for(y = 0; y <= 19; y++)
    for(x = 0; x < 20; x++)
      if(ME.carbonMap.b[y][x] > 1)
	return 1;

  while(lastTime == retrace_count);
  lastTime = retrace_count;  

  return 0;
}

/* CheckLines() *****************
 * Clears horizontal rows using Carbon (flood fill drops) algorithm.
 */
static char CheckLines(int spinMove)
{
  short wiener = 0, chainCount = 0;
  short x, y, hCount, rCount, grayCount;

  wiener = 0;
  for(y = 0; y <= 19; y++)
  {
    hCount = rCount = grayCount = 0;
    for(x = 0; x < 10; x++)
    {
      switch(ME.blockMap.b[y][x])
      {
        case 1:
          hCount++;
          break;
        case 2:
          rCount++;
          break;
        case 3:
          grayCount++;
          break;
      }
    }
    if(hCount + rCount + grayCount == 10)
    {
      // remove the line
      for(x = 0; x < 10; x++)
      {
        ME.blockMap.b[y][x] = 0;
        TeDrawBlock(x, y, 8);
      }

      // score the line
      if(ME.scoreFac && attackMode)
        p[1 - curTurn].coming++;
      ME.scoreFac++;
      ME.score += (int)(ME.scoreFac) * 100;
      ME.lines++;
      wiener++;

      // if H or R line, score double
      if(hCount == 10 || rCount == 10)
      {
        ME.scoreFac++;
        ME.score += (int)(ME.scoreFac) * 100;
	ME.lines++;
        wiener++;
      }
    }
  }
  if(wiener)
  {
    // Handle the combo counter.
//    textprintf(screen, chicago, chainCount * 12, 0, 12, "%d", wiener);
    chainCount++;
    SendSound(gLineSound);
    MarkCarbon(spinMove);
  }
  return wiener;
}

/* DrawPiece() ************************
 * Except for style 3, this probably won't need to be changed too much when
 * someone ports Tetanus to other platforms.  What you'll need to change is
 * TeDrawBlock.
 */
void DrawPiece(short x, short y,
               short inPiece, short inFlip, short color, short style)
{
  short n, j, k;

  switch(style)
  {
    case 0: // erase
      for(n = 0; n < NUM_BLOCKS; n++)
	if(gCBlocks[inPiece][n]) // some pieces do not have all the blocks
	  TeDrawBlock(x + gXBlocks[inPiece][inFlip][n], y + gYBlocks[inPiece][inFlip][n], 0);
      break;
    case 1: // high intensity
      for(n = 0; n < NUM_BLOCKS; n++)
	if(gCBlocks[inPiece][n])
	  TeDrawBlock(x + gXBlocks[inPiece][inFlip][n], y + gYBlocks[inPiece][inFlip][n], color + 8);
      break;
    case 2: // low intensity and copy to ME.blockMap.b
      for(n = 0; n < NUM_BLOCKS; n++)
	if(gCBlocks[inPiece][n])
	{
	  j = y + gYBlocks[inPiece][inFlip][n];
	  if (ME.top > j) ME.top = j;
	  if (j >= 0)
	  {
	    k = x + gXBlocks[inPiece][inFlip][n];
	    ME.blockMap.b[j][k] = color;
	    TeDrawBlock(k, j, ME.blockMap.b[j][k]);
	  }
	}
      break;
    case 3: // next
      // clear out the old next piece
      rectfill(screen, (curTurn ? 160 : 120), 0 + g.yBase,
               (curTurn ? 199 : 159), 23 + g.yBase, curTurn + 1);
      rectfill(screen, (curTurn ? 160 : 120), 24 + g.yBase,
               (curTurn ? 199 : 159), 47 + g.yBase, 0);

      // draw the next piece
      for(n = 0; n < NUM_BLOCKS; n++)
	if(gCBlocks[inPiece][n])
        {
	  k = gXBlocks[inPiece][inFlip][n] * 6 + (curTurn ? 168 : 128);
	  j = gYBlocks[inPiece][0][n] * 5;
	  rectfill(screen, k, j + g.yBase, k + 5, j + 4 + g.yBase,
	           (color == 2) ? 9 : 14);
        }
      if(ME.nextSpecial < 0)
        break;

      if((ME.nextSpecial & 0x0f) < 10)
      {
        inPiece = ME.nextSpecial & 0x0f;
        color = (ME.nextSpecial & 0x10) ? 2 : 1;

        for(n = 0; n < NUM_BLOCKS; n++)
          if(gCBlocks[inPiece][n])
          {
            k = gXBlocks[inPiece][inFlip][n] * 6 + (curTurn ? 168 : 128);
            j = gYBlocks[inPiece][0][n] * 5;
            rectfill(screen, k, j + 24 + g.yBase, k + 5, j + 28 + g.yBase,
                     (color == 2) ? 9 : 14);
          }
      }
      else
        blit(g.tetbits, screen, (ME.nextSpecial - 10) * 24, 40,
             (curTurn ? 168 : 128), 28 + g.yBase, 24, 20);
      break;
  }
}

/* TeFloodFill() ************************
 * This pair of functions fills an area with color.  They have been adapted to
 * the context of Tetanus by marking any contiguous block area with a given
 * unique falling block.
 */
static void TeFloodFill(short x, short y, short c)
{
  FloodFillLoop(x, y, c, ME.carbonMap.b[y][x]);
  ME.carbonMap.b[y][x] = c;
}

static void FloodFillLoop(short x, short y, short c, short fillC)
{
  short  fillL, fillR, i;
  signed char wiener = 1;

  fillL = fillR = x;
  while(wiener)
  {
    ME.carbonMap.b[y][fillL] = c;
    fillL--;
    wiener = (fillL < 0) ? 0 : (ME.carbonMap.b[y][fillL] == fillC);
  }
  fillL++;

  wiener = 1;
  while(wiener)
  {
    ME.carbonMap.b[y][fillR] = c;
    fillR++;
    wiener = (fillR > 9) ? 0 : (ME.carbonMap.b[y][fillR] == fillC);
  }
  fillR--;

  for(i = fillL; i <= fillR; i++)
  {
    if(y > 0)
      if(ME.carbonMap.b[y - 1][i] == fillC)
	FloodFillLoop(i, y - 1, c, fillC);
    if(y < 20)
      if(ME.carbonMap.b[y + 1][i] == fillC)
	FloodFillLoop(i, y + 1, c, fillC);
  }
}

static void FormatTime(signed char  *out, unsigned int seconds)
{
  unsigned int hours, minutes;

  hours = seconds / 3600;
  seconds %= 3600;
  minutes = seconds / 60;
  seconds %= 60;
  if(hours)
    sprintf(out, "%u:%02u:%02u", hours, minutes, seconds);
  else
    sprintf(out, "%u:%02u", minutes, seconds);
}

static void FormatTime10(signed char  *out, unsigned int seconds)
{
  signed char buf[16];

  FormatTime(buf, seconds / 10);
  sprintf(out, "%s.%u", buf, seconds % 10);
}

/* GameOver() **************************
 * Ends a player's game.
 */
static void GameOver(short player)
{
  short i;

  p[player].state = STATE_GAMEOVER;
  p[player].y = SCREEN_H - 1;
  for(i = 0; i < 2; i++)
    textprintf(screen, font, 124 + i * 40, 176, 14 + g.yBase,
        "%3d", p[i].wins);
}

/* GameLoop() *************************
 * Main loop for playing a two-player game of Tetanus.  All multiplayer game
 * loops must be state machines because a linear procedural game loop (used in
 * the 16-bit TET.C) is too modal.
 */
static void GameLoop(void)
{
  short i, j;
  int   lastClock = retrace_count;
  signed char  lastEnter[2] = {1, 1};
  signed char  won = 0;
  signed char  buf[256];

  g.doneTime = 0;

  while (g.doneTime == 0 || lastClock - g.doneTime < 0)
  {
    curTurn = (curTurn + 1) % nPlayers;

    if(curTurn == 0)
    {
      while(lastClock > retrace_count)
        yield_timeslice();
      lastClock++;
    }

    if(g.endTime != 0 && (lastClock - g.endTime) > 0)
      won = 1;
    if(g.winLimit != 0 && ME.wins >= g.winLimit)
      won = 1;

    if(attackMode == 0)
      ME.coming = 0;

    if(ME.state != STATE_INACTIVE)
    {
      // update score; this must be PORTED
      text_mode(16);
      textprintf(screen, font, 124 + curTurn * 16,  80 + curTurn * 8 + g.yBase, 14,
                 "%7d", ME.score);
      textprintf(screen, font, 124 + curTurn * 40, 112 + curTurn * 8 + g.yBase, 14,
                 "%3d", ME.lines);
      FormatTime10(buf, (lastClock - ME.gameStart) / 7);
      textprintf(screen, font, 124 + curTurn * 16, 144 + curTurn * 8 + g.yBase, 14,
                 "%7s", buf);
      if(ME.inverse)
      {
        text_mode(-1);
        textout(screen, small, "INVERSE", 126 + curTurn * 40, 21 + g.yBase, 14);
      }
      g.doneTime = 0;
    }

    switch(ME.state)
    {
      case STATE_INACTIVE:
        // restart key is both flips
        j = ReadKPad(curTurn) | ReadJPad(curTurn);
        i = ((j & 0x0c) == 0x0c);
        ME.coming = 0;
        if(i && !lastEnter[curTurn] && !won)
          NewGame();
        lastEnter[curTurn] = i;
        yield_timeslice();
        break;

      case STATE_GET_NEW_PIECE:
        ME.curPiece = ME.nextPiece;
        ME.curColor = ME.nextColor;

        if(ME.pieceDone == 2) // wants to put a special into action?
        {
          ME.curPiece = ME.nextSpecial & 0x0f;
          ME.curColor = (ME.nextSpecial & 0x10) ? 2 : 1;
          ME.nextSpecial = -1;
        }
        else // choose a new next piece
        {
          ME.nextPiece = rand() % (teflonMode ? NUM_PIECES - 3 : NUM_PIECES);
          ME.nextColor = (rand() & 1) + 1;
        }
        // Draw the next pieces.
        DrawPiece(0, 0, ME.nextPiece, 0, ME.nextColor, 3);

        if(ME.curPiece == PIECE_GEAR || ME.curPiece < 10)
          ME.x = 3;
        else
          ME.x = 4;
        ME.y = -3;
        ME.vis = 1;
        ME.state = STATE_FALLING_PIECE;
        ME.stateTime = retrace_count + 16;
        ME.curFlip = ME.pieceDone = 0;
        DrawPiece(ME.x, ME.y, ME.curPiece, 0, ME.curColor, 1);
        break;

      case STATE_FALLING_PIECE:
        // read the keyboard and joystick
        j = ReadKPad(curTurn) | ReadJPad(curTurn);

        MakeRepeats(ME.repeatTime, j, 18, 2);

        if(won)
          GameOver(curTurn);

        if(ME.nextSpecial >= 0 &&
           (ME.repeatTime[6] == 1 ||
            (ME.repeatTime[4] == 1 && ME.repeatTime[5] > 0) ||
            (ME.repeatTime[5] == 1 && ME.repeatTime[4] > 0))) // action button
        {
          SAMPLE *wav;

          switch(ME.nextSpecial)
          {
            case PIECE_INVERSE:
              p[1 - curTurn].inverse = 1;
              ME.nextSpecial = -1;
              wav = GetResource(g.pix, "INVERSE_WAV");
              if(wav)
                play_sample(wav, 200, curTurn ? 16 : 239, 1000, 0);
              break;
            case PIECE_MAGNET:
              // make new next piece
              ME.nextSpecial = ME.nextPiece;
              if(ME.nextColor == 2)
                ME.nextSpecial |= 0x10;
              ME.nextPiece = rand() % (teflonMode ? NUM_PIECES - 3 : NUM_PIECES);
              ME.nextColor = (rand() & 1) + 1;
              break;
            default:
              ME.repeatTime[6] = 1;
              ME.pieceDone = 2;
              break;
          }
          DrawPiece(0, 0, ME.nextPiece, 0, ME.nextColor, 3);
        }

        // flip left
        if(ME.repeatTime[4 + ME.inverse] == 1 && ME.curPiece != PIECE_SQUARE)
        {
          if(!CheckOverlap(ME.x, ME.y, ME.curPiece, (ME.curFlip + 3) & 3))
          {
            SendSound(gFlipSound);
            if(ME.vis)
              DrawPiece(ME.x, ME.y, ME.curPiece, ME.curFlip, ME.curColor, 0);
            ME.curFlip = (ME.curFlip + 3) & 3;
            ME.vis = 0;
            ME.spinMove = 1;
          }
        }

        // flip right
        if(ME.repeatTime[5 - ME.inverse] == 1 && ME.curPiece != PIECE_SQUARE)
        {
          if(!CheckOverlap(ME.x, ME.y, ME.curPiece, (ME.curFlip + 1) & 3))
  	  {
	    SendSound(gFlipSound);
	    if(ME.vis)
	      DrawPiece(ME.x, ME.y, ME.curPiece, ME.curFlip, ME.curColor, 0);
	    ME.curFlip = (ME.curFlip + 1) & 3;
	    ME.vis = 0;
            ME.spinMove = 1;
          }
        }

        if(ME.repeatTime[2 + ME.inverse] == 1 ||
           ME.repeatTime[2 + ME.inverse] == 17) // move left
        {
          if(!CheckOverlap(ME.x - 1, ME.y, ME.curPiece, ME.curFlip))
          {
            Note(1, 81, 112, 0, 1);
            if(ME.vis)
              DrawPiece(ME.x, ME.y, ME.curPiece, ME.curFlip, ME.curColor, 0);
            ME.x--;
            ME.vis = 0;
            ME.spinMove = 0;
          }
/*
          else if(gCheatCodes[2].flag && ME.repeatTime[1] != 0)
          {
            ME.stateTime = retrace_count;
            ME.ninja = 1;
          }
*/
          else
            ME.repeatTime[2] = 16; // let player slide the block in
        }

        if(ME.repeatTime[1]) // move down
        {
          if(ME.stateTime - retrace_count > 1)
          {
            ME.stateTime = retrace_count + 1;
            ME.score++;
            ME.spinMove = 0;
          }
        }

        if(ME.repeatTime[3 - ME.inverse] == 1 ||
           ME.repeatTime[3 - ME.inverse] == 17) // move right
        {
          if(!CheckOverlap(ME.x + 1, ME.y, ME.curPiece, ME.curFlip))
          {
            Note(1, 81, 112, 0, 1);
  	    if(ME.vis)
              DrawPiece(ME.x, ME.y, ME.curPiece, ME.curFlip, ME.curColor, 0);
	    ME.x++;
	    ME.vis = 0;
            ME.spinMove = 0;
          }
/*
          else if(gCheatCodes[2].flag && ME.repeatTime[1] != 0)
          {
            ME.stateTime = retrace_count;
            ME.ninja = 1;
          }
*/
          else
            ME.repeatTime[3] = 16; // let player slide the block in
        }

        if(j & 0x100) // Pause function
        {
          // stop clocks
          for(j = 0; j < 2; j++)
          {
            p[j].gameStart -= retrace_count;
            p[j].stateTime -= retrace_count;
            for(i = 0; i < 8; i++)
              p[j].repeatTime[i] = 2;
          }
          if(g.endTime)
            g.endTime -= retrace_count;
          lastClock -= retrace_count;

          // pause game
          midi_pause();
          SendSound(gPauseSound);
          if(DA_Pause())
          {
            if(attackMode)
              p[1 - curTurn].wins++;
            GameOver(curTurn);
          }
          lastEnter[curTurn] = 1;
          while(ReadKPad(curTurn) & 0x100);

          // restart clocks
          midi_resume();
          for(j = 0; j < 2; j++)
          {
            p[j].gameStart += retrace_count;
            p[j].stateTime += retrace_count;
          }
          if(g.endTime)
            g.endTime += retrace_count;
          lastClock += retrace_count;
        }

        if(ME.repeatTime[0] == 1) // drop
        {
          ME.vis = 0;
          ME.stateTime = retrace_count;
          DrawPiece(ME.x, ME.y, ME.curPiece, ME.curFlip, ME.curColor, 0);
          do
          {
            ME.y++;
            ME.score++;
            ME.spinMove--;
          } while (!CheckOverlap(ME.x, ME.y, ME.curPiece, ME.curFlip));
          ME.y--;
          ME.score--;
          ME.spinMove++;
          if(ME.spinMove < 0)
            ME.spinMove = 0;
        }

        // is it time to move the piece down 1?
        if(ME.stateTime - retrace_count <= 0)
        {
          if(CheckOverlap(ME.x, ME.y + 1, ME.curPiece, ME.curFlip) != 0 ||
             ME.ninja == 1)
	  {
            if(ME.pieceDone == 0)
              ME.pieceDone = 1;
            SendSound(gDropSound);
            if(ME.ninja == 1)
              ME.ninja = -1;
	    DrawPiece(ME.x, ME.y, ME.curPiece, ME.curFlip, ME.curColor, 2);
            ME.scoreFac = 0;
            ME.chainCount = 0;
            ME.state = STATE_CHECK4LINES;
          }
          else
	  {
	    if(ME.vis)
	      DrawPiece(ME.x, ME.y, ME.curPiece, ME.curFlip, ME.curColor, 0);
	    ME.y++;
	    ME.vis--;
            ME.spinMove = 0;
	    if(raceMode)
              ME.stateTime = retrace_count + 350 /
                             (ME.lines + ME.handiLines + 10);
	    else
	      ME.stateTime = retrace_count + 700 /
                             (ME.lines + ME.handiLines + 20);
	  }
        }

        if(ME.vis != 1)
        {
  	  DrawPiece(ME.x, ME.y, ME.curPiece, ME.curFlip, ME.curColor, 1);
	  ME.vis = 1;
        }
        break;

      case STATE_CHECK4LINES:
        if(CheckLines(ME.spinMove))
        {
          ME.state = STATE_FALL;
          ME.stateTime = retrace_count + 20;
        }
        else
          ME.state = STATE_PUSHUP;
        ME.spinMove = 0;
        break;

      case STATE_FALL:
        if(retrace_count - ME.stateTime >= 0)
        {
          ME.stateTime += 3;
          if(!CarbonStillGoing())
            ME.state = STATE_CHECK4LINES;
        }
        break;

      case STATE_PUSHUP:
        if(retrace_count - ME.stateTime >= 0)
        {
          ME.stateTime += 3;

          if(p[1].coming > p[0].coming)
          {
            p[1].coming -= p[0].coming;
            p[0].coming = 0;
          }
          else
          {
            p[0].coming -= p[1].coming;
            p[1].coming = 0;
          }

          if(ME.coming > 0)
          {
            PushUp();
            ME.coming--;
          }
          else
          {
            if(MarkCarbon(0))
              ME.state = STATE_FALL;
            else if(raceMode && ME.lines >= 40)
            {
              ME.wins++;
              GameOver(curTurn);
            }
            else if(ME.pieceDone == 0)
              ME.state = STATE_FALLING_PIECE;
            else if(ME.top >= 0)
            {
              // remove inverse
              if(ME.scoreFac >= 1)
              {
                ME.inverse = 0;

                // play tetris sound and add an item
                if(ME.scoreFac >= 4)
                {
                  SendSound(gTetrisSound);
                  // If player has nothing in the item box, add an item, but
                  // don't add an inverse in non-versus mode.
                  if(ME.nextSpecial < 0)
                    ME.nextSpecial = PIECE_MAGNET + rand() % (attackMode + 1);
                }
              }
              ME.state = STATE_GET_NEW_PIECE;
            }
            else
            {
              if(attackMode)
                p[1 - curTurn].wins++;
              GameOver(curTurn);
            }
          }
        }
        break;

      case STATE_GAMEOVER:
        text_mode(16);
        textout(screen, chicago, "   GAME OVER   ",
                curTurn ? 200 : 0, ME.y, 15);
        if((ME.y -= 2) >= g.yBase)
          break;

        for(i = 0; i < 2; i++)
          textprintf(screen, font, 124 + i * 40, 176, 14 + g.yBase,
                     "%3d", p[i].wins);
        if(ME.score != 0)
        {
          ME.hsMode = SCORE_TET;
          if(raceMode && ME.lines >= 40)
          {
            int tix = (retrace_count - ME.gameStart) * 10 / 7;

            ME.entry.score[3] = ME.score & 0xffff;
            ME.entry.score[2] = ME.score >> 16;
            ME.entry.score[1] = tix & 0xffff;
            ME.entry.score[0] = tix >> 16;
            if(IsHighScore(SCORE_TET + raceMode, &ME.entry))
              ME.hsMode = SCORE_TET_RACE;
          }
          if(ME.hsMode == SCORE_TET)
          {
            ME.entry.score[3] = (retrace_count - ME.gameStart) / 70;
            ME.entry.score[2] = ME.lines;
            ME.entry.score[1] = ME.score & 0xffff;
            ME.entry.score[0] = ME.score >> 16;
          }
        }

        // If we have a high score, set up the dialog.
        if(ME.score != 0 && IsHighScore(ME.hsMode, &ME.entry))
        {
          ME.state = STATE_HISCORE;
          strcpy(ME.entry.name, "ME?");
          ME.x = 0;
          textout(screen, small, "You have a high score.",
                  curTurn ? 200 : 0, 16 + g.yBase, 12);
          textout(screen, small, "Enter your initials with the",
                  curTurn ? 200 : 0, 24 + g.yBase, 12);
          textout(screen, small, "directions; then push both flips",
                  curTurn ? 200 : 0, 32 + g.yBase, 12);
          ME.y += 2;
          break;
        }

        // Otherwise, prepare for a rejoin.

        rectfill(screen, curTurn ? 200 : 0, 16 + g.yBase,
               curTurn ? 319 : 119, 63 + g.yBase, 16);
        textout(screen, font, "Push both flip",
                curTurn ? 204 : 4, 184 + g.yBase, 30);
        textout(screen, font, "keys to rejoin",
                curTurn ? 204 : 4, 192 + g.yBase, 28);

        // display a tip
        textout(screen, chicago, "TIP",
                curTurn ? 200 : 0, 168 + g.yBase, 8);
        i = rand() % NUM_gTips;
        text_mode(-1);
        for(j = 0; j < 2; j++)
        {
          textout(screen, small, gTips[i][j],
                  (curTurn ? 319 : 119) - text_length(small, gTips[i][j]),
                  168 + j * 8 + g.yBase, 11);
        }

        ME.state = STATE_INACTIVE;
        lastEnter[curTurn] = 1;

        if(won && p[1 - curTurn].state == STATE_INACTIVE)
          g.doneTime = retrace_count;
        else
          g.doneTime = retrace_count + 350;
        break;

      case STATE_HISCORE:
        i = curTurn ? 200 : 0;

        textout(screen, chicago, ME.entry.name, i, 48 + g.yBase, 14);
        hline(screen, ME.x * 8 + i, 63 + g.yBase, ME.x * 8 + i + 7, 15);

        // read the keyboard and joystick
        j = ReadKPad(curTurn) | ReadJPad(curTurn);

        MakeRepeats(ME.repeatTime, j, 18, 2);

        if(ME.repeatTime[2] == 1 || ME.repeatTime[4] == 1) // move left
          if(ME.x > 0)
            ME.x--;

        if(ME.repeatTime[3] == 1 || ME.repeatTime[5] == 1) // move right
          if(ME.x < 2)
            ME.x++;

        if(ME.repeatTime[0] == 1 || ME.repeatTime[0] == 17) // move up
        {
          ME.entry.name[ME.x]--;
          if(ME.entry.name[ME.x] < ' ')
            ME.entry.name[ME.x] += 64;
        }

        if(ME.repeatTime[1] == 1 || ME.repeatTime[1] == 17) // move down
        {
          ME.entry.name[ME.x]++;
          if(ME.entry.name[ME.x] > '_')
            ME.entry.name[ME.x] -= 64;
        }

        if(((j & 0x0c) == 0x0c) || (j & 2)) // exit was pressed
        {
          AddHighScore(ME.hsMode, &(ME.entry));
          ME.state = STATE_GAMEOVER;
          ME.score = 0;
        }
        break;
    }

    // update bombs here
  }

  clear_keybuf();
  if(won)
  {
    short winner;

    rectfill(screen, 0, 184, 319, 199, 16);

    if(p[0].wins > p[1].wins)      // left player won
    {
      winner = 0;
      if(g.handicap < 10)
        g.handicap++;
    }
    else if(p[1].wins > p[0].wins) // right player won
    {
      winner = 1;
      if(g.handicap > -10)
        g.handicap--;
    }
    else             // draw
      winner = -1;

    if(winner >= 0)
    {
      textout(screen, chicago, "    LOSER!     ",
              winner ? 0 : 200, 184, 15);
      rest(250);
      textout(screen, chicago, "    WINNER!    ",
              winner ? 200 : 0, 184, 15);
      play_midi(GetResource(music_dat, "WIN2_MID"), FALSE);
    }
    else
    {
      textout(screen, chicago, "     DRAW!     ",   0, 184, 15);
      textout(screen, chicago, "     DRAW!     ", 200, 184, 15);
    }

    rest(2000);
  }

  fade_out(4);
  ShowHighScores(SCORE_TET);
  fade_in(pal, 4);
  readkey();
  fade_out(4);
}

/* MarkCarbon() ***********************
 * This function marks areas of blocks as either ground or floating by filling
 * contiguous areas with a value denoting to which area each block belongs.
 */
short MarkCarbon(int spinMove)
{
  signed char markX, markY, q = 0;

  if(spinMove)
    SendSound(gTetrisSound);

  for(markY = 0; markY < 20; markY++)
    for(markX = 0; markX < 10; markX++)
      ME.carbonMap.b[markY][markX] = -(ME.blockMap.b[markY][markX] != 0);
  for(markX = 0; markX < 10; markX++)
    ME.carbonMap.b[20][markX] = -1;

  for(markY = 20; markY >= 0; markY--)
    for(markX = 0; markX < 10; markX++)
      if(ME.carbonMap.b[markY][markX] == -1)
      {
	q++;
        if(spinMove)
        {
          if(markY == 20) // the bottom row should be all 1's
            q = 1;
          ME.carbonMap.b[markY][markX] = q;
        }
        else
          TeFloodFill(markX, markY, q);
      }
  return (q > 1);
}

/* NewGame() ***************************
 * Sets up the grid and other parameters for a game of Tetanus.
 */
void NewGame(void)
{
  short x, y;

  if(midiPresent)
    play_midi(g.bgm, TRUE);

  ME.nextPiece = rand() % (teflonMode ? NUM_PIECES - 3 : NUM_PIECES);
  ME.nextColor = (rand() & 1) + 1;
  for(y = 0; y < 20; y++)
  {
    for(x = 0; x < 10; x++)
    {
      ME.blockMap.b[y][x] = 0;
      TeDrawBlock(x, y, 0);
    }
    ME.blkX[y] = -1;
  }

  ME.score = ME.lines = ME.coming = ME.ninja = ME.inverse = 0;
  ME.top = 99;
  ME.state = STATE_GET_NEW_PIECE;
  ME.nextSpecial = -1;
  srand(ME.gameStart = retrace_count);
  ME.repeatTime[6] = 5;

  textout(screen, chicago, "Score:", 128,  64 + g.yBase, 15);
  textout(screen, chicago, "Lines:", 128,  96 + g.yBase, 15);
  textout(screen, chicago, "Time:",  132, 128 + g.yBase, 15);
  textout(screen, chicago, "Wins:",  132, 160 + g.yBase, 15);

  textprintf(screen, font, 124 + curTurn * 40, 176 + g.yBase, 14, "%3d", ME.wins);

  set_palette(pal);

  // add lines for handicap
  y = g.handicap;
  if(curTurn == 1) // negative handicaps hurt player 2
    y = -y;
  if(y > 0)
    ME.handiLines = y * 10;
  else
    ME.handiLines = 0;
  while(y-- > 0)
    PushUp();

  for(y = 0; y < 8; y++)
    ME.repeatTime[y] = 1;
}

/* CheckOverlap() *********************
 * Is the piece overlapping something?
 */
char CheckOverlap(short x, short y, short inPiece, short inFlip)
{
  short g, j, k;

  for(g = 0; g < NUM_BLOCKS; g++)
    if(gCBlocks[inPiece][g]) // some pieces do not have all the blocks
    {
      j = y + gYBlocks[inPiece][inFlip][g];
      k = x + gXBlocks[inPiece][inFlip][g];
      if(k < 0 || k > 9 || j > 19)
	return 1;
      else if(j >= 0 && ME.blockMap.b[j][k])
	  return 1;
    }
  return 0;
}

void PushUp(void)
{
  short x, e, d, g = 0;

  for(x = 0; x < 10; x++)
  {
    e = 66;
    for (g = 1; g < 20; g++)
    {
      d = e;
      e = ME.blockMap.b[g][x];
      if(e != d)
      {
	ME.blockMap.b[g - 1][x] = e;
	TeDrawBlock(x, g - 1, e);
      }
    }
    ME.blockMap.b[19][x] = 3;
  }

  ME.blockMap.b[19][rand() % 10] = 0;
  for(x = 0; x < 10; x++)
    TeDrawBlock(x, 19, ME.blockMap.b[19][x]);
}

/* ScrollDown() ************************
 * Nintendo's Tetris does this instead of CarbonStillGoing(). I'll put it back
 * in when I implement Purist Mode.
 */
/*
void ScrollDown(short top, short bottom)
{
  short x, e, d, g;

  for(x = 0; x < 10; x++)
  {
    e = 66;
    for (g = bottom; g >= top; g--)
    {
      d = e;
      if(!g)
	e = 0;
      else
	e = ME.blockMap.b[g - 1][x];
      if(e != d || g == bottom)
      {
	if(e == -1)
	  e = 0;
	ME.blockMap.b[g][x] = e;
	TeDrawBlock(x, g, e);
      }
    }
  }
  top++;
}
*/

/* TeDrawBlock() **********************
 * Draws a colored block to the screen.  It's moved off this way to make the
 * program easier to port to Mac OS, MS Windows, X Windows, etc.
 */
static void TeDrawBlock(short x, short y, short colour)
{
  int yDest = y * 10 + g.yBase;

  if(y >= 0 && y < 20)
  {
    if(colour == 0 && g.backbits != NULL)
      blit(g.backbits, screen, x * 12 + 200 * curTurn, y * 10,
           x * 12 + 200 * curTurn, yDest, 12, 10);
    else
      blit(g.tetbits, screen, colour * 12, 0,
           x * 12 + 200 * curTurn, yDest, 12, 10);
  }
}

/* TeShutDown() ***********************
 * Release memory, unpatch traps, and put up exit screen.
 */
void TeShutDown(void)
{
  // free all the memory
  if(g.pix)
    unload_datafile(g.pix);
  remove_event();
}

/* TeStartUp() *************************
 * Loads Tetanus's data.
 */
short TeStartUp(void)
{
  g.pix = load_skinned_datafile("tetanus.dat");
  if(!g.pix)
  {
    textout(screen, chicago, "This skin contains no tetanus.dat", 0, 100, 0);
    readkey();
    return 32;
  }
  g.tetbits = GetResource(g.pix, "TET_PCX");
  if(!g.tetbits)
  {
    unload_datafile(g.pix);
    textout(screen, chicago, "TETANUS.DAT is corrupted", 0, 100, 0);
    readkey();
    return 32;
  }
  if(jukeboxMode >= 0)
    g.bgm = music_dat[jukeboxMode].dat;
  else
    g.bgm = GetResource(music_dat, "NEWTET_MID");

  g.backbits = GetResource(g.pix, "BKGND_PCX");
  g.yBase = (SCREEN_H - 200) / 2;

  fade_out(4);
  InitPlayEvent(136);

  return 0;
}


static int TitleScr_update(DIALOG *d, int unused1)
{
  show_mouse(NULL);

  text_mode(d[5].bg);
  g.winLimit = d[5].d2;
  if(g.winLimit)
    textprintf(screen, font, 164, 124, d[5].fg, "%2d wins ", (int)g.winLimit);
  else
    textout(screen, font, "no limit", 164, 124, d[5].fg);

  text_mode(d[6].bg);
  g.timeLimit = d[6].d2;
  if(g.timeLimit)
    textprintf(screen, font, 164, 136, d[6].fg, "%2d:00   ", (int)g.timeLimit);
  else
    textout(screen, font, "no limit", 164, 136, d[6].fg);

  text_mode(d[7].bg);
  g.handicap = d[7].d2 - 10;
  if(g.handicap < 0)
    textout(screen, font, "easy for 1P", 164, 148, d[7].fg);
  else if(g.handicap > 0)
    textout(screen, font, "easy for 2P", 164, 148, d[7].fg);
  else
    textout(screen, font, "even match ", 164, 148, d[7].fg);

  show_mouse(screen);
  return D_O_K;
}

/* TitleScr() *************************
 * Set the basic rules and win conditions for the game.
 */
static short TitleScr(void)
{
  DIALOG dlg[] =
  {
   /* (dialog proc)     (x)   (y)   (w)   (h)   (fg)  (bg)  (key) (flags)  (d1)  (d2)  (dp) */
   { d_clear_proc,      0,    0,    0,    0,    15,   16,   0,    0,       0,    0,    NULL },
   { d_button_proc,     164,  186,  48,   12,   15,   8,    13,   D_EXIT,  0,    0,    "Play"},
   { d_button_proc,     100,  186,  48,   12,   15,   8,    27,   D_EXIT,  0,    0,    "Exit"},
   { DY_check_proc,     8,    112,  128,  8,    15,   16,   'v',  0,       0,    0,    "&Vs. mode" },
   { DY_check_proc,     144,  112,  128,  8,    15,   16,   'r',  0,       0,    0,    "&Race to 40" },
   { d_slider_proc,     64,   124,  96,   8,    15,   16,   'w',  0,       20,   0,    NULL, TitleScr_update, dlg},
   { d_slider_proc,     64,   136,  96,   8,    15,   16,   'i',  0,       20,   0,    NULL, TitleScr_update, dlg},
   { d_slider_proc,     64,   148,  96,   8,    15,   16,   'h',  0,       20,   0,    NULL, TitleScr_update, dlg},
   { DY_check_proc,     8,    100,  128,  8,    15,   16,   't',  0,       0,    0,    "\"&Teflon\" mode" },

   { DY_bitmap_proc,    76,   0,    168,  32,   0,    0,    0,    0,       0,    0,    GetResource(g.pix, "TETLOGO_PCX") },
   { d_text_proc,       256,  16,   64,   16,   15,   16,   0,    0,       0,    0,    "Tetanus", chicago},
   { d_text_proc,       8,    36,   320,  8,    15,   16,   0,    0,       0,    0,    "See Options for key bindings."},
   { d_text_proc,       8,    56,   128,  8,    15,   16,   0,    0,       0,    0,
     "Rules: Use your keys or joystick to control the falling pieces. If you make a complete", small},
   { d_text_proc,       8,    64,   128,  8,    15,   16,   0,    0,       0,    0,
     "horizontal row of blocks with no gaps, it disappears, and everything above it falls.", small},
   { d_text_proc,       8,    126,  56,   8,    15,   16,   0,    0,       0,    0,    "&Win limit", small},
   { d_text_proc,       8,    138,  56,   8,    15,   16,   0,    0,       0,    0,    "T&ime limit", small},
   { d_text_proc,       8,    150,  56,   8,    15,   16,   0,    0,       0,    0,    "&Handicap", small},
   { d_text_proc,       0,    176,  128,  8,    15,   16,   0,    0,       0,    0,
   "Use arrows to move, then press Enter to play.", small},
   { NULL,              0,    0,    0,    0,    0,    0,    0,    0,       0,    0,    NULL }
  };

  DIALOG_PLAYER *player;
  short button;

  if(attackMode)
    dlg[3].flags |= D_SELECTED;
  if(raceMode)
    dlg[4].flags |= D_SELECTED;
  dlg[5].d2 = g.winLimit;
  dlg[6].d2 = g.timeLimit;
  dlg[7].d2 = g.handicap + 10;
  if(teflonMode)
    dlg[8].flags |= D_SELECTED;

  clear_keybuf();

  set_palette(black_palette);
  show_mouse(screen);
  player = init_dialog(dlg, 1);

  if(update_dialog(player))
  {
    TitleScr_update(dlg, 5);
    fade_in(pal, 2);
    while(update_dialog(player))
      ;
  }

  button = shutdown_dialog(player);
  fade_out(4);
  show_mouse(NULL);
  clear(screen);
  set_palette(pal);

  if(button == 2)
    return 1;

  attackMode = (dlg[3].flags & D_SELECTED) ? 1 : 0;
  raceMode = (dlg[4].flags & D_SELECTED) ? 1 : 0;
  teflonMode = (dlg[8].flags & D_SELECTED) ? 1 : 0;

  // If the user set a time limit, play for n min * 60 s/min * 70 retrace/s
  if(g.timeLimit)
    g.endTime = retrace_count + 4200 * dlg[6].d2;
  else
    g.endTime = 0;

  p[0].wins = p[1].wins = 0;

  return 0;
}

