/**************************************\
* KA.C                                 *
* F**by's Avalanche for DOSArena       *
* Copr. 1999 Damian Yerrick            *
\**************************************/

/* Notice

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation; either version 2 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be useful and fun, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the License for more details.

You should have received a copy of the License along with this program,
in the file COPYING; if not, write to the Free Software Foundation, Inc. /
59 Temple Place - Suite 330 / Boston, MA  02111-1307 / USA
or view the License online at http://www.gnu.org/copyleft/gpl.html

The author can be reached by
  usps:Damian Yerrick / Box 398 / 5500 Wabash Avenue / Terre Haute, IN 47803
  mailto:dosarena@pineight.8m.com
    http://come.to/yerrick

DOSArena is a trademark of Damian Yerrick. All other trademarks belong to
their respective owners.

*/

// INCLUDES
#define alleg_3d_unused
#define alleg_joystick_unused
#define alleg_flic_unused
#include <string.h>
#include "dosarena.h"

// constants
enum {
  STATE_INACTIVE,
  STATE_GET_NEW_PIECE,
  STATE_FALLING_PIECE,
  STATE_CHECK_LINES,
  STATE_FALL,
  STATE_BOULDERS,
  STATE_GAMEOVER,
  STATE_HISCORE
};

#define ATTACK_STRENGTH 128

// data structures and global variables

typedef struct KAGlobals
{
  DATAFILE *ta_dat;
  BITMAP   *kabits;
  BITMAP   *title;
  int      endTime, doneTime;
  char     attackMode, handicap, winLimit, timeLimit;
} KAGlobals;
static KAGlobals g;

typedef struct Player
{
  HighScore entry;

  int score, coming, stateTime, gameStart;

  short x, y, state, blocks;
  short wins, handiLines;

  char curFlip, vis, chainCount, r2;
  char next[2], cur[2];

  char repeatTime[8];
  char grid[12][6];
} Player;
static Player p[2];

static int curTurn = 0;
#define ME p[curTurn]

// data tables /////////////////////////

static const short flipCos[4] = {1, 0, -1, 0};
static const short flipSin[4] = {0, 1, 0, -1};

static const SoundRec gFlipSound[] =
{
  {74, 1}, {0, 1}, {74, 1}, {255, 255}
};

static const SoundRec gDropSound[] =
{
  {42, 1}, {40, 1}, {36, 1}, {255, 255}
};

static const SoundRec gPauseSound[] =
{
  {83, 2}, {79, 2}, {83, 2}, {79, 10}, {255, 255}
};

// functions ///////////////////////////

static void GameLoop(void);
static char KACheckPiece(short x, short y, short flip);
static void KADrawBlock(short x, short y, short c);
static void KADrawPiece(short x, short y, short flip, short front, short back);
static short KAFloodFill(short x, short y, short c);
static void KAExit(void);
static short KAInit(void);
static void NewGame(void);
static void Redraw(void);
static char TitleScr(void);

void KAMain(void)
{
  if(KAInit())
    return;

  while(TitleScr() == 0)
  {
    if(g.attackMode)
    {
      curTurn = 1;
      NewGame();
    }
    curTurn = 0;
    NewGame();
    GameLoop();
  }

  KAExit();
}

/* KAInit() ****************************
 * Loads the graphics for Vitamins 2.
 */
static short KAInit(void)
{
  clear(screen);
  set_palette(pal);
  g.ta_dat = load_skinned_datafile("ta.dat");
  if(g.ta_dat == NULL)
  {
    textout(screen, chicago, "This skin contains no ta.dat", 0, 0, 15);
    readkey();
    return 126;
  }

  g.kabits = GetResource(g.ta_dat, "katiles_pcx");
  g.title = GetResource(g.ta_dat, "katitle_pcx");
  if(g.kabits == NULL || g.title == NULL)
  {
    textout(screen, chicago, "ta.dat is missing something", 0, 0, 15);
    KAExit();
    readkey();
    return 126;
  }

  if(jukeboxMode >= 0)
    play_midi(music_dat[jukeboxMode].dat, TRUE);
  InitPlayEvent(136);

  return 0;
}

static void KAExit(void)
{
  if(g.ta_dat != NULL)
    unload_datafile(g.ta_dat);
}

/* CheckLines() ************************
 * Checks for groups of four in a row, then makes everything fall.
 */
static void CheckLines(void)
{
  short foundTotal, comboCount, y, x, ff, srcY;
  char k;

  if(ME.stateTime - retrace_count > 0)
    return;

  switch(ME.state)
  {
    case STATE_CHECK_LINES:
      foundTotal = 0;
      comboCount = -3;
      for(y = 0; y < 12; y++)
        for(x = 0; x < 6; x++)
        {
          k = ME.grid[y][x];
          if(k > 0 && k < 7)                 // If there is a block here...
          {
            ff = KAFloodFill(x, y, k | 16);  // look for more blocks
            if(ff >= 4)
            {
              KAFloodFill(x, y, 15);         // highlight them
              foundTotal += ff;              // Add them into the appropriate
              comboCount += ff;              // score formula
              if(ff == 4)
                comboCount--;                // Fours are worth 3
              ME.blocks += ff;
            }
          }
        }
      if(foundTotal)
        Redraw();

      // clear highlighted blocks at next redraw, and clear boulders
      for(y = 0; y < 12; y++)
        for(x = 0; x < 11; x++)
        {
          if(ME.grid[y][x] == 15)
          {
            ME.grid[y][x] = 0;
            if(x > 0 && ME.grid[y][x - 1] == 7)
              ME.grid[y][x - 1] = 0;
            if(x < 5 && ME.grid[y][x + 1] == 7)
              ME.grid[y][x + 1] = 0;
            if(y > 0 && ME.grid[y - 1][x] == 7)
              ME.grid[y - 1][x] = 0;
            if(y <11 && ME.grid[y + 1][x] == 7)
              ME.grid[y + 1][x] = 0;
          }
          else
            ME.grid[y][x] &= 0x0f;
        }

      if(foundTotal)
      {
        int now = GetNow();
        int add2Score;

        if(ME.chainCount)
          comboCount += 4 << ME.chainCount;
        if(comboCount == 0)
          comboCount = 1;
        ME.chainCount++;
        add2Score = 10 * foundTotal * comboCount;
      
        textprintf(screen, font, curTurn ? 200 : 0, 180, 15,
                   " + %4d *%5d", 10 * foundTotal, comboCount);

        ME.score += add2Score;
        if(g.attackMode)
          p[1 - curTurn].coming += add2Score;

        // play clear sound
        for(y = 0; y < 12; y += 2)
        {
          Note(1, y + 70 + 2 * ME.chainCount, 125 - 4 * y, now++, 2);
          Note(1, y + 75 + 2 * ME.chainCount, 125 - 4 * y, now++, 2);
        }

        ME.stateTime += 35;
      }

      if(foundTotal)
        ME.state = STATE_FALL;
      else
      {
        textout(screen, font, "              ", curTurn ? 200 : 0, 180, 15);
        ME.state = STATE_BOULDERS;
      }

      break;

    case STATE_FALL:
      foundTotal = 0;

      for(x = 0; x < 6; x++)
      {
        // move to first hole
        ff = 11;
        while((ff >= 0) && ME.grid[ff][x])
          ff--;

        srcY = ff;

        // move everything down
        while(srcY >= 0)
        {
          if(ME.grid[srcY][x]) // there's a tile here
          {
            // move it down
            ME.grid[ff][x] = ME.grid[srcY][x];
            ME.grid[srcY][x] = 0;

            ff--;
            foundTotal = 1;
          }
          srcY--;
        }
      }

      Redraw();
      ME.stateTime += 20;
      ME.state = STATE_CHECK_LINES;
      break;
  }
}

/* KAFloodFill() ************************
 * Couldn't use floodfill() because it doesn't count pixels, and the game
 * logic needs to know if four or more pixels have been written.
 */
static short FloodFillLoop(short x, short y, short c, short fillC)
{
  short fillL, fillR, i;
  short fillTotal;
  char wiener = 1;

  fillL = fillR = x;
  while(wiener)
  {
    ME.grid[y][fillL] = c;
    fillL--;
    wiener = (fillL < 0) ? 0 : (ME.grid[y][fillL] == fillC);
  }
  fillL++;

  wiener = 1;
  while(wiener)
  {
    ME.grid[y][fillR] = c;
    fillR++;
    wiener = (fillR > 5) ? 0 : (ME.grid[y][fillR] == fillC);
  }
  fillR--;

  fillTotal = fillR - fillL + 1;

  for(i = fillL; i <= fillR; i++)
  {
    if(y > 0)
      if(ME.grid[y - 1][i] == fillC)
        fillTotal += FloodFillLoop(i, y - 1, c, fillC);
    if(y < 11)
      if(ME.grid[y + 1][i] == fillC)
        fillTotal += FloodFillLoop(i, y + 1, c, fillC);
  }

  return fillTotal;
}

static short KAFloodFill(short x, short y, short c)
{
  short rVal = FloodFillLoop(x, y, c, ME.grid[y][x]);
  ME.grid[y][x] = c;
  return rVal;
}

static void Try2Flip(unsigned int f4)
{
  if(!KACheckPiece(ME.x, ME.y, f4))
  {
    SendSound(gFlipSound);
    if(ME.vis)
      KADrawPiece(ME.x, ME.y, ME.curFlip, 0, 0);
    ME.curFlip = f4;
    ME.vis = 0;
  }
  else
//   o
//  *O     ->    *oO
  if(!KACheckPiece(ME.x, ME.y, f4 ^ 2))
  {
    SendSound(gFlipSound);
    if(ME.vis)
      KADrawPiece(ME.x, ME.y, ME.curFlip, 0, 0);
    ME.curFlip = f4;
    ME.vis = 0;
    ME.x -= flipCos[f4];
    ME.y -= flipSin[f4];
  }
  else
//   o           oO
//  *O*    ->    * *
  if(!KACheckPiece(ME.x + flipCos[(int)ME.curFlip],
                   ME.y + flipSin[(int)ME.curFlip], f4))
  {
    SendSound(gFlipSound);
    if(ME.vis)
      KADrawPiece(ME.x, ME.y, ME.curFlip, 0, 0);
    ME.vis = 0;
    ME.x += flipCos[(int)ME.curFlip];
    ME.y += flipSin[(int)ME.curFlip];
    ME.curFlip = f4;
  }
  else
//  *o           *oO
//  *O*    ->    * *
  if(!KACheckPiece(ME.x + flipCos[(int)ME.curFlip],
                   ME.y + flipSin[(int)ME.curFlip], f4 ^ 2))
  {
    SendSound(gFlipSound);
    if(ME.vis)
      KADrawPiece(ME.x, ME.y, ME.curFlip, 0, 0);
    ME.vis = 0;
    ME.x += flipCos[(int)ME.curFlip] - flipCos[f4];
    ME.y += flipSin[(int)ME.curFlip] - flipSin[f4];
    ME.curFlip = f4;
  }
  else
//  *o*          *O*
//  *O*    ->    *o*
  {
    SendSound(gFlipSound);
    if(ME.vis)
      KADrawPiece(ME.x, ME.y, ME.curFlip, 0, 0);
    ME.vis = 0;
    ME.x += flipCos[(int)ME.curFlip];
    ME.y += flipSin[(int)ME.curFlip];
    ME.curFlip ^= 2;
  }
}

static void FormatTime(char *out, unsigned int seconds)
{
  unsigned int hours, minutes;

  hours = seconds / 3600;
  seconds %= 3600;
  minutes = seconds / 60;
  seconds %= 60;
  if(hours)
    sprintf(out, "%u:%02u:%02u", hours, minutes, seconds);
  else
    sprintf(out, "%u:%02u", minutes, seconds);
}

static void FormatTime10(char *out, unsigned int seconds)
{
  char buf[16];

  FormatTime(buf, seconds / 10);
  sprintf(out, "%s.%u", buf, seconds % 10);
}

/* GameOver() **************************
 * Ends a player's game.
 */
static void GameOver(short player)
{
  short i;

  p[player].state = STATE_GAMEOVER;
  p[player].y = 198;
  for(i = 0; i < 2; i++)
    textprintf(screen, font, 124 + i * 40, 176, 14, "%3d", p[i].wins);
}

/* GameLoop() **************************
 * Main loop for playing Furby's Avalanche.
 */
void GameLoop(void)
{
  char won = 0, lastEnter[2] = {1, 1};
  int i, j;
  int lastClock = retrace_count;

  g.doneTime = 0;
  if(g.timeLimit)
    g.endTime = retrace_count + 4200 * g.timeLimit;
  else
    g.endTime = 0;

  do {
    // take turns
    curTurn = !curTurn;
    if(curTurn == 0)
    {
      while(lastClock > retrace_count)
        yield_timeslice();
      lastClock++;
    }

    if(g.endTime != 0 && (lastClock - g.endTime) > 0)
      won = 1;
    if(g.winLimit != 0 && ME.wins >= g.winLimit)
      won = 1;

    if(ME.state != STATE_INACTIVE)
    {
      char buf[32];

      // update score; this must be PORTED
      text_mode(16);
      textprintf(screen, font, 124 + curTurn * 16,  80 + curTurn * 8, 14,
                 "%7d", ME.score);
      FormatTime10(buf, (lastClock - ME.gameStart) / 7);
      textprintf(screen, font, 124 + curTurn * 16, 144 + curTurn * 8, 14,
                 "%7s", buf);
      g.doneTime = 0;
    }

    switch(ME.state)
    {
      case STATE_INACTIVE:
        // restart key is both flips
        j = ReadKPad(curTurn + 1 - gLaptopControl) | ReadJPad(curTurn);
        i = ((j & 0x0c) == 0x0c);
        if(i && !lastEnter[curTurn] && !won)
          NewGame();
        lastEnter[curTurn] = i;
        break;

      case STATE_GET_NEW_PIECE:
        ME.x = 2;
        ME.y = 0;
        ME.vis = 0;
        ME.curFlip = 3;
        for(i = 0; i < 2; i++)
        {
          ME.cur[i] = ME.next[i];
          ME.next[i] = rand() % 5 + 1;
          KADrawBlock(curTurn ? -2 : 7, 1 - i, ME.next[i]);
        }

        ME.state = STATE_FALLING_PIECE;
        ME.stateTime = retrace_count + 16;
        break;

      case STATE_FALLING_PIECE:
        // read the keyboard and joystick
        j = ReadKPad(curTurn + 1 - gLaptopControl) | ReadJPad(curTurn);
        MakeRepeats(ME.repeatTime, j, 18, 2);

        if(won)
        {
          GameOver(curTurn);
          break;
        }

        if(ME.repeatTime[4] == 1) // flip left
          Try2Flip((ME.curFlip + 3) & 0x03);

        if(ME.repeatTime[5] == 1) // flip right
          Try2Flip((ME.curFlip + 1) & 0x03);

        if(ME.repeatTime[2] == 1 || ME.repeatTime[2] == 17) // move left
        {
          if(!KACheckPiece(ME.x - 1, ME.y, ME.curFlip))
          {
            Note(1, 81, 112, 0, 1);
            if(ME.vis)
              KADrawPiece(ME.x, ME.y, ME.curFlip, 0, 0);
            ME.x--;
            ME.vis = 0;
          }
          else
            ME.repeatTime[2] = 16; // let player slide the block in
        }

        if(ME.repeatTime[1]) // move down
        {
          if(ME.stateTime - retrace_count > 1)
          {
            ME.stateTime = retrace_count + 1;
            ME.score++;
          }
        }

        if(ME.repeatTime[3] == 1 || ME.repeatTime[3] == 17) // move right
        {
          if(!KACheckPiece(ME.x + 1, ME.y, ME.curFlip))
          {
            Note(1, 81, 112, 0, 1);
  	    if(ME.vis)
              KADrawPiece(ME.x, ME.y, ME.curFlip, 0, 0);
	    ME.x++;
	    ME.vis = 0;
          }
          else
            ME.repeatTime[3] = 16; // let player slide the block in
        }

        if(j & 0x100) // Pause function
        {
          // stop clocks
          for(j = 0; j < 2; j++)
          {
            p[j].gameStart -= retrace_count;
            p[j].stateTime -= retrace_count;
            for(i = 0; i < 8; i++)
              p[j].repeatTime[i] = 2;
          }
          if(g.endTime)
            g.endTime -= retrace_count;
          lastClock -= retrace_count;

          // pause game
          midi_pause();
          SendSound(gPauseSound);
          if(DA_Pause())
          {
            if(g.attackMode)
              p[1 - curTurn].wins++;
            GameOver(curTurn);
          }
          text_mode(16);
          while(ReadKPad(curTurn) & 0x100);
          lastEnter[curTurn] = 1;

          // restart clocks
          midi_resume();
          for(j = 0; j < 2; j++)
          {
            p[j].gameStart += retrace_count;
            p[j].stateTime += retrace_count;
          }
          if(g.endTime)
            g.endTime += retrace_count;
          lastClock += retrace_count;
        }

        if(ME.repeatTime[6] == 1) // drop
        {
          ME.vis = 0;
          ME.stateTime = retrace_count;
          KADrawPiece(ME.x, ME.y, ME.curFlip, 0, 0);
          do
          {
            ME.y++;
            ME.score++;
          } while (!KACheckPiece(ME.x, ME.y, ME.curFlip));
          ME.y--;
          ME.score--;
        }

        if(ME.stateTime - retrace_count <= 0)
        {
          if(KACheckPiece(ME.x, ME.y + 1, ME.curFlip) != 0)
	  {
            SendSound(gDropSound);

            // add the piece
            if(ME.y >= 0)
              ME.grid[ME.y][ME.x] = ME.cur[0];
            if(ME.y + flipSin[(int)ME.curFlip] >= 0)
              ME.grid[ME.y + flipSin[(int)ME.curFlip]][ME.x + flipCos[(int)ME.curFlip]]
                    = ME.cur[1];
            ME.vis = 0; // force a redraw
            ME.chainCount = 0;
            ME.state = STATE_FALL;
          }
          else
	  {
	    if(ME.vis)
	      KADrawPiece(ME.x, ME.y, ME.curFlip, 0, 0);
	    ME.y++;
	    ME.vis--;
            ME.stateTime = retrace_count + 2800 / (ME.blocks + ME.handiLines);
	  }
        }

        if(ME.vis != 1)
        {
  	  KADrawPiece(ME.x, ME.y, ME.curFlip, ME.cur[0], ME.cur[1]);
  	  ME.vis = 1;
        }
        break;

      case STATE_CHECK_LINES:
      case STATE_FALL:
        CheckLines();
        break;

      case STATE_BOULDERS:
        if(p[0].coming > p[1].coming)
        {
          p[0].coming -= p[1].coming;
          p[1].coming = 0;
        }
        else
        {
          p[1].coming -= p[0].coming;
          p[0].coming = 0;
        }

        if(ME.grid[0][2])
        {
          if(g.attackMode)
            p[1 - curTurn].wins++;
          GameOver(curTurn);
          break;
        }

        if(ME.coming < ATTACK_STRENGTH)
        {
          ME.state = STATE_GET_NEW_PIECE;
          break;
        }

        j = ME.coming / ATTACK_STRENGTH;

        for(i = 0; i < 6; i++)
        {
          if((ME.grid[0][i] == 0) &&
             (j + i >= 6 || rand() % 6 < j + i))
          {
            ME.grid[0][i] = 7;
            ME.coming -= ATTACK_STRENGTH;
            j--;
          }
          if(j <= 0)
            i = 6;
        }

        ME.stateTime += 5;
        ME.state = STATE_FALL;

        break;

      case STATE_GAMEOVER:
        text_mode(16);
        textout(screen, chicago, "   GAME OVER   ",
                curTurn ? 200 : 0, ME.y, 15);
        if((ME.y -= 2) < 0)
        {
          if(ME.score != 0)
          {
            ME.entry.score[3] = (retrace_count - ME.gameStart) / 70;
            ME.entry.score[2] = ME.blocks;
            ME.entry.score[1] = ME.score & 0xffff;
            ME.entry.score[0] = ME.score >> 16;
          }

          if(ME.score != 0 && IsHighScore(SCORE_KA, &ME.entry))
          {
            ME.state = STATE_HISCORE;
            strcpy(ME.entry.name, "ME?");
            ME.x = 0;
            textout(screen, small, "You have a high score.",
                    curTurn ? 200 : 0, 16, 12);
            textout(screen, small, "Enter your initials with the",
                    curTurn ? 200 : 0, 24, 12);
            textout(screen, small, "directions; then push both flips",
                    curTurn ? 200 : 0, 32, 12);
            ME.y += 2;
          } else
          {
            rectfill(screen, curTurn ? 200 : 0, 16,
                     curTurn ? 319 : 119, 63, 16);
            textout(screen, font, "Push both flip",
                    curTurn ? 204 : 4, 184, 30);
            textout(screen, font, "keys to rejoin",
                    curTurn ? 204 : 4, 192, 28);
/*
            // display a tip
            textout(screen, chicago, "TIP",
                    curTurn ? 200 : 0, 168, 8);
            i = rand() % NUM_gTips;
            text_mode(-1);
            for(j = 0; j < 2; j++)
            {
              textout(screen, small, gTips[i][j],
                      (curTurn ? 319 : 119) - text_length(small, gTips[i][j]),
                      168 + j * 8, 11);
            }
*/
            ME.state = STATE_INACTIVE;
            lastEnter[curTurn] = 1;

            if(won && p[1 - curTurn].state == STATE_INACTIVE)
              g.doneTime = retrace_count;
            else
              g.doneTime = retrace_count + 350;
          }
        }
        break;

      case STATE_HISCORE:
        i = curTurn ? 200 : 0;

        textout(screen, chicago, ME.entry.name, i, 48, 14);
        hline(screen, ME.x * 8 + i, 63, ME.x * 8 + i + 7, 15);

        // read the keyboard and joystick
        j = ReadKPad(curTurn + 1 - gLaptopControl) | ReadJPad(curTurn);

        MakeRepeats(ME.repeatTime, j, 18, 2);

        if(ME.repeatTime[2] == 1 || ME.repeatTime[4] == 1) // move left
          if(ME.x > 0)
            ME.x--;

        if(ME.repeatTime[3] == 1 || ME.repeatTime[5] == 1) // move right
          if(ME.x < 2)
            ME.x++;

        if(ME.repeatTime[0] == 1 || ME.repeatTime[0] == 17) // move up
        {
          ME.entry.name[ME.x]--;
          if(ME.entry.name[ME.x] < ' ')
            ME.entry.name[ME.x] += 64;
        }

        if(ME.repeatTime[1] == 1 || ME.repeatTime[1] == 17) // move down
        {
          ME.entry.name[ME.x]++;
          if(ME.entry.name[ME.x] > '_')
            ME.entry.name[ME.x] -= 64;
        }

        if(((j & 0x0c) == 0x0c) || (j & 2)) // exit was pressed
        {
          AddHighScore(SCORE_KA, &(ME.entry));
          ME.state = STATE_GAMEOVER;
          ME.score = 0;
        }
        break;
    }

  } while(g.doneTime == 0 || lastClock - g.doneTime < 0);

  clear_keybuf();
  if(won)
  {
    short winner;

    rectfill(screen, 0, 184, 319, 199, 16);

    if(p[0].wins > p[1].wins)      // left player won
    {
      winner = 0;
      if(g.handicap < 10)
        g.handicap++;
    }
    else if(p[1].wins > p[0].wins) // right player won
    {
      winner = 1;
      if(g.handicap > -10)
        g.handicap--;
    }
    else             // draw
      winner = -1;

    if(winner >= 0)
    {
      textout(screen, chicago, "    WINNER!    ",
              winner ? 200 : 0, 184, 15);
      textout(screen, chicago, "    LOSER!     ",
              winner ? 0 : 200, 184, 15);
      play_midi(GetResource(music_dat, "WIN2_MID"), FALSE);
    }
    else
    {
      textout(screen, chicago, "     DRAW!     ",   0, 184, 15);
      textout(screen, chicago, "     DRAW!     ", 200, 184, 15);
    }

    rest(2000);
  }

  fade_out(4);
  ShowHighScores(SCORE_KA);
  fade_in(pal, 4);
  readkey();
  fade_out(4);

}

static void NewGame(void)
{
  int x, y;

  for(y = 0; y < 12; y++)
    for(x = 0; x < 6; x++)
      ME.grid[y][x] = 0;
  ME.score = ME.coming = ME.blocks = 0;
  ME.stateTime = ME.gameStart = retrace_count;
  ME.state = STATE_BOULDERS;
  ME.next[0] = 1 + rand() % 5;
  ME.next[1] = 1 + rand() % 5;
  rectfill(screen, curTurn ? 200 : 0, 0, curTurn ? 319 : 199, 199, 16);
  Redraw();

  text_mode(16);
  textout(screen, chicago, "Score:", 128,  64, 15);
  textout(screen, chicago, "Lines:", 128,  96, 15);
  textout(screen, chicago, "Time:",  132, 128, 15);
  textout(screen, chicago, "Wins:",  132, 160, 15);

  for(y = 0; y < 2; y++)
    textprintf(screen, font, 124 + y * 40, 176, 14, "%3d", p[y].wins);

  // add lines for handicap
  y = g.handicap;
  if(curTurn == 1) // negative handicaps hurt player 2
    y = -y;
  if(y > 0)
  {
    ME.handiLines = 40 + y * 40;
    ME.coming = ATTACK_STRENGTH * 3 * y;
  }
  else
  {
    ME.handiLines = 40;
    ME.coming = 0;
  }

  for(y = 0; y < 8; y++)
    ME.repeatTime[y] = 1;
}

static void Redraw(void)
{
  short x, y;

  for(y = 0; y < 12; y++)
    for(x = 0; x < 6; x++)
    {
      KADrawBlock(x, y, ME.grid[y][x]);
/*
      textattr(7);
      gotoxy(x * 3 + 1, y + 1);
      printf("%d", (int)ME.grid[y][x]);
*/
    }
}


static char KACheckPiece(short x, short y, short flip)
{
  if(x < 0 || x > 5 || x + flipCos[flip] > 5 || x + flipCos[flip] < 0 ||
     y > 11 || y + flipSin[flip] > 11)
    return 1;
  if(y >= 0 && ME.grid[y][x])
    return 1;
  if(y + flipSin[flip] >= 0)
    if(ME.grid[y + flipSin[flip]][x + flipCos[flip]])
      return 1;
  return 0;
}

static void KADrawBlock(short x, short y, short c)
{
  if(y >= 0 && y < 12)
  {
    short x1 = x * 16 + (curTurn ? 212 : 12);
    short y1 = y * 14 + 7;

    blit(g.kabits, screen, 0, (c & 0x0f) * 14, x1, y1, 16, 14);
    if(c & 0x20)
      rect(screen, x1, y1, x1 + 15, y1 + 13, 7);
  }
}

static void KADrawPiece(short x, short y, short flip, short front, short back)
{
  // draw the front piece as highlighted
  KADrawBlock(x, y, front ? (front | 0x20) : 0);
  KADrawBlock(x + flipCos[flip], y + flipSin[flip], back);
}

static int Options_update(DIALOG *d, int unused1)
{
  scare_mouse();

  text_mode(d[3].bg);
  g.winLimit = d[3].d2;
  if(g.winLimit)
    textprintf(screen, font, 164, 124, d[5].fg, "%2d wins ", (int)g.winLimit);
  else
    textout(screen, font, "no limit", 164, 124, d[5].fg);

  text_mode(d[4].bg);
  g.timeLimit = d[4].d2;
  if(g.timeLimit)
    textprintf(screen, font, 164, 136, d[6].fg, "%2d:00   ", (int)g.timeLimit);
  else
    textout(screen, font, "no limit", 164, 136, d[6].fg);

  text_mode(d[5].bg);
  g.handicap = d[5].d2 - 10;
  if(g.handicap < 0)
    textout(screen, font, "easy for 1P", 164, 148, d[7].fg);
  else if(g.handicap > 0)
    textout(screen, font, "easy for 2P", 164, 148, d[7].fg);
  else
    textout(screen, font, "even match ", 164, 148, d[7].fg);

  g.attackMode = (d[2].flags & D_SELECTED) ? 1 : 0;

  unscare_mouse();
  return D_O_K;
}

static int Options2(DIALOG *unused1)
{
  show_mouse(NULL);
  Options();
  show_mouse(screen);
  return D_REDRAW;
}

static int OptionsScreen(DIALOG *unused1)
{
  DIALOG dlg[] =
  {
   /* (dialog proc)     (x)   (y)   (w)   (h)   (fg)  (bg)  (key) (flags)  (d1)  (d2)  (dp) */
   { d_clear_proc,      0,    0,    0,    0,    15,   16,   0,    0,       0,    0,    NULL },
   { d_button_proc,     192,  186,  48,   12,   15,   8,    13,   D_EXIT,  0,    0,    "OK"},
   { DY_check_proc,     8,    112,  128,  8,    15,   16,   'v',  0,       0,    0,    "&Vs. mode" },
   { d_slider_proc,     64,   124,  96,   8,    15,   16,   0,    0,       20,   0,    NULL, Options_update, dlg},
   { d_slider_proc,     64,   136,  96,   8,    15,   16,   0,    0,       20,   0,    NULL, Options_update, dlg},
   { d_slider_proc,     64,   148,  96,   8,    15,   16,   0,    0,       20,   0,    NULL, Options_update, dlg},
   { DY_action_proc,    48,   186,  64,   12,   15,   8,    'm',  D_EXIT,  0,    0,    "&More...", Options2},
   { d_text_proc,       8,    126,  56,   8,    15,   16,   0,    0,       0,    0,    "Win limit", small},
   { d_text_proc,       8,    138,  56,   8,    15,   16,   0,    0,       0,    0,    "Time limit", small},
   { d_text_proc,       8,    150,  56,   8,    15,   16,   0,    0,       0,    0,    "Handicap", small},
   { d_text_proc,       0,    0,    320,  16,   14,   16,   0,    0,       0,    0,    "VITAMINS 2: F**BY'S AVALANCHE"},
   { NULL,              0,    0,    0,    0,    0,    0,    0,    0,       0,    0,    NULL }
  };

  DIALOG_PLAYER *player;

  if(g.attackMode)
    dlg[2].flags |= D_SELECTED;
  dlg[3].d2 = g.winLimit;
  dlg[4].d2 = g.timeLimit;
  dlg[5].d2 = g.handicap + 10;
  player = init_dialog(dlg, 1);

  if(update_dialog(player))
  {
    Options_update(dlg, 0);
    while(update_dialog(player))
      ;
  }

  Options_update(dlg, 0);
  shutdown_dialog(player);

  return D_REDRAW;
}

static char TitleScr(void)
{
  DIALOG dlg[] =
  {
   /* (dialog proc)     (x)   (y)   (w)   (h)   (fg)  (bg)  (key) (flags)  (d1)  (d2)  (dp) */
   { d_clear_proc,      0,    0,    0,    0,    15,   16,   0,    0,       0,    0,    NULL },
   { d_button_proc,     192,  186,  48,   12,   15,   8,    13,   D_EXIT,  0,    0,    "Play"},
   { d_button_proc,     128,  186,  48,   12,   15,   8,    27,   D_EXIT,  0,    0,    "Exit"},
   { DY_action_proc,    48,   186,  64,   12,   15,   8,    'o',  D_EXIT,  0,    0,    "&Options", OptionsScreen},

   { DY_bitmap_proc,    0,    0,    320,  124,  0,    0,    0,    0,       0,    0,    g.title },
   { d_text_proc,       8,    124,  320,  8,    15,   16,   0,    0,       0,    0,    "See options screen for key bindings."},
   { d_text_proc,       8,    168,  320,  8,    15,   16,   0,    0,       0,    0,
     "Rules: Use your keys to control the falling pieces. Put four identical tiles together.", small},
   { d_text_proc,       0,    176,  128,  8,    15,   16,   0,    0,       0,    0,
   "Use arrows to move, then press Enter to play or Esc to exit.", small},
   { NULL,              0,    0,    0,    0,    0,    0,    0,    0,       0,    0,    NULL }
  };

  DIALOG_PLAYER *player;
  short button;

  clear_keybuf();

  set_palette(black_palette);
  show_mouse(screen);
  player = init_dialog(dlg, 1);

  if(update_dialog(player))
  {
    fade_in(pal, 4);
    while(update_dialog(player))
      ;
  }

  button = shutdown_dialog(player);
  fade_out(4);
  show_mouse(NULL);
  clear(screen);
  set_palette(pal);
  if(button == 2)
    return 1;

  p[0].wins = p[1].wins = 0;

  return 0;
}

