/**************************************\
* HISCORE.C                            *
* DOSArena high score system           *
* Copr. 1999 Damian Yerrick            *
\**************************************/

/* Notice

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation; either version 2 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be useful and fun, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the License for more details.

You should have received a copy of the License along with this program,
in the file COPYING; if not, write to the Free Software Foundation, Inc. /
59 Temple Place - Suite 330 / Boston, MA  02111-1307 / USA
or view the License online at http://www.gnu.org/copyleft/gpl.html

The author can be reached by
  usps:Damian Yerrick / Box 398 / 5500 Wabash Avenue / Terre Haute, IN 47803
  mailto:dosarena@pineight.8m.com
    http://come.to/yerrick

DOSArena is a trademark of Damian Yerrick. All other trademarks belong to
their respective owners.

*/

#include "dosarena.h"

typedef struct ScoreBlock
{
  unsigned short gameID, cheatFlags, r[6];
  HighScore      score[5];
} ScoreBlock; // 256 bytes

typedef struct ScoreType
{
  unsigned short nType[4];
  char           name[5][20];
} ScoreType;

enum {
  NTYPE_HIGHWORD = 0,
  NTYPE_SCORE,
  NTYPE_SECONDS,
  NTYPE_CSEC,
  
  NTYPE_GOLF = 0x100
};

static ScoreType types[nGames] =
{
  {
    {NTYPE_HIGHWORD, NTYPE_SCORE, NTYPE_SCORE, NTYPE_SECONDS + NTYPE_GOLF},
    {"", "Score", "Lines", "Time", "Tetanus normal"}
  },
  {
    {
      NTYPE_HIGHWORD + NTYPE_GOLF,
      NTYPE_CSEC + NTYPE_GOLF,
      NTYPE_HIGHWORD, 
      NTYPE_SCORE
    },
    {"", "Time", "", "Score", "Tetanus Race to 40"}
  },
  {
    {NTYPE_HIGHWORD, NTYPE_SCORE, NTYPE_SCORE, NTYPE_SECONDS + NTYPE_GOLF},
    {"", "Score", "Cleared", "Time", "Insane (Puzzle)"}
  },
  {
    {NTYPE_HIGHWORD, NTYPE_SCORE, NTYPE_SCORE, NTYPE_SECONDS + NTYPE_GOLF},
    {"", "Score", "Cleared", "Time", "Insane (Action)"}
  },
  {
    {NTYPE_HIGHWORD, NTYPE_SCORE, NTYPE_SCORE, NTYPE_SECONDS + NTYPE_GOLF},
    {"", "Score", "Cleared", "Time", "Insane (Rising)"}
  },
  {
    {NTYPE_HIGHWORD, NTYPE_SCORE, NTYPE_SCORE, NTYPE_SECONDS + NTYPE_GOLF},
    {"", "Score", "Cleared", "Time", "Zeus"}
  },
  {
    {NTYPE_HIGHWORD, NTYPE_SCORE, NTYPE_SCORE, NTYPE_SECONDS + NTYPE_GOLF},
    {"", "Score", "Cleared", "Time", "Vitamin 2"}
  },
  {
    {NTYPE_HIGHWORD, NTYPE_SCORE, NTYPE_HIGHWORD, NTYPE_HIGHWORD},
    {"", "Score", "", "", "ABKey"}
  }
};

#define LONGSCORE(x) (x) >> 16, (x) & 0xffff

static ScoreBlock highScore[nGames] =
{
  {
    SCORE_TET, 0, {0},
    {
      {{LONGSCORE(25180), 144,   0}, "Damian Yerrick"},
      {{LONGSCORE(16736), 102,   0}, "Sam Seabold"},
      {{LONGSCORE(13681),  98, 615}, "@$$master"},
      {{LONGSCORE( 5138),  31,   0}, "Scott Bennett"},
      {{LONGSCORE(    1),   0,   0}, "This space intentionally left blank."}
    }
  },
  {
    SCORE_TET_RACE, 0, {0},
    {
      {{LONGSCORE(15168), LONGSCORE(8299)}, "Damian Yerrick"},
      {{LONGSCORE(19724), LONGSCORE(7957)}, "Sam Seabold"},
      {{300, 0, 0, 0}, "bronze"},
      {{400, 0, 0, 0}, "fourth"},
      {{500, 0, 0, 0}, "fifth"}
    }
  },
  {
    SCORE_INSANE_PUZZLE, 0, {0},
    {
      {{0, 686,  0,   0}, "Damian Yerrick"},
      {{0, 460,  0,   0}, "Richard Smith"},
      {{0, 292, 82,  48}, "Anony Mouse"},
      {{0, 271, 72,  41}, "jeremy scchebelt"},
      {{0, 268, 95, 285}, "Laine Hillier"}
    }
  },
  {
    SCORE_INSANE_ACTION, 0, {0},
    {
      {{0, 5008,    0,  0}, "Damian Yerrick"},
      {{0, 2719, 1270,  0}, "Sam Seabold 'Materia Keeper'"},
      {{0,  897,  444,  0}, "Tom Seabold"},
      {{0,  292,  143, 49}, "Brian Woodbury"},
      {{0,  288,  136, 48}, "Eric 'Titz' Titzer"}
    }
  },
  {
    SCORE_INSANE_TA, 0, {0},
    {
      {{0, 2947, 1320,   0}, "Sam Seabold"},
      {{0, 1993, 1077,   0}, "Damian Yerrick"},
      {{0, 1359,  594, 169}, "andy MORel"},
      {{0,  653,  323,   0}, "rico"},
      {{0,  608,  256,  69}, "Justin Evoy"}
    }
  },
  {
    SCORE_ZEUS, 0, {0},
    {
      {{LONGSCORE(126100), 395, 366}, "Damian Yerrick"},
      {{LONGSCORE( 91800), 394,   0}, "Sam Seabold"},
      {{LONGSCORE(   300),   0,   0}, "noddy"},
      {{LONGSCORE(   200),   0,   0}, "looks like"},
      {{LONGSCORE(   100),   0,   0}, "pinocchio"},
    }
  },
  {
    SCORE_KA, 0, {0},
    {
      {{LONGSCORE(16963), 360, 350}, "Damian Yerrick"},
      {{LONGSCORE(12390), 345, 357}, "Sam Seabold"},
      {{LONGSCORE( 4568), 169, 292}, "(19"},
      {{LONGSCORE( 4452),  72, 105}, "Ku Klux Klan"},
      {{LONGSCORE(    0),   0,   0}, "How much wood could a woodchuck chuck.."},
    }
  },
  {
    SCORE_ABKEY, 0, {0},
    {
      {{LONGSCORE(344),   0, 0}, "Damian Yerrick"},
      {{LONGSCORE(321),   0, 0}, "Kyle Thomas"},
      {{LONGSCORE(313),   0, 0}, "abigail"},
      {{LONGSCORE(232),   0, 0}, "Leo Morand"},
      {{LONGSCORE(143),   0, 0}, "El Chulo"}
    }
  }
};

static int golf[4];

// fns ///////////////////////////////////////////////////////////////////////

/* CompareHighScores() *****************
 * Comparison function for qsort()
 */
static int CompareHighScores(const void *c1, const void *c2)
{
  short i;

  const HighScore *left = c1;
  const HighScore *right = c2;

  for(i = 0; i < 4; i++)
  {
    if(left->score[i] < right->score[i])
      return golf[i];
    else if(left->score[i] > right->score[i])
      return -golf[i];
  }
  return 0;
}

static void SortHighScores(unsigned short gameID)
{
  short i;

  for(i = 0; i < 4; i++)
    golf[i] = (types[gameID].nType[i] & NTYPE_GOLF) ? -1 : 1;

  qsort(highScore[gameID].score, 5, sizeof(HighScore), CompareHighScores);
}

static void FormatTime(char *out, unsigned int seconds)
{
  unsigned int hours, minutes;

  hours = seconds / 3600;
  seconds %= 3600;
  minutes = seconds / 60;
  seconds %= 60;
  if(hours)
    sprintf(out, "%u:%02u:%02u", hours, minutes, seconds);
  else
    sprintf(out, "%u:%02u", minutes, seconds);
}

static void FormatTime100(char *out, unsigned int seconds)
{
  char buf[16];

  FormatTime(buf, seconds / 100);
  sprintf(out, "%s.%02u", buf, seconds % 100);
}

static void WriteScores(void)
{
  FILE *fp = fopen("dosarena.sco", "wb");

  if(fp == NULL)
    return;

  fwrite(highScore, nGames, sizeof(ScoreBlock), fp);
  fclose(fp);
}

/* IsHighScore() ***********************
 * Is this score a high score?  Use this function in games (e.g. Tetanus)
 * where you have to know before you let the player enter his or her initials.
 */
char IsHighScore(unsigned short gameID, HighScore *entry)
{
  SortHighScores(gameID);
  return (CompareHighScores(entry, &highScore[gameID].score[4]) < 0);
}

/* ShowHighScores() ********************
 * Shows the high scores for one game.
 */
void ShowHighScores(unsigned short gameID)
{
  int i, j, y;
  unsigned int thisScore = 0;
  char lastWas32;
  char buf[32];

  SortHighScores(gameID);
  clear_to_color(screen, 15);
  text_mode(15);
  
  // draw name of game
  textout_centre(screen, chicago, types[gameID].name[4], 160, 4, 16);

  // draw column heads
  for(i = 0; i < 4; i++)
    textout(screen, small, types[gameID].name[i], 160 + i * 40, 28, 1);

  for(j = 0; j < 5; j++)
  {
    // draw player name
    y = j * 8 + 40;
    textout(screen, small, highScore[gameID].score[j].name, 2, y, 16 + j * 2);

    // draw scores
    lastWas32 = 0;
    for(i = 0; i < 4; i++)
    {
      int len;
      
      // figure high word
      if(lastWas32)
        thisScore = (thisScore << 16) | highScore[gameID].score[j].score[i];
      else
        thisScore = highScore[gameID].score[j].score[i];
      lastWas32 = 0;

      switch(types[gameID].nType[i] & 0x00ff)
      {
        case NTYPE_HIGHWORD:
          lastWas32 = 1;
          buf[0] = 0;
          break;
        case NTYPE_SCORE:
          sprintf(buf, "%d", thisScore);
          break;
        case NTYPE_SECONDS:
          FormatTime(buf, thisScore);
          break;
        case NTYPE_CSEC:
          FormatTime100(buf, thisScore);
          break;
      }
      // print the score
      len = text_length(small, buf);
      textout(screen, small, buf, 198 + 40 * i - len, y, 16 + j * 2);
    }
  }
  yield_timeslice();
}

/* AddHighScore() **********************
 * If a score is good enough, it prompts the player to enter his or her name.
 */
void AddHighScore(unsigned short gameID, HighScore *entry)
{
  DIALOG dlg[] =
  {
   // (dialog proc)     (x)   (y)   (w)   (h)   (fg)  (bg)  (key) (flags)  (d1)  (d2)  (dp)
   { d_button_proc,     164,  186,  48,   12,   1,    14,   13,   D_EXIT,  0,    0,    "OK"},
   { d_edit_proc,       4,    176,  312,  8,    30,   8,    0,    0,       39,   0,    entry->name },
   { d_text_proc,       4,    164,  60,   8,    16,   15,   0,    0,       0,    0,    "Enter your name:"},
   { DY_idle_proc,      0,    0,    0,    0,    0,    0,    0,    0,       0,    0,    NULL },
   { NULL,              0,    0,    0,    0,    0,    0,    0,    0,       0,    0,    NULL }
  };
  short i;

  // Set up the comparison function.
  SortHighScores(gameID);

  // Negative indicates a better score.
  if(CompareHighScores(entry, &highScore[gameID].score[4]) < 0)
  {
    // If the caller wants to put up the name dialog...
    if(entry->name[0] == 0)
    {
      // Add the high score.
      highScore[gameID].score[4] = *entry;
      highScore[gameID].score[4].name[0] = 26;
      highScore[gameID].score[4].name[1] = 0;
      ShowHighScores(gameID);

      // Ask for player's name.
      do_dialog(dlg, 1);

      // Stick it in.
      for(i = 0; i < 5; i++)
        if(highScore[gameID].score[i].name[0] == 26)
          highScore[gameID].score[i] = *entry;
    }
    else // if the caller already has a name...
      highScore[gameID].score[4] = *entry;
  }
  WriteScores();
}

/* ShowAllHighScores() *****************
 * Shows all the high scores.
 */
void ShowAllHighScores(void)
{
  short i = 0;
  char done = 0;

  while(done == 0)
  {
    ShowHighScores(i);
    textout_centre(screen, small, "Push left and right to switch or Esc to exit.",
                   160, 88, 4);
    switch(readkey() >> 8)
    {
      case KEY_ENTER:
      case KEY_ESC:
        done = 1;
        break;
      case KEY_LEFT:
        i--;
        if(i < 0)
          i += nGames;
        break;
      case KEY_RIGHT:
        i++;
        if(i >= nGames)
          i -= nGames;
        break;
    }
  }
}

void ReadScores(void)
{
  ScoreBlock blk;
  FILE *fp = fopen("dosarena.sco", "rb");

  if(fp == NULL)
    return;

  // read each high score block...
  while(fread(&blk, sizeof(blk), 1, fp))
  {
    // if it's one of ours, write it
    if(blk.gameID < nGames)
      highScore[blk.gameID] = blk;
  }
  fclose(fp);
}

unsigned short GetCheats(unsigned short gameID)
{
  return highScore[gameID].cheatFlags;
}

void SetCheats(unsigned short gameID, unsigned short cheats)
{
  highScore[gameID].cheatFlags = cheats;
}
