/*

    guwame.c:  Grand Unified Whack A Mole Engine
    Copyright  2000 Damian Yerrick

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    mailto:guwame@pineight.8m.com
      http://come.to/yerrick

*/

#include "dosarena.h"

// Set this to the number of moles necessary to stop a bullet.
#define SHOOT_THROUGH 2

// Set to define the maximum number of moles on screen.
#define MAX_MOLES 36

static int mrsd_times[34] = {
     0,  215,  430,  645,  845, 1040, 1140, 1220,
  1285, 1350, 1415, 1470, 1525, 1580, 1635, 1690,
  1740, 1790, 1840, 1886, 1933, 1973, 2013, 2054,
  2095, 2133, 2171, 2206, 2240, 2270, 2300, 2330,
  2360, -1
};

void clear2color(BITMAP *bmp, int color);
void klear(BITMAP *bmp);

enum {
  STATE_INACTIVE,
  STATE_MOLE,
  STATE_DYING
};

typedef struct MoleCell
{
  int x, y;
  int moleType;
  int state, clock;
} MoleCell;

typedef struct Globals
{
  int hit, miss;
  int offX, offY;
  int lastBeat;
  DATAFILE *dat;
  RGB *pal;
  FONT *lucidmono, *lucid;
  BITMAP *backbuffer;
  BITMAP *icandy[2];
  BITMAP *molePix[5];

  MoleCell cells[MAX_MOLES];
} Globals;

typedef struct Timers
{
  int odo, trip;
} Timers;



static Globals g;
static volatile Timers tim;

static void guwame_timerint(void)
{
  tim.odo++;
  tim.trip++;
}
END_OF_FUNCTION(guwame_timerint);

static void install_guwame_timerint()
{
  LOCK_FUNCTION(guwame_timerint);
  LOCK_VARIABLE(tim);
  install_int(guwame_timerint, 10);
}

int MoleCompare(const void *left, const void *right)
{
  return ((MoleCell *)left)->y - ((MoleCell *)right)->y;
}

static const char copy_msg[12][2][24] =
{
  {
    "I   a m   a n   e x i t", ""
  },
  {
    "They're waiting for the", "right moment to attack."
  },
  {
    "v i s i t", "p i n  e i g h t . c o m"
  },
  {
    "Music by Disney and", "nine inch noses"
  },
  {
    "Graphics borrowed from", "the World Wide Web"
  },
  {
    "Code by Damian Yerrick", "None stolen"
  },
  {
    "", "(don't go away)"
  },
  {
    "Why is 'p' in Hampster?", "Why is 'a' in Pearson?"
  }
};

static void CopyMessageTo(BITMAP *dest, int n, int c)
{
  textout_centre(dest, g.lucid, (char *)copy_msg[n][0], 128,  76, c);
  textout_centre(dest, g.lucid, (char *)copy_msg[n][1], 128, 100, c);
}

static void CopyMessage(unsigned now)
{
  static unsigned lastTime = 1000;
  static unsigned curMsg;

  now /= 32;
  if(now != lastTime)
    {
      if((now & 1) == 0)
	CopyMessageTo(g.icandy[0], curMsg, 255);
      else
      switch(now)
      {
        case 18:
          curMsg = 6;
        case 19:
          curMsg = 1;
          break;
        default:
          curMsg = (rand() >> 4) % 8;
          break;
      }
    }
  if(now & 1)
    CopyMessageTo(g.backbuffer, curMsg, -1);
  lastTime = now;
}



void transform(BITMAP *src, BITMAP *dst, int originX, int originY);
static void UpdateScreen()
{
  int i, width, height;
  int originX, originY = 0;

  originY++;

  qsort(g.cells, MAX_MOLES, sizeof(MoleCell), MoleCompare);

  /* handle distracting eye-candy */


  originX = mouse_x - g.offX;
  originY = mouse_y - g.offY;

  circlefill(g.icandy[0], originX, originY, 4, 255);
  transform(g.icandy[0], g.icandy[1], originX, originY);
  for(height = 0; height < 192; height++)
  {
    unsigned char *srcLine = g.icandy[0]->line[height];
    unsigned char *dstLine = g.backbuffer->line[height];
    int x;

    for(x = 0; x < 256; x++)
      dstLine[x] = (srcLine[x] >> 4) + 16;
  }


  for(i = 0; i < MAX_MOLES; i++)
    {
      BITMAP *pix = g.molePix[g.cells[i].moleType];

      switch(g.cells[i].state)
        {
        case STATE_INACTIVE:
  	  break;
        case STATE_MOLE:
	  height = pix->h * g.cells[i].y / 128;
	  width = pix->w * g.cells[i].y / 128;
	  stretch_sprite(g.backbuffer, pix,
		         g.cells[i].x - width / 2, g.cells[i].y - height,
		         width, height);
          if(g.cells[i].clock < tim.odo)
	    {
	      g.cells[i].state = STATE_INACTIVE;
	      g.miss++;
	    }
	  break;
        case STATE_DYING:
          height = pix->h * g.cells[i].y * (25 - tim.odo + g.cells[i].clock) / (128 * 25);
	  width = pix->w * g.cells[i].y / 128;
	  stretch_sprite(g.backbuffer, pix,
		         g.cells[i].x - width / 2, g.cells[i].y - height,
		         width, height);
          circlefill(g.backbuffer, g.cells[i].x, g.cells[i].y - height / 2,
                     tim.odo - g.cells[i].clock, 4); // 4 = blood
          if(25 < (tim.odo - g.cells[i].clock))
	    g.cells[i].state = STATE_INACTIVE;
	  break;
        }
    }
}

static void NewMole(void)
{
  int i, y;

  for(i = 0; i < MAX_MOLES; i++)
    {
      if(g.cells[i].state == STATE_INACTIVE)
	break;
    }
  if(i == MAX_MOLES)
    return;

  g.cells[i].x = rand() % 256;
  y = (rand() * 128ULL / RAND_MAX + 128);
  g.cells[i].y = y * y / 256;
  g.cells[i].state = STATE_MOLE;
  g.cells[i].clock = tim.odo + 230;
  g.cells[i].moleType = rand() % 5;
}

static int double_times[] = {
  128, 240, 400, 928, 1344, 6661666
};

static void Mousing(int x, int y)
{
  int b, i, height, width;
  int found = 0;
  static int last_b = 0;
  SAMPLE *shot = GetResource(g.dat, "SHOT");

  b = mouse_b;
  i = (b && !last_b);
  last_b = b;
  if(!i)
    return;

  if(shot)
    play_sample(shot, 255, 128, 1000, FALSE);

  for(i = MAX_MOLES - 1; i >= 0; i--)
    {
      int moleHt = g.molePix[g.cells[i].moleType]->h;
      int moleWid = g.molePix[g.cells[i].moleType]->w; 

      switch(g.cells[i].state)
        {
        case STATE_INACTIVE:
  	  break;
        case STATE_MOLE:
	case STATE_DYING:
	  height = moleHt * g.cells[i].y / 128;
	  width = moleWid * g.cells[i].y / 128;
	  if(y > g.cells[i].y - height && y < g.cells[i].y &&
	     x > g.cells[i].x - width / 2 && x < g.cells[i].x + width / 2)
	    {
	      found++;
	      if(g.cells[i].state == STATE_MOLE)
		{
		  g.hit++;
                  g.cells[i].clock = tim.odo;
		  g.cells[i].state = STATE_DYING;
		}
	    }
	  break;
        }

      if(found >= SHOOT_THROUGH)
	break;

    }
}

static void InitGame();
static void Play()
{
  int done = 0;
  int last_midi_pos = 1, last_time = 0, last_odo = 0;
  int current_time;
  int multMask = 15; // if current_time (16th note) & multMask == 0 then new mole
  int multIndex = 0;
  int mx, my;

  InitGame();
  tim.trip = 0;

  while(done == 0)
    {
      if(keypressed())
	{
          int k = readkey();
          if((k & 0xff) == 27)
            done = 1;
	}

      if(midi_pos != last_midi_pos)
	{
	  last_midi_pos = midi_pos;
          tim.trip = 0;
	}

      current_time = (last_midi_pos - 1) * 4 + tim.trip / 15;
      mx = mouse_x;
      my = mouse_y;

      UpdateScreen();
      if(current_time >= 544 && current_time < 928)
	CopyMessage(current_time);

      text_mode(-1);
      textprintf(g.backbuffer, g.lucid, 83, 0, -1, "hit%3d    miss%3d",
                 g.hit, g.miss);
      if(current_time != last_time)
	{
	  last_time = current_time;
          if((last_time & multMask) == 0 &&
             (last_time < 544 || last_time >= 928) &&
             (last_time < 1344))
	    NewMole();
          if(last_time >= 1200 && last_time < 1344)
            NewMole();
	  if(current_time >= double_times[multIndex])
	    {
	      multIndex++;
	      multMask >>= 1;
	    }
	}
      if(last_time >= 1344)
        textout_centre(g.backbuffer, g.lucid, "Game Over",
                       128, 48, -1);

      Mousing(mouse_x - g.offX, mouse_y - g.offY);
      show_mouse(NULL);

      acquire_screen();
      blit(g.backbuffer, screen, 0, 0, g.offX, g.offY, 256, 192);
      release_screen();
      show_mouse(screen);
      while(last_odo == tim.odo)
        yield_timeslice();
      last_odo = tim.odo;

      if(current_time > 1370  || g.miss >= (g.hit + 10))
	done = 1;
    }

  show_mouse(NULL);

  acquire_screen();
  textout_centre(screen, g.lucid, "Game Over", SCREEN_W / 2, g.offY + 48, -1);
  textout_centre(screen, g.lucid, "press a key", SCREEN_W / 2, g.offY + 92, -1);
  if(g.miss < g.hit + 10 && current_time > 1224)
  {
    int accuracy = g.hit * 1000 / (g.hit + g.miss);

    textprintf(screen, g.lucid, g.offX + 53, g.offY + 72, -1,
               "accuracy:%3d.%d%%", accuracy / 10, accuracy % 10);
    release_screen();
  }
  else
  {
    textprintf(screen, g.lucid, g.offX + 8, g.offY + 72, -1,
               "%3d hits not good enough!", g.hit);
    release_screen();
    stop_midi();
  }
  readkey();
  acquire_screen();
  klear(screen);
  textout(screen, font, "application closing", 0, 0, -1);
  release_screen();
}

static void KillRandomMole(void)
{
  int n = rand() % MAX_MOLES, tries = MAX_MOLES;

  while(g.cells[n].state != STATE_MOLE)
  {
    n = (n + 1) % MAX_MOLES;
    tries--;
    if(tries == 0)
      return;
  }

  if(g.cells[n].state == STATE_MOLE)
  {
    g.cells[n].state = STATE_DYING;
    g.cells[n].clock = tim.odo;
  }
}

static void InitGame()
{
  SAMPLE *dedo = GetResource(g.dat, "DEDODEDO");
  SAMPLE *shot = GetResource(g.dat, "SHOT");
  MIDI *mrsd = GetResource(g.dat, "MRSD");
  int i = 0;
  int spin_state = 0;

  klear(screen);
  rectfill(screen, g.offX, g.offY, g.offX + 255, g.offY + 191, 19);

  set_palette(g.pal);

  if(dedo)
    play_sample(dedo, 255, 127, 1000, FALSE);
  tim.trip = 0;

  text_mode(-1);
  acquire_screen();
  textout_centre(screen, g.lucid,    "Once upon a time,", SCREEN_W / 2, g.offY + 12, -1);
  textout_centre(screen, g.lucid, "Hampton Hampster and his", SCREEN_W / 2, g.offY + 30, -1);
  textout_centre(screen, g.lucid,   "rodent buddies were", SCREEN_W / 2, g.offY + 48, -1);
  textout_centre(screen, g.lucid, "dancing the night away.", SCREEN_W / 2, g.offY + 66, -1);
  release_screen();

  /*
   * stick expensive initializations in here; we have ten seconds of cinema
   */

  while((!keypressed()) && (tim.trip < 1000))
  {
    if((tim.trip % 50) >= 25)
    {
      if(spin_state == 0)
      {
        spin_state = 1;
        rectfill(g.backbuffer, 0, 100, 255, 191, 19);
        for(i = 0; i < 5; i++)
        {
          draw_sprite(g.backbuffer, g.molePix[i],
                      25 + 50 * i - g.molePix[i]->w / 2,
                      190 - g.molePix[i]->h);
        }
        blit(g.backbuffer, screen, 0, 100, g.offX, g.offY + 100, 256, 92);
      }
    }
    else
    {
      if(spin_state == 1)
      {
        spin_state = 0;
        rectfill(g.backbuffer, 0, 100, 255, 191, 19);
        for(i = 0; i < 5; i++)
        {
          draw_sprite_h_flip(g.backbuffer, g.molePix[i],
                      25 + 50 * i - g.molePix[i]->w / 2,
                      190 - g.molePix[i]->h);
        }
        blit(g.backbuffer, screen, 0, 100, g.offX, g.offY + 100, 256, 92);
      }
    }
    yield_timeslice();
  }

  acquire_screen();
  textout_centre(screen, g.lucid, "Until something went", SCREEN_W / 2, g.offY + 96, -1);
  textout_centre(screen, g.lucid, "horribly wrong...", SCREEN_W / 2, g.offY + 118, -1);
  release_screen();
  if(keypressed())
    readkey();

  tim.trip = i = 0;
  stop_sample(dedo);

  while(mrsd_times[i] >= 0)
  {
    while(tim.trip < mrsd_times[i] && !keypressed())
    {
      if(i >= 3)
      {
        UpdateScreen();
        vsync();
        acquire_screen();
        blit(g.backbuffer, screen, 0, 0, g.offX, g.offY, 256, 192);
        release_screen();
      }
    }
    //printf("%5d\n", i);
    if(i >= 2)
    {
      KillRandomMole();
    }
    if(shot)
      play_sample(shot, 255, 128, 1000, FALSE);
    NewMole();
    if((i & 3) == 3)
      NewMole();
    i++;
    if(keypressed())
      {
 	    readkey();
        break;
      }
  }

  klear(screen);
  set_mouse_sprite(GetResource(g.dat, "XHAIR"));
  set_mouse_sprite_focus(16, 16);
  show_mouse(screen);
  play_midi(mrsd, FALSE);
  while(midi_pos < 1);

  g.hit = g.miss = 0;
  clear_keybuf();
}

static int Init()
{
  int badLoad = 0, i;

  install_guwame_timerint();
  install_sound(DIGI_AUTODETECT, MIDI_AUTODETECT, NULL);

  g.dat = load_skinned_datafile("whack.dat");
  if(g.dat == NULL)
  {
    alert("whack: whack.dat:", "No such file or directory", "Is it installed right?", "No", NULL, 13, 0);
    remove_int(guwame_timerint);
    return 1;
  }

  for(i = 0; i < 5; i++)
  {
    char x[] = "MOLES";

    x[4] = '1' + i;
    g.molePix[i] = GetResource(g.dat, x);
    if(g.molePix[i] == NULL)
      badLoad = 1;
  }
  g.pal = GetResource(g.dat, "PCPAL");
  g.lucidmono = GetResource(g.dat, "LUCIDM");
  g.lucid = GetResource(g.dat, "LUCID");
  if(g.pal == NULL || g.lucidmono == NULL)
    badLoad = 1;
  if(g.lucid == NULL)
    g.lucid = g.lucidmono;
  if(GetResource(g.dat, "MRSD") == NULL ||
     GetResource(g.dat, "XHAIR") == NULL ||
     GetResource(g.dat, "DEDODEDO") == NULL)
    badLoad = 1;

  if(badLoad)
  {
    alert("whack: whack.dat:", "Missing essential data", "Please reinstall.", "/mnt/cdrom", "tar xzvf", 13, 27);
    unload_datafile(g.dat);
    remove_int(guwame_timerint);
    return 1;
  }

  g.pal[0].r = g.pal[0].g = g.pal[0].b = 0;

  g.icandy[0] = create_bitmap(256, 192);
  g.icandy[1] = create_bitmap(256, 192);
  g.backbuffer = create_bitmap(256, 192);
  g.offX = (SCREEN_W - 256) / 2;
  g.offY = (SCREEN_H - 192) / 2;
  tim.odo = tim.trip = 0;

  return 0;
}

void WhackMain(void)
{
  if(Init())
    return;

  stop_midi();
  Play();
  set_mouse_sprite(NULL); // restore arrow

  Credits(
  "H a m p s t e r d e a t h\n"
  "GrandUnifiedWhackAMoleEngine b2\n"
  "Copyright © 2000  Damian Yerrick\n"
  "GUWAME comes with ABSOLUTELY NO WARRANTY.\n"
  "This is free software, and you are\n"
  "welcome to redistribute it under certain\n"
  "conditions; read the COPYING file for details.\n\n"
  "Music by  Disney  and  nine inch noses\n"
  "Graphics borrowed from the World Wide Web\n"
  );
}
