/**************************************\
* drm.c                                *
* Vitamins, a falling pills game       *
*                                      ********************************\
* Copyright 2000 Damian Yerrick                                        *
*                                                                      *
* This program is free software; you can redistribute it and/or modify *
* it under the terms of the GNU General Public License as published by *
* the Free Software Foundation; either version 2 of the License, or    *
* (at your option) any later version.                                  *
*                                                                      *
* This program is distributed in the hope that it will be useful,      *
* but WITHOUT ANY WARRANTY; without even the implied warranty of       *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
* GNU General Public License for more details.                         *
*                                                                      *
* You should have received a copy of the GNU General Public License    *
* along with this program; if not, write to the Free Software          *
* Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.            *
* Or view the License online at http://www.gnu.org/copyleft/gpl.html   *
*                                                                      *
* Damian Yerrick's World Wide Web pages are located at                 *
*   http://pineight.8m.com                                             *
*                                                                      *
\**********************************************************************/


/* Curiouser and curiouser...
 *
 * Nintendo has U.S. Patent 5265888 on the game of Dr. Mario.
 * So what?  Like Nintendo is going to sue me.  What can it get
 * damages-wise out of a free software author?  Plus, I'm not even
 * cloning Dr. Mario.  I'm cloning Dr. Rudy and Dr. Macinto;
 * sue their authors first.
 */

/* System Requirements
 *
 * This program requires:
 *   GNU C compiler    <http://www.delorie.com/djgpp/>
 *   Allegro library   <http://www.talula.demon.co.uk/allegro/>
 *   Allegro-compatible video card capable of 512x384 or 640x480
 *     in 256 colors
 */

/* Language note
 *
 * This program is written in GNU C, which has some C++ features
 * (C++ comments, enums...).  I'm using these features so that in
 * two months, when I shall have learned C++, I will be able to
 * port the program to C++.
 */

#include <time.h>
#include "dosarena.h"
#include "wave.h"


//
// CONSTANTS
//

typedef enum PlayerState
{
  STATE_INACTIVE,
  STATE_NEW_PIECE,
  STATE_FALLING_PIECE,
  STATE_CHECK,
  STATE_CHECKED,
  STATE_FALL,
  STATE_BOULDERS,
  STATE_GAMEOVER
} PlayerState;

static const int BLOCK_WID = 20, BLOCK_HT = 20;
static const int FIELD_WID = 8, FIELD_HT = 16;

//
// CLASS DECLARATIONS
//

typedef struct Player
{
  struct Context *context;
  int originX; // on background
  int originY; // on background
  fixed x;     // 0 = left; 7 << 16 = right
  fixed y;     // 0 = bottom; 15 << 16 = top
  int refreshBottom, refreshTop; // rows of frontbuf to redraw
  int coveredBottom, coveredTop; // rows of backbuf to redraw
  int piece1, piece2; // color of primary and secondary piece
  int next1, next2; // next piece coming
  PlayerState state;
  int clock;
  int flip; // 90deg anticlockwise rotation from east
  int nVIR; // viruses left to kill
  int virScore; // score for next virus
  int chainCount;
  int level;
  int initialLevel;
  int nPieces; // number of pieces dropped determines speed of game
  int score;
  int comingHead, comingTail;
  int wins;

  int seed; // for random numbers.

  char coming[16];
  char going[16];
  signed char repeatTime[8];
  unsigned char field[16][8];
} Player;

typedef struct Context
{
  DATAFILE *pix; // load_datafile(drm_dat)
  RGB *pal;
  FONT *lucid, *lucidgray; // Lucida Console from ttf2pcx
  BITMAP *backbuf; // back buffer
  BITMAP *bkgnd; // current background
  BITMAP *tiles; // 20x20 tiles that make up the graphics
  RLE_SPRITE *rletiles[64];
  int xOff, yOff;
  int now; // last time the game was updated
  int preferredRes;
  int puyo;
  int garbage; // playing a 2p game with garbage?
  int autoHandicap; // playing a 2p game with autohandicap?
  int meterColor;
  int balance; // 0 = all digi; 512 = all midi
  BITMAP *bkgnds[2];
  Player p[2];
} Context;

// split timers off into a global to make them volatile
typedef struct Timers
{
  int odo, trip, pause;
} Timers;


//
// MORE CONSTANTS
//

static const char program_name[] = "Vitamins";
static const int flipCos[8] = {1, 0, -1, 0, 1, 0, -1, 0};
static const int flipSin[8] = {0, 1, 0, -1, 0, 1, 0, -1};

static const Effect moveSnd[] =
{
  {0x20, 85,  0, 5},
  {0x00,  0,  0, 0}
};

static const Effect flipSnd[] =
{
  {0x23, 77, 77, 5},
  {0x00,  0,  0, 6},
  {0x23, 77, 77, 5},
  {0x03,  0, 77, 5},
  {0x00,  0,  0, 0}
};

static const Effect dropSnd[] =
{
  {0x03,  0, 42, 5},
  {0x03,  0, 40, 6},
  {0x03,  0, 36, 5},
  {0x00,  0,  0, 0}
};

static const Effect clearSnd[] =
{
  {0x03,  0, 62, 4},
  {0x03,  0, 64, 4},
  {0x03,  0, 62, 4},
  {0x03,  0, 64, 4},
  {0x03,  0, 62, 4},
  {0x03,  0, 64, 4},
  {0x04,  0, 62, 4},
  {0x04,  0, 64, 4},
  {0x04,  0, 62, 4},
  {0x04,  0, 64, 4},
  {0x00,  0,  0, 0}
};

static const Effect virusSnd[] =
{
  {0x03,  0, 70, 4},
  {0x03,  0, 71, 4},
  {0x03,  0, 86, 4},
  {0x03,  0, 72, 4},
  {0x03,  0, 83, 4},
  {0x03,  0, 73, 4},
  {0x03,  0, 79, 4},
  {0x03,  0, 74, 4},
  {0x03,  0, 77, 4},
  {0x04,  0, 71, 4},
  {0x04,  0, 86, 4},
  {0x04,  0, 72, 4},
  {0x04,  0, 83, 4},
  {0x00,  0,  0, 0}
};

static const Effect sendSnd[] =
{
  {0x22, 73, 71, 4},
  {0x22, 74, 72, 4},
  {0x22, 75, 73, 4},
  {0x22, 76, 74, 4},
  {0x22, 77, 75, 4},
  {0x22, 78, 76, 4},
  {0x22, 79, 77, 4},
  {0x22, 80, 78, 4},
  {0x00,  0,  0, 0}
};

static const Effect winSnd[] =
{
  {0x22, 73, 71, 3},
  {0x22, 74, 72, 3},
  {0x22, 75, 73, 3},
  {0x22, 76, 74, 3},
  {0x22, 77, 75, 3},
  {0x22, 78, 76, 3},
  {0x22, 79, 77, 3},
  {0x22, 80, 78, 3},
  {0x00,  0,  0, 7},
  {0x22, 65, 68,40},
  {0x22, 66, 69,40},
  {0x42, 50, 69,40},
  {0x00,  0,  0, 0}
};

static const Effect loseSnd[] =
{
  {0x04,  0, 44, 4},
  {0x04,  0, 43, 4},
  {0x04,  0, 42, 4},
  {0x04,  0, 41, 4},
  {0x04,  0, 40, 4},
  {0x04,  0, 39, 4},
  {0x04,  0, 38, 4},
  {0x04,  0, 37, 4},
  {0x00,  0,  0, 0}
};

static const char credits[] =
"\n"
"first of all, what Vitamins 2000 is NOT\n"
"  it's not a Tetris® product\n"
"  it's not a Nintendo® product\n"
"  it's not proprietary software!\n"
"\n"
"original concept and design\n"
"  Dr. Mario(TM) by Yamamoto, Harada, and Yamanaka\n"
"  at Nintendo Co. Ltd. (What's 5265888?)\n"
"\n"
"program\n"
"  lead coder: Damian Yerrick\n"
"                           damian@pineight.8m.com\n"
"  cross-platform qa: Pinocchio Poppins\n"
"                        pinocchio@pineight.8m.com\n"
"\n"
"powered by\n"
"  GNU C Compiler\n"
"                              http://gcc.gnu.org/\n"
"  Allegro by Shawn Hargreaves\n"
"           http://www.talula.demon.co.uk/allegro/\n"
"  and one of these C libraries\n"
"    (DOS) DJGPP libc by DJ Delorie\n"
"                    http://www.delorie.com/djgpp/\n"
"    (GNU) glibc by Free Software Foundation\n"
"                              http://www.gnu.org/\n"
"    (Win) Microsoft® Visual C++ Runtime\n"
"                        http://www.microsoft.com/\n"
"\n"
"\n"
"Don't be a slave to Micro$oft; use free software.\n"
"Only you can end software hoarding.\n"
"\n"
"                 GNU's not UNIX.\n"
"\n"
"This game is 100% themable.  Pick up Grabber,"
"the theme editor for Allegro-based games, at\n"
"                           http://come.to/yerrick\n"
;



//
// KEEP GLOBAL VARIABLES TO A MINIMUM
//

static Context g;
static volatile Timers tim;
static char drm_dat[256] = "dr.dat";


//
// FUNCTIONS
//

static void drm_timerint(void);
void clear2color(BITMAP *bmp, int color);
void klear(BITMAP *bmp);
void *GetResource(DATAFILE *dat, const char *name);
static void DrawScanline(Player *this, int sline);
static void ExtendRefresh(Player *this, int scanline);
static void ExtendCovered(Player *this, int scanline);
static void BlitScanlines(Player *this);
static void Uncover(Player *this);
static void DrawCapsule(Player *p);
static void MakeViruses(Player *p, int nVIR);
static void NewGame(Context *g, int nPlayers);
static void NewLevel(Context *g, int nPlayers);
static char CheckOverlap(Player *this, int x, int y, int flip);
static void Try2Flip(Player *this, unsigned int f4);
static int Play(int nPlayers);
static void Copyright(void);
static void NotGlobals(void);
static int GetRLETiles();
static int init_drm(void);

void GetSamples(void);
void TestWaveforms(void);


/* ReadKPad() **************************
 * Reads an area of the keyboard.
 * 0x80 up
 * 0x40 down
 * 0x20 left
 * 0x10 right
 * 0x08 left trigger
 * 0x04 right trigger
 * 0x02 A trigger
 * 0x01 B trigger
 */
static int ReadPad(int which)
{
  return ReadKPad(which) | ReadJPad(which);
}


/* drm_timerint() ***********************
 * Handle the 100 fps timer.
 */
static void drm_timerint(void)
{
  if(!tim.pause)
  {
    tim.odo++;
    tim.trip++;
  }
}
END_OF_FUNCTION(drm_timerint);


/* clear2color() ***********************
 * The Windows clear_to_color() has an overrun.  This doesn't.
 */
void clear2color(BITMAP *bmp, int color)
{
  rectfill(bmp, 0, 0, bmp->w - 1, bmp->h - 1, color);
}


/* klear() *****************************
 * The windows clear() calls the Windows clear_to_color() (see above).
 */
void klear(BITMAP *bmp)
{
  clear2color(bmp, 0);
}


/* GetResourceObj() ********************
 * Get a resource control structure by name from a datafile.
 */
DATAFILE *GetResourceObj(DATAFILE *dat, const char *name)
{
  /*
  int i;

  if(dat == NULL)
    return NULL;
  for(i=0; dat[i].type != DAT_END; i++)
    if(stricmp(get_datafile_property(dat+i, DAT_ID('N','A','M','E')),
	       name) == 0)
      return &(dat[i]);
  return NULL;
  */

  return find_datafile_object(dat, (char *)name);
}


/****************************************************************************/

/* DrawScanline() **********************
 * Draws a line of blocks in a player's field.
 * In C++, this would be Player::DrawScanline(sline)
 */
static void DrawScanline(Player *this, int sline)
{
  int x, y;
  unsigned i;

  /* erase what's already there */
  x = this->originX;
  y = this->originY - (sline + 1) * BLOCK_HT;
  blit(this->context->bkgnd, this->context->backbuf,
       x, y, x, y, FIELD_WID * BLOCK_WID, BLOCK_HT);

  if(sline < 0 || sline >= FIELD_HT)
    return;

  /* blit the blocks */
  for(i = 0; i < FIELD_WID; i++)
  {
    unsigned blockHere = this->field[sline][i];

    if(blockHere != 0 )
    {
#if 0
      unsigned blitorg_x, blitorg_y;
      blitorg_y = (blockHere / 8) * BLOCK_HT;
      blitorg_x = (blockHere % 8) * BLOCK_WID;
      masked_blit(this->context->tiles, this->context->backbuf,
                  blitorg_x, blitorg_y,
                  x + i * BLOCK_WID, y, BLOCK_WID, BLOCK_HT);
#else
      draw_rle_sprite(this->context->backbuf,
                      this->context->rletiles[blockHere],
                      x + i * BLOCK_WID, y);
#endif
    }
  }
}


/* ExtendRefresh() *********************
 * Marks an area as dirty, meaning it needs to be copied to screen.
 */
static void ExtendRefresh(Player *this, int scanline)
{
  if(scanline >= FIELD_HT)
    scanline = FIELD_HT - 1;
  if(scanline < 0)
    scanline = 0;

  if(this->refreshTop < scanline)
    this->refreshTop = scanline;
  if(this->refreshBottom > scanline)
    this->refreshBottom = scanline;
}


/* ExtendCovered() ********************
 * Marks an area as covered with a sprite.  Also marks it as dirty.
 */
static void ExtendCovered(Player *this, int scanline)
{
  if(scanline >= FIELD_HT)
    scanline = FIELD_HT - 1;
  if(scanline < 0)
    scanline = 0;

  if(this->coveredTop < scanline)
    this->coveredTop = scanline;
  if(this->coveredBottom > scanline)
    this->coveredBottom = scanline;
  ExtendRefresh(this, scanline);
}


/* BlitScanlines() *********************
 * Refreshes dirty scanlines and
 * marks them as clean.
 */
static void BlitScanlines(Player *this)
{
  int x, y, height;

  /* erase what's already there */
  x = this->originX;
  y = this->originY - (this->refreshTop + 1) * BLOCK_HT;
  height = (this->refreshTop + 1 - this->refreshBottom) * BLOCK_HT;
  acquire_screen();
  blit(this->context->backbuf, screen,
       x, y, x + this->context->xOff, y + this->context->yOff,
       FIELD_WID * BLOCK_WID, height);
  release_screen();
  this->refreshTop = 0;
  this->refreshBottom = 15;
}


/* Uncover() ***************************
 * Redraws covered scanlines and marks them as
 * uncovered and dirty.
 */
static void Uncover(Player *this)
{
  int i;

  if(this->coveredBottom > this->coveredTop)
    return;

  ExtendRefresh(this, this->coveredBottom);
  ExtendRefresh(this, this->coveredTop);

  for(i = this->coveredBottom; i <= this->refreshTop; i++)
    DrawScanline(this, i);

  this->coveredTop = 0;
  this->coveredBottom = 15;
}


/* DrawCapsule() ***********************
 * Draws the falling vitamin capsule.
 */
static void DrawCapsule(Player *p)
{
  int xloc = p->originX + fmul(BLOCK_WID, p->x);
  int yloc = p->originY - fmul(BLOCK_HT, p->y + 0xc000);
  int scanbase = (p->y - 0x4000) >> 16;

  ExtendCovered(p, scanbase);
  ExtendCovered(p, scanbase + 1);

  switch(p->flip)
  {
  case 0:
    draw_rle_sprite(p->context->backbuf,
                    p->context->rletiles[p->piece1 * 8 + 1],
                    xloc, yloc);
    draw_rle_sprite(p->context->backbuf,
                    p->context->rletiles[p->piece2 * 8 + 2],
                    xloc + BLOCK_WID, yloc);
    break;
  case 1:
    draw_rle_sprite(p->context->backbuf,
                    p->context->rletiles[p->piece1 * 8 + 4],
                    xloc, yloc);
    draw_rle_sprite(p->context->backbuf,
                    p->context->rletiles[p->piece2 * 8 + 3],
                    xloc, yloc - BLOCK_HT);
    ExtendCovered(p, scanbase + 2);
    break;
  case 2:
    draw_rle_sprite(p->context->backbuf,
                    p->context->rletiles[p->piece1 * 8 + 2],
                    xloc, yloc);
    draw_rle_sprite(p->context->backbuf,
                    p->context->rletiles[p->piece2 * 8 + 1],
                    xloc - BLOCK_WID, yloc);
    break;
  case 3:
    draw_rle_sprite(p->context->backbuf,
                    p->context->rletiles[p->piece1 * 8 + 3],
                    xloc, yloc);
    draw_rle_sprite(p->context->backbuf,
                    p->context->rletiles[p->piece2 * 8 + 4],
                    xloc, yloc + BLOCK_HT);
    ExtendCovered(p, scanbase - 1);
    break;
  }
}


/* DrawMeters() ************************
 * Draws various meters and the next piece.
 */
static void DrawMeters(Player *this)
{
  /* erase what's already there */
  int x = this->originX;
  int y = this->originY - (FIELD_HT + 2) * BLOCK_HT;

  text_mode(-1);

  /* erase an area behind the score area */
  blit(this->context->bkgnd, this->context->backbuf,
       x, y, x, y, 8 * BLOCK_WID, 2 * BLOCK_HT);

  /* draw text meters */
  textprintf(g.backbuf, g.lucid, x, y,
             this->context->meterColor, "%5d", this->score);
  textprintf(g.backbuf, g.lucid, x + 5 * BLOCK_WID, y,
             this->context->meterColor, "%5d", this->nVIR);
  textprintf(g.backbuf, g.lucid, x, y + BLOCK_HT,
             this->context->meterColor, "%5d", this->level);
  textprintf(g.backbuf, g.lucid, x + 5 * BLOCK_WID, y + BLOCK_HT,
             this->context->meterColor, "%5d", this->wins);

  /* erase what's already there */

  /* draw next capsule */
  draw_rle_sprite(this->context->backbuf,
                  this->context->rletiles[this->next1 * 8 + 1],
                  x + (FIELD_WID / 2 - 1) * BLOCK_WID, y);
  draw_rle_sprite(this->context->backbuf,
                  this->context->rletiles[this->next2 * 8 + 2],
                  x + (FIELD_WID / 2) * BLOCK_WID, y);

  /* copy it to screen */
  blit(this->context->backbuf, screen,
       x, y, x + this->context->xOff, y + this->context->yOff,
       8 * BLOCK_WID, 2 * BLOCK_HT);
}


/* Random() ****************************
 * Random number generator.
 */
static int Random(Player *p)
{
  p->seed = p->seed * 1103515245 + 12345;
  return(p->seed >> 16) & 0x7fff;
}


/* MakeViruses() ***********************
 * Add nVIR viruses to player's field.
 */
static void MakeViruses(Player *p, int nVIR)
{
  int x, y;
  int maxY = (176 + nVIR) / 20;
  int c = Random(p) % 3 + 1;
  int targetVir;
  int failures = 200;

  text_mode(0);

  while(nVIR > 0)
  {
    targetVir = 8 * c + 5;
    x = Random(p) % FIELD_WID;
    y = Random(p) % maxY;
    if(p->field[y][x] != 0)
      x = x ^ 4;
    if(p->field[y][x] != 0)
      x = x ^ 2;
    if(p->field[y][x] != 0)
      x = x ^ 1;
    if(p->field[y][x] != 0)
    {
      continue;
    }


    // If it absolutely can't place any more viruses, bail.
    if(failures < 0)
    {
      p->nVIR -= nVIR;
      nVIR = 0;
      break;
    }


    // Make sure there aren't any three-in-a-rows
    // (oox xoo)
    if(x >= 2 &&
       p->field[y][x - 1] == targetVir &&
       p->field[y][x - 2] == targetVir)
    {
      failures--;
      continue;
    }

    if(x < FIELD_WID - 2 &&
       p->field[y][x + 1] == targetVir &&
       p->field[y][x + 2] == targetVir)
    {
      failures--;
      continue;
    }

    if(y >= 2 &&
       p->field[y - 1][x] == targetVir &&
       p->field[y - 2][x] == targetVir)
    {
      failures--;
      continue;
    }

    if(y < FIELD_HT - 2 &&
       p->field[y + 1][x] == targetVir &&
       p->field[y + 2][x] == targetVir)
    {
      failures--;
      continue;
    }

    // Allow some three-in-a-rows (oxo) for virus levels > 20 or any other
    // level that fails to generate a unique virus dozens of times.

    if(failures > 150)
    {
      if(y >= 1 && y < FIELD_HT - 1 &&
         p->field[y - 1][x] == targetVir &&
         p->field[y + 1][x] == targetVir)
      {
        failures--;
        continue;
      }

      if(x >= 1 && x < FIELD_WID - 1 &&
         p->field[y][x - 1] == targetVir &&
         p->field[y][x + 1] == targetVir)
      {
        failures--;
        continue;
      }
    }


    /* Funny, the name of one of the most popular programming
     * languages on the planet means "Move to the next color."
     */
    c++;
    if(c > 3)
      c -= 3;

    p->field[y][x] = targetVir;
    nVIR--;
  }
}


/* NewLevel() **************************
 * Initialize variables for a new level.
 */
static void NewLevel(Context *g, int nPlayers)
{
  int i, x, y, randSeed;

  blit(g->bkgnd, screen, 0, 0, g->xOff, g->yOff, 512, 384);
  blit(g->bkgnd, g->backbuf, 0, 0, 0, 0, 512, 384);

  randSeed = 1103515245 * time(NULL) + 12345;

  for(i = 0; i < nPlayers; i++)
  {
    Player *me = &(g->p[1 - i]);
    me->refreshBottom = me->coveredBottom = 0;
    me->refreshTop = me->coveredTop = FIELD_HT - 1;
    me->state = STATE_NEW_PIECE;
    me->nPieces = me->comingHead = me->comingTail = me->chainCount = 0;
    me->nVIR = me->level * 4 + 4;
    for(y = 0; y < FIELD_HT; y++)
      for(x = 0; x < FIELD_WID; x++)
        me->field[y][x] = 0;
    for(y = 0; y < 8; y++)
      me->repeatTime[y] = 1;

    me->seed = randSeed;
    if(g->puyo)
      me->comingHead = me->nVIR;
    else
      MakeViruses(me, me->nVIR);
    me->seed = randSeed;
    me->next1 = Random(me) % 3 + 1;
    me->next2 = Random(me) % 3 + 1;
  }
  g->now = tim.odo;
}


/* NewGame() ***************************
 * Initialize variables for a new game.
 */
void NewGame(Context *g, int nPlayers)
{
  int i;
  MIDI *song = GetResource(music_dat, "DUELTRIS_MID");
  int R, G, B;

  if(jukeboxMode >= 0)
    play_midi(music_dat[jukeboxMode].dat, TRUE);
  else if(song)
    play_midi(song, TRUE);

  for(i = 0; i < 2; i++)
  {
    if(nPlayers == 2)
      g->p[i].originX = 48 + 256 * i;
    else
      g->p[i].originX = i ? 176 : -160;
    g->p[i].originY = 364;
    g->p[i].score = 0;
    g->p[i].context = g;
    g->p[i].nVIR = 0;
  }

  R = g->pal[31 + nPlayers].r << 2;
  G = g->pal[31 + nPlayers].g << 2;
  B = g->pal[31 + nPlayers].b << 2;

  g->meterColor = makecol(R, G, B);

#if USE_BKGND
  blit(g->bkgnds[nPlayers - 1], g->bkgnd, 0, 0, 0, 0, 512, 384);
#else
  klear(g->bkgnd);
#endif
}


/* CheckOverlap() **********************
 * Checks if a piece is overlapping the walls of the glass
 * or blocks.
 */
static char CheckOverlap(Player *this, int x, int y, int flip)
{
  y >>= 16;
  x >>= 16;
  if(x < 0 || x >= FIELD_WID ||
     x + flipCos[flip] < 0 || x + flipCos[flip] >= FIELD_WID ||
     y < 0 || y + flipSin[flip] < 0)
    return 1;
  if(y < FIELD_HT && this->field[y][x])
    return 1;
  if(y + flipSin[flip] < FIELD_HT)
    if(this->field[y + flipSin[flip]][x + flipCos[flip]])
      return 1;
  return 0;
}


/* Try2Flip ****************************
 * Tries several positions for rotating a piece into place.
 */
static void Try2Flip(Player *this, unsigned int f4)
{
  int cosine = itofix(flipCos[f4]);
  int sine = itofix(flipSin[f4]);
  int curCos = itofix(flipCos[this->flip]);
  int curSin = itofix(flipSin[this->flip]);

  if(!CheckOverlap(this, this->x, this->y, f4))
  {
    this->flip = f4;
  }
  else
//   o
//  *O     ->    *oO
  if(!CheckOverlap(this, this->x, this->y, f4 ^ 2))
  {
    this->x -= cosine;
    this->y -= sine;
    this->flip = f4;
  }
  else
//   o           oO
//  *O*    ->    * *
  if(!CheckOverlap(this, this->x + curCos,
                   this->y + curSin, f4))
  {
    this->x += curCos;
    this->y += curSin;
    this->flip = f4;
  }
  else
//  *o           *oO
//  *O*    ->    * *
  if(!CheckOverlap(this, this->x + curCos,
                   this->y + curSin, f4 ^ 2))
  {
    this->x += curCos - cosine;
    this->y += curSin - sine;
    this->flip = f4;
  }
  else
//  *o*          *O*
//  *O*    ->    *o*
  {
    this->x += curCos;
    this->y += curSin;
    this->flip ^= 2;
  }

  PlayEffect(flipSnd);
}


/* PushComing() ************************
 * Pushes a piece onto the coming queue.
 */
static void PushComing(Player *this, int color)
{
  this->coming[this->comingHead++] = color;
  this->comingHead &= 15;
}


/* PopComing() *************************
 * Pulls a piece from the coming queue.
 */
static int PopComing(Player *this)
{
  int color = this->coming[this->comingTail++];
  this->comingTail &= 15;
  return color;
}


void SetBlock(Player *this, int x, int y, int c)
{
  if(x >= 0 && x < FIELD_WID && y >= 0 && y < FIELD_HT)
  {
    ExtendCovered(this, y);
    this->field[y][x] = c;
  }
}

/* CheckLines() ************************
 * Checks for four blocks in a row vertically or horizontally.
 * Returns number of lines found.
 */
static int CheckLines(Player *this)
{
  int x, y, j, c, found = 0, foundVir = 0;

  for(y = 0; y < FIELD_HT; y++)
  {
    for(x = 0; x < FIELD_WID - 3; x++)
    {
//      if((this->field[y][x] & 7) == 7)
//        continue;
      c = this->field[y][x] & 070;
      if(c >= 010 && c < 050)
      {
        for(j = 1; j < 4; j++)
          if((this->field[y][x + j] & 070) != c)
            j = 99;

        if(j < 10)
        {
          found = 1;
          if(this->context->garbage)
            this->going[this->chainCount++] = c;

          j = 0;

          while(x + j < FIELD_WID &&
                (this->field[y][x + j] & 070) == (c & 070))
          {
            c = this->field[y][x + j];
            SetBlock(this, x + j, y, c | 07);
            if(y > 0 && this->field[y - 1][x + j] == 050)
            {
              SetBlock(this, x + j, y - 1, 0);
              this->nVIR--;
              this->score += this->virScore;
              this->virScore <<= 1;
            }
            if(y < FIELD_HT - 1 && this->field[y + 1][x + j] == 050)
            {
              SetBlock(this, x + j, y + 1, 0);
              this->nVIR--;
            }

            if((c & 07) == 5) // virus
            {
              this->nVIR--;
              foundVir = 1;
              this->score += this->virScore;
              this->virScore <<= 1;
            }
            j++;
          }
          // remove boulders on end caps
          if(x + j < FIELD_WID && this->field[y][x + j] == 050)
          {
            SetBlock(this, x + j, y, 0);
            this->nVIR--;
            this->score += this->virScore;
            this->virScore <<= 1;
          }
          if(x >= 1 && this->field[y][x - 1] == 050)
          {
            SetBlock(this, x - 1, y, 0);
            this->nVIR--;
            this->score += this->virScore;
            this->virScore <<= 1;
          }
          x += j - 1; // go to end of cleared row
        }
      }
    }
  }

  for(x = 0; x < FIELD_WID; x++)
  {
    for(y = 0; y < FIELD_HT - 3; y++)
    {
//      if((this->field[y][x] & 7) == 7)
//        continue;
      c = this->field[y][x] & 070;
      if(c >= 010 && c < 050)
      {
        for(j = 1; j < 4; j++)
          if((this->field[y + j][x] & 070) != c)
            j = 99;

        if(j < 10)
        {
          found = 1;
          if(this->context->garbage)
            this->going[this->chainCount++] = c;

          j = 0;

          while(y + j < FIELD_HT &&
                (this->field[y + j][x] & 070) == (c & 070))
          {
            c = this->field[y + j][x];
            SetBlock(this, x, y + j, c | 07);
            if(x > 0 && this->field[y + j][x - 1] == 050)
            {
              SetBlock(this, x - 1, y + j, 0);
              this->nVIR--;
              this->score += this->virScore;
              this->virScore <<= 1;
            }
            if(x < FIELD_WID - 1 && this->field[y + j][x + 1] == 050)
            {
              SetBlock(this, x + 1, y + j, 0);
              this->nVIR--;
              this->score += this->virScore;
              this->virScore <<= 1;
            }

            if((c & 07) == 5) // virus
            {
              this->nVIR--;
              foundVir = 1;
              this->score += this->virScore;
              this->virScore <<= 1;
            }
            j++;
          }
          // remove boulders on end caps
          if(y + j < FIELD_HT && this->field[y + j][x] == 050)
          {
            SetBlock(this, x, y + j, 0);
            this->nVIR--;
            this->score += this->virScore;
            this->virScore <<= 1;
          }
          if(y >= 1 && this->field[y - 1][x] == 050)
          {
            SetBlock(this, x, y - 1, 0);
            this->nVIR--;
            this->score += this->virScore;
            this->virScore <<= 1;
          }
          y += j - 1; // go to end of cleared row
        }
      }
    }
  }

  if(foundVir)
    PlayEffect(virusSnd);
  else if(found)
    PlayEffect(clearSnd);

  return found;
}


static void CheckLinesCleanup(Player *this)
{
  int x, y;

  // remove cleared blocks
  for(y = 0; y < FIELD_HT; y++)
    for(x = 0; x < FIELD_WID; x++)
      if((this->field[y][x] & 7) == 7)
        SetBlock(this, x, y, 0);

  // close up open half-pills
  for(y = 0; y < FIELD_HT; y++)
    for(x = 0; x < FIELD_WID; x++)
      switch(this->field[y][x] & 7)
      {
      case 1: // left half
        if((this->field[y][x + 1] & 7) != 2)
          SetBlock(this, x, y, this->field[y][x] & 070);

        break;

      case 2: // right half
        if((this->field[y][x - 1] & 7) != 1)
          SetBlock(this, x, y, this->field[y][x] & 070);

        break;

      case 3: // top half
        if((this->field[y - 1][x] & 7) != 4)
          SetBlock(this, x, y, this->field[y][x] & 070);

        break;

      case 4: // bottom half
        if((this->field[y + 1][x] & 7) != 3)
          SetBlock(this, x, y, this->field[y][x] & 070);

        break;

      }
}


/* Fall() ******************************
 * Moves floating pills down one space.
 */
static int Fall(Player *this)
{
  int x, y;
  int fallen = 0;

  for(y = 0; y < FIELD_HT - 1; y++)
    for(x = 0; x < FIELD_WID; x++)
      if(this->field[y][x] == 0 && this->field[y + 1][x] != 0 &&
         (this->field[y + 1][x] & 7) != 5 && (this->field[y + 1][x] & 7) != 2)
      {
        // Handle horizontal capsules.
        if((this->field[y + 1][x] & 7) == 1)
        {
          if(this->field[y][x + 1]) // If there isn't space for the pill,
            x++; // skip it.
          else
          {
            this->field[y][x] = this->field[y + 1][x];
            this->field[y + 1][x] = 0;
            x++;
            this->field[y][x] = this->field[y + 1][x];
            this->field[y + 1][x] = 0;
            ExtendCovered(this, y);
            ExtendCovered(this, y + 1);
            fallen = 1;
          }
        }
        else // Handle everything else.
        {
          this->field[y][x] = this->field[y + 1][x];
          this->field[y + 1][x] = 0;
          ExtendCovered(this, y);
          ExtendCovered(this, y + 1);
          fallen = 1;
        }
      }

  return fallen;
}


/* LockInPiece() ***********************
 * Locks a piece into place.
 */
static void LockInPiece(Player *p)
{
  int x = p->x >> 16;
  int y = p->y >> 16;

  ExtendCovered(p, y);

  switch(p->flip)
  {
  case 0:
    if(y < FIELD_HT)
    {
      p->field[y][x]     = p->piece1 * 8 + 1;
      p->field[y][x + 1] = p->piece2 * 8 + 2;
    }
    break;
  case 1:
    if(y + 1 < FIELD_HT)
    {
      p->field[y][x]     = p->piece1 * 8 + 4;
      p->field[y + 1][x] = p->piece2 * 8 + 3;
      ExtendCovered(p, y + 1);
    }
    else if(y < FIELD_HT)
      p->field[y][x]     = p->piece1 * 8 + 0;
    break;
  case 2:
    if(y < FIELD_HT)
    {
      p->field[y][x]     = p->piece1 * 8 + 2;
      p->field[y][x - 1] = p->piece2 * 8 + 1;
    }
    break;
  case 3:
    if(y < FIELD_HT)
    {
      p->field[y][x]     = p->piece1 * 8 + 3;
      p->field[y - 1][x] = p->piece2 * 8 + 4;
    }
    else if(y - 1 < FIELD_HT)
      p->field[y - 1][x] = p->piece2 * 8 + 0;
    ExtendCovered(p, y - 1);
    break;
  }

  PlayEffect(dropSnd);
  p->virScore = 1;
  p->chainCount = 0;

}


/* Play() ******************************
 * Play a level of Dr. M.  Possible return values:
 * 0: quit;  1: p[0] won;  2: p[1] won
 */
static int Play(int nPlayers)
{
  int done = 0, curTurn = 1, rVal = 0, redraws = 2, j;
  Player *me;

  NewLevel(&g, nPlayers);

  while(!done)
  {
    me = &(g.p[curTurn]);

    if(keypressed())
    {
      switch(readkey() >> 8)
      {
      
      case KEY_ESC:
        tim.pause = 1;
        klear(screen);
        textout_centre(screen, g.lucidgray,
                       "game paused.  [esc] resume, [o]ptions, or [q]uit",
                       256 + g.xOff, g.yOff + 160, -1);

        switch(readkey() >> 8)
        {
        case KEY_Q:
          done = 1;
          break;
        case KEY_A:
        case KEY_O:
          Options();
          break;
        default:
          break;
        }

        set_palette(g.pal);
        for(j = 0; j < 2; j++)
          DrawMeters(&(g.p[j]));
        tim.pause = 0;
        blit(g.backbuf, screen, 0, 0, g.xOff, g.yOff, 512, 384);
        break;

      case KEY_F4:
        // debug: introduce a 1-second delay
        rest(1000);
        save_bitmap("drm.bmp", screen, pal);
        break;
      }
    }

    if(me->state != STATE_INACTIVE)
    {
      /* should update player's meters here */
    }

    switch(me->state)
    {
      case STATE_INACTIVE:
        break;

      case STATE_NEW_PIECE:
        if(me->field[FIELD_HT - 1][FIELD_WID / 2 - 1] ||
           me->field[FIELD_HT - 1][FIELD_WID / 2])
        {
          me->state = STATE_GAMEOVER;
          if(nPlayers == 2)
            rVal = 2 - curTurn;
          break;
        }
          
        if(me->chainCount >= 2)
        {
          if(g.puyo)
          {
            g.p[1 - curTurn].comingHead +=
                (me->chainCount - 1) * (me->chainCount - 1);
            PlayEffect(sendSnd);
          }
          else
          {
            PlayEffect(sendSnd);
            do
            {
              me->chainCount--;
              PushComing(&(g.p[1 - curTurn]), me->going[me->chainCount]);
            } while(me->chainCount);
          }
        }

        if(g.puyo)
        {
          if(me->comingHead)
          {
            me->state = STATE_BOULDERS;
            break;
          }
        }
        else
        {
          if(((me->comingHead - me->comingTail) & 0x0f) < 2)
            me->comingTail = me->comingHead = 0;
          else
          {
            j = rand() % (FIELD_WID - 4);
            me->field[FIELD_HT - 1][j] = PopComing(me);
            me->field[FIELD_HT - 1][j + 4] = PopComing(me);
            if(me->comingTail != me->comingHead)
              me->field[FIELD_HT - 1][j + 2] = PopComing(me);
            if(me->comingTail != me->comingHead)
              me->field[FIELD_HT - 1][(j + 6) % FIELD_WID] = PopComing(me);
            me->state = STATE_CHECKED;
            break;
          }
        }

        me->x = (FIELD_WID / 2 - 1) << 16;
        me->y = (FIELD_HT << 16) - 0x1000;
        me->flip = 0;

        me->piece1 = me->next1;
        me->piece2 = me->next2;

        me->next1 = Random(me) % 3 + 1;
        me->next2 = Random(me) % 3 + 1;

        me->state = STATE_FALLING_PIECE;
        me->clock = g.now;
        me->nPieces++;
        DrawMeters(me);

        // fall through

      case STATE_FALLING_PIECE:
        j = ReadPad(curTurn);
        if(nPlayers == 1)
          j |= ReadPad(1 - curTurn);
        MakeRepeats(me->repeatTime, j, 18, 2);

        text_mode(16);
        textprintf(screen, g.lucidgray, me->originX + g.xOff, g.yOff + me->originY + 1, -1,
                   "%2d", me->comingHead - me->comingTail);

        /* check for keypresses and act on them */
        if(me->repeatTime[4] == 1) // flip left
          Try2Flip(me, (me->flip + 1) & 0x03);

        if(me->repeatTime[5] == 1) // flip right
          Try2Flip(me, (me->flip + 3) & 0x03);

        if(me->repeatTime[2] == 1 || me->repeatTime[2] == 17) // move left
        {
          if(!CheckOverlap(me, me->x - 0x10000, me->y, me->flip))
          {
            PlayEffect(moveSnd);
            me->x -= 0x10000;
          }
          else
            me->repeatTime[2] = 16; // let player slide the block in
        }

        if(me->repeatTime[1]) // move down
          me->y &= 0xffffc000;

        if(me->repeatTime[3] == 1 || me->repeatTime[3] == 17) // move right
        {
          if(!CheckOverlap(me, me->x + 0x10000, me->y, me->flip))
          {
            PlayEffect(moveSnd);
            me->x += 0x10000;
          }
          else
            me->repeatTime[3] = 16; // let player slide the block in
        }

        me->y -= 1310 + me->nPieces * 10;
        if(CheckOverlap(me, me->x, me->y, me->flip))
        {
          me->y += 1310 + me->nPieces * 10;
          LockInPiece(me);
          me->state = STATE_CHECK;
          me->clock = tim.odo + 25;
        }

        DrawCapsule(me);
        break;

      case STATE_CHECK:
        if(me->clock - tim.odo > 0)
          break;
        if(CheckLines(me))
          me->state = STATE_CHECKED;
        else
        {
          CheckLinesCleanup(me);
          me->state = STATE_NEW_PIECE;
        }
        
        me->clock = tim.odo + 25;
        break;

      case STATE_CHECKED:
        if(me->clock - tim.odo > 0)
          break;
        CheckLinesCleanup(me);
        DrawMeters(me);
        if(me->nVIR <= 0 && (!g.puyo || !g.garbage))
        {
          done = 1;
          rVal = curTurn + 1;
        }
        me->state = STATE_FALL;

      case STATE_FALL:
        if(me->clock - tim.odo > 0)
          break;
        if(Fall(me))
        {
          if(g.puyo && me->comingHead)
            me->state = STATE_BOULDERS;
        }
        else
          me->state = STATE_CHECK;

        me->clock = tim.odo + 15;
        break;

      case STATE_GAMEOVER:
        {
          BITMAP *bmp = GetResource(g.pix, "GAMEOVER");
          if(bmp)
          {
            draw_sprite(screen, bmp,
                        me->originX + (FIELD_WID * BLOCK_WID - bmp->w) / 2 + g.xOff,
                        me->originY - bmp->h + g.yOff);
          }
        }
        done = 1;
        break;

      // should only be sent to this state if in puyo mode
      case STATE_BOULDERS:
        if(g.garbage)
        {
          if(g.p[0].comingHead > g.p[1].comingHead)
          {
            g.p[0].comingHead -= g.p[1].comingHead;
            g.p[1].comingHead = 0;
          }
          else
          {
            g.p[1].comingHead -= g.p[0].comingHead;
            g.p[0].comingHead = 0;
          }
        }
        if(me->comingHead)
        {
          int i;

          for(i = 0; i < FIELD_WID; i++)
          {
            if((me->field[FIELD_HT - 1][i] == 0) &&
               (me->comingHead > FIELD_WID * 3 || (Random(me) & 1)))
            {
              me->field[FIELD_HT - 1][i] = 050; // row 5 col 0 is boulder
              me->comingHead--;
            }
            if(me->comingHead <= 0)
              i = FIELD_WID;
          }
        }

        me->state = STATE_FALL;

        break;

#if 0
      /* code left in from Furby's Avalanche */

      case STATE_GAMEOVER:
        text_mode(16);
        textout(screen, chicago, "   GAME OVER   ",
                curTurn ? 200 : 0, ME.y, 15);
        if((ME.y -= 2) < 0)
        {
          if(ME.score != 0)
          {
            ME.entry.score[3] = (tim.odo - ME.gameStart) / 70;
            ME.entry.score[2] = ME.blocks;
            ME.entry.score[1] = ME.score & 0xffff;
            ME.entry.score[0] = ME.score >> 16;
          }

          {
            rectfill(screen, curTurn ? 200 : 0, 16,
                     curTurn ? 319 : 119, 63, 16);
            textout(screen, font, "Push both flip",
                    curTurn ? 204 : 4, 184, 30);
            textout(screen, font, "keys to rejoin",
                    curTurn ? 204 : 4, 192, 28);

            ME.state = STATE_INACTIVE;
            lastEnter[curTurn] = 1;

            if(won && p[1 - curTurn].state == STATE_INACTIVE)
              g.doneTime = tim.odo;
            else
              g.doneTime = tim.odo + 350;
          }
        }
        break;

#endif
    }



    if(nPlayers > 1)
      curTurn = (curTurn + 1) % nPlayers;
    if(curTurn == 1)
    {
      // do all processing that applies to global variables

      while(g.now - tim.odo >= 0)
      {
        yield_timeslice();
        redraws = nPlayers;
      }
      g.now++;
      PollEffect();
    }

    if(redraws > 0)
    {
      BlitScanlines(me);
      Uncover(me);
      redraws--;
    }
  }

  clear_keybuf();
  text_mode(-1);

  if(rVal)
    PlayEffect(winSnd);
  else
    PlayEffect(loseSnd);

  if(rVal)
  {
    g.p[rVal - 1].wins++;
    DrawMeters(&(g.p[rVal - 1]));
  }

  j = 100;
  done = 0;
  while(!done)
  {
    while(g.now >= tim.odo)
      yield_timeslice();
    g.now++;
    if(j >= 0)
      j--;
    else if(keypressed())
    {
      readkey();
      done = 1;
    }
    else if((ReadPad(0) | ReadPad(1)) & 0x0c)
      done = 1;
    PollEffect();
    if(j == 0)
    {
      clear_keybuf();
      textout_centre(screen, g.lucidgray, "press a key", 256 + g.xOff, g.yOff + 364, -1);
    }
  }

  if(!rVal)
    {
      MIDI *song = GetResource(music_dat, "BARBIE_MID");
      if(song != 0 && jukeboxMode < 0)
        play_midi(song, FALSE);
    }


  return rVal;
}


static void ReadTextBuf(char *theText)
{
  DIALOG dlg[] =
  {
   /* (dialog proc)     (x)   (y)   (w)   (h)   (fg)  (bg)  (key) (flags)  (d1)  (d2)  (dp) */
   { d_box_proc,        0,    0,    512,  384,  0,    0,    0,    0,       0,    0,    NULL },
   { d_textbox_proc,    -1,   -1,   SCREEN_W+1,354,-1,0,    0,    0,       0,    0,    NULL },
   { d_button_proc,     384,  360,  64,   20,   -1,   0,    13,   D_EXIT,  0,    0,    "OK"},
   { NULL }
  };
  FONT *oldfont = font;

  font = g.lucidgray;

  dlg[1].dp = theText;
  do_dialog(dlg, 1);

  font = oldfont;
}

/* Copyright() *************************
 * Displays basic copyright information about this program.
 */
static void Copyright(void)
{
  int lastOdo = tim.odo;
  int kee;
  char buf[65536];

  MIDI *barbie = GetResource(music_dat, "BARBIE_MID");
  if(barbie != 0 && jukeboxMode < 0)
    play_midi(barbie, FALSE);

  klear(screen);

  // I normally set color 0 to hot pink (so I can tell transparent
  // areas from opaque areas), but I want it to show up on screen as black.
  g.pal[0].r = g.pal[0].g = g.pal[0].b = 0;
  set_palette(g.pal);
  text_mode(0);

  acquire_screen();
  textout(screen, g.lucidgray, (char *)program_name, 5 + g.xOff, g.yOff + 16, -1);
  textout(screen, g.lucidgray, "Copyright © 2000 Damian Yerrick", 5 + g.xOff, g.yOff + 40, -1);
  textout(screen, g.lucidgray, "Comes with ABSOLUTELY NO WARRANTY.", 5 + g.xOff, g.yOff + 56, -1);
  textout(screen, g.lucidgray, "This is free software, and you are welcome to", 5 + g.xOff, g.yOff + 72, -1);
  textout(screen, g.lucidgray, "redistribute it under certain conditions;", 5 + g.xOff, g.yOff + 88, -1);
  textout(screen, g.lucidgray, "read the file COPYING for details.", 5 + g.xOff, g.yOff + 104, -1);
  textout_centre(screen, g.lucidgray, "Press C for more credit information", 256 + g.xOff, g.yOff + 348, -1);
  textout_centre(screen, g.lucidgray, "Press a key to continue", 256 + g.xOff, g.yOff + 364, -1);
  release_screen();

  while(!keypressed())
  {
    PollEffect();
    while(lastOdo == tim.odo)
      yield_timeslice();
    lastOdo = tim.odo;
  }

  kee = readkey() & 0xff;
  if(kee == 'C' || kee == 'c')
  {
    DATAFILE *dat = GetResourceObj(g.pix, "THEME_TXT");

    strcpy(buf, credits);
    strcat(buf, "\n________________________________________________\n\n");
    if(dat)
      strncat(buf, dat->dat, dat->size);
    ReadTextBuf(buf);
  }
  klear(screen);
}


/* NotGlobals() ************************
 * Frees resources when the program exits.
 */
static void NotGlobals(void)
{
  if(g.pix)
    unload_datafile(g.pix);
  if(g.backbuf)
    destroy_bitmap(g.backbuf);
  remove_int(drm_timerint);
  set_color_depth(8);
  set_gfx_mode(GFX_AUTODETECT, 320, scanlines, 0, 0);
}


/* GetRLETiles() ***********************
 * Transforms the tiles into RLE sprites to work around a bug
 * in Allegro 3.9.32's masked_blit().
 */
static int GetRLETiles()
{
  int i = 0, x, y;
  BITMAP *bmp = create_bitmap(BLOCK_WID, BLOCK_HT);
  char octals[32];

  if(bmp == NULL)
    return -1;

  text_mode(-1);
  for(y = 0; y < 8 * BLOCK_HT; y += BLOCK_HT)
    for(x = 0; x < 8 * BLOCK_WID; x += BLOCK_WID)
    {
      blit(g.tiles, bmp, x, y, 0, 0, BLOCK_WID, BLOCK_HT);
      sprintf(octals, "%2o", i);
//      textout(bmp, g.lucid, octals, 0, 3, makecol(255, 255, 255));
      g.rletiles[i] = get_rle_sprite(bmp);
      if(g.rletiles[i] == NULL)
        return -1;
      i++;
    }
  return 0;
}

/* init_drm() **************************
 * Sets up the enVIRonment in which the game runs.
 * Most of this will be moved to Globals::Globals()
 */
static int init_drm(void)
{
  int ok;

  /* install timer handler */
  LOCK_FUNCTION(drm_timerint);
  LOCK_VARIABLE(tim);
  install_int(drm_timerint, 10);

  /* go to graphics mode */

  /* load datafiles */
  set_color_conversion(COLORCONV_NONE);
  g.pix = load_skinned_datafile(drm_dat);
  if(g.pix == NULL)
  {
    NotGlobals();
    alert("Could not open data file", drm_dat, strerror(errno), "OK", 0, 13, 0);
    return 1;
  }

/* old standalone vitamins allowed windowed use; had to play nice with wm
  ok = desktop_color_depth();
  if(ok)
#if defined(ALLEGRO_WINDOWS)
    set_color_depth(16);
#else
    set_color_depth(8);
#endif
*/
  if(scanlines == 200)
    g.preferredRes = 512;
  else
    g.preferredRes = 640;

  // open window
  g.preferredRes &= ~31;
  if(g.preferredRes < 512)
  {
#if defined(ALLEGRO_WINDOWS)
    if(set_gfx_mode(GFX_DIRECTX_WIN, 512, 384, 0, 0) < 0)
      if(set_gfx_mode(GFX_GDI, 520, 410, 0, 0) < 0)
#endif
        g.preferredRes = 512;
  }

  if(g.preferredRes >= 512)
    if(set_gfx_mode(GFX_AUTODETECT, g.preferredRes, g.preferredRes * 3/4, 0, 0) < 0)
      if(set_gfx_mode(GFX_AUTODETECT, 640, 480, 0, 0) < 0)
        if(set_gfx_mode(GFX_AUTODETECT, 800, 600, 0, 0) < 0)
          if(set_gfx_mode(GFX_AUTODETECT, 1024, 768, 0, 0) < 0)
            {
              alert("Could not go to 512x384 or higher res.",
                    "Be sure your graphics driver supports it",
#ifdef ALLEGRO_DOS
                    "(you need VESA and no WinNT)",
#else
                    "",
#endif

                    "OK", 0, 13, 0);
              return 1;
            }

  /* get stuff from datafiles */
  g.tiles = GetResource(g.pix, "PILL_L20");
  g.pal = GetResource(g.pix, "PCPAL");
  g.lucid = GetResource(g.pix, "LUCID_MONO");
  g.lucidgray = GetResource(g.pix, "LUCID");
  g.bkgnds[0] = GetResource(g.pix, "BKGND1");
  g.bkgnds[1] = GetResource(g.pix, "BKGND");
  if(g.tiles == NULL || g.pal == NULL || g.lucid == NULL ||
     g.bkgnds[0] == NULL || g.bkgnds[1] == NULL ||
     GetResource(g.pix, "TITLE") == NULL)
  {
    alert("The datafile is fucked up;", "reinstall?", "", "OK", 0, 13, 0);
    exit(1);
  }

  g.balance = 255;

  if(g.lucidgray == NULL)
    g.lucidgray = g.lucid;

  // convert colors
  g.pal[0].r = g.pal[0].b = 63;
  g.pal[0].g = 0;
  set_palette(g.pal);
  set_color_conversion(COLORCONV_TOTAL);
  fixup_datafile(g.pix);

  /* create a back buffer to eliminate flicker */
  g.backbuf = create_bitmap(512, 384);
  /* create a front buffer to center the screen */
  g.xOff = (SCREEN_W - 512) / 2;
  g.yOff = (SCREEN_H - 384) / 2;

  g.bkgnd = create_bitmap(512, 384);
  ok = GetRLETiles();
  if(g.backbuf == NULL || g.bkgnd == NULL || ok < 0)
  {
    char buf[1024] = "";
    NotGlobals();
    sprintf(buf, "%s could not allocate memory for buffers.\n"
            "Try closing some windows.\n",
            program_name);
    allegro_message(buf);
    exit(1);
  }

  GetSamples();

  return 0;
}


static int Options2P_redraw(DIALOG *dlg, int unused1)
{
  sprintf(dlg[8].dp, "%2d", dlg[6].d2);
  sprintf(dlg[9].dp, "%2d", dlg[7].d2);

  dlg[8].flags |= D_DIRTY;
  dlg[9].flags |= D_DIRTY;
  return D_O_K;
}

static int Options2P(Context *g)
{
  FONT *oldfont = font;
  int rval;
  char levelstr1p[8] = " 0";
  char levelstr2p[8] = " 0";

  DIALOG dlg[] =
  {
   /* (dialog proc)     (x)   (y)   (w)   (h)   (fg)  (bg)  (key) (flags)  (d1)  (d2)  (dp) */
   { d_shadow_box_proc, 0,    0,    256,  192,  0,    30,   0,    0,       0,    0,    NULL },
   { d_button_proc,     160,  168,  80,   20,   16,   30,   13,   D_EXIT,  0,    0,    "Play"},
   { d_button_proc,     64,   168,  80,   20,   16,   30,   27,   D_EXIT,  0,    0,    "Cancel"},
   { DY_check_proc,     32,   40,   200,  16,   16,   30,   'h',  0,       0,    0,    "Auto &handicap"},
   { DY_check_proc,     32,   60,   200,  16,   16,   30,   'g',  0,       0,    0,    "&Garbage"},
   { DY_check_proc,     32,   80,   200,  16,   16,   30,   'b',  0,       0,    0,    "&Boulders mode"},
   { d_slider_proc,     32,   134,  80,   16,   16,   30,   0,    0,       20,   0,    NULL, Options2P_redraw, dlg},
   { d_slider_proc,     144,  134,  80,   16,   16,   30,   0,    0,       20,   0,    NULL, Options2P_redraw, dlg},
   { d_text_proc,       112,  134,  20,   16,   16,   30,   0,    D_EXIT,  0,    0,    levelstr1p},
   { d_text_proc,       224,  134,  20,   16,   16,   30,   0,    D_EXIT,  0,    0,    levelstr2p},
   { d_text_proc,       32,   8,    192,  16,   16,   30,   0,    D_EXIT,  0,    0,    "2 Player Game"},
   { d_text_proc,       32,   114,  192,  16,   16,   30,   0,    D_EXIT,  0,    0,    "Initial level"},
   { NULL,              0,    0,    0,    0,    0,    0,    0,    0,       0,    0,    NULL }
  };

  if(g->autoHandicap)
    dlg[3].flags |= D_SELECTED;
  if(g->garbage)
    dlg[4].flags |= D_SELECTED;
  if(g->puyo)
    dlg[5].flags |= D_SELECTED;
  dlg[6].d2 = g->p[0].initialLevel;
  dlg[7].d2 = g->p[1].initialLevel;

  font = g->lucid;
  centre_dialog(dlg);
  set_dialog_color(dlg, makecol(0, 0, 0), makecol(255, 255, 255));
  Options2P_redraw(dlg, 0);
  rval = popup_dialog(dlg, 1);
  font = oldfont;

  g->autoHandicap = (dlg[3].flags & D_SELECTED) ? 1 : 0;
  g->garbage      = (dlg[4].flags & D_SELECTED) ? 1 : 0;
  g->puyo         = (dlg[5].flags & D_SELECTED) ? 1 : 0;
  g->p[0].initialLevel = g->p[0].level = dlg[6].d2;
  g->p[1].initialLevel = g->p[1].level = dlg[7].d2;

  return (rval == 1) ? 0 : -1;
}


static int Options1P_redraw(DIALOG *dlg, int unused1)
{
  sprintf(dlg[5].dp, "%2d", dlg[4].d2);

  dlg[5].flags |= D_DIRTY;
  return D_O_K;
}

static int Options1P(Context *g)
{
  FONT *oldfont = font;
  int rval;
  char levelstr1p[8] = " 0";

  DIALOG dlg[] =
  {
   /* (dialog proc)     (x)   (y)   (w)   (h)   (fg)  (bg)  (key) (flags)  (d1)  (d2)  (dp) */
   { d_shadow_box_proc, 0,    0,    256,  192,  0,    30,   0,    0,       0,    0,    NULL },
   { d_button_proc,     160,  168,  80,   20,   16,   30,   13,   D_EXIT,  0,    0,    "Play"},
   { d_button_proc,     64,   168,  80,   20,   16,   30,   27,   D_EXIT,  0,    0,    "Cancel"},
   { DY_check_proc,     32,   80,   200,  16,   16,   30,   'b',  0,       0,    0,    "&Boulders mode"},
   { d_slider_proc,     32,   134,  160,  16,   16,   30,   0,    0,       20,   0,    NULL, Options1P_redraw, dlg},
   { d_text_proc,       192,  134,  20,   16,   16,   30,   0,    D_EXIT,  0,    0,    levelstr1p},
   { d_text_proc,       32,   8,    192,  16,   16,   30,   0,    D_EXIT,  0,    0,    "1 Player Game"},
   { d_text_proc,       32,   114,  192,  16,   16,   30,   0,    D_EXIT,  0,    0,    "Initial level"},
   { NULL,              0,    0,    0,    0,    0,    0,    0,    0,       0,    0,    NULL }
  };

  if(g->puyo)
    dlg[3].flags |= D_SELECTED;
  dlg[4].d2 = g->p[1].initialLevel;

  font = g->lucid;
  centre_dialog(dlg);
  set_dialog_color(dlg, makecol(0, 0, 0), makecol(255, 255, 255));
  Options1P_redraw(dlg, 0);
  rval = popup_dialog(dlg, 1);
  font = oldfont;

  g->autoHandicap = 1;
  g->garbage = 0;
  g->puyo = (dlg[3].flags & D_SELECTED) ? 1 : 0;
  g->p[1].initialLevel = g->p[1].level = dlg[4].d2;

  return (rval == 1) ? 0 : -1;
}

static void Ending(void)
{
  int white = makecol(255, 255, 255);
  int lightgray = makecol(170, 170, 170);
  int bsodblue = makecol(0, 0, 170);

  clear2color(screen, bsodblue);
  text_mode(lightgray);
  textout_centre(screen, font, " Drm ", 256 + g.xOff, g.yOff + 144, bsodblue);
  text_mode(bsodblue);
  textout(screen, font, "This application has performed an illegal operation", 48 + g.xOff, g.yOff + 162, white);
  textout(screen, font, "and will be shut down.", 48 + g.xOff, g.yOff + 171, white);
  textout(screen, font, "If the problem persists, contact the program", 48 + g.xOff, g.yOff + 189, white);
  textout(screen, font, "vendor.", 48 + g.xOff, g.yOff + 198, white);

  textout_centre(screen, font, "Press the Any key to continue", 256 + g.xOff, g.yOff + 225, white);
  textout_centre(screen, font, "or C-M-DEL to restart your computer", 256 + g.xOff, g.yOff + 234, white);

  textout(screen, font, "WARNING: Pressing C-M-DEL will restart your", 48 + g.xOff, g.yOff + 261, white);
  textout(screen, font, "computer.  You will lose unsaved information in", 48 + g.xOff, g.yOff + 270, white);
  textout(screen, font, "all programs that are running.", 48 + g.xOff, g.yOff + 279, white);

  readkey();
  g.pal[0].r = g.pal[0].g = g.pal[0].b = 0;
  set_palette(g.pal);
  klear(screen);

  text_mode(0);
  textout(screen, g.lucidgray, "haven't coded the ending yet", 0, 0, -1);
  textout(screen, g.lucidgray, "the game coding department is waiting on the", 0, 16, -1);
  textout(screen, g.lucidgray, "demo coding department to do the end cinema.", 0, 32, -1);
  textout(screen, g.lucidgray, "what happens is jean-louis gassée of be inc. pops", 0, 64, -1);
  textout(screen, g.lucidgray, "up.  linus torvalds of transmeta corp. kills him", 0, 80, -1);
  textout(screen, g.lucidgray, "(how?).         steve jobs of apple computer comes", 0, 96, -1);
  textout(screen, g.lucidgray, "in and shoots linus.  while he's dancing around,", 0, 112, -1);
  textout(screen, g.lucidgray, "william h. gates iii comes in and axe murders jobs.", 0, 128, -1);
  textout(screen, g.lucidgray, "i've given enough away AOLready.", 0, 160, -1);
  textout(screen, g.lucidgray, "now type vc to view the credits", 0, 368, -1);
  readkey();
}

static int TitleScr(void)
{
  BITMAP *title = GetResource(g.pix, "TITLE");
  int play = 0, lastOdo = tim.odo, lastMouse = 0;

  blit(title, screen, 0, 0, g.xOff, g.yOff, 512, 384);

  show_mouse(screen);

  while(!play)
  {
    if(mouse_b)
      lastMouse = 1;
    else if(lastMouse == 1)
    {
      if(mouse_y < SCREEN_H / 2 - 100)
      {
        if(mouse_x < SCREEN_W / 2)
          play = -1;
        else
        {
          Options();
          lastMouse = 0;
        }
      }
      else if(mouse_x < SCREEN_W / 2)
        play = 1;
      else
        play = 2;
    }

    if(keypressed())
    {
      switch(readkey() >> 8)
      {
      case KEY_1:
        play = 1;
        break;
      case KEY_2:
        play = 2;
        break;
      case KEY_ESC:
        play = -1;
        break;
      }
    }
    while(lastOdo == tim.odo)
      yield_timeslice();
    lastOdo = tim.odo;
    PollEffect();
  }

  show_mouse(NULL);

  return play;
}


void DrmMain(void)
{
  int nPlayers, winner;

  g.preferredRes = 512;

  if(init_drm())
    return;
  Copyright();

  while((nPlayers = TitleScr()) > 0)
  {
    if(nPlayers == 2)
    {
      if(Options2P(&g) < 0)
        continue;
    }
    else
    {
      if(Options1P(&g) < 0)
        continue;
    }

    NewGame(&g, nPlayers);
    while((winner = Play(nPlayers)) > 0)
    {
      if(g.autoHandicap)
        g.p[winner - 1].level++;
      if(g.p[winner - 1].level >= 25)
      {
        klear(screen);
        PlayEffect(NULL);
        Ending();
        Copyright();
        break;
      }
    }
  }

  NotGlobals();
}
