/*
   * Based * on dc6con by Ryan Smith, for the understanding of the dc6 format,
   but none of his code is in there (just the structs).
   As for me, you can do whatever you want with this code

   Paul Siramy

   Creation : 15 March 2002

   Updates :
      * 12 december 2005
        source code heavily revised for MSVC6 and allegro 4.0.3
*/


// ========================================================================================
// INCLUDES
// ========================================================================================
#include <stdio.h>

// Disable warning message "unreferenced inline function has been removed"
#pragma warning( disable : 4514 )

#define USE_CONSOLE
#include <allegro.h>



// ========================================================================================
// DATA TYPES
// ========================================================================================

// new data type for conveniance
typedef unsigned char UBYTE; // 1 byte

// dc6 headers
#pragma pack(1)

   typedef struct
   {
      long  version;        // 0x00000006
      long  sub_version;    // 0x00000001
      long  zeros;          // 0x00000000
      UBYTE termination[4]; // 0xEEEEEEEE or 0xCDCDCDCD
      long  directions;     // 0x000000xx
      long  frames_per_dir; // 0x000000xx
   } DC6_HEADER_S;

   typedef struct
   {
	   long flip;
	   long width;
	   long height;
	   long offset_x;
	   long offset_y; // from bottom border, NOT upper border
	   long zeros;
	   long next_block;
	   long length;
   } DC6_FRAME_HEADER_S;

#pragma pack()



// ========================================================================================
// GLOBAL VARIABLES
// ========================================================================================
PALETTE            dc6_pal;
DC6_HEADER_S       dc6_header;
long               * dc6_frame_ptr    = NULL;
DC6_FRAME_HEADER_S * dc6_frame_header = NULL;
BITMAP             * dc6_bmp          = NULL,
                   * final_bmp        = NULL;

// colormaps miscelaneous datas

#define DC6CMAPS_FAMILY  6
#define DC6CMAPS_TINT   21

UBYTE colormap[DC6CMAPS_FAMILY][DC6CMAPS_TINT][256];

char cmap_name[DC6CMAPS_FAMILY][20] = {
        {"grey.dat"},         // 1
        {"grey2.dat"},        // 2
//        {"brown.dat"},
//        {"gold.dat"},
        {"greybrown.dat"},    // 5
        {"invgrey.dat"},      // 6
        {"invgrey2.dat"},     // 7
        {"invgreybrown.dat"}  // 8
     };
int cmap_idx[DC6CMAPS_FAMILY] = {1, 2, 5, 6, 7, 8};



// ========================================================================================
// output on both stdout and stderr
// ========================================================================================
void double_output(char * string)
{
   fprintf(stdout, "%s", string);
   fflush(stdout);

   fprintf(stderr, "%s", string);
   fflush(stderr);
}


// ========================================================================================
// read dc6 main header and all frame headers
//    (we read all frame headers tough we'll only use the 1st frame later)
//
// return 0 on sucess, -1 if error
// ========================================================================================
int read_dc6_headers(FILE * in)
{
   long i, nb, size;
   

   // get main dc6 header
   fread( & dc6_header,  sizeof(DC6_HEADER_S), 1, in);

   // get total number of frames in the dc6
   nb = dc6_header.directions * dc6_header.frames_per_dir;

   // table of in-file pointers
   size = sizeof(long) * nb;
   dc6_frame_ptr = (long *) malloc(size);
   if (dc6_frame_ptr == NULL)
   {
      printf("read_dc6_headers(), ERROR : not enough memory for %li in-file pointers\n", nb);
      return -1;
   }
   memset(dc6_frame_ptr, 0, size);

   // fill table of in-file pointers
   fread(dc6_frame_ptr, size, 1, in);

   // table of frame headers
   size = sizeof(DC6_FRAME_HEADER_S) * nb;
   dc6_frame_header = (DC6_FRAME_HEADER_S *) malloc(size);
   if (dc6_frame_header == NULL)
   {
      free(dc6_frame_ptr);
      printf("read_dc6_headers(), ERROR : not enough memory for %li frame headers\n", nb);
      return -1;
   }

   // fill table of frame headers
   for (i=0; i < nb; i++)
   {
      fseek(in, dc6_frame_ptr[i], SEEK_SET);
      fread( & dc6_frame_header[i], sizeof(DC6_FRAME_HEADER_S), 1, in);
   }

   // end
   return 0;
}



// ========================================================================================
// decompress the pixels of 1 frame
// ========================================================================================
void decompress_dc6(FILE * in, int frame)
{
   DC6_FRAME_HEADER_S * fh;
   long               i, i2, pos;
   int                c, c2, x, y;
   

   fh = & dc6_frame_header[frame];

   if ((fh->width <= 0) || (fh->height <= 0))
      return;

   dc6_bmp = create_bitmap(fh->width, fh->height);
   if (dc6_bmp == NULL)
      return;

   clear(dc6_bmp);

   pos = dc6_frame_ptr[frame] + sizeof(DC6_FRAME_HEADER_S);
   fseek(in, pos, SEEK_SET);

   x = 0;
   y = fh->height - 1;

   for (i=0; i < fh->length; i++)
   {
      c = fgetc(in);

      if (c == 0x80)
      {
         x = 0;
         y--;
      }
      else if (c & 0x80)
         x += c & 0x7F;
      else
      {
         for (i2=0; i2 < c; i2++)
         {
            c2 = fgetc(in);
            i++;
            putpixel(dc6_bmp, x, y, c2);
            x++;
         }
      }
   }
}



// ========================================================================================
// load the standard Diablo II act 1 palette
// ========================================================================================
void load_palette(void)
{
   FILE * in;
   int  i;
   char * d2pal = "datas\\act1.dat";

   
   in = fopen(d2pal, "rb");
   if (in == NULL)
   {
      printf("\nload_palette(), ERROR : can't open palette : %s\n", d2pal);
      exit(1);
   }

   printf("load palette %s\n", d2pal);
   for (i=0; i < 256; i++)
   {
      // order is BGR, not RGB
      dc6_pal[i].b = fgetc(in) >> 2;
      dc6_pal[i].g = fgetc(in) >> 2;
      dc6_pal[i].r = fgetc(in) >> 2;
      /*
         >> 2 is needed because the internal range of the RGB componants
         in allegro (8 bpp mode) is from 0 to 63 (because of the mode 13h),
         not 0 to 255. Since this program only output bitmaps for the purpose
         of tests, and don't create dc6 files, it's ok.
      */
   }
   fclose(in);
}



// ========================================================================================
// build the final bitmap with all variations of the colomaped items
// ========================================================================================
void make_final_bmp(char * filename)
{
   char tmp_str[80];
   int  w, h, ww, hh, x, y, f, t, c;


   w  = dc6_bmp->w;
   h  = dc6_bmp->h;
   ww = w * (DC6CMAPS_TINT + 1) + 180;
   hh = h * DC6CMAPS_FAMILY + 30;
   final_bmp = create_bitmap(ww, hh);
   if (final_bmp == NULL)
      return;
   clear(final_bmp);

   for (f=0; f < DC6CMAPS_FAMILY; f++)
   {
      for (t=0; t < (DC6CMAPS_TINT + 1); t++)
      {
         if (t==0)
         {
            if (f==0)
               blit(dc6_bmp, final_bmp, 0, 0, 0, 16, w, h);
         }
         else
         {
            for (y=0; y < h; y++)
            {
               for (x=0; x < w; x++)
               {
                  c = getpixel(dc6_bmp, x, y);
                  if (c)
                     putpixel(final_bmp, w*t+x, h*f+y+16, colormap[f][t-1][c]);
               }
            }
         }
      }
      sprintf(tmp_str, "%i : %s", cmap_idx[f], cmap_name[f]);
      textout(final_bmp, font, tmp_str, (w * t) + 8, (h * f) + (h / 2) + 12, 255);
   }
   for (t=1; t < (DC6CMAPS_TINT + 1); t++)
   {
      sprintf(tmp_str, "%02i", t-1);
      textout(final_bmp, font, tmp_str, (w * t) + (w / 2) - 8, (h * f) + 20, 255);
   }
   textout(final_bmp, font, filename, 4, 4, 255);
}



// ========================================================================================
// entry point
// ========================================================================================
int main(void)
{
   struct al_ffblk ff;
   FILE   * in;
   char * cmap_dir = "datas\\colormaps", tmp[80], tmp2[80];
   int  f, t, i, done;


   // reassign stdout to a text file
   freopen("log.txt", "w", stdout);

   // standard Diablo II act 1 palette
   load_palette();

   // all colormaps
   double_output("loading colormap :\n");
   for (f=0; f < DC6CMAPS_FAMILY; f++)
   {
      sprintf(tmp, "%s\\%s", cmap_dir, cmap_name[f]);
      in = fopen(tmp, "rb");
      if (in == NULL)
      {
         double_output("can't open ");
         double_output(tmp);
         double_output("\n");
         exit(1);
      }
      double_output("   ");
      double_output(tmp);
      double_output("\n");
      for (t=0; t < DC6CMAPS_TINT; t++)
      {
         for (i=0; i < 256; i++)
            colormap[f][t][i] = fgetc(in);
      }
      fclose(in);
   }

   // allegro inits
   allegro_init();
   set_color_depth(8);
   
   // process all dc6 one by one
   strcpy(tmp, "dc6\\*.dc6");
   done = al_findfirst(tmp, & ff, FA_RDONLY | FA_HIDDEN | FA_SYSTEM | FA_ARCH);
   double_output("processing :\n");
   while ( ! done)
   {
      sprintf(tmp2, "dc6\\%s", ff.name);
      in = fopen(tmp2, "rb");
      if (in == NULL)
      {
         printf("can't open %s\n", tmp2);
         exit(1);
      }
      double_output("   ");
      double_output(tmp2);
      double_output("\n");
      if (read_dc6_headers(in) != 0)
      {
         fclose(in);
         exit(1);
      }
      decompress_dc6(in, 0);
      fclose(in);

      make_final_bmp(ff.name);

      strcpy(tmp2, ff.name);
      tmp2[strlen(tmp2)-4] = 0;
      strcat(tmp2, ".pcx");
      save_bitmap(tmp2, final_bmp, dc6_pal);

      destroy_bitmap(final_bmp);
      destroy_bitmap(dc6_bmp);
      free(dc6_frame_header);
      free(dc6_frame_ptr);

      final_bmp        = NULL;
      dc6_bmp          = NULL;
      dc6_frame_header = NULL;
      dc6_frame_ptr    = NULL;

      done = al_findnext(&ff);
   }
   double_output("all done\n");

   // end
   return 0;
}
