#include "PlayArea.h"
#include <time.h>
#include <stdio.h>


SPlayArea::SPlayArea() {
	SetW(450);
	SetH(450);
	Reset();

	timer = -1;

	// Setup the number of frames the snake will wait before crashing
	// int a wall (different for each speed level)
	maxWait[0] = 2;
	maxWait[1] = 2;
	maxWait[2] = 2;
	maxWait[3] = 3;
	maxWait[4] = 3;
	maxWait[5] = 4;
	maxWait[6] = 5;
	maxWait[7] = 6;

	editMode = false;


	// load the samples
	LoadSamples();
}


SPlayArea::~SPlayArea() {
}


void SPlayArea::LoadSettings(const char *file) {
	// Create some beautifull colours
	set_config_file(file);
	char str[32];

	ustrncpy(str, get_config_string("COLORS", "wall1", "60,120,140"), 32);
	set_config_string("COLORS", "wall1", str);
	color[WALL][0] = StringToColor(str);
	ustrncpy(str, get_config_string("COLORS", "wall2", "100,160,200"), 32);
	set_config_string("COLORS", "wall2", str);
	color[WALL][1] = StringToColor(str);
	ustrncpy(str, get_config_string("COLORS", "wall3", "40,80,100"), 32);
	set_config_string("COLORS", "wall3", str);
	color[WALL][2] = StringToColor(str);

	ustrncpy(str, get_config_string("COLORS", "floor1", "240,240,220"), 32);
	set_config_string("COLORS", "floor1", str);
	color[FLOOR][0] = StringToColor(str);
	ustrncpy(str, get_config_string("COLORS", "floor2", "244,244,224"), 32);
	set_config_string("COLORS", "floor2", str);
	color[FLOOR][1] = StringToColor(str);
	ustrncpy(str, get_config_string("COLORS", "floor3", "236,236,216"), 32);
	set_config_string("COLORS", "floor3", str);
	color[FLOOR][2] = StringToColor(str);

	ustrncpy(str, get_config_string("COLORS", "snake1", "240,80,120"), 32);
	set_config_string("COLORS", "snake1", str);
	color[SNAKE][0] = StringToColor(str);
	ustrncpy(str, get_config_string("COLORS", "snake2", "248,120,140"), 32);
	set_config_string("COLORS", "snake2", str);
	color[SNAKE][1] = StringToColor(str);
	ustrncpy(str, get_config_string("COLORS", "snake3", "200,60,80"), 32);
	set_config_string("COLORS", "snake3", str);
	color[SNAKE][2] = StringToColor(str);
	
	ustrncpy(str, get_config_string("COLORS", "food1", "120,180,60"), 32);
	set_config_string("COLORS", "food1", str);
	color[FOOD][0] = StringToColor(str);
	ustrncpy(str, get_config_string("COLORS", "food2", "140,210,75"), 32);
	set_config_string("COLORS", "food2", str);
	color[FOOD][1] = StringToColor(str);
	ustrncpy(str, get_config_string("COLORS", "food3", "100,150,45"), 32);
	set_config_string("COLORS", "food3", str);
	color[FOOD][2] = StringToColor(str);
	
	ustrncpy(str, get_config_string("COLORS", "border", "0,0,0"), 32);
	set_config_string("COLORS", "border", str);
	color[BORDER][0] = StringToColor(str);

	flush_config_file();
}



void SPlayArea::LoadSamples() {
	DATAFILE *samples = load_datafile("samples.dat");
	if (!samples) return;

	DATAFILE *d;
	int i;
	char *sampleName[] = { 	"SAMPLE_CRASH",
							"SAMPLE_EAT",
							"SAMPLE_START",
							"SAMPLE_LIFE",
							"SAMPLE_WALL" };


	// Look for each sample inside the dat file and load it if it exists
	for (i=0; i<5; i++) {
		d = find_datafile_object(samples, sampleName[i]);
		if (d != NULL) {
			theSkin->SetSample((SAMPLE *)d->dat, SAMPLE_CRASH+i, true);
		}
	}

	unload_datafile(samples);
}


// Convert a string to color where string is in standard def_col format (e.g. '64,128,200')
int SPlayArea::StringToColor(char *str) {
	char *tok;
	tok = ustrtok(str, ", ;");	int r = (int)(uatof(tok));
	tok = ustrtok(0, ", ;");	  int g = (int)(uatof(tok));
	tok = ustrtok(0, ", ;");	  int b = (int)(uatof(tok));
	return makecol(r, g, b);
}


void SPlayArea::MsgDraw() {
	BITMAP *bmp = Parent()->dscreen;
	if (!bmp)	return;

	int r,g,b;
	r = getr(color[FLOOR][0])*0.75f;
	g = getg(color[FLOOR][0])*0.75f;
	b = getb(color[FLOOR][0])*0.75f;

	// Draw the 50x50 grid: each tile has its own colour
	int iy, ix;
	for (iy=0; iy<50; iy++)	{
		for (ix=0; ix<50; ix++) {
			if (editMode) {
				int col = (matrix[ix][iy] == WALL) ? color[WALL][0] : color[FLOOR][0];
				int sh1 = (matrix[ix][iy] == WALL) ? color[WALL][1] : color[FLOOR][1];
				int sh2 = (matrix[ix][iy] == WALL) ? color[WALL][2] : color[FLOOR][2];
				MAS3DFrame(bmp, x()+ix*9, y()+iy*9, x()+ix*9+8, y()+iy*9+8, col, sh1, sh2);
				rect(bmp, x()+ix*9, y()+iy*9, x()+ix*9+8, y()+iy*9+8, makecol(r,g,b));
			}
			else {
				MAS3DFrame(bmp, x()+ix*9, y()+iy*9, x()+ix*9+8, y()+iy*9+8, color[matrix[ix][iy]][0], color[matrix[ix][iy]][1], color[matrix[ix][iy]][2]);
			}
		}
	}

	r *= 0.75f;
	g *= 0.75f;
	b *= 0.75f;
	if (editMode) {
		MAS3DFrame(bmp, x()+startx*9, y()+starty*9, x()+(startx+1)*9, y()+(starty+1)*9, color[SNAKE][0], color[SNAKE][1], color[SNAKE][2]);
		for (ix=0; ix<50; ix += 5) {
			hline(bmp, x(), y()+ix*9, x2()-1, makecol(r,g,b));
			vline(bmp, x()+ix*9, y(), y2()-1, makecol(r,g,b));
		}
	}

	rect(Parent()->dscreen, x()-1, y()-1, x2(), y2(), color[BORDER][0]);

	MakeClean();
	InvalidateRect();
}


bool SPlayArea::MsgWantfocus() {
	return true;
}


void SPlayArea::MsgClick(void) {
	if (!editMode)
		return;

	int tx, ty;

	tx = (mouse_x - x())/9;
	ty = (mouse_y - y())/9;

	if (mouse_b & 1) {
		if (!(starty == ty && startx >= tx && startx <= tx + 3)) {
			if (matrix[tx][ty] == FLOOR)
				matrix[tx][ty] = WALL;
			else
				matrix[tx][ty] = FLOOR;
		}
	}
	else {
		if (tx >= 3 && matrix[tx][ty] != WALL
					&& matrix[tx-1][ty] != WALL
					&& matrix[tx-2][ty] != WALL
					&& matrix[tx-3][ty] != WALL)
		{
			startx = tx;
			starty = ty;
		}
	}

	MsgDraw();
	Parent()->DRSRedraw();
}



bool SPlayArea::MsgChar(int c) {
	if (!handledLastKey)
		return true;

	int cc = c>>8;

	if (!isPlaying) {
		if (cc == KEY_SPACE) {
			theSkin->PlaySample(SAMPLE_START);
			isPlaying = true;
		}
		else if (cc == KEY_LEFT || cc == KEY_RIGHT || cc == KEY_UP || cc == KEY_DOWN)
			return true;
		else
			return false;
	}

	lastdir = direction;
	if		(cc == KEY_LEFT)	direction = MOVE_LEFT;
	else if (cc == KEY_RIGHT)	direction = MOVE_RIGHT;
	else if (cc == KEY_UP)		direction = MOVE_UP;
	else if (cc == KEY_DOWN)	direction = MOVE_DOWN;
	else return false;

	// You can't reverse your direction (you can only turn left or right)
	if (lastdir * direction == MOVE_LEFT * MOVE_RIGHT)	direction = lastdir;
	if (lastdir * direction == MOVE_UP * MOVE_DOWN)		direction = lastdir;

	handledLastKey = false;
	return true;
}


void SPlayArea::MoveSnake() {
	// If no direction is defined, we can't move
	if (!direction) return;

	int old_score = score;

	// If the snake is about to crash then wait for a moment or two to give the
	// player one last chance to change the direction
	if (!CanMoveNext() && wait == -1) {
		theSkin->PlaySample(SAMPLE_WALL);
		wait = maxWait[speed-1];
		score -= (3*speed + growthLevel) * 10;
		if ((score % 10000) > (old_score % 10000)) score = old_score - old_score%10000;
		return;
	}

	if (wait > 0) {
		wait--;
		return;
	}

	wait = -1;

	int i;

	// See if we ate something
	if (HaveFood())	{
		theSkin->PlaySample(SAMPLE_EAT);
		growth += growthLevel;		// make the snake grow some more
		there_is_food = false;		// make a note that there is no food in the playfield
		drop = 10;					// we'll wait this long before dropping the next piece of food

		// increase the total score
		old_score = score;
		float ds = (speed+1)*(speed+2)*(speed+growthLevel);	// the number of points awarded for eating a piece of
		ds = ds * ((600.0f - steps) / 100.0f);	// food depends on speed, growth level and the number
		ds /= 6.0f;										// of steps taken
		ds = ds >= 1.0f ? ds : 0.0f;
		score += (int)ds;
		steps = 0;

		// go to next level after every xxx pieces of food
		pieces++;
		if (pieces >= maxPieces) {
			level++;
			// when all the levels are finnished start at the beginning only this
			// time the player has to eat 10 more pieces per level
			if (level > MAXLEVEL) {
				level = 0;
				maxPieces += 10;
			}
			pieces = 0;
			RestartLevel(level);
			score += (speed + growthLevel)*100;
		}

		// award an extra life every xxx points
		if ((score % 10000) < (old_score % 10000)) {
			theSkin->PlaySample(SAMPLE_LIFE);
			lives++;
		}
	}
	else {
		steps++;
	}

	// Clear the last tile of the snake body if we are not growing; increase length if we are
	if (growth > 0) {
		growth--;
		length++;
	}
	else {
		DrawTile(x() + 9*snake[length-1][0], y() + 9*snake[length-1][1], color[FLOOR][0], color[FLOOR][1], color[FLOOR][2], BLEND, 20);
		matrix[snake[length-1][0]][snake[length-1][1]] = FLOOR;
	}

	// Move each segment of the snake forward (except the first one, we'll do that later)
	for (i=length-1; i>0; i--) {
		snake[i][0] = snake[i-1][0];
		snake[i][1] = snake[i-1][1];
	}

	// Get the direction vector
	int dx, dy;
	GetDxDy(&dx, &dy);


	// Now do the first segment (the head)
	snake[0][0] += dx;
	snake[0][1] += dy;

	// Do the food dropping routine
	if (!there_is_food)	{
		drop--;
		if (!drop) {
			DropFood();
			there_is_food = true;
			DrawTile(x() + 9*foodx, y() + 9*foody, color[FOOD][0], color[FOOD][1], color[FOOD][2], BLEND, 20);
		}
	}

	// No comment
	if (!CanMove())
		Crashed();

	// Move the head of the snake on a new tile
	int animation = NONE;
	if (dx == 1)	animation = ROLL_RIGHT;	else
	if (dx == -1)	animation = ROLL_LEFT;	else
	if (dy == 1)	animation = ROLL_DOWN;	else
	if (dy == -1)	animation = ROLL_UP;
	matrix[snake[0][0]][snake[0][1]] = SNAKE;
	DrawTile(x() + 9*snake[0][0], y() + 9*snake[0][1], color[SNAKE][0], color[SNAKE][1], color[SNAKE][2], animation, (10-speed)>>1);

	// We have just handled a keypress
	handledLastKey = true;
}


// See if the move we made was valid (the head of the snake has to be either on the
// floor on a food piece)
bool SPlayArea::CanMove() {
	if (!isPlaying)
		return true;

	// Wrap around the playfield if necessary
	snake[0][0] += 50;
	snake[0][1] += 50;
	snake[0][0] %= 50;
	snake[0][1] %= 50;

	if (matrix[snake[0][0]][snake[0][1]] != FLOOR && matrix[snake[0][0]][snake[0][1]] != FOOD)
		return false;

	return true;
}


// See if the snake is about to crash if doesn't change direction
bool SPlayArea::CanMoveNext() {
	if (!isPlaying)
		return true;

	int nextx, nexty;
	nextx = snake[0][0];
	nexty = snake[0][1];

	int dx, dy;
	GetDxDy(&dx, &dy);
	nextx += dx;
	nexty += dy;

	nextx += 50;
	nexty += 50;
	nextx %= 50;
	nexty %= 50;

	if (matrix[nextx][nexty] != FLOOR && matrix[nextx][nexty] != FOOD)
		return false;

	return true;
}


// See if we just ate something (if the head is on top of a food piece)
bool SPlayArea::HaveFood() {
	if (snake[0][0] == foodx && snake[0][1] == foody) {
		foodx = -1;		// this will remove the food piece from the playfield
		foody = -1;
		return true;
	}
	else
		return false;
}


// Drop a piece of food on a random floor tile
void SPlayArea::DropFood() {
	// First clear any leftovers
	if (foodx != -1)
		matrix[foodx][foody] = FLOOR;

	srand((unsigned)time(NULL));
	bool found = false;
	// Look for an empty floor tile
	while (!found) {
		foodx = rand() % 50;
		foody = rand() % 50;
		if (matrix[foodx][foody] == FLOOR)
			found = true;
	}

	// Drop the food
	matrix[foodx][foody] = FOOD;
}


// Get the direction vector (how many tiles to move on the x and y axis
// for a particular direction)
void SPlayArea::GetDxDy(int *dx, int *dy) {
	switch (direction) {
	case MOVE_UP:
		*dx = 0;
		*dy = -1;
		break;
	case MOVE_LEFT:
		*dx = -1;
		*dy = 0;
		break;
	case MOVE_RIGHT:
		*dx = 1;
		*dy = 0;
		break;
	case MOVE_DOWN:
		*dx = 0;
		*dy = 1;
		break;
	default:
		*dx = 0;
		*dy = 0;
		break;
	};
}


// Reset the playfield and the snake before starting a new game
void SPlayArea::Reset() {
	gameOver = false;	// we're just starting a game
	length = 4;			// initial length
	score = 0;			// initial score
	speed = 4;			// default speed
	lives = 3;			// initial number of lives
	growthLevel = 3;	// default growth level
	level = 0;
	pieces = 0;
	wait = -1;
	maxPieces = 20;
	RestartLevel(level);
}


// Load an arena
void SPlayArea::LoadArena(int l) {
	DATAFILE *levelDat = load_datafile("levels.dat");
	if (!levelDat) return;

	char *data = (char *)levelDat[l].dat;

	int tile, i=0;
	for (int y=0; y<50; y++) {
		for (int x=0; x<50; x++) {
			tile = data[i++];
			if (tile != WALL) tile = FLOOR;
			matrix[x][y] = tile;
		}
	}

	startx = data[i++];
	starty = data[i++];

	unload_datafile(levelDat);
}

void SPlayArea::RestartLevel(int l) {
	direction = MOVE_RIGHT;
	lastdir = 0;
	growth = 0;
	length = 4;
	isPlaying = false;			// we're not playing until the user wants to
	drop = 10;					// we'll wait this long before dropping food
	there_is_food = false;		// there is no food yet
	foodx = -1;
	foody = -1;
	steps = 0;					// we haven't taken any steps yet
	gameOver = false;

	// Initialize the snake array with -1's
	int i;
	for (i=0; i<1000; i++) {
		snake[i][0] = -1;
		snake[i][1] = -1;
	}

	LoadArena(l);

	// Calculate dx and dy (according to direction)
	handledLastKey = true;
	int dx, dy;
	GetDxDy(&dx, &dy);

	// Put a snake into the playfield
	int cx, cy;
	cx = startx;
	cy = starty;
	for (i=0; i<length; i++) {
		matrix[cx][cy] = SNAKE;		// the matrix represents the arena
		snake[i][0] = cx;			// the array represents the snake (0 byte is the x
		snake[i][1] = cy;			// and the 1 byte is the y coordinate)
		cx -= dx;
		cy -= dy;
	}

	if (Parent()) {
		if (Parent()->dscreen) {
			MsgDraw();
			Parent()->DRSRedraw();
		}
	}
}

// Do some reseting etc. when the snake has crashed into a wall or something
void SPlayArea::Crashed() {
	int len = length;
	theSkin->PlaySample(SAMPLE_CRASH);
	RestartLevel(level);
	growth = len - 4;		// calculate how much the snake will grow
	length = 4;

	// See if we have any lives left
	lives--;				// we crashed -> we lose a life
	if (lives < 0) {		// we have no lives left :(
		lives = 0;
		gameOver = true;
		isPlaying = true;	// this has to be true for some reason
	}
}


void SPlayArea::Save(char *file) {
	FILE *f = fopen(file, "wb");
	if (!f) return;

	int tile;

	for (int y=0; y<50; y++) {
		for (int x=0; x<50; x++) {
			tile = matrix[x][y];
			tile = tile == WALL ? tile : FLOOR;
			fputc(tile, f);
		}
	}

	fputc(startx, f);
	fputc(starty, f);

	fclose(f);
}


void SPlayArea::Load(char *file) {
	FILE *f = fopen(file, "rb");
	if (!f) return;

	int tile;

	for (int y=0; y<50; y++) {
		for (int x=0; x<50; x++) {
			tile = fgetc(f);
			if (tile != WALL) tile = FLOOR;
			matrix[x][y] = tile;
		}
	}

	startx = fgetc(f);
	starty = fgetc(f);

	fclose(f);

	if (Parent()) {
		if (Parent()->dscreen) {
			MsgDraw();
			Parent()->DRSRedraw();
		}
	}
}


void SPlayArea::MsgTimer(int t) {
	if (t == timer) {
		Parent()->HandleEvent(*this, MSG_TIMER);
	}
}
