#include "CTerminal.h"
#include <limits>

#include "Logfile.hpp"

using namespace std;

CTerminal::CTerminal()
{
    m_str_max_length = 40;
    m_state = STATE_IDLE;
    m_prompt_str = ">: ";
    m_border_dist = 5;
    m_string_count = 0;
    m_akt_string = 0;
    m_string_x = m_first_string_y = m_border_dist;
    m_pinput_str = NULL;
    m_delay_count = 0;
    m_delayed_state = STATE_IDLE;
    m_type_delay_ticks_std = m_type_delay_ticks = 7;
    m_repeat_count = 0;

    m_cursor_count = 0;
    m_cursor_on_ticks = 15;
    m_cursor_off_ticks = 15;
    m_cursor_on = false;
    m_wordwrap = true;
    m_input_after_typing = false;
}

CTerminal::~CTerminal()
{
    for(int i = 0; i < m_strings.size(); ++i)
    {
        if (m_strings[i])
        {
            delete m_strings[i];
            m_strings[i] = NULL;
        }
    }
}

void CTerminal::InitRes(DATAFILE * dat)
{
}

void CTerminal::Size_Changed(void) 
{
    ResizeStringArray();
}

void CTerminal::ScrollStringArray(void)
{
    string* s = m_strings[0];
    for (int i = 0 ; i < m_string_count-1 ; ++i)
    {
        m_strings[i] = m_strings[i+1];
    }
    m_strings[m_string_count-1] = s;
    m_strings[m_string_count-1]->clear();
    if(m_akt_string > 0) --m_akt_string;
}

void CTerminal::NewLine(void)
{
    m_akt_string++;
    if(m_akt_string >= m_string_count) ScrollStringArray();
}

void CTerminal::ResizeStringArray(void)
{
    int string_count = (H() - 2 * m_border_dist) / text_height(font);
    m_first_string_y = (H() - string_count * text_height(font)) / 2;
    if(string_count > m_strings.size())
    {
        int i = m_strings.size();
        // Resize the vector
        m_strings.resize(string_count);
        // add the missing strings
        for( ; i < string_count; ++i)
        {
            m_strings[i] = new string();
        }
    }
    else if(string_count < m_strings.size())
    {
        // scroll strings up until actual string is the last visible
        while(m_akt_string > string_count-1) ScrollStringArray();
        // delete unnecessary strings
        for(int i = string_count ; i < m_strings.size(); ++i)
        {
            if(m_strings[i])
            {
                delete m_strings[i];
                m_strings[i] = NULL;
            }
        }
        // Resize the vector
        m_strings.resize(string_count);
    }
    m_string_count = string_count;
    m_strings.resize(m_string_count);
    m_str_max_length = (W() - 2 * m_border_dist) / text_length(font, "X");
    m_string_x = (W() - m_str_max_length * text_length(font, "X"))/2;
}

void CTerminal::Clear(void)
{
    for(int i = 0; i < m_strings.size(); ++i)
    {
        m_strings[i]->clear();
    }
    m_akt_string = 0;
}

void CTerminal::Input(void)
{
    m_finished = false;
    m_state = STATE_INPUT;
    if(m_strings[m_akt_string]->size()) NewLine();
    m_pinput_str = m_strings[m_akt_string];
    *m_pinput_str = m_prompt_str;
    m_cursor_count = m_cursor_off_ticks;
    m_cursor_on = false;
    clear_keybuf();
}

void CTerminal::Type(const std::string &str, bool input_after_typing)
{
    m_finished = false;
    m_input_after_typing = input_after_typing;
    m_type_stream << str;
    m_type_delay_ticks = m_type_delay_ticks_std;
    m_state = STATE_TYPING;
}

void CTerminal::Repeat(const std::string &str, int count, bool input_after_typing)
{
    m_finished = false;
    m_input_after_typing = input_after_typing;
    m_type_stream << str;
    m_repeat_count = count;
    m_type_delay_ticks = m_type_delay_ticks_std;
    m_state = STATE_REPEATING;
}
    
void CTerminal::Delay(int state, int ticks)
{
    m_delay_count = ticks -1;
    m_delayed_state = state;
    m_state = STATE_DELAY;
}

void CTerminal::Stop(void)
{
    m_state = STATE_IDLE;
    m_input_after_typing = false;
    m_type_stream.rdbuf()->str("");
    if(m_strings[m_akt_string]->size() > 0)
    {
        NewLine();
    }
}

void CTerminal::Update(void)
{
    switch(m_state)
    {
        case STATE_IDLE:
            // Nothing to do
            break;
        case STATE_DELAY:
            if(m_delay_count > 0) --m_delay_count;
            if(m_delay_count == 0)
            {
                m_state = m_delayed_state;
            }
            break;
        case STATE_TYPING:
        case STATE_REPEATING:
            UpdateTyping();
            break;
        case STATE_INPUT:

            if(m_cursor_count)
            {
                m_cursor_count--;
            }
            else
            {
                if(m_cursor_on)
                {
                    m_cursor_count = m_cursor_off_ticks;
                    m_cursor_on = false;
                }
                else
                {
                    m_cursor_count = m_cursor_on_ticks;
                    m_cursor_on = true;
                }
            }

            UpdateInput();
            break;
    }
}

void CTerminal::UpdateTyping(void)
{
    bool repeat;
    // Init delay for next char
    int char_delay_ticks = m_type_delay_ticks;
    do
    {
        repeat = false;
        
        ios::int_type c;
        c = m_type_stream.get();
        if(c == ios::traits_type::eof())
        {
            if(m_state == STATE_REPEATING)
            {
                if(m_repeat_count == 0)
                {
                    // Clear the stream
                    m_type_stream.rdbuf()->str("");
                    m_type_stream.clear();
                    if(m_input_after_typing)
                    {
                        Input();
                    }
                    else
                    {
                        m_finished = true;
                        m_state = STATE_IDLE;
                    }
                }
                else
                {
                    if(m_repeat_count > 0) m_repeat_count--;
                    // Start string again
                    m_type_stream.clear();
                    m_type_stream.seekg(0);
                    repeat = true;
                }
            }
            else
            {
                // last char was extracted -> typing is finished
                // Clear the stream
                m_type_stream.rdbuf()->str("");
                m_type_stream.clear();
                if(m_input_after_typing)
                {
                    Input();
                }
                else
                {
                    m_finished = true;
                    m_state = STATE_IDLE;
                }
            }
        }
        else
        {
            switch(c)
            {
                case '\n':
                    NewLine();
                    c = 0;      // don't add this to output, but consider char delay
                    break;
                case '\r':
                    // Ignore it, but repeat with next character, no delay for this
                    repeat = true;
                    break;
                case '<':
                    // Its a control tag
                    c = m_type_stream.get();
                    if(c == '<')
                    {
                        // same char again -> type it as is
                        break;
                    }
                    else
                    {
                        // its a control tag
                        // select the function...
                        switch(c)
                        {
                            case 'd':
                                {
                                    // Read delay for next char
                                    m_type_stream >> char_delay_ticks;
                                    // don't repeat, consider delay for next char
                                }
                                break;
                            case 's':
                                {
                                    // Read new speed (char delay) for typing
                                    int speed;
                                    m_type_stream >> speed;
                                    if(speed)
                                    {
                                        m_type_delay_ticks = speed;
                                        char_delay_ticks = speed;
                                    }
                                    else
                                    {
                                        // Speed is 0 -> reset to standard
                                        m_type_delay_ticks = m_type_delay_ticks_std;
                                        char_delay_ticks = m_type_delay_ticks;
                                    }
                                    // continue with next char without delay
                                    repeat = true;
                                }
                                break;
                        }
                        // Overread everything until the closing tag
                        m_type_stream.ignore(numeric_limits<int>::max(), '>');
                        // Don't type the control character
                        c = 0;
                    }
                    break;
            }
            if(!repeat)
            {
                if(c)
                {
                    // Add character to actual string
                    (*(m_strings[m_akt_string])) += c;
                    if(m_wordwrap)
                    {
                        // If char is a space then check if the following word fits into the line.
                        // Otherwise start a new line.
                        if(c == ' ')
                        {
                            int l = NextWordLength();
                            if(m_strings[m_akt_string]->size() + l > m_str_max_length)
                            {
                                NewLine();
                            }
                        }
                    }
                    else
                    {
                        if(m_strings[m_akt_string]->size() + 1 > m_str_max_length)
                        {
                            NewLine();
                        }
                    }
                }
                // Init delay for next character
                Delay(m_state, char_delay_ticks);
            }
        }
    } while (repeat);
}


int CTerminal::NextWordLength(void)
{
    int p = m_type_stream.tellg();
    ios::iostate state = m_type_stream.rdstate();
    int l = 0;
    bool again = true;
    while(again)
    {
        ios::int_type c = m_type_stream.get();
        if(c == ios::traits_type::eof()) break;
        switch(c)
        {
            case ' ':
            case '\n':
            case '\r':
                again = false;
                break;
            case '<':
                c = m_type_stream.get();
                if(c == ios::traits_type::eof()) 
                {
                    again = false;
                    break;
                }
                else if(c == '<')
                {
                    l++;
                }
                else
                {
                    m_type_stream.ignore(numeric_limits<int>::max(), '>');
                }
                break;
            default:
                l++;
        }
    }
    m_type_stream.clear(state);
    m_type_stream.seekg(p);
    return l;
}

void CTerminal::UpdateInput(void)
{
    if(keypressed())
    {
        int c  = readkey();
        int sc = (c & 0xFF00) >> 8;
        c &= 0x00FF;
        switch(sc)
        {
            case KEY_ENTER:
                {
                    m_finished = true;
                    m_state = STATE_IDLE;
                }
                break;
            case KEY_BACKSPACE:
                {
                    int i = m_pinput_str->length();
                    if(i > m_prompt_str.length())
                    {
                        *m_pinput_str = m_pinput_str->substr(0, i-1);
                    }
                }
                break;
            default:
                if(isprint(c))
                {
                    if(m_pinput_str->length() < m_str_max_length -1)    // -1 for cursor
                    {
                        *m_pinput_str += c;
                    }
                }
        }
    }
}

std::string CTerminal::InputString(void)
{
    return m_pinput_str->substr(m_prompt_str.length());
}

void CTerminal::InputString(const std::string &str)
{
    *m_pinput_str = m_prompt_str + str; 
};
        


void CTerminal::Draw(BITMAP* dest)
{
    int y = m_y + m_first_string_y;
    int cursor_y = 0;
    int x = m_x + m_border_dist;
    for(int i = 0; i < m_string_count; ++i)
    {
        textout_ex(dest, font, m_strings[i]->c_str(), x, y, makecol(0,170,0), -1);
        if(m_strings[i] == m_pinput_str) cursor_y = y; 
        y += text_height(font);
    }
    if(m_state == STATE_INPUT && m_cursor_on)
    {
        // Draw cursor
        int x1 = x + text_length(font, m_pinput_str->c_str());
        rectfill(dest, x1, cursor_y, x1 + text_length(font, " "), cursor_y + text_height(font), makecol(0,170,0));
    }
    rect(dest, m_x, m_y, m_x + m_w -1, m_y + m_h -1, makecol(0, 128, 0));
}
