#include "CCounter.h"
#include "CApplication.h"
#include "Datafile_objects.h"
#include "Logfile.hpp"

#include <sstream> 
#include "fblend.h"
#include "math.h"



#define RED_RAISEDAWN_STEP  2.0f
#define RED_AMPLITUDE       24.0f
#define RED_MIDPOINT        128.0f
#define RED_WAVE_STEP       (M_PI / 64.0f)

#define BAD_SOUND_VOL_SPAWN     50.0f
#define BAD_SOUND_START_SPEED   500.0f
#define BAD_SOUND_END_SPEED     1000.0f
//#define BAD_SOUND_UP_STEP       0.1f
#define BAD_SOUND_UP_STEP       0.35f
#define BAD_SOUND_DOWN_STEP     1.5f


#define SND_DID_DELAY       20  /*approx. 333ms*/

//static 
const int CCounter::numbers_datafile_objects[5/*Digit, 0 = lowest*/][19/*Number 0 to 9 and glyph 1 to 9*/] =
{
    {
        DAT_SEC_1_0, 
        DAT_SEC_1_1, 
        DAT_SEC_1_2, 
        DAT_SEC_1_3, 
        DAT_SEC_1_4, 
        DAT_SEC_1_5, 
        DAT_SEC_1_6, 
        DAT_SEC_1_7, 
        DAT_SEC_1_8, 
        DAT_SEC_1_9,
        DAT_SEC_1_H1, 
        DAT_SEC_1_H2, 
        DAT_SEC_1_H3, 
        DAT_SEC_1_H4, 
        DAT_SEC_1_H5, 
        DAT_SEC_1_H6, 
        DAT_SEC_1_H7, 
        DAT_SEC_1_H8, 
        DAT_SEC_1_H9
    },
    {
        DAT_SEC_10_0, 
        DAT_SEC_10_1, 
        DAT_SEC_10_2, 
        DAT_SEC_10_3, 
        DAT_SEC_10_4, 
        DAT_SEC_10_5, 
        DAT_SEC_10_6, 
        DAT_SEC_10_7, 
        DAT_SEC_10_8, 
        DAT_SEC_10_9,
        DAT_SEC_10_H1, 
        DAT_SEC_10_H2, 
        DAT_SEC_10_H3, 
        DAT_SEC_10_H4, 
        DAT_SEC_10_H5, 
        DAT_SEC_10_H6, 
        DAT_SEC_10_H7, 
        DAT_SEC_10_H8, 
        DAT_SEC_10_H9
    },
    {
        DAT_MIN_1_0, 
        DAT_MIN_1_1, 
        DAT_MIN_1_2, 
        DAT_MIN_1_3, 
        DAT_MIN_1_4, 
        DAT_MIN_1_5, 
        DAT_MIN_1_6, 
        DAT_MIN_1_7, 
        DAT_MIN_1_8, 
        DAT_MIN_1_9,
        DAT_MIN_1_H1, 
        DAT_MIN_1_H2, 
        DAT_MIN_1_H3, 
        DAT_MIN_1_H4, 
        DAT_MIN_1_H5, 
        DAT_MIN_1_H6, 
        DAT_MIN_1_H7, 
        DAT_MIN_1_H8, 
        DAT_MIN_1_H9
    },
    {
        DAT_MIN_10_0, 
        DAT_MIN_10_1, 
        DAT_MIN_10_2, 
        DAT_MIN_10_3, 
        DAT_MIN_10_4, 
        DAT_MIN_10_5, 
        DAT_MIN_10_6, 
        DAT_MIN_10_7, 
        DAT_MIN_10_8, 
        DAT_MIN_10_9,
        DAT_MIN_10_H1, 
        DAT_MIN_10_H2, 
        DAT_MIN_10_H3, 
        DAT_MIN_10_H4, 
        DAT_MIN_10_H5, 
        DAT_MIN_10_H6, 
        DAT_MIN_10_H7, 
        DAT_MIN_10_H8, 
        DAT_MIN_10_H9
    },
    {
        DAT_MIN_100_0, 
        DAT_MIN_100_1, 
        DAT_MIN_100_2, 
        DAT_MIN_100_3, 
        DAT_MIN_100_4, 
        DAT_MIN_100_5, 
        DAT_MIN_100_6, 
        DAT_MIN_100_7, 
        DAT_MIN_100_8, 
        DAT_MIN_100_9,
        DAT_MIN_100_H1, 
        DAT_MIN_100_H2, 
        DAT_MIN_100_H3, 
        DAT_MIN_100_H4, 
        DAT_MIN_100_H5, 
        DAT_MIN_100_H6, 
        DAT_MIN_100_H7, 
        DAT_MIN_100_H8, 
        DAT_MIN_100_H9
    },
};

//static 
const int CCounter::numbers_glyph_overrolls[5] =
{
    // Number of overrolls when rolling to glyphs for each digit
    2,
    4,
    3,
    1,
    5
};

//static 
const int CCounter::m_glyph_roll_delay[5] =
{
    // Number of frames before a roll to a glyph starts
    40,
    0,
    60,
    90,
    20
};

//static 
const int CCounter::counter_crushed[COUNTER_CRUSHED_PICS] =
{
    DAT_CRUSH_01,
    DAT_CRUSH_02,
    DAT_CRUSH_03,
    DAT_CRUSH_04,
    DAT_CRUSH_05,
    DAT_CRUSH_06,
    DAT_CRUSH_07,
    DAT_CRUSH_08,
    DAT_CRUSH_09,
    DAT_CRUSH_10,
    DAT_CRUSH_11,
    DAT_CRUSH_12,
    DAT_CRUSH_13,
    DAT_CRUSH_14,
    DAT_CRUSH_15,
    DAT_CRUSH_16,
    DAT_CRUSH_17,
    DAT_CRUSH_18,
    DAT_CRUSH_19,
    DAT_CRUSH_20,
    DAT_CRUSH_21,
    DAT_CRUSH_22,
    DAT_CRUSH_23,
    DAT_CRUSH_24
};


CCounter::CCounter()
{
    m_state = STOPPED;
    m_x = 0;
    m_y = 0;
    m_value = 0;
    m_to_value = 0;
    m_chassis = NULL;
    for (int i = 0 ; i < COUNTER_CRUSHED_PICS ; i++) m_pic_crushed[i] = NULL;
    m_crushed_pic = 0;
    m_rewind_requested = false;
    m_snd_tick = NULL;
    m_snd_did = NULL;
    m_snd_alert = NULL;
    m_snd_rewind = NULL;
    m_snd_rewind_end = NULL;
    m_snd_bad_sound = NULL;
    m_bad_sound_speed = BAD_SOUND_START_SPEED;
    m_badsound_state = BADSOUND_OFF;
    SetNumberPositions(0, 0);
    m_updates_per_second = 60;
    m_update_counter = 0;
    m_snd_did_delay = 0;
    m_expired = false;
}

CCounter::~CCounter()
{
}
        

void CCounter::BadsoundRaise(void)
{
    if(m_badsound_state == BADSOUND_OFF)
    {
        m_bad_sound_speed = BAD_SOUND_START_SPEED;
        m_badsound_state = BADSOUND_RISING;
        play_sample(m_snd_bad_sound, 0, 127, (int) m_bad_sound_speed, 1);
    }
}

void CCounter::BadsoundDawn(void)
{
    if(m_badsound_state != BADSOUND_OFF)
    {
        m_badsound_state = BADSOUND_DAWNING;
    }
}

void CCounter::BadsoundUpdate(void)
{
    switch(m_badsound_state)
    {
        case BADSOUND_RISING:
            m_bad_sound_speed += BAD_SOUND_UP_STEP;
            if(m_bad_sound_speed >= BAD_SOUND_END_SPEED)
            {
                m_bad_sound_speed = BAD_SOUND_END_SPEED;
                m_badsound_state = BADSOUND_FULLSPEED;
                Crush();
            }
            break;
        case BADSOUND_OFF:
        case BADSOUND_FULLSPEED:
            return;
            break;
        case BADSOUND_DAWNING:
            m_bad_sound_speed -= BAD_SOUND_DOWN_STEP;
            if(m_bad_sound_speed <= BAD_SOUND_START_SPEED)
            {
                m_bad_sound_speed = BAD_SOUND_START_SPEED;
                m_badsound_state = BADSOUND_OFF;
                stop_sample(m_snd_bad_sound);
                return;
            }
            break;
    }
    int vol = (int)(255.0f * (m_bad_sound_speed - BAD_SOUND_START_SPEED) / BAD_SOUND_VOL_SPAWN);
    if(vol > 255) vol = 255;
    adjust_sample(m_snd_bad_sound, vol, 127, (int) m_bad_sound_speed, 1);
}

void CCounter::Crush(void)
{
    if(m_state == STOPPED)
    {
        m_crushed_pic = 0;
        m_update_counter = 0;
        m_state = CRUSHING;
        m_finished = false;
    }
}

void CCounter::DeCrush(void)
{
    if(m_state == CRUSHING || m_state == CRUSHED)
    {
        m_crushed_pic = COUNTER_CRUSHED_PICS-1;
        m_update_counter = 0;
        m_state = DECRUSHING;
        m_finished = false;
    }
}

void CCounter::InitRes(DATAFILE * dat)
{
    for (int n = 0; n < 5; n++)
    {
        for (int m = 0; m < 19; m++)
        {
            if(dat == NULL)
            {
                m_number[n].SetBitmap(m, NULL);
            }
            else
            {
                if(m_number[n].SetBitmap(m, (BITMAP *)dat[numbers_datafile_objects[n][m]].dat ) == NULL)
                {
                    // Object not set!
                    std::stringstream s;
                    s << "Datafile Error! Bitmap for number " << m << " in digit " << n << " is NULL.";
                    throw(s.str());
                }
            }
        }
        m_number[n].GlyphRollDelay(m_glyph_roll_delay[n]);
    }
    if(dat == NULL)
    {
        m_chassis = NULL;
        m_snd_did = NULL;
        m_snd_tick = NULL;
        m_snd_alert = NULL;
        m_snd_rewind = NULL;
        m_snd_rewind_end = NULL;
        m_snd_bad_sound = NULL;
        for (int i = 0 ; i < COUNTER_CRUSHED_PICS ; i++) m_pic_crushed[i] = NULL;
    }
    else
    {
        if(( m_chassis = (BITMAP *)dat[DAT_COUNTDOWN].dat) == NULL)
        {
            // Object not set!
            throw(std::string("Datafile Error! Bitmap for chassis is NULL."));
        }
        for (int i = 0 ; i < COUNTER_CRUSHED_PICS ; i++) 
        {
            m_pic_crushed[i] = (BITMAP *)dat[counter_crushed[i]].dat;
            if(m_pic_crushed[i] == NULL)
            {
                // Object not set!
                std::stringstream s;
                s << "Datafile Error! Bitmap for crushed[" << i << "] is NULL.";
                throw(s.str());
            }
        }
        if(( m_snd_did = (SAMPLE *)dat[DAT_SND_DID].dat) == NULL)
        {
            // Object not set!
            throw(std::string("Datafile Error! 'Did' sample for chassis is NULL."));
        }
        if(( m_snd_tick = (SAMPLE *)dat[DAT_SND_TICK].dat) == NULL)
        {
            // Object not set!
            throw(std::string("Datafile Error! Tick sample for chassis is NULL."));
        }
        if(( m_snd_alert = (SAMPLE *)dat[DAT_SND_ALERT].dat) == NULL)
        {
            // Object not set!
            throw(std::string("Datafile Error! Alert sample for chassis is NULL."));
        }
        if(( m_snd_rewind = (SAMPLE *)dat[DAT_SND_REWIND].dat) == NULL)
        {
            // Object not set!
            throw(std::string("Datafile Error! Rewind sample for chassis is NULL."));
        }
        if(( m_snd_rewind_end = (SAMPLE *)dat[DAT_SND_REWIND_END].dat) == NULL)
        {
            // Object not set!
            throw(std::string("Datafile Error! Rewind-end sample for chassis is NULL."));
        }
        if(( m_snd_bad_sound = (SAMPLE *)dat[DAT_SND_BAD_SOUND].dat) == NULL)
        {
            // Object not set!
            throw(std::string("Datafile Error! Bad-Sound sample for chassis is NULL."));
        }
    }
}


int CCounter::GetH(void)
{
    if(m_chassis) return m_chassis->h; else return 0;
}

int CCounter::GetW(void)
{
    if(m_chassis) return m_chassis->w; else return 0;
}


void CCounter::SetPos(int x, int y)
{
    m_x = x;
    m_y = y;
    SetNumberPositions(x, y);
}


// Sec.   1er:  251     31
// Sec.  10er:  196     31
// Min.   1er:  126     31
// Min.  10er:   70     31
// Min. 100er:   13     31
void CCounter::SetNumberPositions(int x, int y)
{
    m_number[0].SetPos(x + 251, y + 31);
    m_number[1].SetPos(x + 196, y + 31);
    m_number[1].Modulo(6);
    m_number[2].SetPos(x + 126, y + 31);
    m_number[3].SetPos(x + 70,  y + 31);
    m_number[4].SetPos(x + 13,  y + 31);
}

void CCounter::Value(int value, bool with_flip) 
{ 
    m_value = value % 100000; 
    ValueToNumbers(value, with_flip);
    m_expired = false;
    BadsoundDawn();
    DeCrush();
}

void CCounter::SplitValueToDigits(int value, int digits[5])
{
    for (int i = 0; i < 5 ; i++)
    {
        digits[i] = value % 10;
        value /= 10;
    }
}

bool CCounter::ValueToNumbers(int value, bool with_flip)
{
    int digits[5];
    bool changed = false;
    SplitValueToDigits(value, digits);
    for (int i = 0; i < 5 ; i++)
    {
        changed |= m_number[i].Value() != digits[i];
        if(with_flip)
        {
            m_number[i].NextValue(digits[i]);
        }
        else
        {
            m_number[i].Value(digits[i]);
        }
    }
    return changed;
}

void CCounter::GlyphsToNumbers(int glyphs)
{
    int digits[5];
    SplitValueToDigits(glyphs, digits);
    for (int i = 0; i < 5 ; i++)
    {
        m_number[i].RollToGlyph(digits[i], numbers_glyph_overrolls[i]);
    }
}

void CCounter::Start(void)
{
    m_state = COUNTING;
    m_update_counter = 0;
    m_snd_did_delay = 0;
    m_expired = false;
}

void CCounter::Stop(void)
{
    m_state = STOPPED;
    m_snd_did_delay = 0;
    stop_sample(m_snd_rewind);
}

void CCounter::RollToGlyphs(int glyphs)
{
    m_state = ROLLING;
    m_value = glyphs;
    GlyphsToNumbers(glyphs);
    play_sample(m_snd_rewind, 255, 127, 1000, 1);
}


void CCounter::Rewind(int to_value)
{
    m_to_value = to_value;
    switch(m_state)
    {
        case CRUSHING:
        case CRUSHED:
        case DECRUSHING:
            m_finished = false;
            m_rewind_requested = true;
            DeCrush();
            break;
        default:
            if(m_value != to_value)
            {
                m_finished = false;
                m_state = ROLLING;
                m_value = to_value;
                ValueToNumbers(to_value);
                play_sample(m_snd_rewind, 255, 127, 1000, 1);
            }
            else
            {
                m_state = STOPPED;
                m_finished = true;
            }
    }
    m_expired = false;
    m_snd_did_delay = 0;
    BadsoundDawn();
}
        
void CCounter::Update(void)
{
    switch(m_state)
    {
        case STOPPED:
            break;
        case COUNTING:
            m_update_counter++;
            if(m_update_counter >= m_updates_per_second)
            {
                m_update_counter = 0;
                Tick();
            }
            break;
        case ROLLING:
            {
                bool finished = true;
                for(int i = 0; i < 5; i++)
                {
                    finished &= m_number[i].IsFinished();
                }
                m_finished = finished;
                if(finished)
                {
                    stop_sample(m_snd_rewind);
                    play_sample(m_snd_rewind_end, 255, 127, 1000, 0);
                    m_state = STOPPED;
                    if(m_expired)
                    {
                        BadsoundRaise();
                    }
                }
            }
            break;
        case CRUSHING:
            m_update_counter++;
            if(m_update_counter >= COUNTER_CRUSHED_FRAMES_PER_PIC)
            {
                m_update_counter = 0;
                if(m_crushed_pic >= COUNTER_CRUSHED_PICS-1)
                {
                    m_crushed_pic = COUNTER_CRUSHED_PICS-1;
                    m_state = CRUSHED;
                    m_finished = true;
                }
                else
                {
                    m_crushed_pic++;
                }
            }
            
            break;
        case CRUSHED:
            break;
        case DECRUSHING:
            m_update_counter++;
            if(m_update_counter >= COUNTER_CRUSHED_FRAMES_PER_PIC)
            {
                m_update_counter = 0;
                if(m_crushed_pic <= 0)
                {
                    m_crushed_pic = 0;
                    m_state = STOPPED;
                    if(m_rewind_requested)
                    {
                        Rewind(m_to_value);
                        m_rewind_requested = false;
                    }
                    else
                    {
                        m_finished = true;
                    }
                }
                else
                {
                    m_crushed_pic--;
                }
            }
            break;
    }
    BadsoundUpdate();
    for (int i = 0; i < 5 ; i++)
    {
        m_number[i].Update();
    }
    if(m_snd_did_delay)
    {
        m_snd_did_delay--;
        if(m_snd_did_delay == 0)
        {
            play_sample(m_snd_did, 255, 127, 1000, 0);
        }
    }
}

void CCounter::Tick(void)
{
    if(m_value == 0) 
    {
        Stop();
        m_expired = true;
        return;
    }
    m_value--;
    if((m_value % 100) > 60) m_value -= 40;
    int display_value = m_value;
    if(!CApplication::TickSecondsEnabled() && display_value > 400)
    {
        display_value = m_value / 100;
        if(m_value % 100) display_value++;
        display_value *= 100;
    }
    if(ValueToNumbers(display_value)) play_sample(m_snd_tick, 255, 127, 1000, 0);
    if(m_value <= 400)
    {
        if(m_value >= 100)
        {
            //if(m_value % 2) play_sample(m_snd_did, 255, 127, 1000, 0);
            if(m_value % 2 == 0) m_snd_did_delay = SND_DID_DELAY;
        }
        else
        {
            if((m_value <= 10) || (m_value % 2))
            {
                play_sample(m_snd_alert, 255, 127, 1000, 0);
            }
        }
    }
}

void CCounter::Draw(BITMAP* dest)
{
    switch(m_state)
    {
        case CRUSHING:
        case CRUSHED:
        case DECRUSHING:
            blit(m_pic_crushed[m_crushed_pic], dest, 0, 0, m_x + (rand()%3)-1, m_y + (rand()%3)-1, m_pic_crushed[m_crushed_pic]->w, m_pic_crushed[m_crushed_pic]->h);
            break;
        default:
            // Draw the counter chassis
            blit(m_chassis, dest, 0, 0, m_x, m_y, m_chassis->w, m_chassis->h);
            // Draw the numbers into the chassis
            for (int i = 0; i < 5 ; i++)
            {
                m_number[i].Draw(dest);
            }
    }
}
        
