#include "CMainScreen.h"
#include "Datafile_objects.h"
#include "CMyException.hpp"
#include "CApplication.h"


#include "Logfile.hpp"

#include "allegro.h"
#include "fblend.h"
#include <string>
#include <sstream>

using namespace std;


#define COUNTER_START_VALUE     10800

void CMainScreen::InitInstance(void)
{
    m_snd_end = NULL;
    m_lost_counter = NULL;
    m_terminal = NULL;
    m_keylock = NULL;
    m_state = STATE_STARTUP;
    try
    {
        m_lost_counter = new CCounter;
    }
    catch(...){}
    MYASSERTVALIDMSG(m_lost_counter, "MainScreen failed creating CCouter object.");
    try
    {
        m_terminal = new CTerminal;
    }
    catch(...){}
    MYASSERTVALIDMSG(m_terminal, "MainScreen failed creating CTerminal object.");
    try
    {
        m_keylock = new CKeylock;
    }
    catch(...){}
    MYASSERTVALIDMSG(m_keylock, "MainScreen failed creating CKeylock object.");
}

CMainScreen::CMainScreen()
{
    InitInstance();
}

CMainScreen::~CMainScreen()
{
    if (m_lost_counter) delete m_lost_counter;
    m_lost_counter = NULL;
    if (m_terminal) delete m_terminal;
    m_terminal = NULL;
    if (m_keylock) delete m_keylock;
    m_keylock = NULL;
}
        
void CMainScreen::InitRes(DATAFILE * dat)
{
    if(dat == NULL)
    {
        m_snd_end = NULL;
    }
    else
    {
        if(( m_snd_end = (SAMPLE *)dat[DAT_SND_END].dat) == NULL)
        {
            // Object not set!
            throw(std::string("Datafile Error! 'End' sample for Mainscreen is NULL."));
        }
    }

    m_lost_counter->InitRes(dat);
    m_terminal->InitRes(dat);
    m_keylock->InitRes(dat);
    
    m_terminal->Pos(0, m_lost_counter->GetH());
    m_terminal->Size(screen->w, screen->h - m_lost_counter->GetH());
    
    m_keylock->SetPositions(screen->w, screen->w - m_keylock->W(), screen->h - m_keylock->H());
}

bool CMainScreen::IsCode(const string & s)
{
    stringstream ss;
    ss << s;
    int i;
    ss >> i;
    if(i != 4)  return false;
    ss >> i;
    if(i != 8)  return false;
    ss >> i;
    if(i != 15)  return false;
    ss >> i;
    if(i != 16)  return false;
    ss >> i;
    if(i != 23)  return false;
    ss >> i;
    if(i != 42)  return false;
    ios::int_type c;
    do
    {
        c = ss.get();
        if(c == ios::traits_type::eof()) break;
        if(c != ' ') return false;
    }while (1);
    
    return true;
}

void CMainScreen::Delay(e_state state, int ticks)
{
    m_delay_count = ticks -1;
    m_delayed_state = state;
    m_state = STATE_DELAY;
}

void CMainScreen::Reset(void)
{
    m_state = STATE_STARTUP;
    m_finished = false;
}

void CMainScreen::Update(void)
{
    m_lost_counter->Update();
    m_terminal->Update();
    m_keylock->Update();
    switch(m_state)
    {
        case STATE_DELAY:
            if(m_delay_count > 0) --m_delay_count;
            if(m_delay_count == 0)
            {
                m_state = m_delayed_state;
            }
            break;
        case STATE_STARTUP:
            m_lost_counter->Value(COUNTER_START_VALUE, false);
            m_lost_counter->Start();
            m_state = STATE_LOOP;
            m_terminal->Input();
            // Fall through!!!
        case STATE_LOOP:
            if(m_terminal->IsFinished())
            {
                m_terminal->NewLine();
                ProcessTerminalInput_Loop();
            }
            if(m_lost_counter->IsExpired())
            {
                m_state = STATE_COUNTER_EXPIRED;
                m_lost_counter->RollToGlyphs(1234);
            }
            break;        
        case STATE_REWIND:
            m_terminal->Stop();
            m_keylock->MoveOut();
            m_lost_counter->Rewind(COUNTER_START_VALUE);
            m_state = STATE_REWINDING_;
            // fall through
        case STATE_REWINDING_:
            if(!m_lost_counter->IsRewinding())
            {
                m_terminal->Clear();
                m_terminal->WordWrap(true);
                Delay(STATE_RESTART_COUNTER, 60);
            }
            break;
        case STATE_RESTART_COUNTER:
            m_lost_counter->Start();
            m_terminal->Input();
            m_state = STATE_LOOP;
            break;
        case STATE_EXIT_MESSAGE:
            if(m_terminal->IsFinished())
            {
                Delay(STATE_FINISHED, 40);
                if(m_lost_counter->IsCounting()) m_lost_counter->Stop();
                play_sample(m_snd_end, 255, 127, 1000, 0);
            }
            break;
        case STATE_COUNTER_EXPIRED:
            if(m_terminal->IsFinished())        // input from the terminal
            {
                m_terminal->NewLine();
                ProcessTerminalInput_Loop();
            }
            if(m_lost_counter->IsFinished())    // rolling to glyphs finished?
            {
                m_terminal->Stop();             // stop terminal input or output
                m_terminal->Clear();            // and clear the contents
                m_terminal->WordWrap(false);    // switch wordwrap off
                m_terminal->Repeat("<s5>System Failure ", 25, true);
                m_state = STATE_COUNTER_EXPIRED_LOOP;
            }
            break;
        case STATE_COUNTER_EXPIRED_LOOP:
            if(m_terminal->IsFinished())
            {
                m_terminal->NewLine();
                ProcessTerminalInput_Expired();
            }
            if(m_lost_counter->IsCrushed() && m_keylock->IsOut())
            {
                m_keylock->MoveIn();            // move the keylock in
            }
            if(m_keylock->IsKeyTurned())
            {
                //m_terminal->WordWrap(true);
                m_state = STATE_REWIND;
            }
            break;
        case STATE_TYPING_NEW_CHANCE:
            if(m_terminal->IsFinished())
            {
                m_state = STATE_REWIND;
            }
            break;
        case STATE_FINISHED:
            m_finished = true;
            break;        
    }
}


void CMainScreen::ProcessSetCmd(stringstream &input)
{
    stringstream response;
    string what;
    int ivalue;
    input >> what;
    if(CApplication::CheatsEnabled() && what == "counter")
    {
        input >> ivalue;
        if(!input.fail())
        {
            m_lost_counter->Value(ivalue, false);
        }
    }
    if(CApplication::CheatsEnabled() && what == "glyphs")
    {
        input >> ivalue;
        if(!input.fail())
        {
            m_lost_counter->RollToGlyphs(ivalue);
        }
    }
    if(what == "tick_seconds")
    {
        if(input.eof())
        {
            response << (CApplication::TickSecondsEnabled() ? 1 : 0);
        }
        else 
        {
            input >> ivalue;
            if(!input.fail())
            {
                CApplication::TickSecondsEnabled(ivalue != 0);
            }
        }
    }
    if(what == "mute")
    {
        if(input.eof())
        {
            response << (CApplication::SoundEnabled() ? 0 : 1);
        }
        else 
        {
            input >> ivalue;
            if(!input.fail())
            {
                CApplication::SoundEnabled(ivalue == 0);
            }
        }
    }
    if(what == "cheats")
    {
        if(input.eof())
        {
            response << (CApplication::CheatsEnabled() ? 1 : 0);
        }
        else 
        {
            input >> ivalue;
            if(!input.fail())
            {
                CApplication::CheatsEnabled(ivalue != 0);
            }
        }
    }
    if(what == "volume")
    {
        if(input.eof())
        {
            response << CApplication::DigiVolume() << ", " << CApplication::MidiVolume();
        }
        else 
        {
            input >> ivalue;
            if(!input.fail())
            {
                CApplication::DigiVolume(ivalue);
                CApplication::MidiVolume(ivalue);
            }
        }
    }
    if(input.fail())
    {
        response << "Parameter error.";
    }
    if(response.str().size())
    {
        m_terminal->Type(response.str(), true);
    }
    else
    {
        m_terminal->Input();
    }
}

void CMainScreen::ProcessTerminalInput_Loop(void)
{
    stringstream input;
    input << m_terminal->InputString();
    string s;
    input >> s;
    if(IsCode(input.str()))
    {
        if(m_lost_counter->Value() <= 400 || m_lost_counter->IsExpired())
        {
            m_state = STATE_REWIND;
        }
        else
        {
            m_terminal->Type("Sorry, too early...", true);
        }
    }
    else if(s == "set")
    {
        ProcessSetCmd(input);
    }
    else if(CApplication::CheatsEnabled() && s == "stop")
    {
        m_lost_counter->Stop();
        m_terminal->Input();
    }
    else if(CApplication::CheatsEnabled() && s == "start")
    {
        m_lost_counter->Start();
        m_terminal->Input();
    }
    else if(s == "exit" || s == "quit" || s == "escape")
    {
        m_terminal->Type("You want to abandon your duty?<d60>\n"
                         "Really<s30> ...<s0><d60> OK.<d30>\n"
                         "YOU are responsible for what will happen<s30> ...<s0><d30>\n");
        m_state = STATE_EXIT_MESSAGE;
    }
    else if(s == "clear" || s == "cls")
    {
        m_terminal->Clear();
        m_terminal->Input();
    }
    else if(s == "help" || s == "duty" || s == "task")
    {
        m_terminal->Type("Your task is to enter the code before the counter gets down to zero.<d60>\n"
                         "The code is<s30> <s2>4 <d30>8 <d30>15 <d30>16 <d30>23 <d30>42<s0><d60>\n"
                         "Do not use the computer for anything else than entering the code!!!<d60>\n", 
                         true);
    }
    else
    {
        switch(rand()%7)
        {
            case 1:
                m_terminal->Type("Hello?<d60>", true);
                break;
            case 2:
                m_terminal->Type("Who is this?<d60>", true);
                break;
            case 3:
                m_terminal->Type("You are compromising the integrity of this project!<d60>", true);
                break;
            case 4:
                m_terminal->Type("Did you forget the rules<s30>\?\?\?<s0><d60>", true);
                break;
            case 5:
                m_terminal->Type("What are you doing, <d30>recall your duty!<d60>", true);
                break;
            case 6:
                m_terminal->Type("No one knows what will happen when you continue doing what you are doing...<d60>", true);
                break;
            default:
                m_terminal->Type("Syntax Error.", true);
        }
    }
}

void CMainScreen::ProcessTerminalInput_Expired(void)
{
    stringstream input;
    input << m_terminal->InputString();
    string s;
    input >> s;
    if(IsCode(input.str()))
    {
        m_terminal->Type("You want another chance?\n<d45>OK. \n<d30>Do better this time...<d30>");
        m_state = STATE_TYPING_NEW_CHANCE;
    }
    else if(s == "set")
    {
        ProcessSetCmd(input);
    }
    else if(s == "exit" || s == "quit" || s == "escape")
    {
        m_lost_counter->FadeSoundOut();
        m_keylock->MoveOut();
        m_terminal->Type("You failed!<d60>\n"
                         "Escaping is the best for you...\n<d60>");
        m_state = STATE_EXIT_MESSAGE;
    }
    else if(s == "help" || s == "duty" || s == "task")
    {
        m_terminal->Type("You can exit.<d60>\n"
                         "You can try to enter the code.<d60>\n"
                         "Who knows what will happen...<d60>\n", 
                         true);
    }
    else
    {
        switch(rand()%6)
        {
            case 1:
                m_terminal->Type("You failed. This task was too hard for you!<d60>", true);
                break;
            case 2:
                m_terminal->Type("No one can help you now.<d60>", true);
                break;
            case 3:
                m_terminal->Type("Who knows what will happen...<d60>", true);
                break;
            case 4:
                m_terminal->Type("Who are the others<d30> and what are they doing?<d60>", true);
                break;
            case 5:
                m_terminal->Type("You are lost.<d30> No one will come and help you!<d60>", true);
                break;
            default:
                m_terminal->Type("What are you trying\?\?\?<d60>", true);
        }
    }
}

void CMainScreen::Draw(BITMAP* dest)
{
    if(m_lost_counter)
    {
        m_lost_counter->Draw(dest);
    }
    if(m_terminal)
    {
        m_terminal->Draw(dest);
    }
    if(m_keylock)
    {
        m_keylock->Draw(dest);
    }
}

