/// \file resource.h
/// Resource manager

#ifndef RESOURCE_H_INCLUDED
#define RESOURCE_H_INCLUDED

#include <string>
#include <map>
#include <fstream>
#include <allegro.h>

// File format: (hex)
// Off | Len | Desc
//   0 |   2 | 0xAFDF (magic number)
//   2 |   1 | 1 (file version)
//   3 |   2 | Number of entries
//   5 |   * | Contents table
//   * |   * | Data
// Contents table:
// Off | Len | Desc
//   0 |   * | Content name (NULL-terminated)
//   * |   4 | Offset to data

class ResourceManager
{
	public: // Types
		enum { MagicNumber = 0xAFDF };
		enum { FileVersion = 1 };

		class Resource
		{
			public: // Public functions
				virtual ~Resource() = 0;
				virtual void Save(std::iostream* f) = 0;
				virtual void Load(std::iostream* f) = 0;
		};
		
	public: // Public static functions
		static PACKFILE* CreatePackfile(std::iostream* file);
		
	protected: // Protected variables
		std::fstream* file;
		std::map<std::string, unsigned int> contents;
		std::map<std::string, Resource*> pairs;
		
	public: // Public functions
		ResourceManager();
		~ResourceManager();
		
		bool Load(const std::string& filename);
		void Save(const std::string& filename);
		void Unload();
		
		void Remove(const std::string& name, bool fromContents = false);
		void Add(const std::string& name, Resource* data);
		Resource* Retrieve(const std::string& name, Resource* (*Factory)());
		template <class T> T* Retrieve(const std::string& name, Resource* (*Factory)() = T::Factory)
		{
			return (T*) Retrieve(name, Factory);
		}
};

template<class t> class BasicResource: public ResourceManager::Resource
{
	public: // Public static functions
		static ResourceManager::Resource* Factory()
		{ return new BasicResource<t>; }
		
	public: // Public variables
		t data;
		
	public: // Public functions
		BasicResource()
		{ }
		BasicResource(t data): data(data)
		{ }
		
		void Save(std::iostream* f)
		{ f->write((char*) &data, sizeof(t)); }
		void Load(std::iostream* f)
		{ f->read((char*) &data, sizeof(t)); }
};

class StringResource: public ResourceManager::Resource
{
	public: // Public static function
		static Resource* Factory()
		{ return new StringResource; }
		
	public: // Public variables
		std::string data;
		
	public: // Public functions
		StringResource();
		StringResource(const std::string& data);
		
		void Save(std::iostream* f);
		void Load(std::iostream* f);
};

class BitmapResource: public ResourceManager::Resource
{
	public: // Public static function
		static Resource* Factory()
		{ return new BitmapResource; }
		
	public: // Public variables
		BITMAP* data;
		
	public: // Public functions
		BitmapResource();
		BitmapResource(BITMAP* data);
		~BitmapResource();
		
		void Save(std::iostream* f);
		void Load(std::iostream* f);
};

class FontResource: public BitmapResource
{
	public: // Public static function
		static Resource* Factory()
		{ return new FontResource; }
		
	public: // Public variables
		FONT* data;
		
	public: // Public functions
		FontResource();
		FontResource(BITMAP* data);
		~FontResource();
		
		void Load(std::iostream* f);
};

#endif

// The end
