/// \file resource_jpeg.cpp
/// Resource manager plugin for JPEGs

#include "jpgalleg/jpgalleg.h"
#include "resource_jpeg.h"

using namespace std;

#define ESTIMATED_MAX_JPG_SIZE(bmp)		(((bmp)->w * (bmp)->h * 3) + 1024)

static unsigned int igetl(istream* f)
{ int i = f->get(); i |= f->get() << 8; i |= f->get() << 16; i |= f->get() << 24; return i; }

static void iputl(unsigned int s, ostream* f)
{ f->put(s); f->put(s >> 8); f->put(s >> 16); f->put(s >> 24); }

JpegResource::JpegResource()
{
}

JpegResource::JpegResource(BITMAP* data): BitmapResource(data)
{
}

void JpegResource::Save(std::iostream* f)
{
	int size = ESTIMATED_MAX_JPG_SIZE(data);
	char* buffer = new char [size];
	save_memory_jpg(buffer, &size, data, NULL);
	iputl(size, f);
	f->write(buffer, size);
	delete buffer;
}

void JpegResource::Load(std::iostream* f)
{
	int size = igetl(f);
	char* buffer = new char [size];
	f->read(buffer, size);
	data = load_memory_jpg(buffer, size, NULL);
	delete buffer;
}

MaskedJpegResource::MaskedJpegResource()
{
}

MaskedJpegResource::MaskedJpegResource(BITMAP* data): JpegResource(data)
{
}

void MaskedJpegResource::Save(std::iostream* f)
{
	JpegResource::Save(f);
	char* buffer = new char [(data->w * data->h + 7) / 8];
	memset(buffer, 0, (data->w * data->h + 7) / 8);
	int byte = 0, bit = 0;
	for(int y = 0; y < data->h; y++)
	{
		for(int x = 0; x < data->w; x++)
		{
			if(getpixel(data, x, y) == bitmap_mask_color(data))
				buffer[byte] |= 1 << (7 - bit);
			bit++;
			if(bit > 7)
			{
				byte++;
				bit = 0;
			}
		}
	}
	f->write(buffer, (data->w * data->h + 7) / 8);
	delete buffer;
}

void MaskedJpegResource::Load(std::iostream* f)
{
	JpegResource::Load(f);
	char* buffer = new char [(data->w * data->h + 7) / 8];
	f->read(buffer, (data->w * data->h + 7) / 8);
	int byte = 0, bit = 0;
	for(int y = 0; y < data->h; y++)
	{
		for(int x = 0; x < data->w; x++)
		{
			if(buffer[byte] & (1 << (7 - bit)))
				putpixel(data, x, y, bitmap_mask_color(data));
			bit++;
			if(bit > 7)
			{
				byte++;
				bit = 0;
			}
		}
	}
	delete buffer;
}

// The end
