/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
#include <iostream>
#include <cmath>
#include <ctime>
#include <string>
#include <sstream>
#include "config.h"
#include "allegro.h"
#include "compile_settings.h"
#include "language.h"

#define ALLEGRO_LOGO_DEF PACKAGE_DATA_DIR"/"PACKAGE"/allegro_logo.bmp"
#define CONFIRM_SOUND PACKAGE_DATA_DIR"/"PACKAGE"/confirm.wav"
#define GAME_FONT PACKAGE_DATA_DIR"/"PACKAGE"/font_fairlight.pcx"
#define GAME_GRAPHICS PACKAGE_DATA_DIR"/"PACKAGE"/gfx.bmp"
#define JAMYSKIS_SOUND PACKAGE_DATA_DIR"/"PACKAGE"/jamyskis.wav"
#define JAMYSKIS_LOGO PACKAGE_DATA_DIR"/"PACKAGE"/jamyskis_logo.bmp"
#define LINUX_LOGO PACKAGE_DATA_DIR"/"PACKAGE"/linux.bmp"
#define NAUGHTS_LOGO PACKAGE_DATA_DIR"/"PACKAGE"/naughts_logo.bmp"
#define PLACED_SOUND PACKAGE_DATA_DIR"/"PACKAGE"/placed.wav"
#define TYPECHAR_SOUND PACKAGE_DATA_DIR"/"PACKAGE"/typechar.wav"


/* Variable initialization
The enum command defines the various codes to the array of symbols
grid = the game grid of 9x9 to allow for all possibilities
player = the current player number
turn = turn number
winner = the number of the winner (0 when nobody)
x = selected horizontal square
y = selected vertical square
entry_valid = flag to test if space entered is valid
boardsize = size of board between 3 and 9
boardsize_entry = the text entered to determine the size of the board
p1_name = Player 1's name
p2_name = Player 2's name
entry = the square entered by the player
symbol = the list of symbols available to select
letters = the list of nine characters available, plus the lower-case equivalents
validrow = counter to validate if a row has been won
validcolumn = counter to validate if a column has been won
validdiag = counter to validate if diagonal has been won
risk_assessment = table to tell computer which squares are dangerous
canplace = either 0 or 1 to tell if a piece can now be placed in alternate
version
removed = the most recent turn to have a piece removed - used to ensure that a
player is not forced to remove a piece twice*/

enum{EMPTY_SQUARE,NAUGHT,CROSS};

int grid[10][10], player, turn, winner, x, y, entry_valid, validrow;
int validcolumn, validdiag, xcheck, canplace, removed, lastx, lasty;
int playercount, game_type, score, time_taken;
int boardsize=3;
int enhanced=0; 
int difficulty=12;
int bgoffset;
string p1_name, p2_name, entry, boardsize_entry_string, playercount_entry_string;
string difficulty_entry_string, enhanced_entry_string;
char boardsize_entry[1], playercount_entry[1], difficulty_entry[1], keyselected;
char letters[10]="ABCDEFGHI";
char *board_alphabet = letters;
char symbol[4]=" OX";
double margin_of_error;

FONT *gamefont;
BITMAP *titlelogo;
SAMPLE *typesound,*confirmsound,*placesound,*jamyskissound;
BITMAP *jamyskis,*allegro,*linux_logo;
BITMAP *grid_centre,*grid_top,*grid_bottom,*grid_left,*grid_right;
BITMAP *cross,*naught,*spritemap;
BITMAP *title_logo,*title_letters[15];
BITMAP *doublebuffer;
PALETTE jamy_pal,alleg_pal,game_pal,title_pal,linux_pal;

using namespace std;

/*Display the board*/	

void display_board()
{		
	char letterspointer[2]={' ','\0'};
	
	for(int vertical=0; vertical<boardsize-1; vertical++)
	{
		draw_sprite(doublebuffer,grid_top,400-(boardsize*24)+(vertical*48),330-(boardsize*24));
		draw_sprite(doublebuffer,grid_bottom,400-(boardsize*24)+(vertical*48),378-(boardsize*24)+((boardsize-1)*48));
		textout_ex(doublebuffer,gamefont,letterspointer,386-(boardsize*24)+(vertical*48),300-(boardsize*24),2,-1);
	};
	
	for(int horizontal=0; horizontal<boardsize-1; horizontal++)
	{
		draw_sprite(doublebuffer,grid_left,352-(boardsize*24),378-(boardsize*24)+(horizontal*48));
		draw_sprite(doublebuffer,grid_right,400-(boardsize*24)+((boardsize-1)*48),378-(boardsize*24)+(horizontal*48));
	};
	
	for(int vertical=0; vertical<boardsize-1; vertical++)
	{
		for(int horizontal=0; horizontal<boardsize-1; horizontal++)
		{
			draw_sprite(doublebuffer,grid_centre,400-(boardsize*24)+(horizontal*48),378-(boardsize*24)+(vertical*48));
		};
	};
	
	for(int vertical=0; vertical<boardsize; vertical++)
	{
		letterspointer[0]=letters[vertical];
		textout_ex(doublebuffer,gamefont,letterspointer,386-(boardsize*24)+(vertical*48),305-(boardsize*24),6,-1);
		
		for(int horizontal=0; horizontal<boardsize; horizontal++)
		{
			letterspointer[0]=(char) horizontal+49;
			textout_ex(doublebuffer,gamefont,letterspointer,332-(boardsize*24),360-(boardsize*24)+(horizontal*48),6,-1);
			
			switch(grid[horizontal][vertical])
			{
				case NAUGHT: draw_sprite(doublebuffer,naught,376-(boardsize*24)+(horizontal*48),354-(boardsize*24)+(vertical*48)); break;
				case CROSS: draw_sprite(doublebuffer,cross,376-(boardsize*24)+(horizontal*48),354-(boardsize*24)+(vertical*48)); break;
			};
		};
	};
};


void display_player_notification()
{
	string playernotification;
	string scorenotification;

	stringstream scorebuffer;	
	
	switch (player)
	{
		case 1: playernotification=p1_name+_GO; break;
		case 2: playernotification=p2_name+_GO; break;
	};
	
	scorebuffer << "SCORE: " << score;
	
	scorenotification=scorebuffer.str();
	
	textout_ex(doublebuffer,gamefont,playernotification.c_str(),0,0,7,-1);
	
	if(enhanced==2)
	{
		textout_ex(doublebuffer,gamefont,scorenotification.c_str(),0,560,7,-1);
	}
};


void check_for_quit()
{
	if((key_shifts&KB_CTRL_FLAG)&&(key[KEY_C]))
	{
		cout << "Thank you for playing!\n";
		allegro_exit();
		destroy_bitmap(screen);
		abort();
	};		
};
END_OF_FUNCTION(check_for_quit);

string string_entry(int x,int y,int maxlength,int text_color)
{
	char keyboardarray[73]="ABCDEFGHIJKLMNOPQRSTUVWXYZ01234567890123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ";
	bool completed, key_released;
	int keycheck;
	string string_entry;
	
	keycheck=0;
	completed=false;
	
	rectfill(screen,x,y+2,x+20,y+28,text_color);
	
	do
	{
		clear_to_color(doublebuffer,2);
		
		for(int checkpatternx=-80; checkpatternx<800; checkpatternx=checkpatternx+160)
		{
			for(int checkpatterny=-80; checkpatterny<600; checkpatterny=checkpatterny+160)
			{
				rectfill(doublebuffer,checkpatternx+bgoffset,checkpatterny+bgoffset,checkpatternx+80+bgoffset,checkpatterny+80+bgoffset,1);
				rectfill(doublebuffer,checkpatternx+80+bgoffset,checkpatterny+80+bgoffset,checkpatternx+160+bgoffset,checkpatterny+160+bgoffset,1);
			}
		}
		
		bgoffset++;
		if(bgoffset>=80)
		{
			bgoffset=0;
		};

		display_player_notification();		
		display_board();
		
		textout_ex(doublebuffer,gamefont,string_entry.c_str(),x,y,text_color,-1);
		rectfill(doublebuffer,x+((unsigned int)string_entry.length()*20),y+2,x+((unsigned int)string_entry.length()*20)+20,y+28,text_color);
		
		if(canplace==0)
		{			
			textout_ex(doublebuffer,gamefont,_ENTERREMOVESPACE.c_str(),0,25,7,-1);
		}
		else
		{
			textout_ex(doublebuffer,gamefont,_ENTERSPACE.c_str(),0,25,7,-1);
		}
		
		blit(doublebuffer,screen,0,0,0,0,800,600);
		
		time_taken++;
		
		for(int keyrange=1;keyrange<46;keyrange++)
		{
			if (key[keyrange])
			{
				keycheck=keyrange;
			};
		};
		
		if(key[KEY_BACKSPACE])
		{
			keycheck=63;
		};
		
		if(key[KEY_ENTER])
		{
			keycheck=67;
		};

		if (keycheck>0)
		{	
			if(keycheck<47&&string_entry.length()<(unsigned) maxlength)
			{
				string_entry=string_entry+keyboardarray[keycheck-1];
			};
			
			if(keycheck==63&&string_entry.length()>0)
			{
				string_entry=string_entry.erase(string_entry.length()-1,string_entry.length());
			};
			
			if(keycheck==67)
			{
				completed=true;
			};
						
			play_sample(typesound,255,100,1000,0);
		};

		key_released=true;
		do
		{} while (key[keycheck]);
		
		keycheck=0;
	} while(completed==false);

	rectfill(screen,x,y,x+(maxlength*25)+25,y+28,0);
	return string_entry;
	
};

	

/*Check if there is a winner*/

void determine_winner()
{
	/*Horizontal checks*/

	for(int checkrow=0; checkrow<boardsize; checkrow++)
	{
		validrow=0;
		for(int checkcolumn=0; checkcolumn<boardsize; checkcolumn++)
		{
			if (grid[checkcolumn][checkrow]==grid[0][checkrow]&&grid[0][checkrow]!=EMPTY_SQUARE)
			{
				validrow++;
			};
		};
		
		if (validrow==boardsize)
		{
			winner=grid[0][checkrow];
			checkrow=boardsize;
		};
	};


	/*Vertical checks*/
	
	for(int checkcolumn=0; checkcolumn<boardsize; checkcolumn++)
	{
		validcolumn=0;
		for(int checkrow=0; checkrow<boardsize; checkrow++)
		{
			if (grid[checkcolumn][checkrow]==grid[checkcolumn][0]&&grid[checkcolumn][0]!=EMPTY_SQUARE)
			{
				validcolumn++;
			};
		};
		
		if (validcolumn==boardsize)
		{
			winner=grid[checkcolumn][0];
			checkcolumn=boardsize;
		};
	};


	/*Diagonal check top-left to bottom-right*/
	
	validdiag=0;
	for(int checkdiag=0; checkdiag<boardsize; checkdiag++)
	{
		if (grid[checkdiag][checkdiag]==grid[0][0])
		{
			validdiag++;
		};
		
		if (validdiag==boardsize)
		{
			winner=grid[0][0];
			checkdiag=boardsize;
		};
	};
	
	
	/*Diagonal check top-right to bottom-left*/
	
	validdiag=0;
	for(int checkdiag2=0; checkdiag2<boardsize; checkdiag2++)
	{
		
		if (grid[abs(boardsize-checkdiag2-1)][checkdiag2]==grid[boardsize-1][0])
		{
			validdiag++;
		};
		
		if (validdiag==boardsize)
		{
			winner=grid[boardsize-1][0];
			checkdiag2=boardsize;
		};
	};

	if (winner==0&&turn==(boardsize*boardsize)+1&&enhanced==0)
	{
		winner=3;
	};
	
};

bool check_if_key_pressed(int keynumber)
{
	bool keyrange[KEY_MAX];
	
	for (int keyt=0; keyt<KEY_MAX; keyt++) 
	{
		if (key[keyt]) 
		{
			keyrange[keyt]=true;
		}
		
		if (!key[keyt])
		{
			keyrange[keyt]=false;
		};
	};

	return keyrange[keynumber];
};


/*Enter string for difficulty level and convert into number*/

void check_difficulty()
{
	strcpy(difficulty_entry, difficulty_entry_string.c_str());
	int difficulty_valid=0;
	for(unsigned int testlength=0;testlength<strlen(difficulty_entry);testlength++)
	{
		if (isdigit(difficulty_entry[testlength]))
		{
			difficulty_valid++;
		};
	};
	
	if ((unsigned int) difficulty_valid==strlen(difficulty_entry))
	{
		difficulty=atoi(difficulty_entry);
	};
};


/*Set piece on board if able - if place is already taken, set entry_valid to 0*/

void set_piece_on_board()
{
	int linetest;
	
	if (grid[x][y]>0||x>boardsize-1||y>boardsize-1)
	{
		entry_valid=0;
	};
	
	if (entry_valid==2)
	{
		turn++;
		grid[x][y]=player;
		player++;
		if (player>2) {player=1;};
	}
	else
	{
		if(player==1)
		{
			textout_ex(screen,gamefont,_ENTRYINVALID.c_str(),0,80,7,0);
			rest(1000);
		}
	};
	
// Check vertically
				
	linetest=0;
				
	for(int checky=0; checky<boardsize; checky++)
	{		
		if(grid[x][checky]==CROSS)
		{
			linetest++;
		}
	}
				
	if(linetest==boardsize-1)
	{
		score=score+750;
	}
	
// Check horizontally
				
	linetest=0;
				
	for(int checkx=0; checkx<boardsize; checkx++)
	{	
		if(grid[checkx][y]==CROSS)
			{
				linetest++;
			}
	}
				
	if(linetest==boardsize-1)
	{
		score=score+750;
	}

// Check diagonally left to right
				
	if(x==y)
	{			
		linetest=0;
				
		for(int checkx=0; checkx<boardsize; checkx++)
		{			
			if(grid[checkx][checkx]==CROSS)
			{
				linetest++;
			}
		}
				
		if(linetest==boardsize-1)
		{
			score=score+750;
		}
	}
				
// Check diagonally right to left
				
	if(x==boardsize-y-1)
	{			
		linetest=0;
				
		for(int checkx=0; checkx<boardsize; checkx++)
		{				
			if(grid[checkx][boardsize-checkx-1]==CROSS)
			{
				linetest++;
			}
		}
				
		if(linetest==boardsize-1)
		{
			score=score+750;
		}
	}
};

void remove_piece_from_board()
{
	if (grid[x][y]!=player||x>boardsize-1||y>boardsize-1)
	{
		entry_valid=0;
	};
	
	if (entry_valid==2)
	{
		grid[x][y]=0;
		canplace=1;
		removed=turn;
	}
	else
	{
		if(player==1)
		{
			textout_ex(screen,gamefont,_ENTRYINVALID.c_str(),0,80,7,0);
			rest(1000);
		};
	};
	
	lastx=x;
	lasty=y;
}
	


/*Check validity of space entry - if true set entry_valid to 2*/

void check_entry_valid()
{
	entry_valid=0;

	if (entry.length()==2)
	{
		switch (entry[0])
		{
			case 'A': x=0; entry_valid++; break;
			case 'a': x=0; entry_valid++; break;
			case 'B': x=1; entry_valid++; break;
			case 'b': x=1; entry_valid++; break;
			case 'C': x=2; entry_valid++; break;
			case 'c': x=2; entry_valid++; break;
			case 'D': x=3; entry_valid++; break;
			case 'd': x=3; entry_valid++; break;
			case 'E': x=4; entry_valid++; break;
			case 'e': x=4; entry_valid++; break;
			case 'F': x=5; entry_valid++; break;
			case 'f': x=5; entry_valid++; break;
			case 'G': x=6; entry_valid++; break;
			case 'g': x=6; entry_valid++; break;
			case 'H': x=7; entry_valid++; break;
			case 'h': x=7; entry_valid++; break;
			case 'I': x=8; entry_valid++; break;
			case 'i': x=8; entry_valid++; break;
		};
		
		switch (entry[1]) 
		{
			case '1': y=0; entry_valid++; break;
			case '2': y=1; entry_valid++; break;
			case '3': y=2; entry_valid++; break;
			case '4': y=3; entry_valid++; break;
			case '5': y=4; entry_valid++; break;
			case '6': y=5; entry_valid++; break;
			case '7': y=6; entry_valid++; break;
			case '8': y=7; entry_valid++; break;
			case '9': y=8; entry_valid++; break;
		};
	};
	
	if(x==lastx&&y==lasty)
	{
		entry_valid=0;
	}
};


/*init_game is used to establish all of the variables
While the array is set by default to 0 in all squares, better safe than sorry
Also used to set player 1 as first player (player=1), to establish no current
winner (winner=0), to set the turn number as 1 (turn=1)*/

void init_game(int screenmode)
{		
	allegro_init();
	
	install_timer();
	install_sound(DIGI_AUTODETECT,MIDI_NONE,NULL);
	install_keyboard();
	install_int(check_for_quit,5);
	
	set_keyboard_rate(0,0);
	
	define_strings();
	for(int loop1=0;loop1<8;loop1++)
	{
		for(int loop2=0;loop2<8;loop2++)
		{
			grid[loop1][loop2]=EMPTY_SQUARE;
		};
	};
	
	playercount=0;
	player=1;
	winner=0;
	turn=1;
	removed=0;
	lastx=10;
	lasty=10;
	bgoffset=0;
	score=0;
	
	doublebuffer=create_bitmap(800,600);
	
	srand(unsigned(time(0)));
	
	// Each data loader checks for both /usr/local/share/naughts_and_crosses and ./data
	// for the presence of data files. This is to cover the possibility that the file
	// may be located locally or in the shared resources directory depending on the method
	// of distribution.
	
	gamefont=load_bitmap_font("./data/font_fairlight.pcx",0,0);
	
	if(!gamefont)
	{
		gamefont=load_bitmap_font(GAME_FONT,0,0);
	}
	
	if(!gamefont)
	{
		cout << "Could not load game font (font_fairlight.pcx)\n";
		allegro_exit();
		destroy_bitmap(screen);
		abort();
	};
	

	jamyskis=load_bitmap("./data/jamyskis_logo.bmp",jamy_pal);
	allegro=load_bitmap("./data/allegro_logo.bmp",alleg_pal);
	spritemap=load_bitmap("./data/gfx.bmp",game_pal);
	title_logo=load_bitmap("./data/naughts_logo.bmp",title_pal);
	linux_logo=load_bitmap("./data/linux.bmp",linux_pal);
	
	jamyskissound=load_wav("./data/jamyskis.wav");
	
	if(!jamyskis)
	{
		jamyskis=load_bitmap(JAMYSKIS_LOGO,jamy_pal);
	}
	
	if(!allegro)
	{
		allegro=load_bitmap(ALLEGRO_LOGO_DEF,alleg_pal);
	}
	
	if(!spritemap)
	{
		spritemap=load_bitmap(GAME_GRAPHICS,game_pal);
	}
	
	if(!title_logo)
	{
		title_logo=load_bitmap(NAUGHTS_LOGO,title_pal);
	}
	
	if(!linux_logo)
	{
		linux_logo=load_bitmap(LINUX_LOGO,linux_pal);
	}
	
	if(!jamyskissound)
	{
		jamyskissound=load_wav(JAMYSKIS_SOUND);
	}
	
	if(!jamyskis||!allegro||!spritemap||!title_logo||!linux_logo)
	{
		cout << "Could not load images" << endl;
		allegro_exit();
		destroy_bitmap(screen);
		abort();
	}		
	
	cross=create_bitmap(48,48);
	naught=create_bitmap(48,48);
	grid_top=create_bitmap(48,48);
	grid_bottom=create_bitmap(48,48);
	grid_left=create_bitmap(48,48);
	grid_right=create_bitmap(48,48);
	grid_centre=create_bitmap(48,48);
	
	blit(spritemap,cross,0,0,0,0,48,48);
	blit(spritemap,naught,48,0,0,0,48,48);
	blit(spritemap,grid_bottom,96,0,0,0,48,48);
	blit(spritemap,grid_left,144,0,0,0,48,48);
	blit(spritemap,grid_right,192,0,0,0,48,48);
	blit(spritemap,grid_top,240,0,0,0,48,48);
	blit(spritemap,grid_centre,288,0,0,0,48,48);
	
	title_letters[0]=create_bitmap(65,111);  //N
	title_letters[1]=create_bitmap(55,111);  //A
	title_letters[2]=create_bitmap(50,111);  //U
	title_letters[3]=create_bitmap(50,111);  //G
	title_letters[4]=create_bitmap(50,111);  //H
	title_letters[5]=create_bitmap(30,111);  //T
	title_letters[6]=create_bitmap(50,111);  //S
	title_letters[7]=create_bitmap(80,111);  //&
	title_letters[8]=create_bitmap(70,111);  //C
	title_letters[9]=create_bitmap(40,111);  //R
	title_letters[10]=create_bitmap(50,111); //O
	title_letters[11]=create_bitmap(50,111); //S
	title_letters[12]=create_bitmap(45,111); //S
	title_letters[13]=create_bitmap(60,111); //E
	title_letters[14]=create_bitmap(50,111); //S
	
	blit(title_logo,title_letters[0],0,0,0,0,65,111);
	blit(title_logo,title_letters[1],65,0,0,0,55,111);
	blit(title_logo,title_letters[2],120,0,0,0,50,111);
	blit(title_logo,title_letters[3],170,0,0,0,50,111);
	blit(title_logo,title_letters[4],220,0,0,0,50,111);
	blit(title_logo,title_letters[5],270,0,0,0,30,111);
	blit(title_logo,title_letters[6],300,0,0,0,50,111);
	blit(title_logo,title_letters[7],350,0,0,0,80,111);
	blit(title_logo,title_letters[8],430,0,0,0,70,111);
	blit(title_logo,title_letters[9],500,0,0,0,40,111);
	blit(title_logo,title_letters[10],540,0,0,0,50,111);
	blit(title_logo,title_letters[11],590,0,0,0,50,111);
	blit(title_logo,title_letters[12],640,0,0,0,45,111);
	blit(title_logo,title_letters[13],685,0,0,0,60,111);
	blit(title_logo,title_letters[14],745,0,0,0,50,111);
	
	set_color_depth(8);
	switch(screenmode)
	{
		case 0:	set_gfx_mode(GFX_AUTODETECT,800,600,0,0); break;
		case 1: set_gfx_mode(GFX_AUTODETECT_WINDOWED,800,600,0,0); break;
		case 2: set_gfx_mode(GFX_AUTODETECT_FULLSCREEN,800,600,0,0); break;
	};
	
	clear_to_color(screen,0);
	
	textout_ex(screen,gamefont,"Loading...",230,200,15,0);
	
	typesound=load_wav("./data/typechar.wav");
	confirmsound=load_wav("./data/confirm.wav");
	placesound=load_wav("./data/placed.wav");
	
	if(!typesound)
	{
		typesound=load_wav(TYPECHAR_SOUND);
	}
	
	if(!confirmsound)
	{
		confirmsound=load_wav(CONFIRM_SOUND);
	}
	
	if(!placesound)
	{
		placesound=load_wav(PLACED_SOUND);
	}
	
	if(!typesound||!confirmsound||!placesound)
	{
		cout << "Could not load sound effects\n";
		allegro_exit();
		destroy_bitmap(screen);
		abort();
	};
	
	clear_to_color(screen,0);
	set_palette(black_palette);
	rest(2000);
	
	blit(jamyskis,screen,0,0,0,0,800,600);
	
	fade_in(jamy_pal,5);
	if(jamyskissound)
	{
		play_sample(jamyskissound,255,128,1000,0);
	}
	rest(2000);
	fade_out(5);
	
	blit(allegro,screen,0,0,0,0,800,600);
	
	fade_in(alleg_pal,5);
	rest(2000);
	fade_out(5);
	
	blit(linux_logo,screen,0,0,0,0,800,600);
	
	fade_in(linux_pal,5);
	rest(2000);
	fade_out(5);
	
	destroy_bitmap(jamyskis);
	destroy_bitmap(allegro);
};

void reset_game()
{
	for(int boardx=0; boardx<10; boardx++)
	{
		for(int boardy=0; boardy<10; boardy++)
		{
			grid[boardx][boardy]=EMPTY_SQUARE;
		}
	}
	
	p1_name="";
	p2_name="";
	
	playercount=0;
	player=1;
	winner=0;
	turn=1;
	removed=0;
	lastx=10;
	lasty=10;
	
	score=0;
}

/*Update the title logo*/

void update_title_logo(int letter_offset, int offset_direction)
{		
	for(int evenletters=0; evenletters<8; evenletters=evenletters+2)
	{
		draw_sprite(doublebuffer,title_letters[evenletters],130+(evenletters*65),60-letter_offset);
	};
	
	for(int oddletters=1; oddletters<7; oddletters=oddletters+2)
	{
		draw_sprite(doublebuffer,title_letters[oddletters],130+(oddletters*65),60+letter_offset);
	};
	
	for(int evenletters=0; evenletters<8; evenletters=evenletters+2)
	{
		draw_sprite(doublebuffer,title_letters[evenletters+8],130+(evenletters*65),140-letter_offset);
	};
	
	for(int oddletters=1; oddletters<7; oddletters=oddletters+2)
	{
		draw_sprite(doublebuffer,title_letters[oddletters+8],130+(oddletters*65),140+letter_offset);
	};
	
	draw_sprite(doublebuffer,title_letters[7],600,100);
	
	blit(doublebuffer,screen,0,0,0,0,800,600);
	set_palette(title_pal);
};

/*Display intro text and gather board size and names*/

bool options_entry()
{
	bool title_screen_active=true;
	string menulist[10];
	int menu_limit;
	int current_menu=1;
	int current_item=0;
	int letter_offset=0;
	int offset_direction=1;
	int selection=0;
	stringstream stringbuffer;
	int keycheck;
	bool key_released=false;
	char keyboardarray[73]="ABCDEFGHIJKLMNOPQRSTUVWXYZ01234567890123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ";

	
	while(title_screen_active)
	{	
		clear_to_color(doublebuffer,129);
		
		for(int checkpatternx=-80; checkpatternx<800; checkpatternx=checkpatternx+160)
		{
			for(int checkpatterny=-80; checkpatterny<600; checkpatterny=checkpatterny+160)
			{
				rectfill(doublebuffer,checkpatternx+bgoffset,checkpatterny+bgoffset,checkpatternx+80+bgoffset,checkpatterny+80+bgoffset,130);
				rectfill(doublebuffer,checkpatternx+80+bgoffset,checkpatterny+80+bgoffset,checkpatternx+160+bgoffset,checkpatterny+160+bgoffset,130);
			}
		}
		
		bgoffset++;
		if(bgoffset>=80)
		{
			bgoffset=0;
		};
		
		letter_offset=letter_offset+offset_direction;
	
		if(letter_offset==20&&offset_direction==1)
		{
			offset_direction=-1;
		}
	
		if(letter_offset==-20&&offset_direction==-1)
		{
			offset_direction=1;
		};

		switch(current_menu)
		{
			case 1:
				menulist[0]=_START_1UP_MATCH;
				menulist[1]=_START_2UP_MATCH;
				menulist[2]=_OPTIONS;
				menulist[3]=_QUIT_TO_DESKTOP;
				menu_limit=3;
				break;
			
			case 2:
				stringbuffer << difficulty << '\0';
				menulist[0]=_DIFFICULTY+stringbuffer.str();
				stringbuffer.str("");
				stringbuffer << boardsize << '\0';
				menulist[1]=_BOARDSIZE+stringbuffer.str();
				stringbuffer.str("");
				switch(enhanced)
				{
					case 0: stringbuffer << _TRADITIONAL; break;
					case 1: stringbuffer << _ENHANCED; break;
					case 2: stringbuffer << _ARCADE; break;
				}
				menulist[2]=_ENHANCED_GAME+stringbuffer.str();
				stringbuffer.str("");
				menulist[3]=_BACK_MAIN_MENU;
				menu_limit=3;
				break;
				
			case 3:
				menulist[0]=_PLAYER1_ENTERNAME+p1_name;
				menulist[1]=_BEGIN_GAME;
				menulist[2]=_BACK_MAIN_MENU;
				menu_limit=2;
				if(playercount==2)
				{
					menulist[1]=_PLAYER2_ENTERNAME+p2_name;
					menulist[2]=_BEGIN_GAME;
					menulist[3]=_BACK_MAIN_MENU;
					menu_limit=3;
				};
				break;
		};
		
		//Display the menu

		for(int listmenuitems=0; listmenuitems<menu_limit+1; listmenuitems++)
		{
			if(listmenuitems==current_item)
			{
				textout_ex(doublebuffer,gamefont,menulist[listmenuitems].c_str(),150,280+(listmenuitems*40),0,128);
			}
			else
			{
				textout_ex(doublebuffer,gamefont,menulist[listmenuitems].c_str(),150,280+(listmenuitems*40),128,-1);
			}
		};

		textout_ex(doublebuffer,gamefont,_WRITTENBY.c_str(),130,500,127,-1);
		textout_ex(doublebuffer,gamefont,_GPLNOTICE.c_str(),50,540,127,-1);
		textout_ex(doublebuffer,gamefont,_VERSIONNUMBER.c_str(),700,0,82,-1);

		update_title_logo(letter_offset,offset_direction);
		
		if(key[KEY_UP]||key[KEY_DOWN]||key[KEY_ENTER]||key[KEY_LEFT]||key[KEY_RIGHT])
		{
			if(key[KEY_UP]) {selection=1;}
			if(key[KEY_DOWN]) {selection=2;}
			if(key[KEY_ENTER]) {selection=3;}
			if(key[KEY_LEFT]) {selection=4;}
			if(key[KEY_RIGHT]) {selection=5;}
		}
		
		if(!key[KEY_UP]&&!key[KEY_DOWN]&&!key[KEY_ENTER]&&!key[KEY_LEFT]&&!key[KEY_RIGHT]&&selection>0)
		{
			switch(selection)
			{
				case 1:
					if(current_item>0)
					{
						current_item--;
						play_sample(typesound,255,100,1000,0);
					}; break;
				case 2:
					if(current_item<menu_limit)
					{
						current_item++;
						play_sample(typesound,255,100,1000,0);
					}; break;
				case 3:
					
					
				
					if(current_menu==1&&current_item==0)
					{
						current_menu=3;
						current_item=0;
						playercount=1;
						play_sample(confirmsound,255,100,1000,0);
					};
					
					if(current_menu==1&&current_item==1)
					{
						current_menu=3;
						current_item=0;
						playercount=2;
						play_sample(confirmsound,255,100,1000,0);
					};
					
					if(current_menu==1&&current_item==2)
					{
						current_menu=2;
						current_item=0;
						play_sample(confirmsound,255,100,1000,0);
					};

					if(current_menu==1&&current_item==3)
					{
						play_sample(confirmsound,255,100,1000,0);
						fade_out(2);
						return(false);
					};
						
					if(current_menu==2&&current_item==3)
					{
						current_menu=1;
						current_item=0;
						play_sample(confirmsound,255,100,1000,0);
					};
					
					if(current_menu==3&&current_item==playercount)
					{
						title_screen_active=false;
						play_sample(confirmsound,255,100,1000,0);
					};
					
					if(current_menu==3&&current_item==playercount+1)
					{
						current_menu=1;
						current_item=0;
						play_sample(confirmsound,255,100,1000,0);
					}; break;
					
				case 4:
					if(current_menu==2)
					{
						switch(current_item)
						{
							case 0:
								difficulty--;
								play_sample(typesound,255,100,1000,0);
								if(difficulty<0)
								{
									difficulty=0;
								}
								break;
							case 1:
								boardsize--;
								play_sample(typesound,255,100,1000,0);
								if(boardsize<3)
								{
									boardsize=3;
								}
								break;
							case 2:
								play_sample(typesound,255,100,1000,0);
								enhanced--;
								if(enhanced<0)
								{
									enhanced=0;
								};
								break;
						}
					}
					break;
				case 5:
					if(current_menu==2)
					{
						switch(current_item)
						{
							case 0:
								difficulty++;
								play_sample(typesound,255,100,1000,0);
								if(difficulty>20)
								{
									difficulty=20;
								}
								break;
							case 1:
								boardsize++;
								play_sample(typesound,255,100,1000,0);
								if(boardsize>9)
								{
									boardsize=9;
								}
								break;
							case 2:
								enhanced++;							
								if(enhanced>2)
								{
									enhanced=2;
								};
								play_sample(typesound,255,100,1000,0);
								break;
						}
					}
					break;
			}
			selection=0;
		};
		
		keycheck=0;
		clear_keybuf();

		for(int keyrange=1;keyrange<46;keyrange++)
		{
			if (key[keyrange])
			{
				keycheck=keyrange;
			};
		};
		
		if(key[KEY_BACKSPACE])
		{
			keycheck=63;
		};

		if (keycheck>0&&current_menu==3)
		{	
			if(keycheck<47&&p1_name.length()<12&&current_item==0&&key_released==true)
			{
				p1_name=p1_name+keyboardarray[keycheck-1];
				key_released=false;
				play_sample(typesound,255,100,1000,0);
			};
			
			if(keycheck<47&&p2_name.length()<12&&current_item==1&&playercount==2&&key_released==true)
			{
				p2_name=p2_name+keyboardarray[keycheck-1];
				key_released=false;
				play_sample(typesound,255,100,1000,0);
			};
			
			if(keycheck==63&&current_item==0&&p1_name.length()>0&&key_released==true)
			{
				p1_name=p1_name.erase(p1_name.length()-1,p1_name.length());
				key_released=false;
				play_sample(typesound,255,100,1000,0);
			};
			
			if(keycheck==63&&current_item==1&&p2_name.length()>0&&playercount==2&&key_released==true)
			{
				p2_name=p2_name.erase(p2_name.length()-1,p2_name.length());
				key_released=false;
				play_sample(typesound,255,100,1000,0);
			};
		
			keycheck=0;
		}
		else
		{
			key_released=true;
		}
	}
	
	margin_of_error=(double) (20-difficulty)/21;
	
	if (playercount==1)
	{
		p2_name="COMPUTER";
	};
	
	fade_out(2);
	return(true);
};


/*Output winner*/

void winner_output()
{	
	string winnerresult;
	
	clear_to_color(screen,0);
	display_board();
	
	bool breakout=false;
	
	switch (winner) 
	{
		case 1: winnerresult=_WINNER+p1_name; break;
		case 2: winnerresult=_WINNER+p2_name; break;
		case 3: winnerresult=_DRAWN; break;
	};
	
	do
	{
	} while (key[KEY_ENTER]);
	
	clear_keybuf();
	
	do
	{
		clear_to_color(doublebuffer,2);
			
		for(int checkpatternx=-80; checkpatternx<800; checkpatternx=checkpatternx+160)
		{
			for(int checkpatterny=-80; checkpatterny<600; checkpatterny=checkpatterny+160)
			{
				rectfill(doublebuffer,checkpatternx+bgoffset,checkpatterny+bgoffset,checkpatternx+80+bgoffset,checkpatterny+80+bgoffset,1);
				rectfill(doublebuffer,checkpatternx+80+bgoffset,checkpatterny+80+bgoffset,checkpatternx+160+bgoffset,checkpatterny+160+bgoffset,1);
			}
		}
			
		bgoffset++;
		if(bgoffset>=80)
		{
			bgoffset=0;
		};
		textout_ex(doublebuffer,gamefont,winnerresult.c_str(),0,0,7,-1);
		textout_ex(doublebuffer,gamefont,_EXITPHRASE.c_str(),0,25,7,-1);	
		display_board();
		
		blit(doublebuffer,screen,0,0,0,0,800,600);
		
		for(int keycheck=1; keycheck<120; keycheck++)
		{
			if(key[keycheck])
			{
				breakout=true;
			}
		}
	} while(!breakout);

};


/*  AI Routines for placing a piece */

void ai_routine_place()
{
		int attractiveness[10][10];
		int best_value;;
		int attempts, attempt_limit;
		int naught_available, cross_available;
		int linetest;
		int middle;
		float rounddown;

		/*Set values for availability of lines*/
		
		for (int reset=0; reset<boardsize; reset++)
		{
			for (int reset2=0; reset2<boardsize; reset2++)
			{
				attractiveness[reset][reset2]=0;
			};
		};
		
		/*Check if on diagonal*/
		
		/*for (int testx=0; testx<boardsize; testx++)
		{
			attractiveness[testx][testx]=attractiveness[testx][testx]+(boardsize/2);
			attractiveness[boardsize-testx-1][testx]=attractiveness[boardsize-testx-1][testx]+(boardsize/2);
		};*/
		
		if(boardsize%2==0)
		{
			attractiveness[(boardsize/2)-1][(boardsize/2)-1]=attractiveness[(boardsize/2)-1][(boardsize/2)-1]+(boardsize*2);
			attractiveness[(boardsize/2)-1][(boardsize/2)]=attractiveness[(boardsize/2)-1][(boardsize/2)]+(boardsize*2);
			attractiveness[(boardsize/2)][(boardsize/2)-1]=attractiveness[(boardsize/2)][(boardsize/2)-1]+(boardsize*2);
			attractiveness[(boardsize/2)][(boardsize/2)]=attractiveness[(boardsize/2)][(boardsize/2)]+(boardsize*2);
		}
		
		if(boardsize%2!=0)
		{
			//The following apparently needless code is to accommodate the fact
			//that some compilers round down, and some round up. It is for this
			//reason that I first convert to floating point, strip off the .5,
			//and then convert back to int.
			
			rounddown=(float) boardsize/2;
			rounddown=rounddown-0.5;
			middle=(int) rounddown;
			
			attractiveness[middle][middle]=attractiveness[middle][middle]+(boardsize*2);
		};	
		
		/*Check for exclusivity*/
		
		for (int testx=0; testx<boardsize; testx++)
		{
			for (int testy=0; testy<boardsize; testy++)
			{
				// Check vertically
				
				naught_available=false;
				cross_available=false;
				linetest=0;
				
				for(int checky=0; checky<boardsize; checky++)
				{
					if(grid[testx][checky]==NAUGHT)
					{
						linetest++;
						naught_available=true;
					}
					
					if(grid[testx][checky]==CROSS)
					{
						linetest++;
						cross_available=true;
					}
				}
				
				if(cross_available&&!naught_available)
				{
					attractiveness[testx][testy]=attractiveness[testx][testy]+(boardsize*2);
				};
				
				if(!cross_available&&naught_available)
				{
					attractiveness[testx][testy]=attractiveness[testx][testy]+boardsize;
				}
				
				if(naught_available&&!cross_available&&linetest==boardsize-1)
				{
					attractiveness[testx][testy]=attractiveness[testx][testy]+(boardsize*4);
				}

				// Check horizontally
				
				naught_available=false;
				cross_available=false;
				linetest=0;
				
				for(int checkx=0; checkx<boardsize; checkx++)
				{
					if(grid[checkx][testy]==NAUGHT)
					{
						linetest++;
						naught_available=true;
					}
					
					if(grid[checkx][testy]==CROSS)
					{
						linetest++;
						cross_available=true;
					}
				}
				
				if(cross_available&&!naught_available)
				{
					attractiveness[testx][testy]=attractiveness[testx][testy]+(boardsize*2);
				};
				
				if(!cross_available&&naught_available)
				{
					attractiveness[testx][testy]=attractiveness[testx][testy]+boardsize;
				}
				
				if(naught_available&&!cross_available&&linetest==boardsize-1)
				{
					attractiveness[testx][testy]=attractiveness[testx][testy]+(boardsize*4);
				}

				// Check diagonally left to right
				
				if(testx==testy)
				{			
					naught_available=false;
					cross_available=false;
					linetest=0;
				
					for(int checkx=0; checkx<boardsize; checkx++)
					{
						if(grid[checkx][checkx]==NAUGHT)
						{
							linetest++;
							naught_available=true;
						}
						
						if(grid[checkx][checkx]==CROSS)
						{
							linetest++;
							cross_available=true;
						}
					}
				
					if(cross_available&&!naught_available)
					{
						attractiveness[testx][testy]=attractiveness[testx][testy]+(boardsize*2);
					};
				
					if(!cross_available&&naught_available)
					{
						attractiveness[testx][testy]=attractiveness[testx][testy]+boardsize;
					}
				
					if(naught_available&&!cross_available&&linetest==boardsize-1)
					{
						attractiveness[testx][testy]=attractiveness[testx][testy]+(boardsize*4);
					}
				}
				
				// Check diagonally right to left
				
				if(testx==boardsize-testy-1)
				{			
					naught_available=false;
					cross_available=false;
					linetest=0;
				
					for(int checkx=0; checkx<boardsize; checkx++)
					{
						if(grid[checkx][boardsize-checkx-1]==NAUGHT)
						{
							linetest++;
							naught_available=true;
						}
						
						if(grid[checkx][boardsize-checkx-1]==CROSS)
						{
							linetest++;
							cross_available=true;
						}
					}
				
					if(cross_available&&!naught_available)
					{
						attractiveness[testx][testy]=attractiveness[testx][testy]+(boardsize*2);
					};
				
					if(!cross_available&&naught_available)
					{
						attractiveness[testx][testy]=attractiveness[testx][testy]+boardsize;
					}
				
					if(naught_available&&!cross_available&&linetest==boardsize-1)
					{
						attractiveness[testx][testy]=attractiveness[testx][testy]+(boardsize*4);
					}
				}
			}
		}
									
		/*Check for best value*/
	
		best_value=0;
	
		for (int ex=0; ex<boardsize; ex++)
		{
			for (int ey=0; ey<boardsize; ey++)
			{
				if (attractiveness[ex][ey]>best_value)
				{
					best_value=attractiveness[ex][ey];
				};
			};
		};

		/*Scan for fitting square, if attempt_limit is reached, drop attractiveness*
	      requirement by 1*/

		attempt_limit=15;	
		attempts=0;
	
		while ((double) rand()/RAND_MAX<margin_of_error)
		{
			best_value--;
		};
	
		do
		{
			x=rand()%boardsize;
			y=rand()%boardsize;
			entry_valid=0;
		
			if (grid[x][y]==EMPTY_SQUARE&&attractiveness[x][y]>=best_value)
			{
				entry_valid=2;
			}
			else
			{
				entry_valid=0;
			};
		
			attempts++;

			if (x==lastx&&y==lasty)
			{
				entry_valid=0;
			}
		
			if (attempts>attempt_limit)
			{
				best_value--;
				attempts=0;
			};
		} while (entry_valid==0);
}


/* AI routines for removing a piece */

void ai_routine_remove()
{
	int usefulness[10][10];
	int least_useful;
	int attempts, attempt_limit;
	int naught_available, cross_available;
	int linetest;
	int middle;
	float rounddown;

	/*Set values for availability of lines*/
	
	for (int reset=0; reset<boardsize; reset++)
	{
		for (int reset2=0; reset2<boardsize; reset2++)
		{
			usefulness[reset][reset2]=0;
		};
	};
	
	/*Check if on diagonal*/
	
	/*for (int testx=0; testx<boardsize; testx++)
	{
		usefulness[testx][testx]=usefulness[testx][testx]+(boardsize/2);
		usefulness[boardsize-testx-1][testx]=usefulness[boardsize-testx-1][testx]+(boardsize/2);
	};*/
	
	if(boardsize%2==0)
	{
		usefulness[(boardsize/2)-1][(boardsize/2)-1]=usefulness[(boardsize/2)-1][(boardsize/2)-1]+(boardsize*2);
		usefulness[(boardsize/2)-1][(boardsize/2)]=usefulness[(boardsize/2)-1][(boardsize/2)]+(boardsize*2);
		usefulness[(boardsize/2)][(boardsize/2)-1]=usefulness[(boardsize/2)][(boardsize/2)-1]+(boardsize*2);
		usefulness[(boardsize/2)][(boardsize/2)]=usefulness[(boardsize/2)][(boardsize/2)]+(boardsize*2);
	}
	
	if(boardsize%2!=0)
	{
		//The following apparently needless code is to accommodate the fact
		//that some compilers round down, and some round up. It is for this
		//reason that I first convert to floating point, strip off the .5,
		//and then convert back to int.
		
		rounddown=(float) boardsize/2;
		rounddown=rounddown-0.5;
		middle=(int) rounddown;
		
		usefulness[middle][middle]=usefulness[middle][middle]+(boardsize*2);
	};
		
		
	/*Check for exclusivity*/
	
		/*Check for exclusivity*/
		
		for (int testx=0; testx<boardsize; testx++)
		{
			for (int testy=0; testy<boardsize; testy++)
			{
				// Check vertically
				
				naught_available=false;
				cross_available=false;
				linetest=0;
				
				for(int checky=0; checky<boardsize; checky++)
				{
					if(grid[testx][checky]==NAUGHT)
					{
						linetest++;
						naught_available=true;
					}
					
					if(grid[testx][checky]==CROSS)
					{
						linetest++;
						cross_available=true;
					}
				}
				
				if(cross_available&&!naught_available)
				{
					usefulness[testx][testy]=usefulness[testx][testy]+(boardsize*2);
				};
				
				if(!cross_available&&naught_available)
				{
					usefulness[testx][testy]=usefulness[testx][testy]+boardsize;
				}
				
				if(naught_available&&!cross_available&&linetest==boardsize-1)
				{
					usefulness[testx][testy]=usefulness[testx][testy]+(boardsize*8);
				}

				// Check horizontally
				
				naught_available=false;
				cross_available=false;
				linetest=0;
				
				for(int checkx=0; checkx<boardsize; checkx++)
				{
					if(grid[checkx][testy]==NAUGHT)
					{
						linetest++;
						naught_available=true;
					}
					
					if(grid[checkx][testy]==CROSS)
					{
						linetest++;
						cross_available=true;
					}
				}
				
				if(cross_available&&!naught_available)
				{
					usefulness[testx][testy]=usefulness[testx][testy]+(boardsize*2);
				};
				
				if(!cross_available&&naught_available)
				{
					usefulness[testx][testy]=usefulness[testx][testy]+boardsize;
				}
				
				if(naught_available&&!cross_available&&linetest==boardsize-1)
				{
					usefulness[testx][testy]=usefulness[testx][testy]+(boardsize*8);
				}

				// Check diagonally left to right
				
				if(testx==testy)
				{			
					naught_available=false;
					cross_available=false;
					linetest=0;
				
					for(int checkx=0; checkx<boardsize; checkx++)
					{
						if(grid[checkx][checkx]==NAUGHT)
						{
							linetest++;
							naught_available=true;
						}
						
						if(grid[checkx][checkx]==CROSS)
						{
							linetest++;
							cross_available=true;
						}
					}
				
					if(cross_available&&!naught_available)
					{
						usefulness[testx][testy]=usefulness[testx][testy]+(boardsize*2);
					};
				
					if(!cross_available&&naught_available)
					{
						usefulness[testx][testy]=usefulness[testx][testy]+boardsize;
					}
				
					if(naught_available&&!cross_available&&linetest==boardsize-1)
					{
						usefulness[testx][testy]=usefulness[testx][testy]+(boardsize*8);
					}
				}
				
				// Check diagonally right to left
				
				if(testx==boardsize-testy-1)
				{			
					naught_available=false;
					cross_available=false;
					linetest=0;
				
					for(int checkx=0; checkx<boardsize; checkx++)
					{
						if(grid[checkx][boardsize-checkx-1]==NAUGHT)
						{
							linetest++;
							naught_available=true;
						}
						
						if(grid[checkx][boardsize-checkx-1]==CROSS)
						{
							linetest++;
							cross_available=true;
						}
					}
				
					if(cross_available&&!naught_available)
					{
						usefulness[testx][testy]=usefulness[testx][testy]+(boardsize*2);
					};
				
					if(!cross_available&&naught_available)
					{
						usefulness[testx][testy]=usefulness[testx][testy]+boardsize;
					}
				
					if(naught_available&&!cross_available&&linetest==boardsize-1)
					{
						usefulness[testx][testy]=usefulness[testx][testy]+(boardsize*8);
					}
				}
			}
		}
	
	/*Check for best value*/

	least_useful=usefulness[0][0];

	for (int ex=0; ex<boardsize; ex++)
	{
		for (int ey=0; ey<boardsize; ey++)
		{
			if (usefulness[ex][ey]<least_useful)
			{
				least_useful=usefulness[ex][ey];
			};
		};
	};
		/*Scan for fitting square, if attempt_limit is reached, drop attractiveness*
          requirement by 1*/
	
	attempt_limit=1500;	
	attempts=0;

	while ((double) rand()/RAND_MAX<margin_of_error)
	{
		least_useful++;
	};

	do
	{
		x=rand()%boardsize;
		y=rand()%boardsize;
		entry_valid=0;
	
		if (grid[x][y]==CROSS&&usefulness[x][y]<=least_useful)
		{
			entry_valid=2;
		}
		else
		{
			entry_valid=0;
		};
	
		attempts++;
	
		if (attempts>attempt_limit)
		{
			least_useful++;
			attempts=0;
		};
		
		if(x==lastx&&y==lasty)
		{
			entry_valid=0;
		}
	} while (entry_valid==0);
}



/* Routine to allow player to enter move */

void player_routine_place()
{
	time_taken=0;
	clear_to_color(screen,0);
	display_player_notification();
	display_board();
	entry=string_entry(0,50,2,7);
	play_sample(placesound,255,100,1000,0);
}


/* Routine to allow player to remove piece */

void player_routine_remove()
{
	clear_to_color(screen,0);
	display_player_notification();
	display_board();
	textout_ex(screen,gamefont,_ENTERREMOVESPACE.c_str(),0,25,7,0);
	entry=string_entry(0,50,2,7);
	play_sample(placesound,255,100,1000,0);
}


/* Check if game should be ended early because noone can win */

void determine_draw()
{
	bool can_be_won=false;
	bool naught_available, cross_available;
	
	// Rows
	for (int row=0; row<boardsize; row++)
	{
		naught_available=false;
		cross_available=false;
			
		for (int column=0; column<boardsize; column++)
		{
			if (grid[column][row]==NAUGHT)
			{
				naught_available=true;
			};
			
			if (grid[column][row]==CROSS)
			{
				cross_available=true;
			};
		};
		
		if (cross_available==false||naught_available==false)
		{
			can_be_won=true;
		};
	};
			
			
	
	// Columns
	for (int column=0; column<boardsize; column++)
	{
		naught_available=false;
		cross_available=false;
			
		for (int row=0; row<boardsize; row++)
		{
			if (grid[column][row]==NAUGHT)
			{
				naught_available=true;
			};
			
			if (grid[column][row]==CROSS)
			{
				cross_available=true;
			};
		};
		
		if (cross_available==false||naught_available==false)
		{
			can_be_won=true;
		};
	};
	
	// Diagonals
	
	naught_available=false;
	cross_available=false;
	
	for (int testdiag=0; testdiag<boardsize; testdiag++)
	{
		if (grid[testdiag][testdiag]==NAUGHT)
		{
			naught_available=true;
		}
		
		if (grid[testdiag][testdiag]==CROSS)
		{
			cross_available=true;
		};
	};
	
	if (cross_available==false||naught_available==false)
	{
		can_be_won=true;
	};
	
	naught_available=false;
	cross_available=false;
	
	for (int testdiag=0; testdiag<boardsize; testdiag++)
	{
		if (grid[boardsize-testdiag-1][testdiag]==NAUGHT)
		{
			naught_available=true;
		}
		
		if (grid[boardsize-testdiag-1][testdiag]==CROSS)
		{
			cross_available=true;
		};
	};
	
	if (cross_available==false||naught_available==false)
	{
		can_be_won=true;
	};
	
	if (can_be_won==false&&enhanced==0)
	{
		winner=3;
	};
	
	if (can_be_won==false&&enhanced==2)
	{
		for(int resetx=0; resetx<boardsize; resetx++)
		{
			for(int resety=0; resety<boardsize; resety++)
			{
				grid[resetx][resety]=EMPTY_SQUARE;
			}
		};
		
		score=score-500;
	}
};


/*Display state, gather entry input and test for validity*/

bool game_loop()
{
	float rounddown;
	int middle;
	
	clear_to_color(screen,0);
	display_player_notification();
	display_board();
	fade_in(game_pal,2);
	
	do 
	{		
	/* Test if "alternative" version has been selected, and if so, check if the time
   has come to get a piece removed, and do so if it is */
			
		canplace=1;
		
		if (enhanced==1&&turn>(boardsize*2)+2&&removed<turn&&boardsize>3)
		{
			canplace=0;
		};
		
		if(enhanced==1&&boardsize==3&&turn>6&&removed<turn)
		{
			canplace=0;
		}
		
		if (canplace==0)
		{
			if (player==1)
			{
				player_routine_remove();
				check_entry_valid();
			};
			
			if (player==2&&playercount==1)
			{
				ai_routine_remove();
			};
			
			if (player==2&&playercount==2)
			{
				player_routine_remove();
				check_entry_valid();
			};
				
			remove_piece_from_board();
		};

		display_player_notification();
		
		
/* Get requested square, check validity of input and place the piece on the
   board */

		if (canplace==1)
		{
			if (player==1)
			{
				player_routine_place();
				check_entry_valid();
			};
			
			if (player==2&&playercount==1)
			{
				ai_routine_place();
			};
			
			if (player==2&&playercount==2)
			{
				player_routine_place();
				check_entry_valid();
			};

			if(time_taken<200)
			{
				score=score+100;
			}
			
			if(turn<2&&player==1&&boardsize%2==0&&(x==boardsize/2||x==(boardsize/2)-1)&&(y==boardsize/2||y==(boardsize/2)-1))
			{
				score=score+1000;
			}
			
			if(turn<2&&player==1&&boardsize%2!=0)
			{
				rounddown=(float) boardsize/2;
				rounddown=rounddown-0.5;
				middle=(int) rounddown;
				
				if(x==middle&&y==middle)
				{
					score=score+1000;
				}
			}
			
			set_piece_on_board();
				
			determine_winner();
		};
		
		determine_draw();
	} while (winner==0);

	return(true);
}
