//--------------------------------------------------------------------------
//
//  Tetris Unlimited
//  Copyright (C) 2001-2003  Oscar Giner Martnez
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//
//--------------------------------------------------------------------------


#include "stdafx.h"

#include "defines.h"
#include "ExcepcionTU.h"
#include "Video.h"

//##ModelId=3FB2143F01C9
Video::Video()
:modo_actual(automatico)
{
}

//##ModelId=3FB2143F01D1
Video::~Video()
{
}

//##ModelId=3FB2143F01D2
bool Video::Iniciar(bool aceleracion_hw, ModoActualizacion modo)
{
	if (aceleracion_hw && (gfx_capabilities & (GFX_HW_FILL | GFX_HW_VRAM_BLIT)))
	{
		aceleracion_hw = true;
	}
	else
	{
		aceleracion_hw = false;
	}

	if (modo == automatico)
	{
		if (aceleracion_hw && gfx_capabilities & GFX_CAN_TRIPLE_BUFFER)
		{
			modo = triple_buffer;
		}
		else
		{
			enable_triple_buffer();
			if (aceleracion_hw && gfx_capabilities & GFX_CAN_TRIPLE_BUFFER)
			{
				modo = triple_buffer;
			}
			else if (aceleracion_hw)
			{
				modo = page_flipping;
			}
			else
			{
				modo = doble_buffer;
			}
		}
	}

	switch (modo)
	{
	case doble_buffer:
		if (aceleracion_hw)
		{
			dummy = create_video_bitmap(SCREEN_W, SCREEN_H);
			buffer = create_video_bitmap(SCREEN_W, SCREEN_H);
			if (!buffer)
			{
				destroy_bitmap(dummy);
				dummy = NULL;
				buffer = create_bitmap(SCREEN_W, SCREEN_H);
				aceleracion_hw = false;
			}
		}
		else
		{
			buffer = create_bitmap(SCREEN_W, SCREEN_H);
			if (!buffer) return false;
		}
		sub_buffer = create_sub_bitmap(buffer, 0, 0, SCREEN_W, SCREEN_H);
		sub_screen = create_sub_bitmap(screen, 0, 0, SCREEN_W, SCREEN_H);
		modo_actual = doble_buffer;
		break;
	case page_flipping:
		page[0] = create_video_bitmap(SCREEN_W, SCREEN_H);
		if (!page[0]) return false;
		page[1] = create_video_bitmap(SCREEN_W, SCREEN_H);
		if (!page[1])
		{
			destroy_bitmap(page[0]);
			return false;
		}
		pagina_actual = 0;
		clear_bitmap(page[0]);
		show_video_bitmap(page[0]);
		sub_page[0] = create_sub_bitmap(page[0], 0, 0, SCREEN_W, SCREEN_H);
		sub_page[1] = create_sub_bitmap(page[1], 0, 0, SCREEN_W, SCREEN_H);
		modo_actual = page_flipping;
		break;
	case triple_buffer:
		if (!(gfx_capabilities & GFX_CAN_TRIPLE_BUFFER))
		{
			enable_triple_buffer();
			if (!(gfx_capabilities & GFX_CAN_TRIPLE_BUFFER))
			{
				return false;
			}
		}
		page[0] = create_video_bitmap(SCREEN_W, SCREEN_H);
		if (!page[0]) return false;
		page[1] = create_video_bitmap(SCREEN_W, SCREEN_H);
		if (!page[1])
		{
			destroy_bitmap(page[0]);
			return false;
		}
		page[2] = create_video_bitmap(SCREEN_W, SCREEN_H);
		if (!page[1])
		{
			destroy_bitmap(page[0]);
			destroy_bitmap(page[1]);
			return false;
		}
		pagina_actual = 0;
		clear_bitmap(page[0]);
		request_video_bitmap(page[0]);
		sub_page[0] = create_sub_bitmap(page[0], 0, 0, SCREEN_W, SCREEN_H);
		sub_page[1] = create_sub_bitmap(page[1], 0, 0, SCREEN_W, SCREEN_H);
		sub_page[2] = create_sub_bitmap(page[2], 0, 0, SCREEN_W, SCREEN_H);
		modo_actual = triple_buffer;
		break;
	case automatico:
		throw new ExcepcionTU("Error interno en Video::Iniciar()", -1);
	}

	hw_accel = aceleracion_hw;

	vista_x = 0;
	vista_y = 0;
	vista_w = SCREEN_W;
	vista_h = SCREEN_H;
	contador_iniciar_salida = 0;

	return true;
}

//##ModelId=3FB2143F01D5
void Video::Finalizar()
{
	Vaciar();

	show_video_bitmap(screen);

	switch (modo_actual)
	{
	case doble_buffer:
		if (hw_accel)
		{
			destroy_bitmap(dummy);
		}
		destroy_bitmap(sub_buffer);
		destroy_bitmap(buffer);
		destroy_bitmap(sub_screen);
		break;
	case page_flipping:
		destroy_bitmap(sub_page[0]);
		destroy_bitmap(sub_page[1]);
		destroy_bitmap(page[0]);
		destroy_bitmap(page[1]);
		break;
	case triple_buffer:
		destroy_bitmap(sub_page[0]);
		destroy_bitmap(sub_page[1]);
		destroy_bitmap(sub_page[2]);
		destroy_bitmap(page[0]);
		destroy_bitmap(page[1]);
		destroy_bitmap(page[2]);
		break;
	case automatico:
		throw new ExcepcionTU("Error interno en Video::Finalizar()", -1);
	}

	modo_actual = automatico;
}

//##ModelId=3FB2143F01DB
void Video::ModificarColorFondo(int color)
{
	fondo = color;
}

//##ModelId=3FB2143F01DD
void Video::ModificarVista(int vista_w, int vista_h)
{
	this->vista_w = vista_w;
	this->vista_h = vista_h;
	vista_x = (SCREEN_W - vista_w)/2;
	vista_y = (SCREEN_H - vista_h)/2;

	switch (modo_actual)
	{
	case doble_buffer:
		destroy_bitmap(sub_buffer);
		sub_buffer = create_sub_bitmap(buffer, vista_x, vista_y, vista_w, vista_h);
		destroy_bitmap(sub_screen);
		sub_screen = create_sub_bitmap(screen, vista_x, vista_y, vista_w, vista_h);
		break;
	case page_flipping:
		destroy_bitmap(sub_page[0]);
		destroy_bitmap(sub_page[1]);
		sub_page[0] = create_sub_bitmap(page[0], vista_x, vista_y, vista_w, vista_h);
		sub_page[1] = create_sub_bitmap(page[1], vista_x, vista_y, vista_w, vista_h);
		break;
	case triple_buffer:
		destroy_bitmap(sub_page[0]);
		destroy_bitmap(sub_page[1]);
		destroy_bitmap(sub_page[2]);
		sub_page[0] = create_sub_bitmap(page[0], vista_x, vista_y, vista_w, vista_h);
		sub_page[1] = create_sub_bitmap(page[1], vista_x, vista_y, vista_w, vista_h);
		sub_page[2] = create_sub_bitmap(page[2], vista_x, vista_y, vista_w, vista_h);
		break;
	case automatico:
		throw new ExcepcionTU("Error interno en Video::ModificarVista()", -1);
	}
}

//##ModelId=3FB2143F029A
void Video::RepintarFondo(BITMAP *bmp)
{
	acquire_bitmap(bmp);
	rectfill(bmp, 0, 0, SCREEN_W, vista_x, fondo);
	rectfill(bmp, 0, vista_y+vista_h, SCREEN_W, SCREEN_H, fondo);
	rectfill(bmp, 0, vista_y, vista_x, vista_y+vista_h, fondo);
	rectfill(bmp, vista_x+vista_w, vista_y, SCREEN_W, vista_y+vista_h, fondo);
	release_bitmap(bmp);
}

//##ModelId=3FB2143F01EF
void Video::RepintarFondo()
{
	switch(modo_actual)
	{
	case doble_buffer:
		RepintarFondo(screen);
		break;
	case page_flipping:
		RepintarFondo(page[0]);
		RepintarFondo(page[1]);
		break;
	case triple_buffer:
		RepintarFondo(page[0]);
		RepintarFondo(page[1]);
		RepintarFondo(page[2]);
		break;
	case automatico:
		throw new ExcepcionTU("Error interno en Video::RepintarFondo()", -1);
	}
}

//##ModelId=3FB2143F01E5
void Video::Rellenar(int color)
{
	switch(modo_actual)
	{
	case doble_buffer:
		acquire_bitmap(sub_screen);
		clear_to_color(sub_screen, color);
		release_bitmap(sub_screen);
		acquire_bitmap(sub_buffer);
		clear_to_color(sub_buffer, color);
		release_bitmap(sub_buffer);
		break;
	case page_flipping:
		acquire_bitmap(sub_page[0]);
		clear_to_color(sub_page[0], color);
		release_bitmap(sub_page[0]);
		acquire_bitmap(sub_page[1]);
		clear_to_color(sub_page[1], color);
		release_bitmap(sub_page[1]);
		break;
	case triple_buffer:
		acquire_bitmap(sub_page[0]);
		clear_to_color(sub_page[0], color);
		release_bitmap(sub_page[0]);
		acquire_bitmap(sub_page[1]);
		clear_to_color(sub_page[1], color);
		release_bitmap(sub_page[1]);
		acquire_bitmap(sub_page[2]);
		clear_to_color(sub_page[2], color);
		release_bitmap(sub_page[2]);
		break;
	case automatico:
		throw new ExcepcionTU("Error interno en Video::Rellenar()", -1);
	}
}

//##ModelId=3FB2143F01E7
void Video::Vaciar(void)
{
	switch(modo_actual)	
	{
	case doble_buffer:
		acquire_bitmap(sub_screen);
		clear_bitmap(sub_screen);
		release_bitmap(sub_screen);
		acquire_bitmap(sub_buffer);
		clear_bitmap(sub_buffer);
		release_bitmap(sub_buffer);
		break;
	case page_flipping:
		acquire_bitmap(sub_page[0]);
		clear_bitmap(sub_page[0]);
		release_bitmap(sub_page[0]);
		acquire_bitmap(sub_page[1]);
		clear_bitmap(sub_page[1]);
		release_bitmap(sub_page[1]);
		break;
	case triple_buffer:
		acquire_bitmap(sub_page[0]);
		clear_bitmap(sub_page[0]);
		release_bitmap(sub_page[0]);
		acquire_bitmap(sub_page[1]);
		clear_bitmap(sub_page[1]);
		release_bitmap(sub_page[1]);
		acquire_bitmap(sub_page[2]);
		clear_bitmap(sub_page[2]);
		release_bitmap(sub_page[2]);
		break;
	case automatico:
		throw new ExcepcionTU("Error interno en Video::Vaciar()", -1);
	}
}

//##ModelId=3FB2143F01F0
BITMAP* Video::ObtenerPantalla() const
{
	switch (modo_actual)
	{
	case doble_buffer:
		return sub_buffer;
		break;
	case page_flipping:
		return sub_page[1-pagina_actual];
		break;
	case triple_buffer:
		return sub_page[(pagina_actual+1)%3];
		break;
	default:
		throw new ExcepcionTU("Imposible obtener el bitmap a la pantalla", -1);
		break;
	}
}

//##ModelId=3FB2143F01F2
BITMAP* Video::ObtenerPaginaVisible() const
{
	switch (modo_actual)
	{
	case doble_buffer:
		return sub_screen;
		break;
	case page_flipping:
		return sub_page[pagina_actual];
		break;
	case triple_buffer:
		return sub_page[pagina_actual];
		break;
	default:
		throw new ExcepcionTU("Imposible obtener el bitmap a la pagina visible", -1);
		break;
	}
}

//##ModelId=3FB2143F01F9
void Video::IniciarSalida()
{
	acquire_bitmap(ObtenerPantalla());
	contador_iniciar_salida++;
}

//##ModelId=3FB2143F01FA
void Video::FinalizarSalida()
{
	if (contador_iniciar_salida > 0)
	{
		release_bitmap(ObtenerPantalla());
		contador_iniciar_salida--;
	}
}

//##ModelId=3FB2143F01FB
void Video::Actualizar()
{
	while (contador_iniciar_salida > 0)
	{
		FinalizarSalida();
	}

	switch (modo_actual)
	{
	case doble_buffer:
		acquire_screen();
		blit(sub_buffer, sub_screen, 0, 0, 0, 0, vista_w, vista_h);
		release_screen();
		break;
	case page_flipping:
		pagina_actual = 1-pagina_actual;
		show_video_bitmap(page[pagina_actual]);
		break;
	case triple_buffer:
		while(poll_scroll());
		pagina_actual = (pagina_actual+1)%3;
		request_video_bitmap(page[pagina_actual]);
		break;
	case automatico:
		throw new ExcepcionTU("Error interno en Video::Actualizar()", -1);
	}
}

//##ModelId=3FB2143F01FC
void Video::CopiarABuffers()
{
	switch(modo_actual)
	{
	case doble_buffer:
		break;
	case page_flipping:
		blit (sub_page[pagina_actual], sub_page[1-pagina_actual], 0, 0, 0, 0, vista_w, vista_h);
		break;
	case triple_buffer:
		blit (sub_page[pagina_actual], sub_page[(pagina_actual+1)%3], 0, 0, 0, 0, vista_w, vista_h);
		blit (sub_page[pagina_actual], sub_page[(pagina_actual+2)%3], 0, 0, 0, 0, vista_w, vista_h);
		break;
	case automatico:
		throw new ExcepcionTU("Error interno en Video::CopiarABuffers()", -1);
	}
}

//##ModelId=3FB2143F0203
void Video::CopiarFrameAnterior()
{
	switch(modo_actual)
	{
	case doble_buffer:
		break;
	case page_flipping:
		blit (sub_page[pagina_actual], sub_page[1-pagina_actual], 0, 0, 0, 0, vista_w, vista_h);
		break;
	case triple_buffer:
		blit (sub_page[pagina_actual], sub_page[(pagina_actual+1)%3], 0, 0, 0, 0, vista_w, vista_h);
		break;
	case automatico:
		throw new ExcepcionTU("Error interno en Video::CopiarFrameAnterior()", -1);
	}
}

//##ModelId=3FB2143F0204
BITMAP *Video::CrearBitmap(int w, int h, TipoBitmap tipo)
{
	BITMAP *ret = NULL;

	switch (tipo)
	{
	case hw_blit:
		if (hw_accel)
		{
			ret = create_video_bitmap(w, h);
		}
		break;
	case hw_trans_blit:
		if (hw_accel && (gfx_capabilities & GFX_HW_VRAM_BLIT_MASKED))
		{
			ret = create_video_bitmap(w, h);
		}
		break;
	case no_hw:
		break;
	}

	if (!ret)
	{
		ret = create_bitmap(w, h);
	}

	if (!ret)
	{
		throw new ExcepcionTU("Error al crear el bitmap", -1);
	}

	return ret;
}

//##ModelId=3FB2143F020E
void Video::DestruirBitmap(BITMAP *bmp)
{
	destroy_bitmap(bmp);
}

//##ModelId=3FB2143F0210
void Video::PonerBitmap(BITMAP *bmp)
{
}

//##ModelId=3FB2143F0217
BITMAP *Video::CargarPng(const string &file, TipoBitmap tipo)
{
	BITMAP *aux;
	BITMAP *ret = NULL;

	set_color_conversion(COLORCONV_NONE);
	aux = load_png(file.c_str(), NULL);
	if (!aux)
	{
		stringstream st;
		st << "Error al cargar el png " <<  file;
		throw new ExcepcionTU(st.str(), -1);
	}

	set_color_conversion(COLORCONV_TOTAL | COLORCONV_DITHER);
	if (bitmap_color_depth(aux) == 32)
	{
		ret = aux;
	}
	else
	{
		ret = CrearBitmap(aux->w, aux->h, tipo);

		if (ret)
		{
			blit (aux, ret, 0, 0, 0, 0, aux->w, aux->h);
		}
		destroy_bitmap(aux);
	}

	return ret;
}
