#include "map.h"
#include <stdio.h>
#include <stdlib.h>     /* srand, rand */
#include <assert.h>
#include <algorithm>
#include <allegro5/allegro_primitives.h>
#include "util.h"


#include <iostream>

//---------------- forward declarations ------------------------------
bool FSort(const BasicTile * a_pC1, const BasicTile * a_pC2);


//------------------- Basic Tile implementation ---------------------


//--------------------------------------------------------------------
// Constructor  : BasicTile.BasicTile
// Description  : Constructs a tile
//--------------------------------------------------------------------

BasicTile::BasicTile()
{
  m_nOffsetX        =  0;
  m_nOffsetY        =  0;
  m_nBitmapOffsetX  =  -1;
  m_nBitmapOffsetY  =  -1;
  m_pBitmap         =  NULL;
  m_blOccupied      = false;
  m_blPathMarker    = false;
  m_pParent         = NULL;
  m_dF              = 0;
  m_nDijkstra       = -1;
  m_pBitmap         = NULL;
  m_pEngine         = NULL;
  m_nX              = 0;
  m_nY              = 0;
  m_nTraverseCost   = 1;
  m_pOccupier       = NULL;
}


//--------------------------------------------------------------------
// Destructor   : BasicTile.~BasicTile
// Description  : Destroys a tile
//--------------------------------------------------------------------

BasicTile::~BasicTile()
{


}

//--------------------------------------------------------------------
// Function     : BasicTile.SetOffset
// Description  : Set's the grabbing offset for a tile
// Virtual      : No
//--------------------------------------------------------------------

void BasicTile::SetOffset(int a_nOffsetX,int a_nOffsetY)
{
   m_nBitmapOffsetX = a_nOffsetX;
   m_nBitmapOffsetY = a_nOffsetY;
}


//--------------------------------------------------------------------
// Function     : BasicTile.SetBitmap
// Description  : Set's the bitmap for a tile
// Virtual      : No
//--------------------------------------------------------------------

void BasicTile::SetBitmap(ALLEGRO_BITMAP  * a_pBitmap)
{
   m_pBitmap = a_pBitmap;
}


//--------------------------------------------------------------------
// Function     : BasicTile.Draw
// Description  : Draws a tile at position a_nScreenX, a_nScreenY
// Virtual      : Yes
//
// parameters
//--------------
// a_nScreenX   : tile X position on screen in pixels (LEFT)
// a_nScreenY   : tile Y position on screen in pixels (TOP)
//
//--------------------------------------------------------------------

void BasicTile::Draw(int a_nScreenX,int a_nScreenY)
{
  a_nScreenX+=m_nOffsetX;
  a_nScreenY+=m_nOffsetY;

  if (m_pBitmap == NULL)
  {
    // No bitmap, draw a placeholder
    ALLEGRO_COLOR  Fill;

    if (m_nTraverseCost < 0)
    {
        Fill   = al_map_rgb(255,0,0);
    }
    else if (m_blOccupied)
    {
        Fill   = al_map_rgb(100,100,255);
    }
    else
    {
        Fill   = al_map_rgb(0,0,0);
    }

    ALLEGRO_COLOR  Border = al_map_rgb(0,255,0);

    al_draw_filled_rectangle(a_nScreenX,a_nScreenY, a_nScreenX+m_pEngine->m_nTileWidth, a_nScreenY+m_pEngine->m_nTileHeight,Fill);
    al_draw_rectangle(a_nScreenX,a_nScreenY, a_nScreenX+m_pEngine->m_nTileWidth, a_nScreenY+m_pEngine->m_nTileHeight,Border,1);
  }
  else
  {
    if ((m_nBitmapOffsetX == -1)&&(m_nBitmapOffsetY == -1))
    {
      // single tile
      al_draw_bitmap (m_pBitmap, a_nScreenX, a_nScreenY,0);
    }
    else
    {
      // tilesheet
      al_draw_bitmap_region (m_pBitmap,m_nBitmapOffsetX, m_nBitmapOffsetY, m_pEngine->m_nTileWidth, m_pEngine->m_nTileHeight, a_nScreenX, a_nScreenY,0);
    }

  }

  if (m_blPathMarker)
  {
      al_draw_circle(a_nScreenX+m_pEngine->m_nTileWidth/2, a_nScreenY+m_pEngine->m_nTileHeight/2,m_pEngine->m_nTileHeight/4,al_map_rgb(255,255,255),1);
  }
}


//--------------------------------------------------------------------
// Function     : BasicTile.GetX
// Description  : Gets the X coordinate of the tile in Cell positions
// Virtual      : no
//--------------------------------------------------------------------

int BasicTile::GetX()
{
    return m_nX;
}

//--------------------------------------------------------------------
// Function     : BasicTile.GetY
// Description  : Gets the Y coordinate of the tile in Cell positions
// Virtual      : no
//--------------------------------------------------------------------

int BasicTile::GetY()
{
    return m_nY;
}


//--------------------------------------------------------------------
// Function     : BasicTile.ResetPathMarker
// Description  : Resets the circular markers indicating this tile
//                is part of a map
// Virtual      : no
//--------------------------------------------------------------------

void BasicTile::ResetPathMarker()
{
    m_blPathMarker = false;
}

//--------------------------------------------------------------------
// Function     : BasicTile.SetPathMarker
// Description  : Sets the circular markers indicating this tile
//                is part of a map
// Virtual      : no
//--------------------------------------------------------------------

void BasicTile::SetPathMarker()
{
    m_blPathMarker = true;
}


//--------------------------------------------------------------------
// Function     : BasicTile.IsOccupied
// Description  : Tells if a tile is occupied by an entity (Sprite)
//                other than a_pOccupier
// Virtual      : no
// Returns      : true if tile is occupied by another entity than
//                a_pOccupier
//
// parameters
//--------------
// a_pOccupier  : pointer to entity to be ignored
//                this might already be occupying this tile,
//                but it is not another entity
//
//--------------------------------------------------------------------

bool BasicTile::IsOccupied(CSprite * a_pOccupier)
{
    if ((m_pOccupier == NULL)||(m_pOccupier == a_pOccupier))
    {
        return false;
    }
    else
    {
        return true;
    }
}


//--------------------------------------------------------------------
// Function     : BasicTile.GetOccupier
// Description  : Gets the entity occupying this tile
// Virtual      : no
// Returns      : a pointer to the occupying entity
//--------------------------------------------------------------------

CSprite * BasicTile::GetOccupier()
{
    return m_pOccupier;
}

//--------------------------------------------------------------------
// Function     : BasicTile.SetOccupied
// Description  : Sets the entity occupying this tile
// Virtual      : no
//
// parameters
//--------------
// a_pOccupier  : a pointer to the entity claiming occupation
//
//--------------------------------------------------------------------

void BasicTile::SetOccupied(CSprite * a_pOccupier)
{
    if ((m_pOccupier== NULL)||(m_pOccupier == a_pOccupier))
    {
        m_blOccupied = true;
        m_pOccupier = a_pOccupier;
    }
}

//--------------------------------------------------------------------
// Function     : BasicTile.ResetOccupied
// Description  : Resets the occupied state of this tile
// Virtual      : no
//
// parameters
//--------------
// a_pOccupier  : a pointer to the entity that was occupying this tile
//
//--------------------------------------------------------------------

void BasicTile::ResetOccupied(CSprite * a_pOccupier)
{
    if (a_pOccupier == m_pOccupier)
    {
        m_blOccupied = false;
        m_pOccupier = NULL;
    }
}


//--------------------------------------------------------------------
// Function     : BasicTile.IsAdjectend
// Description  : Checks if a tile is adjectend to this tile
// Virtual      : no
// Returns      : true if tiles are adjectend
//
// parameters
//--------------
// a_pTile      : a pointer to a tile to check
//
//--------------------------------------------------------------------

bool BasicTile::IsAdjectend(BasicTile * a_pTile)
{
    if ((abs(m_nX - a_pTile->m_nX)<=1) && (abs(m_nY - a_pTile->m_nY)<=1))
    {
        return true;
    }
    else
    {
        return false;
    }
}



//---------------- Basic Engine implementation -----------------


//--------------------------------------------------------------
// Constructor  : BasicEngine::BasicEngine
// Description  : Constructs a tile engine
//-------------------------------------------------------------
BasicEngine::BasicEngine()
{
  m_nTileWidth=32;
  m_nTileHeight=32;
  m_nYStep = m_nTileHeight;
  SCREENCENTER_X = 400;
  SCREENCENTER_Y = 300;

  m_pMutex = al_create_mutex();
  m_AstarThread = al_create_thread(Func_Thread, this);
  al_start_thread(m_AstarThread);
}


//--------------------------------------------------------------
// Destructor   : BasicEngine::~BasicEngine
// Description  : frees all used memory
//-------------------------------------------------------------
BasicEngine::~BasicEngine()
{
  al_destroy_thread(m_AstarThread);
  al_destroy_mutex(m_pMutex);

  std::vector<std::vector< BasicTile * > >::iterator p=m_lstCell.begin();
  while (p!=m_lstCell.end())
  {
    std::vector< BasicTile * >::iterator q = (*p).begin();
    while (q!=(*p).end())
    {
      BasicTile * hulp = *q;
      (*p).erase(q++);
      delete hulp;
    }
    m_lstCell.erase(p++);
  }
}


//--------------------------------------------------------------------
// Function     : BasicEngine.GetTileWidth
// Description  : Gets the width of the tiles
// Virtual      : no
// Returns      : the tile width in pixels
//--------------------------------------------------------------------

int BasicEngine::GetTileWidth()
{
    return m_nTileWidth;
}

//--------------------------------------------------------------------
// Function     : BasicEngine.GetTileHeight
// Description  : Gets the height of the tiles
// Virtual      : no
// Returns      : the tile height in pixels
//--------------------------------------------------------------------

int BasicEngine::GetTileHeight()
{
    return m_nTileHeight;
}


//--------------------------------------------------------------------
// Function     : BasicEngine.GetTileAt
// Description  : Gets the tile at map position
// Virtual      : no
// Returns      : a pointer to the tile
//
// parameters
//--------------
// a_nX         : X map position
// a_nY         : Y map position
//--------------------------------------------------------------------

BasicTile * BasicEngine::GetTileAt(int a_nx,int a_nY)
{
    return m_lstCell[a_nx][a_nY];
}


//--------------------------------------------------------------
// Function     : BasicEngine.Draw
// Description  : Draws a tile map centered around a_nCamX,a_nCamY
// Virtual      : Yes
//
// parameters
//--------------
// a_nCamX         : X Camera position
// a_nCamY         : Y Camera position
//-------------------------------------------------------------
void BasicEngine::Draw(int a_nCamX,int a_nCamY)
{
  int DrawX;
  int DrawY;
  unsigned int startx,endx,starty,endy;

   a_nCamX -= SCREENCENTER_X;
   a_nCamY -= SCREENCENTER_Y;

  //Calculate start and end for nested loops
  if ((a_nCamX-SCREENCENTER_X)<=0)
  {
      startx = 0;
  }
  else
  {
     startx=(a_nCamX-SCREENCENTER_X)/m_nTileWidth;
     if (startx>m_lstCell.size()) startx=m_lstCell.size();
  }

  endx=((a_nCamX+SCREENCENTER_X)/m_nTileWidth)+1;
  if (endx>m_lstCell.size()) endx=m_lstCell.size();


  if ((a_nCamY-SCREENCENTER_Y)<=0)
  {
      starty = 0;
  }
  else
  {
     starty=((a_nCamY-SCREENCENTER_Y)/m_nYStep);
     if (starty>m_lstCell.size()) starty=m_lstCell.size();
  }

  endy=((a_nCamY+SCREENCENTER_Y)/m_nYStep)+1;
  if (endy>m_lstCell.size()) endy=m_lstCell.size();

  int sy=(starty*m_nYStep);
  int h_sx=startx*m_nTileWidth;
  int sx;

  for (unsigned int y=starty;y<endy;y++)
    {
    sx=h_sx;
    for (unsigned int x=startx;x<endx;x++)
        {
          DrawX=int(SCREENCENTER_X+(sx-a_nCamX));
          DrawY=int(SCREENCENTER_Y+(sy-a_nCamY));
          sx+=32;
          if ((x>=0) && (x<(unsigned int)m_nMapWidth) && (y >=0) & (y < (unsigned int)m_nMapHeight))
          {
              m_lstCell[x][y]->Draw(DrawX,DrawY);
          }
          else
          {


          }
        }
    sy+=m_nYStep;
    }

}


//--------------------------------------------------------------------
// Function     : BasicEngine.TileToEngineX
// Description  : Gets the corresponding engine position from a tile
// Virtual      : no
// Returns      : the engine X position in pixels
//
// parameters
//--------------
// a_nTileX         : tile X position
//--------------------------------------------------------------------

double BasicEngine::TileToEngineX(int a_nTileX)
{
    return a_nTileX * m_nTileWidth + m_nTileWidth / 2;
}


//--------------------------------------------------------------------
// Function     : BasicEngine.TileToEngineY
// Description  : Gets the corresponding engine position from a tile
// Virtual      : no
// Returns      : the engine Y position in pixels
//
// parameters
//--------------
// a_nTileY         : tile Y position
//--------------------------------------------------------------------

double BasicEngine::TileToEngineY(int a_nTileY)
{
    return a_nTileY * m_nTileHeight + m_nTileHeight / 2;
}


//--------------------------------------------------------------------
// Function     : BasicEngine.TileToEngineY
// Description  : Gets the corresponding engine position from a tile
// Virtual      : no
// Returns      : the engine position in pixels
//
// parameters
//--------------
// a_vecTilePosition : tile position
//--------------------------------------------------------------------

Vector2D BasicEngine::TileToEngine(Vector2D a_vecTilePosition)
{
    a_vecTilePosition.SetX(a_vecTilePosition.GetX() * m_nTileWidth + m_nTileWidth/2);
    a_vecTilePosition.SetY(a_vecTilePosition.GetY() * m_nTileHeight + m_nTileHeight/2);
    return a_vecTilePosition;
}



Vector2D BasicEngine::EngineToTile(Vector2D a_vecEnginePosition)
{
    int x = a_vecEnginePosition.GetX();
    int y = a_vecEnginePosition.GetY();

    x = x / m_nTileWidth;
    y = y / m_nTileHeight;

    Vector2D result(x,y);

    return result;

}

//--------------------------------------------------------------------
// Function     : BasicEngine.GetMapWidth
// Description  : Gets the width of the map
// Virtual      : no
// Returns      : the width of the map in tiles
//--------------------------------------------------------------------
int BasicEngine::GetMapWidth()
{
    return m_nMapWidth;
}

//--------------------------------------------------------------------
// Function     : BasicEngine.GetMapHeight
// Description  : Gets the height of the map
// Virtual      : no
// Returns      : the height of the map in tiles
//--------------------------------------------------------------------

int BasicEngine::GetMapHeight()
{
    return m_nMapHeight;
}

//-----------------------------------------------------------------------------
// Function     : BasicEngine.GenerateTest
// Description  : Generates a testmap with Width a_nWidth and height a_nHeight
// Virtual      : Yes
//
// parameters
//--------------
// a_nWidth     : the width of the map in tiles
// a_nHeight    : the height of the map in tiles
// a_pBitmap    : a tilesheet
//-----------------------------------------------------------------------------
void BasicEngine::GenerateTest(int a_nWidth, int a_nHeight, ALLEGRO_BITMAP* a_pBitmap)
{
  m_nMapWidth = a_nWidth;
  m_nMapHeight = a_nHeight;

  for (int y=0; y<m_nMapHeight;y++)
  {
    std::vector<BasicTile * > tempvector;
    for (int x=0;x< m_nMapWidth;x++)
    {
      BasicTile * pTile = new BasicTile();
      pTile->m_nX = y;
      pTile->m_nY = x;
      if (a_pBitmap!=NULL)
      {
        pTile->SetBitmap(a_pBitmap);
        if((x==0)||(y==0))
        {
          pTile->SetOffset(0,320);

        }
        else
        {
          pTile->SetOffset(0,256);
        }
      }
      pTile->m_pEngine = this;
      tempvector.push_back(pTile);
    }
    m_lstCell.push_back(tempvector);
  }

  for (int y=0; y < a_nHeight; y++)
  {
       for (int x=0; x < a_nWidth; x++)
       {
          if ((x == 0) || (x == a_nWidth-1)|| (y == 0)|| (y == a_nHeight-1)||
              ((x == a_nWidth/2) && ((y > 2 ) && (y < a_nHeight - 2) ))||
               ((y == a_nHeight/2) && ((x < (a_nWidth/2) -1) || (x > (a_nWidth/2) +1))) )
          {
             m_lstCell[x][y]->m_nTraverseCost = -1;
          }

       }
  }
}


//-----------------------------------------------------------------------------
// Function     : BasicEngine.FindPath
// Description  : tries to find a path between Start and End
// Virtual      : No
// Returns      : the shortest path between start and end, or an empty list
//                if no path was found returns a list of pointers to the tiles
//
// parameters
//--------------
// a_nStartX    : The start position X
// a_nStartY    : The start position Y
// a_nEndX      : the goal X
// a_nEndY      : the goal Y
//-----------------------------------------------------------------------------
std::vector<BasicTile *> BasicEngine::FindPath(int a_nStartX, int a_nStartY,int a_nEndX, int a_nEndY)
{
   ClearAstar();
   bool blPathFound = false;
   std::vector<BasicTile *> ResultPath;
   ClearAstar();                                                     // clear astar values
   BasicTile * pEndPath = NULL;                                      // pointer to end of path
   BasicTile * pCurrent = m_lstCell[a_nStartX][a_nStartY];           // set current pointer to start of path
   BasicTile * pStartPath = pCurrent;                                // pointer to start of path
   pCurrent->m_dF = 0;                                               // F cost of start tile is 0
   m_lstOpenList.push_back(pCurrent);                                // push to open list

   while ( (!m_lstOpenList.empty()) && (!blPathFound))               // while open list not empty and no path is found
   {
       pCurrent = m_lstOpenList.front();                             // get the first/next node
       m_lstOpenList.pop_front();                                    // remove first node from list
       m_lstClosedList.push_back(pCurrent);                          // push note to closed list

       std::vector<BasicTile *> vSuccessors;                         // get the possible successors
       vSuccessors = GetAdjectend(pCurrent->m_nX, pCurrent->m_nY);

       for (size_t i=0; i < vSuccessors.size(); i++)                 // loop through each successor
       {
            BasicTile * pSuc = vSuccessors[i];
            if ((pSuc->m_nX == a_nEndX) && (pSuc->m_nY == a_nEndY) && (!blPathFound))  // path found, terminate !
            {
                pSuc->m_pParent = pCurrent;
                pEndPath = pSuc;
                blPathFound = true;
            }
            else if (! IsInClosed(pSuc))                             // ignore if in closed
            {
                if (IsInOpen(pSuc))                                  // if in open list, see what path is shortest
                {
                    double H = abs(a_nEndX - pSuc->m_nX) + abs(a_nEndY - pSuc->m_nY);
                    double dCurF;
                    if (IsDiagonal(pCurrent, pSuc))
                    {
                        dCurF = pCurrent->m_dF + (((double)pSuc->m_nTraverseCost) * 1.8) + H;
                    }
                    else
                    {
                        dCurF = pCurrent->m_dF + pSuc->m_nTraverseCost + H;
                    }

                    if (dCurF < pSuc->m_dF)                            // shorter path, update F and Parent
                    {
                        pSuc->m_dF = dCurF;
                        pSuc->m_pParent = pCurrent;                    // set successor parent to current node
                        m_lstOpenList.sort(FSort);
                    }

                }
                else  // if not in open list, calculate F and add it to the open list
                {
                    double H = abs(a_nEndX - pSuc->m_nX) + abs(a_nEndY - pSuc->m_nY);
                    if (IsDiagonal(pCurrent, pSuc))
                    {
                        pSuc->m_dF = pCurrent->m_dF + (((double)pSuc->m_nTraverseCost) * 1.8) + H;
                    }
                    else
                    {
                         pSuc->m_dF = pCurrent->m_dF + pSuc->m_nTraverseCost + H;
                    }

                    pSuc->m_pParent = pCurrent;      // set successor parent to current node
                    m_lstOpenList.push_back(pSuc);
                    m_lstOpenList.sort(FSort);
                }
            }
        }
   }

   if (blPathFound)  // build the path
   {
       BasicTile * pTile = pEndPath;
       while ((pTile != pStartPath)&&(pTile != NULL))
       {
           ResultPath.push_back(pTile);
           pTile = pTile->m_pParent;
       }
       ResultPath.push_back(pTile);
   }
   std::reverse(ResultPath.begin(),ResultPath.end());
   return ResultPath;
}


//-----------------------------------------------------------------------------
// Function     : BasicEngine.IsInOpen
// Description  : checks if tile is on OPEN list
// Virtual      : No
// Returns      : true if tile is in OPEN list
//
// parameters
//--------------
// a_pTile      : Pointer to the tile to check
//-----------------------------------------------------------------------------

bool BasicEngine::IsInOpen(BasicTile * a_pTile)
{
    bool blFound = false;
    std::list<BasicTile *>::const_iterator p = m_lstOpenList.begin();

    while ((!blFound) && (p != m_lstOpenList.end()))
    {
        if ((*p) == a_pTile)
        {
            blFound = true;
        }
        p++;
    }

    return blFound;
}

//-----------------------------------------------------------------------------
// Function     : BasicEngine.IsInClosed
// Description  : checks if tile is on CLOSED list
// Virtual      : No
// Returns      : true if tile is in CLOSED list
//
// parameters
//--------------
// a_pTile      : Pointer to the tile to check
//-----------------------------------------------------------------------------

bool BasicEngine::IsInClosed(BasicTile * a_pTile)
{
    bool blFound = false;
    std::list<BasicTile *>::const_iterator p = m_lstClosedList.begin();

    while ((!blFound) && (p != m_lstClosedList.end()))
    {
        if ((*p) == a_pTile)
        {
            blFound = true;
        }
        p++;
    }

    return blFound;
}


//-----------------------------------------------------------------------------
// Function     : BasicEngine.ClearDijkstra
// Description  : clears the Dijkstra calculations for every tile
// Virtual      : No
//-----------------------------------------------------------------------------

void BasicEngine::ClearAstar()
{
   m_lstOpenList.erase(m_lstOpenList.begin(),m_lstOpenList.end());
   m_lstClosedList.erase(m_lstClosedList.begin(),m_lstClosedList.end());
   for (int y = 0; y < m_nMapHeight; y++)
    {
        for (int x = 0; x < m_nMapWidth; x++)
        {
            m_lstCell[x][y]->m_pParent  = NULL;
            m_lstCell[x][y]->m_dF       = 0.0;
        }
    }
}




//-----------------------------------------------------------------------------
// Function     : BasicEngine.GenerateDijkstra
// Description  : generates Dijkstra calculations for every tile with
//                "goal" as target
// Virtual      : No
//
// parameters
//--------------
// a_nGoalX    : X Tile coordinate of goal
// a_nGoalY    : Y Tile coordinate of tile
//-----------------------------------------------------------------------------

void BasicEngine::GenerateDijkstra(int a_nGoalX, int a_nGoalY)
{
    ClearDijkstra();
    std::list<BasicTile *> ToBeVisited;

    BasicTile * pCurrent = m_lstCell[a_nGoalX][a_nGoalY];
    ToBeVisited.push_back(pCurrent);
    pCurrent->m_nDijkstra = 0;

    while (!ToBeVisited.empty())
    {
       pCurrent = ToBeVisited.front();
       ToBeVisited.pop_front();

       std::vector<BasicTile *> vSuccessors;
       vSuccessors = GetAdjectend(pCurrent->m_nX, pCurrent->m_nY);
       for (size_t i=0; i < vSuccessors.size(); i++)
       {
           if (vSuccessors[i]->m_nDijkstra < 0)
           {
               vSuccessors[i]->m_nDijkstra = pCurrent->m_nDijkstra + vSuccessors[i]->m_nTraverseCost;
               BasicTile * pSuc = vSuccessors[i];
               ToBeVisited.push_back(pSuc);
           }

       }

    }
}


//-----------------------------------------------------------------------------
// Function     : BasicEngine.ClearDijkstra
// Description  : clears the Dijkstra calculations for every tile
// Virtual      : No
//-----------------------------------------------------------------------------

void BasicEngine::ClearDijkstra()
{
    for (int y = 0; y < m_nMapHeight; y++)
    {
        for (int x = 0; x < m_nMapWidth; x++)
        {
            if (m_lstCell[x][y]->m_nTraverseCost < 0)
            {
               m_lstCell[x][y]->m_nDijkstra = INT_MAX;
            }
            else
            {
               m_lstCell[x][y]->m_nDijkstra = -1;
            }

        }
    }
}


//-----------------------------------------------------------------------------
// Function     : BasicEngine.GetAdjectend
// Description  : Gets a list of adjectend tiles to the given position
// Virtual      : No
// Returns      : a list of pointers to the tiles adjectend to the given position
//
// parameters
//--------------
// a_nX            : X tile position
// a_nY            : Y tile position
// blAllowDiagonal : allow diagonal adjectend tiles
//-----------------------------------------------------------------------------

std::vector<BasicTile *> BasicEngine::GetAdjectend(int a_nX, int a_nY, bool blAllowDiagonal)
{
    std::vector<BasicTile *> Result;
    for (int y = a_nY -1; y <= a_nY +1; y++)
    {
        for (int x = a_nX -1; x <= a_nX +1; x++)
        {
             if ((x >= 0) && (x < m_nMapWidth) && (y >= 0) && (y < m_nMapHeight) && ((x != a_nX) || (y != a_nY )))
             {
                 if ((x == a_nX) || (y == a_nY))
                 {
                     if ((m_lstCell[x][y]->m_nTraverseCost > 0) && (!m_lstCell[x][y]->m_blOccupied))
                     {
                        Result.push_back(m_lstCell[x][y]);
                     }
                 }
                 else if (blAllowDiagonal)
                 {
                     if ((m_lstCell[x][y]->m_nTraverseCost > 0) && (!m_lstCell[x][y]->m_blOccupied))
                     {
                        if ((m_lstCell[x][a_nY]->m_nTraverseCost > 0) && (! m_lstCell[x][a_nY]->m_blOccupied) &&
                            (m_lstCell[a_nX][y]->m_nTraverseCost > 0) && (! m_lstCell[a_nX][y]->m_blOccupied))
                        {
                            Result.push_back(m_lstCell[x][y]);
                        }
                     }
                 }
             }
        }
    }
    return Result;
}



//-----------------------------------------------------------------------------
// Function     : BasicEngine.IsDiagonal
// Description  : checks if tiles are diagonally connected
// Virtual      : No
// Returns      : true if diagonal
//
// parameters
//--------------
// a_pTile1     : Tile 1 to check
// a_pTile2     : Tile 2 to check
//-----------------------------------------------------------------------------

bool BasicEngine::IsDiagonal(BasicTile * a_pTile1, BasicTile * a_pTile2)
{
    return ((a_pTile1->m_nX != a_pTile2->m_nX) && (a_pTile1->m_nY != a_pTile2->m_nY));
}


//-----------------------------------------------------------------------------
// Function     : BasicEngine.LineOfSight
// Description  : checks if there is a free line of sight
// Virtual      : No
// Returns      : true if there is a free line of sight
//
// parameters
//--------------
// a_nX1        : X coord of startposition
// a_nY1        : Y coord of start position
// a_nX2        : X coord of end position
// a_nY2        : Y coord of end position
//-----------------------------------------------------------------------------

bool BasicEngine::LineOfSight(int a_nX1, int a_nY1, int a_nX2, int a_nY2)
{
  int i,dx,dy,sdx,sdy,dxabs,dyabs,x,y,px,py;

  dx=a_nX2-a_nX1;      /* the horizontal distance of the line */
  dy=a_nY2-a_nY1;      /* the vertical distance of the line */
  dxabs=abs(dx);
  dyabs=abs(dy);
  sdx=sgn(dx);
  sdy=sgn(dy);
  x=dyabs>>1;
  y=dxabs>>1;
  px=a_nX1;
  py=a_nY1;
  bool blBlocked = false;

  if (dxabs >= dyabs) /* the line is more horizontal than vertical */
  {
    i = 0;
    while ((i<dxabs) && (!blBlocked))
    {
      y+=dyabs;
      if (y>=dxabs)
      {
        y-=dxabs;
        py+=sdy;
        if ((m_lstCell[px][py]->m_nTraverseCost <= 0) || ((m_lstCell[px][py]->m_blOccupied) && (px != a_nX1) && (py != a_nY1) && (px != a_nX2) && (py != a_nY2)) ||
            (m_lstCell[px+sdx][py-sdy]->m_nTraverseCost <= 0) || ((m_lstCell[px+sdx][py-sdy]->m_blOccupied) && (px+sdx != a_nX1) && (py-sdy != a_nY1) && (px+sdx != a_nX2) && (py-sdy != a_nY2)))
        {
          blBlocked = true;
        }
      }
      px+=sdx;
      if ((m_lstCell[px][py]->m_nTraverseCost <= 0) || ((m_lstCell[px][py]->m_blOccupied) && (px != a_nX1) && (py != a_nY1) && (px != a_nX2) && (py != a_nY2)))
      {
          blBlocked = true;
      }
      i++;
    }
  }
  else /* the line is more vertical than horizontal */
  {
    i = 0;
    while ((i < dyabs) && (!blBlocked))
    {
      x+=dxabs;
      if (x>=dyabs)
      {
        x-=dyabs;
        px+=sdx;
        if ((m_lstCell[px][py]->m_nTraverseCost <= 0) || ((m_lstCell[px][py]->m_blOccupied) && (px != a_nX1) && (py != a_nY1) && (px != a_nX2) && (py != a_nY2)) ||
            (m_lstCell[px-sdx][py+sdy]->m_nTraverseCost <= 0) || ((m_lstCell[px-sdx][py+sdy]->m_blOccupied) && (px-dx != a_nX1) && (py+sdy != a_nY1) && (px-sdx != a_nX2) && (py+sdy != a_nY2)))
        {
          blBlocked = true;
        }
      }
      py+=sdy;
      if ((m_lstCell[px][py]->m_nTraverseCost <= 0) || ((m_lstCell[px][py]->m_blOccupied) && (px != a_nX1) && (py != a_nY1) && (px != a_nX2) && (py != a_nY2)))
      {
          blBlocked = true;
      }
      i++;
    }
  }

  return !blBlocked;
}



bool BasicEngine::EnqueueAstar(int a_nStartX, int a_nStartY,int a_nGoalX, int a_nGoalY, CSprite * a_pCaller)
{
     AstarTask task;
     task.startX = a_nStartX;
     task.startY = a_nStartY;
     task.goalX  = a_nGoalX;
     task.goalY  = a_nGoalY;
     task.pCaller = a_pCaller;

     al_lock_mutex(m_pMutex);
        m_queueAstar.push(task);
     al_unlock_mutex(m_pMutex);
     return true;
}

void BasicEngine::ResetPathMarkers()
{
    for (int y = 0; y < m_nMapHeight; y++)
    {
        for (int x = 0; x < m_nMapWidth; x++)
        {
            m_lstCell[x][y]->ResetPathMarker();
        }
    }
}



bool FSort(const BasicTile * a_pTile1, const BasicTile * a_pTile2)
{
     return ((a_pTile1->m_dF) < (a_pTile2->m_dF));
}



void *  BasicEngine::Func_Thread(ALLEGRO_THREAD *thr, void *arg)
{
   BasicEngine * pEngine  = (BasicEngine *) arg;


   while (!al_get_thread_should_stop(thr))
   {
       std::vector<BasicTile *> result;

       if (!pEngine->m_queueAstar.empty())
       {
           AstarTask task;
           al_lock_mutex(pEngine->m_pMutex);
             task = pEngine->m_queueAstar.front();
             pEngine->m_queueAstar.pop();
           al_unlock_mutex(pEngine->m_pMutex);

           if (!task.pCaller->IsRemovePending())
           {
               result = pEngine->FindPath(task.startX,task.startY,task.goalX,task.goalY);

               if (!result.empty())
               {
                  al_lock_mutex(task.pCaller->m_pMutex);
                  task.pCaller->m_vPath = result;
                  task.pCaller->m_blHasPath = true;
                  al_unlock_mutex(task.pCaller->m_pMutex);
               }
               else
               {
                  task.pCaller->m_vPath.clear();
                  task.pCaller->m_blHasPath = false;
                  task.pCaller->OnCannotReach();
               }
           }
       }
       else
       {

       }


   }
   return NULL;
}


