




#include "allegro5/allegro.h"
#include "allegro5/allegro_font.h"
#include "allegro5/allegro_ttf.h"
#include "allegro5/allegro_primitives.h"

#include <stdio.h>


void AssertHandler(const char* expr , const char* file , int line , const char* func) {
    int i = 0;
    
    printf("%s failed in file %s on line %d in function %s\n" , expr , file , line , func);
    
    i /= 0;
    return;
}



void CreateUstrFromCodePoints(unsigned int* cbuf , int size , ALLEGRO_USTR** pstr) {
    if (!cbuf)    {return;}
    if (size < 0) {return;}
    if (!pstr)    {return;}
    ALLEGRO_USTR* ustr = *pstr;
    if (ustr) {al_ustr_free(ustr);}
    *pstr = al_ustr_new("");
    ustr = *pstr;
    for (int i = 0 ; i < size ; ++i) {
        al_ustr_append_chr(ustr , cbuf[i]);
    }
    return;
}



int main(int argc , char** argv) {

    /// Declarations
    
    int ww = 800;
    int wh = 600;
    
    bool redraw = true;
    bool quit = false;
    
    bool caret_blink = false;
    float caret_duration = 0.5f;
    float elapsed_time = 0.0f;
    
    /// For holding the current string. Just allocate more than we'll need instead of messing around with dynamic arrays
    const int MAXNCHARS = 24;
    unsigned int intcharbuf[MAXNCHARS + 1];
    ALLEGRO_USTR* ustr = al_ustr_new("");

    int caret = 0;
    int nchars = 0;
    
    /// For holding stored strings
    const int MAXNLINES = 30;
    ALLEGRO_USTR* ustrs[MAXNLINES];
    int nlines = 0;

    bool insert_mode = true;
    
    ALLEGRO_DISPLAY* d = 0;
    ALLEGRO_EVENT_QUEUE* q = 0;
    ALLEGRO_TIMER* t = 0;
    ALLEGRO_FONT* f = 0;
    
    const int tx = ww/2;
    const int ty = 64;
    
    /// Code

    (void)argc;
    (void)argv;
    
    memset(intcharbuf , 0 , (MAXNCHARS + 1)*sizeof(unsigned int));
    memset(ustrs , 0 , MAXNLINES*sizeof(ALLEGRO_USTR*));
    
    if (!al_init()) {return 1;}
    if (!al_init_font_addon()) {return 2;}
    if (!al_init_ttf_addon()) {return 3;}
    if (!al_init_primitives_addon()) {return 4;}
    
    if (!al_install_keyboard()) {return 10;}
    if (!al_install_mouse()) {return 11;}
    
    al_register_assert_handler(AssertHandler);
    
    al_set_new_display_flags(ALLEGRO_OPENGL | ALLEGRO_WINDOWED);
    
    d = al_create_display(ww,wh);
    q = al_create_event_queue();
    t = al_create_timer(1.0 / 60.0);
    f = al_load_ttf_font("Consola.ttf" , -24 , 0);
    
    if (!d || !q || !t || !f) {return -1;}
    
    
    al_register_event_source(q , al_get_keyboard_event_source());
    al_register_event_source(q , al_get_mouse_event_source());
    al_register_event_source(q , al_get_display_event_source(d));
    al_register_event_source(q , al_get_timer_event_source(t));

    al_start_timer(t);
    
    while(!quit)
    {
        if(redraw)
        {
            al_clear_to_color(al_map_rgb(55,111,222));
            
            /// Draw text label
            al_draw_text(f , al_map_rgb(0,0,64) , tx , ty - al_get_font_line_height(f) - 5 , ALLEGRO_ALIGN_LEFT , "Enter text here :");
            
            /// Draw our text box and text
            {
                int tbx,tby,tbw,tbh;
                al_get_ustr_dimensions(f , ustr , &tbx , &tby , &tbw , &tbh);
                int x = tx + tbx;
                int y = ty + tby;
                al_draw_filled_rectangle(x , y , x + tbw , y + tbh , al_map_rgb(0,0,0));
                al_draw_rectangle(x - 5 , y - 5 , x + tbw + 5 , y + tbh + 5 , al_map_rgb(255,255,255) , 5.0f);
            }
            al_draw_ustr(f , al_map_rgb(255,255,255) , tx , ty , ALLEGRO_ALIGN_LEFT , ustr);
            
            /// Draw our caret
            if (!caret_blink) {
                ALLEGRO_USTR* sub = al_ustr_dup_substr(ustr , 0 , al_ustr_offset(ustr , caret));
                
                int cx = al_get_ustr_width(f , sub);
                
                al_draw_text(f , al_map_rgb(0,255,0) , tx + cx - al_get_text_width(f , "|")/2 , ty , 0 , "|");
                
                al_ustr_free(sub);
            }

            /// Draw our stored strings
            {
                int lh = 1.5*al_get_font_ascent(f);

                /// Column 1 lower left
                int y = wh - (MAXNLINES/2 + 1)*lh;
                int x = ww/3;
                int align = ALLEGRO_ALIGN_CENTER;
                for (int i = 0 ; i < MAXNLINES/2 && i < nlines ; ++i) {
                    al_draw_ustr(f , al_map_rgb(0,255,255) , x , y , align , ustrs[i]);
                    y += lh;
                }
                
                /// Column 2 lower right
                x = 2*ww/3;
                y = wh - (MAXNLINES/2 + 1)*lh;
                for (int i = MAXNLINES/2 ; i < nlines ; ++i) {
                    al_draw_ustr(f , al_map_rgb(0,255,255) , x , y , align , ustrs[i]);
                    y += lh;
                }
            }
            /// Draw some debug info
            al_draw_textf(f , al_map_rgb(255,127,0) , 10 , 10 , ALLEGRO_ALIGN_LEFT , "MAXNCHARS = %d" , MAXNCHARS);
            al_draw_textf(f , al_map_rgb(255,127,0) , 10 , 30 , ALLEGRO_ALIGN_LEFT , "nchars    = %d" , nchars);
            al_draw_textf(f , al_map_rgb(255,127,0) , 10 , 50 , ALLEGRO_ALIGN_LEFT , "nlines    = %d" , nlines);
            al_draw_textf(f , al_map_rgb(255,127,0) , 10 , 70 , ALLEGRO_ALIGN_LEFT , "caret     = %d" , caret);
            al_draw_textf(f , al_map_rgb(255,127,0) , 10 , 90 , ALLEGRO_ALIGN_LEFT , "Insert    = %s" , (insert_mode?"On":"Off"));
            
            al_flip_display();
            redraw = false;
        }
        do
        {
            ALLEGRO_EVENT ev;
            al_wait_for_event(q, &ev);

            if (ev.type == ALLEGRO_EVENT_DISPLAY_CLOSE) {
                quit = true;
            }
            if (ev.type == ALLEGRO_EVENT_KEY_DOWN && ev.keyboard.keycode == ALLEGRO_KEY_ESCAPE) {
                quit = true;
            }
            if(ev.type == ALLEGRO_EVENT_TIMER)
            {
                elapsed_time += al_get_timer_speed(t);
                if (elapsed_time >= caret_duration) {
                    caret_blink = !caret_blink;
                    elapsed_time = 0;
                }
                redraw = true;
            }
            if (ev.type == ALLEGRO_EVENT_MOUSE_BUTTON_DOWN && ev.mouse.button == 1) {
                /// Find new caret position if mouse is over string
                int uw = al_get_ustr_width(f , ustr);
                int mx = ev.mouse.x;
                int y = ev.mouse.y - ty;
                if (y >= 0 && y <= al_get_font_line_height(f)) {
                    int x = mx - tx;
                    if (x >= 0 && x <= uw) {
                        float cw = (float)uw/nchars;
                        caret = (int)(x / cw);
                        caret_blink = false;
                        elapsed_time = 0.0f;
                    }
                }
            }
            if (ev.type == ALLEGRO_EVENT_KEY_DOWN) {
                if (ev.keyboard.keycode == ALLEGRO_KEY_INSERT) {
                    insert_mode = !insert_mode;
                }
            }
            if (ev.type == ALLEGRO_EVENT_KEY_CHAR) {
                
                if (ev.keyboard.keycode == ALLEGRO_KEY_ENTER) {
                    /// Add the current string to our set of strings and clear
                    CreateUstrFromCodePoints(intcharbuf , nchars , &ustr);
                    
                    /// Insert the string in the front of the array
                    if (nlines == MAXNLINES) {
                        al_ustr_free(ustrs[MAXNLINES - 1]);/// So we don't leak memory
                    }
                    for (int i = nlines ; i >= 1 ; --i) {
                        ustrs[i] = ustrs[i - 1];
                    }
                    ustrs[0] = ustr;
                    nlines++;
                    if (nlines > MAXNLINES) {
                        nlines = MAXNLINES;
                    }
                    ustr = al_ustr_new("");
                    memset(intcharbuf , 0 , sizeof(unsigned int)*(MAXNCHARS + 1));
                    nchars = 0;
                    caret = 0;
                }
                else if (ev.keyboard.keycode == ALLEGRO_KEY_BACKSPACE) {
                    /// Delete character before caret position
                    if (caret >= 1) {
                        for (int crt = caret - 1 ; crt < MAXNCHARS ; ++crt) {
                            intcharbuf[crt] = intcharbuf[crt + 1];
                        }
                        --nchars;
                        if (nchars == 0) {
                            memset(intcharbuf , 0 , MAXNCHARS*sizeof(unsigned int));
                        }
                        --caret;
                    }
                }
                else if (ev.keyboard.keycode == ALLEGRO_KEY_DELETE) {
                    if (caret < nchars) {
                        /// Delete character at the caret position
                        for (int crt = caret ; crt < nchars ; ++crt) {
                            intcharbuf[crt] = intcharbuf[crt + 1];
                        }
                        --nchars;
                        if (nchars == 0) {
                            memset(intcharbuf , 0 , MAXNCHARS*sizeof(unsigned int));
                        }
                        if (nchars < 0) {
                            nchars = 0;
                        }
                    }
                }
                else if (ev.keyboard.keycode == ALLEGRO_KEY_LEFT) {
                    /// Move caret left
                    caret--;
                    if (caret < 0) {caret = 0;}
                    caret_blink = false;
                    elapsed_time = 0.0f;
                }
                else if (ev.keyboard.keycode == ALLEGRO_KEY_RIGHT) {
                    /// Move caret right
                    caret++;
                    if (caret >= nchars) {
                        caret = nchars;
                    }
                    caret_blink = false;
                    elapsed_time = 0.0f;
                }
                else {
                    /// We have a character, add it to our buffer
                    unsigned int uc = ev.keyboard.unichar;
                    
                    printf("%X typed\n" , uc);
                    
                    if (uc >= 32) {
                        if (insert_mode) {
                            /// Move all chars right from the caret on
                            for (int crt = MAXNCHARS - 1 ; crt >= caret ; --crt) {
                                intcharbuf[crt] = intcharbuf[crt - 1];
                            }
                            /// We're inserting a new character, so bump the char count
                            nchars += 1;
                            if (nchars > MAXNCHARS) {
                                nchars = MAXNCHARS;
                            }
                        }
                        else {
                            /// Overwrite mode
                            if (caret == nchars && nchars < MAXNCHARS) {
                                nchars++;/// We're on the right end of the string, increase the char count
                            }
                        }
                        /// Add our new character
                        intcharbuf[caret++] = uc;
                        if (caret >= MAXNCHARS) {
                            caret = MAXNCHARS;
                        }
                    }
                }
                
                /// Refresh our USTR
                CreateUstrFromCodePoints(intcharbuf , nchars , &ustr);
                
            }
        }
        while(!al_is_event_queue_empty(q));

    }
    
    for (int i = 0 ; i < nlines ; ++i) {
        al_ustr_free(ustrs[i]);
        ustrs[i] = 0;
    }
    nlines = 0;
    al_ustr_free(ustr);
    ustr = 0;
    nchars = 0;

    al_uninstall_system();
    
    return 0;
}
