
#include <stdio.h>
#include <allegro5/allegro.h>
#include <allegro5/allegro_audio.h>
#include <allegro5/allegro_primitives.h>
#include <allegro5/allegro_acodec.h>
#include "kiss_fft.h"
#include "kiss_fftr.h"

/* Comment out the following line to use 16-bit audio */
#define WANT_8_BIT_DEPTH

#define SCRW 1024
#define SCRH 800

#define BUFFSIZE (SCRW*8)

const ALLEGRO_AUDIO_DEPTH audio_depth = ALLEGRO_AUDIO_DEPTH_UINT8;
typedef uint8_t* audio_buffer_t;
const uint8_t sample_center = 128;
const int8_t min_sample_val = 0x80;
const int8_t max_sample_val = 0x7f;
const int sample_range = 0xff;
const int sample_size = 1;

const unsigned int samples_per_fragment = BUFFSIZE;

const unsigned int frequency = 44100;
const unsigned int max_seconds_to_record = 60 * 5;

const unsigned int playback_fragment_count = 4;
const unsigned int playback_samples_per_fragment = 4096;

double cardfreq = 44100;

float unbiased[BUFFSIZE];
float array[BUFFSIZE];
float magnitudes[BUFFSIZE];
//float buf[BUFFSIZE];	//used to rebuild time from frequency

kiss_fft_cpx* copycpx(float *mat, int nframe)
{
	int i;
	kiss_fft_cpx *mat2;
	mat2=(kiss_fft_cpx*)KISS_FFT_MALLOC(sizeof(kiss_fft_cpx)*nframe);
	kiss_fft_scalar zero;
	memset(&zero,0,sizeof(zero) );
	for(i=0; i<nframe ; i++)
	{
		mat2[i].r = mat[i];
		mat2[i].i = zero;
	}
	return mat2;
}

int main(int argc, const char **argv)
{
	ALLEGRO_AUDIO_RECORDER *r;
	ALLEGRO_AUDIO_STREAM *s;

	ALLEGRO_EVENT_QUEUE *q;
	ALLEGRO_DISPLAY *d;
	float maxmag;
	float ultimatemag = 0.0;
	float sum;

	int prev = 0;

	al_init();

	if (!al_init_primitives_addon())
	{
		fprintf(stderr,"Unable to initialize primitives addon\n");
	}

	if (!al_install_keyboard())
	{
		fprintf(stderr,"Unable to install keyboard\n");
	}

	if (!al_install_audio())
	{
		fprintf(stderr,"Unable to initialize audio addon\n");
	}

	if (!al_init_acodec_addon())
	{
		fprintf(stderr,"Unable to initialize acodec addon\n");
	}

	/* Note: increasing the number of channels will break this demo. Other
	* settings can be changed by modifying the constants at the top of the
	* file.
	*/
	r = al_create_audio_recorder(1000, samples_per_fragment, frequency,
	audio_depth, ALLEGRO_CHANNEL_CONF_1);
	if (!r)
	{
		fprintf(stderr,"Unable to create audio recorder\n");
	}

	s = al_create_audio_stream(playback_fragment_count,
	playback_samples_per_fragment, frequency, audio_depth,
	ALLEGRO_CHANNEL_CONF_1);
	if (!s)
	{
		fprintf(stderr,"Unable to create audio stream\n");
	}

	al_reserve_samples(0);
	al_set_audio_stream_playing(s, false);
	al_attach_audio_stream_to_mixer(s, al_get_default_mixer());

	q = al_create_event_queue();

	/* Note: the following two options are referring to pixel samples, and have
	* nothing to do with audio samples. */
	al_set_new_display_option(ALLEGRO_SAMPLE_BUFFERS, 1, ALLEGRO_SUGGEST);
	al_set_new_display_option(ALLEGRO_SAMPLES, 8, ALLEGRO_SUGGEST);

	d = al_create_display(SCRW, SCRH);

	al_register_event_source(q, al_get_audio_recorder_event_source(r));
	al_register_event_source(q, al_get_audio_stream_event_source(s));
	al_register_event_source(q, al_get_display_event_source(d));
	al_register_event_source(q, al_get_keyboard_event_source());

	//create necessary fft buffers
	kiss_fft_cpx out_cpx[BUFFSIZE],*cpx_buf;
	kiss_fftr_cfg fft = kiss_fftr_alloc(BUFFSIZE*2 ,0 ,0,0);
	//kiss_fftr_cfg ifft = kiss_fftr_alloc(BUFFSIZE*2,isinverse,0,0);

	al_start_audio_recorder(r);

	while (true)
	{
		ALLEGRO_EVENT e;

		al_wait_for_event(q, &e);

		if (e.type == ALLEGRO_EVENT_AUDIO_RECORDER_FRAGMENT)
		{

			ALLEGRO_AUDIO_RECORDER_EVENT *re = al_get_audio_recorder_event(&e);
			audio_buffer_t input = (audio_buffer_t) re->buffer;
			int sample_count = re->samples;
			const int R = sample_count / BUFFSIZE;
			int i;


			al_clear_to_color(al_map_rgb(0,0,0));

			//draw the original waveform (time domain)
			for (i = 1; i < SCRW; ++i)
			{
				int j, c = 0;

				/* Take the average of R samples so it fits on the screen */
				for (j = i * R; j < i * R + R && j < sample_count; ++j)
				{
					c += input[j] - sample_center;
				}
				c /= R;

				/* Draws a line from the previous sample point to the next */
				al_draw_line(i - 1,
					     SCRH/4 + ( ( (prev - min_sample_val) / (float) sample_range) * SCRH/2 - 256),
					     i,
					     SCRH/4 + ( ( (c - min_sample_val) / (float) sample_range) * SCRH/2 - 256),
					     al_map_rgb(255,255,255), 1.2);

				prev = c;
			}

			//copy to another buffer and remove any bias
			sum = 0.0;
			for(i=0;i<BUFFSIZE;i++)
			{
				unbiased[i] = input[i];
				sum += unbiased[i];
			}

			sum /= (float)BUFFSIZE;

			for(i=0;i<BUFFSIZE;i++)
			{
				unbiased[i] -= sum;
			}

			maxmag = 0.0;
			//filter out "low input" like noise gate
			for(i=0;i<BUFFSIZE;i++)
			{
				if(maxmag < unbiased[i])
					maxmag = unbiased[i];
			}
			if(maxmag < 2.0)
				goto noise_gate;

			//copy to another buffer while clamping to a triangular "window"
			//Wild Ass Guess:  Pinching off the ends to zero simulates having the sinusoids
			//begin and end at the beginning and end of the buffer to avoid distorting the fft
			//while still keeping the buffer size at a power of two.
			for(i=0;i<BUFFSIZE/2;i++)
			{
				float ttmp = (float)i/(BUFFSIZE/2);
				array[i] = unbiased[i] * ttmp;
				array[BUFFSIZE - i - 1] *= ttmp;	//mirror it on the second half
			}

			cpx_buf = copycpx(array,BUFFSIZE);
			kiss_fftr(fft,(kiss_fft_scalar*)cpx_buf, out_cpx);

			//kiss_fftri(ifft,out_cpx,(kiss_fft_scalar*)out );  //the inverse, which we're not using

			//convert the complex numbers to scalar magnitude via Pythagoras
			maxmag = 0.0;
			for(i=1;i<BUFFSIZE/4;i++)
			{
				magnitudes[i] =  0.005 * (-(sqrt(out_cpx[i].i*out_cpx[i].i) + (out_cpx[i].r * out_cpx[i].r))); //*(float)1.0/i );
				if(magnitudes[i] < maxmag)
					maxmag = magnitudes[i];
			}

			maxmag = -200.0/maxmag;

			for(i=0;i<BUFFSIZE/4;i++)
			{
				magnitudes[i] *= maxmag;
			}

			//draw the frequency domain
			/* Changed this to only include the first 1/4 of the bins because of the max expected frequency
			 * compared to the Nyquist frequency
			for(i=1;i<BUFFSIZE/2 - 1;i++)
			{
				al_draw_line( (i - 1) * 2,
					      SCRH/2 + magnitudes[i-1],
					     i * 2,
					SCRH/2 + magnitudes[i],
					al_map_rgb(255,0,0), 1.2);
			}
			*/

			for(i=1;i<SCRW;i++)
			{
				al_draw_line( (i - 1) * 8,
						SCRH/2 + magnitudes[i-1],
						i * 8,
						SCRH/2 + magnitudes[i],
						al_map_rgb(255,0,0), 1.2);
			}

noise_gate:
			al_flip_display();
			//printf("%f\n",maxmag);
			if(ultimatemag < maxmag)
				ultimatemag = maxmag;
		}

		else if (e.type == ALLEGRO_EVENT_DISPLAY_CLOSE)
		{
			break;
		}
		else if (e.type == ALLEGRO_EVENT_KEY_CHAR)
		{
			if (e.keyboard.unichar == 27)
			{
				/* pressed ESC */
				break;
			}
		}
	}

	/* clean up */
	al_destroy_audio_recorder(r);
	al_destroy_audio_stream(s);
	kiss_fft_cleanup();
	free(fft);
	printf("biggest mag was %f\n",ultimatemag);
	return 0;
}
