
//This shows the bare time domain waveform from the microphone.

#include <stdio.h>
#include <allegro5/allegro.h>
#include <allegro5/allegro_audio.h>
#include <allegro5/allegro_primitives.h>
#include <allegro5/allegro_acodec.h>

/* Comment out the following line to use 16-bit audio */
#define WANT_8_BIT_DEPTH

#define SCRW 1024
#define SCRH 800

const ALLEGRO_AUDIO_DEPTH audio_depth = ALLEGRO_AUDIO_DEPTH_UINT8;
typedef uint8_t* audio_buffer_t;
const uint8_t sample_center = 128;
const int8_t min_sample_val = 0x80;
const int8_t max_sample_val = 0x7f;
const int sample_range = 0xff;
const int sample_size = 1;

const unsigned int samples_per_fragment = SCRW;

const unsigned int frequency = 44100;
const unsigned int max_seconds_to_record = 60 * 5;

const unsigned int playback_fragment_count = 4;
const unsigned int playback_samples_per_fragment = 4096;

double cardfreq = 44100;

int main(int argc, const char **argv)
{
	ALLEGRO_AUDIO_RECORDER *r;
	ALLEGRO_AUDIO_STREAM *s;

	ALLEGRO_EVENT_QUEUE *q;
	ALLEGRO_DISPLAY *d;

	int prev = 0;

	al_init();

	if (!al_init_primitives_addon())
	{
		fprintf(stderr,"Unable to initialize primitives addon\n");
	}

	if (!al_install_keyboard())
	{
		fprintf(stderr,"Unable to install keyboard\n");
	}

	if (!al_install_audio())
	{
		fprintf(stderr,"Unable to initialize audio addon\n");
	}

	if (!al_init_acodec_addon())
	{
		fprintf(stderr,"Unable to initialize acodec addon\n");
	}

	/* Note: increasing the number of channels will break this demo. Other
	* settings can be changed by modifying the constants at the top of the
	* file.
	*/
	r = al_create_audio_recorder(1000, samples_per_fragment, frequency,
	audio_depth, ALLEGRO_CHANNEL_CONF_1);
	if (!r)
	{
		fprintf(stderr,"Unable to create audio recorder\n");
	}

	s = al_create_audio_stream(playback_fragment_count,
	playback_samples_per_fragment, frequency, audio_depth,
	ALLEGRO_CHANNEL_CONF_1);
	if (!s)
	{
		fprintf(stderr,"Unable to create audio stream\n");
	}

	al_reserve_samples(0);
	al_set_audio_stream_playing(s, false);
	al_attach_audio_stream_to_mixer(s, al_get_default_mixer());

	q = al_create_event_queue();

	/* Note: the following two options are referring to pixel samples, and have
	* nothing to do with audio samples. */
	al_set_new_display_option(ALLEGRO_SAMPLE_BUFFERS, 1, ALLEGRO_SUGGEST);
	al_set_new_display_option(ALLEGRO_SAMPLES, 8, ALLEGRO_SUGGEST);

	d = al_create_display(SCRW, SCRH);

	al_set_window_title(d, "The royal whatzzit");

	al_register_event_source(q, al_get_audio_recorder_event_source(r));
	al_register_event_source(q, al_get_audio_stream_event_source(s));
	al_register_event_source(q, al_get_display_event_source(d));
	al_register_event_source(q, al_get_keyboard_event_source());

	al_start_audio_recorder(r);

	while (true)
	{
		ALLEGRO_EVENT e;

		al_wait_for_event(q, &e);

		if (e.type == ALLEGRO_EVENT_AUDIO_RECORDER_FRAGMENT)
		{
			/* We received an incoming fragment from the microphone. In this
			* example, the recorder is constantly recording even when we aren't
			* saving to disk. The display is updated every time a new fragment
			* comes in, because it makes things more simple. If the fragments
			* are coming in faster than we can update the screen, then it will be
			* a problem.
			*/
			ALLEGRO_AUDIO_RECORDER_EVENT *re = al_get_audio_recorder_event(&e);
			audio_buffer_t input = (audio_buffer_t) re->buffer;
			int sample_count = re->samples;
			const int R = sample_count / SCRW;
			int i;


			al_clear_to_color(al_map_rgb(0,0,0));

			/* Draw a pathetic visualization. It draws exactly one fragment
			* per frame. This means the visualization is dependent on the
			* various parameters. A more thorough implementation would use this
			* event to copy the new data into a circular buffer that holds a
			* few seconds of audio. The graphics routine could then always
			* draw that last second of audio, which would cause the
			* visualization to appear constant across all different settings.
			*/
			for (i = 1; i < SCRW; ++i)
			{
				int j, c = 0;

				/* Take the average of R samples so it fits on the screen */
				for (j = i * R; j < i * R + R && j < sample_count; ++j)
				{
					c += input[j] - sample_center;
				}
				c /= R;

				/* Draws a line from the previous sample point to the next */
				al_draw_line(i - 1,
					     SCRH/2 + ( ( (prev - min_sample_val) / (float) sample_range) * SCRH/2 - 256),
					     i,
					     SCRH/2 + ( ( (c - min_sample_val) / (float) sample_range) * SCRH/2 - 256),
					     al_map_rgb(255,255,255), 1.2);

				prev = c;
			}

			al_flip_display();
		}

		else if (e.type == ALLEGRO_EVENT_DISPLAY_CLOSE)
		{
			break;
		}
		else if (e.type == ALLEGRO_EVENT_KEY_CHAR)
		{
			if (e.keyboard.unichar == 27)
			{
				/* pressed ESC */
				break;
			}
		}
	}

	/* clean up */
	al_destroy_audio_recorder(r);
	al_destroy_audio_stream(s);
	return 0;
}
