import allegro5.allegro;
import allegro5.allegro_audio;
import std.exception;
import std.stdio;
import std.algorithm;
import std.math;
import std.random;

//Attempt to simulate the chip used in the BBC Computer: http://members.casema.nl/hhaydn/howel/parts/76489.htm

ALLEGRO_VOICE* voice;
ALLEGRO_MIXER* mixer;
ALLEGRO_AUDIO_STREAM* stream;

enum bufferCount = 2;
enum samples = 256;
enum samfreq = 44100;
enum maxAmpPerChannel = 2000;

struct OSCILLATOR {
	float period = 0;
	float periodLeft = 0;
	float value = 0;
	float amplitude = 0;
}

OSCILLATOR oscillators[4];

enum NOISE_TRACK = 1, NOISE_PERIODIC = 2;
int noiseBits;
int noisePeriod;

enum valueLossFactor = 0.999f;

//Simple low-pass filter softens the sound a little.
float filterPos = 0;
enum filterBrightness = 0.5f;		//0 to 1

void initSound() {
	try {
		enforce(al_install_audio(), "al_install_audio");
		enforce(stream = al_create_audio_stream(bufferCount, samples, samfreq, ALLEGRO_AUDIO_DEPTH.ALLEGRO_AUDIO_DEPTH_INT16, ALLEGRO_CHANNEL_CONF.ALLEGRO_CHANNEL_CONF_1), "al_create_audio_stream");
		enforce(voice = al_create_voice(samfreq, ALLEGRO_AUDIO_DEPTH.ALLEGRO_AUDIO_DEPTH_INT16, ALLEGRO_CHANNEL_CONF.ALLEGRO_CHANNEL_CONF_1), "al_create_voice");
		if (!al_attach_audio_stream_to_voice(stream, voice)) {
			writeln("Oops: there may be latency!");
			enforce(mixer = al_create_mixer(samfreq, ALLEGRO_AUDIO_DEPTH.ALLEGRO_AUDIO_DEPTH_INT16, ALLEGRO_CHANNEL_CONF.ALLEGRO_CHANNEL_CONF_1), "al_create_mixer");
			enforce(al_attach_audio_stream_to_mixer(stream, mixer), "al_attach_audio_stream_to_mixer");
			enforce(al_attach_mixer_to_voice(mixer, voice), "al_attach_mixer_to_voice");
		}

		foreach (i; 0..4) setNote(i, 0, 0);
	}
	catch (Exception e) {
		writeln("Error initialising sound! " ~ e.toString());
		stopSound();
	}
}

void updateSound() {
	if (stream is null) return;

	if (noiseBits & NOISE_TRACK) {
		oscillators[0].period = oscillators[1].period;
		oscillators[0].periodLeft = oscillators[1].periodLeft;
	}

	while (true) {
		auto frag = (cast(short*)al_get_audio_stream_fragment(stream))[0..samples];
		if (frag.ptr == null) return;

		foreach (i; 0..samples) {
			float sum = 0;
			
			/*Test code for viewing the waveform
			{
				import textbuffer;
				scr[(frag[i]+32768)*h/65536][i*w/samples] = '#';
			}
			//*/

			foreach (channel, ref o; oscillators) {
				float mixedValue;
				if (--o.periodLeft <= 0) {
					float oldValue = o.value;
					float t = -o.periodLeft;
					float newValue;
					o.periodLeft += o.period;
					if (channel == 0) {
						if (noiseBits & NOISE_PERIODIC) {
							noisePeriod = (noisePeriod + 1) % 17;
							newValue = (noisePeriod == 0 ? o.amplitude : -o.amplitude);
						} else {
							newValue = (uniform(0,2) == 0 ? o.amplitude : -o.amplitude);
						}
					} else {
						newValue = (o.value < 0 ? o.amplitude : -o.amplitude);
					}
					if (sgn(newValue) != sgn(oldValue)) o.value = newValue;
					mixedValue = oldValue + (o.value - oldValue) * t;
				} else {
					mixedValue = o.value;
				}
				o.value *= valueLossFactor;
				sum += mixedValue;
			}

			filterPos += (sum - filterPos) * filterBrightness;
			frag[i] = cast(short)filterPos;
		}

		al_set_audio_stream_fragment(stream, frag.ptr);
	}
}

void stopSound() {
	al_destroy_voice(voice);
	al_destroy_mixer(mixer);
	al_destroy_audio_stream(stream);
	al_uninstall_audio();
}

//API

void setNote(int channel, int pitch, int volume) {
	//15 volume levels, 0 to 15
	oscillators[channel].amplitude = volume * (maxAmpPerChannel / 15f);

	//Pitch is BBC computer style.
	if (channel == 0) {
		noiseBits = 0;
		if ((pitch & 4) == 0) noiseBits |= NOISE_PERIODIC;
		switch (pitch & 3) {
			case 0: oscillators[0].period = 512 * samfreq / 4000000f; break;
			case 1: oscillators[0].period = 1024 * samfreq / 4000000f; break;
			case 2: oscillators[0].period = 2048 * samfreq / 4000000f; break;
			default: noiseBits |= NOISE_TRACK; break;	//period will be synced with channel 1 in updateNoise()
		}
	} else {
		//53 is Middle C. Make channels 2 and 3 slightly different from 1.
		//The chip has a 10-bit period register, so limit our precision similarly.
		int periodMicroseconds = cast(int)((1024 - (channel >> 1) * 8) / pow(2, pitch / 48f)) * 16;
		oscillators[channel].period = periodMicroseconds * (samfreq / 4000000f);
	}
}

void cutLouderThan(int volume) {
	float amp = volume * (maxAmpPerChannel / 15);
	foreach (ref o; oscillators) {
		if (o.amplitude > amp) o.amplitude = 0;
	}
}
